# Copyright (c) 2014-2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model
from CliModel import Int
from CliModel import Float
from CliModel import Enum
from CliModel import Str
from CliModel import Bool
from CliModel import Submodel
from CliModel import Dict
from CliModel import List
from CliModel import GeneratorList
from CliModel import GeneratorDict
from CliMode.Intf import IntfMode
from ArnetModel import Ip4Address
from IntfModel import Interface
from HumanReadable import formatTimeInterval
from Ark import ReversibleDict
from Ark import timestampToStr
from DeviceNameLib import kernelIntfFilter
import Tac
import socket
import struct
import os
import ctypes
import ConfigMount
import LazyMount
import OspfConsts
from RibCapiLib import timestampToLocalTime
import PyRibAmiClient

ospfConfigDir = None
ospfStatusDir = None

def ospfConfig():
   return ospfConfigDir

def ospfStatus():
   return ospfStatusDir

#-------------------------------------------------------------------------------
# Ospf Options Model
#-------------------------------------------------------------------------------
OSPF_OPTIONS_MAP = { 'T' : 'multitopologyCapability',
                     'E' : 'externalRoutingCapability',
                     'MC' : 'multicastCapability',
                     'NP' : 'nssaCapability',
                     'L' : 'linkLocalSignaling',
                     'DC' : 'demandCircuitsSupport',
                     'O' : 'opaqueLsaSupport',
                     'DN' : 'doNotUseInRouteCalc' }

class OspfOptions( Model ):
   multitopologyCapability = Bool( default=False,
                                   help='Originating router supports'
                                        ' Multitopology OSPF' )
   externalRoutingCapability = Bool( default=False,
                                     help='Originating router capable of accepting'
                                          ' AS External LSAs' )
   multicastCapability = Bool( default=False,
                               help='Originating router supports multicast'
                                    ' extensions to OSPF' )
   nssaCapability = Bool( default=False,
                          help='Originating router supports Type-7'
                               ' NSSA-External-LSAs' )
   linkLocalSignaling = Bool( default=False,
                              help='OSPF packet contains Link-Local Signaling'
                                   ' data block' )
   demandCircuitsSupport = Bool( default=False,
                                 help='Originating router supports OSPF'
                                      ' over Demand Circuits' )
   opaqueLsaSupport = Bool( default=False,
                            help='Originating router supports Type 9,10 and 11'
                                 ' Opaque LSAs' )
   doNotUseInRouteCalc = Bool( default=False,
                               help='A type 3,5 or 7 LSA is sent from'
                                    ' a PE router to a CE router' )

   def render( self ):
      optionsStr = ''
      for key, option in OSPF_OPTIONS_MAP.items():
         if getattr( self, option ):
            optionsStr += key + ' '
      print '  Options is {}'.format( optionsStr.strip() 
                                      if optionsStr else '(null)' )

   def processData( self, data ):
      if 'options' in data:
         optionsPresent = data.pop( 'options' ).split()
         for option in optionsPresent:
            setattr( self, OSPF_OPTIONS_MAP[ option ], True )

#-------------------------------------------------------------------------------
# "show ip ospf [<instance-id>] [vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
MAX_METRIC_UNSET_REASON_MAP = { 1 : 'maxMetricUnsetTimerExpired',
                                2 : 'maxMetricUnsetBgpConverged',
                                3 : 'maxMetricUnsetBgpTimerExpired',
                                4 : 'maxMetricUnsetUserConfig' }

GR_SPEAKER_EXIT_REASON_MAP = { 0 : 'lastExitNone',
                               1 : 'lastExitSuccess',
                               2 : 'lastExitHelperAborted',
                               3 : 'lastExitNeighborStateChanged',
                               4 : 'lastExitGracePeriodExpired',
                               5 : 'lastExitDisabled' }

grSpkrExitReasonStr = {
   'lastExitNone'                 : 'None',
   'lastExitSuccess'              : 'Success',
   'lastExitHelperAborted'        : 'Helper aborted restart',
   'lastExitNeighborStateChanged' : 'Neighbor state changed',
   'lastExitGracePeriodExpired'   : 'Grace period expired',
   'lastExitDisabled'             : 'Graceful restart disabled' }

GR_RESTART_REASON_MAP = { 0 : 'Unknown',
                          1 : 'RIB restart',
                          2 : 'SSO',
                          3 : 'ASU2' }

grRestartTypeStr = {
   'Unknown'     : 'Unplanned',
   'RIB restart' : 'Unplanned',
   'SSO'         : 'Unplanned',
   'ASU2'        : 'Planned' }

# nospf_gr_state_t and ospf3_gr_state_t
GR_STATE_MAP = { 0 : 'none',
                 1 : 'graceful',
                 2 : 'signaled',
                 3 : 'pending',
                 4 : 'exiting' }

LSA_TYPE_MAP = ReversibleDict( { 1 : 'routerLsa',
                                 2 : 'networkLsa',
                                 3 : 'summaryLsa',
                                 4 : 'summaryAsbrLsa',
                                 5 : 'asExternalLsa',
                                 7 : 'nssaLsa',
                                 9 : 'opaqueType9Lsa',
                                 10: 'opaqueType10Lsa',
                                 11: 'opaqueType11Lsa' } )

ATS_NOT_TRANSLATOR = 0
ATS_TRANSLATOR = 1
ATS_EXTENDED_TRANSLATOR = 2
ATS_STOP_TRANSLATOR = 3

AREA_ZERO = '0.0.0.0'
NUM_MS_PER_SEC = 1000

def printOspfLsaHeader():
   print '\n%-4s %-15s %-15s %-11s %-12s %-8s' % ( 'Type', 'Link ID',
                                                   'ADV Router', 'Age', 'Seq#',
                                                   'Checksum' )
# We allow time to stand still for testing.
if 'CLI_NOW_IS' in os.environ:
   cur_time = float( os.environ[ 'CLI_NOW_IS' ] )
   def now():
      return cur_time
else:
   now = Tac.now

def renderBool( value ):
   return "Yes" if value else "No"

def renderSeconds( seconds ):
   return "%d second%s" % ( seconds, "" if int( seconds ) == 1 else "s" )

class OspfShowInstAreaRangeModel( Model ):
   address = Ip4Address( help='Area range address' )
   maskLen = Int( help='Area range mask length' )
   cost = Int( help='Area range cost' )
   doNotAdvertise = Bool( default=False, optional=True,
                          help='Do not advertise the area range. The cost value '
                               'is invalid if this flag is set to true' )

   def getKey( self, data ):
      return data[ 'address' ]

   def processData( self, data ):
      if data[ 'cost' ] == 4294967295:
         self.doNotAdvertise = True
      return data

   def render( self ):
      if self.doNotAdvertise:
         print "      %s/%d DoNotAdvertise" % ( self.address, self.maskLen )
      else:
         print "      %s/%d Cost %d Advertise" % (
               self.address, self.maskLen, self.cost )
       

class OspfShowInstAreaNssaModel( Model ):
   nssaTransAlways = Bool( default=False, 
                           help='Nssa translate always flag for area' )
   nssaTransState = Enum( values=( 'notTranslator', 'translator', 
                                   'extendedTranslator', 'stopTranslator' ),
                          optional=True, 
                          help='Nssa translate state flag for area' )
   nssaTransInt = Int( optional=True, help='Nssa translate interval' )

   def processModel( self, data ):
      if 'nssaTransInt' in data:
         self.nssaTransInt = data.pop( 'nssaTransInt' )
      if 'nssaTransAlways' in data:
         self.nssaTransAlways = ord( data.pop( 'nssaTransAlways' ) ) != 0
      if 'nssaTransState' in data:
         if ord( data[ 'nssaTransState' ] ) == ATS_NOT_TRANSLATOR:
            self.nssaTransState = 'notTranslator'
         elif ord( data[ 'nssaTransState' ] ) == ATS_TRANSLATOR:
            self.nssaTransState =  'translator'
         elif ord( data[ 'nssaTransState' ] ) == ATS_EXTENDED_TRANSLATOR:
            self.nssaTransState = 'extendedTranslator'
         elif ord( data[ 'nssaTransState' ] ) == ATS_STOP_TRANSLATOR:
            self.nssaTransState = 'stopTranslator'
         data.pop( 'nssaTransState' )

class OspfShowInstAreaLsaInfoModel( Model ):
   treeSize = Int( help='Area LSA tree size (number of LSA)' )
   asExternalCksum = Int( help='Area AS external LSA checksum sum' )

class OspfShowInstAreaOpaqueLsaInfoModel( Model ):
   opaqueTreeSize = Int( help='Area opaque link tree size '
                              '(number of opaque link LSA)' )
   opaqueCksum = Int( help='Area opaque link LSA checksum sum' )

class OspfShowInstAreaScopeOpaqueLsaInfoModel( Model ):
   opaqueTreeSize = Int( help='Area opaque scope LSA tree size '
                              '(number of opaque LSA)' )
   opaqueCksumSum = Int( help='Area opaque scope LSA checksum sum' )

class OspfShowInstAreaModel( Model ):
   areaId = Ip4Address( help='OSPF area ID' )
   numIntf = Int( help='Number of interfaces in the area' )
   stubArea = Bool( default=False, help='This area is a stub area' )
   normalArea = Bool( default=False, help='This area is a normal area' )
   teEnabled = Bool( default=False, 
                     help='Traffic engineering enabled in the area' )
   authType = Enum( values=( 'Simple', 'MD5', 'SHA1', 'SHA256', 'SHA384',
                             'SHA512' ), optional=True,
                    help='Area authentication type' )
   spfCount = Int( help='Area SPF count' )
   lsaInformation = Submodel( valueType=OspfShowInstAreaLsaInfoModel,
                              help='Area LSA information' )
   opaqueLsaInformation = Submodel( valueType=OspfShowInstAreaOpaqueLsaInfoModel,
                                    help='Area link opaque scope LSA information' )
   opaqueAreaLsaInformation = Submodel( 
                                valueType=OspfShowInstAreaScopeOpaqueLsaInfoModel,
                                help='Opaque area scope LSA information' )
   nssaInformation = Submodel( valueType=OspfShowInstAreaNssaModel, optional=True, 
                               help='Area NSSA information' )
   rangeList = GeneratorDict( keyType=Ip4Address, 
                              valueType=OspfShowInstAreaRangeModel, 
                              help='Dictionary of address ranges in this OSPF '
                                   'area keyed by the area range address' )
   areaFiltersConfigured = Bool( default=False, optional=True, 
                       help='Area Summary Filters configured' )
   areaFilterPrefixList = Str( optional=True, help='Area Filter prefix-list' )

   def getKey( self, data ):
      return data[ 'areaId' ]

   def processData( self, data ):
      if 'stubArea' in data:
         data[ 'stubArea' ] = ord( data[ 'stubArea' ] ) != 0
      if 'normalArea' in data:
         data[ 'normalArea' ] = ord( data[ 'normalArea' ] ) != 0
      if data[ 'authType' ] == 'None':
         data.pop( 'authType' )
      lsaInfo = OspfShowInstAreaLsaInfoModel()
      lsaInfo.treeSize = data.pop( 'treeSize' )
      lsaInfo.asExternalCksum = data.pop( 'asExternalCksum' )
      data[ 'lsaInformation' ] = lsaInfo
      opaqueLsaInfo = OspfShowInstAreaOpaqueLsaInfoModel()
      opaqueLsaInfo.opaqueTreeSize = data.pop( 'opaqueTreeSize' )
      opaqueLsaInfo.opaqueCksum = data.pop( 'opaqueCksum' )
      data[ 'opaqueLsaInformation' ] = opaqueLsaInfo
      
      opaqueAreaLsaInfo = OspfShowInstAreaScopeOpaqueLsaInfoModel()
      opaqueAreaLsaInfo.opaqueTreeSize = data.pop( 'opaqueAreaTreeSize' )
      opaqueAreaLsaInfo.opaqueCksumSum = data.pop( 'opaqueAreaCksumsum' )
      data[ 'opaqueAreaLsaInformation' ] = opaqueAreaLsaInfo
      
      if 'nssaArea' in data and ord( data[ 'nssaArea' ] ):
         nssa = OspfShowInstAreaNssaModel()
         nssa.processModel( data )
         data[ 'nssaInformation' ] = nssa
      data[ 'teEnabled' ] = ord( data[ 'teEnabled' ] ) != 0
      if 'areaFiltersConfigured' in data:
         data[ 'areaFiltersConfigured' ] = \
            ord( data[ 'areaFiltersConfigured' ] ) != 0

      return data

   def render( self ):
      if self.areaId == AREA_ZERO:
         print " Area BACKBONE(%s)" % self.areaId
      else:
         print " Area %s" % self.areaId

      print " Number of interface in this area is %d" % \
            self.numIntf

      if self.normalArea:
         print "   It is a normal area"
      if self.nssaInformation:
         print "   It is a NSSA area"
      if self.stubArea:
         print "   It is a stub area"

      if self.teEnabled:
         print "   Traffic engineering is enabled"
      else:
         print "   Traffic engineering is disabled"

      print "   Area has %s authentication " % self.authType
      print "   SPF algorithm executed %d times" % self.spfCount
      print "   Number of LSA %d. Checksum Sum %d" % ( 
            self.lsaInformation.treeSize, self.lsaInformation.asExternalCksum )
      print "   Number of opaque link LSA %d. Checksum Sum %d" % ( 
            self.opaqueLsaInformation.opaqueTreeSize, 
            self.opaqueLsaInformation.opaqueCksum )
      print "   Number of opaque area LSA %d. Checksum Sum %d" % (
                   self.opaqueAreaLsaInformation.opaqueTreeSize,
                   self.opaqueAreaLsaInformation.opaqueCksumSum )
      
      if self.nssaInformation:
         if self.nssaInformation.nssaTransState == 'notTranslator':
            action = "No"
         else:
            action = "Perform"

         extTranslate = ""
         if self.nssaInformation.nssaTransState == 'extendedTranslator':
            extTranslate = " (stability interval)"

         always = ""
         if self.nssaInformation.nssaTransAlways:
            always = ", translate always"

         print "   %s type-7/type-5 LSA translation%s%s" % ( 
               action, extTranslate, always )
         print "   nssa-translator-stability-interval is %s" % \
               self.nssaInformation.nssaTransInt

      printRangeListHeader = True
      for _, rangeModel in self.rangeList:
         if printRangeListHeader is True:
            print "   Area ranges are"
            printRangeListHeader = False
         rangeModel.render()

      if  self.areaFiltersConfigured:
         print "   Area summary filters configured"

      if self.areaFilterPrefixList:
         print "   Area summary filter prefix-list %s" % \
               self.areaFilterPrefixList


class OspfShowInstMaxLsaInfoModel( Model ):
   maxLsa = Int( help='Maximum number of LSAs allowed' )
   maxLsaThreshold = Int( help='Max LSA threshold for warning message' )
   maxLsaWarningOnly = Bool( default=False, help='Max LSA warning-only option' )
   maxLsaIgnoreTime = Int( help='Max LSA ignore time (mins)' )
   maxLsaResetTime = Int( help='Max LSA reset time (mins)' )
   maxLsaAllowedIgnoreCount = Int( help='Max LSA allowed ignore count' )
   maxLsaCurrentIgnoreCount = Int( help='Max LSA current ignore count' )
   maxLsaIgnoring = Bool( default=False, help='Ignoring max LSA condition' )
   maxLsaIgnoreTimeEnd = Float( optional=True, help='Max LSA ignore time end' )

   def processModel( self, data ):
      status = ospfStatus()
      config = ospfConfig()

      instanceId = data[ 'instanceId' ]
      instanceConfig = config.instanceConfig.get( instanceId )
      vrf = data.get( 'vrf', instanceConfig.vrfName )
      vrfStatus = status.get( vrf, None )
      if vrfStatus is not None:
         instanceStatus = vrfStatus.instanceStatus.get( instanceId, None )
      else:
         instanceStatus = None
      if instanceConfig.maxLsa is not None:
         self.maxLsa = instanceConfig.maxLsa
      else:
         self.maxLsa = 0
      if instanceConfig.maxLsaThreshold is not None:
         self.maxLsaThreshold = instanceConfig.maxLsaThreshold
      else:
         self.maxLsaThreshold = 0
      if instanceConfig.maxLsaWarningOnly:
         self.maxLsaWarningOnly = instanceConfig.maxLsaWarningOnly
      else:
         self.maxLsaWarningOnly = instanceConfig.maxLsaWarningOnlyDefault
      if instanceConfig.maxLsaIgnoreTime is not None:
         self.maxLsaIgnoreTime = instanceConfig.maxLsaIgnoreTime
      else:
         self.maxLsaIgnoreTime = 0
      if instanceConfig.maxLsaResetTime is not None:
         self.maxLsaResetTime = instanceConfig.maxLsaResetTime
      else:
         self.maxLsaResetTime = 0
      if instanceConfig.maxLsaIgnoreCount is not None:
         self.maxLsaAllowedIgnoreCount = instanceConfig.maxLsaIgnoreCount
         if instanceStatus is not None:
            self.maxLsaCurrentIgnoreCount = instanceStatus.maxLsaIgnoreCount
         else:
            self.maxLsaCurrentIgnoreCount = 0
      else:
         self.maxLsaAllowedIgnoreCount = 0
         self.maxLsaCurrentIgnoreCount = 0
      if instanceStatus is not None and instanceStatus.maxLsaIgnoring:
         self.maxLsaIgnoring = True
         if ( not instanceStatus.maxLsaIgnoreCount >
               instanceConfig.maxLsaIgnoreCount ):
            self.maxLsaIgnoreTimeEnd = \
               instanceStatus.maxLsaIgnoreTimeEnd - now()
      else:
         self.maxLsaIgnoring = False

class OspfShowInstSpfInfoModel( Model ):
   spfInterval = Int( help='SPF interval' )
   spfStartInterval = Int( help='SPF start interval' )
   spfHoldInterval = Int( help='SPF hold interval' )
   spfCurrHoldInterval = Int( help='SPF current hold interval' )
   spfMaxWaitInterval = Int( help='SPF max wait interval' )
   lastSpf = Int( help='Last SPF run (secs)' )
   nextSpf = Int( help='Next SPF run (msecs)' )

   def processModel( self, data ):
      self.lastSpf = data.pop( 'lastSpf' )
      self.nextSpf = data.pop( 'nextSpf' )
      self.spfStartInterval = data.pop( 'spfStartInterval' )
      self.spfHoldInterval = data.pop( 'spfHoldInterval' )
      self.spfCurrHoldInterval = data.pop( 'spfCurrHoldInterval' )
      self.spfMaxWaitInterval = data.pop( 'spfMaxWaitInterval' )
      # spfInterval is deprecated by SPF throttling timers.  Since this was a
      # non-optional attribute in the model we can't remove it.  We set its
      # value from spfCurrHoldInterval.
      self.spfInterval = self.spfCurrHoldInterval / 1000

class OspfShowInstLsaInfoModel( Model ):
   lsaArrivalInterval = Int( help='LSA arrival interval' )
   lsaStartInterval = Int( help='LSA start interval' )
   lsaHoldInterval = Int( help='LSA hold interval' )
   lsaMaxWaitInterval = Int( help='LSA max wait interval' )
   numLsa = Int( help='Number of LSAs' )

   def processModel( self, data ):
      self.numLsa = data.pop( 'numLsa' )
      self.lsaArrivalInterval = data.pop( 'lsaArrivalInterval' )
      self.lsaStartInterval = data.pop( 'lsaStartInterval' )
      self.lsaHoldInterval = data.pop( 'lsaHoldInterval' )
      self.lsaMaxWaitInterval = data.pop( 'lsaMaxWaitInterval' )

class OspfShowInstExternalLsaInfoModel( Model ):
   treeSize = Int( help='OSPF instance external LSA tree size '
                        '(number of external LSAs' )
   asExternalCksum = Int( help='AS external LSA checksum' )

class OspfShowInstOpaqueLsaInfoModel( Model ):
   opaque = Bool( default=False, help='Opaque LSAs supported' )
   opaqueTreeSize = Int( help='OSPF instance opaque AS LSA tree size '
                              '(number of opaque AS LSAs' )
   opaqueCksum = Int( help='Opaque AS LSA checksum' )

class OspfShowInstMaxMetricTypeInfoModel( Model ):
   maxMetricType = Int( help='Max metric type')
   maxMetricOnStartUpTime = Int( optional=True, 
                                 help='Time interval on startup when maximum metric '
                                      'is advertised (secs)' )
   maxMetricTimerLeft = Int( optional=True, help='Max metric timer left (secs)' )
   maxMetricWaitBgp = Int( optional=True, help='Max metric while Bgp is converging' )
   maxMetricActive = Bool( help='Max metric advertisements being '
                                'currently active or not' )

class OspfShowInstMaxMetricUnsetInfoModel( Model ):
   maxMetricUnsetReason = Enum( optional=True,
                                values = ( 'maxMetricUnsetTimerExpired',
                                           'maxMetricUnsetBgpNotConfigured', 
                                           'maxMetricUnsetBgpConverged',
                                           'maxMetricUnsetBgpTimerExpired', 
                                           'maxMetricUnsetUserConfig' ),
                                help='Max metric unset reason' )
   maxMetricUnsetOrigDuration = Int( optional=True, 
                                     help='Max metric originate duration (secs)' )
   maxMetricUnsetTstamp = Int( help='Max metric unset timestamp' )

class OspfShowInstGracefulRestartInfoModel( Model ):
   gracefulRestart = Bool( help='Graceful restart mode enabled' )
   gracePeriod = Float( help='Maximum time for graceful-restart to complete (secs)' )
   plannedOnly = Bool( help='Restrict graceful-restarts to planned events' )
   state = Enum( optional=True,
                 values=tuple( GR_STATE_MAP.values() ),
                 help='Graceful restart speaker state' )
   restartExpirationTime = Float( help='Time current graceful restart expires' )

   lastExitReason = Enum( optional=True,
                          values=tuple( GR_SPEAKER_EXIT_REASON_MAP.values() ),
                          help='Reason for last graceful restart exit' )
   lastExitTime = Float( help='Time last graceful restart exited' )
   helperMode = Bool( default=True, help='Graceful restart helper mode' )
   helperLooseLsaCheck = Bool( default=True,
                  help='Graceful restart helper allows topology changes' )
   lastRestartReason = Enum( optional=True,
                             values=tuple( GR_RESTART_REASON_MAP.values() ),
                             help='Specific reason for last graceful restart' )
   restartDuration = Int( help='Graceful restart duration in microseconds' )

class OspfShowInstModel( Model ):
   instanceId = Int( help='OSPF instance ID' )
   maxLsaInformation = Submodel( valueType=OspfShowInstMaxLsaInfoModel,
                                 help='Max LSA information' )
   routerId = Ip4Address( optional=True, help='OSPF Router id' )
   referenceBandwidth = Int( optional=True, help='Auto-cost reference bandwidth '
                                                 'in Mbps' )
   asbr = Bool( default=False, help='This OSPF instance is an AS Border Router' )
   abr = Bool( default=False, help='This OSPF instance is an Area Border Router' )
   spfInformation = Submodel( valueType=OspfShowInstSpfInfoModel, optional=True,
                              help='SPF information' )
   lsaInformation = Submodel( valueType=OspfShowInstLsaInfoModel, optional=True,
                              help='LSA information' )
   externalLsaInformation = Submodel( valueType=OspfShowInstExternalLsaInfoModel, 
                                      optional=True,
                                      help='External LSA information' )
   opaqueLsaInformation = Submodel( valueType=OspfShowInstOpaqueLsaInfoModel, 
                                    optional=True, help='Opaque LSA information' )
   vrf = Str( optional=True, help='VRF name' )
   maxMetricTypeInformation = Submodel( valueType=OspfShowInstMaxMetricTypeInfoModel,
                                        optional=True,
                                        help='Max metric type information' )
   maxMetricUnsetInformation = Submodel(
                               valueType=OspfShowInstMaxMetricUnsetInfoModel, 
                               optional=True, help='Max metric unset information' )
   maxMetricExt = Int( optional=True, help='Max metric for external LSA' )
   maxMetricIncStub = Int( optional=True, help='Max metric including stubs' )
   maxMetricSum = Int( optional=True, help='Max metric for summary LSA' )
   numAreas = Int( default=0, help='Number of areas' )
   numNormalAreas = Int( default=0, help='Number of normal areas' )
   numStubAreas = Int( default=0, help='Number of stub areas' )
   numNssaAreas = Int( default=0, help='Number of nssa areas' )
   adjacencyExchangeStartThreshold = Int( optional=True,
                                          help='Adjacency exchange-start threshold' )
   lsaRetransmissionThreshold = Int( default=10,
                                     help='LSA retransmission threshold' )
   ecmpMaximumNexthops = Int( optional=True,
                              help='Maximum number of ECMP nexthops' )
   floodPacing = Int( help='Flood packet pacing timer in msecs' ) 
   areaList = GeneratorDict( optional=True, keyType=Ip4Address,
                             valueType=OspfShowInstAreaModel, 
                             help='List of areas in this OSPF instance' )
   gracefulRestartInfo = Submodel( valueType=OspfShowInstGracefulRestartInfoModel,
                                   optional=True,
                                   help='Graceful restart information' )
   shutDown = Bool( default=False,
                    help='Indicates if the OSPFv2 instance is currently shut down' )
   numBackboneNeighbors = Int( default=0, help='Number of backbone neighbors' )
   teRouterId = Ip4Address( optional=True, 
                            help='OSPF traffic engineering router ID' )
   tunnelRoutesEnabled = Bool( default=False, optional=True,
                               help='OSPF routes over GRE tunnel enabled' )

   def processData( self, data ):
      maxLsaInfo = OspfShowInstMaxLsaInfoModel()
      maxLsaInfo.processModel( data )
      data[ 'maxLsaInformation' ] = maxLsaInfo

      instanceConfig = ospfConfig().instanceConfig.get( data[ 'instanceId'] )
      if instanceConfig.floodPacing is not None: 
         data[ 'floodPacing' ] = instanceConfig.floodPacing 

      if 'routerId' in data:
         if 'asbr' in data:
            data[ 'asbr' ] = ord( data[ 'asbr' ] ) != 0
         if 'abr' in data:
            data[ 'abr' ] = ord( data[ 'abr' ] ) != 0

         spfInfo = OspfShowInstSpfInfoModel()
         spfInfo.processModel( data )
         data[ 'spfInformation' ] = spfInfo

         extLsaInfo = OspfShowInstExternalLsaInfoModel()
         extLsaInfo.treeSize = data.pop( 'treeSize' )
         extLsaInfo.asExternalCksum = data.pop( 'asExternalCksum' )
         data[ 'externalLsaInformation' ] = extLsaInfo

         opaqueLsaInfo = OspfShowInstOpaqueLsaInfoModel()
         if 'opaque' in data:
            opaqueLsaInfo.opaque = ord( data.pop( 'opaque' ) ) != 0
         opaqueLsaInfo.opaqueTreeSize = data.pop( 'opaqueTreeSize' )
         opaqueLsaInfo.opaqueCksum = data.pop( 'opaqueCksum' )
         data[ 'opaqueLsaInformation' ] = opaqueLsaInfo

         lsaInfo = OspfShowInstLsaInfoModel()
         lsaInfo.processModel( data )
         data[ 'lsaInformation' ] = lsaInfo

         if 'maxMetricType' in data:
            maxMetricTypeInfo = OspfShowInstMaxMetricTypeInfoModel()
            maxMetricTypeInfo.maxMetricType = data.pop( 'maxMetricType' )
            if 'maxMetricStartTimer' in data:
               maxMetricTypeInfo.maxMetricOnStartUpTime = \
                     data.pop( 'maxMetricStartTimer' )
            if 'maxMetricTimerLeft' in data:
               maxMetricTypeInfo.maxMetricTimerLeft = \
                     data.pop( 'maxMetricTimerLeft' )
            if 'maxMetricWaitBgp' in data:
               maxMetricTypeInfo.maxMetricWaitBgp = data.pop( 'maxMetricWaitBgp' )
            if 'maxMetricActive' in data:
               maxMetricTypeInfo.maxMetricActive = \
                                 ord( data.pop( 'maxMetricActive' ) ) != 0
            data[ 'maxMetricTypeInformation' ] = maxMetricTypeInfo

         if 'maxMetricUnsetTstamp' in data:
            maxMetricUnsetInfo = OspfShowInstMaxMetricUnsetInfoModel()
            maxMetricUnsetInfo.maxMetricUnsetTstamp = \
                  data.pop( 'maxMetricUnsetTstamp' )
            if 'maxMetricUnsetReason' in data:
               maxMetricUnsetInfo.maxMetricUnsetReason = \
                  MAX_METRIC_UNSET_REASON_MAP[ data.pop( 'maxMetricUnsetReason' ) ]
            if 'maxMetricUnsetOrigDuration' in data:
               maxMetricUnsetInfo.maxMetricUnsetOrigDuration = \
                     data.pop( 'maxMetricUnsetOrigDuration' )
            data[ 'maxMetricUnsetInformation' ] = maxMetricUnsetInfo

         gracefulRestartInfo = OspfShowInstGracefulRestartInfoModel()
         gracefulRestartInfo.gracefulRestart = instanceConfig.gracefulRestart
         gracefulRestartInfo.gracePeriod = instanceConfig.grGracePeriod
         gracefulRestartInfo.plannedOnly = instanceConfig.grPlannedOnly
         gracefulRestartInfo.helperMode = instanceConfig.grHelper
         if instanceConfig.grHelper:
            gracefulRestartInfo.helperLooseLsaCheck = \
               instanceConfig.grHelperLooseLsaChk
         if 'grState' in data:
            gracefulRestartInfo.state = GR_STATE_MAP[ data.pop( 'grState' ) ]
         if 'grTimeRemaining' in data:
            gracefulRestartInfo.restartExpirationTime  = \
               data.pop( 'grTimeRemaining' ) + Tac.utcNow() 
         if 'grLastExitReason' in data:
            gracefulRestartInfo.lastExitReason = \
               GR_SPEAKER_EXIT_REASON_MAP[ data.pop( 'grLastExitReason' ) ]
         if 'grLastExitTime' in data:
            # time reported is seconds since last exit
            gracefulRestartInfo.lastExitTime = Tac.utcNow() - \
               data.pop( 'grLastExitTime' )
         if 'grLastSpecificReason' in data:
            gracefulRestartInfo.lastRestartReason = \
               GR_RESTART_REASON_MAP[ data.pop( 'grLastSpecificReason' ) ]
         if 'grDuration' in data:
            gracefulRestartInfo.restartDuration = data.pop ( 'grDuration' )

         data[ 'gracefulRestartInfo' ] = gracefulRestartInfo
         data[ 'lsaRetransmissionThreshold' ] = instanceConfig.\
                                                lsaRetransmissionThreshold
         data[ 'tunnelRoutesEnabled' ] = ord( data[ 'tunnelRoutesEnabled' ] ) != 0
      return data

   def render( self ):
      if self.routerId:
         print 'OSPF instance %d with ID %s VRF %s' % (
               self.instanceId, self.routerId, self.vrf )
      else:
         print 'OSPF instance %d ' % self.instanceId

      if self.shutDown:
         print '  Instance is disabled due to shutdown command'
         return
      
      if self.teRouterId:
         print " Traffic engineering router ID is %s" % self.teRouterId
      
      if self.opaqueLsaInformation and self.opaqueLsaInformation.opaque:
         print  " Supports opaque LSA"
      
      if self.maxLsaInformation:
         warnOnly = ''
         if self.maxLsaInformation.maxLsaWarningOnly:
            warnOnly = ' (Warning-only)'

         print 'Maximum number of LSA allowed %s%s' % (
            self.maxLsaInformation.maxLsa,
            warnOnly )
         print "  Threshold for warning message %d%%" % \
               self.maxLsaInformation.maxLsaThreshold
         print '  Ignore-time %d minutes, reset-time %d minutes' % ( 
               self.maxLsaInformation.maxLsaIgnoreTime, 
               self.maxLsaInformation.maxLsaResetTime )
         print '  Ignore-count allowed %d, current %d' % ( 
               self.maxLsaInformation.maxLsaAllowedIgnoreCount, 
               self.maxLsaInformation.maxLsaCurrentIgnoreCount )
         if self.maxLsaInformation.maxLsaIgnoring:
            if self.maxLsaInformation.maxLsaIgnoreTimeEnd is None:
               print '  Permanently ignoring all neighbors due to max-lsa', \
                     'limit'
            else:
               print '  Ignoring all neighbors due to max-lsa limit,', \
                     'time remaining: %s' % formatTimeInterval(
                  self.maxLsaInformation.maxLsaIgnoreTimeEnd )

      if self.routerId:
         if self.asbr:
            print " It is an autonomous system boundary router",
         else:
            print " It is not an autonomous system boundary router",

         if self.abr:
            print "and is an area border router"
         else:
            print "and is not an area border router"

         if self.maxMetricTypeInformation:
            print " Originating router-LSAs with maximum metric"
            if self.maxMetricTypeInformation.maxMetricOnStartUpTime: 
               print "    Condition: on startup for %d seconds," % \
                     self.maxMetricTypeInformation.maxMetricOnStartUpTime,
               if self.maxMetricTypeInformation.maxMetricActive:
                  print "State: active, time left: %d secs" % \
                        self.maxMetricTypeInformation.maxMetricTimerLeft
               else:
                  print "State: inactive"
            elif self.maxMetricTypeInformation.maxMetricWaitBgp:
               print "    Condition: on startup while BGP is converging,",
               if self.maxMetricTypeInformation.maxMetricActive:
                  print "State: active"
               else:
                  print "State: inactive"
            else:
               print "   Condition: always, State: active"

         if self.maxMetricIncStub:
            print "     Advertise stub links with maximum metric in router-LSAs"

         if self.maxMetricSum:
            print "     Advertise summary-LSAs with metric %d" % \
                  self.maxMetricSum

         if self.maxMetricExt:
            print "     Advertise external-LSAs with metric %d" % \
                  self.maxMetricExt

         if self.maxMetricUnsetInformation:
            print "    Unset reason:",
            if self.maxMetricUnsetInformation.maxMetricUnsetReason == \
                  'maxMetricUnsetTimerExpired':
               print "timer expired,",
            elif self.maxMetricUnsetInformation.maxMetricUnsetReason == \
                  'maxMetricUnsetBgpNotConfigured':
               print "BGP not configured,",
            elif self.maxMetricUnsetInformation.maxMetricUnsetReason == \
                  'maxMetricUnsetBgpConverged':
               print "BGP Converged,",
            elif self.maxMetricUnsetInformation.maxMetricUnsetReason == \
                  'maxMetricUnsetBgpTimerExpired':
               print "BGP Default Timer Expired,",
            elif self.maxMetricUnsetInformation.maxMetricUnsetReason == \
                  'maxMetricUnsetUserConfig':
               print "User Configuration,",
            else:
               print "Unknown,",
            print "Originated for: %d seconds" % \
                  self.maxMetricUnsetInformation.maxMetricUnsetOrigDuration
            print "    Unset time: %s" % formatTimeInterval( 
                  self.maxMetricUnsetInformation.maxMetricUnsetTstamp )

         print " Initial SPF schedule delay %d msecs" % \
               self.spfInformation.spfStartInterval
         print " Minimum hold time between two consecutive SPFs %d msecs" % \
               self.spfInformation.spfHoldInterval
         print " Current hold time between two consecutive SPFs %d msecs" % \
               self.spfInformation.spfCurrHoldInterval
         print " Maximum wait time between two consecutive SPFs %d msecs" % \
               self.spfInformation.spfMaxWaitInterval
         print " Minimum LSA arrival %d msecs" % \
               self.lsaInformation.lsaArrivalInterval
         print " Initial LSA throttle delay %d msecs" % \
               self.lsaInformation.lsaStartInterval
         print " Minimum hold time for LSA throttle %d msecs" % \
               self.lsaInformation.lsaHoldInterval
         print " Maximum wait time for LSA throttle %d msecs" % \
               self.lsaInformation.lsaMaxWaitInterval
         if self.floodPacing:
            print " Interface flood pacing timer %d msecs" % \
                self.floodPacing
         print " Number of external LSA %d, Checksum sum %d" % ( 
               self.externalLsaInformation.treeSize, 
               self.externalLsaInformation.asExternalCksum )
         print " Number of opaque AS LSA %d, Checksum sum %d" % ( 
               self.opaqueLsaInformation.opaqueTreeSize, 
               self.opaqueLsaInformation.opaqueCksum )
         print " Number of areas in this router is %d." % \
               self.numAreas,
         print "%d normal, %d stub, %d nssa" % ( 
               self.numNormalAreas, self.numStubAreas, self.numNssaAreas )
         print " Number of LSA %d" % self.lsaInformation.numLsa
         if self.referenceBandwidth:
            print " Reference bandwidth is %d Mbps" % (
               self.referenceBandwidth )
         print " Time since last SPF %d secs" % self.spfInformation.lastSpf
         if self.spfInformation.nextSpf:
            if ( self.spfInformation.nextSpf / NUM_MS_PER_SEC ) > 0:
               print " Scheduled SPF in %d secs" % ( 
                     self.spfInformation.nextSpf / NUM_MS_PER_SEC )
            else:
               print " Scheduled SPF in %d msecs" % \
                     self.spfInformation.nextSpf
         else:
            print " No Scheduled SPF"
         if self.adjacencyExchangeStartThreshold:
            print " Adjacency exchange-start threshold is %d" \
                  % self.adjacencyExchangeStartThreshold
         if self.ecmpMaximumNexthops:
            print " Maximum number of next-hops supported in ECMP is %d" \
                  % self.ecmpMaximumNexthops

         print " Retransmission threshold for LSA is %d" \
               % self.lsaRetransmissionThreshold

         print " Number of backbone neighbors is %d" \
               % self.numBackboneNeighbors

         if self.tunnelRoutesEnabled:
            print " Routes over tunnel interfaces enabled"
         else:
            print " Routes over tunnel interfaces disabled"

         if self.gracefulRestartInfo:
            if not self.gracefulRestartInfo.gracefulRestart:
               print " Graceful-restart is not configured"
            else:
               print " Graceful-restart %sis configured, grace-period %d seconds" \
                     % ( "(Planned-Only) " if self.gracefulRestartInfo.plannedOnly 
                     else "", self.gracefulRestartInfo.gracePeriod )
            # 'gii set ospf restart' command can be used to initiate planned 
            # graceful restart, even if gr is not configured.
            if self.gracefulRestartInfo.gracefulRestart or \
               ( self.gracefulRestartInfo.state != 'none' ):
               expires = ""
               if self.gracefulRestartInfo.restartExpirationTime is not None:
                  timeSecs = int( max( 0, 
                                  self.gracefulRestartInfo.restartExpirationTime - \
                                  Tac.utcNow() ) )
                  expires = "expires in %s seconds" % ( timeSecs )

               print "   State:",
               if self.gracefulRestartInfo.state == 'graceful':
                  print "In progress, " + expires
               elif self.gracefulRestartInfo.state == 'signaled':
                  print "In progress, signaled, " + expires
               elif self.gracefulRestartInfo.state == 'pending':
                  print "Planned pending"
               elif self.gracefulRestartInfo.state == 'exiting':
                  print "Exiting"
               else:
                  print "Inactive"

            if self.gracefulRestartInfo.lastExitReason is not None and \
               self.gracefulRestartInfo.lastExitTime is not None:
               timeSecs = int( Tac.utcNow() - self.gracefulRestartInfo.lastExitTime )
               timeHMS = formatTimeInterval( timeSecs )
               print "   Last graceful restart completed %s ago," % ( timeHMS ),
               print "status:",
               print grSpkrExitReasonStr[ self.gracefulRestartInfo.lastExitReason ]

               if self.gracefulRestartInfo.restartDuration is not None and \
                  self.gracefulRestartInfo.lastRestartReason is not None:

                  grPlanned = grRestartTypeStr[
                     self.gracefulRestartInfo.lastRestartReason ]
                  grReason = grPlanned + " restart, reason " + \
                     self.gracefulRestartInfo.lastRestartReason

                  timeSecs = self.gracefulRestartInfo.lastExitTime - \
                             ( self.gracefulRestartInfo.restartDuration / 1000000 )
                  timeAbsolute = timestampToStr( timeSecs, False, Tac.utcNow() )

                  print "   %s, initiated at %s, duration %1.2f secs" % \
                        ( grReason, timeAbsolute, ( float
                        ( self.gracefulRestartInfo.restartDuration ) / 1000000 ) )

            if self.gracefulRestartInfo.helperMode:
               print " Graceful-restart-helper mode is enabled%s" % ( \
                  " (strict LSA checking)" \
                  if not self.gracefulRestartInfo.helperLooseLsaCheck else "" )
            else:
               print " Graceful-restart-helper mode is disabled"
         for _, areaModel in self.areaList:
            areaModel.render()

      print ''

#-------------------------------------------------------------------------------
# "show ip ospf [<instance-id>] neighbor [<interface-type> <interface-number>]
#                        [<neighbor-id>]
#                        [detail]
#                        [ vrf vrf-name|default|all ]"
#-------------------------------------------------------------------------------
BFD_STATE_MAP = { 0 : 'adminDown',
                  1 : 'init',
                  2 : 'down',
                  3 : 'up' }

GR_HELPER_EXIT_REASON_MAP = { 0 : 'grExitNone',
                              1 : 'grExitAdjNotFull',
                              2 : 'grExitNetworkChanged',
                              3 : 'grExitGracePeriodExpired',
                              4 : 'grExitHelperDisabled',
                              5 : 'grExitGracefulRestartInProgress',
                              6 : 'grExitGraceLsaFlushed' }

grHlprExitReasonStr = {
   'grExitNone'              : 'None',
   'grExitAdjNotFull'        : 'Neighbor state not FULL',
   'grExitNetworkChanged'    : 'Network changed',
   'grExitGracePeriodExpired': 'Grace period expired',
   'grExitHelperDisabled'    : 'Helper disabled',
   'grExitGracefulRestartInProgress' : 'Graceful restart is in-progress',
   'grExitGraceLsaFlushed'   : 'Grace LSA flushed' }

ADJACENCY_STATE_MAP = ReversibleDict( { 'DOWN' : 'down',
                                        'ATTEMPT' : 'attempt',
                                        'INIT' : 'init',
                                        '2 WAYS' : '2Ways',
                                        'EXCH START' : 'exchStart',
                                        'EXCHANGE' : 'exchange',
                                        'LOADING' : 'loading',
                                        'FULL' : 'full',
                                        'GRACEFUL RESTART' : 'gracefulRestart' } )

NGB_INQUIRED_STATE_MAP = { 'down'            : 0,
                           'attempt'         : 1,
                           'init'            : 2,
                           '2-ways'          : 3,
                           'exch-start'      : 4,
                           'exchange'        : 5,
                           'loading'         : 6,
                           'full'            : 7,
                           'graceful-restart': 8 }

NEIGHBOR_OPTIONS_MAP = ReversibleDict( { 'multitopologyCapability' : 'T',
                                         'externalRoutingCapability' : 'E',
                                         'multicastCapability' : 'MC',
                                         'nssaCapability' : 'NP',
                                         'linkLocalSignaling' : 'L',
                                         'demandCircuitsSupport' : 'DC',
                                         'opaqueLsaSupport' : 'O',
                                         'doNotUseInRouteCalc' : 'DN' } )

entryFormatStr = '%-15s %-8s %-8s %-3s %-22s %-11s %-15s %s'

bdrHelp = "OSPF neighbor backup designated router id"

adjacencyStateHelp = '''OSPF neighbor adjacency state.
  down -  No hello received from neighbor
  attempt - Router attempting to establish connection 
  init - Hello received but two-way communication not established
  2Ways - Bi-directional communication established between two routers
  exchStart - Exchange of link state information between routers 
              and their DR and BDR started
  exchange - Routers exchanging database descriptor packets
  loading - Exchange of link state information occuring
  full - Routers are fully adjacent with each other
  gracefulRestart - Neighbor is gracefully restarting'''

drStateHelp = '''OSPF neighbor DR state.
  DR - Neighbor is the designated router
  BDR - Neighbor is the backup designated router
  DROTHER - Neighbor is neither DR or the BDR '''

bfdStateHelp = '''OSPF neighbor BFD state.
  adminDown - Session taken down for administrative purposes 
  init - Waiting for session establishment
  up - Session established successfully
  down - Session is down or has just been created'''

# Performs a reverse DNS lookup for the routerId, when the
# name-lookup is enabled, and DNS server has been configured
def ospfRouterIdToStr( routerId, maxLen = 0 ):
   config = ospfConfig()
   if config.nameLookup:
      try:
         result = socket.gethostbyaddr( '%s' % routerId )
         routerName = result[ 0 ]
      except: # pylint: disable-msg=W0702
         return None
   else:
      return None

   if maxLen:
      return  routerName[ 0 : maxLen ]
   return None

class OspfSegmentRoutingNeighborInfo( Model ):
   srGbBase = Int( help='SR Global Block base' )
   srGbRange = Int( help='SR Global Block range' )

   def renderEntry( self ):
      print '  Segment Routing enabled'
      print '    SRGB Base: %d Range: %d' % ( self.srGbBase, self.srGbRange )

class OspfNeighborDetails( Model ):
   areaId = Ip4Address( help='OSPF neighbor area identifier' ) 
   designatedRouter = Ip4Address( help='OSPF neighbor designated router identifier' )
   backupDesignatedRouter = Ip4Address( help= bdrHelp ) 
   numberOfStateChanges = Int( help='OSPF neighbor number of state changes' )
   stateTime = Float( help='OSPF neighbor current state time' )
   inactivityDefers = Int( help='OSPF neighbor inactivity deferred counter' ) 
   retransmissionCount = Int( help='OSPF neighbor retransmission count' ) 
   bfdState = Enum( values=tuple( BFD_STATE_MAP.values() ), help=bfdStateHelp )
   bfdRequestSent = Bool( default=False, help='OSPF neighbor BFD request is sent' ) 
   grHelperTimer = Float( \
      help='Timestamp of OSPF neighbor graceful period expiration' )
   grNumAttempts = Int( help='OSPF neighbor graceful restart attempts' )
   grExitReason = Enum( optional=True,
                        values=tuple( GR_HELPER_EXIT_REASON_MAP.values() ),
                        help='OSPF neighbor graceful restart exit reason' )
   grLastRestartTime = Float( \
      help='Timestamp of the OSPF neighbor last graceful restart time' )
   srNeighborInfo = Submodel( valueType=OspfSegmentRoutingNeighborInfo,
                              optional=True, help='OSPF SR neighbor details' )

class OspfNeighborsEntry( Model ):
   routerId = Str( help='OSPF neighbor router identifier' )
   routerName = Str( optional=True, help='OSPF neighbor router domain name' )
   interfaceAddress = Ip4Address( help='OSPF neighbor interface address' )
   interfaceName = Interface( help='OSPF neighbor interface name' )
   priority = Int( help='OSPF neighbor priority' )
   adjacencyState = Enum( values=tuple ( ADJACENCY_STATE_MAP.values() ),
                          help=adjacencyStateHelp )
   drState = Enum( values=( 'DR', 'BDR', 'DROTHER' ), help=drStateHelp )
   _vrf = Str( help='Private attribute for VRF name' )
   _instanceId = Int( help='Private attribute for instance ID' )
   options = Submodel( valueType=OspfOptions,
                       help='OSPF neighbor options' )
   inactivity = Float( help='Timestamp of OSPF neighbor inactivity/dead timer' )
   details = Submodel( valueType=OspfNeighborDetails, optional=True,
                       help='OSPF neighbor details' )

   def processData( self, data ):
      routerId = data[ 'routerId' ]
      data[ 'routerName' ] = ospfRouterIdToStr( routerId, 15 )

      self.interfaceName = kernelIntfFilter( data.pop( 'interfaceName' ) )

      state = data.pop( 'state' )
      if state in ADJACENCY_STATE_MAP:
         data[ 'adjacencyState' ] = ADJACENCY_STATE_MAP[ state ]

      intfAddr = data[ 'interfaceAddress']
      if intfAddr == data[ 'drRouterId' ]:
         data[ 'drState' ] =  'DR'
      elif intfAddr == data[ 'bdrRouterId' ]: 
         data[ 'drState' ] =  'BDR'
      elif data[ 'drRouterId' ] != "0.0.0.0":
         data[ 'drState' ] = 'DROTHER'

      self.options = OspfOptions()
      self.options.processData( data )

      data[ 'inactivity' ] = Tac.utcNow() + data[ 'inactivity' ]

      details = OspfNeighborDetails()
      details.areaId = data.pop( 'areaId' )
      details.designatedRouter = data.pop( 'drRouterId' )
      details.backupDesignatedRouter = data.pop( 'bdrRouterId' )
      details.numberOfStateChanges = data.pop( 'numberOfStateChanges' )
      details.stateTime = Tac.utcNow() - data.pop( 'stateTime' )
      details.inactivityDefers = data.pop( 'inactivityDefers' )
      details.retransmissionCount = data.pop( 'retransmissionCount' )

      bfdState = data.pop( 'bfdState' )
      if bfdState in BFD_STATE_MAP:
         details.bfdState = BFD_STATE_MAP[ bfdState ]
      details.bfdRequestSent = ord( data.pop( 'bfdRequestSent' ) ) != 0

      if 'grHelperTimer' in data:
         details.grHelperTimer = data.pop( 'grHelperTimer' ) - Tac.utcNow()

      details.grNumAttempts = data.pop( 'grNumAttempts' )

      if 'grExitReason' in data:
         exitReason = data.pop( 'grExitReason' )
         if exitReason in GR_HELPER_EXIT_REASON_MAP:
            details.grExitReason = GR_HELPER_EXIT_REASON_MAP[ exitReason ]

      if 'grLastRestartTime' in data:
         lastRestartTime = data.pop( 'grLastRestartTime' )
         if lastRestartTime is not None:
            details.grLastRestartTime = Tac.utcNow() - lastRestartTime

      if 'srGbBase' in data and 'srGbRange' in data:
         details.srNeighborInfo = OspfSegmentRoutingNeighborInfo()
         details.srNeighborInfo.srGbBase = data.pop( 'srGbBase' )
         details.srNeighborInfo.srGbRange = data.pop( 'srGbRange' )

      data[ 'details' ] = details

      return data

   def renderEntry( self, _detailsPresent ):
      dead_timer = True 
      routerName = self.routerName
      state = self.adjacencyState
      intfName = self.interfaceName.stringValue

      # Router domain name ( if it exists ) will
      # be printed instead of the router ip
      if routerName is not None:
         self.routerId = routerName

      if self.options.demandCircuitsSupport and state != "down":
         dead_timer = False

      if dead_timer:
         inactivity = max( 0, self.inactivity - Tac.utcNow() )
         inactivity = formatTimeInterval( inactivity )
      else:
         inactivity = "OFF"

      revdAdjStateMap = ADJACENCY_STATE_MAP.reverse()
      state = revdAdjStateMap[ state ]

      if not _detailsPresent:
         if self.drState is not None:
            state += "/" + self.drState

         print entryFormatStr % ( self.routerId, self._instanceId, self._vrf,
                                  self.priority, state, inactivity,
                                  self.interfaceAddress, intfName )
      else:
         interfaceAddr = self.interfaceAddress
         priority = self.priority
         areaId = getattr( self.details, 'areaId' )
         drRouterId = getattr( self.details, 'designatedRouter' )
         bdrRouterId = getattr( self.details, 'backupDesignatedRouter' )
         numStateChanges = getattr( self.details, 'numberOfStateChanges' )
         inactDefer = getattr( self.details, 'inactivityDefers' )
         rmxCount = getattr( self.details, 'retransmissionCount' )

         stateTime = int( Tac.utcNow() - getattr( self.details, 'stateTime' ) )
         stateTime = formatTimeInterval( stateTime )

         bfdState = getattr( self.details, 'bfdState' )
         bfdState = bfdState[ 0 ].upper() + bfdState[ 1: ]

         print 'Neighbor %s, instance %s, VRF %s, interface address %s' % (
            self.routerId, self._instanceId, self._vrf, interfaceAddr )
         print '  In area %s interface %s' % ( areaId, intfName )
         print '  Neighbor priority is %s, State is %s,' \
               ' %s state changes' % ( priority, state, numStateChanges )

         if state == "FULL":
            print '  Adjacency was established %s ago' % ( stateTime )

         print '  Current state was established %s ago' % ( stateTime )
         print '  DR IP Address %s BDR IP Address %s' % ( drRouterId,
                                                       bdrRouterId )  
         self.options.render()

         if dead_timer:
            print '  Dead timer is due in %s' % ( inactivity )
         else:
            print '  Dead timer is OFF'

         print '  Inactivity timer deferred %s %s' % ( inactDefer,
                                                       'time' if inactDefer == 1
                                                       else 'times' )
         print '  LSAs retransmitted %s %s to this' \
               ' neighbor' % ( rmxCount, 'time' if rmxCount == 1 else 'times' )
         
         if getattr( self.details, 'bfdRequestSent' ):
            print '  Bfd request is sent and the state is %s' % ( bfdState )     

         if getattr( self.details, 'grHelperTimer' ):
            grHelperTimerSecs = max( 0, self.details.grHelperTimer + Tac.utcNow() )
            grHelperTimerHMS = formatTimeInterval( grHelperTimerSecs )
            print '  Graceful-restart-helper mode is Active'
            print '  Graceful-restart-helper timer is due in %s' % ( 
                  grHelperTimerHMS )
         else:
            print '  Graceful-restart-helper mode is Inactive'

         print '  Graceful-restart attempts: %s' % ( self.details.grNumAttempts )
         if getattr( self.details, 'grExitReason' ):
            print '  Graceful-restart-helper last exit reason:',
            print grHlprExitReasonStr[ self.details.grExitReason ]

         if getattr( self.details, 'grLastRestartTime' ):
            grLastRestartTimeSecs = int( Tac.utcNow() \
                                         - self.details.grLastRestartTime )
            grLastRestartTimeHMS = formatTimeInterval( grLastRestartTimeSecs )
            print '  Graceful-restart last attempt %s ago' % ( grLastRestartTimeHMS )

         if getattr( self.details, 'srNeighborInfo' ):
            self.details.srNeighborInfo.renderEntry()

class OspfNeighborsInstance( Model ):
   ospfNeighborEntries = GeneratorList( valueType=OspfNeighborsEntry,
                                        help='List of OSPF neighbor entries' )
   _firstVrf = Bool( default=False,
                     help='Private attribute to indicate that this model'
                          ' is part of the first VRF to be returned' )
   _firstInst = Bool( default=False,
                      help='Private attribute to indicate that this model is part'
                           ' of the first instance of this VRF to be returned' )
   _detailsPresent = Bool( default=True,
                           help='Private attribute to indicate that the'
                                ' details submodel is present' )

   def render( self ):
      if not self._detailsPresent:
         if self._firstVrf and self._firstInst:
            print entryFormatStr % ( 'Neighbor ID', 'Instance', 'VRF', 'Pri',
                                     'State', 'Dead Time', 'Address', 'Interface' )

      for entry in self.ospfNeighborEntries:
         entry.renderEntry( self._detailsPresent )

#------------------------------------------------------------------------------
# 'show ip ospf neighbor summary'
#------------------------------------------------------------------------------
class OspfNeighborSummary( Model ):
   numDown = Int( default=0, help='Number of neighbors in Down state' )
   numGracefulRestart = Int( default=0,
                             help='Number of neighbors in Graceful Restart state' )
   numInit = Int( default=0, help='Number of neighbors in Init state' )
   numLoading = Int( default=0, help='Number of neighbors in Loading state' )
   numAttempt = Int( default=0, help='Number of neighbors in Attempt state' )
   numFull = Int( default=0, help='Number of neighbors in Full state' )
   numExchange = Int( default=0, help='Number of neighbors in Exchange state' )
   num2Ways = Int( default=0, help='Number of neighbors in 2 Ways state' )
   numExchStart = Int( default=0,
                       help='Number of neighbors in Exchange Start state' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_NEIGHBORS_SUMMARY

   def renderEntry( self ):
      values = [ ( self.numDown, "DOWN" ),
                 ( self.numGracefulRestart, "GRACEFUL RESTART" ),
                 ( self.numInit, "INIT" ),
                 ( self.numLoading, "LOADING" ),
                 ( self.numAttempt, "ATTEMPT" ),
                 ( self.numFull, "FULL" ),
                 ( self.numExchange, "EXCHANGE" ),
                 ( self.num2Ways, "2 WAYS" ),
                 ( self.numExchStart, "EXCH START" ), ]

      for value, name in values:
         print "%7d neighbors are in state %s" % ( value, name )

class OspfNeighborSummaryInstance( Model ):
   neighborSummary = Submodel( valueType=OspfNeighborSummary,
                               help='OSPF neighbor summary information' )
   _vrf = Str( help='Private attribute for VRF name' )
   _instanceId = Int( help='Private attribute for OSPF instance ID' )

   def render( self ):
      print 'OSPF router with (Instance ID %d) (VRF %s)' % ( self._instanceId,
                                                             self._vrf )
      self.neighborSummary.renderEntry()

#------------------------------------------------------------------------------
# 'show ip ospf neighbor [ interface ] [ routerid ] adjacency-changes'
#------------------------------------------------------------------------------
class OspfNeighborAdjChangesEntry( Model ):
   neighborRouterId = Ip4Address( help='OSPF neighbor router identifier' )
   neighborAddress = Ip4Address( help='OSPF neighbor address' )
   interfaceName = Interface( help='OSPF interface where adjacency is established' )
   adjEstablished = Bool( help='OSPF neighbor interface is established' )
   dropReason = Str( optional=True,
                     help='Reason for the dropped interface adjacency' )
   adjChangeTimeStamp = Float( help='Time when interface adjacency changes in UTC' )
   routerName = Str( optional=True, help='OSPF neighbor router domain name' )

   def processData( self, data ):
      self.interfaceName = kernelIntfFilter( data[ 'intf-name' ] )
      self.adjEstablished = ord( data[ 'established' ] ) != 0
      self.adjChangeTimeStamp = float( data[ 'time' ] )
      self.routerName = ospfRouterIdToStr( data[ 'neighborRouterId' ], 15 )

      return data

   def renderEntry( self ):
      if self.routerName is not None:
         neighborRouter = self.routerName
      else:
         neighborRouter = self.neighborRouterId
      time = timestampToStr( self.adjChangeTimeStamp, False, Tac.utcNow() )
      if self.adjEstablished:
         adjacency = "established"
      else:
         adjacency = "dropped: %s" % self.dropReason
      print "[%s] %s, interface %s adjacency %s" % (
            time, neighborRouter, self.interfaceName, adjacency )

class OspfNeighborAdjChanges( Model ):
   routerId = Ip4Address( help='OSPF router identifier' )
   _vrf = Str( help='Private attribute for VRF name' )
   _instanceId = Int( help='Private attribute for instance ID' )
   shutDown = Bool( default=False,
                    help='Indicates if the OSPF instance is currently shut down' )
   adjacencyChanges = GeneratorList( valueType=OspfNeighborAdjChangesEntry,
                                     help='List of OSPF adjacency changes' )
   def render( self ):
      print 'OSPF instance %d with ID %s, VRF %s' % ( self._instanceId,
                                                      self.routerId, self._vrf )
      if self.shutDown:
         print '  Instance is disabled due to shutdown command'
         return

      for entry in self.adjacencyChanges:
         entry.renderEntry()

#------------------------------------------------------------------------------
# "show ip ospf [<instance-id>] border-routers [vrf vrf-name|default|all]"
#------------------------------------------------------------------------------
class OspfBorderRoutersType( Model ):
   asbr = Bool( default=False,
                help='Router is an autonomous system boundary router' )
   abr = Bool( default=False,
               help='Router is an area border router' )

   def processData( self, data ):
      if 'asbr' in data:
         self.asbr = ord( data.pop( 'asbr' ) ) != 0
      if 'abr' in data:
         self.abr = ord( data.pop( 'abr' ) ) != 0

   def renderEntry( self, routerId, areaId ):
      asbr = 'ASBR' if self.asbr else ''
      abr = 'ABR' if self.abr else ''
      state = " ".join( [ asbr, abr ] ).lstrip()      

      print '%-15s %-15s %s' % ( routerId, areaId, state )

class OspfBorderRoutersEntry( Model ):
   borderRouters = Dict( keyType=Ip4Address,
                         valueType=OspfBorderRoutersType,
                         help='Dictionary of OSPF border routers type'
                              ' indexed by router-id' )
   _areaId = Ip4Address( help='area identifier' )

   def getKey( self, data ):
      assert 'areaId' in data and data[ 'areaId' ]
      return data[ 'areaId' ]

   def overrideHierarchy( self, data ):
      readNext = True
      if not self.borderRouters:
         self._areaId = data[ 'areaId' ]         
         routerType = OspfBorderRoutersType()
         routerType.processData( data )
         self.borderRouters[ data[ 'routerId' ] ] = routerType
         return ( data, readNext )
      elif str( self._areaId ) == self.getKey( data ):
         routerType = OspfBorderRoutersType()
         routerType.processData( data )
         self.borderRouters[ data[ 'routerId' ] ] = routerType
         return ( data, readNext )
      else:
         return ( data, False )

   def renderEntry( self, areaId ):
      for routerId, routerType in self.borderRouters.iteritems():
         routerType.renderEntry( routerId, areaId )

class OspfBorderRoutersInstance( Model ):
   routerId = Ip4Address( help='OSPF router identifier' )
   vrf = Str( help='VRF name' )
   _instanceId = Int( help='Private attribute for instance ID' )
   shutDown = Bool( default=False,
                    help='Indicates if the OSPF instance is currently shut down' )
   ospfBorderRouterEntries = GeneratorDict( keyType=Ip4Address,
                                            valueType=OspfBorderRoutersEntry,
                                            help='Dictionary of OSPF border router '
                                                 'entries indexed by area-id' )
   def render( self ):
      print 'OSPF instance %d with ID %s, VRF %s' % ( self._instanceId,
                                                      self.routerId, self.vrf )
      if self.shutDown:
         print '  Instance is disabled due to shutdown command'
         return
      print '%-15s %-15s %4s' % ( 'Router ID', 'Area', 'Type' )
      for areaId, entry in self.ospfBorderRouterEntries:
         entry.renderEntry( areaId )

class OspfDatabaseSummary( Model ):
   numRouter = Int( default=0, help="Number of router LSAs" )
   numNetwork = Int( default=0, help="Number of network LSAs" )
   numSummary = Int( default=0, help="Number of summary LSAs" )
   numSummaryAsbr = Int( default=0, help="Number of summary ASBR LSAs" )
   numAsex = Int( default=0, help="Number of AS-external LSAs" )
   numNssa = Int( default=0, help="Number of type-7 NSSA LSAs" )
   numOpaqueLink = Int( default=0, help="Number of opaque link LSAs" )
   numOpaqueArea = Int( default=0, help="Number of opaque area LSAs" )
   numOpaqueAs = Int( default=0, help="Number of opaque AS LSAs" )

   _vrf = Str( help='Private attribute for VRF name' )
   _instanceId = Int( help='Private attribute OSPF instance ID' )
   _routerId = Str( help='Private attribute for routerId' )

   def render( self ):
      toPrint = [ ( "Count", "LSA Type" ) ]
      values = [ ( self.numRouter, "Router" ),
                 ( self.numNetwork, "Network" ),
                 ( self.numSummary, "Summary Net" ),
                 ( self.numSummaryAsbr, "Summary ASBR" ),
                 ( self.numNssa, "Type-7 Ext" ),
                 ( self.numOpaqueArea, "Opaque Area" ),
                 ( self.numAsex, "Type-5 Ext" ),
                 ( self.numOpaqueAs, "Opaque AS" ),
                 ( self.numOpaqueLink, "Opaque Link" ), ]

      # add up all the values
      total = sum( num for num, _ in values )

      toPrint.extend( values )
      toPrint.append( ( total, "Total" ) )

      # This output mimics the output of cliribd except
      # always print the instance header
      if self._routerId:
         print 'OSPF Router with ID(%s) (Instance ID %d) (VRF %s)' % \
            ( self._routerId, self._instanceId, self._vrf )
      else:
         print 'OSPF Router with (Instance ID %d) (VRF %s)' % \
            ( self._instanceId, self._vrf )

      print " "
      for value, name in toPrint:
         name = name.ljust( 12, " " ) # line up the columns
         text = "  {name}  {value}".format( name=name, value=value )
         print text
      print " "

class OspfShowSummaryModel( Model ):
   # -------------------------------------------------------------------------------
   # Private, duplicate classes with iterables instead of generators. We cannot reuse
   # the original models due to limitations discussed in AID 3458.
   class _OspfNeighborsInstanceAcc( Model ):
      ospfNeighborEntries = List( valueType=OspfNeighborsEntry,
                                 help='List of OSPF neighbor entries' )

   class _OspfShowInstAreaModelAcc( Model ):
      areaId = Ip4Address( help='OSPF area ID' )
      stubArea = Bool( default=False, help='This area is a stub area' )
      normalArea = Bool( default=False, help='This area is a normal area' )
      lsaInformation = Submodel( valueType=OspfShowInstAreaLsaInfoModel,
                                 help='Area LSA information' )
      nssaInformation = Submodel( valueType=OspfShowInstAreaNssaModel, optional=True,
                                 help='Area NSSA information' )

      def getKey( self, data ):
         return data[ 'areaId' ]

      def processData( self, data ):
         if 'stubArea' in data:
            data[ 'stubArea' ] = ord( data[ 'stubArea' ] ) != 0
         if 'normalArea' in data:
            data[ 'normalArea' ] = ord( data[ 'normalArea' ] ) != 0

         lsaInfo = OspfShowInstAreaLsaInfoModel()
         lsaInfo.treeSize = data.pop( 'treeSize' )
         lsaInfo.asExternalCksum = data.pop( 'asExternalCksum' )
         data[ 'lsaInformation' ] = lsaInfo

         if 'nssaArea' in data and ord( data[ 'nssaArea' ] ):
            nssa = OspfShowInstAreaNssaModel()
            nssa.processModel( data )
            data[ 'nssaInformation' ] = nssa

         return data

   class _OspfShowInstModelAcc( Model ):
      instanceId = Int( help='OSPF instance ID' )
      maxLsaInformation = Submodel( valueType=OspfShowInstMaxLsaInfoModel,
                                    help='Max LSA information' )
      routerId = Ip4Address( optional=True, help='OSPF Router id' )
      asbr = Bool( default=False, help='This OSPF instance is an AS Border Router' )
      abr = Bool( default=False, help='This OSPF instance is an Area Border Router' )
      spfInformation = Submodel( valueType=OspfShowInstSpfInfoModel, optional=True,
                                 help='SPF information' )
      lsaInformation = Submodel( valueType=OspfShowInstLsaInfoModel, optional=True,
                                 help='LSA information' )
      vrf = Str( optional=True, help='VRF name' )
      areaList = Dict( optional=True, keyType=Ip4Address,
                     valueType=OspfShowInstAreaModel, 
                     help='List of areas in this OSPF instance' )
      shutDown = Bool( default=False,
                       help='Indicates if the OSPF instance is currently shut down' )

      def processData( self, data ):
         maxLsaInfo = OspfShowInstMaxLsaInfoModel()
         maxLsaInfo.processModel( data )
         data[ 'maxLsaInformation' ] = maxLsaInfo

         if 'routerId' in data:
            if 'asbr' in data:
               data[ 'asbr' ] = ord( data[ 'asbr' ] ) != 0
            if 'abr' in data:
               data[ 'abr' ] = ord( data[ 'abr' ] ) != 0

            spfInfo = OspfShowInstSpfInfoModel()
            spfInfo.processModel( data )
            data[ 'spfInformation' ] = spfInfo

            lsaInfo = OspfShowInstLsaInfoModel()
            lsaInfo.processModel( data )
            data[ 'lsaInformation' ] = lsaInfo

         return data
   # -------------------------------------------------------------------------------

   inst = Submodel( valueType=_OspfShowInstModelAcc,
                        help='OSPF instance information' )
   neighbors = Submodel( valueType=_OspfNeighborsInstanceAcc,
                         help='Neighbor information' )
   lsasByArea = Dict( valueType=OspfDatabaseSummary,
                         help='Database info by area' )

   def render( self ):
      # Without an OSPF router-id configured, it does not make sense to print an OSPF
      # summary, so simply return.  If OSPF is shutdown we'll proceed so we can
      # report this to the user.
      if not self.inst.routerId and not self.inst.shutDown:
         return

      def areaTypeString( area ):
         if area.normalArea:
            return "normal"
         if area.nssaInformation:
            return "nssa"
         if area.stubArea:
            return "stub"
         assert False
         return None

      def borderRouter( inst ):
         if inst.asbr and inst.abr:
            return 'ASBR ABR'
         if inst.asbr:
            return 'ASBR'
         if inst.abr:
            return 'ABR'
         return None

      def routerId( inst ):
         if inst.routerId is not None:
            return inst.routerId
         else:
            return 'unknown'

      inst = self.inst

      br = borderRouter( inst )
      instId = inst.instanceId

      if br:
         print "OSPF instance {} with ID {}, VRF {}, {}".format( instId,
                                                                 routerId( inst ),
                                                                 inst.vrf, br )
      else:
         print "OSPF instance {} with ID {}, VRF {}".format( instId,
                                                             routerId( inst ),
                                                             inst.vrf )

      if inst.shutDown:
         print '  Instance is disabled due to shutdown command'
         return

      if inst.spfInformation:
         print 'Time since last SPF: {} s'.format( inst.spfInformation.lastSpf )

      if inst.maxLsaInformation and inst.lsaInformation:
         print 'Max LSAs: {}, Total LSAs: {}'.format( inst.maxLsaInformation.maxLsa,
                                                      inst.lsaInformation.numLsa )

      if not inst.areaList:
         return

      # External Type 5 LSAs are area-independent so pick one area and print the
      # value. The choice of the first area is reasonable but arbitrary.
      numAsex = self.lsasByArea.itervalues().next().numAsex
      print 'Type-5 Ext LSAs: {}'.format( numAsex )

      areaFormat = '{:<16} {:<6} {:<6} {:<4} ({:<4}) {:<7} {:<7} {:<7} {:<7} {:<7}'
      print areaFormat.format( 'ID', 'Type', 'Intf', 'Nbrs', 'full', 'RTR LSA',
                               'NW LSA', 'SUM LSA', 'ASBR LSA', 'TYPE-7 LSA' )

      # Compute a dictionary of lists of neighbor states indexed by area id
      ngbStates = {}
      for ngb in self.neighbors.ospfNeighborEntries:
         areaId = str( ngb.details.areaId )

         if areaId not in ngbStates:
            ngbStates[ areaId ] = []
         ngbStates[ areaId ].append( ngb.adjacencyState )

      for areaId, area in inst.areaList.iteritems():
         areaId = str( areaId )

         lsaInfo = self.lsasByArea[ areaId ]

         areaType = areaTypeString( area )

         numNgbrs = 0
         numFull = 0

         if areaId in ngbStates:
            areaStates = ngbStates[ areaId ]

            numNgbrs = len( areaStates )
            numFull = len( [ state for state in areaStates if state == 'full' ] )

         rtrLsa = lsaInfo.numRouter
         nwLsa = lsaInfo.numNetwork
         sumLsa = lsaInfo.numSummary
         asbrLsa = lsaInfo.numSummaryAsbr
         type7Lsa = lsaInfo.numNssa

         print areaFormat.format( areaId, areaType, area.numIntf, numNgbrs, numFull,
                                  rtrLsa, nwLsa, sumLsa, asbrLsa, type7Lsa )

      print # end in a newline so multiple VRFs are spaced nicely
   
class OspfMplsLdpSyncEntry( Model ):
   instanceId = Int( help="OSPF Instance ID" )
   interface = Interface( help="OSPF interface" )
   ldpEnabled = Bool( help="Ldp enabled" )
   syncRequired = Bool( help="Sync required" )
   syncAchieved = Enum( values=( "Not applicable", "Yes", "No" ),
                        help="Sync achieved" )
   igpDelay = Int( optional=True, help="IGP link notify delay" )
   holddownTime = Int( optional=True, help="IGP holddown time" )

   def render( self ):
      print "Interface: %s; OSPF Instance: %s" % (
         str( self.interface ).strip("'"), self.instanceId )
      print " Ldp Enabled: %s" % renderBool( self.ldpEnabled )
      print " SYNC information:"
      print "   Required: %s, Achieved: %s" % (
         renderBool( self.syncRequired ), self.syncAchieved )
      if self.ldpEnabled:
         print "   IGP Notify Delay: %s" % (
            "immediate" if self.igpDelay == 0 else \
            renderSeconds( self.igpDelay ) )
         print "   Holddown time: %s" % (
            "infinite" if self.holddownTime == 0 else \
            renderSeconds( self.holddownTime ) )
         
class OspfMplsLdpSyncModel( Model ):
   interfaces = Dict( keyType=Interface, valueType=OspfMplsLdpSyncEntry,
                      help="OSPF MPLS LDP Sync status" )
 
   def render( self ):
      for intf in sorted( self.interfaces, key=lambda x: x[0] ):
         self.interfaces[ intf ].render()


#-------------------------------------------------------------------------------
# "show ip ospf [<instance-id>]
#               interface [ {<interface-type> <interface-number>} |
#                           brief ]
#               [ vrf vrf-name|default|all ]"
#-------------------------------------------------------------------------------

ospfIntfEntryFormatString = ( '   %-12.12s %-8.8s %-10.10s %-15.15s '
                              '%-18.18s %-5.5s %-10.10s %s' )

BFD_CONFIG_STATE_MAP = { 'process' : { 0 : 'default',
                                       1 : 'enabled',
                                       2 : 'disabled' },
                         'render' : { 'default' : '',
                                      'enabled' : ', BFD Enabled',
                                      'disabled' : ', BFD Disabled' } }
                         
MPLS_LDP_IGP_SYNC_MAP = { True : 'On',
                          False : 'Off' }

INTERFACE_STATE_MAP = ReversibleDict( { 'down' : 'Down',
                                        'loopback' : 'Loopback',
                                        'waiting' : 'Waiting',
                                        'p2p' : 'P2P',
                                        'dr' : 'DR',
                                        'backupDr' : 'Backup DR',
                                        'drOther' : 'DR Other' } )

INTERFACE_TYPE_MAP = ReversibleDict( { 'broadcast' : 'Broadcast',
                                       'p2p' : 'Point-To-Point', 
                                       'nbma' : 'Non-Broadcast Multi-Access',
                                       'p2mp' : 'Point-to-Multipoint',
                                       'virtual' : 'Virtual' } )

INTERFACE_AUTH_MAP = ReversibleDict( {
        'simple' : 'Simple authentication',
        'md5' : 'Message-digest authentication',
        'sha1' : 'Message-digest sha1 authentication',
        'sha256' : 'Message-digest sha256 authentication',
        'sha384' : 'Message-digest sha384 authentication',
        'sha512' : 'Message-digest sha512 authentication' } )


class OspfInterfaceEntry( Model ):
   state = Enum( values=tuple( INTERFACE_STATE_MAP.keys() ), 
                 help='Interface state' )
   priority = Int( optional=True, help='Interface priority' )
   cost = Int( help='Interface metric' )
   neighborCount = Int( help='Neighbour count' )
   designatedRouter = Ip4Address( optional=True, help='Designated router ID' )
   backupDesignatedRouter = Ip4Address( optional=True, 
                                        help='Backup designated router ID' )
   helloInterval = Int( help='Hello interval in seconds' )
   deadInterval = Int( help='Dead interval in seconds' ) 
   retransmitInterval = Int( help='Retransmit interval in seconds' )
   transmitDelay = Int( help='Transmit delay in seconds' )
   options = Submodel( help='Interface optional capabilities', 
                       valueType=OspfOptions )
   area = Ip4Address( help='Area assigned to the interface' )
   interfaceAddress = Ip4Address( help='Interface IP address' )
   interfaceMask = Int( help='Interface IP mask length' )
   interfaceType = Enum( values=tuple( INTERFACE_TYPE_MAP.keys() ), 
                         help='Network interface type' )
   _instanceId = Int( help='Private attribute for instance ID' )
   _vrf = Str( help='Private attribute for VRF name' )
   authentication = Enum( optional=True, values=tuple( INTERFACE_AUTH_MAP.keys() ), 
                          help='Authentication algorithm' )
   authKeyId = Int( optional=True, help='Authentication Key ID' )
   mplsLdpIgpSync = Bool( optional=True,
                          help='MPLS LDP-IGP Synchronization, ensures that LDP '
                               'is fully established before IGP path '
                               'is used for switching. Uses global setting '
                               'as default.' )
   bfdState = Enum( values=( 'default', 'enabled', 'disabled' ), 
                    help='BFD configuration state' )
   passive = Bool( default=False, help='Passive state' )
   teEnabled = Bool( default=False, help='Traffic engineering state')

   def getKey( self, data ):
      assert data[ 'ifname' ]
      ifName = kernelIntfFilter( data.pop( 'ifname' ) )
      return ifName

   def processData( self, data):
      self.options = OspfOptions()
      self.options.processData( data )

      if data[ 'state' ]:
         revIntfStateMap = INTERFACE_STATE_MAP.reverse()
         data[ 'state' ] = revIntfStateMap[ data[ 'state' ] ]

      if data[ 'interfaceType' ]:
         revIntfTypeMap = INTERFACE_TYPE_MAP.reverse()
         data[ 'interfaceType' ] = revIntfTypeMap[ data[ 'interfaceType' ] ]

      if data[ 'authentication' ]:
         if data[ 'authentication' ] == 'No authentication':
            data[ 'authentication' ] = None
         else:
            revIntfAuthMap = INTERFACE_AUTH_MAP.reverse()
            data[ 'authentication' ] = revIntfAuthMap[ data [ 'authentication' ] ]
      
      if 'bfdState' in data:
         data[ 'bfdState' ] = BFD_CONFIG_STATE_MAP[ 'process' ][ data[ 'bfdState' ] ]
      
      if 'passive' in data:
         self.passive = ord( data.pop( 'passive' ) ) != 0

      if 'mplsLdpIgpSync' in data:
         self.mplsLdpIgpSync = data.pop( 'mplsLdpIgpSync' ) != 0
         
      self.teEnabled = ord( data.pop( 'teEnabled' ) ) != 0

      return data

   def renderEntry( self, ifname, brief ):
      if brief:
         fullIpAddr = '{}/{}'.format( self.interfaceAddress, self.interfaceMask )
         state = INTERFACE_STATE_MAP[ self.state ]
         print ospfIntfEntryFormatString % ( IntfMode.getShortname( ifname ),
                                             self._instanceId, self._vrf, 
                                             self.area, fullIpAddr, self.cost,
                                             state, self.neighborCount )
      else:
         # mimic cliribd output
         print '{} is up'.format( ifname )
         print '  Interface Address {ip}/{mask}, instance {instance}, ' \
               'VRF {vrf}, Area {area}' \
               .format( ip=self.interfaceAddress, mask=self.interfaceMask,
                        instance=self._instanceId, vrf=self._vrf, area=self.area )
         ifType = INTERFACE_TYPE_MAP[ self.interfaceType ]
         print '  Network Type {}, Cost: {}'.format( ifType, self.cost )
         if self.priority:
            pri = ', Priority {}'.format( self.priority )
         else:
            pri = ''
         bfd = BFD_CONFIG_STATE_MAP[ 'render' ][ self.bfdState ]

         state = INTERFACE_STATE_MAP[ self.state ]
         print '  Transmit Delay is {} sec, State {}{}{}' \
               .format( self.transmitDelay, state, pri, bfd )
              
         if self.designatedRouter:
            print '  Designated Router is {}'.format( self.designatedRouter )
         else:
            print '  No Designated Router on this network'
         
         if self.backupDesignatedRouter:
            print '  Backup Designated Router is {}' \
                  .format( self.backupDesignatedRouter )
         else:
            print '  No Backup Designated Router on this network'

         print '  Timer intervals configured, Hello {}, Dead {}, Retransmit {}' \
               .format( self.helloInterval, self.deadInterval, 
                        self.retransmitInterval )
         passive = ' (Passive Interface)' if self.passive else ''
         print '  Neighbor Count is {}{}'.format( self.neighborCount, passive )
         if self.authentication:
            auth = INTERFACE_AUTH_MAP[ self.authentication ]
            if self.authKeyId:
               print '  {}, using key id {}' \
                  .format( auth, self.authKeyId )
            else:
               print '  {}'.format( auth )
         else:
            print '  No authentication'

         if self.mplsLdpIgpSync != None:
            igp_sync = MPLS_LDP_IGP_SYNC_MAP[ self.mplsLdpIgpSync ]
            print '  MPLS LDP Sync: {}'.format( igp_sync )

         if self.teEnabled:
            print '  Traffic engineering is enabled'
         else:
            print '  Traffic engineering is disabled'
         
class OspfInterface( Model ):
   interfaces = GeneratorDict( keyType=Interface, valueType=OspfInterfaceEntry,
                               help='Dictionary of OSPF Interfaces indexed by '
                                    'the interface name' )
   _brief = Bool( default=False,
                  help='Private attribute to indicate that it is a brief instance'
                       ' of the show ip ospf interface command' )
   _firstInst = Bool( default=False,
                      help='Private attribute to indicate that this model is part'
                           ' of the first instane of this VRF to be returned' )
   _firstVrf = Bool( default=False,
                     help='Private attribute to indicate that this model'
                          ' is part of the first VRF to be returned' )

   def render( self ):
      if self._brief:
         if self._firstVrf and self._firstInst:
            print ospfIntfEntryFormatString % ( 'Interface', 'Instance', 'VRF',
                                                'Area', 'IP Address', 'Cost',
                                                'State', 'Nbrs' )
      for ifname, intf in self.interfaces:
         intf.renderEntry(ifname, self._brief)
         
#------------------------------------------------------------------------------
# show ip ospf [<instance-id>] counters [vrf vrf-name|default|all]
#------------------------------------------------------------------------------
LSA_TYPE_COUNTERS_MAP = \
      { 'newLsaReceived'                : 'New LSAs received',
        'selfOriginatedLsaNew'          : 'New LSAs self-originated',
        'selfOriginatedLsaUpdated'      : 'Updated self-originated LSAs',
        'selfOriginatedLsaRefreshed'    : 'Refreshed self-originated LSAs',
        'lsaInLsdb'                     : 'LSAs in link state database' }
lsaTypeFormatStr = '%-30s %6s %6s %6s %6s %6s %6s %6s %6s %6s'

class OspfCountersLsaEntry( Model ):
   numRouter = Int( default=0, help="Number of router LSAs" )
   numNetwork = Int( default=0, help="Number of network LSAs" )
   numSummary = Int( default=0, help="Number of summary LSAs" )
   numSummaryAsbr = Int( default=0, help="Number of summary ASBR LSAs" )
   numAsex = Int( default=0, help="Number of AS-external LSAs" )
   numNssa = Int( default=0, help="Number of type-7 NSSA LSAs" )
   numOpaqueLink = Int( default=0, help="Number of opaque link LSAs" )
   numOpaqueArea = Int( default=0, help="Number of opaque area LSAs" )
   numOpaqueAs = Int( default=0, help="Number of opaque AS LSAs" )

   def getKey( self, data ):
      return str( data[ 'counterName' ] )

   def renderEntry( self, counterName ):
      print lsaTypeFormatStr % ( LSA_TYPE_COUNTERS_MAP[ counterName ],
                                 self.numRouter, self.numNetwork, self.numSummary,
                                 self.numSummaryAsbr, self.numAsex, self.numNssa,
                                 self.numOpaqueLink, self.numOpaqueArea,
                                 self.numOpaqueAs )

class OspfCountersInstance( Model ):
   _firstVrf = Bool( default=False,
                     help='Private attribute to indicate that this model'
                          ' is part of the first VRF to be returned' )
   _firstInst = Bool( default=False,
                      help='Private attribute to indicate that this model is part'
                           ' of the first instance of this VRF to be returned' )
   _instanceId = Int( help='Private attribute for OSPF instance ID' )
   _vrf = Str( help='Private attribute for VRF name' )
   lastResetTime = Float( optional=True,
                          help='UTC timestamp of last counters reset' )
   opaqueLsaReceived = Int( help='Number of opaque LSAs received' )
   selfOriginatedLsaReceived = Int( help='Number of self-originated LSAs received' )
   selfOriginatedLsaReceivedUpdated = \
         Int( help='Number of self-originated LSAs received and updated' )
   selfOriginatedLsaReceivedDeleted = \
         Int( help='Number of self-originated LSAs received and deleted' )
   spfCount = Int( help='Number of SPF runs' )
   lsaTypeCounters = GeneratorDict( keyType=str, valueType=OspfCountersLsaEntry,
                                    help='Dictionary of LSA type counters keyed'
                                         ' by counter name' )

   def processData( self, data ):
      if 'lastResetTime' in data:
         self.lastResetTime = Tac.utcNow() -  data.pop( 'lastResetTime' )
      return data

   def render( self ):
      if not self._instanceId:
         return
      if self._firstVrf and self._firstInst:
         print "Instance counters"
         print "-----------------"
      print "\nVRF: %s" % self._vrf
      print "OSPF instance: %d" % self._instanceId
      if self.lastResetTime != None:
         timeSecs = int( Tac.utcNow() - self.lastResetTime )
         timeDHMS = formatTimeInterval( timeSecs )
         print "The counters were last reset %s ago\n" % timeDHMS
      else:
         print "The counters have never been reset\n"
      formatStr = '%-19s%23s%5s'
      print formatStr % ( "Counter Description", "", "Value")
      print formatStr % ( "-------------------", "", "-----")
      formatStr = '%-29s%13s%5d'
      print formatStr % ( "Opaque LSAs received", "", self.opaqueLsaReceived )
      print formatStr % ( "Self-originated LSAs received", "",
                          self.selfOriginatedLsaReceived )
      print formatStr % ( "   Updated", "", self.selfOriginatedLsaReceivedUpdated )
      print formatStr % ( "   Deleted", "", self.selfOriginatedLsaReceivedDeleted )
      print formatStr % ( "SPF runs", "", self.spfCount )
      print lsaTypeFormatStr % ( '', 'RTR', 'NTW', 'ASIP', 'ASBR', 'ASEX', 'NSSA',
                                 'OPQ9', 'OPQ10', 'OPQ11' )
      for counterName, counter in self.lsaTypeCounters:
         counter.renderEntry( counterName )
         
#------------------------------------------------------------------------------
# show ip ospf [<instance-id>] counters interface
#                             [ifName|summary] [vrf vrf-name|default|all]
#------------------------------------------------------------------------------
class OspfCountersInterfaceEntry( Model ):
   lastResetTime = Float( optional=True,
                          help="UTC timestamp of the last time counters were reset" )
   multicastLsUpdatesSent = Int( default=0,
                                 help="Number of multicast LS updates sent" )
   delayedLsAcksSent = Int( default=0,
                            help="Number of delayed LS acknowledgements sent" )
   lsAcksReceived = Int( default=0, help="Number of LS acknowledgements received" )
   lsUpdatesInFloodQueue = Int( default=0,
                                help="Number of LS updates currently"
                                     " in the flood queue" )

   def getKey( self, data ):
      assert data[ 'ifname' ]
      ifName = kernelIntfFilter( data.pop( 'ifname' ) )
      return ifName

   def processData( self, data ):
      if 'lastResetTime' in data:
         self.lastResetTime = Tac.utcNow() -  data.pop( 'lastResetTime' )
      return data

   def renderEntry( self, ifname ):
      print "\nStatistics for interface %s:" % ifname
      if self.lastResetTime != None:
         timeSecs = int( Tac.utcNow() - self.lastResetTime )
         timeDHMS = formatTimeInterval( timeSecs )
         print "The counters were last reset %s ago\n" % timeDHMS
      else:
         print "The counters have never been reset\n"
      formatStr = '%-19s%23s%5s'
      print formatStr % ( "Counter Description", "", "Value")
      print formatStr % ( "-------------------", "", "-----")
      formatStr = '%-35s%7s%5d'
      print formatStr % ( "Multicast LS updates sent", "",
                          self.multicastLsUpdatesSent )
      print formatStr % ( "Delayed LS acknowledgements sent", "",
                          self.delayedLsAcksSent )
      print formatStr % ( "LS acknowledgements received", "",
                          self.lsAcksReceived )
      print formatStr % ( "LS updates currently in flood queue", "",
                          self.lsUpdatesInFloodQueue )

class OspfCountersInterface( Model ):
   interfaces = GeneratorDict( keyType=Interface,
                               valueType=OspfCountersInterfaceEntry,
                               help="Dictionary of OSPF interfaces indexed by "
                                    "the interface name" )
   _summary = Bool( default=False,
                    help="Private attribute to indicate that a consolidated"
                         " summary of all interfaces is to be displayed" )
   _firstInst = Bool( default=False,
                      help="Private attribute to indicate that this model is part"
                           " of the first instance of this VRF to be returned" )
   _firstVrf = Bool( default=False,
                     help="Private attribute to indicate that this model"
                          " is part of the first VRF to be returned" )
   _instanceId = Int( help="Instance ID" )
   _vrf = Str( help="VRF name" )

   def render( self ):
      if not self._instanceId:
         return
      if self._firstVrf and self._firstInst:
         print "Interface counters"
         print "------------------"
      if self._summary:
         print "\nVRF: %s" % self._vrf
         print "OSPF instance: %d" % self._instanceId
         sumMulticastLsUpdatesSent = 0
         sumDelayedLsAcksSent = 0
         sumLsAcksReceived = 0
         sumLsUpdatesInFloodQueue = 0
         for ifname, interface in self.interfaces:
            sumMulticastLsUpdatesSent += interface.multicastLsUpdatesSent
            sumDelayedLsAcksSent += interface.delayedLsAcksSent
            sumLsAcksReceived += interface.lsAcksReceived
            sumLsUpdatesInFloodQueue += interface.lsUpdatesInFloodQueue
         print "\nStatistics summary for all interfaces:\n"
         formatStr = '%-19s%23s%5s'
         print formatStr % ( "Counter Description", "", "Value")
         print formatStr % ( "-------------------", "", "-----")
         formatStr = '%-35s%7s%5d'
         print formatStr % ( "Multicast LS updates sent", "",
                             sumMulticastLsUpdatesSent )
         print formatStr % ( "Delayed LS acknowledgements sent", "",
                             sumDelayedLsAcksSent )
         print formatStr % ( "LS acknowledgements received", "",
                             sumLsAcksReceived )
         print formatStr % ( "LS updates currently in flood queue", "",
                             sumLsUpdatesInFloodQueue )
      else:
         # Sometimes OSPF instances exist without being configured on any interface.
         # The following makes sure that the header is printed only if
         # at least one intf exists in the instance.
         firstIntf = True
         for ifname, interface in self.interfaces:
            if firstIntf:
               print "\nVRF: %s" % self._vrf
               print "OSPF instance: %d" % self._instanceId
               firstIntf = False
            interface.renderEntry( ifname )

#------------------------------------------------------------------------------
# show ip ospf [<instance-id>] spf-log [vrf vrf-name|default|all]
#------------------------------------------------------------------------------
class OspfSpfLogEntry( Model ):
   spfStartTime = Float( help='SPF Start time in UTC' )
   spfDuration = Float( help='SPF run duration in milliseconds' )

   def processData( self, data ):
      self.spfStartTime = float( data.pop( 'startTime' ) )
      self.spfDuration = data.pop( 'durationMsec' ) + \
                    ( float( data.pop( 'durationUsec' ) ) \
                      / float( pow ( 10, 3 ) ) )

   def render( self ):
      print "%s\t\t%.3f" % ( timestampToLocalTime ( self.spfStartTime ),
                             self.spfDuration )

class OspfSpfLog( Model ):
   routerId = Ip4Address( optional=True, help='OSPF router identifier' )
   _vrf = Str( help='Private attribute for VRF name' )
   _instanceId = Int( help='Private attribute for instance ID' )
   spflogEntries = GeneratorList( valueType=OspfSpfLogEntry,
                                  help='OSPF SPF Log entries' )
   def render( self ):
      if not self.routerId:
         return
      print "OSPF instance %d with ID %s, VRF %s" % ( self._instanceId,
                                                      self.routerId, self._vrf )
      print "When\t\tDuration(msec)"
      for logEntry in self.spflogEntries:
         logEntry.render()

#------------------------------------------------------------------------------
# "show ip ospf retransmission queue [vrf vrf-name|default|all]"
# Legacy:
# "show ip ospf retransmission-list [vrf vrf-name|default|all]"
#------------------------------------------------------------------------------
class OspfLsaEntry( Model ):
   lsaType = Enum( values=tuple( LSA_TYPE_MAP.values() ),
                   help='LSA type' )
   linkStateId = Ip4Address( help='LSA link state identifier' )
   advertisingRouter = Ip4Address( help='LSA advertising router' )
   age = Int( help='LSA age in seconds' )
   sequenceNumber = Int( help='LSA sequence number' )
   checksum = Int( help='LSA checksum' )

   def processData( self, data ):
      lsaType = int( data.pop( 'lsaType' ) )
      self.lsaType = LSA_TYPE_MAP.get( lsaType )
      assert self.lsaType is not None, 'Invalid value %s for lsaType' % lsaType
      return data

   def render( self ):
      lsaType = LSA_TYPE_MAP.reverse()[ self.lsaType ]
      print '%-4d %-15s %-15s %-11d 0x%-12x 0x%-8x' % ( lsaType, self.linkStateId,
                                                        self.advertisingRouter,
                                                        self.age,
                                                        self.sequenceNumber,
                                                        self.checksum )

class OspfLsaListBase( Model ):
   _vrf = Str( help='VRF name' )
   _instanceId = Int( help='Private attribute for instance ID' )
   interfaceName = Interface( help='OSPF neighbor interface name' )
   interfaceAddress = Ip4Address( help='OSPF neighbor interface address' )

   def getKey( self, data ):
      return data[ 'routerId' ]

   def processData( self, data ):
      self.interfaceName = kernelIntfFilter( data.pop( 'interfaceName' ) )
      return data

   def renderModel( self, routerId ):
      print 'Neighbor %s instance %d vrf %s interface %s address %s' % \
            ( routerId, self._instanceId, self._vrf,
              self.interfaceName.stringValue, self.interfaceAddress )

class OspfLsaRetransmissionInstance( OspfLsaListBase ):
   nextRetransmissionTime = Float( optional=True,
                                   help='Next retransmission time in UTC' )
   retransmitQueueLength = Int( help='LSA retransmission queue length' )
   retransmissions = GeneratorList( valueType=OspfLsaEntry,
                                    help='LSA retransmission list' )

   def processData( self, data ):
      data = OspfLsaListBase.processData( self, data )
      if 'nextRetransmissionTime' in data:
         nextRetransmissionSec = data.pop( 'nextRetransmissionTime' ) / 1000.
         self.nextRetransmissionTime = Tac.utcNow() + nextRetransmissionSec
      return data

   def renderModel( self, routerId ):
      OspfLsaListBase.renderModel( self, routerId )
      if not self.nextRetransmissionTime:
         print 'LSA retransmission not currently scheduled. Queue length is %d' % \
               self.retransmitQueueLength
      else:
         nextRetransmissionMsec = ( self.nextRetransmissionTime - Tac.utcNow() ) * \
                                  1000
         print 'LSA retransmission due in %d msec. Queue length is %d' % \
               ( nextRetransmissionMsec, self.retransmitQueueLength )

      printOspfLsaHeader()
      for retransmissionEntry in self.retransmissions:
         retransmissionEntry.render()

class OspfLsaRetransmissionList( Model ):
   neighbors = GeneratorDict( keyType=Ip4Address,
                              valueType=OspfLsaRetransmissionInstance,
                              help='Dictionary of retransmission list keyed '
                                   'by the neighbor\'s router ID')

   def render( self ):
      for routerId, neighborModel in self.neighbors:
         neighborModel.renderModel( routerId )

#------------------------------------------------------------------------------
# "show ip ospf request queue [vrf vrf-name|default|all]"
# Legacy:
# "show ip ospf request-list [vrf vrf-name|default|all]"
#------------------------------------------------------------------------------
class OspfLsaRequestInstance( OspfLsaListBase ):
   requestQueueLength = Int( help='LSA request queue length' )
   requests = GeneratorList( valueType=OspfLsaEntry,
                             help='LSA request list' )

   def renderModel( self, routerId ):
      OspfLsaListBase.renderModel( self, routerId )
      printOspfLsaHeader()
      for requestEntry in self.requests:
         requestEntry.render()

class OspfLsaRequestList( Model ):
   neighbors = GeneratorDict( keyType=Ip4Address,
                              valueType=OspfLsaRequestInstance,
                              help='Dictionary of request list keyed by the '
                                   'neighbor\'s router ID' )

   def render( self ):
      for routerId, neighborModel in self.neighbors:
         neighborModel.renderModel( routerId )

#------------------------------------------------------------------------------
# "show ip ospf database"
#------------------------------------------------------------------------------

printLsaSummaryHeader = "%-15s %-15s %-11s %-12s %s"
printLsaSummaryFormat = "%-15s %-15s %-11s %-12s %-8s"
lsaSummaryFormatString = "%-15s %-15s %-11s %-12s %-8s %s"

areaLsaHeader = "                 %s Link States (Area %s)"
asExternalHeader = "                 Type-5 AS External Link States"
type11OpaqueHeader = "                 Type 11 Opaque LSDB"

hex8Format = '0x{:06x}'
hex4Format = '0x{:02x}'

OSPF_OPTIONS_LIST = [ ( 'T', 'multitopologyCapability' ),
                      ( 'E', 'externalRoutingCapability' ),
                      ( 'MC', 'multicastCapability' ),
                      ( 'NP', 'nssaCapability' ),
                      ( 'L', 'linkLocalSignaling' ),
                      ( 'DC', 'demandCircuitsSupport' ),
                      ( 'O',  'opaqueLsaSupport' ),
                      ( 'DN', 'doNotUseInRouteCalc' )
                    ]

OSPF_DATABASE_LSA_TYPE_MAP = { 
   'router'           : 1,
   'network'          : 2,
   'summary'          : 3,
   'asbr-summary'     : 4,
   'external'         : 5,
   'nssa-external'    : 7,
   'opaque-link'      : 9,
   'opaque-area'      : 10,
   'opaque-as'        : 11
}

OSPF_DATABASE_QUERY_TYPE_MAP = { 
      'external'  : 2,
      'opaque-as' : 3,
      'area'      : 4,
}

LSA_TYPE_TEXT_MAP = { 1: "Router",
                      2: "Network",
                      3: "Summary",
                      4: "ASBR Summary",
                      5: "AS External",
                      7: "NSSA",
                      9: "Opaque Type 9",
                     10: "Opaque Type 10",
                     11: "Opaque Type 11",
                   }

EXCEPTION_TYPE_MAP = { 
   1 : 'announcingMaximumCost',
   2 : 'announcingMaximumMetric'
}

EXCEPTION_TYPE_TEXT_MAP = {
   'announcingMaximumCost'   : 'Announcing maximum link costs',
   'announcingMaximumMetric' : 'Announcing maximum metrics'
}

ROUTER_LSA_LINK_TYPE_MAP = ReversibleDict( { 'Point-to-point' : 'pointToPoint',
                                             'Transit' : 'transitNetwork',
                                             'Stub' : 'stubNetwork',
                                             'Virtual' : 'virtualLink',
                                             'Unknown' : 'unknown' } )

ROUTER_LSA_OPTIONS = { 'AreaBorder' : 'areaBorderRouter',
                       'ASBorder' : 'asBoundaryRouter',
                       'NSSATranslator' : 'nssaTranslation',
                       'VLink' : 'virtualLinkEndpoint' }

TE_LINK_TYPE_MAP = ReversibleDict( { 'Point-to-point' : 'pointToPoint',
                                     'Multi-access' : 'multiAccess',
                                     'Unknown' : 'unknown' } )

metricTypeHelp = '''External metric type (E bit)  
   type1 : External metric Type 1
   type2 : External metric Type 2
'''

def printOspfDatabaseLsaHeader( lsaType ):
   if lsaType == 1:
      print lsaSummaryFormatString % ( "Link ID", "ADV Router", "Age", "Seq#",
                                       "Checksum", "Link count" )
   elif lsaType == 5:
      print lsaSummaryFormatString % ( "Link ID", "ADV Router", "Age", "Seq#",
                                       "Checksum", "Tag" )
   else:
      print printLsaSummaryHeader % ( 'Link ID', 'ADV Router', 'Age',
                                      'Seq#', 'Checksum' )

class OspfDatabaseOptions( OspfOptions ):
   def render( self ):
      keys = []
      for key, option in OSPF_OPTIONS_LIST:
         if getattr( self, option ):
            keys.append( key )
      return ' '.join( keys ) or '(null)'

class OspfRouterLsaLinkEntry( Model ):
   linkType = Enum( ROUTER_LSA_LINK_TYPE_MAP.values(),
                    help="Description of the router link" )
   linkId = Ip4Address( help="Router link ID" )
   linkData = Ip4Address( help="Router link data" )
   numTos = Int( help="Number of TOS metrics for the link" )
   metric = Int( help="Cost of the router link" )
   
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_ROUTER_LINK_ENTRY
   
   def processData( self, data ):
      self.linkType = ROUTER_LSA_LINK_TYPE_MAP[ data.pop( 'linkType' ) ]
      return data
   
   def renderEntry( self ):
      print
      print "    Link connected to: a %s Network" % \
            ROUTER_LSA_LINK_TYPE_MAP.reverse()[ self.linkType ]
      if self.linkType == 'stubNetwork':
         print "     (Link ID) Network/subnet number: %s" % self.linkId
         print "     (Link Data) Network Mask: %s" % self.linkData
      elif self.linkType == 'virtualLink':
         print "     (Link ID) Neighboring Router ID: %s" % self.linkId
         print "     (Link Data) Router Interface address: %s" % self.linkData
      elif self.linkType == 'transitNetwork':
         print "     (Link ID) Designated Router address: %s" % self.linkId
         print "     (Link Data) Router Interface address: %s" % self.linkData
      elif self.linkType == 'pointToPoint':
         print "     (Link ID) Neighboring Router ID: %s" % self.linkId
         print "     (Link Data)  %s" % self.linkData
      else:
         print "     (Link ID)  %s" % self.linkId
         print "     (Link Data)  %s" % self.linkData
      print "      Number of TOS metrics: %s" % self.numTos
      print "       TOS 0 Metrics: %s" % self.metric
      print

class OspfRouterLsaOptions( Model ):
   areaBorderRouter = Bool( default=False,
                            help='This is an area border router' )
   asBoundaryRouter = Bool( default=False,
                            help='This is an autonomous system boundary '
                                 'router' )
   nssaTranslation = Bool( default=False,
                           help='Translate nssa lsas to as external lsas' )
   virtualLinkEndpoint = Bool( default=False,
                               help='This is an endpoint for virtual links in '
                                    'the area' )
   
   def processData( self, data ):
      optionsList = data.pop( 'options', '' ).split()
      for option in optionsList:
         setattr( self, ROUTER_LSA_OPTIONS[ option ], True )

class OspfRouterLsa( Model ):
   numRtrLinks = Int( default=0, help="Number of router links" )
   rtrLsaOptions = Submodel( valueType=OspfRouterLsaOptions,
                       help='Router lsa options' )
   routerLsaLinks = List( valueType=OspfRouterLsaLinkEntry,
                          help='Links for this router' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_ROUTER_ENTRY
   
   def processData( self, data ):
      self.rtrLsaOptions = OspfRouterLsaOptions()
      if 'options' in data:
         self.rtrLsaOptions.processData( data )
      return data

   def renderEntry( self ):
      print "  Number of Links: %s" % self.numRtrLinks
      for routerLsaLink in self.routerLsaLinks:
         routerLsaLink.renderEntry()

class OspfNetworkLsa( Model ):
   networkMask = Ip4Address( help='IP address mask for the network' )
   attachedRouters = List( valueType=Ip4Address,
                           help='List of routers attached to the network' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_NETWORK_ENTRY

   def overrideHierarchy( self, data ):
      readNext = True
      if not self.networkMask:
         self.networkMask = data.pop( 'networkMask' )
         self.attachedRouters = []
         return ( data, readNext )
      elif 'attachedRouterId' in data:
         self.attachedRouters.append( data.pop( 'attachedRouterId' ) )
         return ( data, readNext )
      return ( data, False )

   def renderEntry( self ):
      print "  Network Mask: %s" % self.networkMask
      
      for attachedRouterId in self.attachedRouters:
         print "        Attached Router: %s" % ( attachedRouterId )

class OspfExternalLsa( Model ):
   networkMask = Ip4Address( help='IP address mask for the advertised '
                                  'destination' )
   metricType = Enum( values=( 'type1', 'type2' ), help=metricTypeHelp )
   metric = Int( help='Route cost' )
   forwardingAddress = Ip4Address( help='Forwarding address' )
   externalRouteTag = Int( help='External route tag' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_EXTERNAL_ENTRY

   def processData( self, data ):
      metricType = data.pop( 'metricType' )
      self.metricType = 'type1' if metricType == 1 else 'type2'
      return data

   def renderEntry( self ):
      metricType = '1' if self.metricType == 'type1' else '2'
      
      print "  Network Mask: %s" % self.networkMask
      print "        Metric Type: %s" % metricType
      print "        Metric: %u" % self.metric
      print "        Forwarding Address: %s" % self.forwardingAddress
      print "        External Route Tag: %s" % self.externalRouteTag

class OspfSummaryLsa( Model ):
   networkMask = Ip4Address( help='IP address mask for the destination' )
   metric = Int( help='Route cost' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_SUMMARY_ENTRY

   def renderEntry( self ):
      print "    Mask: %s" % self.networkMask
      print "    Metric: %s" % self.metric

# bw is in bps, convert it to mpbs for renderding
def bw_best_value_units( bw ):
   bw = ( bw * 1.0 ) / ( 10 **6 )
   value = bw / 1000 if ( bw >= 1000 ) else bw
   unit = "Gbps" if bw >= 1000 else "Mbps"
   return value, unit

def Int32ToIp( intId ):
   return socket.inet_ntoa( struct.pack( '<I', int( intId ) ) )

def updateList( inputList, mapToIp=True ):
   if inputList is None:
      return None
   ips = [ ctypes.c_uint32( i ).value for i in inputList ]
   if mapToIp:
      ips = [ Int32ToIp( i ) for i in ips ]
   return ips

class TeReservablePriorityBandwidth( Model ):
   priority0 = Int( help="Reservable bandwidth (bps) for priority level 0" )
   priority1 = Int( help="Reservable bandwidth (bps) for priority level 1" )
   priority2 = Int( help="Reservable bandwidth (bps) for priority level 2" )
   priority3 = Int( help="Reservable bandwidth (bps) for priority level 3" )
   priority4 = Int( help="Reservable bandwidth (bps) for priority level 4" )
   priority5 = Int( help="Reservable bandwidth (bps) for priority level 5" )
   priority6 = Int( help="Reservable bandwidth (bps) for priority level 6" )
   priority7 = Int( help="Reservable bandwidth (bps) for priority level 7" )
      
   def renderEntry( self ):
      bwInfo = "    TE class {0}: {1:0.2f} {2}"
      bw0 = bw_best_value_units( self.priority0 )
      bw1 = bw_best_value_units( self.priority1 )
      bw2 = bw_best_value_units( self.priority2 )
      bw3 = bw_best_value_units( self.priority3 )
      bw4 = bw_best_value_units( self.priority4 )
      bw5 = bw_best_value_units( self.priority5 )
      bw6 = bw_best_value_units( self.priority6 )
      bw7 = bw_best_value_units( self.priority7 )
      
      print "   Unreserved BW:"
      print "{}\t{}\t{}".format( bwInfo.format( 0, bw0[ 0 ], bw0[ 1 ] ),
                                 bwInfo.format( 1, bw1[ 0 ], bw1[ 1 ] ),
                                 bwInfo.format( 2, bw2[ 0 ], bw2[ 1 ] ) )
      print "{}\t{}\t{}".format( bwInfo.format( 3, bw3[ 0 ], bw3[ 1 ] ),
                                 bwInfo.format( 4, bw4[ 0 ], bw4[ 1 ] ),
                                 bwInfo.format( 5, bw5[ 0 ], bw5[ 1 ] ) )
      print "{}\t{}".format( bwInfo.format( 6, bw6[ 0 ], bw6[ 1 ] ),
                             bwInfo.format( 7, bw7[ 0 ], bw7[ 1 ] ) )

class RtrAddressTlv( Model ):
   teRouterId = Ip4Address( help="Traffic Engineering IPv4 Router-Id" )
   
   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_TE_RTR_ADDR_TLV_ENTRY
   
   def renderEntry( self ):
      print "   TE router ID: %s" % self.teRouterId

class LinkAddressTlv( Model ):
   linkType = Enum( TE_LINK_TYPE_MAP.values(),
                    help="Description of the link" )
   linkId = Ip4Address( help="Ip Address of the other end of the link" )
   teMetric = Int( help="The cost of the link", optional=True )
   adminGroup = Int( help="Administrative group assigned to the link",
                     optional=True )
   maxLinkBw = Int( help="Maximum bandwidth (bps) that can be used on the link",
                    optional=True )
   maxReservableBw = Int( help="Maximum bandwidth (bps) that can be reserved "
                               "on the Link", optional=True )
   localIpAddresses = List( valueType=Ip4Address, optional=True,
                            help='IP addresses of the link' )
   remoteIpAddresses = List( valueType=Ip4Address, optional=True,
                             help='IP addresses of the neighbors of the link' )
   srlgs = List( valueType=long, optional=True,
                 help='Shared risk link group identifiers' )
   maxReservablePriorityBw = Submodel( valueType=TeReservablePriorityBandwidth,
                                       help="Maximum bandwidth reservable "
                                            "for a priority",
                                       optional=True )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_TE_LINK_TLV_ENTRY

   def processData( self, data ):
      self.linkType = TE_LINK_TYPE_MAP[ data.pop( 'linkType' ) ]
         
      maxLinkBw = data.pop( 'maxLinkBw', None )
      if maxLinkBw is not None:
         self.maxLinkBw = int( maxLinkBw * ( 10**6 ) )
         
      maxReservableBw = data.pop( 'maxReservableBw', None )
      if maxReservableBw is not None:
         self.maxReservableBw = int( maxReservableBw * ( 10**6 ) )

      self.localIpAddresses = updateList( data.pop( 'localIpAddrs', None ) )
         
      self.remoteIpAddresses = updateList( data.pop( 'remoteIpAddrs', None ) )

      self.srlgs = updateList( data.pop( 'srlg', None ), mapToIp=False )

      if 'maxUnreservedBw0' in data: 
         maxReservablePriorityBw = TeReservablePriorityBandwidth()
         for i in range( 0, 8 ):
            maxUnresvBw = data.pop( 'maxUnreservedBw{}'.format( i ), None )
            if maxUnresvBw is not None:
               setattr( maxReservablePriorityBw, 'priority{}'.format( i ),
                        int( maxUnresvBw * ( 10**6 ) ) )
            else:
               setattr( maxReservablePriorityBw, 'priority{}'.format( i ),
                        None )
         
         if maxReservablePriorityBw.priority0 is not None:
            self.maxReservablePriorityBw = maxReservablePriorityBw

      return data
   
   def renderEntry( self ):
      print "   Link Type: %s" % \
                  TE_LINK_TYPE_MAP.reverse()[ self.linkType ]
      
      print "   Link Id: %s" % self.linkId
      
      if self.localIpAddresses:
         print "   Interface Address: %s" % self.localIpAddresses[ 0 ]
         for ipAddr in self.localIpAddresses[ 1: ]:
            print "   Interface Address: %s" % ipAddr
      
      if self.remoteIpAddresses:
         print "   Neighbor Address: %s" % self.remoteIpAddresses[ 0 ]
         for ipAddr in self.remoteIpAddresses[ 1: ]:
            print "   Neighbor Address: %s" % ipAddr

      if self.teMetric is not None:
         print "   TE default metric: %s" % self.teMetric
      
      if self.maxLinkBw is not None:
         bestValueUnit = bw_best_value_units( self.maxLinkBw )
         print "   Maximum link BW: %0.2f %s" % ( bestValueUnit[ 0 ],
                                                  bestValueUnit[ 1 ] )
      if self.maxReservableBw is not None:
         bestValueUnit = bw_best_value_units( self.maxReservableBw )
         print "   Maximum reservable link BW: %0.2f %s" % \
                                                   ( bestValueUnit[ 0 ],
                                                     bestValueUnit[ 1 ] )
      
      if self.adminGroup is not None:
         print "   Administrative group (Color): 0x%x" % self.adminGroup
      
      if self.srlgs:
         print "   Shared Risk Link Group ID: %s" % self.srlgs[ 0 ]
         for srlg in self.srlgs[ 1: ]:
            print "   Shared Risk Link Group ID: %s" % srlg

      if self.maxReservablePriorityBw:
         self.maxReservablePriorityBw.renderEntry()

class UnsupportedSubTlv( Model ):
   tlvType = Int( help="TLV type" )
   tlvLength = Int( help="TLV length" )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_TE_UNSUPP_SUB_TLV_ENTRY
   
   def renderEntry( self ):
      if self.tlvType is not None and self.tlvLength is not None:
         fmt = "   Unsupported sub-TLV: Type: {} Length: {}"
         print fmt.format( self.tlvType, self.tlvLength )

class OspfTrafficEngineeringInfo( Model ):
   tlvName = Enum( values=( 'routerAddress', 'link', 'unsupported' ),
                   help='Top Level TLV Name', optional=True )
   tlvType = Int( help='Top Level TLV type', optional=True )
   tlvLength = Int( help='Top Level TLV length', optional=True )
   malformedTlv = Bool( help='The TLV is not parsable', default=False )
   rtrTlvInfo = Submodel( valueType=RtrAddressTlv, help="Router Address TLV",
                          optional=True )
   linkTlvInfo = Submodel( valueType=LinkAddressTlv, help="Link TLV",
                           optional=True )
   unsupportedSubTlvInfo = List( valueType=UnsupportedSubTlv, 
                                 help="Unsupported sub-TLV information",
                                 optional=True )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_TE_TLV_ENTRY
   
   def processData( self, data ):
      if 'malformedTlv' in data:
         self.malformedTlv = data.pop( 'malformedTlv' ) != 0
         data = {}

      if 'tlvType' in data:
         if data[ 'tlvType' ] == 1:
            self.tlvName = 'routerAddress'
         elif data[ 'tlvType' ] == 2:
            self.tlvName = 'link'
         else:
            self.tlvName = 'unsupported'
   
      return data

   def renderEntry( self ):
      if self.malformedTlv:
         print "   This is a malformed Traffic Engineering TLV"
         print
         return

      if self.tlvType not in ( 1, 2 ):
         fmt = "   Unsupported TLV: Type: {} Length: {}\n"
         print fmt.format( self.tlvType, self.tlvLength )
         return

      if self.rtrTlvInfo:
         self.rtrTlvInfo.renderEntry()
      elif self.linkTlvInfo:
         self.linkTlvInfo.renderEntry()
      if self.unsupportedSubTlvInfo:
         for unsupportedSubTlv in self.unsupportedSubTlvInfo:
            unsupportedSubTlv.renderEntry()

      print 

class OspfLsa( Model ):
   lsaType = Enum( values=LSA_TYPE_MAP.values(), help='LSA type' )
   age = Int( help='LSA age in seconds' )
   linkStateId = Ip4Address( help='LSA link state id' )
   advertisingRouter = Ip4Address( help='LSA advertising router' )
   sequenceNumber = Int( help='LSA sequence number' )
   checksum = Int( help='LSA checksum' )
   length = Int( help='LSA length' )
   options = Submodel( valueType=OspfDatabaseOptions,
                       help='OSPF LSA options' )
   doNotAge = Bool( help='LSA will not be aged' )
   exceptionFlag = Enum( values=EXCEPTION_TYPE_MAP.values(), optional=True,
                         help='Indicates maximum cost being advertised for self'
                              ' originated LSAs' )
   _numLinks = Int( help="Number of router links" )  
   _tag = Int( default=0, optional=True,
               help="External Route Tag" )

   _areaId = Ip4Address( help='LSA area ID' )
   _firstLsa = Bool( help='This is the first write of the LSA type' )
   # Specific LSA submodels
   ospfRouterLsa = Submodel( valueType=OspfRouterLsa, optional=True,
                             help='Router LSA' )
   ospfNetworkLsa = Submodel( valueType=OspfNetworkLsa, optional=True,
                              help='Network LSA' )
   ospfSummaryLsa = Submodel( valueType=OspfSummaryLsa, optional=True,
                              help='Summary LSA or ASBR summary LSA' )
   ospfExternalLsa = Submodel( valueType=OspfExternalLsa, optional=True,
                               help='AS or NSSA external LSA' )
   ospfTeInfo = Submodel( valueType=OspfTrafficEngineeringInfo, optional=True,
                          help='Traffic Engineering TLV data')

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_VERTEX_ENTRY

   def processData( self, data ):
      self.options = OspfDatabaseOptions()
      self.options.processData( data )
      
      exceptionFlag = data.pop( 'exceptionFlag', None )
      if exceptionFlag:
         self.exceptionFlag = EXCEPTION_TYPE_MAP[ exceptionFlag ]
      
      self._numLinks = data.pop( 'numLinks', 0 )

      self._tag = data.pop( 'tag', 0 )
      
      lsaType = int( data.pop( 'lsaType' ) )
      self.lsaType = LSA_TYPE_MAP.get( lsaType )

      self._firstLsa = 'firstLsa' in data
      data.pop( 'firstLsa', None )

      self._areaId = data.pop( 'areaId', None )

      self.doNotAge = bool( data.pop( 'flags', 0 ) & 0x400 )

      return data

   def renderEntry( self, detail ):
      lsaType = LSA_TYPE_MAP.reverse()[ self.lsaType ]
      age = self.age
      if self.doNotAge:
         age = 'DNA'

      if self._firstLsa:
         if lsaType == 5:
            print 
            print asExternalHeader
            print
         elif lsaType == 11:
            print 
            print type11OpaqueHeader
            print
         else:
            print
            print areaLsaHeader % ( LSA_TYPE_TEXT_MAP[ lsaType ],
                                    self._areaId )
            print
      
         if not detail:
            printOspfDatabaseLsaHeader( lsaType )

      if not detail:
         if lsaType == 1:
            print lsaSummaryFormatString % ( self.linkStateId,
                                             self.advertisingRouter, age,
                                       hex8Format.format( self.sequenceNumber ),
                                       hex4Format.format( self.checksum ),
                                       self._numLinks )

         elif lsaType == 5:
            print lsaSummaryFormatString % ( self.linkStateId,
                                             self.advertisingRouter, age,
                                       hex8Format.format( self.sequenceNumber ),
                                       hex4Format.format( self.checksum ),
                                       self._tag )
         else:
            print printLsaSummaryFormat % ( self.linkStateId,
                                            self.advertisingRouter, age,
                                       hex8Format.format( self.sequenceNumber ),
                                       hex4Format.format( self.checksum ) )
      else:
         if self.exceptionFlag is not None:
            print "  Exception Flag: %s" % \
                          EXCEPTION_TYPE_TEXT_MAP[ self.exceptionFlag ]
         print "  LS Age: %s" % age
         print "  Options: (%s)" % self.options.render()
         print "  LS Type: %s Links" % \
                     LSA_TYPE_TEXT_MAP[ lsaType ]
         print "  Link State ID: %s" % self.linkStateId
         print "  Advertising Router: %s" % self.advertisingRouter
         print "  LS Seq Number: %s" % hex8Format.format( self.sequenceNumber )
         print "  Checksum: %s" % hex4Format.format( self.checksum )
         print "  Length: %d" % self.length
         
         if self.ospfRouterLsa:
            self.ospfRouterLsa.renderEntry()
         elif self.ospfNetworkLsa:
            self.ospfNetworkLsa.renderEntry()
         elif self.ospfSummaryLsa:
            self.ospfSummaryLsa.renderEntry()
         elif self.ospfExternalLsa:
            self.ospfExternalLsa.renderEntry()
         elif self.ospfTeInfo:
            self.ospfTeInfo.renderEntry()

class OspfType9LsaList( Model ):
   ospfType9Lsas = GeneratorList( valueType=OspfLsa,
                                  help='List of LSAs for the interface' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_TYPE9

   def renderEntry( self, detail ):
      for lsa in self.ospfType9Lsas:
         lsa.renderEntry( detail )

class OspfInterfaceScopeModel( Model ):
   linkLsas = GeneratorList( valueType=OspfType9LsaList,
                            help='List of LSAs for the area' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_INTERFACE_ENTRY

   def getKey( self, data ):
      ifName = kernelIntfFilter( data.pop( 'interfaceName' ) )
      return ifName

   def renderEntry( self, address, detail ):
      for lsa in self.linkLsas:
         lsa.renderEntry( detail )

class OspfAreaScopeLsa( Model ):
   areaLsas = GeneratorList( valueType=OspfLsa,
                             help='List of LSAs for the area' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_AREA_LSDB
   
   def renderEntry( self, detail ):
      for lsa in self.areaLsas:
         lsa.renderEntry( detail )

class OspfAreaModel( Model ):
   interfaces = GeneratorDict( optional=True,
                               keyType=Interface,
                               valueType=OspfInterfaceScopeModel,
                               help='A mapping between an Interface'
                                    ' and its LSA list' )
   areaDatabase = GeneratorList( valueType=OspfAreaScopeLsa,
                                 help='Area scoped LSAs' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_AREA_ENTRY

   def getKey( self, data ):
      return data[ 'area' ]

   def renderEntry( self, area, detail ):
      for ipAddr, linkLsas in self.interfaces:
         linkLsas.renderEntry( ipAddr, detail )

      for lsa in self.areaDatabase:
         lsa.renderEntry( detail )

class OspfAsExternalLsa( Model ):
   externalLsas = GeneratorList( valueType=OspfLsa,
                                 help='List of AS external LSAs' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_AS_EXTERNAL
   
   def renderEntry( self, detail ):
      for lsa in self.externalLsas:
         lsa.renderEntry( detail )

class OspfOpaqueExternalLsa( Model ):
   type11Lsas = GeneratorList( valueType=OspfLsa,
                               help='Type 11 Opaque LSAs' )

   def getMioAttrId( self ):
      return PyRibAmiClient.MIO_DGET_OSPF_DATABASE_TYPE11
   
   def renderEntry( self, detail ):
      for lsa in self.type11Lsas:
         lsa.renderEntry( detail )

class OspfDatabaseInstance( Model ):   
   _vrf = Str( help='Private attribute for VRF name' )
   _instanceId = Int( help='Private attribute OSPF instance ID' )
   _routerId = Str( help='Private attribute for routerId' )
   _detailsPresent = Bool( default=False, help='Details submodel is present' )

   areas = GeneratorDict( keyType=Ip4Address, valueType=OspfAreaModel,
                                    help='A mapping between an area '
                                         'and its LSA list' )
   externalLsdb = GeneratorList( valueType=OspfAsExternalLsa,
                                 help='List of LSAs with AS scope LSAs' )
   
   type11Lsdb = GeneratorList( valueType=OspfOpaqueExternalLsa,
                               help='List of Opaque LSAs with AS scope' )
 
   def processData( self, data ):
      self._instanceId = data.pop( 'instance' )
      self._routerId = data.pop( 'router_id' )
      self._vrf = data.pop( 'vrf' )
      
   def render( self ):
      print
      print "            OSPF Router with ID(%s) (Instance ID %s) (VRF %s)" % \
                                ( self._routerId, self._instanceId, self._vrf )
      print

      for areaId, areaLsas in self.areas:
         areaLsas.renderEntry( areaId, self._detailsPresent )
      
      for externalLsa in self.externalLsdb:
         externalLsa.renderEntry( self._detailsPresent )

      for type11Lsa in self.type11Lsdb:
         type11Lsa.renderEntry( self._detailsPresent )
#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ospfConfigDir, ospfStatusDir
   ospfConfigDir = ConfigMount.mount( entityManager, OspfConsts.configPath,
                                      OspfConsts.configType, 'w' )
   ospfStatusDir = LazyMount.mount( entityManager, OspfConsts.statusPath,
                                    OspfConsts.statusType, 'ri' )
