#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re
import urlparse

import BasicCli
import CliCommand
import CliMatcher
import CliMode.OpenStack as OSMode
import CliParser
from CliPlugin import ( AaaCli, AclCli, AclCliModel, ControllerCli,
                        ControllerdbLib, IraIpCli, IraIp6Cli,
                        NetworkTopologyModels, VlanCli )
import CliToken.Clear
import CliToken.Ip
import CliToken.Ipv6
import ConfigMount
import IpLibConsts
import LazyMount
import Logging
import MultiRangeRule
import Plugins
from ReversibleSecretCli import reversibleSecretCliExpression
import ShowCommand
import Tac
import Tracing
import Vlan

from OpenStackLib import coalesceList
import OpenStackLogMsgs
from CliPlugin import OpenStackModels

# pkgdeps: rpm OpenStack-lib

__defaultTraceHandle__ = Tracing.Handle( 'OpenStackCli' )
debug = Tracing.trace8

ONEDAY = 86400
FOURHOURS = 14400
openStackIdRe = '[a-zA-Z0-9-]+'

# Module globals set up by the Plugin function below
controllerConfig = None
clusterStatus = None
openStackConfig = None
openStackStatus = None
openStackCliConfig = None
virtualNetworkConfig = None
virtualNetworkStatus = None
openStackAgentStatus = None
osApiStatus = None
nameResolutionConfig = None
nameResolutionStatus = None
portVlanConfigDir = None
globalTopoStatus = None
serviceConfigDir = None
httpServiceConfig = None
cdbMountsComplete = False
aclCheckpoint = None

SwitchInterface = Tac.Type( 'VirtualNetwork::Client::SwitchInterface' )
VNAClients = Tac.Type( 'VirtualNetwork::Client::Clients' )
ClientDomain = Tac.Type( 'VirtualNetwork::Client::ClientDomain' )

# Tokens
# Instance types
INSTANCE_TYPE_VIRTUAL = 'virtual'
INSTANCE_TYPE_BAREMETAL = 'baremetal'
INSTANCE_TYPE_ROUTER = 'router'

#
# Configuration methods.
#

def enterTenantInstanceConfigMode( mode, args ):
   instanceId = args[ 'INSTANCE_ID' ]
   instanceHostId = args.get( 'INSTANCE_HOST_ID' )
   instanceType = args.get( 'INSTANCE_TYPE', 'virtual' )

   def _isInstanceConfiguredAs( asType ):
      ''' Helper function to check whether the instance has been configured as a
          given type. If so, return True. '''
      if asType == INSTANCE_TYPE_VIRTUAL:
         if instanceId in mode.region.vmInstance:
            mode.addError( 'Instance id already configured as a vm' )
            return True
      if asType == INSTANCE_TYPE_BAREMETAL:
         if instanceId in mode.region.baremetalInstance:
            mode.addError( 'Instance id already configured as a baremetal' )
            return True
      if asType == INSTANCE_TYPE_ROUTER:
         if instanceId in mode.region.routerInstance:
            mode.addError( 'Instance id already configured as a router' )
            return True

      # Instance has not been configured as a conflicting type
      return False

   if instanceType not in [ INSTANCE_TYPE_VIRTUAL,
                            INSTANCE_TYPE_BAREMETAL,
                            INSTANCE_TYPE_ROUTER ]:
      mode.addError( 'Unrecognized instance type %s' % instanceType )
      return

   if instanceType == INSTANCE_TYPE_VIRTUAL:
      if( _isInstanceConfiguredAs( INSTANCE_TYPE_BAREMETAL ) or
            _isInstanceConfiguredAs( INSTANCE_TYPE_ROUTER ) ):
         return
   if instanceType == INSTANCE_TYPE_BAREMETAL:
      if( _isInstanceConfiguredAs( INSTANCE_TYPE_VIRTUAL ) or
            _isInstanceConfiguredAs( INSTANCE_TYPE_ROUTER ) ):
         return
   if instanceType == INSTANCE_TYPE_ROUTER:
      if( _isInstanceConfiguredAs( INSTANCE_TYPE_VIRTUAL ) or
            _isInstanceConfiguredAs( INSTANCE_TYPE_BAREMETAL ) ):
         return

   childMode = mode.childMode( TenantInstanceConfigMode,
                               instanceId=instanceId,
                               instanceHostId=instanceHostId,
                               instanceType=instanceType )
   mode.session_.gotoChildMode( childMode )

def openStackSetEnabled( mode, args ):
   if not controllerConfig.enabled:
      mode.addError( "OpenStack service cannot be enabled until the CVX server is "
                     "enabled" )
      return

   openStackCliConfig.enabled = True
   nameResolutionConfig.enabled = True
   openStackServiceConfig = serviceConfigDir.service[ "OpenStack" ]
   openStackServiceConfig.enabled = True
   httpServiceConfig.service[ 'JsonApi' ].enabled = True

def openStackShutdown( mode, args ):
   openStackCliConfig.enabled = False
   nameResolutionConfig.enabled = False
   openStackServiceConfig = serviceConfigDir.service[ "OpenStack" ]
   openStackServiceConfig.enabled = False
   httpServiceConfig.service[ 'JsonApi' ].enabled = False

def openStackClrEnabled( mode, args ):
   openStackShutdown( mode, args )
   nameResolutionConfig.refreshPeriod = nameResolutionConfig.defaultRefreshPeriod

def openStackDefaultEnabled( mode, args ):
   if openStackCliConfig.defaultEnabled:
      openStackSetEnabled( mode, args )
   else:
      openStackClrEnabled( mode, args )

def normalizeSwitchId( switchId ):
   switchId = switchId.replace( "-", "" )
   switchId = switchId.replace( ".", "" )
   switchId = switchId.replace( ":", "" )
   for i in xrange( 5, 0, -1 ):
      switchId = switchId[ 0 : i * 2 ] + ":" + switchId[ i * 2 : ].lower()
   return switchId

def normalizeInterfaceName( interfaceName ):
   if not interfaceName:
      return ( '', False )
   splitIndex = re.search( "\d", interfaceName ).start()
   interfaceType = interfaceName[ 0 : splitIndex ]
   interfaceId = interfaceName[ splitIndex : ]
   interfaceType = interfaceType.lower()
   valid = True
   if interfaceType.startswith( 'et' ):
      interfaceType = "Ethernet"
   elif interfaceType.startswith( 'ma' ):
      interfaceType = "Management"
   else:
      valid = False
   return ( interfaceType + interfaceId, valid )

def osConfigGuard( mode, token ):
   if ( openStackAgentStatus.enabled and
        clusterStatus.status[ 'default' ].isStandaloneOrLeader ):
      return None
   else:
      return "only available on cluster leader with openstack service enabled"

def cdbMountsGuard( mode, token ):
   if cdbMountsComplete:
      return None
   return "CVX not ready"

#-------------------------------------------------------------------------------
# Common Matchers
#-------------------------------------------------------------------------------
regionNameMatcher = CliMatcher.QuotedStringMatcher(
   helpname='name',
   helpdesc='Region name',
   pattern=CliParser.excludePipeTokensPattern
)

vniRangeMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 1, 16777215 ),
   noSingletons=False,
   helpdesc='VNI ID or range(s) of VNI IDs',
   helpname='vniRange'
)

matcherOpenstack = CliMatcher.KeywordMatcher(
   'openstack',
   helpdesc='Show OpenStack service'
)

matcherConfig = CliMatcher.KeywordMatcher(
   'config',
   helpdesc='Dump region configuration'
)

matcherRegion = CliMatcher.KeywordMatcher(
   'region',
   helpdesc='Region configuration'
)

matcherDetail = CliMatcher.KeywordMatcher(
   'detail',
   helpdesc='Display information in detail'
)

nodeUuid = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      'uuid',
      helpdesc='Show openstack agent uuid'
    ),
   guard=osConfigGuard
)

def singleMatch( cls ):
   for k, v in cls.data.items():
      if isinstance( v, str ):
         cls.data[ k ] = CliCommand.singleKeyword( k, v )
      elif isinstance( v, CliMatcher.Matcher ):
         cls.data[ k ] = CliCommand.singleNode( v )
      elif isinstance( v, CliCommand.Node ):
         v.maxMatches_ = 1
   return cls

@singleMatch
class _ShowNetworkFilterExpr( CliCommand.CliExpression ):
   expression = 'network NETWORK_NAME_OR_ID'
   data = {
      'network' : 'Filter results by specified tenant network name or ID',
      'NETWORK_NAME_OR_ID' : CliMatcher.QuotedStringMatcher(
            helpdesc='Tenant network name or id'
      )
   }

@singleMatch
class _ShowInstanceFilterExpr( CliCommand.CliExpression ):
   expression = 'instance INSTANCE_NAME_OR_ID'
   data = {
      'instance' : 'Filter results by specified tenant network name or ID',
      'INSTANCE_NAME_OR_ID' : CliMatcher.QuotedStringMatcher(
         helpdesc='Tenant instance name or id'
      )
   }

@singleMatch
class _ShowVmFilterExpr( CliCommand.CliExpression ):
   expression = 'vm INSTANCE_NAME_OR_ID'
   data = {
      'vm' : 'Filter results by specified VM instance name or ID',
      'INSTANCE_NAME_OR_ID' : CliMatcher.QuotedStringMatcher(
         helpdesc='Tenant instance name or id'
      )
   }

@singleMatch
class _ShowRegionFilterExpr( CliCommand.CliExpression ):
   expression = 'region REGION_NAME'
   data = {
      'region' : 'Filter results by specified region name',
      'REGION_NAME' : regionNameMatcher,
   }

@singleMatch
class _ShowTenantFilterExpr( CliCommand.CliExpression ):
   expression = 'tenant TENANT_NAME_OR_ID'
   data = {
      'tenant' : 'Filter results by specified tenant name or ID',
      'TENANT_NAME_OR_ID' : CliMatcher.QuotedStringMatcher(
         helpdesc='Tenant name or id'
      )
   }

@singleMatch
class _ShowHostFilterExpr( CliCommand.CliExpression ):
   expression = 'host HOSTNAME'
   data = {
      'host' : 'Filter results by specified host name',
      'HOSTNAME' : CliMatcher.QuotedStringMatcher(
         helpdesc='Name of the host'
      )
   }

@singleMatch
class _ShowSwitchFilterExpr( CliCommand.CliExpression ):
   expression = 'switch SWITCH'
   data = {
    'switch' : 'Filter results by switch',
    'SWITCH' : CliMatcher.PatternMatcher(
       pattern='.*',
       helpdesc='Switch hostname or system MAC address',
       helpname='switch' ),
   }

@singleMatch
class _ShowInterfaceFilterExpr( CliCommand.CliExpression ):
   expression = 'interface INTF'
   data = {
      'interface' : 'Filter results by interface',
      'INTF' : CliMatcher.PatternMatcher(
         pattern=r'([A-Za-z]+[0-9]+(\/[0-9]+)*)',
         helpdesc='Name of the interface',
         helpname='interface' ),
   }

@singleMatch
class _ShowVlanFilterExpr( CliCommand.CliExpression ):
   expression = 'vlan VLAN_ID'
   data = {
      'vlan' : 'Filter results by VLAN',
      'VLAN_ID' : CliMatcher.IntegerMatcher(
         1, 4094,
         helpdesc='Identifier for a Virtual LAN' ),
   }

@singleMatch
class _ShowInstanceTypeFilterExpr( CliCommand.CliExpression ):
   expression = 'type INSTANCE_TYPE'
   data = {
      'type' : 'Filter results by specified instance type',
      'INSTANCE_TYPE' : CliMatcher.EnumMatcher(
         {
            'baremetal' : 'The instance is running on baremetal',
            'virtual' : 'The instance is running on a hypervisor',
            'router' : 'The instance is a logical router',
         }
      )
   }

@singleMatch
class _ShowPortFilterExpr( CliCommand.CliExpression ):
   expression = 'port PORT_NAME_OR_ID'
   data = {
      'port' : 'Filter results by specified port name or ID',
      'PORT_NAME_OR_ID' : CliMatcher.QuotedStringMatcher(
         helpdesc='Port name or id'
      )
   }

#--------------------------------------------------------------------------------
# [ no | default ] service openstack
#--------------------------------------------------------------------------------
class ServiceOpenstackCmd( CliCommand.CliCommandClass ):
   syntax = 'service openstack'
   noOrDefaultSyntax = syntax
   data = {
      'service' : ControllerCli.serviceKwMatcher,
      'openstack' : 'Configure OpenStack service'
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( OpenStackConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if not cdbMountsComplete:
         # We may have never mounted the config from controllerdb
         return
      # Remove all the region data
      for region in openStackConfig.region:
         RegionCmd.noOrDefaultHandler( mode, { 'EXISTING_REGION' : region } )
      openStackDefaultEnabled( mode, args )

ControllerCli.CvxConfigMode.addCommandClass( ServiceOpenstackCmd )

#-------------------------------------------------------------------------------
# If CVX is unconfigured, we need to unconfigure as well. Register hooks for
# 'no cvx' to unconfigure OpenStack when CVX is unconfigured.
#-------------------------------------------------------------------------------
def openStackCvxShutHandler( mode, no=False ):
   global portVlanConfigDir
   global globalTopoStatus
   global openStackConfig
   global openStackStatus
   global cdbMountsComplete

   if not no:
      portVlanConfigDir = None
      globalTopoStatus = None
      openStackConfig = None
      openStackStatus = None
      cdbMountsComplete = False

def openStackNoCvxHandler( mode ):
   openStackCvxShutHandler( mode )

   # Using defaults from OpenStack:Config
   typeDriver = Tac.Type( 'OpenStack::TypeDriver' )
   agentModeType = Tac.Type( 'OpenStack::AgentModeType' )
   openStackCliConfig.enabled = openStackCliConfig.defaultEnabled
   openStackCliConfig.vlanTypeDriver = typeDriver.defaultVlan
   openStackCliConfig.agentMode = agentModeType.provision

   virtualNetworkConfig.vlanToVniMap.clear()
   openStackCliConfig.credentials.clear()
   openStackCliConfig.vlanAssignment.clear()
   openStackCliConfig.keystoneAuthUrl.clear()
   openStackCliConfig.mandatoryRegion.clear()
   openStackDefaultEnabled( mode, None )

ControllerCli.addCvxShutdownCallback( openStackCvxShutHandler )
ControllerCli.addNoCvxCallback( openStackNoCvxHandler )

#-------------------------------------------------------------------------------
# Service OpenStack mode
#-------------------------------------------------------------------------------

class OpenStackConfigMode ( OSMode.OpenStackConfigModeBase,
                            BasicCli.ConfigModeBase ):

   name = 'OpenStack configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name=None ):
      OSMode.OpenStackConfigModeBase.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# The '[no | default] shutdown' command,
# in 'service openstack' mode.
#-------------------------------------------------------------------------------

class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Disable OpenStack service',
   }

   handler = openStackShutdown
   noHandler = openStackSetEnabled
   defaultHandler = openStackDefaultEnabled

OpenStackConfigMode.addCommandClass( ShutdownCmd )

#-------------------------------------------------------------------------------
# The 'network type-driver vlan <arista | default> command
#-------------------------------------------------------------------------------

class NetworkTypeDriverVlanCmd( CliCommand.CliCommandClass ):
   syntax = 'network type-driver vlan DRIVER_TYPE'
   data = {
      'network' : 'Configure region network properties',
      'type-driver' : 'Configure the ml2 type driver to be used',
      'vlan' : 'VLAN segmentation type',
      'DRIVER_TYPE' : CliMatcher.EnumMatcher(
         { 'arista' : 'Use the arista VLAN type driver',
           'default' : 'Use the default VLAN type driver' } )
   }

   @staticmethod
   def handler( mode, args ):
      vlanTypeDriver = args[ 'DRIVER_TYPE' ]
      driverType = Tac.Type( "OpenStack::TypeDriver" )
      cliToTypeDriver = { 'arista' : driverType.aristaVlan,
                          'default' : driverType.defaultVlan }
      # Check if this is a no-op
      if cliToTypeDriver[ vlanTypeDriver ] == openStackCliConfig.vlanTypeDriver:
         return
      if vlanTypeDriver == 'default':
         openStackCliConfig.vlanTypeDriver = driverType.defaultVlan
      elif vlanTypeDriver == 'arista':
         if cdbMountsComplete:
            for regionName in openStackConfig.region:
               if not safeToModifyVlanAssignment( mode, regionName ):
                  mode.addError(
                     "Arista VLAN type driver cannot be enabled until all "
                     "regions are in sync"
                  )
                  return
               if regionName in openStackStatus.regionStatus.keys():
                  assignedVlans = []
                  if regionName in openStackCliConfig.vlanAssignment.keys():
                     assignedVlans = \
                        openStackCliConfig.vlanAssignment[ regionName ].vlan
                  allocatedVlans = \
                     openStackStatus.regionStatus[ regionName ].\
                     vlanPool.allocatedVlan
                  if not all( vlan in assignedVlans for vlan in allocatedVlans ):
                     mode.addError(
                        "Arista VLAN type driver cannot be enabled because "
                        "assignment doesn't include all VLANs that are "
                        "already in use for region %s" % regionName )
                     return
         else:
            # Warn that this command is not a good idea to be run while
            # OpenStack is not running.
            mode.addWarning(
               "Changing the type driver while CVX is disabled may "
               "result in data path disruption if the range does not "
               "include all VLANs that are currently in use." )
         openStackCliConfig.vlanTypeDriver = driverType.aristaVlan
      else:
         mode.addError( "Unknown type driver" )

OpenStackConfigMode.addCommandClass( NetworkTypeDriverVlanCmd )

#-------------------------------------------------------------------------------
# [ no | default ] auth role ROLE
#-------------------------------------------------------------------------------
class AuthRoleCmd( CliCommand.CliCommandClass ):
   syntax = 'authentication role ROLE'
   noOrDefaultSyntax = 'authentication role ...'
   data = {
         "authentication" : "Authentication parameters",
         "role" : "API authentication user role",
         "ROLE" : CliMatcher.PatternMatcher(
             pattern=r'.+',
             helpname='WORD',
             helpdesc='User role name for API access' )
   }

   @staticmethod
   def handler( mode, args ):
      openStackCliConfig.authRole = args.get( 'ROLE', '' )

   noOrDefaultHandler = handler

OpenStackConfigMode.addCommandClass( AuthRoleCmd )

#-------------------------------------------------------------------------------
# VLAN to VNI mapping
#-------------------------------------------------------------------------------

def doMapVlanToVni( config, regionName, vlanRange, vniRange ):

   assert len( vlanRange ) == len( vniRange )

   # Possible values for vlanRange and vniRange
   # 1. vlan and vni ranges are not mapped
   # 2. vlan range is a subset of existing vlan range
   # 3. vlan range partially overlaps a exising vlan ranges ( starts and ends in
   #    different vlan ranges )
   # 4. vlan range is a super set of existing vlan ranges

   # Ensure that the start and end are not contained in another range
   vlanRange1 = None
   vlanRange2 = None
   vlanMapToRemove = {}
   clientDomain = ClientDomain( VNAClients.OpenStack, regionName )
   vlanToVniMap = config.newVlanToVniMap( clientDomain ).map
   for existingVlanRange in vlanToVniMap:
      existingVniRange = vlanToVniMap[ existingVlanRange ]
      if existingVlanRange.start != vlanRange[ 0 ] and existingVlanRange.contains(
            vlanRange[ 0 ] ):
         # vlanRange overlaps existingVlanRange.
         # Split the existingVlanRange to accommodate the new vlanRange.
         # Add a new vlan range [ existingVlanRange.start -
         # vlanRange.start ] mark the full existingVlanRange for deletion.
         vlanRange1 = Tac.Value( 'VirtualNetwork::Range',
                                 existingVlanRange.start,
                                 vlanRange[ 0 ] - 1 )
         offset = vlanRange[ 0 ] - existingVlanRange.start - 1
         vniRange1 = Tac.Value( 'VirtualNetwork::Range',
                                existingVniRange.start,
                                existingVniRange.start + offset )
         vlanMapToRemove[ existingVlanRange ] = existingVniRange

      if existingVlanRange.end != vlanRange[ -1 ] and existingVlanRange.contains(
            vlanRange[ -1 ] ):
         # vlanRange overlaps existingVlanRange. Split the existingVlanRange to
         # accommodate # the new existingVlanRange. Add a new vlan range
         # [ vlanRange.start - existingVlanRange.end ] mark the full
         # existingVlanRange for deletion.
         vlanRange2 = Tac.Value( 'VirtualNetwork::Range',
                                 vlanRange[ -1 ] + 1,
                                 existingVlanRange.end )
         offset = vlanRange[ -1 ] - existingVlanRange.start + 1
         vniRange2 = Tac.Value( 'VirtualNetwork::Range',
                                existingVniRange.start + offset,
                                existingVniRange.end )
         vlanMapToRemove[ existingVlanRange ] = existingVniRange

      if existingVlanRange.start in vlanRange and existingVlanRange.end in vlanRange:
         # New vlan range contains older vlan range. Mark that for deletion
         vlanMapToRemove[ existingVlanRange ] = existingVniRange

   if vlanRange1:
      vlanToVniMap[ vlanRange1 ] = vniRange1

   if vlanRange2:
      vlanToVniMap[ vlanRange2 ] = vniRange2

   vlanRangeToAdd = Tac.Value( 'VirtualNetwork::Range',
                               vlanRange[ 0 ],
                               vlanRange[ -1 ] )
   vniRangeToAdd = Tac.Value( 'VirtualNetwork::Range',
                              vniRange[ 0 ],
                              vniRange[ -1 ] )
   vlanToVniMap[ vlanRangeToAdd ] = vniRangeToAdd

   # Remove the vlan to vni mappings which were marked for deletion
   for existingVlanRange, existingVniRange in vlanMapToRemove.items():
      if existingVlanRange != vlanRangeToAdd and \
            existingVlanRange in vlanToVniMap and \
            vlanToVniMap[ existingVlanRange ] == existingVniRange:
         del vlanToVniMap[ existingVlanRange ]

def removeDuplicates( numberList ):
   '''This will return a list with all duplicate numbers removed from the list'''
   tempList = []
   added = set()
   for num in numberList:
      if not num in added:
         added.add( num )
         tempList.append( num )
   return tempList

def parseRangeInput( range1, range2 ):
   ''' This method looks at the two input list and returns a list of ranges.
   For example, if the input is '1, 2, 3, 10, 11' and '101, 102, 103, 104, 105',
   the method returns [ [1, 2, 3], [10, 11] ] and [ [101, 102, 103], [104, 105] ].
   This method basically splits the input list into a list of contiguous ranges. '''

   assert len( range1 ) == len( range2 )

   prevRange1Val = range1[ 0 ]
   prevRange2Val = range2[ 0 ]
   index = 1 # index into the range
   startIndex = 0 # holds the start index of the sub range
   range1List = []
   range2List = []

   while index < len( range1 ):
      # Go through all the numbers in the range
      if ( range1[ index ] != prevRange1Val + 1 or
           range2[ index ] != prevRange2Val + 1 ):
         # If the number in either range is not one more than the previous value,
         # break the list down.
         range1List.append( range1[ startIndex : index ] )
         range2List.append( range2[ startIndex : index ] )
         startIndex = index
      prevRange1Val = range1[ index ]
      prevRange2Val = range2[ index ]
      index += 1

   if startIndex < len( range1 ):
      # Append the last range
      range1List.append( range1[ startIndex : index ] )
      range2List.append( range2[ startIndex : index ] )

   return range1List, range2List

def uniqueVniRange( mode, vniRangeList ):
   ''' This method checks to see if any vni in the vni range is configured in a
       different region.
   '''
   unique = True
   regionList = sorted( virtualNetworkConfig.vlanToVniMap )
   # for region in openStackCliConfig.vlanToVniMap:
   for region in regionList:
      overlappingVnis = []
      for vniRange in vniRangeList:
         for vni in vniRange:
            vniMaps = virtualNetworkConfig.vlanToVniMap[ region ].map.values()
            for regionVniRange in vniMaps:
               if vni >= regionVniRange.start and vni <= regionVniRange.end:
                  overlappingVnis.append( vni )
                  unique = False
      if overlappingVnis:
         overlappingVnis = removeDuplicates( overlappingVnis )
         coalescedOverlappingVnis = coalesceList( overlappingVnis )
         suffix = ''
         if len( overlappingVnis ) > 1:
            suffix = "'s"
         mode.addWarning( "vni%s %s already configured in region %s" % (
                             suffix, coalescedOverlappingVnis, region.domain ) )
   return unique

def doRemoveVlanToVniMap( config, regionName, vlanRange ):
   vr = Tac.Value( 'VirtualNetwork::Range',
                   int( vlanRange[ 0 ] ), int( vlanRange[ -1 ] ) )
   clientDomain = ClientDomain( VNAClients.OpenStack, regionName )
   del config.vlanToVniMap[ clientDomain ].map[ vr ]
   if not config.vlanToVniMap[ clientDomain ].map:
      del config.vlanToVniMap[ clientDomain ]

#-------------------------------------------------------------------------------
# VLAN pool config commands
#-------------------------------------------------------------------------------

def loadVlanAssignment( regionName, vlanRange ):
   if regionName in openStackCliConfig.vlanAssignment.keys():
      del openStackCliConfig.vlanAssignment[ regionName ]
   openStackCliConfig.newVlanAssignment( regionName )
   for vlan in vlanRange:
      openStackCliConfig.vlanAssignment[ regionName ].vlan[ vlan ] = True

def safeToModifyVlanAssignment( mode, regionName ):
   # A VLAN Pool may only be modified or deleted if:
   # 1. This is a standby CVX OR
   # 2. The region is in sync OR
   # 3. The region has never synced and no previous vlan assignment exists
   if not clusterStatus.status[ 'default' ].isStandaloneOrLeader:
      return True
   SyncStatus = Tac.Type( 'OpenStack::SyncStatus' )
   if regionName in openStackStatus.regionStatus.keys() and \
          openStackStatus.regionStatus[ regionName ].syncStatus == \
          SyncStatus.syncComplete:
      return True
   if regionName not in openStackCliConfig.vlanAssignment.keys():
      # Check if the region has ever synced
      if regionName not in openStackStatus.regionStatus.keys():
         return True
      regionStatus = openStackStatus.regionStatus[ regionName ]
      if regionStatus.syncStatus == SyncStatus.unknown and \
             regionStatus.oldSyncStatus == SyncStatus.unknown:
         return True
   return False

#-------------------------------------------------------------------------------
# Region config mode under service openstack mode
#-------------------------------------------------------------------------------

class RegionConfigMode ( OSMode.RegionConfigModeBase, BasicCli.ConfigModeBase ):

   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, regionName, sourceMode ):
      self.region = None
      self.regionName = regionName
      OSMode.RegionConfigModeBase.__init__( self, regionName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def canEnter( self, sourceMode ):
      # The region is being synced by the sync thread or by one of the
      # neutron servers. If the region is being synced, then to enter the
      # region config mode the server sends the 'sync' keyword. If anything
      # else tries to enter the region (without the sync keyword), then the
      # CLI should prevent it.

      # To enter the region, the following conditions must be satisfied:
      # 1. Config.Status == 'unknown'
      #         This is the initial state where any neutron server can start a
      #         sync.
      # 2. Config.Status == 'inProgress'
      #    a. If Status.Status == 'timedout'
      #         A sync from a server timed out. If this happens, then the others
      #         should be able to send regular requests or sync.
      #    b. sourceMode == 'sync'
      #         One of the servers has expressed interest in syncing and is
      #         requesting to enter the region mode as part of that sync.

      if not cdbMountsComplete:
         return True

      self.region = openStackConfig.newRegion( self.regionName )
      SyncStatus = Tac.Type( 'OpenStack::SyncStatus' )

      if( self.region.syncStatus == SyncStatus.unknown or
            self.region.syncStatus == SyncStatus.syncComplete ):
         if sourceMode == 'sync':
            # The region is not syncing. But there was a request to enter the
            # sync mode. This could happen if the server syncing failed to send
            # a heartbeat and continued syncing. Since this can lead to an
            # inconsistent state, reject the request to enter the region.
            return False
         else:
            # Sync in not in progress and the request is to enter the region in
            # a regular mode.
            return True

      elif self.region.syncStatus == SyncStatus.syncInProgress:
         if( self.region.name in openStackStatus.regionStatus and
               openStackStatus.regionStatus[ self.region.name ].syncStatus ==
                  SyncStatus.syncTimedout ):
            # If CVX was supposed to be syncing but if it timedout, then allow
            # entring the region irrespective of the source mode.
            # Change the sync in config to unknown.
            self.region.syncStatus = SyncStatus.unknown
            if sourceMode == 'sync':
               # Once the sync times out, it shouldn't be possible to enter a region
               # in sync mode without restarting the sync
               return False
            else:
               return True
         else:
            # Region has not timed out
            # Sync is in progress.
            if sourceMode == 'sync':
               # The server explicitly requests CVX to enter sync mode.
               return True
            else:
               # The server did not request sync mode explicitly
               if self.region.requester == None or self.region.requester == '':
                  # The region has not been locked. This case is hit when the old
                  # neutron talks to new CVX.
                  return True
               else:
                  return False

      # Couldn't enter the region because
      # 1. Config.Status == syncInProgress and
      # 2. sourceMode != 'sync' and
      # 3. Status.syncStatus != syncTimedout
      # The only way to enter the region again is to
      # 1. Set Config.Status to syncComplete (using 'sync end' CLI command)
      # 2. Specify the 'sync' as the source mode.
      # 3. If the OpenStack agent has been enabled, wait for it to timeout.
      return False

class UserKeyCommand( CliCommand.CliCommandClass ):
   syntax = "username USERNAME tenant TENANT_NAME password SECRET"
   noOrDefaultSyntax = "username ..."
   data = {
      'username' : 'OpenStack user for region',
      'USERNAME' : AaaCli.usernameMatcher,
      'tenant' : 'OpenStack tenant name',
      'TENANT_NAME' : CliMatcher.QuotedStringMatcher(),
      'password' : 'OpenStack user\'s password',
      'SECRET' : reversibleSecretCliExpression( 'SECRET' )
      }

   @staticmethod
   def handler( mode, args ):
      credentials = openStackCliConfig.newCredentials( mode.regionName )
      credentials.username = args[ 'USERNAME' ]
      credentials.tenantName = args[ 'TENANT_NAME' ]
      credentials.password = args[ 'SECRET' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del openStackCliConfig.credentials[ mode.regionName ]

RegionConfigMode.addCommandClass( UserKeyCommand )

#-------------------------------------------------------------------------------
# networks map vlan VLAN_RANGE vni VNI_RANGE
#-------------------------------------------------------------------------------

class RegionNetworksMapCmd( CliCommand.CliCommandClass ):
   syntax = 'networks map vlan VLAN_RANGE vni VNI_RANGE'
   noOrDefaultSyntax = 'networks map vlan VLAN_RANGE ...'
   data = {
      'networks' : 'Openstack network config',
      'map' : 'Map vlan range to vni range',
      'vlan' : 'VLAN range',
      'VLAN_RANGE' : VlanCli.vlanSetMatcher,
      'vni' : 'VNI range',
      'VNI_RANGE' : vniRangeMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      vlanRange = sorted( args[ 'VLAN_RANGE' ].ids )
      vniRange = args[ 'VNI_RANGE' ].values()

      if len( vlanRange ) != len( vniRange ):
         mode.addError( 'Length of vlan range must be equal to vni range' )
         return

      vlanRangeList, vniRangeList = parseRangeInput( vlanRange, vniRange )
      # If a vni exists in another region's mapping, allow map but add a
      # warning stating that the vni/vni's are already configured
      uniqueVniRange( mode, vniRangeList )

      for index in range( len( vlanRangeList ) ):
         doMapVlanToVni( virtualNetworkConfig, mode.regionName,
                         vlanRangeList[ index ], vniRangeList[ index ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vlanRange = sorted( args[ 'VLAN_RANGE' ].ids )
      vlanRangeList, _ = parseRangeInput( vlanRange, vlanRange )
      for vr in vlanRangeList:
         try:
            doRemoveVlanToVniMap( virtualNetworkConfig, mode.region.name, vr )
         except KeyError:
            mode.addError( 'VLAN to VNI mapping does not exist for %d-%d' % (
               int( vr[ 0 ] ), int( vr[ -1 ] ) ) )

RegionConfigMode.addCommandClass( RegionNetworksMapCmd )

#--------------------------------------------------------------------------------
# resource-pool vlan VLAN_RANGE
#--------------------------------------------------------------------------------

class RegionResourcePoolCmd( CliCommand.CliCommandClass ):
   syntax = 'resource-pool vlan VLAN_RANGE'
   noOrDefaultSyntax = 'resource-pool vlan ...'
   data = {
      'resource-pool' : 'Assign a range of VLAN IDs for this region',
      'vlan' : 'VLAN segmentation type',
      'VLAN_RANGE' : VlanCli.vlanSetMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      vlanRange = args[ 'VLAN_RANGE' ].ids
      regionName = mode.regionName
      # Check if this is a no-op
      if regionName in openStackCliConfig.vlanAssignment.keys():
         assignedVlans = frozenset(
            openStackCliConfig.vlanAssignment[ regionName ].vlan.keys() )
         if not vlanRange.symmetric_difference( assignedVlans ):
            return
      if not cdbMountsComplete:
         mode.addWarning(
            "Changing OpenStack's VLAN range assignment while CVX is "
            "disabled may result in data path disruption if range does "
            "not include all VLANs that are currently in use."
         )
      elif not safeToModifyVlanAssignment( mode, regionName ):
         mode.addError( "VLAN assignment cannot be altered until region is in sync" )
         return
      # If we are updating an existing assignment, the new range must contain all
      # allocated VLANs from the assignment range (since this will result in
      # a state where some networks have vlans from outside the assignment range)
      if cdbMountsComplete and regionName in openStackStatus.regionStatus.keys():
         allocatedVlans = \
             openStackStatus.regionStatus[ regionName ].vlanPool.allocatedVlan
         if not all( vlan in vlanRange for vlan in allocatedVlans ):
            mode.addError(
               "Range doesn't include all VLANs that are already in use"
            )
            return
      loadVlanAssignment( regionName, vlanRange )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      regionName = mode.regionName
      # Check if this is a no-op
      if regionName not in openStackCliConfig.vlanAssignment.keys():
         return
      if not safeToModifyVlanAssignment( mode, regionName ):
         mode.addError( "VLAN assignment cannot be altered until region is in sync" )
         return
      if regionName in openStackStatus.regionStatus.keys():
         allocatedVlans = \
             openStackStatus.regionStatus[ regionName ].vlanPool.allocatedVlan
         if len( allocatedVlans ):
            mode.addError( "Cannot remove VLAN assignment while VLANS are in use" )
            return
      if regionName in openStackCliConfig.vlanAssignment.keys():
         del openStackCliConfig.vlanAssignment[ regionName ]

RegionConfigMode.addCommandClass( RegionResourcePoolCmd )

#--------------------------------------------------------------------------------
# provision sync [ ( mandatory | optional ) ]
#--------------------------------------------------------------------------------
class RegionProvisionSyncCmd( CliCommand.CliCommandClass ):
   syntax = 'provision sync [ ( optional | mandatory ) ]'
   noOrDefaultSyntax = 'provision sync ...'
   data = {
      'provision' : 'Configure provisioning prerequisites',
      'sync' : 'Region sync provisioning prerequisites',
      'mandatory' : 'Sync completion is mandatory prior to provisioning',
      'optional' : 'Sync completion is optional prior to provisioning',
   }

   @staticmethod
   def handler( mode, args ):
      if 'mandatory' in args:
         openStackCliConfig.mandatoryRegion[ mode.regionName ] = True
      elif mode.regionName in openStackCliConfig.mandatoryRegion:
         del openStackCliConfig.mandatoryRegion[ mode.regionName ]

   noOrDefaultHandler = handler

RegionConfigMode.addCommandClass( RegionProvisionSyncCmd )

#-------------------------------------------------------------------------------
# The 'keystone auth-url authUrl port authPort version apiVer' command
#-------------------------------------------------------------------------------
AuthApiVer = Tac.Type( 'OpenStack::AuthApiVer' )

class RegionKeystoneAuthUrlAuthurlCmd( CliCommand.CliCommandClass ):
   syntax = 'keystone auth-url URL [ port PORT ] [ version API_VERSION ]'
   noOrDefaultSyntax = 'keystone auth-url ...'
   data = {
      'keystone' : 'OpenStack keystone service',
      'auth-url' : 'OpenStack keystone service',
      'URL' : CliMatcher.PatternMatcher(
         pattern='.+',
         helpdesc='Endpoint URL for Keystone authentication server.',
         helpname='Endpoint URL' ),
      'port' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'port',
            helpdesc='Port number which authentication server listens to' ),
         hidden=True ),
      'PORT' : CliMatcher.IntegerMatcher( 1, 65535,
         helpdesc='Number of the port to use' ),
      'version' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'version',
            helpdesc='Authentication API version' ),
         hidden=True ),
      'API_VERSION' : CliMatcher.EnumMatcher(
         { AuthApiVer.v2 : 'API version 2.0', AuthApiVer.v3 : 'API version 3' }
      )
   }

   @staticmethod
   def handler( mode, args ):
      authUrl = args[ 'URL' ]
      authPort = args.get( 'PORT' )
      apiVer = args.get( 'API_VERSION' )

      # Similar to regex pattern used by django for validating URL
      authUrlRe = (
         r'^https?://'
         r'(?:(?:[A-Z0-9](?:[A-Z0-9-]{0,61}[A-Z0-9])?\.)+[A-Z]{2,6}\.?|'
         r'\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})' # or IP
         r'(?::\d+)?' # optional port
         r'(?:/?|[/?]\S+)$' )
      authRe = re.compile( authUrlRe, re.IGNORECASE )

      # urlparse is very forgiving and parse results can be unexpected for
      # malformed URLs, so validate form first

      if not authRe.match( authUrl ):
         mode.addError( 'Invalid URL %s' % authUrl )
         return

      parsed = urlparse.urlparse( authUrl )

      if authPort and parsed.port:
         if parsed.port != authPort:
            mode.addError(
               'Port %d must match port in URL %d' % ( authPort, parsed.port )
            )
            return
      elif parsed.port:
         authPort = parsed.port

      path = parsed.path
      # normalize path end
      if not path or path[ -1 ] != '/':
         path += '/'

      for version in ( AuthApiVer.v3, AuthApiVer.v2 ):
         if path[ : -1 ].endswith( version ):
            if not apiVer:
               apiVer = version
            elif apiVer != version:
               mode.addError(
                  'Version %s must match vesion in URL %s' %
                  ( apiVer, version )
               )
               return
            break
      else:
         # default to v3
         apiVer = apiVer or AuthApiVer.v3
         # append version to path
         path += '%s/' % apiVer

      if apiVer != AuthApiVer.v3:
         mode.addWarning( "Use of keystone v3 is strongly encouraged" )

      if mode.regionName not in openStackCliConfig.keystoneAuthUrl:
         openStackCliConfig.newKeystoneAuthUrl( mode.regionName )

      keystoneAuthUrl = openStackCliConfig.keystoneAuthUrl[ mode.regionName ]
      keystoneAuthUrl.authUrl = urlparse.urlunsplit(
           (
              parsed.scheme,
              parsed.hostname + ( ':%s' % authPort if authPort else '' ),
              path,
              '',
              ''
           )
      )
      keystoneAuthUrl.apiVersion = apiVer

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.regionName in openStackCliConfig.keystoneAuthUrl:
         del openStackCliConfig.keystoneAuthUrl[ mode.regionName ]

RegionConfigMode.addCommandClass( RegionKeystoneAuthUrlAuthurlCmd )

#------------------------------------------------------------------------------
# 'region RegionName [sync]' (hidden)
# 'no region RegionName' (hidden)
# sub-mode command in 'service openstack' mode.
#------------------------------------------------------------------------------

class RegionCmd( CliCommand.CliCommandClass ):
   syntax = 'region REGION_NAME [ sync ]'
   noOrDefaultSyntax = 'region EXISTING_REGION'
   data = {
      'region' : 'Region configuration',
      'REGION_NAME' : CliMatcher.QuotedStringMatcher(
          helpdesc='Region name',
          pattern=CliParser.excludePipeTokensPattern
      ),
      'sync' : 'Indicate that the region is being synchronized',
      'EXISTING_REGION' : regionNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      regionName = args[ 'REGION_NAME' ]
      sync = args.get( 'sync' )

      childMode = mode.childMode( RegionConfigMode, regionName=regionName,
                                  sourceMode=sync )
      if childMode.canEnter( sourceMode=sync ):
         mode.session_.gotoChildMode( childMode )
      else:
         mode.addError( "Region is being synced" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      regionName = args[ 'EXISTING_REGION' ]
      if not cdbMountsComplete:
         return

      if regionName in openStackConfig.region:
         # delete all entities under the specified region including the region
         clientDomain = ClientDomain( VNAClients.OpenStack, regionName )
         del openStackConfig.region[ regionName ]
         del virtualNetworkConfig.vlanToVniMap[ clientDomain ]
         del openStackCliConfig.credentials[ regionName ]
         del openStackCliConfig.vlanAssignment[ regionName ]
         del openStackCliConfig.mandatoryRegion[ regionName ]
      else:
         mode.addError( 'Invalid region name' )

OpenStackConfigMode.addCommandClass( RegionCmd )

#-------------------------------------------------------------------------------
# Clear commands
#-------------------------------------------------------------------------------
class _ClearOpenStackExpr( CliCommand.CliExpression ):
   expression = 'clear openstack'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'openstack' : 'Clear OpenStack Information'
   }

#-------------------------------------------------------------------------------
# 'clear openstack region RegionName'
#-------------------------------------------------------------------------------
class ClearOpenstackRegionCmd( CliCommand.CliCommandClass ):
   syntax = 'CLEAR region EXISTING_REGION'
   data = {
      'CLEAR' : _ClearOpenStackExpr,
      'region' : 'Region configuration',
      'EXISTING_REGION' : regionNameMatcher
   }

   handler = RegionCmd.noOrDefaultHandler

BasicCli.EnableMode.addCommandClass( ClearOpenstackRegionCmd )

#-------------------------------------------------------------------------------
# 'clear openstack ( ip | ipv6 ) access-list counters'
#-------------------------------------------------------------------------------

class ClearOpenstackIpAccessListCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'CLEAR ( ip | ipv6 ) access-list counters'
   data = {
      'CLEAR' : _ClearOpenStackExpr,
      'ip' : CliToken.Ip.ipMatcherForClear,
      'ipv6' : CliToken.Ipv6.ipv6MatcherForClear,
      'access-list' : AclCli.accessListKwMatcherForServiceAcl,
      'counters' : AclCli.countersKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      AclCli.clearServiceAclCounters(
         mode,
         osApiStatus.aclStatus,
         aclCheckpoint,
         'ipv6' if 'ipv6' in args else 'ip' )

BasicCli.EnableMode.addCommandClass( ClearOpenstackIpAccessListCountersCmd )

#-------------------------------------------------------------------------------
# The 'auth url KeystoneServiceEndPoint user UserName password Password
#      tenant TENANT_NAME' command
#-------------------------------------------------------------------------------

class HiddenRegionAuthCmd( CliCommand.CliCommandClass ):
   syntax = (
      'auth url ENDPOINT user USERNAME password PASSWORD [ tenant TENANT_NAME ]'
   )
   data = {
      'auth' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'auth',
            helpdesc='Specify the keystone service for a region' ),
         guard=osConfigGuard ),
      'url' : 'Specify the keystone service endpoint',
      'ENDPOINT' : CliMatcher.PatternMatcher(
         pattern='.*',
         helpdesc='Authentication service endPoint',
         helpname='endPoint' ),
      'user' : 'Specify the user name for keystone service admin',
      'USERNAME' : CliMatcher.QuotedStringMatcher(
         helpdesc='Keystone admin user name' ),
      'password' : 'Specify the password for keystone service admin',
      'PASSWORD' : CliCommand.Node(
         CliMatcher.QuotedStringMatcher(
            helpdesc='Keystone admin user password' ),
         sensitive=True ),
      'tenant' : 'Specify the tenant name for keystone service admin',
      'TENANT_NAME' : CliMatcher.QuotedStringMatcher(
         helpdesc='Keystone admin user tenant' ),
   }

   hidden = True

   @staticmethod
   def handler( mode, args ):
      authServiceName = 'keystone'

      # Delete the service endpoint if it already exists.
      # Then create the service endpoint with the new information.
      if authServiceName in mode.region.serviceEndPoint:
         del mode.region.serviceEndPoint[ authServiceName ]

      ep = mode.region.newServiceEndPoint( authServiceName )
      ep.authUrl = args[ 'ENDPOINT' ]
      ep.user = args[ 'USERNAME' ]
      ep.password = args[ 'PASSWORD' ]
      ep.tenant = args.get( 'TENANT_NAME', 'service' )

RegionConfigMode.addCommandClass( HiddenRegionAuthCmd )

#------------------------------------------------------------------------------
# The 'sync start interval <interval-value> / end' command
#------------------------------------------------------------------------------
nodeSync = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher(
      'sync',
      helpdesc='Represents the sync status for a region' ),
   guard=osConfigGuard,
   hidden=True
)

class RegionSyncStartCmd( CliCommand.CliCommandClass ):
   syntax = 'sync start'
   data = {
      'sync' : nodeSync,
      'start' : 'Respresents start synchronization trigger from Neutron-ML2',
   }

   @staticmethod
   def handler( mode, args ):
      # Set the Sync Start status
      SyncStatus = Tac.Type( 'OpenStack::SyncStatus' )

      mode.region.syncStatus = SyncStatus.syncInProgress

RegionConfigMode.addCommandClass( RegionSyncStartCmd )

class RegionSyncIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'sync interval SYNC_INTERVAL'
   data = {
      'sync' : nodeSync,
      'interval' : 'Represents sync interval value configured in Neutron-ML2',
      'SYNC_INTERVAL' : CliMatcher.IntegerMatcher(
         0,
         ONEDAY,
         helpdesc='Specify the interval in seconds that is configured in ML2' ),
   }

   @staticmethod
   def handler( mode, args ):
      # Set the Sync Start status
      mode.region.syncInterval = float( args[ 'SYNC_INTERVAL' ] )

RegionConfigMode.addCommandClass( RegionSyncIntervalCmd )

class RegionSyncEndCmd( CliCommand.CliCommandClass ):
   syntax = 'sync end'
   data = {
      'sync' : nodeSync,
      'end' : 'Represents end synchronization trigger from Neutron-ML2'
   }

   @staticmethod
   def handler( mode, args ):
      # Set the Sync Start status
      SyncStatus = Tac.Type( 'OpenStack::SyncStatus' )
      mode.region.requestId = ''
      mode.region.requester = ''
      mode.region.syncStatus = SyncStatus.syncComplete

RegionConfigMode.addCommandClass( RegionSyncEndCmd )

#------------------------------------------------------------------------------
# The 'sync lock' command
# This command locks the region preventing entering the region unless the source
# mode is 'sync'. This command is added to ensure that the old ML2 driver can
# continue to work with newer EOS releases.
#------------------------------------------------------------------------------

class RequestSyncLockCmd( CliCommand.CliCommandClass ):
   syntax = 'sync lock CLIENT_ID REQUEST_ID'
   data = {
      'sync' : nodeSync,
      'lock' : 'Locks the region',
      'CLIENT_ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='Client Id',
         helpname='id' ),
      'REQUEST_ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='Request Id',
         helpname='id' ),
   }

   @staticmethod
   def handler( mode, args ):
      # The region lock can be grabbed only if the region is not in the middle
      # of a sync or if it has timed out.
      SyncStatus = Tac.Type( 'OpenStack::SyncStatus' )
      if mode.region.syncStatus == SyncStatus.syncInProgress:
         if( mode.region.name in openStackStatus.regionStatus and
               openStackStatus.regionStatus[ mode.region.name ].syncStatus !=
                  SyncStatus.syncTimedout ):
            # The region is already being synced.
            # If the OpenStack agent is not running, then the sync status is
            # never updated and the multiple controllers will send all their
            # data to CVX.
            mode.addError( "Region is being synchronized" )
            return
      mode.region.requester = args[ 'CLIENT_ID' ]
      mode.region.requestId = args[ 'REQUEST_ID' ]
      mode.region.syncStatus = SyncStatus.syncInProgress

RegionConfigMode.addCommandClass( RequestSyncLockCmd )

#------------------------------------------------------------------------------
# The 'show sync lock'
#------------------------------------------------------------------------------

class ShowSyncLockCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show sync lock'
   data = {
      'sync' : 'Region sync provisioning prerequisites',
      'lock' : 'Locks for the region'
   }
   cliModel = OpenStackModels.RegionLockOwner
   hidden = True

   @staticmethod
   def handler( mode, args ):
      owner = OpenStackModels.RegionLockOwner()
      if mode.region.name in openStackStatus.regionStatus:
         owner.owner = openStackStatus.regionStatus[ mode.region.name ].owner
         owner.requestId = openStackStatus.regionStatus[ mode.region.name ].requestId
      else:
         owner.owner = ''
         owner.requestId = ''
      return owner

RegionConfigMode.addShowCommandClass( ShowSyncLockCmd )

#------------------------------------------------------------------------------
# The 'sync heartbeat' command
#------------------------------------------------------------------------------

class SyncHeartbeatCmd( CliCommand.CliCommandClass ):
   syntax = 'sync heartbeat'
   data = {
      'sync' : nodeSync,
      'heartbeat' : 'Send a heart beat during sync',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.region.syncHeartbeat = Tac.now()

RegionConfigMode.addCommandClass( SyncHeartbeatCmd )

#------------------------------------------------------------------------------
# Tenant sub-mode
#------------------------------------------------------------------------------

class TenantConfigMode ( OSMode.TenantConfigModeBase,
                         BasicCli.ConfigModeBase ):

   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, tenantId ):
      # Create the region if it does not already exist
      self.region = openStackConfig.newRegion( parent.regionName )

      modeStr = '%s-tenant' % parent.regionName
      OSMode.TenantConfigModeBase.__init__( self, modeStr )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.tenantId = tenantId
      self.region = parent.region
      self.regionName = parent.regionName
      # Create the tenant if it does not already exist
      self.tenant = self.region.newTenant( tenantId )

def cleanupTenantResources( region, tenantId ):
   instanceCollections = [ 'vmInstance', 'dhcpInstance', 'baremetalInstance',
                           'routerInstance' ]
   for ic in instanceCollections:
      collection = getattr( region, ic )
      for instance in collection.values():
         if instance.tenant is None or instance.tenant.id != tenantId:
            continue
         for port in instance.port.values():
            del region.portBinding[ port.id ]
            del region.port[ port.id ]
         del collection[ instance.id ]

   # Cleanup any remaining networks
   for network in region.network.values():
      if network.tenant.id == tenantId:
         for segmentId in network.staticSegment.keys():
            del region.segment[ segmentId ]
         for segmentId in network.dynamicSegment.keys():
            del region.segment[ segmentId ]
         del region.network[ network.id ]

#------------------------------------------------------------------------------
# The '[no] tenant ID'
# sub-mode hidden command in Region config mode
#------------------------------------------------------------------------------

class TenantCmd( CliCommand.CliCommandClass ):
   syntax = 'tenant TENANT_ID'
   noSyntax = syntax
   data = {
      'tenant' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'tenant',
            helpdesc='Tenant configuration'
         ),
         guard=osConfigGuard
      ),
      'TENANT_ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='Tenant id',
         helpname='id'
      ),
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( TenantConfigMode,
                                  tenantId=args[ 'TENANT_ID' ] )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noHandler( mode, args ):
      region = mode.region
      tenantId = args[ 'TENANT_ID' ]
      if tenantId in region.tenant:
         cleanupTenantResources( region, tenantId )
         del region.tenant[ tenantId ]
      else:
         mode.addError( 'Invalid tenant id' )

RegionConfigMode.addCommandClass( TenantCmd )

#------------------------------------------------------------------------------
# Network sub-mode
#------------------------------------------------------------------------------

class NetworkConfigMode( OSMode.NetworkConfigModeBase,
                         BasicCli.ConfigModeBase ):

   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, networkId, networkName ):
      if networkName:
         netName = networkName
      else:
         netName = 'Network'
      modeStr = ( '%s-tenant-%s' % ( parent.regionName, netName ) )
      OSMode.NetworkConfigModeBase.__init__( self, modeStr )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.networkId = networkId
      self.networkName = networkName or ''
      self.tenant = parent.tenant
      self.tenantId = parent.tenantId
      self.region = parent.region
      self.regionName = parent.regionName
      self.shared = False

      # Note that the TenantNetwork entity is created when the
      # segment is specified.

#------------------------------------------------------------------------------
# 'network id TENANT-NETWORK-ID [ name TENANT-NETWORK-NAME' ]
# 'no network id TENANT-NETWORK-ID'
# sub-mode hidden command in tenant config mode
#------------------------------------------------------------------------------

class NetworkCmd( CliCommand.CliCommandClass ):
   syntax = 'network id ID [ name NAME ]'
   noOrDefaultSyntax = 'network id ID'
   data = {
      'network' : 'Tenant network configuration',
      'id' : 'Specify tenant network ID',
      'ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='Tenant network id',
         helpname='id'
      ),
      'name' : 'Specify tenant network name',
      'NAME' : CliMatcher.QuotedStringMatcher(
         helpdesc='Tenant network name',
         helpname='name' ),
   }

   @staticmethod
   def handler( mode, args ):
      networkId = args[ 'ID' ]
      networkName = args.get( 'NAME' )
      childMode = mode.childMode( NetworkConfigMode,
                                  networkId=networkId,
                                  networkName=networkName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      networkId = args[ 'ID' ]
      if networkId in mode.region.network:
         segments = mode.region.network[ networkId ].staticSegment
         for segmentId in segments:
            del mode.region.network[ networkId ].staticSegment[ segmentId ]
            del mode.region.segment[ segmentId ]
         # Remove dynamic segments
         segments = mode.region.network[ networkId ].dynamicSegment
         for segmentId in segments:
            del mode.region.network[ networkId ].dynamicSegment[ segmentId ]
            del mode.region.segment[ segmentId ]
         del mode.region.network[ networkId ]
      else:
         mode.addError( 'Invalid tenant network id' )

TenantConfigMode.addCommandClass( NetworkCmd )

#------------------------------------------------------------------------------
# Tenant Network Segmentation Configuration Commands and Tokens
# '[no] segment SegmentId type SegmentationType id SegmentationId
#     [static|dynamic]'
# command in tenant network config mode
#------------------------------------------------------------------------------

class SegmentCmd( CliCommand.CliCommandClass ):
   syntax = ( 'segment SEGMENT_ID '
              'type ( ( vlan id VLAN_ID ) | ( vxlan id VNI ) )'
              ' [ ( dynamic | static ) ]' )
   noSyntax = 'segment SEGMENT_ID ...'
   data = {
      'segment' : 'Tenant network segment configuration',
      'SEGMENT_ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='Tenant network segment id',
         helpname='id' ),
      'type' : 'Specify segmentation type',
      'vlan' : 'VLAN segmentation type',
      'vxlan' : 'VxLAN segmentation type',
      'id' : 'Specify segmentation type id',
      'VLAN_ID' : VlanCli.vlanIdMatcher,
      'VNI' : CliMatcher.IntegerMatcher(
         1, 16777216,
         helpdesc='Virtual Network Identifier' ),
      'dynamic' : 'Mark the segment as dynamic',
      'static' : 'Mark the segment as static'
   }

   @staticmethod
   def handler( mode, args ):
      segmentId = args[ 'SEGMENT_ID' ]
      segmentationType = 'vlan' if 'vlan' in args else 'vxlan'
      segmentationTypeId = args[ 'VLAN_ID' ].id if 'vlan' in args else args[ 'VNI' ]
      segmentType = args.get( 'dynamic', 'static' )
      # The segmentId parameter is not used for now. The CLI currently supports
      # multiple segments per tenant network but Havana does not support this.
      # Left this way so when OpenStack supports multiple segments per tenant
      # network we hopefully will not have to change the CLI.

      driverType = Tac.Type( "OpenStack::TypeDriver" )
      if segmentationType == 'vlan' and \
             openStackCliConfig.vlanTypeDriver == driverType.aristaVlan:
         vlanPool = None
         if mode.region.name in openStackStatus.regionStatus:
            vlanPool = openStackStatus.regionStatus[ mode.region.name ].vlanPool
         if vlanPool and segmentationTypeId not in ( vlanPool.availableVlan.keys() +
                                                     vlanPool.allocatedVlan.keys() ):
            mode.addError( "VLAN segmentation id %d is not available" %
                           segmentationTypeId )
            Logging.log( OpenStackLogMsgs.CVX_OPENSTACK_INVALID_NETWORK_VLAN,
                         segmentationTypeId,
                         mode.networkId )
            return

      network = mode.region.newNetwork( mode.networkId, mode.tenant )
      network.networkName = mode.networkName

      # Backward compatibility
      # Older drivers send segment id as 1. If the segment id is 1, the network
      # id is used as a segment id.
      # Once the driver is updated, it will send the actual segment id and
      # specify level 0 over writing the existing segment.
      try:
         if int( segmentId ) == 1:
            segmentId = mode.networkId
      except ValueError:
         pass
      segment = mode.region.newSegment( segmentId, segmentationType,
                                        segmentationTypeId, network.id )
      if segmentType == 'static':
         network.staticSegment.addMember( segment )
      elif segmentType == 'dynamic':
         network.dynamicSegment.addMember( segment )

      if mode.shared:
         network.shared = mode.shared

      mode.network = network

   @staticmethod
   def noHandler( mode, args ):
      del mode.region.segment[ args[ 'SEGMENT_ID' ] ]

NetworkConfigMode.addCommandClass( SegmentCmd )

class SharedCmd( CliCommand.CliCommandClass ):
   syntax = 'shared'
   noOrDefaultSyntax = syntax
   data = {
      'shared' : 'Mark a network as shared.',
   }

   @staticmethod
   def handler( mode, args ):
      mode.shared = '__default__' not in args and '__no__' not in args
      if mode.networkId in mode.region.network:
         mode.region.network[ mode.networkId ].shared = mode.shared

   noOrDefaultHandler = handler

NetworkConfigMode.addCommandClass( SharedCmd )

#------------------------------------------------------------------------------
# Vm sub-mode
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# 'vm id VM-ID hostid HOST_ID'
# 'no vm id VM-ID'
# sub-mode hidden commands in tenant config mode
#------------------------------------------------------------------------------
def cleanupInstancePorts( region, instance ):
   for port in instance.port.values():
      del region.portBinding[ port.id ]
      del region.port[ port.id ]

def noInstance( mode, args ):
   instanceId = args[ 'INSTANCE_ID' ]
   region = mode.region
   instanceCollection = None
   if instanceId in region.baremetalInstance:
      instanceCollection = region.baremetalInstance
   elif instanceId in region.vmInstance:
      instanceCollection = region.vmInstance
   elif instanceId in region.dhcpInstance:
      instanceCollection = region.dhcpInstance
   elif instanceId in region.routerInstance:
      instanceCollection = region.routerInstance
   else:
      mode.addWarning( 'Invalid instance id' )
      return
   cleanupInstancePorts( region, instanceCollection[ instanceId ] )
   del instanceCollection[ instanceId ]

class VmInstanceidCmd( CliCommand.CliCommandClass ):
   syntax = 'vm id INSTANCE_ID hostid INSTANCE_HOST_ID'
   noSyntax = 'vm id INSTANCE_ID ...'
   data = {
      'vm' : 'Tenant VM configuration',
      'id' : 'Specify tenant VM ID',
      'INSTANCE_ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='Tenant VM id',
         helpname='id'
      ),
      'hostid' : 'Specify tenant VM host ID',
      'INSTANCE_HOST_ID' : CliMatcher.PatternMatcher(
         pattern='.*',
         helpdesc='Tenant VM host id',
         helpname='hostid'
      ),
   }

   handler = enterTenantInstanceConfigMode
   noHandler = noInstance

TenantConfigMode.addCommandClass( VmInstanceidCmd )

#-------------------------------------------------------------------------------
# Instance sub-mode
#-------------------------------------------------------------------------------

class TenantInstanceConfigMode( OSMode.TenantInstanceConfigModeBase,
                                BasicCli.ConfigModeBase ):

   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, instanceId, instanceHostId, instanceType ):
      modeStr = ( '%s-tenant-instance' % ( parent.regionName ) )
      OSMode.TenantInstanceConfigModeBase.__init__( self, modeStr )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.session_ = session
      self.region = parent.region
      self.instanceType = instanceType
      hostId = instanceHostId or ""
      # Create the instance if it does not already exist
      if instanceType == INSTANCE_TYPE_BAREMETAL:
         self.instance = self.region.newBaremetalInstance( instanceId )
         self.instance.baremetalHostId = hostId
      elif instanceType == INSTANCE_TYPE_VIRTUAL:
         self.instance = self.region.newVmInstance( instanceId )
         self.instance.vmHostId = hostId
      elif instanceType == INSTANCE_TYPE_ROUTER:
         self.instance = self.region.newRouterInstance( instanceId )
         self.instance.routerHostId = hostId

      self.instance.tenant = parent.tenant

   def exit( self ):
      self.session_.gotoParentMode()

#-------------------------------------------------------------------------------
# 'instance id INSTANCE_ID [hostid INSTANCE_HOST_ID]
#    type <baremetal | virtual | router>'
# 'no instance id INSTANCE-ID'
# sub-mode hidden commands in tenant config mode
#-------------------------------------------------------------------------------

class InstanceCmd( CliCommand.CliCommandClass ):
   syntax = ( 'instance id INSTANCE_ID [ hostid INSTANCE_HOST_ID ] '
              'type INSTANCE_TYPE' )
   noSyntax = 'instance id INSTANCE_ID ...'
   data = {
      'instance' : 'Tenant instance configuration',
      'id' : 'Specify tenant instance ID',
      'INSTANCE_ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='Tenant instance id',
         helpname='id'
       ),
      'hostid' : 'Specify tenant instance host ID',
      'INSTANCE_HOST_ID' : CliMatcher.PatternMatcher(
         pattern='.*',
         helpdesc='Tenant instance host id',
         helpname='hostid'
      ),
      'type' : 'Specify tenant instance type',
      'INSTANCE_TYPE' : CliMatcher.EnumMatcher(
         {
            'baremetal' : 'The instance is running on baremetal',
            'virtual' : 'The instance is running on a hypervisor',
            'router' : 'The instance is a logical router'
         }
      )
   }

   handler = enterTenantInstanceConfigMode
   noHandler = noInstance

TenantConfigMode.addCommandClass( InstanceCmd )

#-------------------------------------------------------------------------------
# The 'sync heartbeat' command
#-------------------------------------------------------------------------------

TenantConfigMode.addCommandClass( SyncHeartbeatCmd )

#-------------------------------------------------------------------------------
# Port Configuration Commands and Tokens
#-------------------------------------------------------------------------------
class PortConfigMode( OSMode.PortConfigModeBase,
                      BasicCli.ConfigModeBase ):

   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, portId, binding ):
      modeStr = ( '%s-port' % ( parent.region.name ) )
      OSMode.PortConfigModeBase.__init__( self, modeStr )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.session_ = session
      self.region = parent.region
      self.portId = portId
      self.binding = binding

   def onExit( self ):
      # Backward compatibility
      if 0 not in self.binding.segment:
         # If the port was not configured with any segment, then an older driver
         # is talking to CVX
         port = self.region.port[ self.portId ]
         networkId = port.network.id
         if networkId in self.region.segment:
            # Older drivers use networkId as the segmentId
            segment = self.region.segment[ networkId ]
            self.binding.segment[ 0 ] = segment

#-------------------------------------------------------------------------------
# segment level LEVEL id SEGMENT_ID
#-------------------------------------------------------------------------------
class SegmentLevelLevelIdSegmentidCmd( CliCommand.CliCommandClass ):
   syntax = 'segment level LEVEL id SEGMENT_ID'
   data = {
      'segment' : 'Tenant network segment configuration',
      'level' : 'Segment level',
      'LEVEL' : CliMatcher.IntegerMatcher(
         0,
         10,
         helpdesc='Segment level' ),
      'id' : 'Segment ID',
      'SEGMENT_ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='Tenant network segment id',
         helpname='id' ),
   }

   @staticmethod
   def handler( mode, args ):
      segment = mode.region.segment[ args[ 'SEGMENT_ID' ] ]

      # Add the segment at the specified level
      mode.binding.segment[ int( args[ 'LEVEL' ] ) ] = segment

PortConfigMode.addCommandClass( SegmentLevelLevelIdSegmentidCmd )

#-------------------------------------------------------------------------------
# 'port id PORT-ID [ name PORT-NAME ] network-id TENANT-NETWORK-ID
#     [port-type native | trunk | allowed] [hostid host]
#        [switch-id SWITCH-ID switchport INTERFACE]'
# 'no port id PORT-ID'
# sub-mode hidden command in vm config mode
#-------------------------------------------------------------------------------
class TenantInstancePortCmd( CliCommand.CliCommandClass ):
   syntax = (
      'port id PORT_ID [ name PORT_NAME ] network-id NETWORK_ID '
      '[ type PORT_VLAN_TYPE ] '
      '[ hostid HOST_ID ] [ switch-id SWITCH_ID switchport INTF ]'
   )
   noSyntax = (
      'port id PORT_ID '
      '[ hostid HOST_ID ] [ switch-id SWITCH_ID switchport INTF ]'
   )
   data = {
      'port' : 'Port configuration',
      'id' : 'Specify port ID',
      'PORT_ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='OpenStack identity string for the port',
         helpname='id'
      ),
      'name' : 'Port name configuration',
      'PORT_NAME' : CliMatcher.QuotedStringMatcher(
         helpdesc='Name for the port' ),
      'network-id' : 'Specify port tenant network ID',
      'NETWORK_ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='Port tenant network id',
         helpname='id'
      ),
      'type' : 'Specify port VLAN type',
      'PORT_VLAN_TYPE' : CliMatcher.EnumMatcher(
         {
            'native' : 'Port has native VLAN',
            'access' : 'Port is access VLAN',
            'allowed' : 'Port is allowed VLAN'
         }
      ),
      'hostid' : 'Specify tenant port host ID',
      'HOST_ID' : CliMatcher.PatternMatcher(
         pattern='.*',
         helpdesc='Tenant port host id',
         helpname='hostid'
      ),
      'switch-id' : 'Connected switch',
      'SWITCH_ID' : CliMatcher.PatternMatcher(
         pattern='([0-9A-Fa-f]{2}[-:.]?){5}([0-9A-Fa-f]{2})',
         helpdesc='MAC address used to identify a switch',
         helpname='id'
      ),
      'switchport' : 'Connected switchport',
      'INTF' : CliMatcher.PatternMatcher(
         pattern='([A-Za-z]+[0-9]+(\\/[0-9]+)*)',
         helpdesc='Interface connected to baremetalhost',
         helpname='interface'
      ),
   }

   @staticmethod
   def handler( mode, args ):
      portId = args[ 'PORT_ID' ]
      portName = args.get( 'PORT_NAME' )
      portNetworkId = args[ 'NETWORK_ID' ]
      portVlanType = args.get( 'PORT_VLAN_TYPE' )
      hostId = args.get( 'HOST_ID' )
      switchId = args.get( 'SWITCH_ID' )
      interfaceName = args.get( 'INTF' )

      # Make sure tenant network id is valid
      try:
         if not portVlanType:
            portVlanType = 'allowed'

         if portVlanType == 'access':
            mode.addError( 'Access ports are not supported' )
            return

         if portNetworkId not in mode.region.network:
            mode.addWarning( "Network %s does not exist" % portNetworkId )

         # Create the port if it does not already exist
         # Deleting the port will flap the link
         port = mode.region.newPort( portId, portVlanType )
         port.network = mode.region.network[ portNetworkId ]
         port.instance = mode.instance

         if portName:
            port.portName = portName

         mode.instance.port.addMember( port )

         portBinding = mode.region.newPortBinding( portId, port )

         # If the port is not explicitly associated with a host, use the
         # instance's host
         host = hostId or mode.instance.hostId

         binding = None
         if switchId:
            switchId = normalizeSwitchId( switchId )
            ( interfaceName, valid ) = normalizeInterfaceName( interfaceName )
            if not valid:
               mode.addError( "Invalid interface name" )
               return
            switchInterface = SwitchInterface( switchId, interfaceName )
            binding = portBinding.newPortToSwitchInterfaceBinding(
               switchInterface,
               host )
         else:
            binding = portBinding.newPortToHostBinding( host )

         # Backward compatibility
         # Older drivers send segment id as 1. If the segment id is 1, the
         # network id is used as a segment id.
         # Once the driver is updated, it will send the actual segment id and
         # specify level 0 over writing the existing segment.
         segmentId = portNetworkId
         if segmentId in mode.region.segment:
            segment = mode.region.segment[ segmentId ]
            # Add the segment at the specified level
            binding.segment[ 0 ] = segment

         childMode = mode.childMode( PortConfigMode,
                                     portId=portId,
                                     binding=binding )
         mode.session_.gotoChildMode( childMode )

      except KeyError:
         mode.addError( 'Invalid tenant network id' )

   @staticmethod
   def noHandler( mode, args ):
      portId = args[ 'PORT_ID' ]
      hostId = args.get( 'HOST_ID' )
      switchId = args.get( 'SWITCH_ID' )
      interfaceName = args.get( 'INTF' )

      if portId in mode.region.port:
         port = mode.region.port[ portId ]
         portBinding = mode.region.portBinding[ portId ]
         if hostId or switchId:
            if hostId:
               del portBinding.portToHostBinding[ hostId ]
            if switchId and interfaceName:
               switchId = normalizeSwitchId( switchId )
               ( interfaceName, valid ) = normalizeInterfaceName( interfaceName )
               if not valid:
                  mode.addError( "Invalid interface name" )
                  return
               switchInterface = SwitchInterface( switchId, interfaceName )
               del portBinding.portToSwitchInterfaceBinding[ switchInterface ]
            if( not len( portBinding.portToHostBinding ) and
                not len( portBinding.portToSwitchInterfaceBinding ) ):
               del mode.region.portBinding[ portId ]
               del mode.instance.port[ portId ]
               del mode.region.port[ portId ]
         else:
            del mode.region.portBinding[ portId ]
            del mode.instance.port[ portId ]
            del mode.region.port[ portId ]

         # Delete a VM with no ports and go back to tenant mode
         if not len( mode.instance.port ):
            # The instance needs to be deleted from the tenant.
            # Deleting mode.instance deletes the instance from the mode object
            if mode.instanceType == INSTANCE_TYPE_BAREMETAL:
               del mode.region.baremetalInstance[ mode.instance.id ]
            elif mode.instanceType == INSTANCE_TYPE_VIRTUAL:
               del mode.region.vmInstance[ mode.instance.id ]
            elif mode.instanceType == INSTANCE_TYPE_ROUTER:
               del mode.region.routerInstance[ mode.instance.id ]
            mode.exit()
      else:
         mode.addError( 'Invalid port id' )

TenantInstanceConfigMode.addCommandClass( TenantInstancePortCmd )

#-------------------------------------------------------------------------------
# The 'sync heartbeat' command
#-------------------------------------------------------------------------------

TenantInstanceConfigMode.addCommandClass( SyncHeartbeatCmd )

#-------------------------------------------------------------------------------
# Tenant Network DHCP Configuration Command and Tokens
# 'dhcp id DHCP-ID hostid HOST_ID port-id PORT-ID [ name PORT-NAME ]'
# 'no dhcp id DHCP-ID port-id PORT-ID'
# command in tenant config mode
#-------------------------------------------------------------------------------

def _dhcpInstancePortIdNames( mode, context ):
   return getattr(
      mode.region.dhcpInstance.get( context.sharedResult[ 'EX_INSTANCE' ] ),
      'portId',
      ()
   )

class DhcpCmd( CliCommand.CliCommandClass ):
   syntax = (
      'dhcp id INSTANCE_ID hostid HOST_ID port-id PORT_ID [ name PORT_NAME ]'
   )
   noSyntax = 'dhcp id EX_INSTANCE port-id EX_PORT_ID [ hostid HOST_ID ]'
   data = {
      'dhcp' : 'Tenant network DHCP configuration',
      'INSTANCE_ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='DHCP instance id',
         helpname='id' ),
      'EX_INSTANCE' : CliMatcher.DynamicNameMatcher(
         lambda mode : mode.region.dhcpInstance,
         'DHCP instance id',
         pattern=openStackIdRe,
         helpname='id' ),
      'id' : 'Specify DHCP instance ID',
      'hostid' : 'Specify DHCP host ID',
      'HOST_ID' : CliMatcher.PatternMatcher(
         pattern='.*',
         helpdesc='DHCP host id',
         helpname='hostid' ),
      'port-id' : 'Specify DHCP port ID',
      'PORT_ID' : CliMatcher.PatternMatcher(
         pattern=openStackIdRe,
         helpdesc='OpenStack identity string for the port',
         helpname='id' ),
      'EX_PORT_ID' : CliMatcher.DynamicNameMatcher(
         _dhcpInstancePortIdNames,
         'OpenStack identity string for the port',
         helpname='id',
         passContext=True ),
      'name' : 'Port name configuration',
      'PORT_NAME' : CliMatcher.QuotedStringMatcher(
         helpdesc='Name for the port' ),
   }

   @staticmethod
   def handler( mode, args ):
      if mode.networkId in mode.region.network:
         # Add DHCP Instance information to the Tenant
         dhcpNetwork = mode.region.network[ mode.networkId ]
         dhcpInstance = mode.region.newDhcpInstance( args[ 'INSTANCE_ID' ] )
         dhcpInstance.tenant = mode.tenant
         dhcpInstance.dhcpHostId = args[ 'HOST_ID' ]

         dhcpPort = mode.region.newPort( args[ 'PORT_ID' ], 'allowed' )
         dhcpPort.network = dhcpNetwork
         dhcpPort.instance = dhcpInstance

         if 'PORT_NAME' in args:
            dhcpPort.portName = args[ 'PORT_NAME' ]

         dhcpInstance.port.addMember( dhcpPort )

         portBinding = mode.region.newPortBinding( args[ 'PORT_ID' ], dhcpPort )
         binding = portBinding.newPortToHostBinding( args[ 'HOST_ID' ] )

         # Backward compatibility
         # Older drivers send segment id as 1. If the segment id is 1, the network
         # id is used as a segment id.
         # Once the driver is updated, it will send the actual segment id and
         # specify level 0 over writing the existing segment.
         segmentId = mode.networkId
         if segmentId in mode.region.segment:
            segment = mode.region.segment[ segmentId ]
            # Add the segment at the specified level
            binding.segment[ 0 ] = segment

         childMode = mode.childMode( PortConfigMode,
                                     portId=args[ 'PORT_ID' ],
                                     binding=binding )
         mode.session_.gotoChildMode( childMode )
      else:
         mode.addError( 'Invalid tenant network id' )

   @staticmethod
   def noHandler( mode, args ):
      dhcpInstanceId = args[ 'EX_INSTANCE' ]
      portId = args[ 'EX_PORT_ID' ]
      hostId = args.get( 'HOST_ID' )

      if not dhcpInstanceId in mode.region.dhcpInstance:
         mode.addError( 'Invalid DHCP instance id' )
         return
      dhcpInstance = mode.region.dhcpInstance[ dhcpInstanceId ]
      if not portId in dhcpInstance.port:
         mode.addError( 'Invalid port id' )
         return
      if hostId:
         portBinding = mode.region.portBinding[ portId ]
         del portBinding.portToHostBinding[ hostId ]
         if not len( portBinding.portToHostBinding ):
            del dhcpInstance.port[ portId ]
            del mode.region.portBinding[ portId ]
            del mode.region.port[ portId ]
      else:
         del mode.region.portBinding[ portId ]
         del dhcpInstance.port[ portId ]
         del mode.region.port[ portId ]

      if not len( dhcpInstance.port ):
         del mode.region.dhcpInstance[ dhcpInstanceId ]

NetworkConfigMode.addCommandClass( DhcpCmd )

#-------------------------------------------------------------------------------
# OpenStack graceful reboot commands
#    grace-period <time in seconds>
#    region-sync-timeout <time in seconds>
#-------------------------------------------------------------------------------
class GracePeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'grace-period GRACE_PERIOD'
   noOrDefaultSyntax = 'grace-period ...'
   data = {
      'grace-period' :
         ( 'Set the grace period for which the OpenStack agent '
           'waits for OpenStack region data' ),
      'GRACE_PERIOD' :
         CliMatcher.IntegerMatcher( 0, FOURHOURS,
            helpdesc='grace time in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      gracePeriod = args.get(
         'GRACE_PERIOD',
         openStackCliConfig.defaultGracePeriod
      )
      openStackCliConfig.gracePeriod = float( gracePeriod )

   noOrDefaultHandler = handler

OpenStackConfigMode.addCommandClass( GracePeriodCmd )

class SyncTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'sync-timeout REGION_SYNC_TIMEOUT'
   noOrDefaultSyntax = 'sync-timeout ...'
   data = {
      'sync-timeout' : 'Set a timeout value for a region sync',
      'REGION_SYNC_TIMEOUT' :
         CliMatcher.IntegerMatcher( 0, FOURHOURS,
            helpdesc='sync timeout in seconds' ),
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      pass

   noOrDefaultHandler = handler

OpenStackConfigMode.addCommandClass( SyncTimeoutCmd )

#-------------------------------------------------------------------------------
# OpenStack Show Commands
#
# The show commands simply apply the user supplied discriminant to the
# TAC model creating a CLI model of the data (point in time image) which is
# then passed off to the CAPI model for rendering.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# The 'show openstack config region RegionName'
# hidden command. Used by the ML2 agent during a sync to download all the
# data associated with the specified region.
#-------------------------------------------------------------------------------
def getRegionInstances( region, instanceType ):
   typeToInstanceTypeMap = { 'virtual' : 'vmInstance',
                             'baremetal' : 'baremetalInstance',
                             'dhcp' : 'dhcpInstance',
                             'router' : 'routerInstance' }
   return getattr( region, typeToInstanceTypeMap[ instanceType ] )

def getInstanceConfigModelType( instanceType ):
   typeToModelMap = { 'virtual' : OpenStackModels.VmInstanceConfig,
                      'baremetal' : OpenStackModels.BaremetalInstanceConfig,
                      'dhcp' : OpenStackModels.VmInstanceConfig,
                      'router' : OpenStackModels.RouterInstanceConfig }
   return typeToModelMap[ instanceType ]

def getModelTenantInstances( tenant, instanceType ):
   typeToModelInstancesMap = { 'virtual' : 'tenantVmInstances',
                                'baremetal' : 'tenantBaremetalInstances',
                                'dhcp' : 'tenantVmInstances',
                                'router' : 'tenantRouterInstances' }
   return getattr( tenant, typeToModelInstancesMap[ instanceType ] )

def getInstanceId( instance, instanceType ):
   typeToModelInstanceIdMap = { 'virtual' : 'vmInstanceId',
                                 'baremetal' : 'baremetalInstanceId',
                                 'dhcp' : 'dhcpInstanceId',
                                 'router' : 'routerInstanceId' }
   return getattr( instance, typeToModelInstanceIdMap[ instanceType ] )

def getInstanceHostId( instance, instanceType ):
   typeToModelInstanceIdMap = { 'virtual' : 'vmHostId',
                                 'baremetal' : 'baremetalHostId',
                                 'dhcp' : 'dhcpHostId',
                                 'router' : 'routerHostId' }
   return getattr( instance, typeToModelInstanceIdMap[ instanceType ] )

def buildTenantsModel( region ):
   # Go through all the OpenStack data and build the necessary models
   modelTenants = OpenStackModels.TenantsConfig()
   for tenantId in region.tenant:
      modelTenants.tenants[ tenantId ] = OpenStackModels.TenantConfig(
         tenantId=tenantId
      )

   for network in region.network.values():
      if network.tenant is None or network.tenant.id not in modelTenants.tenants:
         continue
      modelTenant = modelTenants.tenants[ network.tenant.id ]
      modelNetwork = OpenStackModels.NetworkConfig(
         networkId=network.id,
         networkName=network.networkName,
         shared=network.shared )
      if len( network.staticSegment.values() ):
         firstStaticSegment = network.staticSegment.values()[ 0 ]
         modelNetwork.segmentationType = firstStaticSegment.type
         modelNetwork.segmentationTypeId = firstStaticSegment.segmentationId
      else:
         continue

      for s in network.staticSegment.values():
         modelSegment = OpenStackModels.SegmentConfig(
            segmentId=s.id, segmentationType=s.type,
            segmentationTypeId=s.segmentationId,
            networkId=s.networkId
         )
         modelNetwork.staticSegment.append( modelSegment )
      for s in network.dynamicSegment.values():
         modelSegment = OpenStackModels.SegmentConfig(
            segmentId=s.id, segmentationType=s.type,
            segmentationTypeId=s.segmentationId,
            networkId=s.networkId
         )
         modelNetwork.dynamicSegment.append( modelSegment )
      modelTenant.tenantNetworks[ network.id ] = modelNetwork

   for instanceType in [ 'virtual', 'baremetal', 'dhcp', 'router' ]:
      instances = getRegionInstances( region, instanceType )
      modelType = getInstanceConfigModelType( instanceType )
      for instance in instances.values():
         if ( instance.tenant is None or
             instance.tenant.id not in modelTenants.tenants ):
            continue
         modelTenant = modelTenants.tenants[ instance.tenant.id ]
         tenantModelInstances = getModelTenantInstances( modelTenant, instanceType )
         modelInstance = modelType()
         instanceId = getInstanceId( instance, instanceType )
         instanceHostId = getInstanceHostId( instance, instanceType )
         modelInstance.setInstanceId( instanceId )
         modelInstance.setInstanceHostId( instanceHostId )
         tenantModelInstances[ instanceId ] = modelInstance
         modelPorts = modelInstance.getInstancePorts()

         for port in instance.port.values():
            segmentList = []
            modelSwitchports = dict()
            if port.network is None or port.id not in region.portBinding.keys():
               continue
            portBinding = region.portBinding[ port.id ]
            for binding in portBinding.portToSwitchInterfaceBinding.values():
               switchport = binding.switchInterface
               if switchport.switchId not in modelSwitchports:
                  modelSwitchports[ switchport.switchId ] = (
                     OpenStackModels.SwitchportList()
                  )
               modelSwitchports[ switchport.switchId ].switchports.append(
                  switchport.interface )
               for _, segment in sorted( binding.segment.iteritems() ):
                  modelSegment = OpenStackModels.SegmentConfig(
                     segmentId=segment.id,
                     segmentationType=segment.type,
                     segmentationTypeId=segment.segmentationId,
                     networkId=segment.networkId
                  )
                  if modelSegment not in segmentList:
                     segmentList.append( modelSegment )

            hosts = portBinding.portToHostBinding.keys()
            for binding in portBinding.portToHostBinding.values():
               for _, segment in sorted( binding.segment.iteritems() ):
                  modelSegment = OpenStackModels.SegmentConfig(
                     segmentId=segment.id,
                     segmentationType=segment.type,
                     segmentationTypeId=segment.segmentationId,
                     networkId=segment.networkId
                  )
                  if modelSegment not in segmentList:
                     segmentList.append( modelSegment )
            modelPorts[ port.id ] = OpenStackModels.PortConfig(
               portId=port.id,
               networkId=port.network.id,
               portVlanType=port.portVlanType,
               hosts=hosts,
               switches=modelSwitchports,
               segments=segmentList )
   return modelTenants

class ShowOpenstackConfigRegionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack config region REGION_NAME'
   data = {
      'openstack' : matcherOpenstack,
      'config' : matcherConfig,
      'region' : matcherRegion,
      'REGION_NAME' : regionNameMatcher
   }

   cliModel = OpenStackModels.TenantsConfig
   hidden = True

   @staticmethod
   def handler( mode, args ):
      regionName = args[ 'REGION_NAME' ]
      modelTenants = OpenStackModels.TenantsConfig()
      if not cdbMountsComplete:
         # We may have never mounted the config from controllerdb
         mode.addWarning( 'CVX not ready' )
         return modelTenants

      if regionName not in openStackConfig.region:
         mode.addError( 'Invalid region name' )
         return modelTenants

      region = openStackConfig.region[ regionName ]
      return buildTenantsModel( region )

BasicCli.addShowCommandClass( ShowOpenstackConfigRegionCmd )

#-------------------------------------------------------------------------------
# The 'show openstack config region <RegionName> timestamp'
# command.
# Returns the agent's UUID.
# Note: This command used to return the timestamp at which any entity in the
# region was modified.
#-------------------------------------------------------------------------------
class ShowOpenstackConfigRegionTimestampCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack config region REGION_NAME timestamp'
   data = {
      'openstack' : matcherOpenstack,
      'config' : matcherConfig,
      'region' : matcherRegion,
      'REGION_NAME' : regionNameMatcher,
      'timestamp' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'timestamp',
            helpdesc='Display timestamp' ),
         guard=osConfigGuard
      ),
   }
   cliModel = OpenStackModels.RegionTimestamp
   hidden = True

   @staticmethod
   def handler( mode, args ):
      modelRegionTimestamp = OpenStackModels.RegionTimestamp()
      modelRegionTimestamp.regionName = args[ 'REGION_NAME' ]
      modelRegionTimestamp.regionTimestamp = openStackAgentStatus.agentUuid
      return modelRegionTimestamp

BasicCli.addShowCommandClass( ShowOpenstackConfigRegionTimestampCmd )

#-------------------------------------------------------------------------------
# The 'show openstack features'
# hidden command.
# Returns list of supported openstack features in CVX.
#-------------------------------------------------------------------------------
class ShowOpenstackFeaturesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack features'
   data = {
      'openstack' : matcherOpenstack,
      'features' : 'Show openstack features',
   }

   cliModel = OpenStackModels.OpenStackFeatures
   hidden = True

   @staticmethod
   def handler( mode, args ):
      # The key/value in the features dictionary is feature name/version
      features = { 'hierarchical-port-binding' : 1,
                   'baremetal-and-dvr' : 1,
                   'json-api' : 1 }
      return OpenStackModels.OpenStackFeatures( features=features )

BasicCli.addShowCommandClass( ShowOpenstackFeaturesCmd )

#-------------------------------------------------------------------------------
# The 'show openstack agent uuid'
# hidden command.
# Returns the uuid of the agent. This id changes everytime the agent restarts.
#-------------------------------------------------------------------------------
class ShowOpenstackAgentUuidCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack agent uuid'
   data = {
      'openstack' : matcherOpenstack,
      'agent' : 'Show openstack agent status',
      'uuid' : nodeUuid,
   }

   cliModel = OpenStackModels.AgentUuid
   hidden = True

   @staticmethod
   def handler( mode, args ):
      modelAgentUuid = OpenStackModels.AgentUuid()
      modelAgentUuid.uuid = openStackAgentStatus.agentUuid
      return modelAgentUuid

BasicCli.addShowCommandClass( ShowOpenstackAgentUuidCmd )

#-------------------------------------------------------------------------------
# The 'show openstack resource-pool vlan region <region> uuid' hidden command.
#
# Returns the uuid of the vlan assignment. This id changes everytime the vlan
# assignment is altered.
#-------------------------------------------------------------------------------
class ShowOpenstackResourcePoolVlanRegionUuidCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack resource-pool vlan region REGION_NAME uuid'
   data = {
      'openstack' : matcherOpenstack,
      'resource-pool' : 'Show openstack resource pool status',
      'vlan' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'vlan',
            helpdesc='Show openstack VLAN assignment uuid' ),
         guard=osConfigGuard ),
      'region' : matcherRegion,
      'REGION_NAME' : regionNameMatcher,
      'uuid' : nodeUuid,
   }

   cliModel = OpenStackModels.VlanAssignmentUuid
   hidden = True

   @staticmethod
   def handler( mode, args ):
      regionName = args[ 'REGION_NAME' ]
      modelVlanAssignmentUuid = OpenStackModels.VlanAssignmentUuid()
      if regionName in openStackStatus.regionStatus.keys():
         modelVlanAssignmentUuid.uuid = \
             openStackStatus.regionStatus[ regionName ].vlanAssignmentUuid
      else:
         modelVlanAssignmentUuid.uuid = ''
      return modelVlanAssignmentUuid

BasicCli.addShowCommandClass( ShowOpenstackResourcePoolVlanRegionUuidCmd )

#-------------------------------------------------------------------------------
# The 'show openstack config all-regions'
# hidden command. Used by the tests to get all the openstack data stored on CVX
#-------------------------------------------------------------------------------
class ShowOpenstackConfigAllRegionsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack config all-regions'
   data = {
      'openstack' : matcherOpenstack,
      'config' : matcherConfig,
      'all-regions' : 'Dump configuration of all regions',
   }
   cliModel = OpenStackModels.RegionsConfig
   hidden = True

   @staticmethod
   def handler( mode, args ):
      modelRegions = OpenStackModels.RegionsConfig()
      if not cdbMountsComplete:
         # We may have never mounted the config from controllerdb
         mode.addWarning( 'CVX not ready' )
         return modelRegions
      for regionName, region in openStackConfig.region.iteritems():
         modelRegion = OpenStackModels.RegionConfig()
         modelRegion.regionName = regionName
         tenantsConfig = buildTenantsModel( region )
         for tenantId in tenantsConfig.tenants:
            modelRegion.tenants[ tenantId ] = tenantsConfig.tenants[ tenantId ]
         modelRegions.regions[ regionName ] = modelRegion
      return modelRegions

BasicCli.addShowCommandClass( ShowOpenstackConfigAllRegionsCmd )

#-------------------------------------------------------------------------------
# The 'show openstack config networks vni mapping'
# hidden command. Used by the tests to get all the openstack data stored on CVX
#-------------------------------------------------------------------------------
class ShowOpenstackConfigNetworksVniMappingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack config networks vni mapping'
   data = {
      'openstack' : matcherOpenstack,
      'config' : matcherConfig,
      'networks' : 'Openstack network config',
      'vni' : 'VLAN to VNI map',
      'mapping' : 'Show network mapping',
   }
   cliModel = OpenStackModels.RegionVlanToVniMap

   @staticmethod
   def handler( mode, args ):
      modelRegionVlanToVniMap = OpenStackModels.RegionVlanToVniMap()
      for cd, vniMap in virtualNetworkConfig.vlanToVniMap.items():
         if cd.client != 'OpenStack':
            continue
         modelVlanToVniMap = OpenStackModels.VlanToVniMap()
         for vlanRange, vniRange in vniMap.map.items():
            if vlanRange.start != vlanRange.end:
               vlr = "%d-%d" % ( vlanRange.start, vlanRange.end )
               vnr = "%d-%d" % ( vniRange.start, vniRange.end )
            else:
               vlr = "%d" % vlanRange.start
               vnr = "%d" % vniRange.start
            modelVlanToVniMap.mapping[ vlr ] = vnr

         modelRegionVlanToVniMap.vlanToVniMap[ cd.domain ] = modelVlanToVniMap

      return modelRegionVlanToVniMap

BasicCli.addShowCommandClass( ShowOpenstackConfigNetworksVniMappingCmd )

#-------------------------------------------------------------------------------
# The 'show openstack regions [REGION_NAME]'
# command.
#-------------------------------------------------------------------------------
class ShowOpenstackRegionsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack regions [ REGION_NAME ]'
   data = {
      'openstack' : matcherOpenstack,
      'regions' : 'Display regions',
      'REGION_NAME' : regionNameMatcher,
   }

   cliModel = OpenStackModels.RegionServiceEndPoints

   @staticmethod
   def handler( mode, args ):
      regionNameFilter = args.get( 'REGION_NAME' )

      modelRegionServiceEndPoints = OpenStackModels.RegionServiceEndPoints()
      if not cdbMountsComplete:
         # We may have never mounted the config from controllerdb
         mode.addWarning( 'CVX not ready' )
         return modelRegionServiceEndPoints
      if regionNameFilter and regionNameFilter not in openStackConfig.region:
         return modelRegionServiceEndPoints

      for region in openStackConfig.region.values():
         if regionNameFilter and regionNameFilter != region.name:
            continue
         modelRegion = OpenStackModels.Region( regionName=region.name )
         modelRegionServiceEndPoints.regions[ region.name ] = modelRegion
         SyncStatusEnum = Tac.Type( 'OpenStack::SyncStatus' )
         syncStatus = SyncStatusEnum.unknown
         if region.name in openStackStatus.regionStatus:
            syncStatus = openStackStatus.regionStatus[ region.name ].syncStatus
         syncInterval = openStackConfig.region[ region.name ].syncInterval
         modelRegion.syncStatus = syncStatus
         modelRegion.syncInterval = syncInterval

         # Get the configured credentials by CLI or Arista ML2 driver
         # Since the credentials configured by CLI take precedence over Arista
         # ML2 the CLI config is checked first.
         username = ''
         tenantname = ''
         if region.name in openStackCliConfig.credentials:
            username = openStackCliConfig.credentials[ region.name ].username
            tenantname = openStackCliConfig.credentials[ region.name ].tenantName
         elif region.serviceEndPoint and 'keystone' in region.serviceEndPoint:
            username = region.serviceEndPoint[ 'keystone' ].user
            tenantname = region.serviceEndPoint[ 'keystone' ].tenant

         # Get the configured authentication URL by CLI or Arista ML2 driver
         # Since the URL configured by CLI takes precedence over Arista ML2
         # driver the CLI config is checked first.
         authUrl = ''
         if region.name in openStackCliConfig.keystoneAuthUrl:
            keystoneAuthUrl = openStackCliConfig.keystoneAuthUrl[ region.name ]
            authUrl = keystoneAuthUrl.authUrl
         elif region.serviceEndPoint and 'keystone' in region.serviceEndPoint:
            authUrl = region.serviceEndPoint[ 'keystone' ].authUrl

         serviceEndPoint = OpenStackModels.ServiceEndPoint(
            name='keystone',
            authUrl=authUrl,
            username=username,
            tenantname=tenantname
         )
         modelRegion.serviceEndPoints[ 'keystone' ] = serviceEndPoint
         if nameResolutionStatus.region.has_key( region.name ):
            novaUrl = nameResolutionStatus.region[ region.name ].novaUrl
            neutronUrl = nameResolutionStatus.region[ region.name ].neutronUrl
            if novaUrl != '':
               serviceEndPoint = OpenStackModels.ServiceEndPoint(
                  name='nova',
                  authUrl=novaUrl
               )
               modelRegion.serviceEndPoints[ 'nova' ] = serviceEndPoint
            if neutronUrl != '':
               serviceEndPoint = OpenStackModels.ServiceEndPoint(
                  name='neutron',
                  authUrl=neutronUrl
               )
               modelRegion.serviceEndPoints[ 'neutron' ] = serviceEndPoint

      return modelRegionServiceEndPoints

BasicCli.addShowCommandClass( ShowOpenstackRegionsCmd )

#-------------------------------------------------------------------------------
# show openstack tenants [ region REGION_NAME ]
#-------------------------------------------------------------------------------
class ShowOpenstackTenantsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack tenants [ RF ]'
   data = {
      'openstack' : matcherOpenstack,
      'tenants' : 'Display tenants',
      'RF' : _ShowRegionFilterExpr,
   }
   cliModel = OpenStackModels.RegionTenants

   @staticmethod
   def handler( mode, args ):
      regionNameFilter = args.get( 'REGION_NAME' )

      modelRegionTenants = OpenStackModels.RegionTenants()
      if not cdbMountsComplete:
         # We may have never mounted the config from controllerdb
         mode.addWarning( 'CVX not ready' )
         return modelRegionTenants

      if regionNameFilter and regionNameFilter not in openStackConfig.region:
         return modelRegionTenants

      # Set this to true when a network is added to the model.
      # It is possible that no tenants match the filters specified and this
      # prevents just headers from being printed.
      genOutput = False
      for region in openStackConfig.region.values():
         if regionNameFilter:
            # Filter was set for region name
            if regionNameFilter == region.name:
               # If the region has already been processed then we are done.
               if len( modelRegionTenants.regions ):
                  break
            else:
               # Miss on filter
               continue

         modelRegion = OpenStackModels.Region( regionName=region.name )
         modelRegionTenants.regions[ region.name ] = modelRegion
         for tenant in region.tenant.values():
            tenantName = getTenantName( region.name, tenant.id )
            modelTenant = OpenStackModels.Tenant(
               tenantId=tenant.id,
               tenantName=tenantName
            )
            modelRegion.tenants[ tenant.id ] = modelTenant
            genOutput = True

      if not genOutput:
         # Return an empty model
         modelRegionTenants = OpenStackModels.RegionTenants()

      return modelRegionTenants

BasicCli.addShowCommandClass( ShowOpenstackTenantsCmd )

#-------------------------------------------------------------------------------
# show openstack networks
#    [ { ( network NETWORK_NAME_OR_ID ) | ( region REGION_NAME ) |
#        ( tenant TENANT_NAME_OR_ID ) } ]
#    [ detail ]
#-------------------------------------------------------------------------------
def populateTopology( model ):
   model.detail = True
   networkElementCollection = virtualNetworkStatus.networkElementCollection
   if networkElementCollection != None:
      for host in networkElementCollection.networkElement.values():
         # Populate topology edges
         for portName, port in sorted( host.physicalInterface.items() ):
            e = NetworkTopologyModels.DirectedEdge()
            e.fromPort = NetworkTopologyModels.Port(
               name=portName,
               hostid=host.name,
               hostname=host.hostname
            )
            pc = None
            if port.portGroup:
               pcName = port.portGroup.intfName
               if '%s-%s' % ( host, pcName ) in model.neighbors:
                  pc = model.neighbors[ '%s-%s' % ( host, pcName ) ]
               else:
                  pc = NetworkTopologyModels.DirectedEdge()
                  pc.fromPort = NetworkTopologyModels.Port(
                     name=pcName,
                     hostid=host.name,
                     hostname=host.hostname
                  )
            # Maintain a list of connected hosts in the global topology view for use
            # below
            connectedHosts = list()
            if globalTopoStatus != None and \
               host.name in globalTopoStatus.host.keys():
               topoHost = globalTopoStatus.host[ host.name ]
               if portName in topoHost.port:
                  edge = globalTopoStatus.edge.get( topoHost.port[ portName ] )
                  if edge:
                     for connectedPort in edge.toPort.keys():
                        connectedHosts.append( connectedPort.host().hostname )
                        toPort = NetworkTopologyModels.Port(
                           name=connectedPort.name,
                           hostid=connectedPort.host().name,
                           hostname=connectedPort.host().hostname
                        )
                        e.toPort.append( toPort )
                        if pc and toPort not in pc.toPort:
                           pc.toPort.append( toPort )

            # Go through the edges in the openstack topology that aren't in the
            # global topology. These edges are unidirectional, since we have no
            # view of baremetal elements' interfaces, so create artificial
            # baremetal -> switch connections so that render functions can easily
            # determine which switches a baremetal host is connected to
            for networkElement in port.configuredElement.values():
               if networkElement.hostname in connectedHosts:
                  continue
               baremetalPortCount = 1
               baremetalPortName = 'BM-port%d' % baremetalPortCount
               modelNeighborId = '%s-%s' % ( networkElement.hostname,
                                             baremetalPortName )
               # Find an unused baremetal port
               while modelNeighborId in model.neighbors:
                  baremetalPortCount += 1
                  baremetalPortName = 'BM-port%d' % baremetalPortCount
                  modelNeighborId = '%s-%s' % ( networkElement.hostname,
                                                baremetalPortName )
               toPort = NetworkTopologyModels.Port(
                  name=baremetalPortName,
                  hostid=NetworkTopologyModels.getHostName( networkElement.name ),
                  hostname=networkElement.hostname
               )
               e.toPort.append( toPort )
               if pc and toPort not in pc.toPort:
                  pc.toPort.append( toPort )
               otherEdge = NetworkTopologyModels.DirectedEdge()
               otherEdge.fromPort = NetworkTopologyModels.Port(
                  name=baremetalPortName,
                  hostid=NetworkTopologyModels.getHostName( networkElement.name ),
                  hostname=networkElement.hostname
               )
               otherEdge.toPort.append(
                  NetworkTopologyModels.Port(
                     name=portName,
                     hostid=host.name,
                     hostname=host.hostname
                  )
               )
               model.neighbors[ modelNeighborId ] = otherEdge

               # Create an artificial baremetal interface on the baremetal element
               # to match the edge
               if networkElement.hostname not in model.hosts:
                  h = NetworkTopologyModels.Host(
                     name=NetworkTopologyModels.getHostName( networkElement.name ),
                     hostname=networkElement.hostname
                  )
               else:
                  h = model.hosts[ networkElement.hostname ]
               h.port[ baremetalPortName ] = NetworkTopologyModels.Port(
                  name=baremetalPortName,
                  hostid=NetworkTopologyModels.getHostName( networkElement.name ),
                  hostname=networkElement.hostname
               )
               model.hosts[ networkElement.hostname ] = h

            model.neighbors[ '%s-%s' %
                             ( host.hostname, portName ) ] = e
            if pc:
               model.neighbors[ '%s-%s' %
                                ( host.hostname, pcName ) ] = pc

         # Populate topology hosts
         if host.hostname not in model.hosts:
            h = NetworkTopologyModels.Host(
               name=NetworkTopologyModels.getHostName( host.name ),
               hostname=host.hostname
            )
         else:
            h = model.hosts[ host.hostname ]
         for intf in host.physicalInterface.values():
            h.port[ intf.intfName ] = NetworkTopologyModels.Port(
               name=intf.intfName,
               hostid=host.name,
               hostname=host.hostname
            )

         model.hosts[ host.hostname ] = h

def networkMatchesFilter( network, networkFilter ):
   if network is None:
      return False

   if networkFilter:
      # Filter was set for network name|id
      if networkFilter != network.networkName and \
            networkFilter != network.id:
         return False
   return True

def portMatchesFilter( port, portFilter ):
   if portFilter:
      # Filter was set for port name or id
      if ( portFilter != port.portName and
          portFilter != port.id ):
         return False
   return True

def buildNetworkModel( regionName, network ):
   if network is None or not network.staticSegment.values():
      return None

   mappedVni = 0
   segment = network.staticSegment.values()[ 0 ]
   clientDomain = ClientDomain( VNAClients.OpenStack, regionName )
   if segment.type == 'vlan':
      mappedVni = virtualNetworkConfig.vniForVlan( clientDomain,
                     segment.segmentationId )

   shared = network.shared
   modelNetwork = OpenStackModels.Network(
      networkId=network.id,
      networkName=getNetworkName( network.networkName ),
      segmentationType=segment.type,
      segmentationTypeId=segment.segmentationId,
      mappedVni=mappedVni,
      shared=shared
   )
   return modelNetwork

def buildSegmentConfigModel( region, portId ):
   segmentList = []
   if region and portId in region.portBinding.keys():
      portBinding = region.portBinding[ portId ]
      for binding in portBinding.portToSwitchInterfaceBinding.values():
         for _, segment in sorted( binding.segment.iteritems() ):
            modelSegment = OpenStackModels.SegmentConfig(
               segmentId=segment.id,
               segmentationType=segment.type,
               segmentationTypeId=segment.segmentationId,
               networkId=segment.networkId
            )
            if modelSegment not in segmentList:
               segmentList.append( modelSegment )
      if segmentList:
         return segmentList

      for binding in portBinding.portToHostBinding.values():
         for _, segment in sorted( binding.segment.iteritems() ):
            modelSegment = OpenStackModels.SegmentConfig(
               segmentId=segment.id,
               segmentationType=segment.type,
               segmentationTypeId=segment.segmentationId,
               networkId=segment.networkId
            )
            if modelSegment not in segmentList:
               segmentList.append( modelSegment )
   return segmentList

def getInstanceIdModelType( instanceType ):
   typeToInstanceIdModelTypeMap = {
      'virtual' : OpenStackModels.VmInstanceIdName,
      'baremetal' : OpenStackModels.BaremetalInstanceIdName,
      'dhcp' : OpenStackModels.DhcpInstance
   }
   return typeToInstanceIdModelTypeMap[ instanceType ]

def getModelNetworkInstances( modelNetwork, instanceType ):
   typeToInstanceIdTypeMap = { 'virtual' : 'networkVmInstancesIdName',
                               'baremetal' : 'networkBaremetalInstancesIdName',
                               'dhcp' : 'dhcpInstances' }
   return getattr( modelNetwork, typeToInstanceIdTypeMap[ instanceType ] )

def doShowOpenStackNetworks( mode, args ):
   regionNameFilter = args.get( 'REGION_NAME' )
   tenantNameOrIdFilter = args.get( 'TENANT_NAME_OR_ID' )
   networkNameOrIdFilter = args.get( 'NETWORK_NAME_OR_ID' )
   detail = 'detail' in args

   modelTenantNetworks = OpenStackModels.TenantNetworks()
   if not cdbMountsComplete:
      # We may have never mounted the config from controllerdb
      mode.addWarning( 'CVX not ready' )
      return modelTenantNetworks

   if regionNameFilter and regionNameFilter not in openStackConfig.region:
      return modelTenantNetworks

   # Set this to true when a network is added to the model.
   # It is possible that no networks match the filters specified and this
   # prevents just headers from being printed.
   genOutput = False

   for region in openStackConfig.region.values():
      if regionNameFilter:
         # Filter was set for region name
         if regionNameFilter != region.name:
            continue

      modelRegion = OpenStackModels.Region( regionName=region.name )
      modelTenantNetworks.regions[ region.name ] = modelRegion

      for network in region.network.values():
         if network.tenant is None:
            continue
         # Check for network filter
         if networkNameOrIdFilter:
            if not networkMatchesFilter( network, networkNameOrIdFilter ):
               continue

         # Check for tenant filter
         if tenantNameOrIdFilter:
            tenantName = getTenantName( region.name, network.tenant.id )
            if ( tenantNameOrIdFilter != tenantName and
                tenantNameOrIdFilter != network.tenant.id ):
               continue

         if network.tenant.id not in modelRegion.tenants:
            tenantName = getTenantName( region.name, network.tenant.id )
            modelTenant = OpenStackModels.Tenant(
               tenantId=network.tenant.id,
               tenantName=tenantName
            )
            modelRegion.tenants[ network.tenant.id ] = modelTenant

         networkModel = buildNetworkModel( region.name, network )
         if not networkModel:
            continue

         genOutput = True
         modelTenant = modelRegion.tenants[ network.tenant.id ]
         modelNetworks = modelTenant.tenantNetworks
         modelNetworks[ network.id ] = networkModel

      # If the detailed ouput is required, go through all the VM instances and
      # build their models. One possible race condition is when the network is
      # deleted after the network model is built but before the VM models.
      # In that case, the VM port associated with the network is deleted and
      # hence the VM will not appear in the output.
      if detail:
         for instanceType in [ 'virtual', 'baremetal', 'dhcp' ]:
            instances = getRegionInstances( region, instanceType )
            modelType = getInstanceIdModelType( instanceType )

            for instance in instances.values():
               modelInstance = modelType()
               instanceId = getInstanceId( instance, instanceType )

               instanceName = getInstanceName( region.name, instance.tenant.id,
                                               instanceId )
               instanceHostId = getInstanceHostId( instance, instanceType )
               modelInstance.setInstanceId( instanceId )
               modelInstance.setInstanceName( instanceName )
               modelInstance.setInstanceHostId( instanceHostId )

               for port in instance.port.values():
                  if ( not networkNameOrIdFilter or
                      networkMatchesFilter( port.network, networkNameOrIdFilter ) ):
                     if port.network is None or port.network.tenant is None:
                        continue
                     if port.network.id not in region.network.keys():
                        continue
                     if port.network.tenant.id not in modelRegion.tenants:
                        continue
                     modelTenant = modelRegion.tenants[ port.network.tenant.id ]
                     if port.network.id not in modelTenant.tenantNetworks:
                        continue
                     modelNetwork = modelTenant.tenantNetworks[ port.network.id ]

                     networkModelInstances = getModelNetworkInstances(
                        modelNetwork, instanceType )
                     networkModelInstances[ instanceId ] = modelInstance
                     # DHCP must be handled separately here since we print out
                     # all ports for a DHCP instance in 'show openstack networks
                     # detail' (we assume other instances only have a single host)
                     if instanceType == 'dhcp':
                        segments = buildSegmentConfigModel( region, port.id )
                        modelPorts = modelInstance.dhcpPorts
                        modelPorts[ port.id ] = \
                           OpenStackModels.Port(
                              portId=port.id,
                              portName=getPortName( port.portName ),
                              networkId=port.network.id,
                              segments=segments )
                     else:
                        modelSwitchports = modelInstance.switches
                        if port.id not in region.portBinding.keys():
                           continue
                        portBinding = region.portBinding[ port.id ]
                        for binding in \
                              portBinding.portToSwitchInterfaceBinding.values():
                           switchport = binding.switchInterface
                           if switchport.switchId not in modelSwitchports:
                              modelSwitchports[ switchport.switchId ] = (
                                 OpenStackModels.SwitchportList()
                              )
                           modelSwitchports[
                              switchport.switchId ].switchports.append(
                                 switchport.interface )

   if genOutput and detail:
      populateTopology( modelTenantNetworks )

   if not genOutput:
      # Return an empty model
      modelTenantNetworks = OpenStackModels.TenantNetworks()

   return modelTenantNetworks

class ShowOpenstackNetworksCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack networks [ { NF | RF | TF } ] [ detail ]'
   data = {
      'openstack' : matcherOpenstack,
      'networks' : 'Display tenant networks',
      'NF' : _ShowNetworkFilterExpr,
      'RF' : _ShowRegionFilterExpr,
      'TF' : _ShowTenantFilterExpr,
      'detail' : matcherDetail,
   }
   handler = doShowOpenStackNetworks
   cliModel = OpenStackModels.TenantNetworks

BasicCli.addShowCommandClass( ShowOpenstackNetworksCmd )

#-------------------------------------------------------------------------------
# The'show openstack resource-pools [region RegionName]'
# Used to view the resource-pools assigned to OpenStack regions.
#-------------------------------------------------------------------------------

def vlanListToString( vlanList ):
   return coalesceList( sorted( vlanList ) )

class ShowOpenstackResourcePoolsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack resource-pools [ region REGION_NAME ]'
   data = {
      'openstack' : matcherOpenstack,
      'resource-pools' : 'Show VLAN ID assignements',
      'region' : 'Filter by region',
      'REGION_NAME' : regionNameMatcher,
   }

   cliModel = OpenStackModels.RegionPhysicalNetwork

   @staticmethod
   def handler( mode, args ):
      regionFilter = args.get( 'REGION_NAME' )
      modelRegionVlanPool = OpenStackModels.RegionPhysicalNetwork()
      for regionName in openStackCliConfig.vlanAssignment.keys():
         if regionFilter and regionFilter != regionName:
            continue
         modelVlanPool = OpenStackModels.VlanPool()
         defaultPhysNet = OpenStackModels.PhysicalNetworkVlanPool()
         modelVlanPool.assignedVlans = vlanListToString(
            openStackCliConfig.vlanAssignment[ regionName ].vlan )
         if regionName in openStackStatus.regionStatus.keys():
            modelVlanPool.allocatedVlans = vlanListToString(
               openStackStatus.regionStatus[ regionName ].vlanPool.allocatedVlan )
            modelVlanPool.availableVlans = vlanListToString(
               openStackStatus.regionStatus[ regionName ].vlanPool.availableVlan )
         else:
            modelVlanPool.allocatedVlans = ''
            modelVlanPool.availableVlans = ''
         defaultPhysNet.vlanPool[ 'default' ] = modelVlanPool
         modelRegionVlanPool.physicalNetwork[ regionName ] = defaultPhysNet
      return modelRegionVlanPool

BasicCli.addShowCommandClass( ShowOpenstackResourcePoolsCmd )

#-------------------------------------------------------------------------------
# The 'show openstack vms [region RegionName] [tenant TenantName|TenantId]
# [network NetworkName|NetworkId] [host HostName] [vm VmName|vmId] [detail]'
# command and the 'show 'show openstack instances [region RegionName]
# [tenant TenantName|TenantId] [network NetworkName|NetworkId] [host HostName]
# [instance InstanceName|instanceId] [type baremetal|virtual] [detail]' command.
#
# All options can be applied in any order. Detail is the last option.
#-------------------------------------------------------------------------------

def getInstanceModelType( instanceType ):
   typeToInstanceModelTypeMap = {
      'virtual' : OpenStackModels.VmInstance,
      'baremetal' : OpenStackModels.BaremetalInstance,
      'router' : OpenStackModels.RouterInstance,
      'dhcp' : OpenStackModels. DhcpInstance
   }
   return typeToInstanceModelTypeMap[ instanceType ]

def populateOpenStackInstances( mode, args, modelInstanceType ):
   instanceNameOrIdFilter = args.get( 'INSTANCE_NAME_OR_ID' )
   regionNameFilter = args.get( 'REGION_NAME' )
   tenantNameOrIdFilter = args.get( 'TENANT_NAME_OR_ID' )
   networkNameOrIdFilter = args.get( 'NETWORK_NAME_OR_ID' )
   hostNameFilter = args.get( 'HOSTNAME' )
   instanceTypeFilter = args.get( 'INSTANCE_TYPE' )
   detail = 'detail' in args

   modelInstances = modelInstanceType()
   if not cdbMountsComplete:
      # We may have never mounted the config from controllerdb
      mode.addWarning( 'CVX not ready' )
      return modelInstances

   if regionNameFilter and regionNameFilter not in openStackConfig.region:
      return modelInstances

   # Set this to true when a VM is added to the model.
   # It is possible that no VMs match the filters specified and this prevents
   # just headers from being printed.
   genOutput = False

   for region in openStackConfig.region.values():
      if regionNameFilter:
         # Filter was set for region name
         if regionNameFilter != region.name:
            continue

      finishSearching = False

      modelRegion = OpenStackModels.Region( regionName=region.name )
      modelInstances.regions[ region.name ] = modelRegion
      for tenant in region.tenant.values():
         tenantName = getTenantName( region.name, tenant.id )
         if tenantNameOrIdFilter:
            # Filter was set for tenant name|id
            if tenantNameOrIdFilter != tenantName and \
               tenantNameOrIdFilter != tenant.id:
               continue
            if tenantNameOrIdFilter == tenant.id:
               finishSearching = True

         modelTenant = OpenStackModels.Tenant(
            tenantId=tenant.id,
            tenantName=tenantName
         )
         modelRegion.tenants[ tenant.id ] = modelTenant

         # Tenant Loop
         if finishSearching:
            break

      for instanceType in [ 'virtual', 'baremetal', 'router' ]:
         if not instanceTypeFilter or instanceTypeFilter == instanceType:
            instances = getRegionInstances( region, instanceType )
            modelType = getInstanceModelType( instanceType )
            for instance in instances.values():
               if ( instance.tenant is None or
                   instance.tenant.id not in modelRegion.tenants ):
                  continue
               modelTenant = modelRegion.tenants[ instance.tenant.id ]
               tenantModelInstances = getModelTenantInstances( modelTenant,
                                                               instanceType )
               instanceId = getInstanceId( instance, instanceType )
               instanceName = getInstanceName( region.name, instance.tenant.id,
                                               instanceId )
               instanceHostId = getInstanceHostId( instance, instanceType )

               if instanceNameOrIdFilter:
                  # Filter was set for instance name|id
                  if instanceNameOrIdFilter != instanceName and \
                        instanceNameOrIdFilter != instanceId:
                     continue

               if hostNameFilter:
                  # Filter was set for host name
                  if hostNameFilter != instanceHostId:
                     continue

               modelInstance = modelType()
               modelInstance.setInstanceId( instanceId )
               modelInstance.setInstanceName( instanceName )
               modelInstance.setInstanceHostId( instanceHostId )
               tenantModelInstances[ instanceId ] = modelInstance
               modelPorts = modelInstance.getInstancePorts()

               for port in instance.port.values():
                  network = port.network
                  if networkNameOrIdFilter:
                     # Filter was set for network name|id
                     if ( network is None or
                          networkNameOrIdFilter != network.networkName and
                          networkNameOrIdFilter != network.id ):
                        continue

                  # TODO: Enhance show commands to show network segments
                  # Support single segment for now
                  networkModel = buildNetworkModel( region.name, network )
                  if networkModel:
                     modelNetworks = modelTenant.tenantNetworks
                     modelNetworks[ network.id ] = networkModel

                  modelSwitchports = dict()
                  if port.id not in region.portBinding.keys():
                     continue
                  portBinding = region.portBinding[ port.id ]
                  bindings = portBinding.portToSwitchInterfaceBinding.values()
                  for binding in bindings:
                     switchport = binding.switchInterface
                     if switchport.switchId not in modelSwitchports:
                        modelSwitchports[ switchport.switchId ] = (
                            OpenStackModels.SwitchportList()
                        )
                     modelSwitchports[ switchport.switchId ].switchports.append(
                        switchport.interface )

                  hosts = portBinding.portToHostBinding.keys()
                  segments = buildSegmentConfigModel( region, port.id )
                  modelPorts[ port.id ] = OpenStackModels.Port(
                        portId=port.id,
                        portName=port.portName,
                        networkId=port.network.id if network else '',
                        portVlanType=port.portVlanType,
                        hosts=hosts,
                        switches=modelSwitchports,
                        segments=segments )

                  genOutput = True

               if instanceNameOrIdFilter == instanceId:
                  finishSearching = True
                  break

            # Instance Type Loop
            if finishSearching:
               break

      # Region Loop
      if finishSearching:
         break

   if genOutput and detail:
      populateTopology( modelInstances )

   if not genOutput:
      # Return an empty model
      modelInstances = modelInstanceType()

   return modelInstances

class ShowOpenstackVmsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show openstack vms '
              '[ { HF | NF | RF | TF | VF } ] '
              '[ detail ]' )
   data = {
      'openstack' : matcherOpenstack,
      'vms' : 'Display one or more VM instances',
      'HF' : _ShowHostFilterExpr,
      'NF' : _ShowNetworkFilterExpr,
      'RF' : _ShowRegionFilterExpr,
      'TF' : _ShowTenantFilterExpr,
      'VF' : _ShowVmFilterExpr,
      'detail' : matcherDetail,
   }
   cliModel = OpenStackModels.Vms

   @staticmethod
   def handler( mode, args ):
      args[ 'INSTANCE_TYPE' ] = 'virtual'
      modelInstanceType = OpenStackModels.Vms
      return populateOpenStackInstances( mode, args, modelInstanceType )

BasicCli.addShowCommandClass( ShowOpenstackVmsCmd )

class ShowOpenstackInstancesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show openstack instances '
              '[ { HF | NF | RF | TF | IF | TYPEF } ] '
              '[ detail ]' )
   data = {
      'openstack' : matcherOpenstack,
      'instances' : 'Display one or more instances',
      'HF' : _ShowHostFilterExpr,
      'NF' : _ShowNetworkFilterExpr,
      'RF' : _ShowRegionFilterExpr,
      'TF' : _ShowTenantFilterExpr,
      'IF' : _ShowInstanceFilterExpr,
      'TYPEF' : _ShowInstanceTypeFilterExpr,
      'detail' : matcherDetail,
   }
   cliModel = OpenStackModels.Instances
   hidden = True

   @staticmethod
   def handler( mode, args ):
      modelInstanceType = OpenStackModels.Instances
      return populateOpenStackInstances( mode, args, modelInstanceType )

BasicCli.addShowCommandClass( ShowOpenstackInstancesCmd )

#-------------------------------------------------------------------------------
# show openstack vlans
#-------------------------------------------------------------------------------
class ShowOpenstackVlansCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack vlans'
   data = {
      'openstack' : matcherOpenstack,
      'vlans' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'vlans',
            helpdesc='OpenStack Auto-Vlan Provisioning' ),
         guard=ControllerdbLib.controllerGuard ),
   }
   cliModel = OpenStackModels.VlanInfo
   hidden = True

   @staticmethod
   def handler( mode, args ):
      vlanInfo = OpenStackModels.VlanInfo()
      if portVlanConfigDir and globalTopoStatus:
         for switch in portVlanConfigDir:
            switchMac = switch.replace( '-', ':' )
            if switchMac not in globalTopoStatus.host:
               continue
            switchHost = globalTopoStatus.host[ switchMac ]
            switchModel = OpenStackModels.Switch(
               switchId=switchMac,
               hostname=switchHost.hostname
            )
            vlanInfo.switches[ switchMac ] = switchModel
            for intf in portVlanConfigDir[ switch ].portAllowedVlanList:
               switchPort = None
               if intf in switchHost.port:
                  switchPort = switchHost.port[ intf ]
               if not switchPort or not switchPort.portGroup:
                  # Do not include the port if it is part of a port channel.
                  allowedVlans = (
                     portVlanConfigDir[ switch ].portAllowedVlanList[ intf ].vlans
                  )
                  switchModel.interfaces[ intf ] = OpenStackModels.Interface(
                     name=intf,
                     allowedVlans=allowedVlans
                  )

      return vlanInfo

BasicCli.addShowCommandClass( ShowOpenstackVlansCmd )

#-------------------------------------------------------------------------------
# show openstack ip access-list [<acl-name>] [summary] [ dynamic ] [ detail ]
# show openstack ipv6 access-list [<acl-name>] [summary] [ dynamic ] [ detail ]
#-------------------------------------------------------------------------------
class ShowOpenstackAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack ( ip | ipv6 ) access-list [ ACLNAME ]'
   data = {
      'openstack' : matcherOpenstack,
      'ip' : IraIpCli.ipMatcherForShow,
      'ipv6' : IraIp6Cli.ipv6MatcherForShow,
      'access-list' : AclCli.accessListKwMatcherForServiceAcl,
      'ACLNAME' : AclCli.ipOrIpv6AclNameExpression,
   }
   cliModel = AclCliModel.AllAclList

   @staticmethod
   def handler( mode, args ):
      return AclCli.showServiceAcl(
         mode,
         openStackCliConfig.serviceAclTypeVrfMap,
         osApiStatus.aclStatus,
         aclCheckpoint,
         'ipv6' if 'ipv6' in args else 'ip',
         args[ '<aclNameExpr>' ],
         supressVrf=True
      )

BasicCli.addShowCommandClass( ShowOpenstackAccessListCmd )

# -----------------------------------------------------------------------------
# The '[ no | default ] ( ip | ipv6 ) access-group ACL [ in ]' command
# -----------------------------------------------------------------------------
class OpenStackIpAcl( CliCommand.CliCommandClass ):
   syntax = '( ip | ipv6 ) access-group ACL [ in ]'
   noOrDefaultSyntax = '( ip | ipv6 ) access-group'
   data = {
         "ip" : AclCli.ipKwForServiceAclMatcher,
         "ipv6" : AclCli.ipv6KwMatcherForServiceAcl,
         "access-group" : AclCli.accessGroupKwMatcher,
         "ACL" : AclCli.standardIpAclNameMatcher,
         "in" : AclCli.inKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      aclName = args.get( 'ACL', '' )
      key = Tac.Value(
         'Acl::AclTypeAndVrfName',
         'ip' if 'ip' in args else 'ipv6',
         IpLibConsts.DEFAULT_VRF
      )
      openStackCliConfig.serviceAclTypeVrfMap.aclName[ key ] = aclName

   noOrDefaultHandler = handler

OpenStackConfigMode.addCommandClass( OpenStackIpAcl )

#-------------------------------------------------------------------------------
# The 'name-resolution interval [INTERVAL]' and 'name-resolution force' command.
#-------------------------------------------------------------------------------
matcherNameResolution = CliMatcher.KeywordMatcher(
   'name-resolution',
   helpdesc='Configure openstack name resolution' )

class NameResolutionIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'name-resolution interval SECONDS'
   noOrDefaultSyntax = 'name-resolution interval ...'
   data = {
      'name-resolution' : matcherNameResolution,
      'interval' : 'Time interval in seconds between polling OpenStack for names',
      'SECONDS' : CliMatcher.IntegerMatcher(
         0,
         ONEDAY,
         helpdesc='Set the time interval in seconds between name updates, '
                  '0 to disable'
      )
   }

   @staticmethod
   def handler( mode, args ):
      nameResolutionInterval = args[ 'SECONDS' ]

      if int( nameResolutionInterval ) == 0:
         # Disable name resolution in OpenStackPoller. Therefore setting:
         # enabled = False : prevents OpenStackPollerSm from handling changes
         # in region refreshPeriod = Tac.endOfTime: prevents refresh handler
         # in ResolverSm from running
         nameResolutionConfig.enabled = False
         nameResolutionConfig.refreshPeriod = Tac.endOfTime
      else:
         nameResolutionConfig.enabled = True
         nameResolutionConfig.refreshPeriod = int( nameResolutionInterval )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nameResolutionConfig.refreshPeriod = nameResolutionConfig.defaultRefreshPeriod

OpenStackConfigMode.addCommandClass( NameResolutionIntervalCmd )

class NameResolutionForceCmd( CliCommand.CliCommandClass ):
   syntax = 'name-resolution force'
   data = {
      'name-resolution' : matcherNameResolution,
      'force' : 'Get the tenant and VM names from OpenStack immediately',
   }

   @staticmethod
   def handler( mode, args ):
      nameResolutionConfig.refreshRequest = Tac.now()

OpenStackConfigMode.addCommandClass( NameResolutionForceCmd )

#-------------------------------------------------------------------------------
# The 'show openstack physical-network <switch name> <interface intf> command
# These commands provide visibility into VMs attached to the switches
#-------------------------------------------------------------------------------

def buildOpenStackSwitchModel( switchNameOrId, interfaceName, vlanId ):
   assert portVlanConfigDir
   assert globalTopoStatus

   physicalNetwork = OpenStackModels.PhysicalNetwork()

   for switch in portVlanConfigDir:
      switchMac = switch.replace( '-', ':' )
      switchModel = None

      if switchMac in globalTopoStatus.host:
         hostname = globalTopoStatus.host[ switchMac ].hostname
      else:
         continue

      # Switch filter can either be switch name or system MAC id
      if not switchNameOrId or \
            switchNameOrId == switchMac or \
            switchNameOrId == hostname:
         switchModel = OpenStackModels.Switch(
            switchId=NetworkTopologyModels.getHostName( switchMac ),
            hostname=hostname
         )
      else:
         continue

      for intf in portVlanConfigDir[ switch ].portAllowedVlanList:
         # Ignore interfaces that do not match the filter
         if interfaceName and \
               str( interfaceName ).lower() != str( intf ).lower():
            continue

         allowedVlans = portVlanConfigDir[ switch ].portAllowedVlanList[ intf ].vlans

         if vlanId:
            if int( vlanId ) in Vlan.computeVlanRangeSet( allowedVlans ):
               allowedVlans = str( vlanId )
            else:
               # Ignore vlans that do not match the filter
               continue

         # Add the interface to the switch model
         switchModel.interfaces[ intf ] = OpenStackModels.Interface(
            name=intf,
            allowedVlans=allowedVlans
         )

         physicalNetwork.switches[ switchMac ] = switchModel

   return physicalNetwork

def doShowOpenStackPhysicalNetwork( mode, args ):
   physicalNetwork = OpenStackModels.PhysicalNetwork()

   switchNameOrId = args.get( 'SWITCH' )
   interfaceName = args.get( 'INTF' )
   hostName = args.get( 'HOSTNAME' )
   vlanId = args.get( 'VLAN_ID' )

   if interfaceName:
      interfaceName, valid = normalizeInterfaceName( interfaceName )
      if not valid:
         mode.addError( "Invalid interface name" )
         return physicalNetwork

   if not portVlanConfigDir or not globalTopoStatus:
      mode.addWarning( "Topology information is not available" )
      return physicalNetwork

   # Build a physical model of all the switches that are in the OpenStack
   # cluster
   physicalNetwork = buildOpenStackSwitchModel( switchNameOrId, interfaceName,
                                                vlanId )

   if not len( physicalNetwork.switches ):
      mode.addWarning( "Topology information is not available" )
      return physicalNetwork

   modelVms = OpenStackModels.Instances()
   if not cdbMountsComplete:
      # We may have never mounted the config from controllerdb
      mode.addWarning( 'CVX not ready' )
      return modelVms

   for region in openStackConfig.region.values():
      modelRegion = OpenStackModels.Region( regionName=region.name )
      modelVms.regions[ region.name ] = modelRegion
      for tenant in region.tenant.values():
         tenantName = getTenantName( region.name, tenant.id )
         modelTenant = OpenStackModels.Tenant(
            tenantId=tenant.id,
            tenantName=tenantName
         )
         modelRegion.tenants[ tenant.id ] = modelTenant
         modelNetworks = modelTenant.tenantNetworks

      for instanceType in [ 'virtual', 'baremetal', 'router' ]:

         instances = getRegionInstances( region, instanceType )
         modelType = getInstanceModelType( instanceType )
         for instance in instances.values():
            if instance.tenant is None:
               continue
            tenantId = instance.tenant.id
            if tenantId not in modelRegion.tenants:
               continue

            modelTenant = modelRegion.tenants[ tenantId ]
            tenantModelInstances = getModelTenantInstances( modelTenant,
                                                            instanceType )

            instanceId = getInstanceId( instance, instanceType )
            instanceName = getInstanceName( region.name, tenantId,
                                            instanceId )
            instanceHostId = getInstanceHostId( instance, instanceType )

            if hostName:
               # Filter was set for host name
               if hostName != instanceHostId:
                  continue

            modelInstance = modelType()
            modelInstance.setInstanceId( instanceId )
            modelInstance.setInstanceName( instanceName )
            modelInstance.setInstanceHostId( instanceHostId )
            modelPorts = modelInstance.getInstancePorts()

            for port in instance.port.values():
               network = port.network
               if network is None or not network.staticSegment.values():
                  continue

               # TODO: Enhance show commands to show network segments
               # Support single segment for now
               networkSegment = port.network.staticSegment.values()[ 0 ]
               # If the vlan filter is specified, ignore the VM ports which do
               # not match the filter
               if vlanId and \
                  ( networkSegment.type != 'vlan' or
                    ( networkSegment.type == 'vlan' and
                      int( networkSegment.segmentationId ) != int( vlanId ) ) ):
                  continue

               networkModel = buildNetworkModel( region.name, network )
               if networkModel:
                  modelNetworks = modelTenant.tenantNetworks
                  modelNetworks[ network.id ] = networkModel

               modelSwitchports = dict()
               if port.id not in region.portBinding.keys():
                  continue
               portBinding = region.portBinding[ port.id ]
               for binding in portBinding.portToSwitchInterfaceBinding.values():
                  switchport = binding.switchInterface
                  if switchport.switchId not in modelSwitchports:
                     modelSwitchports[ switchport.switchId ] = (
                        OpenStackModels.SwitchportList()
                     )
                  modelSwitchports[ switchport.switchId ].switchports.append(
                     switchport.interface )

               hosts = portBinding.portToHostBinding.keys()
               segments = buildSegmentConfigModel( region, port.id )
               modelPorts[ port.id ] = OpenStackModels.Port(
                  portId=port.id,
                  portName=port.portName,
                  networkId=port.network.id,
                  portVlanType=port.portVlanType,
                  hosts=hosts,
                  switches=modelSwitchports,
                  segments=segments )

            if len( modelPorts ):
               tenantModelInstances[ instanceId ] = modelInstance

      physicalNetwork.regions[ region.name ] = modelRegion

   populateTopology( physicalNetwork )
   return physicalNetwork

class ShowOpenstackPhysicalNetworkCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show openstack physical-network '
              '[ { SF | INTFF | HF | VF } ]'
            )
   data = {
      'openstack' : matcherOpenstack,
      'physical-network' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'physical-network',
            helpdesc='Display physical network used by OpenStack' ),
         guard=ControllerdbLib.controllerGuard ),
      'SF' : _ShowSwitchFilterExpr,
      'INTFF' : _ShowInterfaceFilterExpr,
      'HF' : _ShowHostFilterExpr,
      'VF' : _ShowVlanFilterExpr,
   }
   handler = doShowOpenStackPhysicalNetwork
   cliModel = OpenStackModels.PhysicalNetwork
   hidden = True

BasicCli.addShowCommandClass( ShowOpenstackPhysicalNetworkCmd )

def getTenantName( regionName, tenantId ):
   if ( regionName in nameResolutionStatus.region and
        tenantId in nameResolutionStatus.region[ regionName ].tenant and
        nameResolutionStatus.region[ regionName ].tenant[ tenantId ].tenantName ):
      return nameResolutionStatus.region[ regionName ].tenant[ tenantId ].tenantName
   return "Unknown"

def getInstanceName( regionName, tenantId, instanceId ):

   if regionName in nameResolutionStatus.region:
      region = nameResolutionStatus.region[ regionName ]

      if ( tenantId in region.tenant and
              instanceId in region.tenant[ tenantId ].instanceName and
              region.tenant[ tenantId ].instanceName
            ):
         return region.tenant[ tenantId ].instanceName[ instanceId ]
   return "Unknown"

def getNetworkName( networkName ):
   return "Unknown" if not networkName else networkName

def getPortName( portName ):
   return "Unknown" if not portName else portName

#-------------------------------------------------------------------------------
# show openstack grace-period
#-------------------------------------------------------------------------------
class ShowOpenstackGracePeriodCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack grace-period'
   data = {
      'openstack' : matcherOpenstack,
      'grace-period' : ( 'Show the grace period for which the OpenStack agent '
                        'waits for OpenStack region data' ),
   }
   cliModel = OpenStackModels.GracePeriod

   @staticmethod
   def handler( mode, args ):
      gp = OpenStackModels.GracePeriod()
      gp.gracePeriod = float( openStackCliConfig.gracePeriod )
      return gp

BasicCli.addShowCommandClass( ShowOpenstackGracePeriodCmd )

#-------------------------------------------------------------------------------
# show openstack sync-timeout
#-------------------------------------------------------------------------------
class ShowOpenstackSyncTimeoutCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openstack sync-timeout'
   data = {
      'openstack' : matcherOpenstack,
      'sync-timeout' : 'Show timeout value for a region sync',
   }
   cliModel = OpenStackModels.RegionSyncTimeout
   hidden = True

   @staticmethod
   def handler( mode, args ):
      regionSyncTimeout = OpenStackModels.RegionSyncTimeout()
      regionSyncTimeout.regionSyncTimeout = float( 30 )
      return regionSyncTimeout

BasicCli.addShowCommandClass( ShowOpenstackSyncTimeoutCmd )

#-------------------------------------------------------------------------------
# show openstack segments [
#    { ( host HOSTNAME ) | ( network NETWORK_NAME_OR_ID ) |
#      ( port INSTANCENAMEORIDFILTER ) | ( instance INSTANCENAMEORIDFILTER ) |
#      ( region REGION_NAME ) | ( switch SWITCH ) | ( tenant TENANT_NAME_OR_ID ) } ]
#-------------------------------------------------------------------------------
def doShowOpenStackSegments( mode, args ):
   #import rpdb;rpdb.set_trace()
   hostFilter = args.get( 'HOSTNAME' )
   instanceFilter = args.get( 'INSTANCE_NAME_OR_ID' )
   networkFilter = args.get( 'NETWORK_NAME_OR_ID' )
   portFilter = args.get( 'PORT_NAME_OR_ID' )
   regionFilter = args.get( 'REGION_NAME' )
   tenantFilter = args.get( 'TENANT_NAME_OR_ID' )
   switchFilter = args.get( 'SWITCH' )

   regionSegmentsModel = OpenStackModels.RegionOpenStackSegments()

   if not cdbMountsComplete:
      # We may have never mounted the config from controllerdb
      mode.addWarning( 'CVX not ready' )
      return regionSegmentsModel

   if switchFilter and hostFilter:
      mode.addWarning( "'host' and 'switch' filter cannot be used together" )
      return regionSegmentsModel

   for regionName, region in openStackConfig.region.iteritems():
      if regionFilter and regionFilter != regionName:
         continue

      # Get port from vmInstance, routerInstance, dhcpInstance and baremetal
      instanceModel = OpenStackModels.RegionInstanceBinding()
      for instanceType in [ 'virtual', 'baremetal', 'dhcp', 'router' ]:
         instances = getRegionInstances( region, instanceType )
         for instance in instances.values():
            if instance.tenant is None:
               continue
            tenantName = getTenantName( region.name, instance.tenant.id )
            if ( tenantFilter and tenantFilter != instance.tenant.id and
                  tenantFilter != tenantName ):
               continue

            instanceName = getInstanceName( region.name, instance.tenant.id,
                                            instance.id )
            if ( instanceFilter and instanceFilter != instance.id and
                  instanceFilter != instanceName ):
               continue

            instancePortBindingModel = OpenStackModels.InstancePortBinding()

            for port in instance.port.values():
               if not networkMatchesFilter( port.network, networkFilter ):
                  continue
               if not portMatchesFilter( port, portFilter ):
                  continue

               if port.id not in region.portBinding:
                  continue

               portBinding = region.portBinding[ port.id ]
               modelSwitchports = dict()
               for binding in portBinding.portToSwitchInterfaceBinding.values():
                  if hostFilter:
                     break

                  switchport = binding.switchInterface
                  if ( globalTopoStatus and
                      switchport.switchId in globalTopoStatus.host ):
                     switchHost = globalTopoStatus.host[ switchport.switchId ]
                     switchName = switchHost.hostname
                  else:
                     switchName = switchport.switchId
                  if ( switchFilter and switchFilter != switchName and
                      switchFilter != switchport.switchId ):
                     continue

                  networkSegmentModel = OpenStackModels.NetworkSegment()
                  switchInterface = ' - '.join(
                            ( switchName, switchport.interface ) )
                  for _, segment in sorted( binding.segment.iteritems() ):
                     segmentConfigModel = OpenStackModels.SegmentConfig(
                        segmentId=segment.id,
                        segmentationType=segment.type,
                        segmentationTypeId=segment.segmentationId,
                        networkId=segment.networkId
                     )
                     if segmentConfigModel not in networkSegmentModel.segments:
                        networkSegmentModel.networkId = port.network.id
                        networkSegmentModel.networkName = port.network.networkName
                        networkSegmentModel.segments.append( segmentConfigModel )
                  modelSwitchports[ switchInterface ] = networkSegmentModel

               modelHostports = dict()
               for binding in portBinding.portToHostBinding.values():
                  if switchFilter:
                     break
                  if hostFilter and hostFilter != binding.host:
                     continue
                  networkSegmentModel = OpenStackModels.NetworkSegment()
                  for _, segment in sorted( binding.segment.iteritems() ):
                     segmentConfigModel = OpenStackModels.SegmentConfig(
                        segmentId=segment.id,
                        segmentationType=segment.type,
                        segmentationTypeId=segment.segmentationId,
                        networkId=segment.networkId
                     )
                     if segmentConfigModel not in networkSegmentModel.segments:
                        networkSegmentModel.networkId = port.network.id
                        networkSegmentModel.networkName = port.network.networkName
                        networkSegmentModel.segments.append( segmentConfigModel )
                  modelHostports[ binding.host ] = networkSegmentModel

               if modelSwitchports or modelHostports:
                  instancePortBindingModel.ports[ port.id ] = (
                     OpenStackModels.PortBinding(
                        portId=port.id,
                        hosts=modelHostports,
                        switches=modelSwitchports )
                  )
            if instancePortBindingModel.ports:
               instancePortBindingModel.instanceId = instance.id
               instancePortBindingModel.instanceName = instanceName
               instancePortBindingModel.tenantId = instance.tenant.id
               instancePortBindingModel.tenantName = tenantName
               instanceModel.instances[ instance.id ] = instancePortBindingModel

      regionSegmentsModel.regions[ regionName ] = instanceModel

   return regionSegmentsModel

class ShowOpenstackSegmentsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = (
      'show openstack segments '
      '{ HF | NF | PF | IF | RF | SF | TF }'
   )
   data = {
      'openstack' : matcherOpenstack,
      'segments' : 'Display OpenStack Segments',
      'HF' : _ShowHostFilterExpr,
      'NF' : _ShowNetworkFilterExpr,
      'PF' : _ShowPortFilterExpr,
      'IF' : _ShowInstanceFilterExpr,
      'RF' : _ShowRegionFilterExpr,
      'SF' : _ShowSwitchFilterExpr,
      'TF' : _ShowTenantFilterExpr,
   }

   handler = doShowOpenStackSegments
   cliModel = OpenStackModels.RegionOpenStackSegments
   hidden = True

BasicCli.addShowCommandClass( ShowOpenstackSegmentsCmd )

#--------------------------------------------------------------------------------

def doControllerMounts( controllerdbEm ):
   global portVlanConfigDir
   global globalTopoStatus
   global openStackConfig
   global openStackStatus
   global cdbMountsComplete

   dynVlanPath = 'controllerProxyAgent/v1/portVlanConfigDir/openstack/switch'
   portVlanConfigDir = LazyMount.mount( controllerdbEm, dynVlanPath,
                                              'Tac::Dir', 'ri' )
   globalTopoStatus = LazyMount.mount( controllerdbEm,
                                       "topology/version3/global/status",
                                       "NetworkTopologyAggregatorV3::Status",
                                       "r" )

   openStackConfig = LazyMount.mount( controllerdbEm,
                                      'openstack/openstackConfig',
                                      'OpenStack::OpenStackConfig',
                                      'w' )
   openStackStatus = LazyMount.mount( controllerdbEm,
                                      'openstack/openstackStatus',
                                      'OpenStack::OpenStackStatus',
                                      'r' )
   cdbMountsComplete = True

#
# Plug-in definition.
#
@Plugins.plugin( requires=( 'ControllerdbMgr', ) )
def Plugin( entityManager ):
   global controllerConfig
   global clusterStatus
   global openStackCliConfig
   global virtualNetworkConfig
   global virtualNetworkStatus
   global openStackAgentStatus
   global osApiStatus
   global aclCheckpoint
   global nameResolutionConfig
   global nameResolutionStatus
   global serviceConfigDir
   global httpServiceConfig

   mount = LazyMount.mount
   mountConfig = ConfigMount.mount
   controllerConfig = mount( entityManager,
                             'controller/config',
                             'Controllerdb::Config',
                             'r' )
   clusterStatus = mount( entityManager,
                          'controller/cluster/statusDir',
                          'ControllerCluster::ClusterStatusDir',
                          'r' )
   openStackCliConfig = mountConfig( entityManager,
                                     'mgmt/openstack/config',
                                     'OpenStack::Config',
                                     'w' )
   virtualNetworkConfig = mountConfig( entityManager,
                                       'mgmt/virtualnetwork/config',
                                       'VirtualNetwork::Config',
                                       'w' )
   virtualNetworkStatus = mount( entityManager,
                                 'mgmt/virtualnetwork/status',
                                 'VirtualNetwork::Status',
                                 'r' )
   openStackAgentStatus = mount( entityManager,
                                 'mgmt/openstack/status',
                                 'OpenStack::Status',
                                 'r' )
   osApiStatus = mount( entityManager,
                        'mgmt/openstack/osApiStatus',
                        'JsonApi::ApiStatus',
                        'r' )
   aclCheckpoint = mount( entityManager,
                          'mgmt/openstack/aclCheckpoint',
                          'Acl::CheckpointStatus',
                          'w' )
   nameResolutionConfig = mountConfig( entityManager,
                                       'mgmt/openstack/nameResolutionConfig',
                                       'OpenStack::NameResolution::Config',
                                       'w' )
   nameResolutionStatus = mount( entityManager,
                                 'mgmt/openstack/nameResolutionStatus',
                                 'OpenStack::NameResolution::Status',
                                 'r' )
   httpServiceConfig = mount( entityManager,
                              'mgmt/capi/config',
                              'HttpService::Config',
                              'w' )
   # To let the CVX infrastructure to know that OpenStack service is enabled/disabled
   serviceConfigDir = mountConfig( entityManager,
                                   'controller/service/config',
                                   'Controller::ServiceConfigDir', 'w' )

   debug( "Registering controller mount callback" )
   ControllerdbLib.registerNotifiee( doControllerMounts )
