#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import CliPlugin.OpenFlowCli as OpenFlowCli
import CliPlugin.OpenFlowModel as OpenFlowModel
import ShowCommand
import BasicCliUtil

nodeOpenflow = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'openflow', helpdesc='OpenFlow status' ),
      guard=OpenFlowCli.openFlowSupportedGuard )

def getTableName( mode ):
   tNames = []
   flows = OpenFlowCli.openFlowHwConfig.flowEntry
   for flow in flows:
      tNames.append( OpenFlowCli.openFlowHwConfig.flowEntry[ flow ].tableName )

   return set( tNames )

def flowNameExcludePattern():
   excludeKeywords = [ 'brief', 'matched' ]
   excludePattern = ''.join( BasicCliUtil.notAPrefixOf( k )
                             for k in excludeKeywords )
   pattern = excludePattern + r'[A-Za-z0-9_:{}\[\]-]+'
   return pattern

#--------------------------------------------------------------------------------
# show openflow
#--------------------------------------------------------------------------------
class OpenflowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openflow'
   data = {
      'openflow': nodeOpenflow,
   }
   handler = lambda mode, args: OpenFlowCli.showSummary( mode )
   cliModel = OpenFlowModel.OpenFlowSummary

BasicCli.addShowCommandClass( OpenflowCmd )

#--------------------------------------------------------------------------------
# show openflow flows [ ( FLOWS) | ( table TABLENAME ) ][ matched ][ brief ]
#--------------------------------------------------------------------------------
class OpenflowFlowsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show openflow flows'
              '[ ( FLOWS ) | ( table TABLENAME ) ][ matched ][ brief ]' )
   data = {
      'openflow': nodeOpenflow,
      'flows': 'Show OpenFlow flows',
      'FLOWS': CliMatcher.DynamicNameMatcher(
         lambda mode: OpenFlowCli.openFlowHwConfig.flowEntry,
         'Flow name to retrieve', pattern=flowNameExcludePattern() ),
      'table': 'Show OpenFlow flows in specifc table',
      'TABLENAME': CliMatcher.DynamicNameMatcher(
         getTableName,
         'Table name to retrieve' ),
      'matched': "Show OpenFlow matched-flows",
      'brief': 'Show OpenFlow flows in brief mode',
   }
   handler = OpenFlowCli.showFlows
   cliModel = OpenFlowModel.OpenFlowFlows

BasicCli.addShowCommandClass( OpenflowFlowsCmd )

#--------------------------------------------------------------------------------
# show openflow groups [ GROUP ]
#--------------------------------------------------------------------------------
class OpenflowGroupsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openflow groups [ GROUP ]'
   data = {
      'openflow': nodeOpenflow,
      'groups': 'Show OpenFlow groups',
      'GROUP': CliMatcher.IntegerMatcher( 0x0, 0xffffff00,
                                          helpdesc='Group ID to retrieve' ),
   }
   handler = lambda mode, args: OpenFlowCli.showGroups( mode, args.get( 'GROUP' ) )
   cliModel = OpenFlowModel.OpenFlowGroups

BasicCli.addShowCommandClass( OpenflowGroupsCmd )

#--------------------------------------------------------------------------------
# show openflow ports
#--------------------------------------------------------------------------------
class OpenflowPortsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openflow ports'
   data = {
      'openflow': nodeOpenflow,
      'ports': 'Show OpenFlow port-interface map',
   }
   handler = lambda mode, args: OpenFlowCli.showPorts( mode )
   cliModel = OpenFlowModel.OpenFlowPorts

BasicCli.addShowCommandClass( OpenflowPortsCmd )

#--------------------------------------------------------------------------------
# show openflow profiles
#--------------------------------------------------------------------------------
class OpenflowProfilesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openflow profiles'
   data = {
      'openflow': nodeOpenflow,
      'profiles': 'Show OpenFlow table profiles for current bind mode',
   }
   handler = lambda mode, args: OpenFlowCli.showTableProfiles( mode )
   cliModel = OpenFlowModel.OpenFlowTablesProfile

BasicCli.addShowCommandClass( OpenflowProfilesCmd )

#--------------------------------------------------------------------------------
# show openflow queues
#--------------------------------------------------------------------------------
class OpenflowQueuesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openflow queues'
   data = {
      'openflow': nodeOpenflow,
      'queues': 'Show OpenFlow transmit queue stats',
   }
   handler = lambda mode, args: OpenFlowCli.showQueues( mode )
   cliModel = OpenFlowModel.OpenFlowQ

BasicCli.addShowCommandClass( OpenflowQueuesCmd )

#--------------------------------------------------------------------------------
# show openflow statistics
#--------------------------------------------------------------------------------
class OpenflowStatisticsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show openflow statistics'
   data = {
      'openflow': nodeOpenflow,
      'statistics': 'Show OpenFlow table statistics',
   }
   handler = lambda mode, args: OpenFlowCli.showStatistics( mode )
   cliModel = OpenFlowModel.OpenFlowActivityStats

BasicCli.addShowCommandClass( OpenflowStatisticsCmd )
