#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac
import datetime
import ArnetModel
import Intf
from CliModel import Bool
from CliModel import Enum
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from CliModel import Dict
from IntfModel import Interface
from collections import defaultdict

_bindModes = [ 'bindModeInterface', 'bindModeMonitor', 'bindModeVlan',
               'None' ]

_forwardingPipelines = [ 'flow', 'flow-and-normal' ]
_connectionModes = [ 'active', 'passive' ]
_minimumOFVersions = [ '1.3' ]
_tableProfiles = [ 'full-match', 
                   'l2-match',
                   'vxlan-match',
                   'auto' ]
_controllerRoles = [ 'equal', 'master', 'slave' ]
_status = [ 'flowPending', 'flowCreated', 'flowDeleted', 'flowExpiredHard',
            'flowExpiredIdle', 'flowRejectedBadMatch', 'flowRejectedBadAction',
            'flowRejectedHwTableFull', 'flowDeletedGroupRemoved',
            'flowRejectedOther', 'flowUnknown', 'flowActionsUnsupported',
            'flowRemovedIntfDown', 'flowRejectedTimeoutNotSupported' ]

_prioGroupType = [ 'default', 'low', 'high',
                   'portacl', 'vlan', 'vlanacl', 'ethsrc',
                   'ipv4Fib', 'ipv6Fib', 'vip', 'ethdst', 'ethdstHairpin', 'flood' ]
_tableType = [ 'ifp', 'vfp', 'egress vlan translate', 'bfnIngress1', 'bfnIngress2',
               'bfnIngress3', 'bfnIngress4' ]

packetResMap = { 
   'L2BC':Tac.Type( "OpenFlowTable::PacketResValues" ).L2BC,
   'L2UcKnown':Tac.Type( "OpenFlowTable::PacketResValues" ).L2UcKnown, 
   'L2UcUnknown':Tac.Type( "OpenFlowTable::PacketResValues" ).L2UcUnknown, 
   'L2McKnown':Tac.Type( "OpenFlowTable::PacketResValues" ).L2McKnown, 
   'L2McUnknown':Tac.Type( "OpenFlowTable::PacketResValues" ).L2McUnknown, 
   'L3UcKnown':Tac.Type( "OpenFlowTable::PacketResValues" ).L3UcKnown, 
   'L3UcUnknown':Tac.Type( "OpenFlowTable::PacketResValues" ).L3UcUnknown, 
   'L3McKnown':Tac.Type( "OpenFlowTable::PacketResValues" ).L3McKnown, 
   'L3McUnknown':Tac.Type( "OpenFlowTable::PacketResValues" ).L3McUnknown, 
   }

packetResDisplayMap = dict( 
   [ ( val, key ) for key, val in packetResMap.iteritems() ] )

PacketResType = Tac.Type( "OpenFlowTable::PacketResValues" )

class IpAddrAndBitMask( Model ):
# Mask could be non-contiguous bits
   ip = ArnetModel.Ip4Address( help="Ip address" )
   mask = ArnetModel.Ip4Address( help="Ip address mask" )
   
   def formatStr( self ):
      maskVal = str( "/%s"%self.mask ) if self.mask else ''
      return '%s%s' % ( self.ip, maskVal )

class IpAddr( Model ):
   ip = ArnetModel.Ip4Address( help="Ip address" )
   
   def formatStr( self ):
      return '%s' % ( self.ip )

class Ip6AddrAndBitMask( Model ):
# Mask could be non-contiguous bits
   ip = ArnetModel.Ip6Address( help="Ip6 address" )
   mask = ArnetModel.Ip6Address( help="Ip6 address mask" )
   
   def formatStr( self ):
      maskVal = str( "/%s"%self.mask ) if self.mask else ''
      return '%s%s' % ( self.ip, maskVal )

class IpA6ddr( Model ):
   ip = ArnetModel.Ip6Address( help="Ip6 address" )
   
   def formatStr( self ):
      return '%s' % ( self.ip )

class MacAddrAndBitMask( Model ):
# Mask could be non-contiguous bits
   mac = ArnetModel.MacAddress( help="Mac address" )
   mask = ArnetModel.MacAddress( help="Mac address mask" )
   
   def formatStr( self ):
      maskVal = str( "/%s"%self.mask ) \
          if ( self.mask != "ff:ff:ff:ff:ff:ff" ) else ''
      return '%s%s' % ( self.mac.stringValue, maskVal )

class MacAddr( Model ):
   mac = ArnetModel.MacAddress( help="Mac address" )
   
   def formatStr( self ):
      return '%s' % ( self.mac.stringValue )

class OpenFlowSummary( Model ):
   class BindModeEntries( Model ):
      bindMode = Enum( values=_bindModes,
                       help='Bind mode' )
      bindVlan = Str( help='Bind vlans',
                      optional=True )
      nativeVlan = Str( help='Native vlan',
                        optional=True )
      bindInterfaces = List( help='Bind interfaces',
                             valueType=Interface,
                             optional=True )
   class RvlanEntries ( Model ):
      class Entries ( Model ):
         vlan = Int( help= 'Vlan entry' )
         rvlan = Int( help = 'rvlan entry' )

      rIntf = Interface( help='Recirculation interface' )
      rvlanEntry = List( help='rvlan entries',
                         valueType=Entries )

   class ControllerConfig( Model ):
      controllerAddr = Submodel( valueType=ArnetModel.IpAddrAndPort,
                                help='Connected controller' )
      auxEnabled = Bool( help="Auxiliary channel connected" )
      connectionMode = Enum( values=_connectionModes,
                            help="Switch's network connection mode with controller" )
      
   class ConnectedController( Model ):
      controllerAddr = Submodel( valueType=ArnetModel.IpAddrAndPort,
                                help='Connected controller' )
      controllerRole = Enum( values=_controllerRoles,
                             help='Controller role',
                             optional=True )
      auxConnected = Bool( help="Auxiliary channel connected" )
      negotiatedVersion = Str( 
         help='Version negotiated with the connected controller',
         optional=True )      

# Configuration
   enabled = Bool( help='OpenFlow enabled' )
   description = Str( help='Datapath description' )
   datapathID = Int( help='Datapath ID' )

   controllers = List( help='List of openflow controllers',
                       valueType=ArnetModel.IpAddrAndPort )
   controllersInfo = List( help='List of openflow controllers',
                       valueType=ControllerConfig, 
                       optional=True )
   minimumVersion = Enum( values=_minimumOFVersions,
      help='Minimum OpenFlow protocol version required',
      optional=True )
   connectedCtrl = List( help='List of connected controllers',
                         valueType=ConnectedController,
                         optional=True )
   connectSuccessCount = Int( help='Connect success count' )
   keepalivePeriod = Int( help='Keepalive in seconds' )
   passiveControllerConfig = List( help='List of configured passive controllers',
                                   valueType=ControllerConfig,
                                   optional=True )
   passiveControllerStatus = List( help='List of listening passive controllers',
                                   valueType=ConnectedController,
                                   optional=True )

   flowTableState = Bool( help='Flow table enabled or not' )
   # Flowtable Profile is tableProfileL2Match, full-match or auto
   flowTableProfile = Enum( values=_tableProfiles,
                            help='Table profile' )
   forwardingPipeline = Enum( values=_forwardingPipelines,
                              help='Forwarding pipeline' )
   bindMode = Submodel( valueType=BindModeEntries,
                        help='Bind mode' )
   rvlans = Submodel( valueType=RvlanEntries,
                      help='rvlan entries',
                      optional=True )
   ipRoutingState = Bool( help='Enabled if rvlans' )
   shellCommandAllowed = Bool( help='Shell commands allowed' )
   expiredTtlOutputController = Bool( help='Send expired ttl packets to controller',
                                      optional=True )
   totalPacketCount = Int( help ='Matched packets' )

   def render( self ):

      print "OpenFlow configuration: %s" % ( "Enabled" if self.enabled
                                             else "Disabled" )
      print "DPID: 0x%016x" % self.datapathID
      print "Description: %s" % self.description
      print "Controllers:"
      ctrlList = []
      auxEnabled = False
      for c in self.controllersInfo:
         ctrlList.append( c.controllerAddr.formatStr() )
         for ctrl in self.connectedCtrl:
            if ctrl.controllerAddr == c.controllerAddr and c.auxEnabled:
               auxEnabled = True
      print "  configured: %s" % ( " ".join( ctrlList ) or "<none>" )
      if self.connectedCtrl:
         print "  connected:"
         for c in self.connectedCtrl:
            print "    %s" % c.controllerAddr.formatStr()
            if c.negotiatedVersion != 'Version 1.0': 
               print "      role: %s" % c.controllerRole
            if auxEnabled:
               print "      auxiliary connection %s" % ( "connected" if 
                     c.auxConnected else "not connected" )
            print "      negotiated version: %s" % c.negotiatedVersion
      else:
         print "  connected: <none>"
      print "  connection count: %s" % self.connectSuccessCount
      print "  keepalive period: %d sec" % self.keepalivePeriod
      if self.passiveControllerConfig:
         print "Passive Controllers:"
         configList = []
         for controller in self.passiveControllerConfig:
            configList.append( controller.controllerAddr.formatStr() )
         print "  configured: %s" % " ".join( configList )
         print "  listening:"
         for controller in self.passiveControllerStatus:
            print "    %s" % controller.controllerAddr.formatStr()

      print "Flow table state: %s" \
          % ( "Enabled" if self.flowTableState else "Disabled" )
      print "Flow table profile: %s" % self.flowTableProfile

      print "Forwarding pipeline: %s" % self.forwardingPipeline
      if self.minimumVersion:
         print "Minimum version: %s" % self.minimumVersion
      print "Bind mode: %s" % self.bindMode.bindMode      
      if self.bindMode.bindMode == "bindModeVlan":
         print "  VLANs: %s" % self.bindMode.bindVlan
         print "  native VLAN: %s" % self.bindMode.nativeVlan
      elif self.bindMode.bindMode == "bindModeInterface":
         bindIntfList = Intf.IntfRange.intfListToCanonical(
                        self.bindMode.bindInterfaces )
         print "  interfaces: "
         if bindIntfList:
            for bindIntf in bindIntfList:
               print "    %s" % bindIntf
         else:
            print "    <none>"

      print "IP routing state: %s" \
          % ( "Enabled" if self.ipRoutingState else "Disabled" )
      if self.ipRoutingState:
         print "  recirculation interface: %s" % self.rvlans.rIntf
         for e in self.rvlans.rvlanEntry:
            print "  VLAN %s: routed to/from VLAN %s" % (
               "untagged" if e.vlan == 0 else e.vlan, e.rvlan )
      print "Shell command execution: %s" \
          % ( "Enabled" if self.shellCommandAllowed else "Disabled" )
      if self.expiredTtlOutputController:
         print "Expired TTL packets: Output to Controller"
      print "Total matched: %d packets" \
      % ( self.totalPacketCount if self.enabled else 0 )

class OpenFlowPorts( Model ):
   class Entry( Model ):
      port = Int( help="OpenFlow port" )
      intf = Interface( help="Switch interface" )
      mirror = Bool( help="mirror destination" )

   portInterface = List( help='List of openflow Ports to switch interface',
                         valueType=Entry )

   def render( self ):
      for entry in sorted( self.portInterface, key=lambda e: e.port ):
         print "Port %s: %s%s" % ( entry.port, 
                                   entry.intf.stringValue.replace( "'", "" ), 
                                   " (mirror destination)" \
                                      if entry.mirror else "" )

class OpenFlowFlows( Model ):
   _brief = Bool( help="Show flow info brief" )

   class Flow( Model ):
      class Counter( Model ):
         matchPackets = Int( help="Number of packets that matched flow",
                             optional=True )
         matchBytes = Int( help="Number of bytes that matched flow",
                           optional=True )
      class Status( Model ):
         status = Enum( values=_status, help="Flow status per slice" )
         
      class Match( Model ):
         # All are optional, but if match is set the atleast one match
         # condition should be set
         inInterfaces = List( help="Match input interfaces",
                              valueType=Interface,
                              optional=True )
         inInterfaceCpu = Bool( help="Match input interface CPU",
                                optional=True )
         fieldSetSrc = Str( help="Match source field-Set",
                            optional=True )
         fieldSetDst = Str( help="Match destination field-Set",
                            optional=True )
         macSrc = Submodel( valueType=MacAddrAndBitMask,
                            help="Match source Mac address",
                            optional=True )
         macSrcBr = Submodel( valueType=MacAddrAndBitMask,
                              help="Match source Mac address with router-address",
                              optional=True )
         macDst = Submodel( valueType=MacAddrAndBitMask,
                            help="Match destination Mac address",
                            optional=True )
         macDstBr = Submodel( valueType=MacAddrAndBitMask,
                              help="Match destination Mac address with "
                                   "router-address",
                              optional=True )
         vlan = Int( help="Match Vlan `ID",
                     optional=True )
         vlanMask = Int( help="Match Vlan ID Mask",
                         optional=True )
         vlanInner = Int( help="Match inner Vlan `ID",
                          optional=True )
         vlanInnerMask = Int( help="Match inner Vlan ID Mask",
                              optional=True )
         encapDot1q = Bool( help="Match 802.1Q encapsulation", optional=True)
         vlanUnTagged = Bool(help="Match untagged packets",
                             optional=True )
         vlanPCP = Int( help="Match Vlan PCP",
                        optional=True )
         ethType = Int( help="Match ethernet type",
                        optional=True )
         ethTypeMask = Int( help="Match Ethertype  Mask",
                         optional=True )
         vrfName = Str( help="Match VRF", optional=True )
         ipSrc = Submodel( valueType=IpAddrAndBitMask,
                           help="Match source IP address and mask",
                           optional=True )
         ipDst = Submodel( valueType=IpAddrAndBitMask,
                           help="Match destination IP address and mask",
                           optional=True )
         ip6Src = Submodel( valueType=Ip6AddrAndBitMask,
                            help="Match source IP6 address and mask",
                            optional=True )
         ip6Dst = Submodel( valueType=Ip6AddrAndBitMask,
                            help="Match destination IP6 address and mask",
                            optional=True )

         ipTos = Int( help="Match Ip Tos (as received)",
                      optional=True )
         ipProto = Int( help="Match IP protocol",
                        optional=True )
         ipPortSrc = Int( help="Match TCP/UDP source port",
                          optional=True )
         ipPortDst = Int( help="Match TCP/UDP destination port",
                          optional=True )
         icmpType = Int( help="Match ICMP type",
                         optional=True )
         icmpCode = Int( help="Match ICMP code",
                         optional=True )
         unknownL2V4MulticastAddress = Bool( help="Unknown IPV4 multicast"
                                             "address. Match on the L2 field",
                                             optional=True )
         unknownL3V4MulticastAddress = Bool( help="Unknown IPV4 multicast"
                                             "address. Match on the L3 field",
                                             optional=True )
         tcpUrg = Bool( help="Tcp flag Urg", optional=True )
         tcpAck = Bool( help="Tcp flag Ack", optional=True )
         tcpPsh = Bool( help="Tcp flag Psh", optional=True )
         tcpRst = Bool( help="Tcp flag Rst", optional=True )
         tcpSyn = Bool( help="Tcp flag Syn", optional=True )
         tcpFin = Bool( help="Tcp flag Fin", optional=True )
         ttl = Int( help="Match Time to Live value", optional=True )
         ttlMask = Int( help="Match Time to Live mask", optional=True )
         packetRes = Enum( values=PacketResType.attributes,
                                    help="Match Packet Resolution",
                                    optional=True )
         inputInternalLoopback = Bool( help="Match hardware loopback",
                                       optional=True )
         metaData = Int( help="Match metaData",
                         optional=True )
         def render( self ):
            print "  match:"
            if self.inInterfaces or self.inputInternalLoopback:
               print "    ingress interface: "
               if self.inInterfaces:
                  inIntfList = Intf.IntfRange.intfListToCanonical(
                     self.inInterfaces )
                  for inIntf in inIntfList:
                     print "        %s" % inIntf
               if self.inputInternalLoopback:
                  print "        hardware loopback interface"
            if self.inInterfaceCpu:
               print "    ingress interface: cpu"
            if self.fieldSetSrc:
               print "    source Field-Set: %s" \
                   % ( self.fieldSetSrc )
            if self.fieldSetDst:
               print "    destination Field-Set: %s" \
                   % ( self.fieldSetDst )
            if self.macSrc:
               print "    source Ethernet address: %s" \
                   % ( self.macSrc.formatStr() )
            if self.macSrcBr:
               print "    source Ethernet address: router-address (%s)" \
                   % ( self.macSrcBr.formatStr() )
            if self.macDst:
               print "    destination Ethernet address: %s" \
                   % ( self.macDst.formatStr() )
            if self.macDstBr:
               print "    destination Ethernet address: router-address (%s)" \
                   % ( self.macDstBr.formatStr() )
            if self.vlan is not None:
               if self.vlan == 0:
                  print "    untagged/native VLAN ID"
               elif self.vlanMask == 4095:
                  print "    VLAN ID: %s" % self.vlan
               else:
                  print "    VLAN ID: 0x%x/0x%x" % (
                     self.vlan, self.vlanMask )
            if self.vlanInner is not None:
               if self.vlanInnerMask == 4095:
                  print "    Inner VLAN ID: %s" % self.vlanInner
               else:
                  print "    Inner VLAN ID: 0x%x/0x%x" % (
                     self.vlanInner, self.vlanInnerMask )
            if self.encapDot1q is not None:
               print "    802.1Q Encapsulation"
            if self.vlanUnTagged is not None:
               print "    Vlan Untagged"
            if self.vlanPCP is not None:
               print "    VLAN PCP: %s" % self.vlanPCP
            if self.ethType is not None:
               ethTypeStr = "IPv4" if self.ethType == 0x0800 else \
                            "ARP" if self.ethType == 0x0806 else \
                            "IPv6" if self.ethType == 0x86dd else \
                            ( "0x%x" % self.ethType )
               if self.ethTypeMask is None:
                  self.ethTypeMask = 0xffff
               if self.ethTypeMask == 0xffff:
                  print "    Ethernet type: " + ethTypeStr
               else:
                  print "    Ethernet type: %s mask 0x%x" % \
                        ( ethTypeStr, self.ethTypeMask )
            if self.vrfName is not None:
               print "    VRF: %s" % self.vrfName
            if self.ipSrc:
               print "    source IPv4 address: %s" % self.ipSrc.formatStr()
            if self.ipDst:
               print "    destination IPv4 address: %s" % self.ipDst.formatStr()
            if self.ip6Src:
               print "    source IPv6 address: %s" % self.ip6Src.formatStr()
            if self.ip6Dst:
               print "    destination IPv6 address: %s" % self.ip6Dst.formatStr()

            if self.tcpUrg:
               print "    TCP Flag Urg"
            if self.tcpAck:
               print "    TCP Flag Ack"
            if self.tcpPsh:
               print "    TCP Flag Psh"
            if self.tcpRst:
               print "    TCP Flag Rst"
            if self.tcpSyn:
               print "    TCP Flag Syn"
            if self.tcpFin:
               print "    TCP Flag Fin"
            if self.ipTos:
               print "    IP TOS: %s (received) DSCP: %s (programmed)" \
                  % ( self.ipTos, ( self.ipTos >> 2 ) )
            if self.ipProto:
               print "    IP protocol: " + (
                  "ICMP" if self.ipProto == 1 else
                  "TCP" if self.ipProto == 6 else
                  "UDP" if self.ipProto == 17 else
                  "ICMPv6" if self.ipProto == 58 else
                  ( "%s" % self.ipProto ) )
            if self.ipPortSrc is not None:
               print "    source TCP/UDP port: %s" % self.ipPortSrc
            if self.ipPortDst is not None:
               print "    destination TCP/UDP port: %s" % self.ipPortDst
            if self.icmpType is not None:
               print "    ICMP type: %s" % self.icmpType
            if self.icmpCode is not None:
               print "    ICMP code: %s" % self.icmpCode
            if self.ttl is not None:
               print "    TTL: %s mask 0x%x" % ( self.ttl, self.ttlMask )
            if self.packetRes is not None:
               print "    Packet Resolution: %s" % (
                  packetResDisplayMap[ self.packetRes ] )
            if self.metaData is not None:
               print "    MetaData: %s" % self.metaData

         def renderBrief( self ):
            matchStr = ''
            if self.inInterfaces or self.inputInternalLoopback:
               matchStr += "in_port:"
               if self.inInterfaces:
                  inIntfList = Intf.IntfRange.intfListToCanonical(
                     self.inInterfaces )
                  for inIntf in inIntfList:
                     matchStr += "%s " % inIntf
            if self.macSrc:
               matchStr += "smac:%s " \
                   % ( self.macSrc.formatStr() )
            if self.macDst:
               matchStr += "dmac:%s " \
                   % ( self.macDst.formatStr() )
            if self.vlan is not None:
               matchStr += "vlan:%s " % self.vlan
            if self.vlanPCP is not None:
               matchStr += "vlan_pcp:%s " % self.vlanPCP
            if self.ethType is not None:
               matchStr += "eth_type:%s " % self.ethType
            if self.ipSrc:
               matchStr += "sip:%s " % self.ipSrc.formatStr()
            if self.ipDst:
               print "dip:%s " % self.ipDst.formatStr()
            if self.ip6Src:
               print "sip:%s " % self.ip6Src.formatStr()
            if self.ip6Dst:
               matchStr += "dip:%s " % self.ip6Dst.formatStr()
            if self.ipTos:
               matchStr += "tos:%s " % self.ipTos
            if self.ipProto:
               matchStr += "proto:" + (
                  "ICMP" if self.ipProto == 1 else
                  "TCP" if self.ipProto == 6 else
                  "UDP" if self.ipProto == 17 else
                  "ICMPv6" if self.ipProto == 58 else
                  ( "%s " % self.ipProto ) )
            if self.ipPortSrc is not None:
               matchStr += "src_port:%s " % self.ipPortSrc
            if self.ipPortDst is not None:
               matchStr += "dst_port:%s " % self.ipPortDst
            if self.icmpType is not None:
               matchStr += "icmp_type:%s " % self.icmpType
            if self.icmpCode is not None:
               matchStr += "icmp_code:%s " % self.icmpCode
            if self.ttl is not None:
               matchStr += "ttl:%s/0x%x " % ( self.ttl, self.ttlMask )
            if self.metaData is not None:
               matchStr += "meta_data:%s " % self.metaData

            return matchStr[ :-1 ]

      class Action( Model ):
         # All are optional but if action is used then atleast one action
         # should have been set
         ingrMirrorInterfaces = \
         List( help="Copy ingress to mirror destination interfaces",
               valueType=Interface,
               optional=True )
         egrMirrorInterfaces = \
         List( help="Copy egress to mirror destination interfaces",
               valueType=Interface,
               optional=True )
         macSrc = Submodel( valueType=MacAddr,
                            help="Set source Mac address",
                            optional=True )
         macSrcBr = Submodel( valueType=MacAddr,
                              help="Set source Mac address to router-address",
                              optional=True )
         macDst = Submodel( valueType=MacAddr,
                            help="Set destination Mac address",
                            optional=True )
         macDstBr = Submodel( valueType=MacAddr,
                              help="Set destination Mac address to router-address",
                              optional=True )
         pushVlan = Bool( help="Push Vlanid", optional=True )
         popVlan = Bool( help="Push Vlanid", optional=True )
         vlan = Int( help="Set Vlan ID",
                     optional=True )
         vlanPCP = Int( help="Set Vlan PCP",
                        optional=True )
         addVlanInner = Int( help="Add inner Vlan ID",
                             optional=True )
         addVlanInnerPCP = Int( help="Add inner Vlan PCP",
                                optional=True )
         addVlanOuter = Int( help="Add outer Vlan ID",
                             optional=True )
         addVlanOuterPCP = Int( help="Add outer Vlan PCP",
                                optional=True )
         removeVlanInner = Bool( help="Remove Inner Vlan header", optional=True )

         ipSrc = Submodel( valueType=IpAddr,
                           help="Set source IP address",
                           optional=True )
         ipDst = Submodel( valueType=IpAddr,
                           help="Set destination IP address",
                           optional=True )

         ipTos = Int( help="Set Ipv4 Tos (as received)",
                      optional=True )
         ttlDec = Bool( help="Decrement TTL", optional=True )
         ipPortSrc = Int( help="Set TCP/UDP source port",
                          optional=True )
         ipPortDst = Int( help="Set TCP/UDP destination port",
                          optional=True )
         icmpType = Int( help="Set ICMP type",
                          optional=True )
         icmpCode = Int( help="Set ICMP code",
                          optional=True )

         group = Int( help="Forward to group",
                      optional=True )

         outInterfaces = List( help="Output interfaces",
                               valueType=Interface,
                               optional=True )
         loopback = Bool( help="Output to ingress interface",
                          optional=True )
         outputNormal = Bool( help="Process normally" )
         outputFlood = \
             Bool( help="Flood to all non-blocked interfaces except ingress" )
         outputAll = Bool( help="Flood to all interfaces except ingress" )
         outputController = Bool( help="Send to controller" )
         outputLocal = Bool( help="Send to switch CPU" )
         outputQ = Int( help="Output queue",
                        optional=True )
         outputTrafficClass = Int( help="Output traffic class",
                                   optional=True )
         outputNexthop = List( valueType=ArnetModel.IpGenericAddr,
                                help="Output nexthop IP address",
                                optional=True )
         outputNexthopRec = Bool( 
            help="Recursive resolution of nexthop IP address", 
            optional=True )
         outputNexthopVrfName = Str( help="VRF name", optional=True )
         outputInternalLoopback = Bool( help="Output to hardware loopback",
                                        optional=True )
         outputDrop = Bool( help="Drop packet" )
         goto = Bool( help="Goto Action", optional=True )
         debugCapture = Bool( help="Debug Capture", optional=True )
         localIntf = Str( help="Local intf", optional=True )
         dropTag = Bool( help="Drop Outer Tag", optional=True )
         metaData = Int( help="Set meta data", optional=True )
         nextTblName = Str( help="Next Table Name", optional=True )
         def render( self ):
            print "  actions:"
            if self.ingrMirrorInterfaces:
               print "    copy ingress to mirror dest interfaces: %s" \
                   % ", ".join(sorted( self.ingrMirrorInterfaces ) )
            if self.vlan is not None:
               if self.pushVlan is None:
                  if self.vlan == 0:
                     print "    set VLAN ID to: untagged/native VLAN"
                  else:
                     print "    set VLAN ID to: %s" % self.vlan
               else:
                  print "    push VLAN ID to: %s" % self.vlan
            if self.popVlan is not None:
               print "    pop VLAN ID"
            if self.vlanPCP is not None:
               print "    set VLAN PCP to: %s" % self.vlanPCP
            if self.addVlanInner is not None:
               print "    add inner VLAN header with ID: %s" % self.addVlanInner
            if self.addVlanInnerPCP is not None:
               print "    add inner VLAN header with PCP: %s" % self.addVlanInnerPCP
            if self.addVlanOuter is not None:
               print "    add outer VLAN header with ID: %s" % self.addVlanOuter
            if self.addVlanOuterPCP is not None:
               print "    add outer VLAN header with PCP: %s" % self.addVlanOuterPCP
            if self.removeVlanInner is not None:
               print "    remove inner VLAN header"
            if self.macSrc:
               print "    set source Ethernet address to: %s" \
                   % self.macSrc.formatStr()
            if self.macSrcBr:
               print "    set source Ethernet address to: router-address (%s)" \
                   % self.macSrcBr.formatStr()
            if self.macDst:
               print "    set destination Ethernet address to: %s" \
                   % self.macDst.formatStr()
            if self.macDstBr:
               print "    set destination Ethernet address to: router-address (%s)" \
                   % self.macDstBr.formatStr()
            if self.ipSrc:
               print "    set source IPv4 address to: %s" % self.ipSrc.formatStr()
            if self.ipDst:
               print "    set destination IPv4 address to: %s" \
                   % self.ipDst.formatStr()
            if self.ipTos is not None:
               print "    set IPv4 TOS to: %s (received) DSCP: %s (programmed)" \
                  % ( self.ipTos, ( self.ipTos >> 2 ) )
            if self.ttlDec:
               print "    decrement TTL"
            if self.ipPortSrc:
               print "    set source TCP/UDP port to: %s" % self.ipPortSrc
            if self.ipPortDst:
               print "    set destination TCP/UDP port to: %s" % self.ipPortDst
            if self.icmpType:
               print "    set ICMP type to: %s" % self.icmpType
            if self.icmpCode:
               print "    set ICMP code to: %s" % self.icmpCode
            if self.group:
               print "    forward to group: %s" % self.group
            if self.outInterfaces:
               outIntfList = Intf.IntfRange.intfListToCanonical(
                             self.outInterfaces )
               print "    output interfaces: "
               for outIntf in outIntfList:
                  print "        %s" % outIntf
            if self.outputInternalLoopback:
               print "    output to hardware loopback interface"
            if self.loopback:
               print "    output to ingress interface"
            if self.outputNormal:
               print "    forward normally"
            if self.outputFlood:
               print "    output to all non-blocked interfaces except ingress"
            if self.outputAll:
               print "    output to all interfaces except ingress"
            if self.outputController:
               print "    output to controller"
            if self.outputLocal:
               print "    output to switch CPU interface"
               if self.debugCapture:
                  print "    debug capture"
               if self.localIntf:
                  print "    local intf: %s" % self.localIntf
            if self.dropTag:
               print "    drop outer tag"
            if self.metaData:
               print "    set MetaData %s" % self.metaData
               print "    goto Table   %s" % self.nextTblName
            if self.outputQ is not None:
               print "    output via queue: %s" % self.outputQ
            if self.outputTrafficClass is not None:
               print "    output traffic-class: %s" % self.outputTrafficClass
            if self.outputNexthop:
               nexthopList = [ x.ip.stringValue for x in self.outputNexthop ]
               nexthopStr = ' '.join( nexthopList )
               vrfStr = ""
               if self.outputNexthopVrfName != 'default':
                  vrfStr = "vrf " + self.outputNexthopVrfName + " "
               if self.outputNexthopRec:
                  print "    output nexthop: recursive " + vrfStr + nexthopStr
               else:
                  print "    output nexthop: " + vrfStr + nexthopStr
            if self.outputDrop:
               print "    drop"
            if self.egrMirrorInterfaces:
               print "    copy egress to mirror dest interfaces: %s" \
                   % ", ".join( sorted( self.egrMirrorInterfaces ) )

         def renderBrief( self ):
            actionStr = ''
            if self.ingrMirrorInterfaces:
               actionStr += "ingress_mirror_dst:%s " \
                   % ", ".join( sorted( self.ingrMirrorInterfaces ) )
            if self.vlan is not None:
               if self.pushVlan is None:
                  actionStr += "set_vlan:%s " % self.vlan
               else:
                  actionStr += "push_vlan:%s " % self.vlan
            if self.popVlan is not None:
               actionStr += "pop_vlan "
            if self.vlanPCP is not None:
               actionStr += "set_vlan_pcp:%s " % self.vlanPCP
            if self.addVlanInner is not None:
               actionStr += "set_inner_vlan:%s " % self.addVlanInner
            if self.addVlanInnerPCP is not None:
               actionStr += "set_inner_vlan_pcp:%s " % self.addVlanInnerPCP
            if self.addVlanOuter is not None:
               actionStr += "set_outer_vlan:%s " % self.addVlanOuter
            if self.addVlanOuterPCP is not None:
               actionStr += "add_outer_vlan_pcp:%s " % self.addVlanOuterPCP
            if self.removeVlanInner is not None:
               actionStr += "pop_inner_vlan "
            if self.macSrc:
               actionStr += "set_smac:%s " % self.macSrc.formatStr()
            if self.macSrcBr:
               actionStr += "set_smac:%s " % self.macSrcBr.formatStr()
            if self.macDst:
               actionStr += "set_dmac:%s " % self.macDst.formatStr()
            if self.macDstBr:
               actionStr += "set_dmac:%s " % self.macDstBr.formatStr()
            if self.ipSrc:
               actionStr += "sip:%s " % self.ipSrc.formatStr()
            if self.ipDst:
               actionStr += "dip:%s " % self.ipDst.formatStr()
            if self.ipTos is not None:
               actionStr += "ip_tos:%s " % ( self.ipTos )
            if self.ttlDec:
               actionStr += "dec_ttl "
            if self.ipPortSrc:
               actionStr += "set_sport:%s " % self.ipPortSrc
            if self.ipPortDst:
               actionStr += "set_dport:%s " % self.ipPortDst
            if self.icmpType:
               actionStr += "icmp_type:%s " % self.icmpType
            if self.icmpCode:
               actionStr += "icmp_code:%s " % self.icmpCode
            if self.group:
               actionStr += "output_group:%s " % self.group
            if self.outInterfaces:
               outIntfList = Intf.IntfRange.intfListToCanonical(
                             self.outInterfaces )
               actionStr += "output:"
               for outIntf in outIntfList:
                  actionStr += "%s " % outIntf
            if self.outputInternalLoopback:
               actionStr += "output:hw_loopback "
            if self.loopback:
               actionStr += "output:loopback "
            if self.outputNormal:
               actionStr += "output:forward_normal "
            if self.outputFlood:
               actionStr += "output:flood "
            if self.outputAll:
               actionStr += "output:all "
            if self.outputController:
               actionStr += "output:controller "
            if self.outputLocal:
               if self.debugCapture:
                  actionStr += "output_cpu:debug_capture "
               if self.localIntf:
                  actionStr += "output_cpu:%s " % self.localIntf
            if self.dropTag:
               actionStr += "drop_outer_tag "
            if self.metaData:
               actionStr += "meta_data:%s " % self.metaData
               actionStr += "goto_table:%s " % self.nextTblName
            if self.outputQ is not None:
               actionStr += "output_queue:%s " % self.outputQ
            if self.outputTrafficClass is not None:
               actionStr += "output_traffic-class:%s " % self.outputTrafficClass
            if self.outputNexthop:
               nexthopList = [ x.ip.stringValue for x in self.outputNexthop ]
               nexthopStr = ' '.join( nexthopList )
               vrfStr = ""
               if self.outputNexthopVrfName != 'default':
                  vrfStr = "vrf " + self.outputNexthopVrfName + " "
               if self.outputNexthopRec:
                  actionStr += "output_nexthop:" + vrfStr + nexthopStr
            if self.outputDrop:
               actionStr += "drop "
            if self.egrMirrorInterfaces:
               actionStr += "egress_mirror_dst:%s " \
                   % ", ".join( sorted( self.egrMirrorInterfaces ) )

            return actionStr[ :-1 ]

      match = Submodel( valueType=Match,
                        help='Match values',
                        optional=True )
      action = Submodel( valueType=Action,
                         help='Action set values',
                         optional=True )
      name = Str( help="Flow name" )
      tableName = Str( help="Flow name", optional=True )
      priority = Int( help="Flow priority" )
      priorityGroupType = Enum( values=_prioGroupType,
                                help="Flow priority Group", optional=True )
      tableType = Enum( values=_tableType, help="Flow table", optional=True )
      persistent = Bool( help="Flow persistence", optional=True )
      installation = Bool( help="Install flow after resource allocation",
                           optional=True )
      cookie = Int( help="flow cookie", optional=True )
      idleTimeout = Int( help=" Idle timeout", optional=True )
      hardTimeout = Int( help=" Hard timeout", optional=True )
      matchPackets = Int( help="Number of packets that matched flow",
                          optional=True )
      matchBytes = Int( help="Number of bytes that matched flow",
                        optional=True )
      bridgeMacAddr = Str( help="Bridge MAC Address", optional=True )
      countersPerSlice = Dict( keyType=str, valueType=Counter,
                               help="Dictionary of counters keyed with slice name" )
      statusPerSlice = Dict( keyType=str, valueType=Status,
                               help="Dictionary of status's keyed with slice name" )
      countersAvailable = Bool( help="Hardware counters available", optional=True )

      tableId = Int( help="Table Id", optional=True )

      def render( self ):
         print "Flow %s:" % self.name
         if self.tableName:
            print "  Table Name: %s" % self.tableName
         print "  priority: %s" % self.priority
         if self.priorityGroupType:
            print "  table: %s" % self.priorityGroupType
         print "  cookie: %s (0x%x)" % (self.cookie, self.cookie)
         if self.idleTimeout > 0:
            print "  idle timeout: %s sec" % self.idleTimeout
         if self.hardTimeout > 0:
            print "  hard timeout: %s sec" % self.hardTimeout
         if self.match:
            self.match.render()
         if self.action:
            self.action.render()
         print "  matched: %s packets, %s bytes" % ( self.matchPackets,
                                                     self.matchBytes )

      def renderBrief( self ):
         print "  %s: priority:%s packets:%s match[%s] action(s)[%s]" % \
            ( self.name,
              self.priority,
              self.matchPackets,
              self.match.renderBrief(),
              self.action.renderBrief() )


   flows = List( help='Flows',
                 valueType=Flow )

   def renderBrief( self ):
      table_list = defaultdict( list )
      table_name = defaultdict( list )
      for flow in self.flows:
         table_list[ flow.tableId ].append( flow )
         table_name[ flow.tableId ] = flow.tableName

      for table_id, table_flows in sorted( table_list.items(),
                                           key=lambda( k, v ): k ):
         print "Table ID: %d Table Name: %s:" % ( table_id, table_name[ table_id ] )
         for flow in table_flows:
            flow.renderBrief()

   def renderDetailed( self ):
      for flow in self.flows:
         flow.render()

   def render( self ):
      if self._brief:
         self.renderBrief()
      else:
         self.renderDetailed()

class OpenFlowGroups( Model ):
   class Group( Model ):
      class Bucket( Model ):
         bucketId = Int( help="Bucket Id" )
         macSrc = Submodel( valueType=MacAddr,
                            help="Set source Mac address",
                            optional=True )
         macDst = Submodel( valueType=MacAddr,
                            help="Set destination Mac address",
                            optional=True )
         vlan = Int( help="Set Vlan ID",
                     optional=True )
         ttlDec = Bool( help="Decrement TTL",
                        optional=True )
         outInterfaces = List( help="Output interfaces",
                               valueType=Interface,
                               optional=True )

         def render( self ):
            print "  Bucket %d:" % self.bucketId
            if self.vlan is not None:
               if self.vlan == 0:
                  print "    set VLAN ID to: untagged/native VLAN"
               else:
                  print "    set VLAN ID to: %s" % self.vlan
            if self.macSrc:
               print "    set source Ethernet address to: %s" \
                     % self.macSrc.formatStr()
            if self.macDst:
               print "    set destination Ethernet address to: %s" \
                     % self.macDst.formatStr()
            if self.ttlDec:
               print "    decrement TTL"
            if self.outInterfaces:
               print "    output interfaces : %s" % ", ".join( 
                     sorted( self.outInterfaces ) )

      buckets = List( valueType=Bucket,
                               help='List of buckets in the group',
                               optional=True )
      groupId = Int( help="Group Identifier" )
      status = Str( help="Status of the group" )
      flows = List( valueType=str, help='Flows referring to the group' )

      def render( self ):
         print "Group %d" % self.groupId
         print "  Status : %s" % self.status
         for bucket in self.buckets:
            bucket.render()
         if self.flows:
            print "  Flows referring to the group: "
            print "   ", ", ".join( sorted( self.flows ) )

   groups = List( help='Groups', valueType=Group )

   def render( self ):
      for group in self.groups:
         group.render()

class OpenFlowActivityStats( Model ):
   class Stats( Model ):
      endTime = Int( help="End time of this interval" )
      flowModifications = Int( help="Number of messages modified by flows" )
      packetsOut = Int( help="Number of packets" )
      packetsToController = Int( help="Packets sent to controller" )
      packetsDropped = \
      Int( help="Packets dropped instead of being sent to controller" )
      flowEntryCount = Int( help="Number of flows at the end of this interval" )

      def render( self ):
         timeDelta = Tac.utcNow() - Tac.now()
         t = datetime.datetime.fromtimestamp(
            self.endTime + timeDelta ).strftime( "%Y-%m-%d %H:%M:%S" )
         print "%s  %7d  %10d  %10d  %10d  %10d" % (
            t, self.flowEntryCount, self.flowModifications, self.packetsOut, 
            self.packetsToController,
            self.packetsDropped)

   activityStats = List( help='Activity stats for 5 second intervals',
                         valueType=Stats )
   def render( self ):
      print "                       table       messages processed last "\
          + "5 sec     dropped"
      print "                     entries   (flow_mod)(packet_out) "\
          + "(packet_in) last 5 sec"
      for a in sorted( self.activityStats, key=lambda s: s.endTime ):
         a.render()
      

class OpenFlowQStats( Model ):
   ID = Int( help="Q id" )
   packetCount = Int( help="Packet count" )
   byteCount = Int( help="Byte count" )
   errCount = Int( help="Error count" )

class OpenFlowQ( Model ):
   class PortQ( Model ):
      port = Int( help="Port ID" )
      interface = Interface( help='Interface' )
      stats = List( help="Port Queues",
                    valueType=OpenFlowQStats )
      
   controllerQ = List( help="Controller Queues",
                       valueType=OpenFlowQStats )
   portQ = List( help="Ports",
                 valueType=PortQ )

   def render( self ):
      # Show Controller Q
      for q in sorted( self.controllerQ, key=lambda q: q.ID ):
         print "  Queue %s: %s packets (%s bytes) transmitted, %s dropped" % (
            q.ID, q.packetCount, q.byteCount, q.errCount )
      # Show Ports and their Queues
      for p in sorted( self.portQ, key=lambda p: p.port ):
         print "Port %s (%s):" % ( p.port, p.interface )
         for q in sorted( p.stats, key=lambda q: q.ID ):
            print "  Queue %s: %s packets (%s bytes) transmitted, %s dropped" % (
               q.ID, q.packetCount, q.byteCount, q.errCount )

class OpenFlowTableProfileMatch( Model ):
   inInterface = Bool( "Ingress interface" )
   macSrc = Bool( "Ethernet source address" )
   macDst = Bool( "Ethernet destination address" )
   vlan = Bool( "Vlan ID" )
   vlanPCP = Bool( "VLAN priority" )
   ethType = Bool( "Ethernet type" )
   ipSrc = Bool( "IPv4 source address" )
   ipDst = Bool( "IPv4 destination address" )
   ipTos = Bool( "IPv4 TOS" )
   ipProto = Bool( "IPv4 protocol" )
   ipPortSrc = Bool( "TCP/UDP source port" )
   ipPortDst = Bool( "TCP/UDP destination port" )
   icmpType = Bool( "ICMP type" )
   icmpCode = Bool( "ICMP code" )

   def render( self ):
      # pylint: disable-msg=C0321
      if self.inInterface: print "    ingress interface" 
      if self.macSrc:      print "    source Ethernet address" 
      if self.macDst:      print "    destination Ethernet address" 
      if self.vlan:        print "    VLAN ID" 
      if self.vlanPCP:     print "    VLAN PCP" 
      if self.ethType:     print "    Ethernet type" 
      if self.ipSrc:       print "    source IPv4 address"
      if self.ipDst:       print "    destination IPv4 address" 
      if self.ipTos:       print "    IPv4 TOS" 
      if self.ipProto:     print "    IPv4 protocol" 
      if self.ipPortSrc:   print "    source TCP/UDP port"
      if self.ipPortDst:   print "    destination TCP/UDP port"
      if self.icmpType:    print "    ICMP type"
      if self.icmpCode:    print "    ICMP code"

class OpenFlowTableProfileAction( Model ):
   ingrMirrorInterfaces = Bool( "Copy ingress to mirror destination interfaces" )
   egrMirrorInterfaces = Bool( "Copy egress to mirror destination interfaces" )
   macSrc = Bool( "Set source Mac address" )
   macDst = Bool( "Set destination Mac address" )
   vlan = Bool( "Set Vlan ID" )
   vlanPCP = Bool( "Set Vlan PCP" )
   ipSrc = Bool( "Set source IP address" )
   ipDst = Bool( "Set destination IP address" )
   ipTos = Bool( "Set Ipv4 Tos" )
   ttlDec = Bool( help="Decrement TTL", optional=True )
   ipPortSrc = Bool( "Set TCP/UDP source port" )
   ipPortDst = Bool( "Set TCP/UDP destination port" )
   icmpType = Bool( "Set ICMP type" )
   icmpCode = Bool( "Set ICMP code" )
   outInterfaces = Bool( "Output interfaces" )
   loopback = Bool( "Output to ingress interface" )
   outputNormal = Bool( "Process normally" )
   outputFlood = Bool( help="Flood to all non-blocked interfaces except ingress" )
   outputAll = Bool( "Flood to all interfaces except ingress" )
   outputController = Bool( "Send to controller" )
   outputLocal = Bool( "Send to switch CPU" )
   outputQ = Bool( "Output queue" )
   setGroup = Bool( "Forward to group", optional=True )
   outputDrop = Bool( "Drop packet" )

   def render( self ):
      # pylint: disable-msg=C0321
      if self.ingrMirrorInterfaces: print "    copy ingress to mirror dest "\
             "interfaces" 
      if self.vlan:                 print "    set VLAN ID" 
      if self.vlanPCP:              print "    set VLAN PCP" 
      if self.macSrc:               print "    set source Ethernet address" 
      if self.macDst:               print "    set destination Ethernet address" 
      if self.ipSrc:                print "    set source IPv4 address" 
      if self.ipDst:                print "    set destination IPv4 address" 
      if self.ipTos:                print "    set IPv4 TOS" 
      if self.ttlDec:               print "    decrement TTL"
      if self.ipPortSrc:            print "    set source TCP/UDP port" 
      if self.ipPortDst:            print "    set destination TCP/UDP port"
      if self.icmpType:             print "    set ICMP type"
      if self.icmpCode:             print "    set ICMP code"
      if self.outInterfaces:        print "    output to specific interfaces" 
      if self.loopback:             print "    output to ingress interface" 
      if self.outputNormal:         print "    forward normally" 
      if self.outputFlood:          print "    output to all non-blocked interfaces"\
             " except ingress"
      if self.outputAll:            print "    output to all interfaces except "\
             "ingress" 
      if self.outputController:     print "    output to controller" 
      if self.outputLocal:          print "    output to switch CPU interface" 
      if self.outputQ:              print "    output via queue" 
      if self.setGroup:             print "    forward to group" 
      if self.outputDrop:           print "    drop" 
      if self.egrMirrorInterfaces:  print "    copy egress to mirror dest "\
             "interfaces" 



class  OpenFlowTableProfile ( Model ):
   match = Submodel( valueType=OpenFlowTableProfileMatch,
                     help='Match conditions' )
   wildcardMatch = Submodel( valueType=OpenFlowTableProfileMatch,
                             help='Wildcard match conditions' )
   action = Submodel( valueType=OpenFlowTableProfileAction,
                      help='Actions' )
   profile = Str( help="Table profile" )
   numFlowEntries = Int( help ="Table size entries" )   

   def render( self ):
      print self.profile
      print "  Match fields:"
      self.match.render()
      print "  Wildcard fields:"
      self.wildcardMatch.render()
      print "  Actions:"
      self.action.render()
      print "  Table size: %s entries max" % self.numFlowEntries

class OpenFlowTablesProfile( Model ):
   tables = List( help='Table profiles',
                  valueType=OpenFlowTableProfile )
   def render( self ):
      for t in self.tables:
         t.render()
         # New line between tables
         print
