#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, MultiRangeRule, socket, struct, Tracing, IpLibConsts
from eunuchs.in_h import IPPROTO_TCP, IPPROTO_UDP, IPPROTO_ICMP, IPPROTO_ICMPV6
from eunuchs.if_ether_h import ETH_P_IP, ETH_P_IPV6
import Arnet

t0 = Tracing.Handle( "OpenFlowCliLib" ).trace0

# --------------------------------------------------------------------------
# Utility functions
# --------------------------------------------------------------------------
def copyFlowEntry( srcFlowEntry, dstFields, dstMatch, dstEnabled, dstActions ):
   dstFields.inIntf = srcFlowEntry.match.matched.inIntf
   dstFields.inCpuIntf = srcFlowEntry.match.matched.inCpuIntf
   dstFields.ethSrc = srcFlowEntry.match.matched.ethSrc
   dstFields.ethSrcBr = srcFlowEntry.match.matched.ethSrcBr
   dstFields.ethDst = srcFlowEntry.match.matched.ethDst
   dstFields.ethDstBr = srcFlowEntry.match.matched.ethDstBr
   dstFields.ethType = srcFlowEntry.match.matched.ethType
   dstFields.vlanId = srcFlowEntry.match.matched.vlanId
   dstFields.vlanPri = srcFlowEntry.match.matched.vlanPri
   dstFields.vlanInnerId = srcFlowEntry.match.matched.vlanInnerId
   dstFields.encapDot1q = srcFlowEntry.match.matched.encapDot1q
   dstFields.vrfId = srcFlowEntry.match.matched.vrfId
   dstFields.ipSrc = srcFlowEntry.match.matched.ipSrc
   dstFields.ipDst = srcFlowEntry.match.matched.ipDst
   dstFields.ipTos = srcFlowEntry.match.matched.ipTos
   dstFields.ipProto = srcFlowEntry.match.matched.ipProto
   dstFields.l4Src = srcFlowEntry.match.matched.l4Src
   dstFields.l4Dst = srcFlowEntry.match.matched.l4Dst
   dstFields.icmpType = srcFlowEntry.match.matched.icmpType
   dstFields.icmpCode = srcFlowEntry.match.matched.icmpCode
   dstFields.tcpUrg = srcFlowEntry.match.matched.tcpUrg
   dstFields.tcpAck = srcFlowEntry.match.matched.tcpAck
   dstFields.tcpPsh = srcFlowEntry.match.matched.tcpPsh
   dstFields.tcpRst = srcFlowEntry.match.matched.tcpRst
   dstFields.tcpSyn = srcFlowEntry.match.matched.tcpSyn
   dstFields.tcpFin = srcFlowEntry.match.matched.tcpFin
   dstFields.ip6Src = srcFlowEntry.match.matched.ip6Src
   dstFields.ip6Dst = srcFlowEntry.match.matched.ip6Dst
   dstFields.packetRes = srcFlowEntry.match.matched.packetRes
   dstFields.inputInternalLoopback = srcFlowEntry.match.matched.inputInternalLoopback
   dstFields.trafficStreamSrc = srcFlowEntry.match.matched.trafficStreamSrc
   dstFields.trafficStreamDst = srcFlowEntry.match.matched.trafficStreamDst

   dstMatch.ethSrc = srcFlowEntry.match.ethSrc
   dstMatch.ethSrcMask = srcFlowEntry.match.ethSrcMask
   dstMatch.ethDst = srcFlowEntry.match.ethDst
   dstMatch.ethDstMask = srcFlowEntry.match.ethDstMask
   dstMatch.ethType = srcFlowEntry.match.ethType
   dstMatch.ethTypeMask = srcFlowEntry.match.ethTypeMask
   dstMatch.vlanId = srcFlowEntry.match.vlanId
   dstMatch.vlanIdMask = srcFlowEntry.match.vlanIdMask
   dstMatch.vlanPri = srcFlowEntry.match.vlanPri
   dstMatch.vlanInnerId = srcFlowEntry.match.vlanInnerId
   dstMatch.vlanInnerIdMask = srcFlowEntry.match.vlanInnerIdMask
   dstMatch.vrfName = srcFlowEntry.match.vrfName
   dstMatch.ipSrc = srcFlowEntry.match.ipSrc
   dstMatch.ipSrcMask = srcFlowEntry.match.ipSrcMask
   dstMatch.ipDst = srcFlowEntry.match.ipDst
   dstMatch.ipDstMask = srcFlowEntry.match.ipDstMask
   dstMatch.ipTos = srcFlowEntry.match.ipTos
   dstMatch.ipProto = srcFlowEntry.match.ipProto
   dstMatch.l4Src = srcFlowEntry.match.l4Src
   dstMatch.l4Dst = srcFlowEntry.match.l4Dst
   dstMatch.icmpType = srcFlowEntry.match.icmpType
   dstMatch.icmpCode = srcFlowEntry.match.icmpCode
   dstMatch.tcpUrg = srcFlowEntry.match.tcpUrg
   dstMatch.tcpAck = srcFlowEntry.match.tcpAck
   dstMatch.tcpPsh = srcFlowEntry.match.tcpPsh
   dstMatch.tcpRst = srcFlowEntry.match.tcpRst
   dstMatch.tcpSyn = srcFlowEntry.match.tcpSyn
   dstMatch.tcpFin = srcFlowEntry.match.tcpFin
   dstMatch.ip6Src = srcFlowEntry.match.ip6Src
   dstMatch.ip6SrcMask = srcFlowEntry.match.ip6SrcMask
   dstMatch.ip6Dst = srcFlowEntry.match.ip6Dst
   dstMatch.ip6DstMask = srcFlowEntry.match.ip6DstMask
   dstMatch.packetRes = srcFlowEntry.match.packetRes
   dstMatch.trafficStreamSrc = srcFlowEntry.match.trafficStreamSrc
   dstMatch.trafficStreamDst = srcFlowEntry.match.trafficStreamDst

   for i in srcFlowEntry.match.inIntf:
      dstMatch.inIntf[ i ] = True

   dstEnabled.ingrMirror = srcFlowEntry.actions.enabled.ingrMirror
   dstEnabled.setVlanId = srcFlowEntry.actions.enabled.setVlanId
   dstEnabled.setVlanPri = srcFlowEntry.actions.enabled.setVlanPri
   dstEnabled.addVlanInner = srcFlowEntry.actions.enabled.addVlanInner
   dstEnabled.addVlanOuter = srcFlowEntry.actions.enabled.addVlanOuter
   dstEnabled.removeVlanInner = srcFlowEntry.actions.enabled.removeVlanInner
   dstEnabled.setEthSrc = srcFlowEntry.actions.enabled.setEthSrc
   dstEnabled.setEthSrcBr = srcFlowEntry.actions.enabled.setEthSrcBr
   dstEnabled.setEthDst = srcFlowEntry.actions.enabled.setEthDst
   dstEnabled.setEthDstBr = srcFlowEntry.actions.enabled.setEthDstBr
   dstEnabled.setIpSrc = srcFlowEntry.actions.enabled.setIpSrc
   dstEnabled.setIpDst = srcFlowEntry.actions.enabled.setIpDst
   dstEnabled.setIpTos = srcFlowEntry.actions.enabled.setIpTos
   dstEnabled.setL4Src = srcFlowEntry.actions.enabled.setL4Src
   dstEnabled.setL4Dst = srcFlowEntry.actions.enabled.setL4Dst
   dstEnabled.setIcmpType = srcFlowEntry.actions.enabled.setIcmpType
   dstEnabled.setIcmpCode = srcFlowEntry.actions.enabled.setIcmpCode
   dstEnabled.outputIntf = srcFlowEntry.actions.enabled.outputIntf
   dstEnabled.outputInIntf = srcFlowEntry.actions.enabled.outputInIntf
   dstEnabled.outputNormal = srcFlowEntry.actions.enabled.outputNormal
   dstEnabled.outputFlood = srcFlowEntry.actions.enabled.outputFlood
   dstEnabled.outputAll = srcFlowEntry.actions.enabled.outputAll
   dstEnabled.outputController = srcFlowEntry.actions.enabled.outputController
   dstEnabled.outputLocal = srcFlowEntry.actions.enabled.outputLocal
   dstEnabled.outputQueue = srcFlowEntry.actions.enabled.outputQueue
   dstEnabled.outputDrop = srcFlowEntry.actions.enabled.outputDrop
   dstEnabled.egrMirror = srcFlowEntry.actions.enabled.egrMirror
   dstEnabled.dropTag = srcFlowEntry.actions.enabled.dropTag
   dstEnabled.outputNexthop = srcFlowEntry.actions.enabled.outputNexthop
   dstEnabled.outputInternalLoopback = \
               srcFlowEntry.actions.enabled.outputInternalLoopback

   dstActions.outputQueue = srcFlowEntry.actions.outputQueue
   dstActions.outputCpuQueue = srcFlowEntry.actions.outputCpuQueue
   dstActions.outputTrafficClass = srcFlowEntry.actions.outputTrafficClass
   dstActions.setVlanId = srcFlowEntry.actions.setVlanId
   dstActions.setVlanPri = srcFlowEntry.actions.setVlanPri
   dstActions.addVlanInnerId = srcFlowEntry.actions.addVlanInnerId
   dstActions.addVlanInnerPri = srcFlowEntry.actions.addVlanInnerPri
   dstActions.addVlanOuterId = srcFlowEntry.actions.addVlanOuterId
   dstActions.addVlanOuterPri = srcFlowEntry.actions.addVlanOuterPri
   dstActions.setEthSrc = srcFlowEntry.actions.setEthSrc
   dstActions.setEthDst = srcFlowEntry.actions.setEthDst
   dstActions.setIpSrc = srcFlowEntry.actions.setIpSrc
   dstActions.setIpDst = srcFlowEntry.actions.setIpDst
   dstActions.setIpTos = srcFlowEntry.actions.setIpTos
   dstActions.setL4Src = srcFlowEntry.actions.setL4Src
   dstActions.setL4Dst = srcFlowEntry.actions.setL4Dst
   dstActions.setIcmpType = srcFlowEntry.actions.setIcmpType
   dstActions.setIcmpCode = srcFlowEntry.actions.setIcmpCode
   dstActions.debugCapture = srcFlowEntry.actions.debugCapture
   dstActions.localIntf = srcFlowEntry.actions.localIntf
   dstActions.outputNexthopRec = srcFlowEntry.actions.outputNexthopRec
   dstActions.outputNexthopVrfName = srcFlowEntry.actions.outputNexthopVrfName

   for intf in srcFlowEntry.actions.outputIntf:
      dstActions.outputIntf[ intf ] = True

   for intf in srcFlowEntry.actions.ingrMirrorIntf:
      dstActions.ingrMirrorIntf[ intf ] = True

   for intf in srcFlowEntry.actions.egrMirrorIntf:
      dstActions.egrMirrorIntf[ intf ] = True

   for intf in srcFlowEntry.actions.outputNexthop:
      dstActions.outputNexthop[ intf ] = True

# ------------------------------------------------------------------------
# Context
#
# A context is created per FlowEntryConfigMode and stores (buffers) the
# commands till the user exits the mode or aborts the changes.
#
# If the flow entry exists already, the context contains an editable copy of
# the contents; else it contains a new (editable) copy
# --------------------------------------------------------------------------
class Context( object ):
   def __init__( self, flowHwConfig, hwStatus, flowName, directFlowHwConfig=None,
                 directFlowStatusDir=None ):
      self.flowHwConfig = flowHwConfig
      self.hwStatus = hwStatus
      self.directFlowHwConfig = directFlowHwConfig
      self.directFlowStatusDir = directFlowStatusDir
      self.flowEntryName = flowName
      self.m0 = Tac.Value( "OpenFlowTable::Match" )
      self.f0 = Tac.Value( "OpenFlowTable::MatchFieldSet" )
      self.a0 = Tac.Value( "OpenFlowTable::Actions" )
      self.e0 = Tac.Value( "OpenFlowTable::ActionSet" )
      self.ipProtoExplicit = False
      self.priority = 0
      self.priorityGroupType = \
          Tac.Type( "OpenFlowTable::PriorityGroupType" ).prioGroupTypeDefault
      if self.hwStatus.directFlowVfpEfpSupported:
         self.tableType = \
             Tac.Type( "OpenFlowTable::TableType" ).postLookup
      else:
         self.tableType = \
             Tac.Type( "OpenFlowTable::TableType" ).defaultTable
      self.cookie = 0
      self.idleTimeout = 0
      self.hardTimeout = 0
      self.e0.outputController = False
      # Default action for all flows is process Normally 
      self.e0.outputNormal = True
      self.mode_ = None

      # Used only by DirectFlow which has persistent flows by default. OpenFlow code
      # does not touch this parameter at all.
      self.persistent = True
      self.install = True

      if flowName in self.flowHwConfig.flowEntry:
         copyFlowEntry( self.flowHwConfig.flowEntry[ flowName ],
                        self.f0, self.m0, self.e0,
                        self.a0 )
         self.priority = self.flowHwConfig.flowEntry[ flowName ].priority
         self.priorityGroupType = \
             self.flowHwConfig.flowEntry[ flowName ].priorityGroupType
         self.tableType = \
             self.flowHwConfig.flowEntry[ flowName ].tableType
         self.cookie = self.flowHwConfig.flowEntry[ flowName ].cookie
         self.idleTimeout = self.flowHwConfig.flowEntry[ flowName ].idleTimeout
         self.hardTimeout = self.flowHwConfig.flowEntry[ flowName ].hardTimeout
         self.persistent = self.flowHwConfig.flowEntry[ flowName ].persistent
         self.installation = self.flowHwConfig.flowEntry[ flowName ].install

         # Work around to prevent ipProto config being removed automatically.
         # This gives similar behavior as ethType. Both ipProto and ethType are
         # set automatically for certain match conditions where these fields
         # are prerequisite fields.
         self.ipProtoExplicit = \
             self.flowHwConfig.flowEntry[ flowName ].match.matched.ipProto

   def modeIs( self, mode ):
      self.mode_ = mode

   # Function that checks if the match criteria is supported on the hardware based
   # on tableProfile. For a given attr it checks for the corresponding entry in 
   # matchField and returns. For example in T+ and T2, match on source mac is not
   # supported along with ipv6. So this would return field.ethSrc which is False
   def chkHwSupport( self, attr, field ):
      if attr == "intf" or attr == "cpu":
         return field.inIntf
      elif attr == "srcMac":
         return field.ethSrc
      elif attr == "dstMac":
         return field.ethDst
      elif attr == "vlan":
         return field.vlanId
      elif attr == "vlanInner":
         return field.vlanInnerId
      elif attr == "encapDot1q":
         return field.encapDot1q
      elif attr == "cos":
         return field.vlanPri
      elif attr == "ethType":
         return field.ethType
      elif attr == "srcIp":
         return field.ipSrc
      elif attr == "dstIp":
         return field.ipDst
      elif attr == "srcIpv6":
         return field.ip6Src
      elif attr == "dstIpv6":
         return field.ip6Dst
      elif attr == "tos":
         return field.ipTos
      elif attr == "proto":
         return field.ipProto
      elif attr == "sport":
         return field.l4Src
      elif attr == "dport":
         return field.l4Dst
      elif attr == "type" or attr == "typev6":
         return field.icmpType
      elif attr == "code" or attr == "codev6":
         return field.icmpCode
      elif attr == "urg":
         return field.tcpUrg
      elif attr == "ack":
         return field.tcpAck
      elif attr == "psh":
         return field.tcpPsh
      elif attr == "rst":
         return field.tcpRst
      elif attr == "syn":
         return field.tcpSyn
      elif attr == "fin":
         return field.tcpFin
      elif attr == "packetRes":
         return field.packetRes
      elif attr == "internalLoopback":
         return field.inputInternalLoopback
      else:
         return False

   # Function that checks if a match criteria that has been previously enabled is
   # supported based on tableProfile. For example in T+ and T2, match on ipv6 is
   # not supported along with source mac address (field.ethSrc == False). So if 
   # match on source mac address had been configured previously on the same flow 
   # (self.f0.ethSrc == True) this would return False
   def chkSupportedMatchFieldSet( self, field ):
      if ( ( ( self.f0.inIntf == True or self.f0.inCpuIntf == True ) and 
             field.inIntf == False ) or
           ( self.f0.ethSrc == True and field.ethSrc == False ) or
           ( self.f0.ethDst == True and field.ethDst == False ) or
           ( self.f0.vlanId == True and field.vlanId == False ) or
           ( self.f0.vlanPri == True and field.vlanPri == False ) or
           ( self.f0.vlanInnerId == True and field.vlanInnerId == False ) or
           ( self.f0.encapDot1q == True and field.encapDot1q == False ) or
           ( self.f0.ethType == True and field.ethType == False ) or
           ( self.f0.ipSrc == True and field.ipSrc == False ) or
           ( self.f0.ipDst == True and field.ipDst == False ) or
           ( self.f0.ip6Src == True and field.ip6Src == False ) or
           ( self.f0.ip6Dst == True and field.ip6Dst == False ) or
           ( self.f0.ipTos == True and field.ipTos == False ) or
           ( self.f0.ipProto == True and 
             ( field.ipProto == False or
               field.l4Src == False or field.l4Dst == False or
               field.icmpType == False or field.icmpCode == False ) ) or
           ( self.f0.tcpUrg == True and field.tcpUrg == False ) or
           ( self.f0.tcpAck == True and field.tcpAck == False ) or
           ( self.f0.tcpPsh == True and field.tcpPsh == False ) or
           ( self.f0.tcpRst == True and field.tcpRst == False ) or
           ( self.f0.tcpSyn == True and field.tcpSyn == False ) or
           ( self.f0.tcpFin == True and field.tcpFin == False ) or
           ( self.f0.packetRes == True and field.packetRes == False ) ):
         return False
      else:
         return True

   def chkIpv6MatchSupport( self, attr ):
      v6TableDesc = self.hwStatus.supportedTableDesc[ 'bindModeInterface' ].\
            tableDesc[ 'tableProfileIpv6Match' ]
      v6ProtoPortTableDesc = self.hwStatus.\
          supportedTableDesc[ 'bindModeInterface' ].\
          tableDesc[ 'tableProfileIpv6ProtoPortMatch' ]
      if self.f0.ip6Src or self.f0.ip6Dst:
         if ( self.chkHwSupport( attr, v6TableDesc.matchFields ) or 
              self.chkHwSupport( attr, v6ProtoPortTableDesc.matchFields ) ) :
            t0( " Matches ipv6 vlan or ipv6 Proto table " )
         else:
            self.mode_.addError( "Cannot configure this match with existing " \
                                 "source/destination IPv6 address" )
            return False
      elif attr == "srcIpv6" or attr == "dstIpv6":
         if ( self.chkSupportedMatchFieldSet( v6TableDesc.matchFields ) or
              self.chkSupportedMatchFieldSet( v6ProtoPortTableDesc.matchFields ) ) :
            t0( " Matches ipv6 vlan or ipv6 Proto table " )
         else:
            self.mode_.addError( "Cannot configure match on source/destination " \
                                 "IPv6 address in current flow" )
            return False
      return True

   # Function that sets values in flow entry based on the 'attr' set by the caller.
   # The same function is called for handling all match commands (with different
   # parameters of course).
   def setMatchValue( self, no, attr, value1, value2 ):
      neg = False
      if no is not None:
         neg = True

      # Adding the following condition to check if match on IPv6 is enabled and
      # if enabling the configuration is supported along with that only if this 
      # is not being replayed from the startupConfig
      if not self.mode_.session_.startupConfig():
         if not neg and not self.chkIpv6MatchSupport( attr ):
            return

      # If the ether type is set, ensure it is either IPv4 or the mask is zero for
      # matching on IPv4 fields
      ethTypeIpAttr = ['unknownIpV4SwitchedMulticast', 'unknownIpV4RoutedMulticast',
                       'srcIp', 'dstIp', 'tos', 'sport', 'dport',
                       'code', 'type', 'urg', 'ack', 'psh',
                       'rst', 'syn', 'fin']
      if attr in ethTypeIpAttr:
         # If we are setting an IP attribute and the eth type is not set, auto
         # set it to be IP (else leave it as we can have non-ip eth types match
         # IP fields with the intent of matching bits at the location the IP field
         # would be present.
         if not neg and not self.f0.ethType:
            # Set Ethtype = IP if an IP field is selected
            self.f0.ethType = not neg
            self.m0.ethType = ETH_P_IP
            self.m0.ethTypeMask = 0xffff


      # If the ether type is set, ensure it is either IPv6 or the mask is zero for
      # matching on IPv6 fields
      ethTypeIp6Attr = [ 'srcIpv6', 'dstIpv6', 'codev6', 'typev6' ]
      if attr in ethTypeIp6Attr:
         if not neg and not self.f0.ethType:
            # Set Ethtype = IPv6 if an IPv6 field is selected
            self.f0.ethType = not neg
            self.m0.ethType = ETH_P_IPV6
            self.m0.ethTypeMask = 0xffff

      if attr == "intf":
         self.f0.inIntf = not neg
         if not no:
            if isinstance( value1, MultiRangeRule.IntfList ):
               for i in value1.intfNames():
                  self.m0.inIntf[ i ] = True
            else:
               self.m0.inIntf[ value1.name ] = True
         else:
            if isinstance( value1, MultiRangeRule.IntfList ):
               for i in value1.intfNames():
                  del self.m0.inIntf[ i ]
            else:
               del self.m0.inIntf[ value1.name ]
         if len( self.m0.inIntf ) == 0:
            self.f0.inIntf = False
         else:
            self.f0.inIntf = True
      if attr == "cpu":
         self.f0.inCpuIntf = not neg
      if attr == "vlan":
         self.f0.vlanId = not neg
         self.m0.vlanId = value1
         if value2 is not None:
            self.m0.vlanIdMask = value2
         else:
            self.m0.vlanIdMask = 0xFFF
      if attr == "vlanInner":
         self.f0.vlanInnerId = not neg
         self.m0.vlanInnerId = value1
         if value2 is not None:
            self.m0.vlanInnerIdMask = value2
         else:
            self.m0.vlanInnerIdMask = 0xFFF
      if attr == "encapDot1q":
         self.f0.encapDot1q = not neg
      if attr == "ethType":
         self.f0.ethType = not neg
         self.m0.ethType = value1
         if value2 is None:
            self.m0.ethTypeMask = 0xffff
         else:
            self.m0.ethTypeMask = value2
      if attr == "unknownIpV4SwitchedMulticast":
         self.f0.unknownL2V4MulticastAddress = not neg
         self.m0.unknownL2V4MulticastAddress = not neg
      if attr == "unknownIpV4RoutedMulticast":
         self.f0.unknownL3V4MulticastAddress = not neg
         self.m0.unknownL3V4MulticastAddress = not neg
      if attr == "srcMac":
         self.f0.ethSrc = not neg
         if ( value1 == 'BRIDGE_MAC' ):
            self.f0.ethSrcBr = not neg
            self.m0.ethSrc = "00:00:00:00:00:00"
         else:
            self.f0.ethSrcBr = False
            self.m0.ethSrc = value1
         if value2 is not None:
            self.m0.ethSrcMask = value2
         else:
            self.m0.ethSrcMask = "ff:ff:ff:ff:ff:ff"
      if attr == "dstMac":
         self.f0.ethDst = not neg
         if ( value1 == 'BRIDGE_MAC' ):
            self.f0.ethDstBr = not neg
            self.m0.ethDst = "00:00:00:00:00:00"
         else:
            self.f0.ethDstBr = False
            self.m0.ethDst = value1
         if value2 is not None:
            self.m0.ethDstMask = value2
         else:
            self.m0.ethDstMask = "ff:ff:ff:ff:ff:ff"
      if attr == "cos":
         self.f0.vlanPri = not neg
         self.m0.vlanPri = value1
      if attr == "srcIp":
         self.f0.ipSrc = not neg
         if type( value1 ) is Tac.Type( "Arnet::IpAddrWithFullMask" ):
            self.m0.ipSrc = value1.address
            mask = value1.mask
            self.m0.ipSrcMask = socket.inet_ntoa(struct.pack(">L", mask))
         else:
            self.m0.ipSrc = value1
            if value2 is not None:
               self.m0.ipSrcMask = value2
            else:
               self.m0.ipSrcMask = "255.255.255.255"
      if attr == "dstIp":
         self.f0.ipDst = not neg
         if type( value1 ) is Tac.Type( "Arnet::IpAddrWithFullMask" ):
            self.m0.ipDst = value1.address
            mask = value1.mask
            self.m0.ipDstMask = socket.inet_ntoa(struct.pack(">L", mask))
         else:
            self.m0.ipDst = value1
            if value2 is not None:
               self.m0.ipDstMask = value2
            else:
               self.m0.ipDstMask = "255.255.255.255"
      def ip6AddrMaskFromLen( maskLen ):
         mask = ( 1 << 128 ) - ( 1 << 128 >> maskLen )
         maskBytes = []
         for i in range( 8 ):
            maskBytes.append( ( mask >> ( 16 * i ) ) & 0xffff )
         maskBytes.reverse()
         return Arnet.Ip6Addr( "%x:%x:%x:%x:%x:%x:%x:%x" % tuple( maskBytes ) )
      if attr == "srcIpv6":
         self.f0.ip6Src = not neg
         if type( value1 ) == Tac.Type( "Arnet::Ip6AddrWithMask" ):
            self.m0.ip6Src = value1.address
            self.m0.ip6SrcMask = ip6AddrMaskFromLen( value1.len )
         else:
            self.m0.ip6Src = value1
            if value2 is not None:
               self.m0.ip6SrcMask = value2
            else:
               self.m0.ip6SrcMask = \
                  Arnet.Ip6Addr( "ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff" )
      if attr == "dstIpv6":
         self.f0.ip6Dst = not neg
         if type( value1 ) == Tac.Type( "Arnet::Ip6AddrWithMask" ):
            self.m0.ip6Dst = value1.address
            self.m0.ip6DstMask = ip6AddrMaskFromLen( value1.len )
         else:
            self.m0.ip6Dst = value1
            if value2 is not None:
               self.m0.ip6DstMask = value2
            else:
               self.m0.ip6DstMask = \
                  Arnet.Ip6Addr( "ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff" )
      protoV6Values = [ IPPROTO_ICMPV6 ]
      if attr == "proto":
         if not neg and not self.f0.ethType:
            if value1 in protoV6Values:
               # Set EthType = IPv6 if an IPv6 proto selected
               self.f0.ethType = not neg
               self.m0.ethType = ETH_P_IPV6
               self.m0.ethTypeMask = 0xffff
            else:
               # Set Ethtype = IPv4
               self.f0.ethType = not neg
               self.m0.ethType = ETH_P_IP
               self.m0.ethTypeMask = 0xffff

         self.f0.ipProto = not neg
         self.m0.ipProto = value1
         self.ipProtoExplicit = not neg

         # If the protocol is ICMP, disable the L4 ports, set it to 0 and vice-versa
         if not neg:
            if self.m0.ipProto in ( IPPROTO_ICMP, IPPROTO_ICMPV6 ):
               self.f0.l4Src = False
               self.f0.l4Dst = False
               self.m0.l4Src = 0
               self.m0.l4Dst = 0
            elif self.m0.ipProto in ( IPPROTO_TCP, IPPROTO_UDP ):
               self.f0.icmpType = False
               self.f0.icmpCode = False
               self.m0.icmpType = 0
               self.m0.icmpCode = 0
      if attr == "tos":
         self.f0.ipTos = not neg
         self.m0.ipTos = value1
      if attr == "sport":
         if not neg:
            # If the protocol is not TCP or UDP, set to default ( TCP )
            if ( self.m0.ipProto != IPPROTO_TCP and \
                   self.m0.ipProto != IPPROTO_UDP ):
               self.f0.ipProto = not neg
               self.m0.ipProto = IPPROTO_TCP

         self.f0.l4Src = not neg
         self.m0.l4Src = value1
      if attr == "dport":
         if not neg:
            # If the protocol is not TCP or UDP, set to default (TCP)
            if ( self.m0.ipProto != IPPROTO_TCP and \
                      self.m0.ipProto != IPPROTO_UDP ):
               self.f0.ipProto = not neg
               self.m0.ipProto = IPPROTO_TCP

         self.f0.l4Dst = not neg
         self.m0.l4Dst = value1
      if attr == "type" or attr == "typev6":
         self.f0.icmpType = not neg
         self.m0.icmpType = value1
         if not neg:
            self.f0.ipProto = not neg
            if attr == "typev6":
               self.m0.ipProto = IPPROTO_ICMPV6
            else:
               self.m0.ipProto = IPPROTO_ICMP
      if attr == "code" or attr == "codev6":
         self.f0.icmpCode = not neg
         self.m0.icmpCode = value1
         if not neg:
            self.f0.ipProto = not neg
            if attr == "codev6":
               self.m0.ipProto = IPPROTO_ICMPV6
            else:
               self.m0.ipProto = IPPROTO_ICMP
      if not neg:
         if self.m0.ipProto in ( IPPROTO_ICMP, IPPROTO_ICMPV6 ) and \
            ( self.f0.l4Src == True or self.f0.l4Dst == True ):
            self.f0.l4Src = False
            self.f0.l4Dst = False
            self.m0.l4Src = 0
            self.m0.l4Dst = 0
         if self.m0.ipProto in ( IPPROTO_TCP, IPPROTO_UDP ) and \
            ( self.f0.icmpType == True or self.f0.icmpCode == True ):
            self.f0.icmpType = False
            self.f0.icmpCode = False
            self.m0.icmpType = 0
            self.m0.icmpCode = 0
      if attr == "urg":
         self.f0.tcpUrg = not neg
         self.m0.tcpUrg = not neg
         if not neg:
            self.f0.ipProto = not neg
            self.m0.ipProto = IPPROTO_TCP
      if attr == "ack":
         self.f0.tcpAck = not neg
         self.m0.tcpAck = not neg
         if not neg:
            self.f0.ipProto = not neg
            self.m0.ipProto = IPPROTO_TCP
      if attr == "psh":
         self.f0.tcpPsh = not neg
         self.m0.tcpPsh = not neg
         if not neg:
            self.f0.ipProto = not neg
            self.m0.ipProto = IPPROTO_TCP
      if attr == "rst":
         self.f0.tcpRst = not neg
         self.m0.tcpRst = not neg
         if not neg:
            self.f0.ipProto = not neg
            self.m0.ipProto = IPPROTO_TCP
      if attr == "syn":
         self.f0.tcpSyn = not neg
         self.m0.tcpSyn = not neg
         if not neg:
            self.f0.ipProto = not neg
            self.m0.ipProto = IPPROTO_TCP
      if attr == "fin":
         self.f0.tcpFin = not neg
         self.m0.tcpFin = not neg
         if not neg:
            self.f0.ipProto = not neg
            self.m0.ipProto = IPPROTO_TCP
      if ( not self.f0.l4Src ) and ( not self.f0.l4Dst ) and \
             ( not self.f0.icmpType ) and ( not self.f0.icmpCode ) and \
             ( not self.ipProtoExplicit ) and ( not self.f0.tcpUrg ) and \
             ( not self.f0.tcpAck ) and ( not self.f0.tcpPsh ) and \
             ( not self.f0.tcpRst ) and ( not self.f0.tcpSyn ) and \
             ( not self.f0.tcpFin ):
         self.f0.ipProto = False
         self.m0.ipProto = 0
      if attr == "packetRes":
         self.f0.packetRes = not neg
         if not neg:
            self.m0.packetRes = value1
      if attr == "internalLoopback":
         self.f0.inputInternalLoopback = not neg
      if attr == "trafficStreamSrc":
         self.f0.trafficStreamSrc = not neg
         if not neg:
            self.m0.trafficStreamSrc = value1
      if attr == "trafficStreamDst":
         self.f0.trafficStreamDst = not neg
         if not neg:
            self.m0.trafficStreamDst = value1
      if attr == 'vrfName':
         self.f0.vrfId = not neg
         if not neg:
            self.m0.vrfName = value1
         else:
            self.m0.vrfName = ''

   # Function that sets values in flow entry based on the 'attr' set by the caller.
   # The same function is called for handling all action commands (with different
   # parameters of course).
   def setActionValue( self, no, attr, value, value1=None, value2=None ):
      neg = False
      if no is not None:
         neg = True

      if attr == "vlan":
         self.e0.setVlanId = not neg
         self.a0.setVlanId = value
      if attr == "cos":
         self.e0.setVlanPri = not neg
         self.a0.setVlanPri = value
      if attr == "srcMac":
         self.e0.setEthSrc = not neg
         if ( value == 'BRIDGE_MAC' ):
            self.e0.setEthSrcBr = not neg
            self.a0.setEthSrc = "00:00:00:00:00:00"
         else:
            self.e0.setEthSrcBr = False
            self.a0.setEthSrc = value
      if attr == "dstMac":
         self.e0.setEthDst = not neg
         if ( value == 'BRIDGE_MAC' ):
            self.e0.setEthDstBr = not neg
            self.a0.setEthDst = "00:00:00:00:00:00"
         else:
            self.e0.setEthDstBr = False
            self.a0.setEthDst = value
      if attr == "tos":
         self.e0.setIpTos = not neg
         self.a0.setIpTos = value
      if attr == "flood":
         if neg:
            if ( self.e0.outputFlood==True ): 
               # if output interface is removed process normally
               self.e0.outputNormal = True 
         self.e0.outputFlood = not neg # Set to true or false based on no value
      if attr == "all":
         if neg:
            if ( self.e0.outputAll==True ): 
            # if output interface is removed process normally
               self.e0.outputNormal = True 
         self.e0.outputAll = not neg
      if attr == "nexthop":
         if neg:
            self.e0.outputNexthop = False
            self.a0.outputNexthopRec = False
            for i in self.a0.outputNexthop:
               del self.a0.outputNexthop[ i ]
            # add a delete all option here
         else:
            self.e0.outputNexthop = True
            # first delete all and then add the new stuff
            for i in self.a0.outputNexthop:
               del self.a0.outputNexthop[ i ]
            self.a0.outputNexthopRec = False
            for ipAddr in value2:
               self.a0.outputNexthop[ Arnet.IpGenAddr( ipAddr ) ] = True
            if value is not None:
               self.a0.outputNexthopRec = True
            if value1 is not None:
               self.a0.outputNexthopVrfName = value1
            else:
               self.a0.outputNexthopVrfName = IpLibConsts.DEFAULT_VRF
      if attr == "drop":
         # If not a 'no' command, set all other outputs to false as that signifies
         # a drop action; Process Normally for a "no" command
         if neg:
            self.e0.outputNormal = True
            self.e0.outputDrop = False
         else:
            self.e0.outputDrop = True
            self.e0.outputIntf = False
            self.e0.outputInIntf = False
            self.e0.outputNormal = False
            self.e0.outputNexthop = False
            self.e0.outputFlood = False
            self.e0.outputAll = False
            self.e0.outputController = False
            self.e0.outputLocal = False
            self.e0.outputQueue = False
            for intf in self.a0.outputIntf:
               del self.a0.outputIntf[ intf ]
      if attr == "trafficClass":
         # Directflow supports setting the traffic class while
         # OpenFlow instead supports setting the output queue directly.
         # We therefore overload the outputQueue action flag in the Directflow
         # case to represent the setTrafficClass action.
         # Try not to be confused by the setTrafficClass action flag - this
         # is currently only used by the P4 runtime feature and is never set
         # by CLI.
         self.e0.outputQueue = not neg
         self.a0.outputTrafficClass = value
      if attr == "outIntf":
         if not no:
            if isinstance( value, MultiRangeRule.IntfList ):
               for i in value.intfNames():
                  self.a0.outputIntf[ i ] = True
            else:
               self.a0.outputIntf[ value.name ] = True
         else:
            if isinstance( value, MultiRangeRule.IntfList ):
               for i in value.intfNames():
                  del self.a0.outputIntf[ i ]
            else:
               del self.a0.outputIntf[ value.name ]

         if len( self.a0.outputIntf ) == 0:
            if ( self.e0.outputIntf==True ) and not self.e0.outputInternalLoopback:
               # if output interface is removed process normally
               self.e0.outputNormal = True 
            self.e0.outputIntf = False
         else:
            self.e0.outputIntf = True
      if attr == "local":
         if neg:
            self.a0.debugCapture = False
            self.a0.localIntf = ""
            # Process normally if this is toggled off
            if self.e0.outputLocal == True:
               self.e0.outputNormal = True
         else:
            if value is not None:
               self.a0.debugCapture = True
            if value1 is not None:
               self.a0.localIntf = value1
         self.e0.outputLocal = not neg

      if attr == "internalLoopback":
         if neg:
            # if output interface is removed process normally
            self.e0.outputNormal |= self.e0.outputInternalLoopback
         self.e0.outputInternalLoopback = not neg

      # BUG91907: For the output cpu queue <n> hidden command.
      if attr == "cpuQueue":
         self.a0.outputCpuQueue = value

      # If Any output Action is set then process normal flag is false
      if ( self.e0.outputFlood or
           self.e0.outputAll or
           self.e0.outputLocal or
           self.e0.outputNexthop or
           len( self.a0.outputIntf ) != 0 or
           self.e0.outputInternalLoopback ):
         self.e0.outputNormal = False

      if attr == "ingrM":
         if not no:
            if isinstance( value, MultiRangeRule.IntfList ):
               for i in value.intfNames():
                  self.a0.ingrMirrorIntf[ i ] = True
            else:
               self.a0.ingrMirrorIntf[ value.name ] = True
         else:
            if isinstance( value, MultiRangeRule.IntfList ):
               for i in value.intfNames():
                  del self.a0.ingrMirrorIntf[ i ]
            else:
               del self.a0.ingrMirrorIntf[ value.name ]

         if len( self.a0.ingrMirrorIntf ) == 0:
            self.e0.ingrMirror = False
         else:
            self.e0.ingrMirror = True

      if attr == "egrM":
         if not no:
            if isinstance( value, MultiRangeRule.IntfList ):
               for i in value.intfNames():
                  self.a0.egrMirrorIntf[ i ] = True
            else:
               self.a0.egrMirrorIntf[ value.name ] = True
         else:
            if isinstance( value, MultiRangeRule.IntfList ):
               for i in value.intfNames():
                  del self.a0.egrMirrorIntf[ i ]
            else:
               del self.a0.egrMirrorIntf[ value.name ]

         if len( self.a0.egrMirrorIntf ) == 0:
            self.e0.egrMirror = False
         else:
            self.e0.egrMirror = True

      if attr == "vlanInner":
         self.e0.addVlanInner = not neg
         self.a0.addVlanInnerId = value
         if value1 is not None:
            self.a0.addVlanInnerPri = value1
         else:
            self.a0.addVlanInnerPri = 0

      if attr == "vlanOuter":
         self.e0.addVlanOuter = not neg
         self.a0.addVlanOuterId = value
         if value1 is not None:
            self.a0.addVlanOuterPri = value1
         else:
            self.a0.addVlanOuterPri = 0

      if attr == "removeVlanInner":
         self.e0.removeVlanInner = not neg

      if attr == "dropTag":
         self.e0.dropTag = not neg

   def setOutputDrop ( self ):
      if ( self.e0.outputIntf or self.e0.outputInIntf or
           self.e0.outputNormal or self.e0.outputFlood or
           self.e0.outputAll or self.e0.outputNexthop or 
           self.e0.outputInternalLoopback ):
         # Set the outputDrop to False if we are sending the packet
         # anywhere other than the CPU/Local interface
         self.e0.outputDrop = False
      else:
         # If there is no explicit output specified or if the flow sends
         # traffic to CPU and doesn't send traffic elsewhere, outputDrop=True
         self.e0.outputDrop = True

   def commit( self ):
      # Set the outputDrop action based on all output actions in the flow
      self.setOutputDrop()

      # Replace the entry if it exists; Don't worry about multiple sessions
      # overwriting the same entry etc.
      if self.flowEntryName in self.flowHwConfig.flowEntry:
         flow = self.flowHwConfig.flowEntry[ self.flowEntryName ]
         self.m0.matched = self.f0
         self.a0.enabled = self.e0
         flow.match = self.m0
         flow.actions = self.a0
         flow.idleTimeout = self.idleTimeout
         flow.hardTimeout = self.hardTimeout
         flow.priority = self.priority
         flow.priorityGroupType = self.priorityGroupType
         flow.tableType = self.tableType
         flow.persistent = self.persistent
         flow.install = self.install
         flow.changeCount += 1
      else:
         self.m0.matched = self.f0
         self.a0.enabled = self.e0
         self.flowHwConfig.newFlowEntry( self.flowEntryName, self.m0, self.a0,
                                         self.priority, self.priorityGroupType,
                                         self.tableType,
                                         self.cookie, self.idleTimeout,
                                         self.hardTimeout, self.persistent,
                                         self.install )
         self.flowHwConfig.flowEntry[ self.flowEntryName ].source = 'config'

      # Remove non-persistent expired flows from config.
      if self.directFlowStatusDir:
         for sliceHwStatus in self.directFlowStatusDir.itervalues():
            for flowName in sliceHwStatus.expiredFlow:
               hwConfigFlow = self.directFlowHwConfig.flowEntry.get( flowName )
               # Config may have been deleted
               if ( hwConfigFlow ):
                  name = hwConfigFlow.flowName
                  flowConfig = self.flowHwConfig.flowEntry.get( name )
                  if ( flowConfig and ( not flowConfig.persistent ) ):
                     flowStatus = sliceHwStatus.flowStats.get( flowName )
                     if ( flowStatus and
                          ( flowStatus.status == "flowExpiredHard" or
                            flowStatus.status == "flowExpiredIdle" ) ):
                        del self.flowHwConfig.flowEntry[ name ]
            
      else: 
         for flowName in self.hwStatus.expiredFlow:
            flowConfig = self.flowHwConfig.flowEntry.get( flowName )
            if ( flowConfig and ( not flowConfig.persistent ) ):
               flowStatus = self.hwStatus.flowStats.get( flowName )
               if ( flowStatus and ( flowStatus.status == "flowExpiredHard" or
                                     flowStatus.status == "flowExpiredIdle" ) ):
                  del self.flowHwConfig.flowEntry[ flowName ]

   def abort( self ):
      self.f0 = None
      self.m0 = None
      self.e0 = None
      self.a0 = None
