#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from ArnetModel import IpAddrAndPort
import BasicCli
import CliParser
import CliPlugin.IntfCli as IntfCli
from CliPlugin.EthIntfCli import isVEos
from CliPlugin.OpenFlowModel import OpenFlowSummary, OpenFlowPorts, OpenFlowFlows
from CliPlugin.OpenFlowModel import IpAddrAndBitMask, MacAddrAndBitMask
from CliPlugin.OpenFlowModel import Ip6AddrAndBitMask
from CliPlugin.OpenFlowModel import MacAddr, IpAddr
from CliPlugin.OpenFlowModel import OpenFlowActivityStats, OpenFlowQ, OpenFlowQStats
from CliPlugin.OpenFlowModel import OpenFlowTableProfile, OpenFlowTablesProfile
from CliPlugin.OpenFlowModel import OpenFlowTableProfileMatch
from CliPlugin.OpenFlowModel import OpenFlowTableProfileAction
from CliPlugin.OpenFlowModel import OpenFlowGroups
import CliPlugin.TechSupportCli
import CliMode.OpenFlow
import ConfigMount
import LazyMount
import MultiRangeRule
import Tac
import Toggles.OpenFlowToggleLib
import Tracing

t0 = Tracing.Handle( "OpenFlowCli" ).trace0

config = None
counterConfig = None
status = None
flowModeConfig = None
hwStatus = None
openFlowHwConfig = None
directFlowHwConfig = None
flowAndNormalHwConfig = None
fm = Tac.Type( "OpenFlowTable::FlowMode" )
counterControl = None
hwEpochStatus = None
mirroringConfig = None
entityMib = None

def _openFlowSessionCheckpoint( mode ):
   sessionCheckpoint = mode.session.sessionData( 'OpenFlowCli.sessionCheckpoint',
                                                 None )
   if sessionCheckpoint is None:
      sessionCheckpoint = {}
      mode.session.sessionDataIs( 'OpenFlowCli.sessionCheckpoint',
                                  sessionCheckpoint )
   return sessionCheckpoint

def openFlowSupportedGuard( mode, token ):
   if hwStatus.openFlowSupported:
      return None
   if Toggles.OpenFlowToggleLib.toggleOpenFlowMultiTableModeEnabled():
      return None
   return CliParser.guardNotThisPlatform

def openFlowVeosGuard( mode, token ):
   if isVEos():
      return CliParser.guardNotThisPlatform
   return None

def multitableGuard( mode, token ):
   if Toggles.OpenFlowToggleLib.toggleOpenFlowMultiTableModeEnabled():
      return CliParser.guardNotThisPlatform
   return None

def forwardingPipelineGuard( mode, token ):
   if isVEos():
      return CliParser.guardNotThisPlatform
   if Toggles.OpenFlowToggleLib.toggleOpenFlowMultiTableModeEnabled():
      return CliParser.guardNotThisPlatform
   return None

# Add or remove hidden mirror session for faucet
# Currently, mirror infra requires a session to be created ( no hw
# resource allocated at this time ) before a mirror source can be
# bound to a mirror target. We choose to pre-create hidden session
# here to avoid multi-writer issue. When mirror adds infra to bind
# mirror source to target directly with need seesion creation, this
# pre-creation can be removed.
def addOrRemoveMirrorSession( intfName, add=True ):
   if not Toggles.OpenFlowToggleLib.toggleOpenFlowMultiTableModeEnabled():
      return
   sessionName = intfName.replace( '/', '-' )
   cfg = mirroringConfig
   if add:
      if sessionName in cfg.session:
         return
      session = cfg.session.newMember( sessionName )
      session.targetIntf.newMember( intfName )
      session.destIntf = intfName
      session.isHiddenSession = True
      session.owner = "OpenFlow"
   else:
      if sessionName not in cfg.session:
         return
      session = cfg.session.newMember( sessionName )
      del session.targetIntf[ intfName ]
      del cfg.session[ sessionName ]

# --------------------------------------------------------------------------
# OpenFlow mode
# --------------------------------------------------------------------------
class OpenFlowConfigMode( CliMode.OpenFlow.OpenFlowMode, BasicCli.ConfigModeBase ):

   name = "OpenFlow configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CliMode.OpenFlow.OpenFlowMode.__init__( self, param = " " )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoOpenFlowConfigMode( mode ):
   childMode = mode.childMode( OpenFlowConfigMode )
   mode.session_.gotoChildMode( childMode )
   if flowModeConfig.flowMode == fm.flowModeDirect:
      mode.addWarning( "Enabling OpenFlow will cause DirectFlow to be disabled" )

def clearOpenFlowConfig( mode ):
   # This is triggered when the user issues 'no openflow'. Need to cleanup all config
   config.enabled = False
   if flowModeConfig.flowMode in ( fm.flowModeFlowAndNormal, fm.flowModeOpenFlow ):
      flowModeConfig.flowMode = fm.flowModeNone
   for intf in config.bindInterface:
      addOrRemoveMirrorSession( intf, False )
   config.description = ""
   config.controller.clear()
   config.bindVlan.clear()
   config.bindInterface.clear()
   config.bindMode = "bindModeInterface"
   config.keepalivePeriod = 10
   config.tableProfile = "tableProfileFullMatch"
   config.recircInterface = ""
   config.routedVlan.clear()
   config.defaultFlowEntryActions = config.defaultFlowEntryActionsController()
   config.shellCommandAllowed = False
   config.minimumVersion13 = False
   config.tracerTlv = False
   config.debugModeEnabled = False
# ------------------------------------------------------------------------------
# [no|default] shutdown
# ------------------------------------------------------------------------------
def setEnable( mode, no ):
   t0( "shutdown command, no = %s" % no )
   if no is None or no == 'default':
      # Shutdown
      config.enabled = False
      ofModes = ( fm.flowModeFlowAndNormal, fm.flowModeOpenFlow )
      if flowModeConfig.flowMode in ofModes:
         flowModeConfig.flowMode = fm.flowModeNone
   else:
      # When OpenFlow is enabled, if multiTableMode feature-toggle is set
      # 1. Set flowMode = "openFlowMode"
      # 2. set bindmode = "bindModeInterface"
      # 3. Set pipelineMode = "multiTableMode"
      # 4. Set default table miss flow entry = defaultFlowEntryActionsNone
      #    which translates to "drop"
      if Toggles.OpenFlowToggleLib.toggleOpenFlowMultiTableModeEnabled():
         flowAndNormalHwConfig.enabled = False
         flowModeConfig.flowMode = fm.flowModeOpenFlow
         config.pipelineMode = "multiTableMode"
         config.bindMode = "bindModeInterface"
         config.forwardingPipeline = "flow"
         config.defaultFlowEntryActions = config.defaultFlowEntryActionsNone()
      else:
         config.pipelineMode = "singleTableMode"

         if config.forwardingPipeline == "flowAndNormal":
            flowAndNormalHwConfig.enabled = True
            config.bindMode = "bindModeDirect"
            flowModeConfig.flowMode = fm.flowModeFlowAndNormal
         else:
            flowAndNormalHwConfig.enabled = False
            flowModeConfig.flowMode = fm.flowModeOpenFlow

      if isVEos():
         flowAndNormalHwConfig.enabled = False
         flowModeConfig.flowMode = fm.flowModeOpenFlow
         config.forwardingPipeline = "flow"

      # No Shutdown
      config.enabled = True

   if config.enabled:
      if config.bindMode == "bindModeInterface":
         mode.addWarning( "In interface bind mode, traffic on unbound interfaces"
                          " is not supported" )
      if config.bindMode == "bindModeVlan": 
         mode.addWarning( "In VLAN bind mode, traffic on unbound VLANs is "
                          "not supported" )

# -----------------------------------------------------------------------------------
# [no] description LINE
# -----------------------------------------------------------------------------------
def setDesc( mode, desc="" ):
   t0( "description command, desc = %s" % desc )
   config.description = desc

# -----------------------------------------------------------------------------------
# [no] datapath ID
# -----------------------------------------------------------------------------------
def setDpid( mode, dpidType=Tac.Type( "OpenFlow::DpidType" ).systemMac, dpid=0 ):
   t0( "dpid command, dpidType = %s,  dpid = %d" % (dpidType, dpid ) )
   config.dpidType = dpidType
   config.dpid = dpid

# -----------------------------------------------------------------------------------
# [no] controller tcp:A.B.C.D:N
# -----------------------------------------------------------------------------------
class ControllerConfigMode( CliMode.OpenFlow.ControllerMode, 
      BasicCli.ConfigModeBase ):
   name = "OpenFlow Controller Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, controller ):
      CliMode.OpenFlow.ControllerMode.__init__( self, param=controller  )
      BasicCli.ConfigModeBase.__init__( self, parent, session, controller )

def isValidIpAddr( addr ):
   a = addr.split( '.' )
   return len(a) == 4 and \
       len([v for v in a if len(v) == 0 or int(v) < 0 or int(v) > 255]) == 0

def canonicalizeController( controller ):
   c = controller.split( ':' )
   if len(c) == 3 and c[0] == "tcp" and \
          isValidIpAddr( c[1] ) and c[2] and int( c[2] ) < 0x10000:
      # strip leading zeros
      return "tcp:" + \
          ".".join( [ str( int( a ) ) for a in c[1].split('.') ] ) + \
          ':' + str( int( c[2] ) )
   else:
      return None

def gotoControllerMode( mode, controller ):
   t0( "controller command, controller = %s" % controller )
   con = canonicalizeController( controller )
   if con is None:
      mode.addError( "Invalid controller specification" )
      return
   if not config.controller.has_key( con ):
      maxControllers = 8
      if len(config.controller) >= maxControllers:
         mode.addError( "Maximum number of controllers is " + 
                                      str(maxControllers) )
         return
      config.newController( con )
   childMode = mode.childMode( ControllerConfigMode, controller=con )
   mode.session_.gotoChildMode( childMode )

def noController( mode, controller=None ):
   t0( "no controller command, controller = %s" % controller )
   if controller is None:
      config.controller.clear()
   else:
      con = canonicalizeController( controller )
      if con is None:
         mode.addError( "Invalid controller specification" )
         return
      del config.controller[ con ]

def setAuxConnection( mode, no ):
   t0( "auxiliary connection udp, no = %s" % no )
   controller = config.controller[ mode.controller ]
   controller.enableAuxConnection = not bool( no )

# -----------------------------------------------------------------------------------
# [no] controller version minimum 1.3
# -----------------------------------------------------------------------------------
def minimumVersion13( mode ):
   t0( "Set minimum OF version to be 1.3" )
   if status.controller:
      for c in status.controller:
         if status.controller[ c ].negotiatedVersion == "Version 1.0":
            mode.addError( "Controller connected at 1.0, cannot enable "
                           "this feature" ) 
            return
   config.minimumVersion13 = True

def noMinimumVersion13( mode ):
   t0( "Unset minimum OF version to be 1.3" )
   config.minimumVersion13 = False

# -----------------------------------------------------------------------------------
# [no] exception ttl expired action controller
# -----------------------------------------------------------------------------------
def setExpiredTtlFlow( mode, no ):
   t0( "expired ttl flow, no =", no )
   config.invalidTtlFlow = not bool( no )

# -----------------------------------------------------------------------------------
# forwarding pipeline { flow | flow-and-normal }
# -----------------------------------------------------------------------------------
def setForwardingPipeline( mode, forwardingPipeline="flow" ):
   t0( "forwarding pipeline command, forwardingPipeline= %s" \
          % forwardingPipeline )
   if forwardingPipeline == "flow-and-normal" :
      if config.enabled:
         flowAndNormalHwConfig.enabled = True
         flowModeConfig.flowMode = fm.flowModeFlowAndNormal
      config.forwardingPipeline = "flowAndNormal"
      config.bindMode = "bindModeDirect"
   else:
      if config.enabled:
         flowAndNormalHwConfig.enabled = False
         flowModeConfig.flowMode = fm.flowModeOpenFlow
      config.forwardingPipeline = "flow"
      config.bindMode = "bindModeInterface"

# -----------------------------------------------------------------------------------
# bind mode interface|monitor|vlan|direct
# -----------------------------------------------------------------------------------
def setBindMode( mode, bindMode="interface" ):
   t0( "bind mode command, mode = %s" % bindMode )
   if config.forwardingPipeline == "flowAndNormal" :
      mode.addError( 'Command not supported in this forwarding pipeline' )
      return 
   if bindMode == "vlan":
      config.bindMode = "bindModeVlan"
      if config.enabled: 
         mode.addWarning( "In VLAN bind mode, traffic on unbound VLANs is not "
                          "supported" )
   elif bindMode == "monitor":
      config.bindMode = "bindModeMonitor"
   else:
      config.bindMode = "bindModeInterface"
      if config.enabled: 
         mode.addWarning( "In interface bind mode, traffic on unbound interfaces " 
                          "is not supported" )

# -----------------------------------------------------------------------------------
# [no] bind vlan VLANID
# -----------------------------------------------------------------------------------
def setBindVlan( mode, vlanSet ):
   t0( "bind vlan command, vlanSet = %s" % vlanSet )
   for vid in vlanSet.ids:
      config.bindVlan[ vid ] = True

def noBindVlan( mode, vlanSet=None ):
   t0( "no bind vlan command, vlanSet = %s" % vlanSet )
   if vlanSet is None:
      config.bindVlan.clear()
   else:
      for vid in vlanSet.ids:
         try:
            del config.bindVlan[ vid ]
         except KeyError:
            pass

# -----------------------------------------------------------------------------------
# [no] bind interface INTF
# -----------------------------------------------------------------------------------
def setBindInterface( mode, intf ):
   t0( "bind interface command, intf = %r" % intf )
   if isinstance( intf, MultiRangeRule.IntfList ):
      for i in intf.intfNames():
         config.bindInterface[ i ] = True
         addOrRemoveMirrorSession( i, True )
   else:
      config.bindInterface[ intf.name ] = True
      addOrRemoveMirrorSession( intf.name, True )

def noBindInterface( mode, intf=None ):
   t0( "no bind interface command, intf = %r" % intf )
   if intf is None:
      config.bindInterface.clear()
   elif isinstance( intf, MultiRangeRule.IntfList ):
      for i in intf.intfNames():
         if i in config.bindInterface:
            del config.bindInterface[ i ]
            addOrRemoveMirrorSession( i, False )
   else:
      if intf.name in config.bindInterface:
         del config.bindInterface[ intf.name ]
         addOrRemoveMirrorSession( intf.name, False )

# -----------------------------------------------------------------------------------
# [no] routing vlan VLANID|untagged routed-vlan VLANID
# -----------------------------------------------------------------------------------
def setRoutingVlan( mode, vlanId, routedVlanId ):
   t0( "routing vlan command, vlan = %s, routed-vlan = %s" % (vlanId, routedVlanId) )
   config.routedVlan[vlanId.id] = routedVlanId.id

def noRoutingVlan( mode, vlanId=None ):
   t0( "no routing vlan command, vlan = %s" % vlanId )
   if vlanId is None:
      config.routedVlan.clear()
   else:
      try:
         del config.routedVlan[vlanId.id]
      except KeyError:
         pass

# -----------------------------------------------------------------------------------
# [no] routing recirculation-interface INTERFACE
# -----------------------------------------------------------------------------------
def setRoutingRecircIntf( mode, intf ):
   t0( "routing recirculation-interface command, intf = %s" % intf )
   config.recircInterface = intf.name

def noRoutingRecircIntf( mode ):
   t0( "no routing recirculation-interface command" )
   config.recircInterface = ""

def noRouting( mode ):
   t0( "no routing command" )
   config.recircInterface = ""
   config.routedVlan.clear()

# -----------------------------------------------------------------------------------
# keepalive SECONDS
# -----------------------------------------------------------------------------------
def setKeepalive( mode, period ):
   t0( "keepalive command, period = %s" % period )
   config.keepalivePeriod = period

def noKeepalive( mode ):
   t0( "no keepalive command" )
   config.keepalivePeriod = 10

# -----------------------------------------------------------------------------------
# profile NAME
# -----------------------------------------------------------------------------------
def setTableProfile( mode, profile="full-match" ):
   t0( "profile command, profile = %s" % profile )
   if profile == "l2-match":
      config.tableProfile = "tableProfileL2Match"
   elif profile == "vxlan-match":
      config.tableProfile = "tableProfileVxlanMatch"
   elif profile == "auto":
      config.tableProfile = "tableProfileAutoMatch"
   else :
      config.tableProfile = "tableProfileFullMatch"

# -----------------------------------------------------------------------------------
# default-action controller|drop
# -----------------------------------------------------------------------------------
def defaultDropGuard( mode, token ):
   if isVEos():
      return None

   mode = Tac.Type( 'HwEpoch::RestrictionStatus::Mode' )
   try:
      if hwEpochStatus.feature[ 'oldAclsTcam' ].mode == mode.enabled:
         return None
   except KeyError:
      pass
   return CliParser.guardNotThisPlatform

def setDefaultAction( mode, action="controller" ):
   t0( "default-action command, action = %s" % action )
   if action == "controller":
      actions = config.defaultFlowEntryActionsController()
   else:
      actions = config.defaultFlowEntryActionsNone()
   config.defaultFlowEntryActions = actions

# -----------------------------------------------------------------------------------
# shell-command allowed
# -----------------------------------------------------------------------------------
def setShellCommandAllowed( mode ):
   t0( "shell-command allowed command" )
   config.shellCommandAllowed = True

def noShellCommandAllowed( mode  ):
   t0( "no shell-command allowed command" )
   config.shellCommandAllowed = False

# -----------------------------------------------------------------------------------
# [no] extension tlv tracer
# -----------------------------------------------------------------------------------
def setTracerTlv( mode ):
   t0( "extension tlv tracer" )
   config.tracerTlv = True

def noTracerTlv( mode  ):
   t0( "no extension tlv tracer" )
   config.tracerTlv = False

# -----------------------------------------------------------------------------------
# show openflow
# -----------------------------------------------------------------------------------
# def runShowCommand( command, sysname ):
#    sys.stdout.write( Tac.run( ["OpenFlowShow", "--sysname=%s" % sysname, command],
#                              stdout=Tac.CAPTURE ) )
def showSummary( mode ):
   ret = OpenFlowSummary()
   try:
      flowStats = hwStatus.flowStats
      defaultFlowEntryName = openFlowHwConfig.defaultFlowEntryName
      active = ( flowStats[ defaultFlowEntryName ].status
                 == "flowCreated" )
   except KeyError:
      active = False

   ret.enabled = config.enabled
   ret.description = status.description
   ret.datapathID = status.dpid
   ret.controllers = []
   ret.controllersInfo = []
   ret.connectedCtrl = []
   ret.passiveControllerConfig = []
   ret.passiveControllerStatus = []
   for c in config.controller:
      ipPort = c.split( ':' )
      ipAndPort = IpAddrAndPort()
      ipAndPort.ip = ipPort[1]
      ipAndPort.port = int( ipPort[2] )
      ret.controllers.append( ipAndPort )
      controllerCfg = OpenFlowSummary.ControllerConfig()
      controllerCfg.controllerAddr = ipAndPort
      controllerCfg.auxEnabled = config.controller[ c ].enableAuxConnection
      controllerCfg.connectionMode = config.controller[ c ].connectionMode
      if config.controller[ c ].connectionMode == 'active':
         ret.controllersInfo.append( controllerCfg )
      else:
         ret.passiveControllerConfig.append( controllerCfg )
   ctrlStatus = status.controller
   if ctrlStatus:
      for cin in ctrlStatus:
         c = ctrlStatus.get( cin )
         if c is None:
            continue
         ipPort = cin.split( ':' )
         ipAndPort = IpAddrAndPort()
         try:
            ipAndPort.ip = ipPort[ 1 ]
            ipAndPort.port = int( ipPort[ 2 ] )
         except ValueError:
            continue
         connectedCtrl = OpenFlowSummary.ConnectedController()
         connectedCtrl.controllerAddr = ipAndPort
         connectedCtrl.controllerRole = c.role
         connectedCtrl.auxConnected = c.auxChannelConnected
         connectedCtrl.negotiatedVersion = c.negotiatedVersion
         ret.connectedCtrl.append( connectedCtrl )
   if config.minimumVersion13:
      ret.minimumVersion = "1.3"
   ret.connectSuccessCount = status.connectSuccessCount
   ret.keepalivePeriod = int( config.keepalivePeriod )
   for controller in status.passiveController:
      c = controller.split( ':' )
      ipAndPort = IpAddrAndPort()
      try:
         ipAndPort.ip = c[ 1 ]
         ipAndPort.port = int( c[ 2 ] )
      except ValueError:
         continue
      controllerModel = OpenFlowSummary.ConnectedController()
      controllerModel.controllerAddr = ipAndPort
      controllerModel.auxConnected = False
      ret.passiveControllerStatus.append( controllerModel )
   ret.flowTableState = active
   if hwStatus.tableProfile == "tableProfileL2Match" :
      ret.flowTableProfile = "l2-match"
   elif hwStatus.tableProfile == "tableProfileVxlanMatch":
      ret.flowTableProfile = "vxlan-match"
   elif hwStatus.tableProfile == "tableProfileAutoMatch":
      ret.flowTableProfile = "auto"
   else :
      ret.flowTableProfile = "full-match"
   if hwStatus.forwardingPipeline == "flowAndNormal":
      ret.forwardingPipeline = "flow-and-normal"
   else:
      ret.forwardingPipeline = "flow"

   bindMode = OpenFlowSummary.BindModeEntries()
   bindMode.bindVlan = ''
   bindMode.nativeVlan = ''
   bindMode.bindInterfaces = []
   if hwStatus.bindMode == "bindModeDirect":      
      bindMode.bindMode = "None"
   else:
      bindMode.bindMode = hwStatus.bindMode
   if hwStatus.bindMode == "bindModeVlan":
      bindMode.bindVlan = \
      MultiRangeRule.multiRangeToCanonicalString( hwStatus.bindVlan ) \
          or "<none>"
      bindMode.nativeVlan = str ( openFlowHwConfig.nativeVlan \
                                     or "<none>" ) 
   elif hwStatus.bindMode == "bindModeInterface":
      for i in hwStatus.bindInterface:
         bindMode.bindInterfaces.append( i )
   ret.bindMode = bindMode
   rvlans = sorted( hwStatus.routedVlan.items() )
   if rvlans:
      ret.ipRoutingState = True
      retRvlans = OpenFlowSummary.RvlanEntries()
      retRvlans.rvlanEntry = []
      retRvlans.rIntf = openFlowHwConfig.recircInterface
      for v, rv in rvlans:
         e = OpenFlowSummary.RvlanEntries.Entries()
         e.vlan = v
         e.rvlan = rv
         retRvlans.rvlanEntry.append( e )
      ret.rvlans = retRvlans
   else:
      ret.ipRoutingState = False
   ret.shellCommandAllowed = status.shellCommandAllowed
   ret.expiredTtlOutputController = status.invalidTtlFlow
   ret.totalPacketCount = hwStatus.totalPacketCount
   return ret
    
def showPorts( mode ):
   ret = OpenFlowPorts()
   ret.portInterface = []
# portInterface & mirrorInterface
   pi = dict( status.portInterface.items() )
   for port in pi:
      entry = OpenFlowPorts.Entry()
      entry.port = port
      entry.intf = pi[ port ]
      entry.mirror = True if port in status.portMirrorDest.items() else False 
      ret.portInterface.append( entry )
   return ret

def getFlowPriority( flowName ):
   try:
      flow = openFlowHwConfig.flowEntry[ flowName ]
      if Toggles.OpenFlowToggleLib.toggleOpenFlowMultiTableModeEnabled():
         tablePrio = 100
         if flow.match.matched.metaData:
            # Table 0 should be at top
            tablePrio=100-flow.match.metaData 
         return tablePrio, flow.priority + 1, flowName
      else:
         return flow.priority + 1, flowName
   except KeyError:
      return ( -1, flowName )

def showGroups( mode, groupId=None ):
   ret = OpenFlowGroups()
   ret.groups = []
   statusStrings = { "groupCreated" : "Programmed",
                     "groupDeleted" : "Deleted",
                     "groupRejectedBadAction" : "Rejected due to "
                                                "invalid actions",
                     "groupRejectedHwTableFull" : "Rejected due to "
                                                  "insufficient resources",
                     "groupRejectedOther" : "Rejected" }
   if groupId:
      groups = [ groupId ]
   else:
      groups = sorted( openFlowHwConfig.groupEntry )

   for groupIdx in groups:
      group = openFlowHwConfig.groupEntry.get( groupIdx, None )
      if group:
         stats = hwStatus.groupStats.get( groupIdx, None )
         groupModel = OpenFlowGroups.Group()
         groupModel.groupId = groupIdx
         groupModel.buckets = []
         groupModel.flows = []
         bucketId = 0
         for bkt in group.bucket.values():
            bktModel = OpenFlowGroups.Group.Bucket()
            bktModel.bucketId = bucketId
            bktModel.outInterfaces = []
            bucketId += 1
            if bkt.action.enabled.setEthSrc:
               mac = MacAddr()
               mac.mac = bkt.action.setEthSrc
               bktModel.macSrc = mac
            if bkt.action.enabled.setEthDst:
               mac = MacAddr()
               mac.mac = bkt.action.setEthDst
               bktModel.macDst = mac
            if bkt.action.enabled.setVlanId:
               bktModel.vlan = bkt.action.setVlanId
            bktModel.ttlDec = bkt.action.enabled.decNwTtl
            if bkt.action.enabled.outputIntf:
               for i in bkt.action.outputIntf:
                  bktModel.outInterfaces.append( i )
            groupModel.buckets.append( bktModel )
         groupModel.status = "Pending"
         if stats:
            groupModel.status = statusStrings.get( stats.status, "Rejected" )
            for flow in stats.referringFlows.keys():
               groupModel.flows.append( flow )
         ret.groups.append( groupModel )
   return ret

def getCounters( mode, flowName=None ):
   sessionSnapshot = None
   globalSnapshot = None
   fStats = None

   sessionCheckpoint = _openFlowSessionCheckpoint( mode )
   checkpoint = sessionCheckpoint.get( flowName )
   if checkpoint is not None:
      sessionSnapshot = checkpoint

   if flowName in counterControl.checkpoint:
      globalSnapshot = \
         counterControl.checkpoint.get( flowName )

   if flowName in openFlowHwConfig.flowEntry:
      fStats = hwStatus.flowStats[ flowName ]
   # Create a dummy snapshot that was created at the time of creation of 
   # flowStats
   zero = Tac.newInstance(
      "OpenFlowTable::CounterCheckpoint", "dummy" )

   # Choose the newest of the snapshots
   snapshots = [ zero ]
   if sessionSnapshot is not None:
      snapshots.append( sessionSnapshot )
   if globalSnapshot is not None:
      snapshots.append( globalSnapshot )
   snapshots.sort( key=lambda snapshot: snapshot.timestamp )
   snapshot = snapshots[ -1 ]

   delta = {}
   if fStats is  None:
      delta[ 'packetCount' ] = 0
      delta[ 'byteCount'  ] = 0
      return delta

   currentPacketCount = fStats.packetCount 
   currentByteCount = fStats.byteCount

   snapshotPacketCount = snapshot.packetCount
   snapshotByteCount = snapshot.byteCount

   maxCount = 0xffffffffffffffff
   if currentPacketCount >= snapshotPacketCount:
      delta[ 'packetCount' ] = currentPacketCount - snapshotPacketCount
      delta[ 'byteCount'  ] = currentByteCount - snapshotByteCount
   else:
      delta[ 'packetCount' ] = maxCount - snapshotPacketCount + \
                               currentPacketCount + 1
      delta[ 'byteCount'  ] = maxCount - snapshotByteCount + \
                              currentByteCount + 1

   return delta

def showFlows( mode, args ):
   brief = 'brief' in args
   matchedFlows = 'matched' in args
   flowName = args.get( 'FLOWS' )
   tableName = args.get( 'TABLENAME' )

   ret = OpenFlowFlows()
   ret.flows = []
   # pylint: disable-msg=W0212
   ret._brief = brief
   counterConfig.flowCounterUpdateTime = Tac.now()
   # sort by flow priority, highest to lowest
   if flowName:
      flowsUnSorted = [ flowName ]
   elif tableName:
      flowsUnSorted = [ f for f in openFlowHwConfig.flowEntry \
                        if tableName.upper() in \
                        openFlowHwConfig.flowEntry[ f ].tableName.upper() ]
   else:
      flowsUnSorted = openFlowHwConfig.flowEntry

   flows = sorted( flowsUnSorted, key=getFlowPriority,
                   reverse=True )

   for flowName in flows:
      try:
         flow = openFlowHwConfig.flowEntry[ flowName ]
         stats = hwStatus.flowStats[ flowName ]
         if matchedFlows:
            if getCounters( mode, flowName )[ 'packetCount' ] == 0 :
               continue
      except KeyError:
         continue
      if stats.status != "flowCreated":
         continue
      flowModel = OpenFlowFlows.Flow()
      flowModel.name = flowName
      flowModel.priority = flow.priority
      flowModel.tableName = flow.tableName
      flowModel.cookie = flow.cookie
      flowModel.idleTimeout = int( flow.idleTimeout )
      flowModel.hardTimeout = int( flow.hardTimeout )
      delta = getCounters( mode, flowName )
      flowModel.matchPackets = delta[ 'packetCount' ]
      flowModel.matchBytes = delta[ 'byteCount' ]
      flowModel.tableId = int( flow.tableId )

      match = OpenFlowFlows.Flow.Match()
      match.inInterfaces = []
      if flow.match.matched.inIntf:
         for i in flow.match.inIntf:
            match.inInterfaces.append( i )
      if flow.match.matched.ethSrc:
         mac = MacAddrAndBitMask()
         mac.mac = flow.match.ethSrc
         mac.mask = flow.match.ethSrcMask
         match.macSrc = mac
      if flow.match.matched.ethDst:
         mac = MacAddrAndBitMask()
         mac.mac = flow.match.ethDst
         mac.mask = flow.match.ethDstMask
         match.macDst = mac
      if flow.match.matched.vlanId:
         match.vlan = flow.match.vlanId
         match.vlanMask = flow.match.vlanIdMask
      if flow.match.matched.vlanPri:
         match.vlanPCP = flow.match.vlanPri
      if flow.match.matched.ethType:
         match.ethType = flow.match.ethType
      if flow.match.matched.ipSrc:
         ip = IpAddrAndBitMask()
         ip.ip = flow.match.ipSrc
         ip.mask = flow.match.ipSrcMask
         match.ipSrc = ip
      if flow.match.matched.ipDst:
         ip = IpAddrAndBitMask()
         ip.ip = flow.match.ipDst
         ip.mask = flow.match.ipDstMask
         match.ipDst = ip
      if flow.match.matched.ip6Src:
         ip = Ip6AddrAndBitMask()
         ip.ip = flow.match.ip6Src
         ip.mask = flow.match.ip6SrcMask
         match.ip6Src = ip
      if flow.match.matched.ip6Dst:
         ip = Ip6AddrAndBitMask()
         ip.ip = flow.match.ip6Dst
         ip.mask = flow.match.ip6DstMask
         match.ip6Dst = ip
      if flow.match.matched.ipTos:
         match.ipTos = flow.match.ipTos
      if flow.match.matched.ipProto:
         match.ipProto = flow.match.ipProto
      if flow.match.matched.l4Src:
         match.ipPortSrc = flow.match.l4Src
      if flow.match.matched.l4Dst:
         match.ipPortDst = flow.match.l4Dst
      if flow.match.matched.icmpType:
         match.icmpType = flow.match.icmpType
      if flow.match.matched.icmpCode:
         match.icmpCode = flow.match.icmpCode
      if flow.match.matched.ttl:
         match.ttl = flow.match.ttl
         match.ttlMask = flow.match.ttlMask
      if flow.match.matched.metaData:
         match.metaData = flow.match.metaData
      flowModel.match = match

      action = OpenFlowFlows.Flow.Action()
      action.ingrMirrorInterfaces = []
      action.egrMirrorInterfaces = []
      action.outInterfaces = []
      if flow.actions.enabled.ingrMirror:
         for i in flow.actions.ingrMirrorIntf:
            action.ingrMirrorInterfaces.append( i )
      if flow.actions.enabled.egrMirror:
         for i in flow.actions.egrMirrorIntf:
            action.egrMirrorInterfaces.append( i )
      if flow.actions.enabled.setEthSrc:
         mac = MacAddr()
         mac.mac = flow.actions.setEthSrc
         action.macSrc = mac
      if flow.actions.enabled.setEthDst:
         mac = MacAddr()
         mac.mac = flow.actions.setEthDst
         action.macDst = mac
      if flow.actions.enabled.setVlanId:
         action.vlan = flow.actions.setVlanId
      if flow.actions.enabled.pushVlanId:
         action.vlan = flow.actions.setVlanId
         action.pushVlan = True
      if flow.actions.enabled.popVlanTag:
         action.popVlan = True
      if flow.actions.enabled.setVlanPri:
         action.vlanPCP = flow.actions.setVlanPri
      if flow.actions.enabled.setIpSrc:
         ip = IpAddr()
         ip.ip = flow.actions.setIpSrc
         action.ipSrc = ip
      if flow.actions.enabled.setIpDst:
         ip = IpAddr()
         ip.ip = flow.actions.setIpDst
         action.ipDst = ip
      if flow.actions.enabled.setIpTos:
         action.ipTos = flow.actions.setIpTos
      action.ttlDec = flow.actions.enabled.decNwTtl
      if flow.actions.enabled.setL4Src:
         action.ipPortSrc = flow.actions.setL4Src
      if flow.actions.enabled.setL4Dst:
         action.ipPortDst = flow.actions.setL4Dst
      if flow.actions.enabled.setIcmpType:
         action.icmpType = flow.actions.setIcmpType
      if flow.actions.enabled.setIcmpCode:
         action.icmpCode = flow.actions.setIcmpCode
      if flow.actions.enabled.setGroup:
         action.group = flow.actions.setGroup

      if flow.actions.enabled.outputIntf:
         for i in flow.actions.outputIntf:
            action.outInterfaces.append( i )
      action.loopback = flow.actions.enabled.outputInIntf
      action.outputNormal = flow.actions.enabled.outputNormal
      action.outputFlood = flow.actions.enabled.outputFlood
      action.outputAll = flow.actions.enabled.outputAll
      action.outputController = flow.actions.enabled.outputController
      action.outputLocal = flow.actions.enabled.outputLocal
      if flow.actions.enabled.outputQueue:
         action.outputQ = flow.actions.outputQueue
      if ( flow.actions.enabled.outputIntf or
           flow.actions.enabled.setGroup or
           action.loopback or 
           action.outputNormal or 
           action.outputFlood or 
           action.outputAll or 
           action.outputController or 
           action.outputLocal or
           flow.actions.enabled.outputQueue or
           flow.actions.enabled.gotoTbl or
           openFlowHwConfig.bindMode == "bindModeMonitor" or 
           openFlowHwConfig.forwardingPipeline == "flowAndNormal" ):
         action.outputDrop = False
      else:
         action.outputDrop = True
      if flow.actions.enabled.gotoTbl:
         action.goto = True
         action.nextTblName = flow.actions.nextTblName
      if flow.actions.enabled.metaData:
         action.metaData = flow.actions.metaData
      flowModel.action = action
      
      ret.flows.append( flowModel )
   return ret

def showStatistics( mode ):
   ret = OpenFlowActivityStats()
   ret.activityStats = []
   activityStats5second = dict( status.activityStats5second )
   for i in reversed( sorted( activityStats5second ) ):
      s = activityStats5second[i]
      sM = OpenFlowActivityStats.Stats()
      sM.endTime = int( s.endTime )
      sM.flowModifications = s.flowModCount
      sM.packetsOut = s.packetOutCount
      sM.packetsToController = s.packetInCount
      sM.packetsDropped = s.sendDropCount
      sM.flowEntryCount = s.flowEntryCount
      ret.activityStats.append( sM )

   return ret

def showQueues( mode ):
   ret = OpenFlowQ()
   ret.controllerQ = []
   ret.portQ = []
   controllerQIds = dict( hwStatus.controllerQueueStats.items() )
   if controllerQIds:
      for qId in sorted( controllerQIds ):
         stats = hwStatus.controllerQueueStats[ qId ]
         ctrQ = OpenFlowQStats()
         ctrQ.ID = qId
         ctrQ.packetCount = stats.packetCount
         ctrQ.byteCount = stats.byteCount
         ctrQ.errCount = stats.errorCount
         ret.controllerQ.append( ctrQ )
   pi = dict( status.portInterface.items() )
   for port in sorted( pi ):
      intf = pi[ port ]
      try:
         qs = hwStatus.intfQueueStats[ intf ]
      except KeyError:
         continue
      pQ = OpenFlowQ.PortQ()
      pQ.stats = []
      pQ.port = port
      pQ.interface = intf
      qs = dict( qs.queueStats.items() )
      for queue in sorted( qs ):
         portQStat = OpenFlowQStats()
         stats = qs[ queue ]
         portQStat.ID = queue
         portQStat.packetCount = stats.packetCount
         portQStat.byteCount = stats.byteCount
         portQStat.errCount = stats.errorCount
         pQ.stats.append( portQStat )
      ret.portQ.append( pQ )
   return ret

def matchFieldSet( mf, ai, match ):
   if ai is not None:
      match.inInterface = mf.inIntf
      match.macSrc = mf.ethSrc
      match.macDst = mf.ethDst
      match.vlan = mf.vlanId
      match.vlanPCP = mf.vlanPri
      match.ethType = mf.ethType
      match.ipSrc = mf.ipSrc
      match.ipDst = mf.ipDst
      match.ipTos = mf.ipTos
      match.ipProto = mf.ipProto
      match.ipPortSrc = mf.l4Src
      match.ipPortDst = mf.l4Dst
      match.icmpType = mf.icmpType
      match.icmpCode = mf.icmpCode
   else :
      match.inInterface = False
      match.macSrc = False
      match.macDst = False
      match.vlan = False
      match.vlanPCP = False
      match.ethType = False
      match.ipSrc = False
      match.ipDst = False
      match.ipTos = False
      match.ipProto = False
      match.ipPortSrc = False
      match.ipPortDst = False
      match.icmpType = False
      match.icmpCode = False
   return match

def actionSet( a, action ):
   if a is not None:
      action.ingrMirrorInterfaces = a.ingrMirror
      action.vlan = a.setVlanId
      action.vlanPCP = a.setVlanPri
      action.macSrc = a.setEthSrc
      action.macDst = a.setEthDst
      action.ipSrc = a.setIpSrc
      action.ipDst = a.setIpDst
      action.ipTos = a.setIpTos
      action.ttlDec = a.decNwTtl
      action.ipPortSrc = a.setL4Src
      action.ipPortDst = a.setL4Dst
      action.icmpType = a.setIcmpType
      action.icmpCode = a.setIcmpCode
      action.outInterfaces = a.outputIntf
      action.loopback = a.outputInIntf
      action.outputNormal = a.outputNormal
      action.outputFlood = a.outputFlood
      action.outputAll = a.outputAll
      action.outputController = a.outputController
      action.outputLocal = a.outputLocal
      action.outputQ = a.outputQueue
      action.setGroup = a.setGroup
      action.outputDrop = a.outputIntf
      action.egrMirrorInterfaces = a.egrMirror
   return action

def showTableProfiles( mode ):
   ret = OpenFlowTablesProfile()
   ret.tables = []
   td = hwStatus.supportedTableDesc.get( hwStatus.bindMode )
   tableDesc = dict( td.tableDesc ) if td else dict()
   for profile in sorted( tableDesc.keys() ):
      pM = OpenFlowTableProfile()
      if profile == "tableProfileL2Match":
         pM.profile = "l2-match:"
      elif profile == "tableProfileVxlanMatch":
         pM.profile = "vxlan-match:"
      else:
         pM.profile = "full-match:"
      pM.match = OpenFlowTableProfileMatch()
      pM.match = matchFieldSet( tableDesc[profile].matchFields,
                                tableDesc[profile].matchArpIpSrcDst,
                                pM.match )
      pM.wildcardMatch = OpenFlowTableProfileMatch()
      pM.wildcardMatch = matchFieldSet( tableDesc[profile].wildcardFields,
                                        tableDesc[profile].matchArpIpSrcDst,
                                        pM.wildcardMatch )
      pM.action = OpenFlowTableProfileAction()
      pM.action = actionSet( tableDesc[profile].actions,
                             pM.action )
      pM.numFlowEntries = tableDesc[profile].flowEntries
      ret.tables.append( pM )
   return ret

# -----------------------------------------------------------------------------------
# clear openflow statistics
# -----------------------------------------------------------------------------------
def clearStatistics( mode ):
   if not config.enabled:
      return
   counterConfig.activityStatsClearCount += 1 # pylint: disable-msg=E1103

def clearFlowCounters( mode, name ):
   t0("Clearing counters for flow:", name )
   if name in openFlowHwConfig.flowEntry:
      fStats = hwStatus.flowStats[ name ]
      checkpoint = Tac.newInstance( 'OpenFlowTable::CounterCheckpoint', name )
      sessionCheckpoint = _openFlowSessionCheckpoint( mode )
      sessionCheckpoint[ name ] = checkpoint

      if fStats:
         checkpoint.packetCount = fStats.packetCount
         checkpoint.byteCount = fStats.byteCount
         checkpoint.timestamp = Tac.now()

def openFlowDataLinkExplanation( intfName, mode ):
   if intfName in status.interfacePort:
      return "OpenFlow", "OpenFlow configuration", "OpenFlow"
   else:
      return None, None, None
IntfCli.dataLinkExplanationHook.addExtension( openFlowDataLinkExplanation )

#------------------------------------------------------
# Register OpenFlow show commands into "show tech-support".
#------------------------------------------------------

# Timestamps are made up to maintain historical order within show tech-support
def _openFlowTechSupportCmds():
   if hwStatus.openFlowSupported:
      return [ 'show openflow',
               'show openflow ports',
               'show openflow statistics',
               'show openflow queues',
               'show openflow profiles',
               'show openflow flows' ]
   else :
      return []

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( 
                                '2010-01-01 00:08:00',
                                _openFlowTechSupportCmds )

def Plugin( entMan ):
   global config, status, flowModeConfig, hwStatus, openFlowHwConfig, \
      directFlowHwConfig, flowAndNormalHwConfig, entityMib
   global counterConfig, counterControl, hwEpochStatus, mirroringConfig
   config = ConfigMount.mount( entMan, "openflow/config", "OpenFlow::Config", "w" )
   counterConfig = LazyMount.mount( entMan, "openflow/counterconfig",
                                    "OpenFlow::CounterConfig", "w" )
   status = LazyMount.mount( entMan, "openflow/status", "OpenFlow::Status", "r" )
   flowModeConfig = ConfigMount.mount( entMan, "openflow/hwconfig",
                                 "OpenFlowTable::HwConfig", "w" )
   hwStatus = LazyMount.mount( entMan, "openflow/hwstatus",
                               "OpenFlowTable::HwStatus", "r" )
   openFlowHwConfig = LazyMount.mount( entMan, "openflow/openflowhwconfig",
                                       "OpenFlowTable::OpenFlowHwConfig", "r" )
   directFlowHwConfig = LazyMount.mount( entMan, "openflow/directflowhwconfig",
                                         "OpenFlowTable::DirectFlowHwConfig", "r" )
   flowAndNormalHwConfig = ConfigMount.mount( entMan,
                                              "openflow/flowandnormalhwconfig",
                                              "OpenFlowTable::DirectFlowHwConfig",
                                              "w" )
   counterControl = LazyMount.mount( entMan, "openflow/countercontrol",
                                     "OpenFlowTable::FlowCtrControl", "r" )
   hwEpochStatus = LazyMount.mount( entMan, "hwEpoch/status",
                                    "HwEpoch::Status", "r" )
   mirroringConfig = ConfigMount.mount( entMan, "mirroring/config",
                                        "Mirroring::Config", "w" )
   entityMib = LazyMount.mount( entMan, "hardware/entmib", "EntityMib::Status", "r" )
