# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import glob
import os

import CliModel
import LazyMount

GNMIStatus = None
RestconfStatus = None
NetconfStatus = None
NetconfConfig = None
OctaConfig = None

def printLineItem( label, content ):
   print '{:<20}{:}'.format( '%s:' % label, content )

class CommonEndpointStatus( CliModel.Model ):
   enabled = CliModel.Bool( help="Endpoint is enabled" )
   port = CliModel.Int( help="The port this protocol's server is listening on" )
   vrfName = CliModel.Str( help="The VRF this protocol's server is listening in" )
   error = CliModel.Str( help="error which occurred while starting the endpoint" )

   def render( self ):
      printLineItem( "Enabled", "Yes" if self.enabled else "No" )
      printLineItem( "Server", "running on port %d, in %s VRF" % ( self.port,
         self.vrfName ) \
                      if self.port != 0 else "Not yet running" )
      if self.error:
         printLineItem( "Error", self.error )

class GNMIEndpointStatus( CommonEndpointStatus ):
   sslProfile = CliModel.Str( help="SSL profile name" )
   qosDscp = CliModel.Int( help="The QoS DSCP value" )
   authorization = CliModel.Bool( help="per-RPC authorization" )
   def render( self ):
      if OctaConfig.enabled:
         printLineItem( "Octa", "enabled" )
      super( GNMIEndpointStatus, self ).render()
      printLineItem( "SSL Profile", self.sslProfile if self.sslProfile else "none" )
      printLineItem( "QoS DSCP", self.qosDscp if self.qosDscp != 0 else "none" )
      printLineItem( "Authorization Required",
         "Yes" if self.authorization else "No" )

class RestconfEndpointStatus( CommonEndpointStatus ):
   sslProfile = CliModel.Str( help="SSL profile name" )
   qosDscp = CliModel.Int( help="The QoS DSCP value" )
   def render( self ):
      super( RestconfEndpointStatus, self ).render()
      printLineItem( "SSL Profile", self.sslProfile if self.sslProfile else "none" )
      printLineItem( "QoS DSCP", self.qosDscp if self.qosDscp != 0 else "none" )

class NetconfEndpointStatus( CommonEndpointStatus ):
   pass

class ModelsStatus( CliModel.Model ):
   enabledSmashPaths = CliModel.List( valueType=str,
                                 help="List of enabled Smash paths" )
   disabledSmashPaths = CliModel.List( valueType=str,
                                  help="List of disabled Smash paths" )
   disabledSysdbPaths = CliModel.List( valueType=str,
                                  help="List of disabled Sysdb paths" )
   aftIPv4Unicast = CliModel.Bool( help="IPv4 Unicast is enabled for AFT" )
   aftIPv6Unicast = CliModel.Bool( help="IPv6 Unicast is enabled for AFT" )

   def render( self ):
      if self.aftIPv4Unicast or self.aftIPv6Unicast:
         print 'provider aft'
         if self.aftIPv4Unicast:
            print '  ipv4-unicast'
         if self.aftIPv6Unicast:
            print '  ipv6-unicast'
      print 'provider smash'
      for path in self.enabledSmashPaths:
         print '  path /Smash/{}'.format( path )
      for path in self.disabledSmashPaths:
         print '  path /Smash/{} disabled'.format( path )
      print 'provider sysdb'
      for path in self.disabledSysdbPaths:
         print '  path /Sysdb/{} disabled'.format( path )

#-------------------------------------------------------------------------------
# The "show management api gnmi" command
#-------------------------------------------------------------------------------
def showStatusGNMI( mode, args ):
   model = GNMIEndpointStatus()
   model.sslProfile = GNMIStatus.sslProfile
   model.enabled = GNMIStatus.enabled
   model.port = GNMIStatus.port
   model.qosDscp = GNMIStatus.qosDscp
   model.authorization = GNMIStatus.authorization
   model.vrfName = GNMIStatus.vrfName
   model.error = GNMIStatus.error
   return model

#-------------------------------------------------------------------------------
# The "show management api restconf" command
#-------------------------------------------------------------------------------
def showStatusRestconf( mode, args ):
   model = RestconfEndpointStatus()
   model.sslProfile = RestconfStatus.sslProfile
   model.enabled = RestconfStatus.enabled
   model.port = RestconfStatus.port
   model.qosDscp = RestconfStatus.qosDscp
   model.vrfName = RestconfStatus.vrfName
   model.error = RestconfStatus.error
   return model

#-------------------------------------------------------------------------------
# The "show management api netconf" command
#-------------------------------------------------------------------------------
def showStatusNetconf( mode, args ):
   model = NetconfEndpointStatus()
   model.enabled = False
   model.port = NetconfStatus.port
   model.vrfName = NetconfStatus.vrfName
   model.error = NetconfStatus.error
   for endpoint in NetconfConfig.endpoints.values():
      if not endpoint.enabled:
         continue
      vrfName = endpoint.vrfName
      port = endpoint.port
      sshdConfig = "/etc/ssh/sshd_config"
      if vrfName != "default":
         sshdConfig += "-" + vrfName
      if not os.path.exists( sshdConfig ):
         continue
      for line in open( sshdConfig, "r" ):
         if "Subsystem netconf netconf start-client" in line:
            configs = glob.glob( '/etc/xinetd.d/sshnetconf-' + endpoint.name )
            if configs:
               model.vrfName = vrfName
               model.port = port
               model.enabled = True
               return model
   return model

#-------------------------------------------------------------------------------
# The "show management api models" command
#-------------------------------------------------------------------------------
def showStatusModels( mode, args ):
   model = ModelsStatus()
   model.aftIPv4Unicast = OctaConfig.aftOptions.ipv4Unicast
   model.aftIPv6Unicast = OctaConfig.aftOptions.ipv6Unicast
   smashIncludes = OctaConfig.option.get( 'smashincludes', "" )
   if smashIncludes == "":
      model.enabledSmashPaths = []
   else:
      model.enabledSmashPaths = smashIncludes.split( ',' )
   smashExcludes = OctaConfig.option.get( 'smashexcludes', "" )
   if smashExcludes == "":
      model.disabledSmashPaths = []
   else:
      model.disabledSmashPaths = smashExcludes.split( ',' )
   sysdbExcludes = OctaConfig.option.get( 'sysdbexcludes', "" )
   if sysdbExcludes == "":
      model.disabledSysdbPaths = []
   else:
      model.disabledSysdbPaths = sysdbExcludes.split( ',' )
   return model


# Plug-in definition:
def Plugin( entityManager ):
   global GNMIStatus, RestconfStatus, NetconfStatus, NetconfConfig, OctaConfig
   GNMIStatus = LazyMount.mount( entityManager, "mgmt/gnmi/status",
                                 "Gnmi::Status", "r" )
   RestconfStatus = LazyMount.mount( entityManager, "mgmt/restconf/status",
                                 "Restconf::Status", "r" )
   NetconfStatus = LazyMount.mount( entityManager, "mgmt/netconf/status",
                                 "Netconf::Status", "r" )
   NetconfConfig = LazyMount.mount( entityManager, "mgmt/netconf/config",
                                 "Netconf::Config", "r" )
   OctaConfig = LazyMount.mount( entityManager, "mgmt/octa/config",
                                 "Octa::Config", "r" )
