#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliToken.OpenConfigTokens
import CliPlugin.AclCli as AclCli
import CliPlugin.Gnmi as Gnmi
import CliPlugin.VrfCli as VrfCli
from CliPlugin.AclCli import ( ipKwForServiceAclMatcher, 
                               inKwMatcher, 
                               accessGroupKwMatcher )

matcherPort = CliMatcher.KeywordMatcher( 'port', 
      helpdesc='The port number to listen on' )
matcherProfile = CliMatcher.KeywordMatcher( 'profile', 
      helpdesc='Configure SSL profile' )
matcherSsl = CliMatcher.KeywordMatcher( 'ssl', 
      helpdesc='Configure SSL related options' )
matcherAuthz = CliMatcher.KeywordMatcher( 'authorization', 
      helpdesc='Configure authorization' )
matcherReq = CliMatcher.KeywordMatcher( 'requests', 
      helpdesc='Configure per-RPC authorization' )

#--------------------------------------------------------------------------------
# [ no | default ] vrf VRFNAME 
#--------------------------------------------------------------------------------
class Cmd( CliCommand.CliCommandClass ):
   syntax = ' vrf VRFNAME '
   noOrDefaultSyntax = 'vrf ...'
   data = {
      'vrf': 'Configure VRF',
      'VRFNAME': CliMatcher.DynamicNameMatcher( VrfCli.getVrfNames, 'VRF name' ),
   }
   handler = Gnmi.setVrfName
   noOrDefaultHandler = handler

Gnmi.GnmiTransportConfigMode.addCommandClass( Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] authorization
#--------------------------------------------------------------------------------
class AuthzCmd( CliCommand.CliCommandClass ):
   syntax = ' authorization  requests '
   noOrDefaultSyntax = 'authorization requests'
   data = {
      'authorization': matcherAuthz,
      'requests': matcherReq
   }
   handler = Gnmi.setAuthorization
   noOrDefaultHandler = Gnmi.noAuthorization

Gnmi.GnmiTransportConfigMode.addCommandClass( AuthzCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip access-group ACLNAME [ in ]
#--------------------------------------------------------------------------------
class IpAccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACLNAME [ in ]'
   noOrDefaultSyntax = 'ip access-group ...'
   data = {
      'ip': ipKwForServiceAclMatcher,
      'access-group': accessGroupKwMatcher,
      'ACLNAME': AclCli.standardIpAclNameMatcher,
      'in': inKwMatcher,
   }
   handler = Gnmi.setOpenConfigAcl
   noOrDefaultHandler = Gnmi.noOpenConfigAcl

Gnmi.GnmiTransportConfigMode.addCommandClass( IpAccessGroupAclnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] port PORT
#--------------------------------------------------------------------------------
class PortPortCmd( CliCommand.CliCommandClass ):
   syntax = 'port PORT'
   noOrDefaultSyntax = 'port ...'
   data = {
      'port': matcherPort,
      'PORT': CliToken.OpenConfigTokens.matcherPortValue,
   }
   handler = Gnmi.setPort
   noOrDefaultHandler = handler

Gnmi.GnmiTransportConfigMode.addCommandClass( PortPortCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable protocol',
   }
   handler = Gnmi.shutdown
   noHandler = Gnmi.noShutdown
   defaultHandler = Gnmi.shutdown

Gnmi.GnmiTransportConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ssl profile PROFILENAME
#--------------------------------------------------------------------------------
class SslProfileProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILENAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
      'ssl': matcherSsl,
      'profile': matcherProfile,
      'PROFILENAME': Gnmi.profileNameMatcher,
   }
   handler = Gnmi.setSslProfile
   noOrDefaultHandler = Gnmi.noSslProfile

Gnmi.GnmiTransportConfigMode.addCommandClass( SslProfileProfilenameCmd )
