# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements IP DHCP Relay configuration.
# It contains both interface-specific and non-interface-specific commands.
#-------------------------------------------------------------------------------

import Arnet
import Tac
import CliCommand, CliMatcher, BasicCliModes, LazyMount
import ConfigMount
import IntfCli, IraIpIntfCli
from CliToken.Ip import ipMatcherForConfig
from CliToken.Ip import ipMatcherForConfigIf
from CliToken.Dhcp import dhcpMatcherForConfig
from CliPlugin.IpAddrMatcher import IpAddrMatcher
import DhcpRelayHelperCli
from DhcpRelayHelperCli import circuitIdValueMatcher
from IpLibConsts import DEFAULT_VRF

matcherRelay = CliMatcher.KeywordMatcher( 'relay',
      helpdesc='DHCP Relay' )

dhcpRelayConfig = None
nativeRelayConfig = None

def configConflict( mode ):
   # new relay is configured, old one isn't
   if not dhcpRelayConfig.intfConfig and nativeRelayConfig.intfConfig:
      msg = ( "To configure DHCP relay first remove incompatible" 
              " configuration of the form 'ip helper-address ...'" 
              " or 'ip dhcp relay information option circuit-id ...'" 
              " or 'ip dhcp relay all-subnets'" )
      mode.addWarning( msg )
      return True
   # old relay doesn't have the alwaysOn feature
   if nativeRelayConfig.alwaysOn:
      msg = ( "To configure DHCP relay first remove incompatible"
              " configuration 'ip dhcp relay always-on'" )
      mode.addWarning( msg )
      return True
   # old relay doesn't have the smartRelay feature
   if nativeRelayConfig.smartRelayGlobal:
      msg = ( "To configure DHCP relay first remove incompatible"
              " configuration 'ip dhcp relay all-subnets'" )
      mode.addWarning( msg )
      return True
   return False

# bypasses checks while calling a function in native relay CLI
def callNativeRelay( func, args ):
   DhcpRelayHelperCli.fromForwarder = True
   getattr( DhcpRelayHelperCli, func )( *args )
   DhcpRelayHelperCli.fromForwarder = False

def deleteIntfConfigIfDefault( mode ):
   c = dhcpRelayConfig
   i = mode.intf.name
   conf = c.intfConfig.get( i, None )
   if conf:
      keep = False
      if conf.clientAllowed:
         keep = True
      if conf.serverAllowed:
         keep = True
      if conf.circuitId != "":
         keep = True
      if not keep:
         del c.intfConfig[ i ]


# RoutingProtocolIntfConfigModelet omits management and loopback interfaces
modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

#-------------------------------------------------------------------------------
# The "[no|default] ip dhcp relay [client] [server]" interface command
#-------------------------------------------------------------------------------

relayMatcher = CliMatcher.KeywordMatcher( 'relay', helpdesc='DHCP Relay' )

def setIntfDhcpRelay( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   client = 'client' in args
   server = 'server' in args
   c = dhcpRelayConfig
   i = mode.intf.name
   conf = c.intfConfig.get( i, None )
   if not conf and not no:
      if configConflict( mode ):
         return
      conf = c.intfConfig.newMember( i )
   if conf:
      if client:
         conf.clientAllowed = not no
         # update intfConfig in native relay
         for addr, on in c.serverIp.items():
            if on:
               callNativeRelay( 'setIntfDhcpRelay', ( mode, addr, no ) )
      if server:
         conf.serverAllowed = not no
      deleteIntfConfigIfDefault( mode )

class IpDhcpRelayClientServerCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay [ client ] [ server ]'
   noOrDefaultSyntax = syntax

   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : relayMatcher,
      'client' : 'Service DHCP clients on this interface',
      'server' : 'Process DHCP server responses on this interface',
   }
   
   hidden = True
   handler = setIntfDhcpRelay
   noOrDefaultHandler = handler
   
modelet.addCommandClass( IpDhcpRelayClientServerCmd )

#-------------------------------------------------------------------------------
# The "[no|default] ip dhcp relay circuit-id <circuit>" interface command
#-------------------------------------------------------------------------------
circuitIdMatcher = CliMatcher.KeywordMatcher(
   'circuit-id', helpdesc='Relay Agent (Option 82) Circuit ID for this interface' )

def setIntfCircuitId( mode, args ):
   newId = args[ 'ID' ]
   c = dhcpRelayConfig
   i = mode.intf.name
   conf = c.intfConfig.get( i, None )
   if not conf:
      if configConflict( mode ):
         return
      conf = c.intfConfig.newMember( i )
   conf.circuitId = newId
   # set circuit-id in new relay
   callNativeRelay( 'setIntfCircuitId', ( mode, args ) )

def noIntfCircuitId( mode, args):
   c = dhcpRelayConfig
   i = mode.intf.name
   conf = c.intfConfig.get( i, None )
   if conf:
      conf.circuitId = ""
      deleteIntfConfigIfDefault( mode )
      # delete circit-id in new relay
      callNativeRelay( 'noIntfCircuitId', ( mode, args ) )

class IpDhcpRelayCircuitIdCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay circuit-id ID'
   noOrDefaultSyntax = 'ip dhcp relay circuit-id ...'

   data = {
      'ip' : ipMatcherForConfigIf,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : relayMatcher,
      'circuit-id' : circuitIdMatcher,
      'ID' : circuitIdValueMatcher,
   }

   hidden = True
   handler = setIntfCircuitId
   noOrDefaultHandler = noIntfCircuitId
 
modelet.addCommandClass( IpDhcpRelayCircuitIdCmd )

class OldDhcpRelayIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del dhcpRelayConfig.intfConfig[ self.intf_.name ]

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay log verbose
#--------------------------------------------------------------------------------
def setDhcpRelayLogLevel( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if not no:
      msg = "Log verbose is ignored. This command is left here for backward" \
          + " compatibility reason and has no effect. To enable logging, use" \
          + " 'trace DhcpRelay enable ...' command instead."
      mode.addWarning( msg )
   dhcpRelayConfig.logVerbose = not no

class IpDhcpRelayLogVerboseCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay log verbose'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'log' : 'Specify Logging Level',
      'verbose' : 'More Log Messages',
   }

   handler = setDhcpRelayLogLevel
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpRelayLogVerboseCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay server ADDR
#--------------------------------------------------------------------------------
def vrfIpSrcIntf( vrf, ip ):
   srcIntfId = Tac.Value( 'Arnet::IntfId', '' )
   return Tac.Value( "Ip::Helper::DhcpRelay::VrfIpSrcIntfSrcIp", vrf,
                     Arnet.IpGenAddr( ip ), srcIntfId, Arnet.IpGenAddr( '0.0.0.0' ) )

def setDhcpRelayServer( mode, args ):
   server = args.get( 'ADDR' )
   no = CliCommand.isNoOrDefaultCmd( args )
   c = dhcpRelayConfig
   # intf option is not currently supported
   if configConflict( mode ):
      return
   if no:
      del c.serverIp[ server ]
   else:
      c.serverIp[ server ] = True
   # modify native relay's intfConfig helper-addresses here
   # would be better to use callNativeRelay(), but unable to access
   # intfconfig mode, so setting manually
   for nativeIntfName in nativeRelayConfig.intfConfig:
      nativeIntfConfig = nativeRelayConfig.intfConfig.get( nativeIntfName )
      if no:
         del nativeIntfConfig.serverIp[ vrfIpSrcIntf( \
               DEFAULT_VRF, server ) ] 
      else:
         nativeIntfConfig.serverIp[ vrfIpSrcIntf( \
               DEFAULT_VRF, server ) ] = True

class IpDhcpRelayServerAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'ip dhcp relay server ADDR'
   noOrDefaultSyntax = syntax
   # In the original code, there is some consideration to allowing ADDR to be
   # an interface name, but was disabled because it is low win and has some
   # problems (e.g. if the intf is not routable the configuration will not work
   # and is confusing)
   data = {
      'ip' : ipMatcherForConfig,
      'dhcp' : dhcpMatcherForConfig,
      'relay' : matcherRelay,
      'server' : 'Specify DHCP Server',
      'ADDR' : IpAddrMatcher( helpdesc="DHCP Server's address" ),
   }

   handler = setDhcpRelayServer
   noOrDefaultHandler = handler
   
BasicCliModes.GlobalConfigMode.addCommandClass( IpDhcpRelayServerAddrCmd )


#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global dhcpRelayConfig
   dhcpRelayConfig = ConfigMount.mount( entityManager, "ip/dhcp/relay/config",
                                      "Ip::Dhcp::Relay::Config", "w" )

   # check for incompatible native DhcpRelay agent
   global nativeRelayConfig
   nativeRelayConfig = LazyMount.mount( entityManager, "ip/helper/dhcprelay/config",
                                        "Ip::Helper::DhcpRelay::Config", "r" )

   IntfCli.Intf.registerDependentClass( OldDhcpRelayIntf )
