#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.VrfCli as VrfCli
import CliPlugin.Ntp as Ntp
from CliPlugin.VrfCli import VrfExprFactory
import DscpCliLib
import HostnameCli
import MultiRangeRule
import ReversibleSecretCli

import Toggles.NtpToggleLib

localIntfHelp = 'Configure the interface from which the IP source address is taken'
matcherNtp = CliMatcher.KeywordMatcher( 'ntp',
      helpdesc='Configure NTP' )
vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='Use a specific VRF' )

featureServeMultiVrfEnabled = Toggles.NtpToggleLib.toggleNtpServeMultiVrfEnabled()

#--------------------------------------------------------------------------------
# [ no | default ] ntp authenticate [servers]
#--------------------------------------------------------------------------------
class NtpAuthenticateCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp authenticate [ servers ]'
   noOrDefaultSyntax = 'ntp authenticate ...'
   data = {
      'ntp': matcherNtp,
      'authenticate': 'Require authentication for NTP synchronization',
      'servers': 'Authentication required only for incoming NTP server responses',
   }
   handler = Ntp.doConfigAuth
   noOrDefaultHandler = Ntp.doDisableAuth

BasicCliModes.GlobalConfigMode.addCommandClass( NtpAuthenticateCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ntp bind [ VRF ] INTFS
#--------------------------------------------------------------------------------
class NtpBindCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp bind [ VRF ] INTFS'
   noOrDefaultSyntax = 'ntp bind [ VRF ] [ INTFS ] ...'
   data = {
      'ntp': matcherNtp,
      'bind': 'Configure the interfaces for NTP to listen on',
      'VRF': vrfExprFactory,
      'INTFS': IntfCli.Intf.rangeMatcherWithIpSupport
   }
   hidden = True
   handler = Ntp.doAddBoundIntfs
   noOrDefaultHandler = Ntp.doRemoveBoundIntfs

BasicCliModes.GlobalConfigMode.addCommandClass( NtpBindCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ntp
#--------------------------------------------------------------------------------
class NtpCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ntp'
   data = {
      'ntp': matcherNtp,
   }
   noOrDefaultHandler = Ntp.doDisableNtp

BasicCliModes.GlobalConfigMode.addCommandClass( NtpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ntp force-restarts
#--------------------------------------------------------------------------------
class NtpForceRestartsCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp force-restarts'
   noOrDefaultSyntax = 'ntp force-restarts ...'
   data = {
      'ntp': matcherNtp,
      'force-restarts': 'Force restarts on interface changes',
   }
   hidden = True
   handler = Ntp.enableRestarts
   noOrDefaultHandler = Ntp.disableRestarts

BasicCliModes.GlobalConfigMode.addCommandClass( NtpForceRestartsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ntp local-interface [ VRF ] INTF
#--------------------------------------------------------------------------------
class NtpLocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp ( source | local-interface ) [ VRF ] INTF'
   noOrDefaultSyntax = 'ntp ( source | local-interface ) ...'
   data = {
      'ntp': matcherNtp,
      'local-interface': localIntfHelp,
      'source': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source',
            helpdesc='%s (deprecated)' % localIntfHelp ),
         deprecatedByCmd='ntp local-interface' ),
      'VRF': vrfExprFactory,
      'INTF': IntfCli.Intf.matcherWithIpSupport,
   }
   handler = Ntp.doConfigDefaultSourceIntf
   noOrDefaultHandler = Ntp.doRemoveDefaultSourceIntf

BasicCliModes.GlobalConfigMode.addCommandClass( NtpLocalInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ntp qos dscp DSCP
#--------------------------------------------------------------------------------
DscpCliLib.addQosDscpCommandClass( BasicCliModes.GlobalConfigMode, Ntp.setDscp,
                                   Ntp.noDscp, tokenProto=matcherNtp )

#--------------------------------------------------------------------------------
# [ no | default ] ntp serve all [ vrf VRF ]
#--------------------------------------------------------------------------------
class NtpServeAllCmd( CliCommand.CliCommandClass ):

   if featureServeMultiVrfEnabled:
      syntax = 'ntp serve all [ VRF ]'
      noOrDefaultSyntax = 'ntp serve all [ VRF ] ...'
   else:
      syntax = 'ntp serve all'
      noOrDefaultSyntax = 'ntp serve all ...'
   data = {
      'ntp': matcherNtp,
      'serve': 'Configure the switch as an NTP server',
      'all': 'Service NTP requests received on any interface',
      'VRF': VrfExprFactory( helpdesc='Service NTP requests received on '
                                      'any interface in specified VRF' ),
   }
   handler = Ntp.doNtpServeAll
   noOrDefaultHandler = Ntp.doRemoveNtpServeAll

BasicCliModes.GlobalConfigMode.addCommandClass( NtpServeAllCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ntp serve ip access-group IP_ACL [ vrf VRF ] in
#--------------------------------------------------------------------------------
class NtpServeIpAccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp serve ip access-group IP_ACL [ VRF ] in'
   noOrDefaultSyntax = 'ntp serve ip access-group [ IP_ACL ] [ VRF ] ...'
   data = {
      'ntp': matcherNtp,
      'serve': 'Configure the switch as an NTP server',
      'ip': AclCli.ipKwForServiceAclMatcher,
      'IP_ACL': AclCli.standardIpAclNameMatcher,
      'access-group': AclCli.accessGroupKwMatcher,
      'VRF': AclCli.vrfExprFactoryForConfigAcl,
      'in': AclCli.inKwMatcher,
   }
   handler = Ntp.setNtpIpAcl
   noOrDefaultHandler = Ntp.noNtpIpAcl

BasicCliModes.GlobalConfigMode.addCommandClass( NtpServeIpAccessGroupCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ntp serve ipv6 access-group IP6_ACL [ vrf VRF ] in
#--------------------------------------------------------------------------------
class NtpServeIpv6AccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp serve ipv6 access-group IP6_ACL [ VRF ] in'
   noOrDefaultSyntax = 'ntp serve ipv6 access-group [ IP6_ACL ] [ VRF ] ...'
   data = {
      'ntp': matcherNtp,
      'serve': 'Configure the switch as an NTP server',
      'ipv6': AclCli.ipv6KwMatcherForServiceAcl,
      'IP6_ACL': AclCli.standardIp6AclNameMatcher,
      'access-group': AclCli.accessGroupKwMatcher,
      'VRF': AclCli.vrfExprFactoryForConfigAcl,
      'in': AclCli.inKwMatcher,
   }
   handler = Ntp.setNtpIpAcl
   noOrDefaultHandler = Ntp.noNtpIpAcl

BasicCliModes.GlobalConfigMode.addCommandClass( NtpServeIpv6AccessGroupCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ntp server [ vrf VRF ] HOST OPTIONS
#--------------------------------------------------------------------------------
sharedMatchObj = object()
class NtpServerOptionsExpr( CliCommand.CliExpression ):
   expression = ( '{ ( version VERSION ) | prefer | '
                   ' ( minpoll MIN_POLL ) | ( maxpoll MAX_POLL ) | '
                   ' burst | iburst | ( key KEY_ID ) | '
                   ' ( source | local-interface INTF ) }' )
   data = {
      'version': CliCommand.singleKeyword( 'version',
         helpdesc='Configure the NTP version' ),
      'VERSION': CliMatcher.IntegerMatcher( 1, 4,
         helpdesc='Version of the NTP protocol' ),
      'prefer': CliCommand.singleKeyword( 'prefer',
         helpdesc='Mark this server as preferred' ),
      'minpoll': CliCommand.singleKeyword( 'minpoll',
         helpdesc='Minimum poll interval' ),
      'MIN_POLL': CliMatcher.IntegerMatcher( 3, 17,
         helpdesc='Base-2 logarithm of minimum poll interval' ),
      'maxpoll': CliCommand.singleKeyword( 'maxpoll',
         helpdesc='Maximum poll interval' ),
      'MAX_POLL': CliMatcher.IntegerMatcher( 3, 17,
         helpdesc='Base-2 logarithm of maximum poll interval' ),
      'burst': CliCommand.singleKeyword( 'burst',
         helpdesc='Send a burst of packets instead of the usual one' ),
      'iburst': CliCommand.singleKeyword( 'iburst',
         helpdesc='Send bursts of packets until the server is reached' ),
      'key': CliCommand.singleKeyword( 'key',
         helpdesc='Set a key to use for authentication' ),
      'KEY_ID': CliMatcher.IntegerMatcher( 1, 65534, helpdesc='Key identifier' ),
      'source': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source', helpdesc=localIntfHelp ),
         maxMatches=1, sharedMatchObj=sharedMatchObj ),
      'local-interface': CliCommand.Node( 
         matcher=CliMatcher.KeywordMatcher( 'local-interface',
            helpdesc=localIntfHelp ),
         maxMatches=1, sharedMatchObj=sharedMatchObj ),
      'INTF': IntfCli.Intf.matcherWithIpSupport,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      options = {}
      if args.pop( 'key', None ):
         options[ 'keyId' ] = args.pop( 'KEY_ID' )[ 0 ]
      for i in ( ( 'version', 'VERSION' ), ( 'minpoll', 'MIN_POLL' ),
                 ( 'maxpoll', 'MAX_POLL' ) ):
         if args.pop( i[ 0 ], None ):
            options[ i[ 0 ] ] = args.pop( i[ 1 ] )[ 0 ]
      for i in ( 'prefer', 'burst', 'iburst' ):
         if args.pop( i, None ):
            options[ i ] = True
      if 'INTF' in args:
         options[ 'sourceIntf' ] = args.pop( 'INTF' )[ 0 ].name
         options[ 'sourceUsed' ] = 'source' in args

      args[ 'OPTIONS' ] = options

class NtpServerCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp server [ VRF ] HOST [ OPTIONS ]'
   noOrDefaultSyntax = 'ntp server [ VRF ] HOST ...'
   data = {
      'ntp': matcherNtp,
      'server': 'Configure NTP server to synchronize to',
      'VRF': vrfExprFactory,
      'HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
      'OPTIONS': NtpServerOptionsExpr
   }
   handler = Ntp.doConfigNtpServer
   noOrDefaultHandler = Ntp.doDeleteNtpServer

BasicCliModes.GlobalConfigMode.addCommandClass( NtpServerCmd )

#--------------------------------------------------------------------------------
# [ ( no | default ) ] ntp trusted-key KEY_RANGE
#--------------------------------------------------------------------------------
class NtpTrustedKeyCmd( CliCommand.CliCommandClass ):
   syntax = 'ntp trusted-key KEY_RANGE'
   noOrDefaultSyntax = 'ntp trusted-key ...'
   data = {
      'ntp': matcherNtp,
      'trusted-key': ( 'Configure the set of keys that are accepted for incoming '
                       'messages' ),
      'KEY_RANGE': MultiRangeRule.MultiRangeMatcher( lambda: ( 1, 65534 ),
         noSingletons=False, helpdesc='Key identifier(s)' )
   }
   handler = Ntp.doConfigTrustedKey
   noOrDefaultHandler = Ntp.doNoTrustedKey

BasicCliModes.GlobalConfigMode.addCommandClass( NtpTrustedKeyCmd )

#--------------------------------------------------------------------------------
# [ ( no | default ) ] ntp authentication-key KEY_RANGE
#--------------------------------------------------------------------------------
class AuthKeyCommand( CliCommand.CliCommandClass ):
   syntax = 'ntp authentication-key KEY_ID md5 | sha1 KEY'
   noOrDefaultSyntax = 'ntp authentication-key KEY_ID ...'
   data = {
      'ntp': matcherNtp,
      'authentication-key': 'Define a key to use for authentication',
      'KEY_ID': CliMatcher.IntegerMatcher( 1, 65534, helpdesc='Key identifier' ),
      'md5': 'MD5 hash algorithm',
      'sha1': 'SHA-1 hash algorithm',
      'KEY':
      ReversibleSecretCli.reversibleSecretCliExpression( 'KEY' )
   }
   handler = Ntp.doConfigAuthenKey
   noOrDefaultHandler = Ntp.doDeleteAuthenKey

BasicCliModes.GlobalConfigMode.addCommandClass( AuthKeyCommand )
