# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliToken.Ip
import ConfigMount
import FileCliUtil
import IntfCli
import Url
from CliPlugin.VrfCli import VrfExprFactory, DEFAULT_VRF

networkUrlConfig = None

#-------------------------------------------------------------------------------
# Global config mode commands.
# Below, <proto> can be any of ftp, ping, ssh, tftp, telnet, or traceroute.
# "http" is separate, because the "client" token is mandatory for it.
# "[no] ip <proto> [client] source-interface <interface-name> [vrf <vrf>]"
# "[no] ip <proto> [client] local-interface <interface-name> [vrf <vrf>]"
# "[no] ip http client local-interface <interface-name> [vrf <vrf>]"
# Deprecated commands
# "[no] ip http client source-interface <interface-name> [vrf <vrf>]"
#-------------------------------------------------------------------------------

def setProtocolSrcIntfName( mode, intf, protocol, useSrcIntf, vrfName=None ):
   saveVrfWithArg = bool( vrfName ) # save non-default vrf
   vrfName = vrfName or DEFAULT_VRF
   protoConf = networkUrlConfig.protocolConfig.get( protocol )
   if not protoConf:
      protoConf = networkUrlConfig.protocolConfig.newMember( protocol )

   del protoConf.srcIntf[ vrfName ] # From TACC; no KeyError thrown
   protoConf.srcIntf.newMember( vrfName, intf.name, saveVrfWithArg,
                                useSrcIntf )

def delProtocolSrcIntfName( mode, protocol, vrfName=None ):
   vrfName = vrfName or DEFAULT_VRF
   protoConfig = networkUrlConfig.protocolConfig.get( protocol )
   if protoConfig is not None:
      del protoConfig.srcIntf[ vrfName ]
      if not protoConfig.srcIntf:
         del networkUrlConfig.protocolConfig[ protocol ]

configMode = BasicCli.GlobalConfigMode

# accept interfaces with IP support
srcIntfNameMatcher = IntfCli.Intf.matcherWithIpSupport

sourceIntfHelp = 'The interface to be used for selecting source IP address'
srcIntfKw = CliMatcher.KeywordMatcher( 'source-interface',
   helpdesc=sourceIntfHelp )

srcIntfKwDeprecated = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'source-interface',
                              helpdesc='%s (deprecated)' % sourceIntfHelp ),
   deprecatedByCmd='local-interface' )

localIntfKw = CliMatcher.KeywordMatcher( 'local-interface',
   helpdesc=sourceIntfHelp )

vrfExprFactory = VrfExprFactory(
   helpdesc='VRF instance name' )

def addCmdForProtocol( protocol, clientRequired=False, srcIntfIsDeprecated=False ):
   # There are a lot of variants for these commands
   #
   # 1. The 'client' keyword is optional, but hide the non-client variant
   # 2. http has "local-interface" but "source-interface" is deprecated,
   #    while all others have both, but "local-interface" is hidden

   protoInCaps = protocol.upper()
   protocolKw = "%s configuration" % protoInCaps
   if srcIntfIsDeprecated:
      srcIntfCompat = srcIntfClient = srcIntfKwDeprecated
      localIntfCompat = CliCommand.Node( localIntfKw, hidden=True )
      localIntfClient = localIntfKw
   else:
      srcIntfCompat = CliCommand.Node( srcIntfKw, hidden=True )
      srcIntfClient = srcIntfKw
      localIntfCompat = localIntfClient = CliCommand.Node( localIntfKw, hidden=True )

   def srcIntfHandler( mode, args ):
      intf = args[ 'INTF' ]
      vrfName = args.get( 'VRF' )
      useSrcIntf = not srcIntfIsDeprecated and 'source-interface' in args
      setProtocolSrcIntfName( mode, intf, protocol, useSrcIntf, vrfName=vrfName )

   def noSrcIntfHandler( mode, args ):
      vrfName = args.get( 'VRF' )
      delProtocolSrcIntfName( mode, protocol, vrfName=vrfName )

   class SrcIntfCompatCmd( CliCommand.CliCommandClass ):
      # Compatibility non-client mode
      syntax = "ip %s source-interface | local-interface INTF [ VRF ]" % protocol
      noOrDefaultSyntax = "ip %s source-interface | local-interface [ VRF ] ..." \
                          % protocol
      data = {
         "ip" : CliToken.Ip.ipMatcherForConfig,
         protocol : protocolKw,
         "source-interface" : srcIntfCompat,
         "local-interface" : localIntfCompat,
         "INTF" : srcIntfNameMatcher,
         "VRF" : vrfExprFactory,
      }
      handler = staticmethod( srcIntfHandler )
      noOrDefaultHandler = staticmethod( noSrcIntfHandler )

   class SrcIntfClientCmd( CliCommand.CliCommandClass ):
      # Client mode
      syntax = "ip %s client source-interface | local-interface INTF [ VRF ]" \
               % protocol
      noOrDefaultSyntax = "ip %s client source-interface | local-interface [ VRF ]" \
                          " ..." % protocol
      data = {
         "ip" : CliToken.Ip.ipMatcherForConfig,
         protocol : protocolKw,
         "client" : '%s client configuration' % protoInCaps,
         "source-interface" : srcIntfClient,
         "local-interface" : localIntfClient,
         "INTF" : srcIntfNameMatcher,
         "VRF" : vrfExprFactory,
      }
      handler = staticmethod( srcIntfHandler )
      noOrDefaultHandler = staticmethod( noSrcIntfHandler )

   configMode.addCommandClass( SrcIntfClientCmd )
   if not clientRequired:
      configMode.addCommandClass( SrcIntfCompatCmd )

# The client keyword is optional for protocols other than HTTP
# Be careful here, because only "http" is deprecated,
# but all the other commands support both new and old syntax.
addCmdForProtocol( 'http', clientRequired=True, srcIntfIsDeprecated=True )
for proto in ( 'ftp', 'ping', 'ssh', 'tftp', 'telnet', 'traceroute' ):
   addCmdForProtocol( proto )

vrfExprFactory_ = VrfExprFactory(
      helpdesc='VRF instance name' )

class UrlVrfExpr( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      urlVrfName = name + "_VRF"
      class _UrlVrfExpr( CliCommand.CliExpression ):
         expression = "%s %s" % ( name, urlVrfName )
         data = {
            name: Url.UrlMatcher( lambda fs: fs.fsType == 'network',
                                  'Network URL', acceptSimpleFile=False ),
            urlVrfName: vrfExprFactory_,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            vrf = args.pop( urlVrfName, None )
            if vrf:
               url = args[ name ]
               url.setUrlVrfName( vrf )

      return _UrlVrfExpr

FileCliUtil.registerUrlExprExtension( "VRF", UrlVrfExpr() )

def Plugin( entityManager ):
   global networkUrlConfig
   networkUrlConfig = ConfigMount.mount( entityManager, "mgmt/networkUrl/config",
                                         "Mgmt::NetworkUrl::Config", "w" )
