#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

# pylint: disable=ungrouped-imports

import Plugins, functools, Tac
import LazyMount
import ConfigMount
import CliPlugin.ControllerdbLib
import CliPlugin.ControllerClient
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import ShowCommand
import CliPlugin.NetworkTopology as NetworkTopology
import CliPlugin.NetworkTopologyModels as NetworkTopologyModels
from CliPlugin.ControllerdbLib import controllerGuard
from CliPlugin.ControllerCli import CvxConfigMode
from CliPlugin.ControllerCli import serviceKwMatcher
from CliMode.Topology import TopologyMode
import Toggles.TopologyToggleLib as TopologyToggleLib

status = None
config = None

def forceMountWithActivityLock( func ):
   @functools.wraps( func )
   def _withActivityLock( *args, **kwargs ):

      LazyMount.force( status )
      
      with Tac.ActivityLockHolder():
         return func( *args, **kwargs )

   return _withActivityLock

matcherNetwork = CliMatcher.KeywordMatcher( 'network', 
      helpdesc='Network topology' )

nodeNetwork = CliCommand.Node( 
      matcher=matcherNetwork, 
      guard=controllerGuard )

matcherPhysicalTopology = CliMatcher.KeywordMatcher( 'physical-topology', 
      helpdesc='Physical network topology' )

matcherHostDetails = CliMatcher.KeywordMatcher( 'details', 
      helpdesc='Additional details' )
nodeHostDetailsHidden = CliCommand.Node(
      matcher=matcherHostDetails,
      hidden=True )

matcherHost = CliMatcher.KeywordMatcher( 'host', 
      helpdesc='Network host' )

matcherTopology = CliMatcher.KeywordMatcher( 'topology',
      helpdesc='Topology aggregation service' )
matcherSwitch = CliMatcher.KeywordMatcher( 'switch',
      helpdesc='Network device' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Network device interface' )
matcherNeighbor = CliMatcher.KeywordMatcher( 'neighbor',
      helpdesc='Neighbor device' )
matcherNeighborInterface = CliMatcher.KeywordMatcher( 'neighbor-interface',
      helpdesc='Neighbor interface' )

class TopologyConfigMode( TopologyMode,
                          BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'cvx-topology'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      TopologyMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------------
# service topology
#--------------------------------------------------------------------------------
class ServiceTopologyCmd( CliCommand.CliCommandClass ):
   syntax = 'service topology'
   noOrDefaultSyntax = syntax
   data = {
      'service': serviceKwMatcher,
      'topology': matcherTopology,
   }
   
   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( TopologyConfigMode )
      mode.session_.gotoChildMode( childMode )

   noOrDefaultHandler = handler

if TopologyToggleLib.toggleNetworkTopologyStaticConfigEnabled():
   CvxConfigMode.addCommandClass( ServiceTopologyCmd )

#--------------------------------------------------------------------------------
# network physical-topology switch SWITCH interface INTERFACE neighbor
# NEIGHBOR-HOST [ neighbor-interface NEIGHBOR-INTERFACE ]
#--------------------------------------------------------------------------------
def getSwitchNames( mode ):
   return [ h.hostname for h in status.host.values() ] if status else []

def getNeighborNames( mode, context ):
   if not status:
      return []
   switch = context.sharedResult[ 'SWITCH' ]
   return [ h.hostname for h in status.host.values() if h.hostname != switch ]

def getSwitchIntf( mode, context ):
   if not status:
      return []
   switch = context.sharedResult[ 'SWITCH' ]
   hosts = status.hostsByHostname.get( switch )
   if not hosts or len( hosts.host ) != 1:
      return []
   host = hosts.host.values()[ 0 ]
   return [ p for p in host.port ] if host is not None else []

def getNeighborIntf( mode, context ):
   if not status:
      return []
   neighbor = context.sharedResult[ 'NEIGHBOR-HOST' ]
   hosts = status.hostsByHostname.get( neighbor )
   if not hosts or len( hosts.host ) != 1:
      return []
   host = hosts.host.values()[ 0 ]
   return [ p for p in host.port ] if host is not None else []

class NetworkPhysicalTopologyConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''network physical-topology switch SWITCH interface INTERFACE neighbor 
               NEIGHBOR-HOST [ neighbor-interface NEIGHBOR-INTERFACE ]'''
   noOrDefaultSyntax = syntax
   
   _matcherSwitchObj = CliMatcher.DynamicNameMatcher( getSwitchNames,
      helpdesc='Switch name', pattern=r'.+')
   _matcherNeighborObj = CliMatcher.DynamicNameMatcher( getNeighborNames,
      pattern=r'.+', passContext=True, helpdesc='Neighbor name' )
   _matcherIntfObj = CliMatcher.DynamicNameMatcher( getSwitchIntf,
      passContext=True, helpdesc='Switch interface name' )
   _matcherNeighborIntfObj = CliMatcher.DynamicNameMatcher( getNeighborIntf,
      passContext=True, helpdesc='Neighbor interface name' )
   
   data = {
      'network': nodeNetwork,
      'physical-topology': matcherPhysicalTopology,
      'switch' : matcherSwitch,
      'SWITCH' : CliCommand.Node( _matcherSwitchObj, storeSharedResult=True ),
      'interface' : matcherInterface,
      'INTERFACE' : CliCommand.Node( _matcherIntfObj ),
      'neighbor' : matcherNeighbor,
      'NEIGHBOR-HOST' : CliCommand.Node( _matcherNeighborObj,
                                         storeSharedResult=True ),
      'neighbor-interface' : matcherNeighborInterface,
      'NEIGHBOR-INTERFACE' : CliCommand.Node( _matcherNeighborIntfObj )
   }

   @staticmethod
   def handler( mode, args ):
      neighIntf = args.get( 'NEIGHBOR-INTERFACE', "")
      edge = Tac.newInstance( "NetworkTopologyAggregatorV3::StaticEdge",
            args[ 'SWITCH' ], args[ 'INTERFACE' ], args[ 'NEIGHBOR-HOST' ],
            neighIntf )
      config.staticEdge[ edge ] = True
   
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      neighIntf = args.get( 'NEIGHBOR-INTERFACE', "")
      edge = Tac.newInstance( "NetworkTopologyAggregatorV3::StaticEdge",
            args[ 'SWITCH' ], args[ 'INTERFACE' ], args[ 'NEIGHBOR-HOST' ],
            neighIntf )
      del config.staticEdge[ edge ]

if TopologyToggleLib.toggleNetworkTopologyStaticConfigEnabled():
   TopologyConfigMode.addCommandClass( NetworkPhysicalTopologyConfigCmd )

#--------------------------------------------------------------------------------
# show network physical-topology neighbors [ host HOSTNAME ]
#--------------------------------------------------------------------------------
@forceMountWithActivityLock
def showNeighbors( mode, args ):
   host = args.get( 'HOSTNAME' )
   return NetworkTopology.showNeighbors( status, host )

class NetworkPhysicalTopologyNeighborsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show network physical-topology neighbors [ host HOSTNAME ]'
   data = {
      'network': nodeNetwork,
      'physical-topology': matcherPhysicalTopology,
      'neighbors': 'Show physical topology neighbors',
      'host': matcherHost,
      'HOSTNAME': CliMatcher.PatternMatcher( helpdesc='Name of the host', 
         helpname='HOST', pattern=r'.+' ),
   }
   handler = showNeighbors
   cliModel = NetworkTopologyModels.TopologyNeighbors

BasicCli.addShowCommandClass( NetworkPhysicalTopologyNeighborsCmd )

#--------------------------------------------------------------------------------
# show network physical-topology hosts [ host HOSTNAME ] [ details ]
#--------------------------------------------------------------------------------
#forceMountWithActivityLock
def showHosts( mode, args ):
   host = args.get( 'HOSTNAME' )
   details = 'details' in args
   return NetworkTopology.showHosts( status, host, details )

class NetworkPhysicalTopologyHostsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show network physical-topology hosts [ host HOSTNAME ] [ details ]'
   data = {
      'network': nodeNetwork,
      'physical-topology': matcherPhysicalTopology,
      'host': matcherHost,
      'hosts': 'Show physical topology hosts',
      'HOSTNAME': CliMatcher.PatternMatcher( helpdesc='Name of the host', 
         helpname='HOST', pattern=r'.+' ),
      'details': nodeHostDetailsHidden,
   }
   handler = showHosts
   cliModel = NetworkTopologyModels.TopologyHosts

BasicCli.addShowCommandClass( NetworkPhysicalTopologyHostsCmd )

#--------------------------------------------------------------------------------
# Mount global topology status from Controllerdb
#--------------------------------------------------------------------------------

def doControllerMounts( controllerdbEm ):
   global status 
   status = LazyMount.mount( controllerdbEm, "topology/version3/global/status",
         "NetworkTopologyAggregatorV3::Status", "r" )
   
@Plugins.plugin( requires=( "ControllerdbMgr", ) )
def Plugin( entityManager ):
   global config
   config = ConfigMount.mount( entityManager, "topology/aggregator/config",
         "NetworkTopologyAggregatorV3::Config", "w" )
   
   CliPlugin.ControllerdbLib.registerNotifiee( doControllerMounts )
