#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from CliModel import Dict
from CliModel import List
from CliModel import Model
from CliModel import Submodel
from CliModel import Str
from CliModel import Bool
import Ethernet
import Arnet
import re
import Tac
from TableOutput import createTable
from TableOutput import Format
from itertools import groupby

def getHostName( hostName ):
   r = re.compile( "^([0-9a-fA-F]{2}[:]){5}([0-9a-fA-F]{2})$" )
   if r.match( hostName ):
      return Ethernet.convertMacAddrCanonicalToDisplay( hostName )
   else:
      return hostName

def findHostname( host, useName=True ):
   if host.hostname:
      hostname = host.hostname
   elif useName:
      hostname = host.name
   else:
      return '(unknown)'

   return getHostName( hostname )

# These model classes mirror the tac types in Topology.tac
class Port( Model ):
   name = Str( help='Name of the port, for e.g., Ethernet1, Management1..' )
   hostname = Str( help='Host owning this port' )
   hostid = Str( help='Unique name/id which identifies host owning this port' )
   portChannel = Str( help='Port channel which this port is a member of', 
                      optional=True )

class PortGroup( Model ):
   name = Str( help='Name of the portGroup, for e.g., port-Channel1' )
   memberPorts = List( valueType=str, help='Names of the member ports in ' + 
                       'this port channel' )

class LogicalPort( Model ):
   name = Str( help='Name of the logical port, port channel or physical port' )
   memberPorts = List( valueType=str, help='Names of the member ports in ' + 
                       'this port channel', optional=True )
   
class Host( Model ):
   name = Str( help='Unique name/id which identifies this device' )
   hostname = Str( help='Hostname of the device. This is typically configured.' )
   port = Dict( valueType=Port, help='Ports on this host' )

   details = Bool( help="Details including logical ports and port channels",
                   optional=True )
   logicalPorts = Dict( valueType=LogicalPort, help="Logical ports on this host" )
   portChannels = Dict( valueType=PortGroup, help="Port channels on this host" )

   
class DirectedEdge( Model ):
   fromPort = Submodel( valueType=Port, help='Starting port of the directed edge' )
   toPort   = List( valueType=Port, help='Ending port of the directed edge' )

def _cmpHost( hostA, hostB ):
   return cmp( hostA.hostname, hostB.hostname )

def portNameKey( port ):
   return Arnet.intfNameKey( port.name )

def hostThenInterfaceKey( edge ):
   return ( edge.fromPort.hostname, portNameKey( edge.fromPort ) )
  
class TopologyHosts( Model ):
   hosts = Dict( valueType=Host, help='Set of hosts in topology' )
   
   def renderMemberPort( self, portChannel ):
      tabSpace =  " " * 4 + "%-15s"
      portChannelResult = tabSpace % portChannel.name + ": "
      escapedSpace = len( portChannelResult ) * " "
      newLineCount = 1
      # display 3 members each line
      for memberPort in portChannel.memberPorts:
         memberPort = "%-10s" % memberPort
         if newLineCount == len( portChannel.memberPorts ):
            if newLineCount == 1:
               print portChannelResult + memberPort
            else:
               if newLineCount % 3 == 1 :
                  print escapedSpace + memberPort
               else:
                  print memberPort
         elif newLineCount % 3 == 1 :
            if newLineCount == 1:
               print portChannelResult + memberPort + ",",
            else:
               print escapedSpace + memberPort + ",",
         elif newLineCount % 3 == 0:
            print memberPort + ","
         else:
            print memberPort + ",",
         newLineCount = newLineCount + 1
   
   def renderHostDetails( self, host ):   
      print "Unique Id     : " + getHostName( host.name )
      print "Hostname      : " + host.hostname
      print "Ports :"
      tabSpace =  " " * 4 + "%-15s"
      for port in sorted( host.port.values(), key=portNameKey ):
         portResult = tabSpace % port.name 
         if port.portChannel:
            portResult = portResult + ": " + port.portChannel
         print portResult

      print "Port Groups   :"
      for portChannel in host.portChannels.values():
         self.renderMemberPort( portChannel )

      print "Logical Ports :"
      for logicalPort in host.logicalPorts.values():
         if logicalPort.name.lower().startswith( "port-channel" ):
            self.renderMemberPort( logicalPort )
         else:
            logicalPortResult = tabSpace % logicalPort.name
            print logicalPortResult
      print
            
   def render( self ):
      fmt = "%-20s %s"
      printHead = True
      for h in sorted( self.hosts.values(), cmp=_cmpHost ):
         if not h.details:
            if printHead:
               print fmt % ( "Unique Id", "Hostname" )
               print fmt % ( "-" * 20, "-" * 30 )
               printHead = False
            print fmt % ( getHostName( h.name ), h.hostname )
         else:
            self.renderHostDetails( h ) 
            
def renderEdges( edges ):
   fmt = Format( justify='left' )
   sortedEdges = sorted( edges.values(), key=hostThenInterfaceKey )
   groupedEdges = groupby( sortedEdges, lambda edge : edge.fromPort.hostname )
   for fromHostname, group in groupedEdges:
      print fromHostname
      table = createTable(( "Interface", "Neighbor Intf", "Neighbor Host" ))
      table.formatColumns( fmt, fmt, fmt )
      for edge in group:
         for toPort in sorted( edge.toPort, key=portNameKey ):
            table.newRow( edge.fromPort.name, toPort.name, toPort.hostname )
      print table.output()

class TopologyNeighbors( Model ):
   neighbors = Dict( valueType=DirectedEdge,
                     help='Collection of neighbors in topology' )

   def render( self ):
      renderEdges( self.neighbors )
