# Copyright (c) 2007-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import BasicCli
import BasicCliSession
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.VrfCli as VrfCli
import CliToken.Ip
import CliToken.Ipv6
import ConfigMount
import DscpCliLib
import HostnameCli
import CliPlugin.IntfCli as IntfCli
import LazyMount
import Logging
import ShowCommand
import CliPlugin.NetConfigModels as NetConfigModels
import Tac
from Tracing import t0

#------------------------------------------------------------------------------------
# The "hostname" command, in config mode.
#
#  hostname <hostname>
#  no hostname
#------------------------------------------------------------------------------------

netConfig = None
netStatus = None
dscpConfig = None

Logging.logD(
   id="SYS_CLI_HOSTNAME_INVALID",
   severity=Logging.logError,
   format="Configured hostname \'%s\' is invalid",
   explanation="The configured hostname is invalid and has been ignored. The "
               "hostname must contain only alphanumeric characters, '.' and '-'. "
               "It must begin and end with an alphanumeric character."
               "Maximum characters in hostname is 64.",
   recommendedAction="Configure a valid hostname." )

def doSetHostname( mode, hostname="" ):
   if '_' in hostname:
      mode.addWarning( "The hostname contains '_'. This is not a valid Internet "
                       "hostname. The system might show '%s' instead of '%s' in "
                       "certain cases." % ( hostname.replace( '_', '-' ),
                                            hostname ) )
   netConfig.hostname = hostname
   if not ( mode.session_.inConfigSession() or mode.session_.startupConfig() or
            mode.session_.isStandalone() ):
      # Give the NetworkManager agent a chance to update the hostname according to
      # the new configured value, so that the next time we print a CLI prompt it
      # includes the updated hostname.  But don't wait too long, because if the agent
      # isn't running, or if two CLI processes set the hostname simultaneously, then
      # the hostname may never be updated.
      try:
         Tac.waitFor( lambda: netStatus.hostname == ( hostname or "localhost" ),
                      timeout=5, warnAfter=None, maxDelay=0.1, sleep=True )
      except Tac.Timeout:
         # This could happen if the NetworkManager/SuperServer agent is not
         # running (like when loading the startup config), is overwhelmed,
         # or is buggy. This isn't necessarily a bad thing, but we need to
         # make sure the user isn't confused when the prompt comes back with
         # the old hostname.
         mode.addWarning(
            "Timed out waiting for the hostname to be set. The prompt may "
            "not be updated\nimmediately. Check that the SuperServer agent "
            "is running, and if the problem\ndoes not resolve itself soon, "
            "contact customer support." )

def hostnameValueFunc( mode, match ):
   hostname = match.group()
   # Support _ even if it's not valid hostname
   if not HostnameCli.validateHostname( hostname.replace( '_', '-' ) ):
      mode.addError( HostnameCli.hostnameErrorMsg )
      if mode.session.startupConfig():
         Logging.log( SYS_CLI_HOSTNAME_INVALID, hostname )
      raise CliParser.AlreadyHandledError()

   return hostname

class HostnameCmd( CliCommand.CliCommandClass ):
   syntax = "hostname HOSTNAME"
   noOrDefaultSyntax = "hostname ..."
   data = {
      'hostname': 'Configure the system hostname',
      'HOSTNAME': HostnameCli.HostnameMatcher( helpname="WORD",
                                                helpdesc="The system's hostname",
                                                value=hostnameValueFunc )
      }

   @staticmethod
   def handler( mode, args ):
      doSetHostname( mode, args[ 'HOSTNAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      doSetHostname( mode )

BasicCli.GlobalConfigMode.addCommandClass( HostnameCmd )

#------------------------------------------------------------------------------------
# DNS config command, in config mode.
#
#  ip name-server <server-address> [ <server-address> ... ]
#  no|default ip name-server [ <server-address> ... ]
#
#    where <server-address> is the ipv4 or ipv6 address of a name server.
#
#  Some deprecation is happening
#  ip domain-name <domainname>    ->   dns domain <domainname>
#  no|default ip domain-name      ->   no|default dns domain
#
#  ip domain-list <domainname>
#  no|default ip domain-list <domainname>
#------------------------------------------------------------------------------------

def _nameServerId( netConfig, nameServer ):
   if isinstance( nameServer, str ):
      nameservers = netConfig.nameServer.items()
   else:
      nameservers = netConfig.v6NameServer.items()
   for id, ns in nameservers:
      if nameServer == ns:
         return id
   return None

def _idIsFree( id, netConfig ):
   return not ( id in netConfig.nameServer or
               id in netConfig.v6NameServer )

def _freeNameServerId( netConfig ):
   for id in range( 1, netConfig.maxNameServers + 1 ):
      if _idIsFree( id, netConfig ):
         return id
   return None

def _getNameServerString( nameServer ):
   if isinstance( nameServer, str ):
      return nameServer
   else:
      return nameServer.stringValue

def _compactNameServers( startID, netConfig ):
   # Compact nameservers for consecutive IDs
   maxIDs = Tac.Type( "System::NameServerId" ).max
   for id in range( startID, maxIDs ):
      if _idIsFree( id, netConfig ):
         next = id + 1
         if next in netConfig.nameServer:
            netConfig.nameServer[ id ] = netConfig.nameServer[ next ]
            del netConfig.nameServer[ id + 1 ]
         if next in netConfig.v6NameServer:
            netConfig.v6NameServer[ id ] = netConfig.v6NameServer[ next ]
            del netConfig.v6NameServer[ next ]

def _numConfigNameServers():
   return len( netConfig.nameServer ) + len( netConfig.v6NameServer )

def doAddNameServer( mode, args ):
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )

   # If the user deletes the VRF that contains the configured name
   # servers, we leave the NS config alone but the SuperServer plugin
   # will reject it that config and give none.
   if not _numConfigNameServers():
      netConfig.vrfName = vrfName
   elif vrfName != netConfig.vrfName:
      errorVrf = "Main VRF"
      if netConfig.vrfName != VrfCli.DEFAULT_VRF:
         errorVrf = "VRF " + netConfig.vrfName
      mode.addErrorAndStop(
         ( "Nameservers are supported in only one VRF at a time.  Please " +
           "unconfigure nameservers from %s before continuing." ) % errorVrf )

   nameServerList = args.get( 'V4', [] ) + args.get( 'V6', [] )
   for nameServer in nameServerList:
      id = _nameServerId( netConfig, nameServer )
      if id:
         # nameserver already exists
         continue
      # Get a free nameserver id
      id = _freeNameServerId( netConfig )
      if not id:
         nameServerString = _getNameServerString( nameServer )
         mode.addError(
            "Maximum number of nameservers reached. '%s' not added." %
            nameServerString )
         continue
      if isinstance( nameServer, str ):
         # IPv4 nameServers are of type 'str':
         netConfig.nameServer[ id ] = nameServer
      else:
         netConfig.v6NameServer[ id ] = nameServer
   updateDscpRules()

def doRemoveNameServer( mode, args ):
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )

   nameServerList = args.get( 'V4', [] ) + args.get( 'V6', [] )
   if ( ( netConfig.vrfName != vrfName ) and _numConfigNameServers() and
        ( nameServerList or vrfName != VrfCli.DEFAULT_VRF ) ):
      mode.addErrorAndStop(
         "No 'name-server' configuration found in VRF '%s'." % vrfName )

   if nameServerList:
      for nameServer in nameServerList:
         id = _nameServerId( netConfig, nameServer )
         if not id:
            nameServerString = _getNameServerString( nameServer )
            mode.addError(
               "Nameserver '%s' is not configured" % nameServerString )
            break
         if isinstance( nameServer, str ):
            del netConfig.nameServer[ id ]
         else:
            del netConfig.v6NameServer[ id ]
         _compactNameServers( id, netConfig )
   else:
      # If no nameservers are specified, remove all of them
      for id in netConfig.nameServer.keys():
         del netConfig.nameServer[ id ]
      for id in netConfig.v6NameServer.keys():
         del netConfig.v6NameServer[ id ]

   if _numConfigNameServers() == 0:
      netConfig.vrfName = VrfCli.DEFAULT_VRF
   updateDscpRules()

vrfExprFactory = VrfCli.VrfExprFactory(
   helpdesc='Configure a nameserver in a VRF' )

class NameServerCmd( CliCommand.CliCommandClass ):
   matchObj_ = object()
   maxServers_ = Tac.Type( "System::NameServerId" ).max
   syntax = "ip name-server [ VRF ] { V4 | V6 }"
   noOrDefaultSyntax = "ip name-server [ VRF ] [ { V4 | V6 } ]"
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'name-server': 'Configure the name server address',
      'VRF': vrfExprFactory,
      'V4': CliCommand.Node(
         IpAddrMatcher.IpAddrMatcher( "Domain name server IPv4 address (max 3)" ),
         maxMatches=maxServers_, sharedMatchObj=matchObj_ ),
      'V6': CliCommand.Node(
         Ip6AddrMatcher.Ip6AddrMatcher( "Domain name server IPv6 address (max 3)" ),
         maxMatches=maxServers_, sharedMatchObj=matchObj_ ),
      }

   handler = doAddNameServer
   noOrDefaultHandler = doRemoveNameServer

BasicCli.GlobalConfigMode.addCommandClass( NameServerCmd )

domainNameHelp = 'Configure the DNS domain name'

dnsConfigKw = CliMatcher.KeywordMatcher(
   'dns',
   helpdesc='Domain Name System configuration' )

domainNameMatcher = CliMatcher.PatternMatcher( r'\S+',
                                               helpname="WORD",
                                               helpdesc="DNS domain name" )

def doSetDomainName( mode, domainName="" ):
   # no/default form leads to non-deprecated cmd
   netConfig.domainName = domainName

class DomainNameDeprecated( CliCommand.CliCommandClass ):
   syntax = "ip domain-name DOMAIN_NAME"
   noOrDefaultSyntax = "ip domain-name ..."
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'domain-name': CliCommand.Node(
         CliMatcher.KeywordMatcher( 'domain-name',
                                    helpdesc='%s (deprecated)' % domainNameHelp ),
         deprecatedByCmd='dns domain' ),
      'DOMAIN_NAME': domainNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      doSetDomainName( mode, domainName=args.get( 'DOMAIN_NAME', '' ) )

   noOrDefaultHandler = handler

# (config)# ip domain-name <domain.name>
# deprecated
BasicCli.GlobalConfigMode.addCommandClass( DomainNameDeprecated )

class DnsDomainName( CliCommand.CliCommandClass ):
   syntax = "dns domain DOMAIN_NAME"
   noOrDefaultSyntax = "dns domain ..."
   data = {
      'dns': dnsConfigKw,
      'domain': domainNameHelp,
      'DOMAIN_NAME': domainNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      doSetDomainName( mode, domainName=args.get( 'DOMAIN_NAME', '' ) )

   noOrDefaultHandler = handler

# (config)# dns domain <domain.name>
# [legacy] `ip domain-name <domain.name>`
BasicCli.GlobalConfigMode.addCommandClass( DnsDomainName )

def _numDomains( config ):
   if ( config.domainName
        and config.domainName not in config.domainList.values() ):
      return len( config.domainList ) + 1
   else:
      return len( config.domainList )

def _searchLineLen( config, newDomain ):
   searchDomains = [ newDomain ]
   if ( config.domainName
       and config.domainName not in config.domainList.values() ):
      searchDomains.append( config.domainName )
   searchDomains += config.domainList.values()
   return len( ' '.join( searchDomains ) )

def _domainListId( config, domainName ):
   for domainId, ds in config.domainList.iteritems():
      if domainName == ds:
         return domainId
   return None

def _domainInDomainList( config, domainName ):
   return _domainListId( config, domainName ) is not None

def doAddDomainName( mode, domainName ):
   if not _domainInDomainList( netConfig, domainName ):
      if _numDomains( netConfig ) >= netConfig.maxSearchDomains:
         mode.addError( "Maximum number of domains (%d) exceeded. Domain %s not "
                        "added." % ( netConfig.maxSearchDomains, domainName ) )
      elif _searchLineLen( netConfig, domainName ) >= netConfig.maxSearchChars:
         # Search path character limit seems to be exclusive, not inclusive
         mode.addError( "Maximum number of characters (%d) exceeded. Domain %s "
                        "not added." % ( netConfig.maxSearchChars, domainName ) )
      else:
         netConfig.domainList.enq( domainName )

def doRemoveDomainName( mode, domainName ):
   domainId = _domainListId( netConfig, domainName )
   if domainId is not None:
      del netConfig.domainList[ domainId ]

class DomainList( CliCommand.CliCommandClass ):
   syntax = "ip domain-list DOMAIN_NAME"
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'domain-list': 'Configure domain names to complete unqualified host names',
      'DOMAIN_NAME': CliMatcher.PatternMatcher( r'\S+',
                                                 helpname="WORD",
                                                 helpdesc="A domain name" )
      }

   @staticmethod
   def handler( mode, args ):
      doAddDomainName( mode, args[ 'DOMAIN_NAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      doRemoveDomainName( mode, args[ 'DOMAIN_NAME' ] )

BasicCli.GlobalConfigMode.addCommandClass( DomainList )

def doSetLookupSourceIntf( mode, args ):
   vrfName = args.get( 'VRF', VrfCli.DEFAULT_VRF )
   intf = args.get( 'INTF' )
   t0( 'doSetLookupSourceIntf : vrfName=", vrfName, "intf=', intf )
   if intf:
      netConfig.sourceIntf[ vrfName ] = intf.name
   else:
      del netConfig.sourceIntf[ vrfName ]

domainConfigKw = CliMatcher.KeywordMatcher( 'domain',
                                            helpdesc='Domain configuration' )

class DomainSrcIntf( CliCommand.CliCommandClass ):
   syntax = "ip domain lookup [ VRF ] source-interface INTF"
   noOrDefaultSyntax = "ip domain lookup [ VRF ] source-interface ..."
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'domain': domainConfigKw,
      'lookup': 'Lookup a domain-name ip',
      'VRF': vrfExprFactory,
      'source-interface': 'Interface name to be looked up',
      'INTF': IntfCli.Intf.matcherWithIpSupport
      }

   handler = doSetLookupSourceIntf
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( DomainSrcIntf )

class DomainProxy( CliCommand.CliCommandClass ):
   syntax = "ip domain proxy"
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'domain': domainConfigKw,
      'proxy': 'Enable the Domain Name Service proxy for external hosts'
      }

   @staticmethod
   def handler( mode, args ):
      netConfig.externalDnsProxy = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      netConfig.externalDnsProxy = False

BasicCli.GlobalConfigMode.addCommandClass( DomainProxy )

def updateDscpRules():
   dscpValue = netConfig.dscpValue
   if not dscpValue:
      del dscpConfig.protoConfig[ 'dns' ]
      return

   protoConfig = dscpConfig.newProtoConfig( 'dns' )
   ruleColl = protoConfig.rule
   ruleColl.clear()

   port = 53
   proto = 'udp'
   vrf = netConfig.vrfName
   DscpCliLib.addDscpRule( ruleColl, '0.0.0.0', port,
                           True, vrf, proto, dscpValue )
   DscpCliLib.addDscpRule( ruleColl, '::', port,
                           True, vrf, proto, dscpValue, v6=True )
   for server in netConfig.nameServer.itervalues():
      DscpCliLib.addDscpRule( ruleColl, server, port,
                              False, vrf, proto, dscpValue )
   for server in netConfig.v6NameServer.itervalues():
      DscpCliLib.addDscpRule( ruleColl, server.stringValue, port,
                              False, vrf, proto, dscpValue, v6=True )

def setDscp( mode, args ):
   netConfig.dscpValue = args[ 'DSCP' ]
   updateDscpRules()

def noDscp( mode, args ):
   netConfig.dscpValue = netConfig.dscpValueDefault
   updateDscpRules()

# [no|default] dns qos dscp <dscpValue>
DscpCliLib.addQosDscpCommandClass( BasicCli.GlobalConfigMode, setDscp, noDscp,
                                   dnsConfigKw )

#------------------------------------------------------------------------------------
# hostname config command, in config mode.
#
#  ip host <hostname> A.B.C.D [ A.B.C.D ... ]
#  no ip host [<hostname>] [ A.B.C.D ... ]
#  no ip host
#  ipv6 host <hostname> A.B.C.D [ A.B.C.D ... ]
#  no ipv6 host [<hostname>] [ A.B.C.D ... ]
#  no ipv6 host
#------------------------------------------------------------------------------------
# This function adds one or more ipv4 mappings to a host
# Previous IPv4 mappings are cleared for hostname, if any
def doSetHostnameIp( mode, hostname, ipAddressList ):
   # create a new hostname entry if it doesn't exist already
   netConfig.newHostAddr( hostname )
   for address in netConfig.hostAddr[ hostname ].ipAddr:
      if address not in ipAddressList:
         del netConfig.hostAddr[ hostname ].ipAddr[ address ]
   for address in ipAddressList:
      netConfig.hostAddr[ hostname ].ipAddr[ address ] = True

# This function adds one or more ipv6 mappings to a host
# Previous IPv6 mappings are cleared for hostname, if any
def doSetHostnameIp6( mode, hostname, ip6AddressList ):
   # create a new hostname entry if doesn't exist already
   netConfig.newHostAddr( hostname )
   for address in netConfig.hostAddr[ hostname ].ip6Addr:
      if address not in ip6AddressList:
         del netConfig.hostAddr[ hostname ].ip6Addr[ address ]
   for address in ip6AddressList:
      netConfig.hostAddr[ hostname ].ip6Addr[ address ] = True

# This function removes ipv4 mapping from netconfig
def doRemoveMapping( mode, hostname=None, ipAddressList=None, ipv6=False ):
   if hostname:
      if not hostname in netConfig.hostAddr:
         return
      ipTable = ( netConfig.hostAddr[ hostname ].ip6Addr if ipv6
                  else netConfig.hostAddr[ hostname ].ipAddr )
      ipTable2 = ( netConfig.hostAddr[ hostname ].ipAddr if ipv6
                   else netConfig.hostAddr[ hostname ].ip6Addr )

      if ipAddressList:
         # removing specified ipv4/ipv6 mappings for the host
         for mappedAddress in ipAddressList:
            del ipTable[ mappedAddress ]
      else:
         # removing all ipv4/ipv6 mappings for the host
         ipTable.clear()
      # delete a host object from netconfig which has no mapping
      if not ( ipTable or ipTable2 ):
         del netConfig.hostAddr[ hostname ]

   else:
      # removing all the ipv4/ipv6 mappings( for all hosts)  from netconfig
      for name in netConfig.hostAddr:
         ipTable = ( netConfig.hostAddr[ name ].ip6Addr if ipv6
                     else netConfig.hostAddr[ name ].ipAddr )
         ipTable2 = ( netConfig.hostAddr[ name ].ipAddr if ipv6
                      else netConfig.hostAddr[ name ].ip6Addr )

         ipTable.clear()
         # delete a host object from netconfig which has no mapping
         if not ( ipTable or ipTable2 ):
            del netConfig.hostAddr[ name ]

hostKw = CliMatcher.KeywordMatcher( 'host',
                                    helpdesc='Configure the IP addresses for host' )
hostMatcher = HostnameCli.HostnameMatcher( helpname="WORD",
                                           helpdesc="Name of the host" )

class IpHostCmd( CliCommand.CliCommandClass ):
   syntax = "ip host HOSTNAME { IP }"
   noOrDefaultSyntax = "ip host [ HOSTNAME [ { IP } ] ]"
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'host': hostKw,
      'HOSTNAME': hostMatcher,
      'IP': IpAddrMatcher.IpAddrMatcher( helpdesc="IP address of the host" )
      }

   @staticmethod
   def handler( mode, args ):
      doSetHostnameIp( mode, args[ 'HOSTNAME' ], args[ 'IP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      doRemoveMapping( mode, args.get( 'HOSTNAME' ), args.get( 'IP' ),
                       ipv6=False )

BasicCli.GlobalConfigMode.addCommandClass( IpHostCmd )

class Ip6HostCmd( CliCommand.CliCommandClass ):
   syntax = "ipv6 host HOSTNAME { IP }"
   noOrDefaultSyntax = "ipv6 host [ HOSTNAME [ { IP } ] ]"
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
      'host': hostKw,
      'HOSTNAME': hostMatcher,
      'IP': Ip6AddrMatcher.Ip6AddrMatcher( helpdesc="IPv6 address of the host" )
      }

   @staticmethod
   def handler( mode, args ):
      doSetHostnameIp6( mode, args[ 'HOSTNAME' ], args[ 'IP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      doRemoveMapping( mode, args.get( 'HOSTNAME' ), args.get( 'IP' ),
                       ipv6=True )

BasicCli.GlobalConfigMode.addCommandClass( Ip6HostCmd )

#-----------------------------------------------------------------------------------
# show hostname
#-----------------------------------------------------------------------------------
class ShowHostname( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hostname'
   data = {
            'hostname': 'Show the system hostname',
          }
   cliModel = NetConfigModels.Hostname

   @staticmethod
   def handler( mode, args ):
      return NetConfigModels.Hostname( hostname=netStatus.hostname,
                                    fqdn=netStatus.fqdn )

BasicCli.addShowCommandClass( ShowHostname )

#------------------------------------------------------------------------------------
# Show hostname configuration commands, in unpriv / enable mode
#
#  show hosts
#------------------------------------------------------------------------------------
def doShowHosts( mode, args ):
   hostsInfo = NetConfigModels.SysMgrHostInfo()
   hostsInfo.domainName = netConfig.domainName
   hostsInfo.domainList = netConfig.domainList.values()
   hostsInfo.nameServers = netConfig.nameServer.values()

   for host in netStatus.hostAddr.keys():
      hostsInfo.hosts[ host ] = NetConfigModels.SysMgrHostInfo.IpAddresses()

      for address in netStatus.hostAddr[ host ].ipAddr.keys():
         hostsInfo.hosts[ host ].ipv4Addresses.append( address )

      for address in netStatus.hostAddr[ host ].ip6Addr.keys():
         hostsInfo.hosts[ host ].ipv6Addresses.append( address )
   return hostsInfo

class ShowHosts( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hosts'
   data = {
            'hosts': 'IP domain-name, lookup style, nameservers and host table',
          }
   cliModel = NetConfigModels.SysMgrHostInfo
   handler = doShowHosts

BasicCli.addShowCommandClass( ShowHosts )

#------------------------------------------------------------------------------------
# Show DNS configuration commands, in unpriv / enable mode
#
#  Deprecated command, but still valid
#  show ip domain-name    ->   show dns domain
#  show ip name-server
#------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------
# show ip domain-name
#-----------------------------------------------------------------------------------
showDomainNameHelp = 'Show the system domain name'

def doShowDomainName( mode, args ):
   return NetConfigModels.DomainName( domainName=netConfig.domainName )

class ShowIpDomainName( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip domain-name'
   data = {
            'ip': 'Details related to IPv4',
            'domain-name': CliCommand.Node(
                              matcher=CliMatcher.KeywordMatcher( 'domain-name',
                                 helpdesc='%s (deprecated)' % showDomainNameHelp ),
                              deprecatedByCmd='show dns domain' )
          }
   cliModel = NetConfigModels.DomainName
   handler = doShowDomainName

BasicCli.addShowCommandClass( ShowIpDomainName )

#-----------------------------------------------------------------------------------
# show dns domain
#-----------------------------------------------------------------------------------
class ShowDnsDomain( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dns domain'
   data = {
            'dns': 'Domain Name System configuration',
            'domain': showDomainNameHelp,
          }
   cliModel = NetConfigModels.DomainName
   handler = doShowDomainName

BasicCli.addShowCommandClass( ShowDnsDomain )

#-----------------------------------------------------------------------------------
# show ip name-server
#-----------------------------------------------------------------------------------
def doShowNameServer( mode, args ):
   maxIDs = Tac.Type( "System::NameServerId" ).max
   nameServerInfo = NetConfigModels.SysMgrNameServerInfo()
   for i in xrange( 1, maxIDs + 1 ):
      if i in netConfig.nameServer:
         nameServerInfo.v4NameServers.append( netConfig.nameServer[ i ] )
      if i in netConfig.v6NameServer:
         nameServerInfo.v6NameServers.append(
               netConfig.v6NameServer[ i ].stringValue )
   return nameServerInfo

class ShowIpNameServer( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip name-server'
   data = {
            'ip': CliToken.Ip.ipMatcherForShow,
            'name-server': 'Show name-server configuration',
          }
   cliModel = NetConfigModels.SysMgrNameServerInfo
   handler = doShowNameServer

BasicCli.addShowCommandClass( ShowIpNameServer )

def _hostnameHandler():
   return netStatus.hostname

def Plugin( entityManager ):
   global netConfig
   global netStatus
   global dscpConfig

   netConfig = ConfigMount.mount( entityManager,
                                  "sys/net/config", "System::NetConfig", "w" )
   netStatus = LazyMount.mount( entityManager, Cell.path( "sys/net/status" ),
                                "System::NetStatus", "r" )
   dscpConfig = ConfigMount.mount( entityManager, "mgmt/dscp/config",
                                   "Mgmt::Dscp::Config", "w" )
   BasicCliSession.registerHostnameHandler( _hostnameHandler )
