# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from CliModel import Dict
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Enum
from CliModel import Bool
from ArnetModel import Ip4Address
from ArnetModel import Ip6Address
from Ark import ReversibleDict

MAX_NAME_LEN = 29

class DomainName( Model ):
   domainName = Str( help="The domain name of the switch" )

   def render( self ):
      if self.domainName:
         print( self.domainName )

class SysMgrNameServerInfo( Model ):
   v4NameServers = List( valueType=str,
                         help="List of IPV4 name servers" )
   v6NameServers = List( valueType=str,
                         help="List of IPV6 name servers" )

   def render( self ):
      for addr in self.v4NameServers:
         print( addr )
      for addr in self.v6NameServers:
         print( addr )

class Hostname( Model ):
   hostname = Str( help="The current hostname of the switch" )
   fqdn = Str( help="The fully qualified domain name of the switch" )

   def render( self ):
      if not self.hostname:
         print( "No hostname set" )
      else:
         print( "Hostname:", self.hostname )
         print( "FQDN:    ", self.fqdn )

class SysMgrHostInfo( Model ):
   class IpAddresses( Model ):
      ipv4Addresses = List( valueType=Ip4Address, help="List of IPV4 addresses" )
      ipv6Addresses = List( valueType=Ip6Address, help="List of IPV6 addresses" )

   hosts = Dict( keyType=str, valueType=IpAddresses,
                 help="A mapping between host names and their IP addresses" )
   domainName = Str( help="The domain name that the listed hosts are "
                     "associated with",
                     optional=True )
   domainList = List( valueType=str,
                      help="List of additional domains to lookup domains with" )
   nameServers = List( valueType=str,
                       help="List of domain name servers used to resolve names" )

   def render( self ):
      if self.domainName:
         print( '\nDefault domain is: %s' % self.domainName )
      else:
         print( '\nDefault domain is not configured' )
      if self.domainList:
         print( 'Additional domains are: ' + ', '.join( self.domainList ) )
      print( 'Name/address lookup uses domain service' )

      print( 'Name servers are: ' + ', '.join( self.nameServers ) )

      print( '\nStatic Mappings:\n' )

      print( '%-*s  %s     %s\n' % ( MAX_NAME_LEN, 'Hostname', 'IP', 'Addresses' ) )

      for host, addresses in sorted( self.hosts.items() ):
         self._printAddresses( addresses.ipv4Addresses, host, 'IPV4' )
         self._printAddresses( addresses.ipv6Addresses, host, 'IPV6' )

   def _printAddresses( self, addresses, host, ipType ):
      firstLineFlag = True
      for addr in addresses:
         if ipType == 'IPV6':
            addr = addr.stringValue
         if firstLineFlag:
            print ( self._formatHostMapping( addr, ipType, host ) )
            firstLineFlag = False
         else:
            print ( self._formatHostMapping( addr ) )

   def _formatHostMapping( self, address, ipType='', name='' ):
      return '%-*s %4s    %s' % ( MAX_NAME_LEN, name, ipType, address )
