#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import LazyMount
from CliModel import Bool
from CliModel import List
from CliModel import Dict
from CliModel import Model
from IraIp6Model import IpAddress
from IntfModels import Interface

arpConfig = None
ip6StatusDir = None

class NdProxyPrefixInterface( Model ):
   ndProxyConnected = Bool( help="ND Proxy connected prefix is enabled" )
   ip6Configured = Bool( help="Interface has an IPv6 address configured" )
   ndProxyPrefixes = List( valueType=IpAddress,                                  
                                    help="Prefixes with ND Proxy enabled" )

   def render( self ):
      if self.ndProxyConnected or self.ndProxyPrefixes:
         print "  ND Proxy is enabled"
         if not self.ip6Configured:
            print "  No IPv6 address configured"
         else:
            if self.ndProxyConnected:
               print "    Connected"
            if self.ndProxyPrefixes:
               for prefix in self.ndProxyPrefixes:
                  print "   ", prefix.subnet
      else:
         print "  ND Proxy is disabled"      
         if not self.ip6Configured:
            print "  No IPv6 address configured"

class NdProxyPrefixCollection( Model ):
   ndProxyInterfaces = Dict( keyType=Interface, valueType=NdProxyPrefixInterface,
                              help="A mapping of Interfaces to ND Proxy prefixes" )

   def render( self ):
      if self.ndProxyInterfaces:
         for key in sorted( self.ndProxyInterfaces ):
            print key
            self.ndProxyInterfaces[ key ].render()

def getIntfProxy( intfName ):
   intfProxy = NdProxyPrefixInterface()
   status = ip6StatusDir.intf.get( intfName )
   intfProxy.ip6Configured = status is not None
   arpIntfConfig = arpConfig.arpIntfConfig.get( intfName )
   if not arpIntfConfig:
      intfProxy.ndProxyConnected = False
      return intfProxy
   intfProxy.ndProxyConnected = arpIntfConfig.proxyNdPrefixConnected
   ndPrefixSet = arpIntfConfig.proxyNdPrefix

   for addr in ndPrefixSet:
      dadfailed = ( status and status.dadfailed and addr in status.dadfailed )
      address = IpAddress( address=addr.address.stringValue,
                           subnet=addr.subnet.stringValue,
                           active=( status and addr in status.addr ), 
                           dadfailed=bool( dadfailed ) )
      intfProxy.ndProxyPrefixes.append( address )
   return intfProxy

def showNdProxyPrefix( mode, args ):
   intf = args.get( 'INTF' )
   result = NdProxyPrefixCollection()
   if intf:
      result.ndProxyInterfaces[ intf.name ] = getIntfProxy( intf.name )
   else:
      for intfName in arpConfig.arpIntfConfig:
         result.ndProxyInterfaces[ intfName ] = getIntfProxy( intfName )
   return result

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ip6StatusDir, arpConfig
   ip6StatusDir = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   arpConfig = LazyMount.mount( entityManager, "arp/config", "Arp::Config", "r" )
