#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This file has the show command of the Identity package.
# It's in LLDP to prevent a dependency cycle from forming, as LLDP already depends
# on Identity but this show commands uses LLDP information in the description column.

from __future__ import absolute_import, division, print_function

import itertools

import BasicCli
import CliMatcher
import Intf.IntfRange as IntfRange
import ShowCommand
import LazyMount
import Tac

import CliPlugin.TechSupportCli
from CliPlugin.MacAddr import MacAddrMatcher
from CliPlugin.NbrClassificationCliModel import ( allClassifications,
                                                  NbrClassificationsModel )

nbrClassificationStatus = None
lldpStatus = None

classifications = dict( itertools.izip( allClassifications(),
                                        [ 'Show neighbor classified as phones' ] ) )

class ShowNbrClassification( ShowCommand.ShowCliCommandClass ):
   syntax = '''show neighbor classification [ ( interface INTERFACE ) |
                                              ( mac address MACADDRESS ) |
                                              CLASSIFICATION ]'''

   data = {
      'neighbor': 'Show neighbor information',
      'classification': 'Show neighbor classification information',

      'interface': 'Show neighbor classification information for given interface',
      'INTERFACE': IntfRange.IntfRangeMatcher(),

      'mac': 'Show neighbor classification information for given MAC address',
      'address': 'Show neighbor classification information for given MAC address',
      'MACADDRESS': MacAddrMatcher(),

      'CLASSIFICATION': CliMatcher.EnumMatcher( classifications ),
   }

   cliModel = NbrClassificationsModel

   @staticmethod
   def handler( mode, args ):
      res = NbrClassificationsModel()
      res.populate( mode, nbrClassificationStatus, lldpStatus,
                    args.get( 'CLASSIFICATION' ), args.get( 'INTERFACE' ),
                    args.get( 'MACADDRESS' ) )
      return res

BasicCli.addShowCommandClass( ShowNbrClassification )

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
                                '2019-07-18 05:38:20',
                                lambda: [ 'show neighbor classification' ] )

def Plugin( entityManager ):
   global nbrClassificationStatus
   nbrClassificationStatus = LazyMount.mount( entityManager,
                                 'identity/nbrClassification/status',
                                 'Identity::NbrClassification::Status',
                                 'r' )
   global lldpStatus
   lldpStatus = LazyMount.mount( entityManager,
                                 'l2discovery/lldp/status/all',
                                 'Lldp::AllStatus',
                                 'r' )
