#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import itertools

import Toggles.LldpToggleLib as LldpToggle

import ConfigMount
from TypeFuture import TacLazyType

import BasicCli
import CliParser
import CliCommand
import CliMatcher
from CliPlugin.MacAddr import macAddrMatcher
from CliPlugin.NbrClassificationCliModel import allClassifications

EthAddr = TacLazyType( "Arnet::EthAddr" )
IntfId = TacLazyType( "Arnet::IntfId" )
Neighbor = TacLazyType( "Identity::NbrClassification::Neighbor" )
Source = TacLazyType( "Identity::NbrClassification::Source" )
Classification = TacLazyType( "Identity::NbrClassification::Classification" )

identityCliConfig = None

classifications = dict( itertools.izip( allClassifications(),
                                        [ 'Set classification as a phone' ] ) )

#------------------------------------------------------------------------------------
# [ no | default ] neighbor classification mac address MACADDRESS ( CLASSIFICATION |
#                                                                   none )
#------------------------------------------------------------------------------------
class SetNeighborClassificationCmd( CliCommand.CliCommandClass ):
   syntax = '''neighbor classification mac address MACADDRESS ( CLASSIFICATION |
                                                                none )'''
   noOrDefaultSyntax = 'neighbor classification mac address MACADDRESS ...'
   data = {
      # TODO: agree with cli-review help messages
      'neighbor' : 'Set neighbor details',
      'classification' : 'Set neighbor classification',

      'mac' : 'Set neighbor classification for given MAC address',
      'address' : 'Set neighbor classification for given MAC address',
      'MACADDRESS' : macAddrMatcher,

      'CLASSIFICATION' : CliMatcher.EnumMatcher( classifications ),
      'none' : 'Set as not classified',
   }

   @staticmethod
   def handler( mode, args ):
      phoneKw = [ 'phone', 'telephone' ]
      classificationArg = args.get( 'CLASSIFICATION' )
      mac = args[ 'MACADDRESS' ]

      if classificationArg and classificationArg not in phoneKw:
         raise CliParser.InvalidInputError(
                              "unknown classification %s" % classificationArg )

      classification = Classification()
      if 'none' not in args:
         classification.telephone = classificationArg in phoneKw

      identityCliConfig.neighbor.addMember( Neighbor( mac,
                                                      IntfId(),
                                                      classification,
                                                      Source.sourceCli ) )

   @staticmethod
   def noHandler( mode, args ):
      mac = args[ 'MACADDRESS' ]
      del identityCliConfig.neighbor[ mac ]

   @staticmethod
   def defaultHandler( mode, args ):
      SetNeighborClassificationCmd.noHandler( mode, args )

if LldpToggle.toggleLldpNbrClassificationCliEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( SetNeighborClassificationCmd )

def Plugin( entityManager ):
   global identityCliConfig
   identityCliConfig = ConfigMount.mount( entityManager,
                                        "identity/nbrClassification/cliConfig",
                                        "Identity::NbrClassification::CliConfig",
                                        "w" )
