# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from ArnetModel import MacAddress
from CliModel import Model, DeferredModel, Str, List, Dict
from CliPlugin.IntfCli import Intf
from CliPlugin.MacAddr import compareMacs
from CliPlugin.LldpStatusCli import chassisIdToStr
from IntfModels import Interface
from operator import attrgetter
import Tac
from TypeFuture import TacLazyType
import TableOutput

Source = TacLazyType( "Identity::NbrClassification::Source" )

@Tac.memoize
def allClassifications():
   return [ 'phone' ]

def findLldpRemoteSystem( lldpStatus, intfId, ethAddr ):
   lldpPortStatus = lldpStatus.portStatus.get( intfId )
   if not lldpPortStatus:
      return None
   for lldpRemoteSystem in lldpPortStatus.remoteSystem.values():
      if lldpRemoteSystem.ethAddr == ethAddr:
         return lldpRemoteSystem
   return None

class NbrClassificationModel( DeferredModel ):
   _ethAddr = MacAddress( help="Ethernet address of neighbor" )
   interface = Interface( help="Interface that neighbor is connected to" )
   description = Str( help="Description that helps identify neighbor" )
   classification = List( valueType=str, help="Classifications" )

   def populate( self, mode, neighborStatus, lldpStatus ):
      self._ethAddr = neighborStatus.ethAddr
      self.interface = neighborStatus.intfId
      lldpRemoteSystem = findLldpRemoteSystem( lldpStatus,
                                               neighborStatus.intfId,
                                               neighborStatus.ethAddr )
      if neighborStatus.source == Source.sourceCli:
         self.description = 'User defined' # TODO: agree with cli-review
      elif lldpRemoteSystem is not None:
         self.description = 'Chassis ID {}'.format(
            chassisIdToStr( lldpRemoteSystem.msap.chassisIdentifier,
                            short=False, canonical=False ) )
      else:
         self.description = ''
      if neighborStatus.classification.telephone:
         self.classification.append( 'telephone' )

class NbrClassificationsModel( Model ):
   neighbors = Dict( keyType=MacAddress, valueType=NbrClassificationModel,
         help="A dictionary of neighbors classifications keyed by ethernet address" )

   def populate( self, mode, classificationStatus, lldpStatus,
                 classification, intf, macAddr ):
      def intfMaches( nbrIntf ):
         if not intf:
            return True
         return nbrIntf in intf.intfNames() or nbrIntf in intf.intfNames( True )

      def macAddrMatches( nbrAddr ):
         if not macAddr:
            return True
         return compareMacs( nbrAddr, macAddr )

      def classificationMatches( nbrClassification ):
         if not classification:
            return True
         if classification == 'phone':
            return nbrClassification.telephone
         return False

      if classification and not classification in allClassifications():
         mode.addWarning( "Undefined classification '%s'" % classification )
      for neighbor in classificationStatus.neighbor.values():
         if not intfMaches( neighbor.intfId ):
            continue
         if not macAddrMatches( neighbor.ethAddr ):
            continue
         if not classificationMatches( neighbor.classification ):
            continue
         nbrItem = NbrClassificationModel()
         nbrItem.populate( mode, neighbor, lldpStatus )
         self.neighbors[ neighbor.ethAddr ] = nbrItem

   def render( self ):
      table = TableOutput.createTable( [ 'Port',
                                         'Mac Address',
                                         'Description',
                                         'Classifications' ] )
      fmt = TableOutput.Format( justify='left' )
      fmtWrap = TableOutput.Format( justify='left', wrap=True )
      table.formatColumns( fmt, fmt, fmtWrap, fmtWrap )
      for nbr in sorted( self.neighbors.values(), key=attrgetter( 'interface' ) ):
         # pylint: disable=protected-access
         classification = ', '.join( nbr.classification )
         classification = classification.replace( 'telephone', 'phone' )
         table.newRow( Intf.getShortname( nbr.interface ),
                       nbr._ethAddr.displayString,
                       nbr.description,
                       classification )
      print( table.output() )
