#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from operator import attrgetter
import Tac
from ArnetModel import IpGenericAddress
from CliModel import Model, Dict, List, Enum
import TableOutput

def addrForUI( addr ):
   if addr in ( '0.0.0.0', '::' ):
      return '*'
   else:
      return addr

def tunnelTypeForUI( tunnelType ):
   if tunnelType == "pimSsm":
      return "PIM-SSM"
   elif tunnelType == "pimSm":
      return "PIM-ASM"
   elif tunnelType == "pimBidir":
      return "PIM-BIDIR"
   else:
      return "Unknown"

def tunnelTypeForCapi( tunnelType ):
   '''tunnelTypeFooBar --> fooBar'''

   ttfc = tunnelType.strip( "tunnelType" )
   return ttfc[ 0 ].lower() + ttfc[ 1 : ]

class MulticastEncapSourceGroup( Model ):
   group = IpGenericAddress( help="Destination group address used in tunnel "
                             "encapsulation" )
   source = IpGenericAddress( help="Source address used in tunnel encapsulation" )
   tunnelType = Enum( help='Tunnel type', optional=True,
                      values=( "pimSsm", "pimSm", "pimBidir" ) )

class MulticastTunnelSource( Model ):
   groups = List( help="A list of encapsulating groups for this (S,G)",
                  valueType=MulticastEncapSourceGroup )

class MulticastTunnelGroup( Model ):
   sources = Dict( help="A mapping of source to encapsulating multicast sources "
                   " and groups", keyType=IpGenericAddress,
                   valueType=MulticastTunnelSource )

class MulticastTunnelVlan( Model ):
   groups = Dict( help="A mapping of a multicast group in a VLAN to traffic sources",
                  keyType=IpGenericAddress, valueType=MulticastTunnelGroup )

class MulticastTunnelVlans( Model ):
   vlans = Dict( help="A mapping of VLAN ID to multicast tunnel encapsuslation "
                 "information", keyType=int, valueType=MulticastTunnelVlan )

   def render( self ):

      headers = ( "VLAN", "Group", "Source", "Encap Group", "Encap Source",
                  "Tunnel Type" )

      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )

      table = TableOutput.createTable( headers )
      table.formatColumns( formatLeft, formatLeft, formatLeft, formatLeft,
                           formatLeft )

      for vlan, groups in sorted( self.vlans.items() ):
         for group, sources in sorted( groups.groups.items() ):
            for source, underlayGroups in sorted( sources.sources.items() ):
               underlayGroups = sorted( underlayGroups.groups,
                                        key=attrgetter( 'group', 'source',
                                                        'tunnelType' ) )
               for underlayGroup in underlayGroups:
                  table.newRow( vlan,
                                addrForUI( group ),
                                addrForUI( source ),
                                addrForUI( underlayGroup.group ),
                                addrForUI( underlayGroup.source ),
                                tunnelTypeForUI( underlayGroup.tunnelType ) )

      print table.output()

   def initFromTacModel( self, vlanGroupStatus ):

      for vlanGroup, underlayGroups in \
          vlanGroupStatus.vlanToMulticastGroup.iteritems():
         vlanModel = self.vlans.get( vlanGroup.vlanId )
         if vlanModel is None:
            vlanModel = MulticastTunnelVlan()
            self.vlans[ vlanGroup.vlanId ] = vlanModel

         groupModel = vlanModel.groups.get( vlanGroup.group )
         if groupModel is None:
            groupModel = MulticastTunnelGroup()
            vlanModel.groups[ vlanGroup.group ] = groupModel

         source = "0.0.0.0" if vlanGroup.group.af == "ipv4" else "::"
         sourceModel = groupModel.sources.get( source )
         if sourceModel is None:
            sourceModel = MulticastTunnelSource()
            groupModel.sources[ source ] = sourceModel

         for underlayGroup in underlayGroups.multicastGroup:
            encapGroupModel = MulticastEncapSourceGroup()
            encapGroupModel.group = underlayGroup.group.stringValue
            encapGroupModel.source = underlayGroup.source.stringValue
            encapGroupModel.tunnelType = \
               tunnelTypeForCapi( underlayGroup.tunnelType )
            sourceModel.groups.append( encapGroupModel )

class MulticastFloodVlan( Model ):
   floodGroups = List( help="A list of encapsulating (S,G) for a VLAN",
                       valueType=MulticastEncapSourceGroup )

class MulticastFloodVlans( Model ):
   vlans = Dict( help="A mapping of VLAN ID to multicast tunnel encapsuslation "
                 "information for broadcast, unknown unicast and link-local "
                 "multicast traffic", keyType=int,
                 valueType=MulticastFloodVlan )

   def render( self ):
      headers = ( "VLAN", "Encap Group", "Encap Source", "Tunnel Type" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )

      table = TableOutput.createTable( headers )
      table.formatColumns( formatLeft, formatLeft, formatLeft )

      for vlan, floodGroupColl in sorted( self.vlans.items() ):
         floodGroups = sorted( floodGroupColl.floodGroups,
                               key=attrgetter( 'group', 'source', 'tunnelType' ) )
         for floodGroup in floodGroups:
            table.newRow( vlan,
                          addrForUI( floodGroup.group ),
                          addrForUI( floodGroup.source ),
                          tunnelTypeForUI( floodGroup.tunnelType ) )
      print table.output()

   def initFromTacModel( self, vlanGroupStatus ):
      # vlanGroupStatus is of type Routing::Multicast::IpTunnelVlanGroupStatus
      for vlanFloodGroup in vlanGroupStatus.vlanToFloodGroup.values():
         floodGroupColl = MulticastFloodVlan()
         for underlayRoute in vlanFloodGroup.floodGroup:
            floodGroup = MulticastEncapSourceGroup()
            floodGroup.source = underlayRoute.source.stringValue
            floodGroup.group = underlayRoute.group.stringValue
            floodGroup.tunnelType = tunnelTypeForCapi( underlayRoute.tunnelType )
            floodGroupColl.floodGroups.append( floodGroup )
         self.vlans[ vlanFloodGroup.vlanId ] = floodGroupColl
