#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac
import TableOutput
import Toggles.MssToggleLib as MssToggleLib
from ArnetModel import IpGenericPrefix, IpGenericAddr
from CliModel import Model, Str, Bool, Int, Dict, OrderedDict, Submodel, List, Enum
from MssCliLib import ( abbreviateState, externalState )

policyRunningState = Tac.Type( "Mss::PolicyRunningState" )
POLICY_RUNNING_STATES = [ policyStatus
      for policyStatus in policyRunningState.attributes ]

policyConfigState = Tac.Type( "Mss::PolicyConfigState" )
POLICY_CONFIG_STATES = [ policyConfig
      for policyConfig in policyConfigState.attributes ]

LocalData = Tac.Type( 'MssL3V2::LocalData' )
MssL3MssPolicyModifier = Tac.Type( 'MssL3::MssPolicyModifier' )
MssRuleState = Tac.Type( "MssL3::MssRuleState" )
MssRuleStateV2 = Tac.Type( "MssL3V2::MssPolicyState" )
MssL3V2MssPriority = Tac.Value( 'MssL3V2::MssPriority' )
RULE_STATES = [ ruleState for ruleState in MssRuleState.attributes ]
RULE_STATES_V2 = [ externalState( ruleState ) for ruleState in
                   MssRuleStateV2.attributes ]
RULE_STATES.extend( RULE_STATES_V2 )
RULE_STATES.append( 'N/A' )
RULE_STATES.append( LocalData.internalRedirectPolicyName )
FLOW_STATES = [ externalState( ruleState ) for ruleState in
                MssRuleStateV2.attributes ]
IP_STATES_V2 = FLOW_STATES

MssAction = Tac.Type( "MssL3::MssAction" )
FLOW_ACTIONS = [ ruleState for ruleState in MssAction.attributes ]

mssIpState = Tac.Type( "MssL3::MssIpState" )
IP_STATES = [ ipAddrState for ipAddrState in mssIpState.attributes ]
IP_STATES.extend( IP_STATES_V2 )

EnforcementConsistency = Tac.Type( 'Mss::PolicyEnforcementConsistency' )
ENFORCEMENT_CONSISTENCY = [ mode for mode in EnforcementConsistency.attributes ]

INDENT_LEN = 2
INDENT_BLOCK = ' ' * INDENT_LEN
DIVIDER_LEN = 80
DIVIDER = '-' * DIVIDER_LEN

SHORT_DIVIDER_LEN = 67
SHORT_DIVIDER = '-' * SHORT_DIVIDER_LEN

StatusCode = 'Status Codes: P - Pending, A - Active, E - Error,'
StatusCode += ' AS - Allocation Success,\n              AP - Allocation Pending'
StatusCode += ', AE - Allocation Error, IR - Internal-Redirect'

def getIndent( indent ):
   return INDENT_BLOCK * indent

def printIndentedLine( label=None, value=None, indent=0 ):
   label = ( '' if not label else "%s: " % label )
   value = ( '' if not value else value )
   print "%s%s%s" % ( getIndent( indent ), label, value )

#------------------------------------------------------------------------------
# Model for "show service mss status"
#------------------------------------------------------------------------------
class MssPolicyEnforcementModel( Model ):
   __public__ = False
   group = Bool( help='Group policy enforcement rule enabled' )
   verbatim = Bool( help='Verbtim policy enforcement rule enabled' )

class MssStatusModel( Model ):
   __public__ = False

   enabled = Bool( help='The configured state of the Macro-Segmentation Service.' )
   policyEnforcementRules = Submodel( valueType=MssPolicyEnforcementModel,
                                      help='Policy Enforcement rules' )
   def render( self ):
      mssState = 'Enabled' if self.enabled else 'Disabled'
      print 'State: %s' % mssState

      rules = []
      if self.policyEnforcementRules.group:
         rules.append( 'group' )
      if self.policyEnforcementRules.verbatim:
         rules.append( 'verbatim' )
      print 'Policy Enforcement rules: %s' % ','.join( rules )

#------------------------------------------------------------------------------
# Models for:
# show service mss policy [ name < policy - name > ]
#                         [ source (static|<dyn-src-name>) ]
#                         [ device <device-name>
#                            [ virtual instance < name > ] [ vrf < name > ] ]
#                         [ switch <switch-id> ]
#                         [ ipv4 <PREFIX> ]
#                         [ ipv4 range   <START>    <END>    ]
#                         [ active ]
#                         [ group-redirect ]
#                         [ detail ]
#------------------------------------------------------------------------------
class MssPolicyL2Model( Model ):
   __public__ = False

   config = Enum( values=POLICY_CONFIG_STATES,
         help='The configured state of the policy' )
   status = Enum( values=POLICY_RUNNING_STATES,
         help='The operational state of the policy' )

class SwitchIdModel( Model ):
   __public__ = False

   switchId = Str( help='Switch chassis id' )
   hostname = Str( help='Hostname of the switch', optional=True )

class MssPolicyIpModel( Model ):
   __public__ = False

   state = Enum( values=IP_STATES,
                 help='The status of the offload rule, if any' )
   deleting = Bool( help="Whether the IP is in the process of being deleted" )
   unconvergedSwitches = List( valueType=SwitchIdModel, optional=True,
         help='List of unconverged switches for this intercept IP' )

class MssPolicyModifierModel( Model ):
   __public__ = False

   policyModifier = Enum( values=MssL3MssPolicyModifier.attributes,
                          help='Policy modifier configured for this policy' )

class FlowMatchModel( Model ):
   __public__ = False

   ethertype = Int( help='Ethernet type', optional=True )
   srcIpv4 = IpGenericPrefix( help="Flow source IPv4 address", optional=True )
   dstIpv4 = IpGenericPrefix( help="Flow destination IPv4 address", optional=True )
   srcGroup = Str( help='Source flow group', optional=True )
   dstGroup = Str( help='Destination flow group', optional=True )
   ipProto = Int( help='L4 protocol', optional=True )
   srcL4 = Int( help='Source L4 port', optional=True )
   dstL4 = Int( help='Destination L4 port', optional=True )

class NextHopActionModel( Model ):
   __public__ = False

   v4 = List( valueType=IpGenericAddr, help="IPv4 nextHop list" )

class FlowActionModel( Model ):
   __public__ = False

   nextHop = Submodel(
      help="Redirect action", valueType=NextHopActionModel, optional=True )
   drop = Bool( help="Packets will be dropped", optional=True )
   bypass = Bool( help="Packets will be forwarded normally", optional=True )

class FlowInfoModel( Model ):
   __public__ = False

   match = Submodel( help="Match fields for the flow", valueType=FlowMatchModel )
   action = Submodel( help='The action of the flow', valueType=FlowActionModel )
   priority = Int( help="Flow priority" )
   state = Enum( values=FLOW_STATES, help='The status of the flow' )
   deleting = Bool( help="This flow is in the process of being deleted" )
   unconvergedSwitches = List( valueType=SwitchIdModel, optional=True,
                               help='List of unconverged switches' )

class MssPolicyL3Model( Model ):
   __public__ = False

   tags = Str( help='Policy tags on the service device', optional=True )
   offloadStatus = Enum( values=RULE_STATES,
                         help='The status of the offload rule, if any' )
   redirectStatus = Enum( values=RULE_STATES,
                          help='The status of the redirect rule, if any' )
   dependencies = List( valueType=str, optional=True,
      help="List of policies that this policy depends on" )
   policyModifierSet = List( valueType=MssPolicyModifierModel,
                             help='Policy Modifier' )
   internal = Bool( help="This is an internal MSS policy" )
   ips = Dict( keyType=str, valueType=MssPolicyIpModel, optional=True,
         help='A mapping of intercept IP addresses in this policy to their status' )
   flows = Dict( valueType=FlowInfoModel,
         help='Flows in this policy' )

class MssPolicyNetworkVrfModel( Model ):
   __public__ = False

   policiesL2 = Dict( valueType=MssPolicyL2Model,
         help='A mapping from L2 policy name to its properties', optional=True )

   policiesL3 = OrderedDict( valueType=MssPolicyL3Model,
         help='A mapping from L3 policy name to its properties' )

class MssPolicyVsysModel( Model ):
   __public__ = False

   networkVrfs = Dict( valueType=MssPolicyNetworkVrfModel,
         help='A mapping from a network VRF to policies' )

class MssPolicyDeviceModel( Model ):
   __public__ = False

   virtualInstances = Dict( valueType=MssPolicyVsysModel,
         help='A mapping from a device virtual instances to network VRFs' )

class MssPolicySourceModel( Model ):
   __public__ = False

   devices = Dict( valueType=MssPolicyDeviceModel,
         help='A mapping from a service device to virtual instances' )

class MssPolicyRootModel( Model ):
   __public__ = False
   _detail = Bool( help="If the display is a detailed version" )

   sources = Dict( valueType=MssPolicySourceModel,
         help='A mapping from a source name to service devices' )

   policyEnforcementConsistency = Enum(
      values=ENFORCEMENT_CONSISTENCY, help='The Policy Enfocement Consistency mode' )

   def isDetail( self ):
      return self._detail

   def setDetail( self, detail ):
      self._detail = detail

   def render( self ) :
      v2 = MssToggleLib.toggleMssL3V2Enabled() and \
           self.policyEnforcementConsistency == EnforcementConsistency.strict
      if self._detail:
         self.renderDetail( v2 )
      else:
         self.renderSummary( v2 )

   def renderSummary( self, v2 ) :
      tableOutput = TableOutput.createTable( (
         "Source", "Device", "Policy", "Configured State", "Operational State" ) )

      fmt = TableOutput.Format( justify='center' )
      tableOutput.formatColumns( *( [ fmt ] * 4 ) )
      table = []

      for source, sourceModel in sorted( self.sources.iteritems() ):
         for device, deviceModel in sorted( sourceModel.devices.iteritems() ):
            for _, vsysModel in deviceModel.virtualInstances.iteritems():
               for _, vrfModel in vsysModel.networkVrfs.iteritems():
                  for policy, policyModel in sorted(
                        vrfModel.policiesL2.iteritems() ):
                     table.append( ( source, device, policy,
                                     str( policyModel.config ),
                                     str( policyModel.status ) ) )
      print " " * 16 + "Macro-Segmentation L2 Policy Table\n" + "-" * 79
      for entry in sorted( table, key=lambda x: ( x[ 0 ], x[ 1 ], x[ 2 ], x[ 3 ] ) ):
         tableOutput.newRow( *entry )
      print tableOutput.output()

      if v2:
         l3Comumns = [
            "Policy", "Offload\nstatus", "Redirect\nstatus", "Unconverged\nIPs",
            "Unconverged\nFlows" ]
      else:
         l3Comumns = [
            "Source", "Device", "Policy", "Offload\nstatus", "Redirect\nstatus",
            "Unconverged\nIPs" ]
      print " " * 16 + "Macro-Segmentation L3 Policy Table\n" + "-" * 79

      tableOutput = TableOutput.createTable( l3Comumns )

      fmt = TableOutput.Format( justify='center' )
      tableOutput.formatColumns( *( [ fmt ] * len( l3Comumns ) ) )
      table = []

      totalIps = 0
      totalUnconvergedIps = 0
      totalFlows = 0
      totalUnconvergedFlows = 0
      totalUnallocatedFlows = 0
      sources = sorted( self.sources.iteritems() )
      policiesPrinted = False
      for source, sourceModel in sources: # pylint: disable-msg = R1702
         for device, deviceModel in sorted( sourceModel.devices.iteritems() ):
            for vsys, vsysModel in deviceModel.virtualInstances.iteritems():
               for vrf, vrfModel in vsysModel.networkVrfs.iteritems():
                  for policy, policyModel in vrfModel.policiesL3.iteritems():
                     ipCount = 0
                     unconvergedIpCount = 0
                     flowCount = 0
                     unconvergedFlowCount = 0
                     unallocatedFlowCount = 0
                     for ipModel in sorted( policyModel.ips.itervalues() ):
                        ipCount += 1
                        if v2:
                           if ipModel.state != externalState(
                                 MssRuleStateV2.active ):
                              unconvergedIpCount += 1
                        elif ipModel.state != mssIpState.ipActive :
                           unconvergedIpCount += 1
                     modifierModel = MssPolicyModifierModel()
                     modifierModel.policyModifier = MssL3MssPolicyModifier.verbatim
                     if source != LocalData.zombieSourceName  and \
                        ( LocalData.zombiePolicyName in policy or \
                          modifierModel in policyModel.policyModifierSet or \
                          LocalData.internalRedirectPolicyName in policy ):
                        unconvergedIpStr = 'N/A'
                     else:
                        unconvergedIpStr = \
                              str( unconvergedIpCount ) + " of " + str( ipCount )
                     unconvergedFlowStr = 'N/A'
                     if v2:
                        for flowModel in sorted( policyModel.flows.itervalues() ):
                           flowCount += 1
                           if flowModel.state != externalState(
                                 MssRuleStateV2.active ):
                              unconvergedFlowCount += 1
                              if flowModel.state == externalState(
                                    MssRuleStateV2.dryrunPending ) or \
                                 flowModel.state == externalState(
                                    MssRuleStateV2.dryrunError ):
                                 unallocatedFlowCount += 1
                        if flowCount > 0:
                           unconvergedFlowStr = str( unconvergedFlowCount ) + ' of '
                           unconvergedFlowStr += str( flowCount )
                        redirectStatus = abbreviateState(
                           policyModel.redirectStatus )
                        if LocalData.internalRedirectPolicyName in \
                                                           policyModel.dependencies:
                           redirectStatus += ' [IR]'
                        if LocalData.internalRedirectPolicyName in policy:
                           policy += '*'
                        tableEntries = [
                           policy,
                           abbreviateState( policyModel.offloadStatus ),
                           redirectStatus,
                           unconvergedIpStr, unconvergedFlowStr
                        ]
                     else:
                        tableEntries = [
                           source, device, policy,
                           externalState( policyModel.offloadStatus ),
                           externalState( policyModel.redirectStatus ),
                           unconvergedIpStr
                        ]
                     table.append( tableEntries )

                     if unconvergedIpStr != 'N/A':
                        totalIps += ipCount
                        totalUnconvergedIps += unconvergedIpCount
                     if unconvergedFlowStr != 'N/A':
                        totalFlows += flowCount
                        totalUnconvergedFlows += unconvergedFlowCount
                        totalUnallocatedFlows += unallocatedFlowCount
                  if v2:
                     policiesPrinted = True
                     print "\nSource:", source + ", Device:",
                     print device + ' [' + vsys + '],', # pylint: disable-msg = W0631
                     print 'Net. VRF:', vrf           # pylint: disable-msg = W0631
                     print "-" * tableOutput.printableWidth()
                     for entry in table:
                        tableOutput.newRow( *entry )
                     print tableOutput.output()
                     table = []
                     tableOutput = TableOutput.createTable( l3Comumns )
                     fmt = TableOutput.Format( justify='center' )
                     tableOutput.formatColumns( *( [ fmt ] * len( l3Comumns ) ) )
                     print '-' * tableOutput.printableWidth()

      if not v2:
         table = sorted( table, key=lambda x: ( x[ 0 ], x[ 1 ], x[ 2 ],
                                                x[ 3 ], x[ 4 ], x[ 5 ] ) )
         for entry in table:
            tableOutput.newRow( *entry )
         print tableOutput.output()
      elif policiesPrinted:
         unconvergedIpStr = str( totalUnconvergedIps ) + " of " + str(
            totalIps )
         unconvergedFlowStr = str( totalUnconvergedFlows ) + ' of '
         unconvergedFlowStr += str( totalFlows ) + ' ('
         unconvergedFlowStr += str( totalUnallocatedFlows )
         unconvergedFlowStr += ' Unallocated Resources)'
         print '    ' + 'Total Unconverged IPs:   ' + unconvergedIpStr
         print '    ' + 'Total Unconverged Flows: ' + unconvergedFlowStr
         print '* - MSS internal policy'
         print StatusCode
      else:
         print tableOutput.output()

   def renderDetail( self, v2 ) :
      def renderPolicyL2Model( policyModel, indent ):
         printIndentedLine( "Config", policyModel.config, indent )
         printIndentedLine( "Status", policyModel.status, indent )

      def renderFlowInfoModel( flowModel, flow, policyName, indent ):
         if v2 and flowModel.priority == MssL3V2MssPriority.ipRedirect and \
            LocalData.internalRedirectPolicyName not in policyName:
            return
         printIndentedLine( 'Flow', flow, indent - 1 )
         printIndentedLine( "Priority", flowModel.priority, indent )
         printIndentedLine( "Match", '', indent )
         printIndentedLine( 'EtherType', flowModel.match.ethertype, indent + 1 )
         printIndentedLine( 'Source IPv4 address',
                            flowModel.match.srcIpv4, indent + 1 )
         printIndentedLine( 'Destination IPv4 address',
                            flowModel.match.dstIpv4, indent + 1 )
         if flowModel.match.srcL4:
            printIndentedLine( 'Source TCP/UDP port', flowModel.match.srcL4,
                               indent + 1 )
         if flowModel.match.dstL4:
            printIndentedLine( 'Destination TCP/UDP port', flowModel.match.dstL4,
                               indent + 1 )
         printIndentedLine( "Action", '', indent )
         if flowModel.action.drop:
            actionText = "Drop"
         elif flowModel.action.bypass:
            actionText = "Bypass"
         else:
            actionText = 'NextHop: [ '
            actionText += ', '.join( [
               hop.ip.stringValue  # pylint: disable=no-member
               for hop in flowModel.action.nextHop.v4 ] )
            actionText += ' ]'
         printIndentedLine( '', actionText, indent + 1 )
         stateText = externalState( flowModel.state )
         if v2 and flowModel.deleting:
            stateText += ' [deleting]'
         printIndentedLine( "Status", stateText, indent )
         if flowModel.unconvergedSwitches:
            printIndentedLine( "Unconverged Switches", '', indent + 1 )
            for switch in sorted( flowModel.unconvergedSwitches,
                                  key=lambda switch: switch.switchId ):
               if switch.hostname:
                  switchid = switch.hostname + ' (' + switch.switchId + ')'
               else:
                  switchid = switch.switchId
               printIndentedLine( '', switchid, indent + 2 )

      def renderPolicyL3Model( policyModel, policyName, indent ):
         if v2:
            printIndentedLine(
               "Internal", "True" if policyModel.internal else "False", indent )
         printIndentedLine( "Offload Status",
                            externalState( policyModel.offloadStatus ), indent )

         for flow, flowModel in sorted( policyModel.flows.iteritems() ):
            if flowModel.action.drop or flowModel.action.bypass:
               renderFlowInfoModel( flowModel, flow, policyName, indent + 2 )

         printIndentedLine( "Redirect Status",
                            externalState( policyModel.redirectStatus ), indent )

         for flow, flowModel in sorted( policyModel.flows.iteritems() ):
            if not ( flowModel.action.drop or flowModel.action.bypass ) :
               renderFlowInfoModel( flowModel, flow, policyName, indent + 2 )

         printIndentedLine( "Tags", policyModel.tags, indent )
         if policyModel.policyModifierSet:
            modifiers = \
             [ i.policyModifier.capitalize() for i in policyModel.policyModifierSet ]
            printIndentedLine( "Policy Modifier", ', '.join( modifiers ), indent )
         if policyModel.ips:    # pylint: disable-msg = R1702
            printIndentedLine( "IP Addresses", '', indent )
            for ip, ipModel in sorted( policyModel.ips.iteritems() ):
               isText = ip
               if v2:
                  isText += '  '
                  if ipModel.deleting:
                     isText += ' [deleting]'
               printIndentedLine(
                  externalState( ipModel.state ), isText, indent + 1 )
               if ipModel.unconvergedSwitches:
                  if v2:
                     printIndentedLine( "Unconverged Switches", '', indent + 2 )
                     for switch in sorted( ipModel.unconvergedSwitches,
                                  key=lambda switch: switch.switchId ):
                        if switch.hostname:
                           switchid = switch.hostname + ' (' + switch.switchId + ')'
                        else:
                           switchid = switch.switchId
                        printIndentedLine( '', switchid, indent + 3 )
                  else:
                     printIndentedLine(
                        "Unconverged Switches",
                        ', '.join( [ switch.switchId for switch in
                                     ipModel.unconvergedSwitches ] ), indent + 2 )

      def renderVrfModel( vrfModel, indent ):
         for policy, policyModel in sorted( vrfModel.policiesL2.iteritems() ):
            printIndentedLine( "Policy (L2)", policy, indent )
            renderPolicyL2Model( policyModel, indent + 1 )

         for policy, policyModel in sorted( vrfModel.policiesL3.iteritems() ):
            printIndentedLine( "Policy (L3)", policy, indent )
            renderPolicyL3Model( policyModel, policy, indent + 1 )

      def renderVsysModel( vsysModel, indent ):
         for vrf, vrfModel in sorted( vsysModel.networkVrfs.iteritems() ):
            if v2:
               printIndentedLine( "Network VRF", vrf, indent )
               renderVrfModel( vrfModel, indent + 1 )
            else:
               renderVrfModel( vrfModel, indent )

      def renderDeviceModel( deviceModel, indent ):
         for vsys, vsysModel in sorted( deviceModel.virtualInstances.iteritems() ):
            if v2:
               printIndentedLine( "Virtual Instance", vsys, indent )
               renderVsysModel( vsysModel, indent + 1 )
            else:
               renderVsysModel( vsysModel, indent )

      def renderSourceModel( sourceModel, indent ):
         for device, deviceModel in sorted( sourceModel.devices.iteritems() ):
            printIndentedLine( "Device", device, indent )
            renderDeviceModel( deviceModel, indent + 1 )

      for source, sourceModel in sorted( self.sources.iteritems() ):
         printIndentedLine( value=SHORT_DIVIDER )
         printIndentedLine( "Source", source )
         printIndentedLine( value=SHORT_DIVIDER )
         renderSourceModel( sourceModel, 1 )

#------------------------------------------------------------------------------
# Models for "show service mss vrf [device <device-name>
#                                     [virtual instance <name> [name <name>]]]"
#------------------------------------------------------------------------------
class MssVrfSubnetModel( Model ):
   __public__ = False

   subnets = List( valueType=IpGenericPrefix, help='List of subnets' )

class MssVrfNetworkVRFModel( Model ):
   __public__ = False

   internal = Bool( help='Internal routing table', default=False )
   vnis = List( valueType=int, help='VNIs associated to this VRF' )
   switches = List( valueType=SwitchIdModel,
                    help='List of switches associated to this network VRF' )
   nexthops = Dict( valueType=MssVrfSubnetModel,
                    help='Mapping from firewall nexthop address to its subnets' )

class MssVrfFwVRFModel( Model ):
   __public__ = False

   networkVRFs = Dict( valueType=MssVrfNetworkVRFModel,
                       help='Mapping from Network VRF name to associated routes' )

class MssVrfVInstModel( Model ):
   __public__ = False

   firewallVRFs = Dict( valueType=MssVrfFwVRFModel,
                        help='Mapping from firewall VRF name to its properties' )

class MssVrfDeviceModel( Model ):
   __public__ = False

   virtualInstances = Dict( valueType=MssVrfVInstModel,
                            help='Mapping from virtual instance name to its VRFs' )

class MssVrfSourceModel( Model ):
   __public__ = False

   devices = Dict( valueType=MssVrfDeviceModel,
                   help='Mapping from service device name to its virtual instances' )

class MssVrfModel( Model ):
   __public__ = False

   sources = Dict( valueType=MssVrfSourceModel,
                   help='Mapping from source name to service devices' )

   def render( self ):
      def renderL3IntfModel( ip, l3IntfModel ):
         printIndentedLine( 'Interface', ip, indent=3 )
         printIndentedLine( 'Reachable subnets', ', '.join(
            sorted( s.stringValue for s in l3IntfModel.subnets ) ), indent=4 )

      def renderVrfModel( vrfName, vrfModel ):
         def renderSwitchName( switchModel ):
            return ( switchModel.hostname + ' (' + switchModel.switchId + ')'
                     if switchModel.hostname else switchModel.switchId )
               
         for netVrfName, netVrfModel in sorted( vrfModel.networkVRFs.iteritems() ):
            print "%sVRF: %s, Network VRF: %s" % (
                  getIndent( 2 ), vrfName, netVrfName )
            printIndentedLine( 'VNI', ','.join(
               sorted( str( vni ) for vni in netVrfModel.vnis ) ), indent=3 )
            printIndentedLine( 'Switch', ', '.join(
               sorted( renderSwitchName( s ) for s in netVrfModel.switches ) ),
               indent=3 )

            for ip, l3IntfModel in sorted( netVrfModel.nexthops.iteritems() ):
               renderL3IntfModel( ip, l3IntfModel )

      def renderDeviceModel( devName, devModel ):
         for vinstName, vinstModel in sorted(
               devModel.virtualInstances.iteritems() ):
            print "%sDevice: %s, Virtual Instance: %s" % (
                  getIndent( 1 ), devName, vinstName )
            for vrfName, vrfModel in sorted( vinstModel.firewallVRFs.iteritems() ):
               renderVrfModel( vrfName, vrfModel )

      def renderSourceModel( srcName, srcModel ):
         printIndentedLine( value=SHORT_DIVIDER )
         printIndentedLine( 'Source', srcName )
         printIndentedLine( value=SHORT_DIVIDER )
         for devName, devModel in sorted( srcModel.devices.iteritems() ):
            renderDeviceModel( devName, devModel )

      for srcName, srcModel in sorted( self.sources.iteritems() ):
         renderSourceModel( srcName, srcModel )
