#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# pylint: disable=R1710

import json
import pickle

import BasicCli
import CliCommand
import CliMatcher
import LazyMount
import ShowCommand
import Tac
from Tac import Timeout

from MssPolicyMonitor.Error import ServiceDeviceError
from MssPolicyMonitor.Lib import ( checkSslProfileStatus,
                                   getDeviceConfig,
                                   mergeNeighborsWithIntfMap,
                                   CHKP_MS_PLUGIN,
                                   API_ACCESS_MSG,
                                   TIMEOUT_MSG,
                                   CLI_TIMEOUT,
                                   PATH_FOR_API,
                                   MssDeviceName )

import CliPlugin.ControllerCli
from MssPolicyMonitorModel import ( MacroSegmentationDynamicStatesModel,
                                    MacroSegmentationDeviceSetStatesModel,
                                    MacroSegmentationServiceDevicesModel,
                                    MacroSegmentationServiceDeviceStatesModel,
                                    MacroSegmentationServiceDeviceModel,
                                    MacroSegmentationDynamicModel,
                                    MacroSegmentationGroupMembersModel,
                                    MacroSegmentationGroupMembersInfoModel,
                                    MacroSegmentationHighAvailabilityModel,
                                    MacroSegmentationHighAvailabilityInfoModel,
                                    MacroSegmentationResourcesModel,
                                    MacroSegmentationResourcesInfoModel,
                                    MacroSegmentationDevicePoliciesModel,
                                    MacroSegmentationPoliciesModel,
                                    MacroSegmentationPolicyZone,
                                    MacroSegmentationDeviceNetworkModel,
                                    MacroSegmentationDeviceNeigborModel,
                                    MacroSegmentationStatusModel,
                                    MacroSegmentationPoliciesInfoModel,
                                    MacroSegmentationNetworkModel,
                                    MacroSegmentationNetworkInfoModel,
                                    MacroSegmentationNeighborModel,
                                    MacroSegmentationNeighborInfoModel )

from Toggles import MssToggleLib

mssStatus = None
mssConfig = None
mssMainConfig = None
sslStatus = None

def setJson( deviceDict, servDev=None ):
   deviceDict.pop( 'extAttr', None ) # purge unserializable Tac entity
   jsonDeviceDict = json.dumps( deviceDict )
   jArgs = { 'service-device': servDev }
   jsonArgs = json.dumps( jArgs )
   return jsonDeviceDict, jsonArgs

def getServiceDevices( mode ):
   return [ l for k in mssConfig.deviceSet.iterkeys() \
            for l in mssConfig.deviceSet[ k ].serviceDevice.iterkeys() ]

deviceSetMatcher = CliMatcher.DynamicNameMatcher(
                        lambda mode: [ k for k in mssConfig.deviceSet.iterkeys() ],
                        helpdesc='Device set name',
                        pattern='.+' )

# pylint: disable-msg=W0108
serviceDeviceMatcher = CliMatcher.DynamicNameMatcher(
                           lambda mode: getServiceDevices( mode ),
                           helpdesc='DNS hostname or IP Address of Service Device',
                           pattern='[^:|> ]+' )
cliTimeoutMatcher = CliMatcher.IntegerMatcher( 1, 100, helpdesc='Cli timeout' )
serviceKwMatcher = CliPlugin.ControllerCli.serviceAfterShowKwMatcher
mssKwNode = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'mss',
                             helpdesc='Macro-Segmentation Service' ) )

class ShowMacroSegmentationStatus( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service mss dynamic status'
   data = {
            'service': serviceKwMatcher,
            'mss': mssKwNode,
            'dynamic': 'Select policy source',
            'status': 'MssPolicyMonitor status'
          }
   cliModel = MacroSegmentationServiceDeviceStatesModel

   @staticmethod
   def handler( mode, args ):
      result = MacroSegmentationServiceDeviceStatesModel()

      def populateStatusModel( devStatus ):
         devStat = mssStatus.serviceDeviceStatus[ devStatus ]
         deviceGrpMem = [ servDevId for servDevId in devStat.groupMember ]
         haPeerMgmtIp = devStat.haPeerMgmtIp.v4Addr \
                        if not devStat.haPeerMgmtIp.isAddrZero else ''
         statusMdl = MacroSegmentationStatusModel(
            state=devStat.state,
            deviceSetName=devStat.deviceSetName,
            deviceType=devStat.policySourceType,
            mgmtIp=devStat.mgmtIp.v4Addr,
            haPeerMgmtIp=haPeerMgmtIp,
            aggregationMgr=devStat.isAggregationMgr,
            accessedVia=devStat.accessedViaAggrMgr,
            groupMembers=deviceGrpMem,
            timeLastSeen=devStat.timeLastSeen,
            numPoliciesProcessed=devStat.numPoliciesProcessed )
         return statusMdl

      deviceStatusMdl = MacroSegmentationServiceDeviceModel()
      deviceStatusMdl.deviceStatus = {
         mssStatus.serviceDeviceStatus[ devStatus ].serviceDeviceId: \
         populateStatusModel( devStatus ) \
         for devStatus in mssStatus.serviceDeviceStatus.iterkeys() }
      result.serviceDeviceStates = deviceStatusMdl
      return result

BasicCli.addShowCommandClass( ShowMacroSegmentationStatus )

class ShowMacroSegmentationDynamic( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service mss dynamic'
   data = { 'service': serviceKwMatcher,
            'mss': mssKwNode,
            'dynamic': 'Select policy source'
        }
   cliModel = MacroSegmentationDynamicModel

   @staticmethod
   def handler( mode, args ):
      result = MacroSegmentationDynamicModel()
      dynamicStatesDict = dict()
      dynamicStatesMdl = MacroSegmentationDynamicStatesModel()

      for devSet, dev in mssConfig.deviceSet.iteritems():
         dynamicMdl = MacroSegmentationDeviceSetStatesModel()
         dynamicMdl.deviceType = dev.policySourceType
         dynamicMdl.state = dev.state
         servDevDict = dict()

         for servDev, serv in dev.serviceDevice.iteritems():
            if serv.isAccessedViaAggrMgr:
               continue
            servModel = MacroSegmentationServiceDevicesModel()
            servModel.groupName = serv.group
            if dev.isAggregationMgr:
               servModel.aggregationMgr = True
            else:
               servModel.aggregationMgr = False
               servModel.groupMembers = [ 'N/A' ]
            servDevDict[ servDev ] = servModel
            dynamicMdl.serviceDevices = servDevDict
         dynamicStatesDict[ devSet ] = dynamicMdl
      dynamicStatesMdl.deviceSetStates = dynamicStatesDict
      dynamicStatesMdl.numPoliciesProcessed = mssStatus.numPoliciesProcessed

      result.dynamicStates = dynamicStatesMdl
      return result

BasicCli.addShowCommandClass( ShowMacroSegmentationDynamic )

class ShowMacroSegmentationGroupMembers( ShowCommand.ShowCliCommandClass ):
   syntax = """ show service mss dynamic device-set
                DEVICE_SET device SERVICE_DEVICE group-members
                [ cli-timeout CLI_TIMEOUT ] """
   data = { 'service': serviceKwMatcher,
            'mss': mssKwNode,
            'dynamic': 'Select policy source',
            'device-set': 'A named configuration for a set of devices',
            'DEVICE_SET': deviceSetMatcher,
            'device': 'Service device',
            'SERVICE_DEVICE': serviceDeviceMatcher,
            'group-members': 'Show device-group members for an Aggregation Manager',
            'cli-timeout': 'External API timeout',
            'CLI_TIMEOUT': cliTimeoutMatcher
        }
   cliModel = MacroSegmentationGroupMembersModel

   @staticmethod
   def handler( mode, args ):
      timeout = CLI_TIMEOUT
      if 'CLI_TIMEOUT' in args:
         timeout = args[ 'CLI_TIMEOUT' ]

      result = MacroSegmentationGroupMembersModel()

      # error message added to json errors
      if errorChecking( mode, args, 1 ) == -1:
         return

      if not mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ].isAggregationMgr:
         mode.addError( 'device set %s does not contain an Aggregation Manager'
                        % ( args[ 'DEVICE_SET' ] ) )
         return

      devSet = mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ]

      def getAggrDevice( devSet ):
         servDev = mssConfig.deviceSet[ devSet ].serviceDevice
         return [ x for x in servDev.keys()
                  if not servDev[ x ].isAccessedViaAggrMgr ].pop()

      # is it a device accessed only via an aggr device
      if args[ 'SERVICE_DEVICE' ] in mssStatus.serviceDeviceStatus and \
         mssStatus.serviceDeviceStatus[ args[ 'SERVICE_DEVICE' ] ].\
         accessedViaAggrMgr == getAggrDevice( args[ 'DEVICE_SET' ] ):
         mode.addError( '%s is not an Aggregation Manager' % (
            args[ 'SERVICE_DEVICE' ] ) )
         return

      # is the service device in the device set
      if errorChecking( mode, args, 2 ) == -1:
         return

      serviceDevice = devSet.serviceDevice[ args[ 'SERVICE_DEVICE' ] ]

      servDev = ''
      if not serviceDevice.username or not serviceDevice.encryptedPassword:
         mode.addError( 'Could not log in to device %s,'
                        'check username, password' % (
                           args[ 'DEVICE_SET' ] ) )
         return

      try:
         mode.addWarning( API_ACCESS_MSG )
         deviceDict = getDeviceConfig( devSet, serviceDevice )
         deviceDict[ 'retries' ] = 0
         deviceDict[ 'timeout' ] = timeout
         servDev = None
         
         checkSslProfileStatus( deviceDict, sslStatus )
         jsonDeviceDict, jsonArgs = setJson( deviceDict, servDev=None )

         tacRunTimeout = timeout + 2
         op = Tac.run( [ '/usr/bin/timeout', '-k3', '-s9', str( tacRunTimeout ),
                         'python', '-m', PATH_FOR_API, jsonDeviceDict, jsonArgs,
                         'getAggMgrGroupMembers' ],
                       stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
         pklOp = pickle.loads( op )
         if isinstance( pklOp, ServiceDeviceError ):
            raise pklOp
         else:
            servDev = pklOp

      except ServiceDeviceError as e:
         mode.addError( e )
         return

      except Exception: # pylint: disable=W0703
         mode.addError( TIMEOUT_MSG.format( str( timeout ) ) )
         return

      mdl = MacroSegmentationGroupMembersInfoModel(
         groupName=serviceDevice.group,
         groupMembers=servDev,
         aggregationMgrName=args[ 'SERVICE_DEVICE' ] )

      result.groupMembers = mdl
      return result

BasicCli.addShowCommandClass( ShowMacroSegmentationGroupMembers )

class ShowMacroSegmentationPolicies( ShowCommand.ShowCliCommandClass ):
   syntax = """ show service mss dynamic device-set
                DEVICE_SET device SERVICE_DEVICE policies """
   data = { 'service': serviceKwMatcher,
            'mss': mssKwNode,
            'dynamic': 'Select policy source',
            'device-set': 'A named configuration for a set of devices',
            'DEVICE_SET': deviceSetMatcher,
            'device': 'Service device',
            'SERVICE_DEVICE': serviceDeviceMatcher,
            'policies': 'Policies read from service device that have the MSS tag'
        }
   cliModel = MacroSegmentationDevicePoliciesModel

   @staticmethod
   def handler( mode, args ):
      result = MacroSegmentationDevicePoliciesModel()
      policiesMdl = MacroSegmentationPoliciesModel(
         serviceDevice=args[ 'SERVICE_DEVICE' ] )
      result.devicePolicies = policiesMdl

      if errorChecking( mode, args, 1 ) == -1:
         return

      if not mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ].isAggregationMgr:
         if errorChecking( mode, args, 2 ) == -1:
            return

      if errorChecking( mode, args, 4 ) == -1: # 4
         return result

      serviceDeviceName = args[ 'SERVICE_DEVICE' ]
      serviceDevice = []
      if ( MssToggleLib.toggleMssL3V2Enabled() and
           mssMainConfig.policyEnforcementConsistency == 'strict' ):
         serviceDevice = [ serviceDev for deviceName, serviceDev
                           in mssStatus.devicePolicyList.iteritems()
                           if MssDeviceName.fromString( deviceName, v2=True ).devName
                              == serviceDeviceName ]
      elif serviceDeviceName in mssStatus.devicePolicyList:
         serviceDevice = [ mssStatus.devicePolicyList[ serviceDeviceName ] ]

      if not serviceDevice:
         return result

      def makeZoneInfo( zone ):
         zoneModel = MacroSegmentationPolicyZone()

         if zone:
            zoneModel.name = zone.zoneName if zone else ''
            zoneModel.intf = ', '.join( [ sdl.serviceDeviceIntfStr
                                          for sdl in zone.link.values()
                                          if zone and zone.link ] )
            zoneModel.ip = [ k for k in zone.rawIntercept.keys()
                             if zone and zone.rawIntercept ]
            zoneModel.zoneClass = zone.zoneClass
         else:
            zoneModel.name = ''
            zoneModel.intf = ''
            zoneModel.ip = []
            zoneModel.zoneClass = ''

         return zoneModel

      def makePolicyInfo():
         policiesDict = {}
         for dev in serviceDevice:
            for policy, pol in dev.devicePolicy.iteritems():
               policyInfoMdl = MacroSegmentationPoliciesInfoModel()
               policyInfoMdl.action = pol.action
               policyInfoMdl.src = makeZoneInfo( pol.zoneA )
               policyInfoMdl.dest = makeZoneInfo( pol.zoneB )

               policyInfoMdl.deviceSet = args[ 'DEVICE_SET' ]
               policyInfoMdl.tag = ', '.join( t for t in pol.tag )
               policiesDict[ policy ] = policyInfoMdl

         return policiesDict

      policiesDict = makePolicyInfo()
      policiesMdl.policies = policiesDict
      policiesMdl.serviceDevice = serviceDeviceName
      result.devicePolicies = policiesMdl
      return result

BasicCli.addShowCommandClass( ShowMacroSegmentationPolicies )

class ShowMacroSegmentationNeighbors( ShowCommand.ShowCliCommandClass ):
   syntax = """ show service mss dynamic device-set
                DEVICE_SET device SERVICE_DEVICE neighbors
                [ cli-timeout CLI_TIMEOUT ] """
   data = { 'service': serviceKwMatcher,
            'mss': mssKwNode,
            'dynamic': 'Select policy source',
            'device-set': 'A named configuration for a set of devices',
            'DEVICE_SET': deviceSetMatcher,
            'device': 'Service device',
            'SERVICE_DEVICE': serviceDeviceMatcher,
            'neighbors': 'Service device ethernet interface neighbors',
            'cli-timeout': 'External API timeout',
            'CLI_TIMEOUT': cliTimeoutMatcher
        }
   cliModel = MacroSegmentationDeviceNeigborModel

   @staticmethod
   def handler( mode, args ):
      timeout = CLI_TIMEOUT
      if 'cli-timeout' in args:
         timeout = args[ 'CLI_TIMEOUT' ]

      result = MacroSegmentationDeviceNeigborModel()
      neighborDict = dict()
      neighborMdl = MacroSegmentationNeighborModel()

      if errorChecking( mode, args, 1 ) == -1:
         return

      if not mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ].isAggregationMgr:
         if errorChecking( mode, args, 2 ) == -1:
            return

      else:
         if errorChecking( mode, args, 3 ) == -1:
            return

      devSet = mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ]
      if devSet.isAggregationMgr:
         # there can be only one Aggregation Manager
         aggrMgrDevice = [ x for x in devSet.serviceDevice.keys()
            if not devSet.serviceDevice[ x ].isAccessedViaAggrMgr ].pop()
         serviceDevice = devSet.serviceDevice[ aggrMgrDevice ]
      else:
         serviceDevice = devSet.serviceDevice[ args[ 'SERVICE_DEVICE' ] ]
      if not serviceDevice.username or not serviceDevice.encryptedPassword:
         mode.addError( 'Could not log in to device %s,'
                        'check username, password' % (
                           args[ 'DEVICE_SET' ] ) )
         return

      try:
         mode.addWarning( API_ACCESS_MSG )
         deviceConfigDict = getDeviceConfig( devSet, serviceDevice )
         deviceConfigDict[ 'retries' ] = 0
         deviceConfigDict[ 'timeout' ] = timeout

         deviceId = args[ 'SERVICE_DEVICE' ]

         neighbors = None
         
         checkSslProfileStatus( deviceConfigDict, sslStatus )
         jsonDeviceDict, jsonArgs = setJson( deviceConfigDict,
            servDev=args[ 'SERVICE_DEVICE' ] )

         tacRunTimeout = timeout + 2
         op = Tac.run( [ '/usr/bin/timeout', '-k3', '-s9', str( tacRunTimeout ),
                         'python', '-m', PATH_FOR_API, jsonDeviceDict, jsonArgs,
                         'getInterfaceNeighbors' ],
                       stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
         pklOp = pickle.loads( op )

         if isinstance( pklOp, ServiceDeviceError ):
            raise pklOp
         else:
            neighbors = mergeNeighborsWithIntfMap( pklOp, deviceId, devSet )

      except ServiceDeviceError as e:
         mode.addError( e )
         return

      except Exception: # pylint: disable=W0703
         mode.addError( TIMEOUT_MSG.format( str( timeout ) ) )
         return

      for k, v in neighbors.items():
         neighborInfoMdl = MacroSegmentationNeighborInfoModel()
         neighborInfoMdl.neighborIntf = v[ 'switchIntf' ] \
                     if 'switchIntf' in v and v[ 'switchIntf' ] else ''

         neighborInfoMdl.chassisId = v[ 'switchChassisId' ] \
                             if 'switchChassisId' in v and \
                                v[ 'switchChassisId' ] else ''

         neighborInfoMdl.systemName = v[ 'nborSysName' ] \
                           if 'nborSysName' in v and v[ 'nborSysName' ] else ''

         neighborInfoMdl.managementIp = v[ 'nborMgmtIp' ] \
                          if 'nborMgmtIp' in v and v[ 'nborMgmtIp' ] else ''

         neighborInfoMdl.description = v[ 'nborDesc' ] \
                        if 'nborDesc' in v and v[ 'nborDesc' ] else ''
         neighborDict[ k ] = neighborInfoMdl
      neighborMdl.neighbors = neighborDict
      neighborMdl.serviceDevice = args[ 'SERVICE_DEVICE' ]
      result.deviceIntfNeighbors = neighborMdl
      return result

BasicCli.addShowCommandClass( ShowMacroSegmentationNeighbors )

class ShowMacroSegmentationNetwork( ShowCommand.ShowCliCommandClass ):
   syntax = """ show service mss dynamic device-set
                DEVICE_SET device SERVICE_DEVICE network
                [ cli-timeout CLI_TIMEOUT ] """
   data = { 'service': serviceKwMatcher,
            'mss': mssKwNode,
            'dynamic': 'Select policy source',
            'device-set': 'A named configuration for a set of devices',
            'DEVICE_SET': deviceSetMatcher,
            'device': 'Service device',
            'SERVICE_DEVICE': serviceDeviceMatcher,
            'network': 'Service device network interface information',
            'cli-timeout': 'External API timeout',
            'CLI_TIMEOUT': cliTimeoutMatcher
        }
   cliModel = MacroSegmentationDeviceNetworkModel

   @staticmethod
   def handler( mode, args ):
      timeout = CLI_TIMEOUT
      if 'cli-timeout' in args:
         timeout = args[ 'CLI_TIMEOUT' ]

      result = MacroSegmentationDeviceNetworkModel()
      networksDict = dict()
      networkMdl = MacroSegmentationNetworkModel()

      if errorChecking( mode, args, 1 ) == -1:
         return

      if not mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ].isAggregationMgr:
         if errorChecking( mode, args, 2 ) == -1:
            return

      else:
         if errorChecking( mode, args, 3 ) == -1:
            return

      devSet = mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ]
      if devSet.isAggregationMgr:
         # there can be only one Aggregation Manager
         aggrMgrDevice = [ x for x in devSet.serviceDevice.keys()
                        if not devSet.serviceDevice[ x ].isAccessedViaAggrMgr ].pop()
         serviceDevice = devSet.serviceDevice[ aggrMgrDevice ]
      else:
         serviceDevice = devSet.serviceDevice[ args[ 'SERVICE_DEVICE' ] ]
      if not serviceDevice.username or not serviceDevice.encryptedPassword:
         mode.addError( 'Could not log in to device %s,'
                        'check username, password' % (
                           args[ 'DEVICE_SET' ] ) )
         return

      try:
         mode.addWarning( API_ACCESS_MSG )
         devDict = getDeviceConfig( devSet, serviceDevice )
         devDict[ 'retries' ] = 0
         devDict[ 'timeout' ] = timeout
         
         networks = None
         
         checkSslProfileStatus( devDict, sslStatus )
         jsonDeviceDict, jsonArgs = setJson( devDict,
            servDev=args[ 'SERVICE_DEVICE' ] )

         tacRunTimeout = timeout + 2
         op = Tac.run( [ '/usr/bin/timeout', '-k3', '-s9', str( tacRunTimeout ),
                         'python', '-m', PATH_FOR_API, jsonDeviceDict, jsonArgs,
                         'getInterfacesInfo' ],
                       stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
         pklOp = pickle.loads( op )
         if isinstance( pklOp, ServiceDeviceError ):
            raise pklOp
         else:
            networks = pklOp

      except AttributeError:
         mode.addError( 'type is not valid, please set a valid type' )
         return

      except ServiceDeviceError as e:
         mode.addError( e )
         return

      except Exception: # pylint: disable=W0703
         mode.addError( TIMEOUT_MSG.format( str( timeout ) ) )
         return

      for each in networks:
         for nwMdl, nwMdlInfo in each[ 'interfaces' ].items() :
            networkInfoMdl = MacroSegmentationNetworkInfoModel()
            networkInfoMdl.state = nwMdlInfo[ 'state' ]
            networkInfoMdl.vlan = nwMdlInfo[ 'vlan' ]
            networkInfoMdl.mode = nwMdlInfo[ 'mode' ] if 'mode' in nwMdlInfo else ''
            networkInfoMdl.virtualWire = nwMdlInfo[ 'virtualWire' ]
            networkInfoMdl.zone = nwMdlInfo[ 'zone' ] if nwMdlInfo[ 'zone' ] else ''
            networksDict[ nwMdl ] = networkInfoMdl
            networkMdl.interfaces = networksDict
            networkMdl.serviceDevice = args[ 'SERVICE_DEVICE' ]
            result.deviceNetworks = networkMdl

      return result

BasicCli.addShowCommandClass( ShowMacroSegmentationNetwork )

class ShowMacroSegmentationResources( ShowCommand.ShowCliCommandClass ):
   syntax = """ show service mss dynamic device-set
                DEVICE_SET device SERVICE_DEVICE resources
                [ cli-timeout CLI_TIMEOUT ] """
   data = {
            'service': serviceKwMatcher,
            'mss': mssKwNode,
            'dynamic': 'Select policy source',
            'device-set': 'A named configuration for a set of devices',
            'DEVICE_SET': deviceSetMatcher,
            'device': 'Service device',
            'SERVICE_DEVICE': serviceDeviceMatcher,
            'resources': 'Service device system resource information',
            'cli-timeout': 'External API timeout',
            'CLI_TIMEOUT': cliTimeoutMatcher
        }
   cliModel = MacroSegmentationResourcesModel

   @staticmethod
   def handler( mode, args ):
      timeout = CLI_TIMEOUT
      if 'cli-timeout' in args:
         timeout = args[ 'CLI_TIMEOUT' ]

      result = MacroSegmentationResourcesModel()

      if errorChecking( mode, args, 1 ) == -1:
         return

      if not mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ].isAggregationMgr:
         if errorChecking( mode, args, 2 ) == -1:
            return

      else:
         if errorChecking( mode, args, 3 ) == -1:
            return

      devSet = mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ]
      if devSet.isAggregationMgr:
         # there can be only one Aggregation Manager
         aggrMgrDevice = [ x for x in devSet.serviceDevice.keys()
                        if not devSet.serviceDevice[ x ].isAccessedViaAggrMgr ].pop()
         serviceDevice = devSet.serviceDevice[ aggrMgrDevice ]
      else:
         serviceDevice = devSet.serviceDevice[ args[ 'SERVICE_DEVICE' ] ]
      if not serviceDevice.username or not serviceDevice.encryptedPassword:
         mode.addError( 'Could not log in to device %s,'
                        'check username, password' % (
                           args[ 'DEVICE_SET' ] ) )
         return

      try:
         mode.addWarning( API_ACCESS_MSG )
         deviceConfigDict = getDeviceConfig( devSet, serviceDevice )
         deviceConfigDict[ 'retries' ] = 0
         deviceConfigDict[ 'timeout' ] = timeout

         resources = None
         
         checkSslProfileStatus( deviceConfigDict, sslStatus )
         jsonDeviceDict, jsonArgs = setJson( deviceConfigDict,
            servDev=args[ 'SERVICE_DEVICE' ] )

         tacRunTimeout = timeout + 2
         op = Tac.run( [ '/usr/bin/timeout', '-k3', '-s9', str( tacRunTimeout ),
                         'python', '-m', PATH_FOR_API, jsonDeviceDict, jsonArgs,
                         'getDeviceResources' ],
                       stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
         pklOp = pickle.loads( op )
         if isinstance( pklOp, ServiceDeviceError ):
            raise pklOp
         else:
            resources = pklOp

      except ServiceDeviceError as e:
         mode.addError( e )
         return

      except Exception as e: # pylint: disable=W0703
         mode.addError( TIMEOUT_MSG.format( str( timeout ) ) )
         return

      mdl = MacroSegmentationResourcesInfoModel(
         resourceInfo=resources[ 'resourceInfo' ],
         serviceDeviceName=args[ 'SERVICE_DEVICE' ] )
      result.deviceResources = mdl
      return result

BasicCli.addShowCommandClass( ShowMacroSegmentationResources )

class ShowMacroSegmentationHA( ShowCommand.ShowCliCommandClass ):
   syntax = """ show service mss dynamic device-set
                DEVICE_SET device SERVICE_DEVICE high-availability
                [ cli-timeout CLI_TIMEOUT ] """
   data = { 'service': serviceKwMatcher,
            'mss': mssKwNode,
            'dynamic': 'Select policy source',
            'device-set': 'A named configuration for a set of devices',
            'DEVICE_SET': deviceSetMatcher,
            'device': 'Service device',
            'SERVICE_DEVICE': serviceDeviceMatcher,
             'high-availability': 'Service device high availability information',
            'cli-timeout': 'External API timeout',
            'CLI_TIMEOUT': cliTimeoutMatcher
        }
   cliModel = MacroSegmentationHighAvailabilityModel

   @staticmethod
   def handler( mode, args ):
      timeout = CLI_TIMEOUT
      if 'cli-timeout' in args:
         timeout = args[ 'CLI_TIMEOUT' ]

      result = MacroSegmentationHighAvailabilityModel()
      if errorChecking( mode, args, 1 ) == -1:
         return

      if not mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ].isAggregationMgr:
         if errorChecking( mode, args, 2 ) == -1:
            return

      else:
         if errorChecking( mode, args, 3 ) == -1:
            return

      devSet = mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ]
      if devSet.policySourceType == CHKP_MS_PLUGIN:  # FIXME, remove when supported
         print '** HA mode not supported yet for Check Point devices **'
         return result

      if devSet.isAggregationMgr:
         # there can be only one Aggregation Manager
         aggrMgrDevice = [ x for x in devSet.serviceDevice.keys()
                        if not devSet.serviceDevice[ x ].isAccessedViaAggrMgr ].pop()
         serviceDevice = devSet.serviceDevice[ aggrMgrDevice ]
      else:
         serviceDevice = devSet.serviceDevice[ args[ 'SERVICE_DEVICE' ] ]

      if not serviceDevice.username or not serviceDevice.encryptedPassword:
         mode.addError( 'Could not log in to device %s,'
                        'check username, password' % (
                           args[ 'DEVICE_SET' ] ) )
         return

      try:
         mode.addWarning( API_ACCESS_MSG )
         deviceConfigDict = getDeviceConfig( devSet, serviceDevice )
         deviceConfigDict[ 'retries' ] = 0
         deviceConfigDict[ 'timeout' ] = timeout

         ha = None
         
         checkSslProfileStatus( deviceConfigDict, sslStatus )
         jsonDeviceDict, jsonArgs = setJson( deviceConfigDict,
            servDev=args[ 'SERVICE_DEVICE' ] )

         tacRunTimeout = timeout + 2
         op = Tac.run( [ '/usr/bin/timeout', '-k3', '-s9', str( tacRunTimeout ),
                         'python', '-m', PATH_FOR_API, jsonDeviceDict, jsonArgs,
                         'getHighAvailabilityState' ],
                       stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
         pklOp = pickle.loads( op )
         if isinstance( pklOp, ServiceDeviceError ):
            raise pklOp
         else:
            ha = pklOp

      except ServiceDeviceError as e:
         mode.addError( e )
         return

      except Exception: # pylint: disable=W0703
         mode.addError( TIMEOUT_MSG.format( str( timeout ) ) )
         return

      mdl = MacroSegmentationHighAvailabilityInfoModel( haEnabled=ha.enabled,
                                                        haMode=ha.mode,
                                                        haState=ha.state )
      result.deviceHAState = mdl
      return result

BasicCli.addShowCommandClass( ShowMacroSegmentationHA )

def getGroupMembers( mode, devSet, servDev, timeout ):
   deviceSet = mssConfig.deviceSet[ devSet ]
   s = deviceSet.serviceDevice
   aggrMgr = [ x for x in s.keys()
               if not s[ x ].isAccessedViaAggrMgr ].pop()
   serviceDevice = deviceSet.serviceDevice[ aggrMgr ]

   try:
      deviceDict = getDeviceConfig( deviceSet, serviceDevice )
      deviceDict[ 'retries' ] = 0
      deviceDict[ 'timeout' ] = timeout

      grpMem = None
      
      checkSslProfileStatus( deviceDict, sslStatus )
      jsonDeviceDict, jsonArgs = setJson( deviceDict, servDev=None )

      op = Tac.run( [ '/usr/bin/timeout', '-k3', '-s9', str( timeout ),
                      'python', '-m', PATH_FOR_API, jsonDeviceDict, jsonArgs,
                      'getAggMgrGroupMembers' ],
                    stdout=Tac.CAPTURE, stderr=Tac.DISCARD )
      pklOp = pickle.loads( op )
      if isinstance( pklOp, ServiceDeviceError ):
         raise pklOp
      else:
         grpMem = pklOp

   except ServiceDeviceError:
      return

   except Timeout:
      return

   return servDev in grpMem

def createFormattedCell( table, desc, cols, fmt, space=0 ):
   table.startRow()
   desc = ' ' * space + desc
   table.newFormattedCell( desc, nCols=cols, format=fmt )

def dashes( num=65 ):
   if num == 65:
      return '-' * 65
   else:
      return '-' * num

def errorChecking( mode, args, errorNo, timeout=CLI_TIMEOUT ):
   # does the device set exists?
   if errorNo == 1:
      if args[ 'DEVICE_SET' ] not in mssConfig.deviceSet:
         mode.addError( 'Invalid device-set %s' % ( args[ 'DEVICE_SET' ] ) )
         return -1

   # is the service device in the device set specified?
   if errorNo == 2:
      if args[ 'SERVICE_DEVICE' ] not in mssConfig.deviceSet\
         [ args[ 'DEVICE_SET' ] ].serviceDevice:
         mode.addError( 'service-device %s not in %s' % (
            args[ 'SERVICE_DEVICE' ], args[ 'DEVICE_SET' ] ) )
         return -1

   # is this an Aggregation Manager?
   if errorNo == 3:
      if args[ 'SERVICE_DEVICE' ] in mssConfig.deviceSet\
         [ args[ 'DEVICE_SET' ] ].serviceDevice.keys():
         if mssConfig.deviceSet[ args[ 'DEVICE_SET' ] ].\
            serviceDevice[ args[ 'SERVICE_DEVICE' ] ].isAccessedViaAggrMgr:
            return
         mode.addError( 'This is an Aggregation Manager, '
                        'network/neighbor/resources/high-availability information '
                        'valid only for group-members of this aggregation device' )
         return -1

   if errorNo == 4:
      if args[ 'SERVICE_DEVICE' ] in mssStatus.serviceDeviceStatus.keys():
         if not mssStatus.serviceDeviceStatus[ args[ 'SERVICE_DEVICE' ] ].\
            deviceSetName == args[ 'DEVICE_SET' ]:
            mode.addError( '%s is not accessed via %s' % (
               args[ 'SERVICE_DEVICE' ], args[ 'DEVICE_SET' ] ) )
            return -1
      else:
         return -1


def Plugin( em ):
   global mssStatus
   global mssConfig
   global mssMainConfig
   global sslStatus

   mssStatus = LazyMount.mount( em, "msspolicymonitor/status",
                                "MssPolicyMonitor::Status", "r" )

   mssConfig = LazyMount.mount( em, "msspolicymonitor/config",
                                "MssPolicyMonitor::Config", "r" )

   mssMainConfig = LazyMount.mount( em, "mss/config",
                                    "Mss::Config", "r" )

   sslStatus = LazyMount.mount( em, "mgmt/security/ssl/status",
                                "Mgmt::Security::Ssl::Status", "r" )
