#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
# pylint: disable-msg=W0621

import re
import LazyMount
import ConfigMount
import Tac
import CliParser
import CliCommand
import CliMatcher
import BasicCli
import Arnet
import Toggles.MssToggleLib
import CliMode.MssPolicyMonitor as Mss
from CliPlugin.MssCli import MssConfigMode, addNoMssCallback
from CliPlugin.Ssl import profileNameMatcher, sslMatcher
from MssCliLib import TrafficInspectionCmd
from MssPolicyMonitor.Lib import ( PAN_FW_PLUGIN,
                                   PANORAMA_PLUGIN,
                                   CHKP_MS_PLUGIN,
                                   FORTIMGR_PLUGIN,
                                   UNASSIGNED,
                                   VERBATIM,
                                   OFFLOAD,
                                   REDIRECT,
                                   t0,
                                   createNewDeviceSet,
                                   setPolicySourceType,
                                   createNewServiceDevice,
                                   setAggrMgr,
                                   getEncryptedPasswordFor,
                                   DEFAULT_POLICY_TAG )
import IpAddrMatcher

neighborInterfaceMapType = Tac.Type( 'MssPolicyMonitor::NeighborInterfaceMap' )
defaults = Tac.Type( 'MssPolicyMonitor::CliDefaults' )
mssDefaults = Tac.Type( 'Mss::CliDefaults' )

psrcName = { PANORAMA_PLUGIN: 'Palo Alto Networks Panorama',
             UNASSIGNED: 'unassigned' }

mssConfig = None
mssStatus = None

# helper functions

def hasDuplicateTags( tag, mode, typeToSet ):
   for tagType, modifierTag in mode.ds.modifierTag.iteritems():
      if tagType != typeToSet: 
         if tag in modifierTag.tag:
            mode.addError( "One or more tags already used as %s tag(s)" % tagType )
            return True

   if typeToSet != REDIRECT and tag in mode.ds.policyTag:
      mode.addError( "One or more tags already used as %s tag(s)" % REDIRECT )
      return True
   if typeToSet != OFFLOAD and tag in mode.ds.offloadTag:
      mode.addError( "One or more tags already used as %s tag(s)" % OFFLOAD )
      return True
   return False

def getIntfName( mode ):
   return mode.sd.intfMap.members()

def getMacAddr( mode ):
   return [ i.switchChassisId
            for i in mode.sd.intfMap.itervalues() ]

def getSwitchIntf( mode ):
   return [ i.switchIntf
            for i in mode.sd.intfMap.itervalues() ]

interfacePattern = r'([^\d]+)((\d+)|(\d+/\d+)|(\d+/\d+/\d+))$'
def parseInterface( inputIntf ):
   match = re.match( interfacePattern, inputIntf )
   inputIntfType = match.groups()[ 0 ]
   inputIntfId = match.groups()[ 1 ]

   isExpected = ( lambda intfType:
      intfType.startswith( inputIntfType.lower() ) and
      ( intfType == 'ethernet' or re.match( r'\d+$', inputIntfId ) ) )

   intfTypes = [ 'ethernet', 'port-channel' ]
   try:
      matchedIntfType = next( intfType for intfType in intfTypes
                              if isExpected( intfType ) )
      return ( matchedIntfType + inputIntfId ).title()
   except StopIteration:
      raise CliParser.InvalidInputError()

chassisIdRe = r'\.'.join( [ "[0-9a-fA-F]" * 4 ] * 3 ) \
              + '|' \
              + ':'.join( [ "[0-9a-fA-F]" * 2 ] * 6 )

def mapOnlyDeviceGuard( mode, token ):
   t0( 'calling mapOnlyGuard for ', token )
   if mode.sd.isAccessedViaAggrMgr:
      return 'map-only/member devices cannot configure %s' % ( token )
   return None

# token declaration
usernameMatcher = CliCommand.guardedKeyword(
   'username',
   helpdesc='Username for API authentication for the service device',
   guard=mapOnlyDeviceGuard )
policyMatcher = CliMatcher.KeywordMatcher( 'policy',
      helpdesc='Security policy parameters' )
tagMatcher = CliMatcher.KeywordMatcher( 'tag',
      helpdesc='Policy tag in service device policies to act on' )
domainMatcher = CliMatcher.KeywordMatcher( 'domain',
      helpdesc='Domain to use with MSS' )

hostRe = '([^:|> ]+|%s)' % ( Arnet.Ip6AddrRe )

# for aggregation manager devices: check that there is only one non-map service
# device any number of map-only/member service devices are allowed devices
def isMoreThanOneNonMap( mode ):
   # get the number of non-map service devies
   noOfNonMapServDev = sum( not servDev.isAccessedViaAggrMgr
                            for servDev in mode.ds.serviceDevice.itervalues() )
   # are there more than 1 non-map service devices?
   return noOfNonMapServDev >= 2

def isExactlyOneNonMap( mode, name ):
   # get the number of non-map service devies
   noOfNonMapServDev = sum( not servDev.isAccessedViaAggrMgr
                            for servDev in mode.ds.serviceDevice.itervalues() )
   # is there exactly one non-map service device and creating a new service device
   return noOfNonMapServDev == 1 and name not in mode.ds.serviceDevice.iterkeys()


################################################################################
# The "dynamic palo-alto panorama device-set <name>" command in mss            #
# config mode                                                                  #
################################################################################
class MssDynamic( Mss.MssPolicyMonitorDynamic,
                  BasicCli.ConfigModeBase ):
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name ):
      t0( 'In init of MssDynamic' )
      Mss.MssPolicyMonitorDynamic.__init__( self, name )
      self.ds = createNewDeviceSet( mssConfig, name )
      # legacyVdom is set if `virtual domain` is used during startup
      self.legacyVdom = None
      if not self.ds.policyTag:
         self.ds.policyTag[ DEFAULT_POLICY_TAG ] = True
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def trafficInspectionIs( self, local, outbound ):
      self.ds.trafficInspection = Tac.Value( "Mss::TrafficInspection",
                                             local=local,
                                             outbound=outbound )
   def noTrafficInspection( self ):
      self.ds.trafficInspection = mssDefaults.trafficInspection

def gotoMssDynamic( mode, args ):
   childMode = mode.childMode( MssDynamic, name=args[ 'NAME' ] )
   mode.session_.gotoChildMode( childMode )

def removeMssDynamic( mode, args ):
   t0( 'In remove of dynamic device set !' )
   del mssConfig.deviceSet[ args[ 'NAME' ] ]

#--------------------------------------------------------------------------------
# [ no | default ] dynamic device-set NAME
#--------------------------------------------------------------------------------
class DynamicDeviceSetNameCmd( CliCommand.CliCommandClass ):
   syntax = 'dynamic device-set NAME'
   noOrDefaultSyntax = syntax
   data = {
      'dynamic': 'Select dynamic policy configuration',
      'device-set': 'A named configuration for a set of devices',
      'NAME': CliMatcher.DynamicNameMatcher( 
         lambda mode: mssConfig.deviceSet.keys(),
         helpdesc='A unique device set name' )
   }
   handler = gotoMssDynamic
   noOrDefaultHandler = removeMssDynamic

t0( 'adding the keyword to macro-segmentation config' )
MssConfigMode.addCommandClass( DynamicDeviceSetNameCmd )

def deleteDeviceSet( mode ):
   mssConfig.deviceSet.clear()

addNoMssCallback( deleteDeviceSet )


################################################################
# The "type [ palo-alto ( firewall | panorama ) ]" command in  #
# dynamic config mode                                          #
################################################################
class MssType( CliCommand.CliCommandClass ):
   syntax = ( "type ( ( palo-alto ( panorama | firewall ) ) "
              "| ( fortinet fortimanager ) "
              "| ( check-point management-server ) )" )
   data = { 'type': 'Vendor for service devices in this set, selects MSS Plugin',
            'palo-alto': 'Palo Alto Networks policy source',
            'firewall': 'Palo Alto Networks Firewall',
            'panorama': 'Palo Alto Networks Panorama',
            'fortinet': 'Fortinet policy source',
            'fortimanager': 'Fortinet FortiManager',
            'check-point': 'Check Point policy source',
            'management-server': 'Check Point Management Server' }

   noOrDefaultSyntax = """ type """

   @staticmethod
   def handler( mode, args ):
      if 'palo-alto' in args:
         if 'panorama' in args:
            if isMoreThanOneNonMap( mode ):
               mode.addError( 'Device sets with type Palo Alto Networks Panorama '
                              'only supports one panorama device and zero or more '
                              'map-only/member devices. Keep only one non-map '
                              'service device.' )
               return
            setAggrMgr( mode.ds, True )
            setPolicySourceType( mode.ds, PANORAMA_PLUGIN )

         elif 'firewall' in args:
            setAggrMgr( mode.ds, False )
            setPolicySourceType( mode.ds, PAN_FW_PLUGIN )

      elif 'fortinet' in args and 'fortimanager' in args:
         setAggrMgr( mode.ds, True )
         setPolicySourceType( mode.ds, FORTIMGR_PLUGIN )

      elif 'check-point' in args and 'management-server' in args:
         setAggrMgr( mode.ds, True )
         setPolicySourceType( mode.ds, CHKP_MS_PLUGIN )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setPolicySourceType( mode.ds, defaults.policySourceType )

MssDynamic.addCommandClass( MssType )


#####################################################################
# The "timeout TIMEOUT" command in dynamic config mode              #
#####################################################################
class MssTimeout( CliCommand.CliCommandClass ):
   syntax = """ timeout TIMEOUT """
   noOrDefaultSyntax = """ timeout """
   data = { 'timeout': 'Timeout duration for service device API access',
            'TIMEOUT': CliMatcher.IntegerMatcher( 5, 900,
                          helpdesc='API Timeout in units of seconds' )
      }

   @staticmethod
   def handler( mode, args ):
      mode.ds.timeout = args.get( 'TIMEOUT', defaults.timeout )

   noOrDefaultHandler = handler

MssDynamic.addCommandClass( MssTimeout )


##########################################################################
# The "tag { TAGS }" command in dynamic config mode                      #
##########################################################################
class MssPolicyTag( CliCommand.CliCommandClass ):
   syntax = """ tag { TAGS } """
   noOrDefaultSyntax = """ tag """
   data = { 'tag': tagMatcher,
            'TAGS': CliMatcher.QuotedStringMatcher( helpname='WORD',
                                                    helpdesc='Name of the tag(s)' ),

      }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.ds.policyTag.clear()
      for tag in args.get( 'TAGS', [ DEFAULT_POLICY_TAG ] ):
         mode.ds.policyTag[ tag ] = True

   noOrDefaultHandler = handler

MssDynamic.addCommandClass( MssPolicyTag )

##############################################################################
# The "policy tag redirect { TAGS }" command in dynamic config mode          #
##############################################################################
class MssPolicyRedirectTag( CliCommand.CliCommandClass ):
   syntax = """ policy tag redirect { TAGS } """
   noOrDefaultSyntax = """ policy tag redirect """
   data = { 'policy': policyMatcher,
            'tag': tagMatcher,
            'redirect': 'redirect tags in service device policies',
            'TAGS': CliMatcher.QuotedStringMatcher( helpname='WORD',
                                                    helpdesc='Name of the tag(s)' ),
      }

   @staticmethod
   def handler( mode, args ):
      tags = args[ 'TAGS' ]
      # Make sure that no tag is same as ones in offload rules
      for tag in tags:
         if hasDuplicateTags( tag, mode, REDIRECT ):
            return
      mode.ds.policyTag.clear()
      for tag in tags:
         mode.ds.policyTag[ tag ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ds.policyTag.clear()
      mode.ds.policyTag[ DEFAULT_POLICY_TAG ] = True

MssDynamic.addCommandClass( MssPolicyRedirectTag )

##############################################################################
# The "policy tag offload { TAGS }" command in dynamic config mode           #
##############################################################################
class MssPolicyOffloadTag( CliCommand.CliCommandClass ):
   syntax = """ policy tag offload { TAGS } """
   noOrDefaultSyntax = """ policy tag offload """
   data = { 'policy': policyMatcher,
            'tag': tagMatcher,
            'offload': 'Offload tags in service device policies',
            'TAGS': CliMatcher.QuotedStringMatcher( helpname='WORD',
                                                    helpdesc='Name of the tag(s)' ),
      }

   @staticmethod
   def handler( mode, args ):
      tags = args[ 'TAGS' ]
      # Make sure that no tag is same as ones in redirect rules
      # Also don't allow DEFAULT_POLICY_TAG here
      for tag in tags:
         if tag == DEFAULT_POLICY_TAG:
            mode.addError( 'Default redirect tag %s cannot be used as '
               'offload tag' % ( DEFAULT_POLICY_TAG ) )
            return
         if hasDuplicateTags( tag, mode, OFFLOAD ):
            mode.addError( "One or more tags already used as redirect tag(s)" )
            return
      mode.ds.offloadTag.clear()
      for tag in tags:
         mode.ds.offloadTag[ tag ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.ds.offloadTag.clear()

MssDynamic.addCommandClass( MssPolicyOffloadTag )

##############################################################################
# The "policy tag modifier verbatim TAGLIST" command in dynamic config mode  #
##############################################################################
class MssPolicyVerbatimTag( CliCommand.CliCommandClass ):
   syntax = 'policy tag modifier verbatim { TAGLIST }'
   noOrDefaultSyntax = 'policy tag modifier verbatim [ { TAGLIST } ]'
   data = { 'policy': policyMatcher,
            'tag': tagMatcher,
            'modifier': 'Modifier policy tag worked with primary policy tag',
            'verbatim': 'Verbatim tags in service device policies',
            'TAGLIST': CliMatcher.QuotedStringMatcher(
                                    helpname='TAGLIST',
                                    helpdesc='Name of the verbatim tag(s)' ),
          }

   @staticmethod
   def handler( mode, args ):
      # Make sure that no tag is same as ones in offload rules
      for tag in args[ 'TAGLIST' ]:
         if tag == DEFAULT_POLICY_TAG:
            mode.addError( 'Default redirect tag %s cannot be used as '
                           'verbatim tag' % ( DEFAULT_POLICY_TAG ) )
            return
         if hasDuplicateTags( tag, mode, VERBATIM ):
            return

      verbatimTags = mode.ds.newModifierTag( VERBATIM )
      verbatimTags.tag.clear()
      for tag in args[ 'TAGLIST' ]:
         verbatimTags.tag[ tag ] = True
      verbatimTags.seqNo += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      taglist = args.get( 'TAGLIST' )
      verbatimTags = mode.ds.newModifierTag( VERBATIM )
      if taglist:
         for tag in taglist:
            del verbatimTags.tag[ tag ]
      else:
         verbatimTags.tag.clear()

      verbatimTags.seqNo += 1

MssDynamic.addCommandClass( MssPolicyVerbatimTag )

####################################################################
# The "state ( shutdown | suspend | dry-run | active )" command in #
# dynamic config mode                                              #
####################################################################
class MssState( CliCommand.CliCommandClass ):
   syntax = """ state ( shutdown | suspend | active ) """
   noOrDefaultSyntax = """ state """
   data = { 'state': 'Running state for this device set',
            'shutdown': 'No policy monitoring or network traffic redirection',
            'suspend':
               'No policy monitoring, don\'t change existing traffic redirection',
            'active': 'Active policy monitoring and network traffic redirection'
      }

   @staticmethod
   def handler( mode, args ):
      if 'shutdown' in args:
         state = 1
      elif 'active' in args:
         state = 2
      elif 'suspend' in args:
         state = 4
      else:
         state = defaults.state
      mode.ds.state = state

   noOrDefaultHandler = handler

MssDynamic.addCommandClass( MssState )


#####################################################################
# The "retries RETRIES" command in dynamic config mode            #
#####################################################################
class MssRetries( CliCommand.CliCommandClass ):
   syntax = """ retries RETRIES """
   noOrDefaultSyntax = """ retries """
   data = { 'retries': 'Service device API access retries',
            'RETRIES': CliMatcher.IntegerMatcher( 0, 9,
                            helpdesc='Number of service device api access retries' ),
      }

   @staticmethod
   def handler( mode, args ):
      mode.ds.retries = args[ 'RETRIES' ]

   @staticmethod
   def noHandler( mode, args ):
      mode.ds.retries = 0

   @staticmethod
   def defaultHandler( mode, args ):
      mode.ds.retries = defaults.retries

MssDynamic.addCommandClass( MssRetries )


#######################################################################
# The "interval INTERVAL" command in dynamic config mode            #
#######################################################################
class MssInterval( CliCommand.CliCommandClass ):
   syntax = """ interval INTERVAL """
   noOrDefaultSyntax = """ interval """
   data = { 'interval': 'Service device policy update interval',
            'INTERVAL': CliMatcher.IntegerMatcher( 5, 900,
                       helpdesc='Service device policy update interval in seconds' ),
      }

   @staticmethod
   def handler( mode, args ):
      mode.ds.queryInterval = args.get( 'INTERVAL', defaults.queryInterval )

   noOrDefaultHandler = handler

MssDynamic.addCommandClass( MssInterval )


########################################################################
# The "admin domain DOMAIN" command in dynamic config mode        #
########################################################################
class MssAdminDomainMode( CliCommand.CliCommandClass ):
   syntax = """ admin domain DOMAIN """
   noOrDefaultSyntax = """ admin domain ... """
   data = { 'admin': 'Set administrative domain (ADOM), for Fortinet only',
            'domain': domainMatcher,
            'DOMAIN': CliMatcher.PatternMatcher( '.+',
                                                 helpname='WORD',
                                                 helpdesc='ADOM name' ),
      }

   @staticmethod
   def handler( mode, args ):
      mode.ds.adminDomain = args.get( 'DOMAIN', defaults.adminDomain )

   noOrDefaultHandler = handler

MssDynamic.addCommandClass( MssAdminDomainMode )

########################################################################
# The "virtual domain DOMAIN" command in dynamic config mode           #
# Note: It is recommended to use `virtual instance` instead but we     #
# still want this syntax to be available to not break existing         #
# costumer configuration.                                              #
########################################################################
if Toggles.MssToggleLib.toggleMssL3V2Enabled():
   virtualOldKw = CliCommand.singleKeyword( 'virtual',
                                 "Set virtual domain (VDOM), for Fortinet only.",
                                 deprecatedByCmd='virtual instance' )
else:
   virtualOldKw = CliCommand.singleKeyword( 'virtual',
                                 "Set virtual domain (VDOM), for Fortinet only." )

class MssVirtualDomainMode( CliCommand.CliCommandClass ):
   syntax = """ virtual domain DOMAIN """
   noOrDefaultSyntax = """ virtual domain ... """
   data = { 'virtual': virtualOldKw,
            'domain': domainMatcher,
            'DOMAIN': CliMatcher.PatternMatcher( '.+',
                                                 helpname='WORD',
                                                 helpdesc='VDOM name' ),
      }

   @staticmethod
   def handler( mode, args ):
      vdomName = args.get( 'DOMAIN', defaults.virtualDomain )
      if Toggles.MssToggleLib.toggleMssL3V2Enabled():
         if vdomName != defaults.virtualDomain:
            # If toggle is enabled, we should be in start-up config
            # Set the legacy non-default VDOM in all existing service device
            # with no virtual instance.
            # We keep legacy vdom if service devices are configured later.
            mode.legacyVdom = vdomName

            for sd in mode.ds.serviceDevice.values():
               if sd.isAccessedViaAggrMgr and not sd.virtualInstance:
                  sd.newVirtualInstance( vdomName )
      else: 
         mode.ds.virtualDomain = vdomName

   noOrDefaultHandler = handler

MssDynamic.addCommandClass( MssVirtualDomainMode )


########################################################################
# The "exception device unreachable ( bypass | redirect )" command in  #
# dynamic config mode                                                  #
########################################################################
class MssExceptionMode( CliCommand.CliCommandClass ):
   syntax = """ exception device unreachable ( bypass | redirect ) """
   noOrDefaultSyntax = """ exception device unreachable ... """
   data = { 'exception': 'Exception handling mode',
            'device': 'applies to the service devices in this device-set',
            'unreachable': 'service device control-plane API is unreachable',
            'bypass': 'bypass unreachable service devices',
            'redirect': 'continue redirecting traffic to service devices',
      }

   @staticmethod
   def handler( mode, args ):
      if 'bypass' in args:
         exceptionHandling = 0
      elif 'redirect' in args:
         exceptionHandling = 1
      else:
         exceptionHandling = defaults.exceptionHandling
      mode.ds.exceptionHandling = exceptionHandling

   noOrDefaultHandler = handler

MssDynamic.addCommandClass( MssExceptionMode )


# ##################################################################################
# # # The "[ no | default ] verify cert" command in dynamic config mode            #
# ##################################################################################
# class MssVerifyCert( object ):
#    syntax = """ verify certificate """
#    noOrDefaultSyntax = """ verify certificate """
#    data = { 'verify': 'Verify SSL/TLS certificates',
#             'certificate': 'certificate' }

#    def handler( self, mode, args ):
#       mode.ds.verifyCertificate = True

#    def noOrDefaultHandler( self, mode, args ):
#       mode.ds.verifyCertificate = defaults.verifyCertificate
#
# MssDynamic.addLegacyCommandClass( MssVerifyCert )


#########################################################
# # # The " device <ip-addr-or-hostname>"               #
# # # command in dynamic  config mode                   #
#########################################################
class MssDevice( Mss.MssPolicyMonitorDevice, BasicCli.ConfigModeBase ):
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name, devset, mapOnlyOrMember=None,
                 isMapOnly=False ):
      t0( 'In init of MssDevice' )
      param = ( name, devset, isMapOnly, mapOnlyOrMember )
      Mss.MssPolicyMonitorDevice.__init__( self, param )
      self.sd = createNewServiceDevice( parent.ds, name )
      self.sd.isAccessedViaAggrMgr = isMapOnly
      if parent.legacyVdom and isMapOnly:
         # we are in startup config and `virtual domain` was configured
         self.sd.newVirtualInstance( parent.legacyVdom )
      self.dsName = parent.ds.name
      BasicCli.ConfigModeBase.__init__( self, parent, session )


def gotoMssDevice( mode, args ):
   t0( 'in goto MssDevice' )
   name = args[ 'NAME' ]
   if mode.ds.policySourceType == PANORAMA_PLUGIN and \
      isExactlyOneNonMap( mode, name ):
      mode.addError( 'Device sets with type %s only support one panorama device '
                     'and zero or more map-only/member devices'
                     % ( psrcName[ mode.ds.policySourceType ] ) )
      return
   childMode = mode.childMode( MssDevice, name=name, devset=mode.param_,
                               isMapOnly=False )
   mode.session_.gotoChildMode( childMode )


def gotoMssDeviceMapOnly( mode, args ):
   childMode = mode.childMode( MssDevice, mapOnlyOrMember=args[ 'MAP_OR_MEMBER' ],
                               name=args[ 'NAME' ], devset=mode.param_,
                               isMapOnly=True )
   mode.session_.gotoChildMode( childMode )

def removeMssDevice( mode, args ):
   t0( "In remove of Mss device!" )
   name = args[ 'NAME' ]
   if name in mssConfig.deviceSet[ mode.ds.name ].serviceDevice.keys():
      del mssConfig.deviceSet[ mode.ds.name ].serviceDevice[ name ]

def removeMssDeviceMapOnly( mode, args ):
   t0( "In remove of Mss device!" )
   name = args[ 'NAME' ]
   if name in mssConfig.deviceSet[ mode.ds.name ].serviceDevice.keys():
      del mssConfig.deviceSet[ mode.ds.name ].serviceDevice[ name ]

mapNoMapHelp = 'IP address or DNS hostname of service device management interface'

nonMapNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode:[ k for k, v in mssConfig.deviceSet[ mode.param_ ].
                 serviceDevice.iteritems() if not v.isAccessedViaAggrMgr ],
   mapNoMapHelp, pattern=hostRe )

mapOnlyNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode:[ k for k, v in mssConfig.deviceSet[ mode.param_ ].
                 serviceDevice.iteritems() if v.isAccessedViaAggrMgr ],
   mapNoMapHelp, pattern=hostRe )
deviceKwMatcher = CliMatcher.KeywordMatcher( 'device',
      helpdesc='Add a service device to the device list' )

#--------------------------------------------------------------------------------
# [ no | default ] device NAME
#--------------------------------------------------------------------------------
class GotoMssDeviceCmd( CliCommand.CliCommandClass ):
   syntax = 'device NAME'
   noOrDefaultSyntax = syntax
   data = {
      'device': deviceKwMatcher,
      'NAME': nonMapNameMatcher
   }
   handler = gotoMssDevice
   noOrDefaultHandler = removeMssDevice

t0( 'adding the keyword to macro-segmentation config' )
MssDynamic.addCommandClass( GotoMssDeviceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] device MAP_OR_MEMBER NAME
#--------------------------------------------------------------------------------
class GotoMssDeviceMapOnlyCmd( CliCommand.CliCommandClass ):
   syntax = 'device MAP_OR_MEMBER NAME'
   noOrDefaultSyntax = syntax
   data = {
      'device': deviceKwMatcher,
      'MAP_OR_MEMBER': CliMatcher.EnumMatcher( {
         'map-only': 'Map only devices [deprecated, use member]',
         'member': 'A device in a group or policy layer',
      } ),
      'NAME': mapOnlyNameMatcher
   }
   handler = gotoMssDeviceMapOnly
   noOrDefaultHandler = removeMssDeviceMapOnly

t0( 'adding the keyword to macro-segmentation config' )
MssDynamic.addCommandClass( GotoMssDeviceMapOnlyCmd )

#################################################################################
# #"traffic inspection {local | local outbound}" command in device-set mode     #
# # This config can be overridden in the virtual instance                       #
#################################################################################
if Toggles.MssToggleLib.toggleMssL3V2Enabled():
   MssDynamic.addCommandClass( TrafficInspectionCmd )

##################################################################################
# # The "username USERNAME password ( 0 | 7 ) PASSWORD" command in device <host>#
# # config mode in PAN Panorama                                                  #
##################################################################################
class MssUnamePwd( CliCommand.CliCommandClass ):
   syntax = """ username USERNAME password ( 0 | 7 ) PASSWORD  """
   noOrDefaultSyntax = """ username """
   data = { 'username': usernameMatcher,
            'USERNAME': CliMatcher.PatternMatcher( '.+',
                                                   helpname='WORD',
                                                   helpdesc='Account name string' ),
            'password': 'Password for API authentication for the service device',
            '0': 'Specifies an UNENCRYPTED password will follow',
            '7': 'Specifies an ENCRYPTED password will follow',
            'PASSWORD': CliCommand.Node(
                           matcher=CliMatcher.PatternMatcher( '.+',
                                                              helpname='WORD',
                                                              helpdesc='A single '
                                                                       'word' ),
                           sensitive=True )
   }

   @staticmethod
   def handler( mode, args ):
      mode.sd.username = args.get( 'USERNAME', defaults.username )
      password = args.get( 'PASSWORD', defaults.encryptedPassword )
      if '0' in args:
         password = getEncryptedPasswordFor( password )
      mode.sd.encryptedPassword = password

   noOrDefaultHandler = handler

MssDevice.addCommandClass( MssUnamePwd )


####################################################################################
# # The "protocol ( ( http [ PORT ] ) | ( https [ PORT ]                           #
# #                 [ ( ssl profile PROFILE ) ] ) )" command in device <host>      #
# # config mode in PAN Panorama                                                    #
####################################################################################
class MssProtocol( CliCommand.CliCommandClass ):
   syntax = ( """ protocol """
              """( ( http  [ PORT ] ) """
              """| ( https [ PORT ] [ ( ssl profile PROFILE ) ] ) ) """ )
   noOrDefaultSyntax = """ protocol [ ( https ssl profile [ PROFILE ] ) ] """
   data = { 'protocol': CliCommand.guardedKeyword( 'protocol',
                           helpdesc='Protocol for service device API access',
                           guard=mapOnlyDeviceGuard ),
            'http': 'Use HTTP for service device API access protocol',
            'https': 'Use HTTPS for service device API access protocol',
            'PORT': CliMatcher.IntegerMatcher( 0, 65535,
                                               helpdesc='TCP port number' ),
            'ssl': sslMatcher,
            'profile': 'Select SSL profile',
            'PROFILE': profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if 'http' in args:
         mode.sd.protocol = 1
         mode.sd.sslProfileName = defaults.sslProfileName
         mode.sd.protocolPortNum = args.get( 'PORT', 80 )
      else:
         mode.sd.protocol = 0
         mode.sd.protocolPortNum = args.get( 'PORT', defaults.protocolPortNum )
         mode.sd.sslProfileName = args.get( 'PROFILE', defaults.sslProfileName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'ssl' in args and 'profile' in args:
         mode.sd.sslProfileName = defaults.sslProfileName
      else:
         mode.sd.protocolPortNum = defaults.protocolPortNum
         mode.sd.protocol = defaults.protocol
         mode.sd.sslProfileName = defaults.sslProfileName

MssDevice.addCommandClass( MssProtocol )


#########################################################
# # The "group GROUP" command in device <host>          #
# # config mode in PAN Panorama                         #
#########################################################
class MssDeviceGrp( CliCommand.CliCommandClass ):
   syntax = """ group GROUP  """
   noOrDefaultSyntax = """ group """
   data = { 'group': CliCommand.guardedKeyword( 'group',
                        helpdesc='Name of device group or policy layer '
                                 '(for aggregation managers only)',
                        guard=mapOnlyDeviceGuard ),
            'GROUP': CliMatcher.PatternMatcher( '.+', helpname='WORD',
                                                      helpdesc='Group name' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.sd.group = args[ 'GROUP' ]

   @staticmethod
   def noHandler( mode, args ):
      mode.sd.group = defaults.group

   @staticmethod
   def defaultHandler( mode, args ):
      pass

MssDevice.addCommandClass( MssDeviceGrp )

#################################################################################
# # The "virtual instance <vinst name> " mode in device <host>                #
#################################################################################
class MssVirtualInstance( Mss.MssPolicyMonitorVInst, BasicCli.ConfigModeBase ):
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vinstName ):
      param = ( parent.dsName, parent.sd.ipAddrOrDnsName, vinstName )
      Mss.MssPolicyMonitorVInst.__init__( self, param )
      if vinstName in parent.sd.virtualInstance:
         self.vinstConfig = parent.sd.virtualInstance[ vinstName ]
      else:
         self.vinstConfig = parent.sd.newVirtualInstance( vinstName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def trafficInspectionIs( self, local, outbound ):
      self.vinstConfig.trafficInspection = Tac.Value( "Mss::TrafficInspection",
                                                      local=local,
                                                      outbound=outbound )

   def noTrafficInspection( self ):
      self.vinstConfig.trafficInspection = Tac.Value( "Mss::TrafficInspection" )

class MssDeviceVirtualInstanceCmd( CliCommand.CliCommandClass ):
   syntax = "virtual instance VSYS_NAME"
   noOrDefaultSyntax = syntax

   data = {
         'virtual': 'Add a Virtual System or Domain on the service device',
         'instance' : 'Virtual Instance to use with MSS',
         'VSYS_NAME' : CliMatcher.PatternMatcher( '.+', helpname='WORD',
                                       helpdesc='Virtual System or Domain name' ),
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( MssVirtualInstance, vinstName=args[ 'VSYS_NAME' ] )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del mssConfig.deviceSet[ mode.dsName ].serviceDevice[
            mode.sd.ipAddrOrDnsName ].virtualInstance[ args[ 'VSYS_NAME' ] ]

if Toggles.MssToggleLib.toggleMssL3V2Enabled():
   MssDevice.addCommandClass( MssDeviceVirtualInstanceCmd )

#################################################################################
# # The "network vrf <vrf name> (virtual-router|vrf id) <fw vrf name> "         #
# # command in virtual instance <vinst>                                          #
#################################################################################
networkVrfKw = CliMatcher.KeywordMatcher( 'vrf', helpdesc='Network VRF name' )
firewallVrfKw = CliMatcher.KeywordMatcher( 'vrf', helpdesc='Fortinet VRF ID' )

class MssNetworkVrfCmd( CliCommand.CliCommandClass ):
   syntax = """ network NETVRF_KW NETVRF_NAME
                ( virtual-router | ( FWVRF_KW id ) ) FWVRF_NAME"""
   noOrDefaultSyntax = syntax

   data = {
      'network' : 'Map network VRF name to service device VRF name',
      'NETVRF_KW' : networkVrfKw,
      'NETVRF_NAME' : CliMatcher.DynamicNameMatcher(
         lambda mode: mode.vinstConfig.firewallVrf,
         helpdesc='Network VRF name' ),
      'virtual-router' : 'Palo-Alto virtual router name',
      'FWVRF_KW' : firewallVrfKw,
      'id' : 'Fortinet VRF ID',
      'FWVRF_NAME' : CliMatcher.PatternMatcher( '.+', helpname='WORD',
                                   helpdesc='Service device VRF name or ID' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.vinstConfig.firewallVrf[ args[ 'FWVRF_NAME' ] ] = args[ 'NETVRF_NAME' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del mode.vinstConfig.firewallVrf[ args[ 'FWVRF_NAME' ] ]

MssVirtualInstance.addCommandClass( MssNetworkVrfCmd )

#####################################################################################
# #"traffic inspection {local | local outbound}" command in virtual instance <vinst>#
# # This command overrides traffic inspection mode configuration of device-set      #
#####################################################################################
# Parent mode is behind the feature toggle. So no need to add one here
MssVirtualInstance.addCommandClass( TrafficInspectionCmd )

#################################################################################
# # The "map intf INTF1 switch SWITCH intf INTF2 " command in                   #
# # device <host>                                                               #
# # config mode in PAN Panorama                                                 #
#################################################################################
intfMatcher = CliMatcher.DynamicNameMatcher( getIntfName,
                                             helpdesc='Name of the interface',
                                             pattern='.+' )
class MssIntfMap( CliCommand.CliCommandClass ):
   syntax = """ map device-interface INTF1 switch SWITCH
   interface INTF2  """
   noOrDefaultSyntax = """ map device-interface INTF1 ... """

   data = { 'map': 'Define map of service device to switch links when LLDP'
                   ' not used',
            'device-interface': 'Service device interface name',
            'INTF1': intfMatcher,
            'switch': 'Switch that the service device is attached to',
            'SWITCH': CliMatcher.DynamicNameMatcher( getMacAddr,
                         helpdesc='Switch system MAC address (available from '
                                  'show version)',
                         pattern=chassisIdRe ),
            'interface': 'Switch interface name',
            'INTF2': intfMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf1 = args[ 'INTF1' ]
      try:
         intf2 = parseInterface( args[ 'INTF2' ] )
      except:
         raise CliParser.InvalidInputError()
      intfmap = neighborInterfaceMapType( intf1 )
      intfmap.switchChassisId = args[ 'SWITCH' ]
      intfmap.switchIntf = intf2
      mode.sd.intfMap.addMember( intfmap )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf1 = args[ 'INTF1' ]
      del mode.sd.intfMap[ intf1 ]

MssDevice.addCommandClass( MssIntfMap )


#################################################################################
# # The "management virtual domain DOMAIN" command in device <host>             #
# # config mode in PAN Panorama                                                 #
#################################################################################
class MssManagementIntfVdom( CliCommand.CliCommandClass ):
   syntax = """ management virtual domain DOMAIN """
   noOrDefaultSyntax = """ management virtual domain ... """

   data = { 'management': 'Configuration that applies to device management '
                          'elements',
            'virtual': 'Set virtual domain (VDOM), for Fortinet only',
            'domain': 'Domain to use with MSS',
            'DOMAIN': CliMatcher.PatternMatcher( '.+', helpname='WORD',
                                                 helpdesc='VDOM name' ),
   }

   @staticmethod
   def handler( mode, args ):
      domain = args.get( 'DOMAIN', defaults.mgmtIntfVirtualDomain )
      mode.sd.mgmtIntfVirtualDomain = domain

   noOrDefaultHandler = handler

MssDevice.addCommandClass( MssManagementIntfVdom )

################################################################################
# The "vrf <vrf-name>" command                                                 #
################################################################################
class MssDeviceVrf( Mss.MssPolicyMonitorVrf, BasicCli.ConfigModeBase ):
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      self.vrfName = vrfName
      self.dsName = parent.dsName
      self.sdName = parent.sd.ipAddrOrDnsName
      param = ( self.dsName, self.sdName, vrfName )
      Mss.MssPolicyMonitorVrf.__init__( self, param )
      if vrfName in parent.sd.vrfConfig:
         self.vrfConfig = parent.sd.vrfConfig[ vrfName ]
      else:
         self.vrfConfig = parent.sd.newVrfConfig( vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMssDeviceVrf( mode, args ):
   childMode = mode.childMode( MssDeviceVrf, vrfName=args[ 'VRF_NAME' ] )
   mode.session_.gotoChildMode( childMode )

def removeMssDeviceVrf( mode, args ):
   del mssConfig.deviceSet[ mode.dsName ].serviceDevice[
      mode.sd.ipAddrOrDnsName ].vrfConfig[ args[ 'VRF_NAME' ] ]

#--------------------------------------------------------------------------------
# [ no | default ] vrf VRFNAME
#--------------------------------------------------------------------------------
class VrfVrfnameCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'vrf': 'Service device VRF config',
      'VRF_NAME': CliMatcher.DynamicNameMatcher(
         lambda mode: mode.sd.vrfConfig.keys(), 'VRF name' ),
   }
   handler = gotoMssDeviceVrf
   noOrDefaultHandler = removeMssDeviceVrf

MssDevice.addCommandClass( VrfVrfnameCmd )

###################################################################################
# The "security policy ipv4 exclude-list SUBNETS" command in vrf config mode      #
###################################################################################
class MssPolicyExcludeList( CliCommand.CliCommandClass ):
   syntax = """ security policy ipv4 exclude-list { SUBNETS } """
   noOrDefaultSyntax = """ security policy ipv4 exclude-list """
   data = { 'security': 'Security policy vrf parameters',
            'policy': policyMatcher,
            'ipv4': 'Policy next-hop IPv4 address',
            'exclude-list': 'List of subnets to exclude from this device',
            'SUBNETS': IpAddrMatcher.ipPrefixExpr( 'IP address',
                                                   'Subnet\'s mask value',
                                                   'IPv4 address prefix' ),
      }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.vrfConfig.hostExcludeList.clear()
      for prefix in args.get( 'SUBNETS', [] ):
         try:
            subnet = Tac.Value( 'Arnet::IpGenPrefix', prefix )
            mode.vrfConfig.hostExcludeList[ subnet ] = True
         except IndexError:
            mode.addError( "Invalid IP prefix: " + str( prefix ) )

   noOrDefaultHandler = handler

MssDeviceVrf.addCommandClass( MssPolicyExcludeList )

################################################################################
# The "next-hop address ipv4 <next-hop>" command under the vrf submode         #
################################################################################
class MssRoutingTable( Mss.MssPolicyMonitorRoute, BasicCli.ConfigModeBase ):
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, nextHop ):
      self.vrfName = parent.vrfName
      self.dsName = parent.dsName
      self.sdName = parent.sdName
      param = ( self.dsName, self.sdName, self.vrfName, nextHop )
      Mss.MssPolicyMonitorRoute.__init__( self, param )
      if nextHop in parent.vrfConfig.routingTable:
         self.routingTable = parent.vrfConfig.routingTable[ nextHop ]
      else:
         self.routingTable = parent.vrfConfig.newRoutingTable( nextHop )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoMssRoutingTable( mode, args ):
   nextHop = args[ 'NEXT_HOP' ]
   ip = Tac.Value( "Arnet::IpGenAddr", nextHop )
   childMode = mode.childMode( MssRoutingTable, nextHop=ip )
   mode.session_.gotoChildMode( childMode )

def removeMssRoutingTable( mode, args ):
   nextHop = args[ 'NEXT_HOP' ]
   ip = Tac.Value( "Arnet::IpGenAddr", nextHop )
   del mssConfig.deviceSet[ mode.dsName ].serviceDevice[
      mode.sdName ].vrfConfig[ mode.vrfName ].routingTable[ ip ]

#--------------------------------------------------------------------------------
# [ no | default ] ipv4 address NEXT_HOP
#--------------------------------------------------------------------------------
class Ipv4AddressNexthopCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv4 address NEXT_HOP'
   noOrDefaultSyntax = syntax
   data = {
      'ipv4': 'Next hop IPv4 address (device interface address)',
      'address': 'Next hop IPv4 address (device interface address)',
      'NEXT_HOP': IpAddrMatcher.IpAddrMatcher(
         helpdesc='Route next hop IPv4 address (device interface address)' ),
   }
   hidden = True
   handler = gotoMssRoutingTable
   noOrDefaultHandler = removeMssRoutingTable

MssDeviceVrf.addCommandClass( Ipv4AddressNexthopCmd )

#--------------------------------------------------------------------------------
# [ no | default ] next-hop address ipv4 NEXT_HOP
#--------------------------------------------------------------------------------
class NexthopAddressIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = 'next-hop address ipv4 NEXT_HOP'
   noOrDefaultSyntax = syntax
   data = {
      'next-hop': 'Next hop IPv4 address (device interface address)',
      'address': 'Next hop IPv4 address (device interface address)',
      'ipv4': 'Next hop IPv4 address (device interface address)',
      'NEXT_HOP': IpAddrMatcher.IpAddrMatcher(
         helpdesc='Route next hop IPv4 address (device interface address)' ),
   }
   handler = gotoMssRoutingTable
   noOrDefaultHandler = removeMssRoutingTable

MssDeviceVrf.addCommandClass( NexthopAddressIpv4Cmd )

#####################################################################
# The "route PREFIX" command                                        #
#####################################################################
class MssRoute( CliCommand.CliCommandClass ):
   syntax = 'route PREFIX'
   noOrDefaultSyntax = syntax
   data = { 'route': 'Subnets reachable by this next hop',
            'PREFIX': IpAddrMatcher.ipPrefixExpr( 'IP address',
                                                  'Subnet\'s mask value',
                                                  'IPv4 address prefix' ),
   }

   @staticmethod
   def handler( mode, args ):
      try:
         prefix = args[ 'PREFIX' ]
         prefix = Tac.Value( 'Arnet::IpGenPrefix', prefix )
         mode.routingTable.route[ prefix ] = True
      except IndexError:
         mode.addError( "Invalid IP prefix: " + str( prefix ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'PREFIX' in args:
         prefix = args[ 'PREFIX' ]
         prefix = Tac.Value( 'Arnet::IpGenPrefix', prefix )
         del mode.routingTable.route[ prefix ]

MssRoutingTable.addCommandClass( MssRoute )

def Plugin ( em ):
   global mssConfig, mssStatus
   mssConfig = ConfigMount.mount( em, "msspolicymonitor/config",
                                  "MssPolicyMonitor::Config", "wi" )

   mssStatus = LazyMount.mount( em, "msspolicymonitor/status",
                                  "MssPolicyMonitor::Status", "ri" )
