# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import IntfCli
import MsrpCliLib
from CliModel import Bool, Int, Enum, Dict, Model, Str
from IntfModels import Interface
from MsrpTypes import tacListenerState, tacTalkerState
from MsrpTypes import tacTalkerFailureCode, tacSrClassId, tacSrClassState
from MsrpCliLib import countersFmt, msrpCountersKeys
from MrpCliLib import tacMrpAppState, tacMrpRegState
from MrpCliLib import mrpAppStateFmt, mrpRegStateFmt
from TableOutput import createTable, Format
from ArnetModel import MacAddress

_tStateFmt = { tacTalkerState.NoTalker: '--',
               tacTalkerState.TalkerAdvertise: 'Adv',
               tacTalkerState.TalkerFailed: 'Fail' }

_failCodeFmt = { tacTalkerFailureCode.noFailure: '--',
                 tacTalkerFailureCode.insufficientBandwidth: '1',
                 tacTalkerFailureCode.insufficientResources: '2',
                 tacTalkerFailureCode.insufficientTrafficClassBandwidth: '3',
                 tacTalkerFailureCode.streamIdInUse: '4',
                 tacTalkerFailureCode.streamDestinationAddressInUse: '5',
                 tacTalkerFailureCode.streamPreemptedByHigherRank: '6',
                 tacTalkerFailureCode.latencyHasChanged: '7',
                 tacTalkerFailureCode.egressPortNotAvbCapable: '8',
                 tacTalkerFailureCode.useDifferentDestinationAddress: '9',
                 tacTalkerFailureCode.outOfMsrpResources: '10',
                 tacTalkerFailureCode.outOfMmrpResources: '11',
                 tacTalkerFailureCode.cannotStoreDestinationAddress: '12',
                 tacTalkerFailureCode.priorityIsNotAnSrClass: '13',
                 tacTalkerFailureCode.maxFrameSizeTooLarge: '14',
                 tacTalkerFailureCode.maxFanInPortsLimitReached: '15',
                 tacTalkerFailureCode.firstValueChangedForStreamId: '16',
                 tacTalkerFailureCode.vlanBlockedOnEgress: '17',
                 tacTalkerFailureCode.vlanTaggingDisabledOnEgress: '18',
                 tacTalkerFailureCode.srClassPriorityMismatch: '19' }

_lStateFmt = { None: '',
               tacListenerState.NoListener: '--',
               tacListenerState.ListenerReady: 'Rdy',
               tacListenerState.ListenerReadyFailed: 'RdyFail',
               tacListenerState.ListenerAskingFailed: 'AskFail' }

_srClassIdFmt = { None: '--', 'SrClassA': 'A', 'SrClassB': 'B' }

_intfFmt = lambda intf: '--' if not intf else IntfCli.Intf.getShortname( intf )

_vlanFmt = lambda vlanId: vlanId if vlanId else '--'

_attrTypeFmt = { 1: 'Talker Advertise',
                 2: 'Talker Failed',
                 3: 'Listener',
                 4: 'Domain' }

def printLegend( talkers=True, listeners=True ):
   legendHeading = 'Legend'
   print legendHeading
   print '-' * len( legendHeading )

   talkerLegend = { "Adv": "Talker Advertise", "Fail": "Talker Fail" }
   listenerLegend = { "Rdy": "Listener Ready", "RdyFail": "Listener Ready Failed",
                      "AskFail": "Listener Asking Failed" }

   singleEntryFmt = '%-8s: %-25s'
   doubleEntryFmt = '%-8s: %-25s %-8s: %-25s'
   if talkers:
      keys = sorted( talkerLegend.keys() )
      print doubleEntryFmt % ( keys[0], talkerLegend[ keys[0] ],
                               keys[1], talkerLegend[ keys[1] ] )
   if listeners:
      keys = sorted( listenerLegend.keys() )
      print doubleEntryFmt % ( keys[0], listenerLegend[ keys[0] ],
                               keys[1], listenerLegend[ keys[1] ] )
      print singleEntryFmt % ( keys[2], listenerLegend[ keys[2] ] )
   # XXX Also print bridge failure code
   print

class MsrpInfo( Model ):
   msrpEnabled = Bool( help="Indicates whether Msrp is enabled" )
   maxFanInPorts = Int( help="Maxiumum number of ingress ports "
                             "supporting streaming down, beyond which "
                             "no more Talker streams are allowed" )
   maxFrameSize = Int( help="Maximum allowed frame size" )

   class SrClassGlobal( Model ):
      supported = Bool( help="Indicates whether SR Class is supported" )
      priority = Int( help="Priority associated with the SR Class" )
      bandwidth = Int( help="Maximum bandwidth allowed for SR Class" )

   srClasses = Dict( keyType=str, valueType=SrClassGlobal,
                     help="Globally configured parameters for SR Classes" )

   class Intf( Model ):
      operState = Enum( values=( 'Disabled', 'LinkDown', 'Blocked', 'Active' ),
                        help="Operational Msrp state on this interface: "
                             "Disabled -- Msrp is disabled, "
                             "LinkDown -- Physical link is down, "
                             "Blocked -- STP port state is blocked, "
                             "Active -- Msrp is running successfully" )
      srPvid = Int( help="Default VLAN ID for streams on this interface" )
      
      class SrClass( Model ):
         state = Enum( values=tacSrClassState.attributes,
                       help="The state of an SR Class" )
         talkerAdvertise = Int( help="Number of TalkerAdvertise of a given SR Class "
                                     "received on the interface" )
         talkerFailed = Int( help="Number of TalkerFailed of a given SR Class "
                                  "received on the interface" )
         listenerReady = Int( help="Number of ListenerReady and ListenerReadyFailed "
                                   "of a given SR Class received on the interface" )
         listenerAskFailed = Int( help="Number of ListenerAskingFailed of a given "
                                       "SR Class received on the interface" )
         bandwidth = Int( help="Maximum bandwidth allowed for the SR Class" )
         bandwidthUnit = Enum( values=( 'pps', 'bps', 'kbps', 'Mbps' ),
                               help="The unit of bandwidth for SR Classes" )

      srClasses = Dict( keyType=str, valueType=SrClass,
                        help="Msrp information for a given SR Class" )

   interfaces = Dict( keyType=Interface, valueType=Intf,
                      help="Msrp information for a port" )

   def render( self ):
      _msrpEnabledDispFmt = { True : 'Enabled', False: 'Disabled' }
      _fanInDispFmt = lambda x: x if ( x != 0 ) else 'No limit'

      # Global MSRP information
      print
      print '  MSRP Global Status :', _msrpEnabledDispFmt[ self.msrpEnabled ]
      print '  Max Frame Size :', self.maxFrameSize
      print '  Max Fan-In Ports :', _fanInDispFmt( self.maxFanInPorts )
      print

      headings = ( 'Class', 'Supported', 'Priority', ( 'Delta', ( 'Bandwidth', ) ) )
      table = createTable( headings, indent=2 )

      f1 = Format( justify='left' )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )

      f2 = Format( justify='right' )
      f2.noPadLeftIs( True )
      f2.padLimitIs( True )
      table.formatColumns( f2, f2, f2, f2 )

      _classSupportDispFmt = { True: 'Y', False: 'N' }
      _classIdDispFmt = { 'SrClassA': 'A', 'SrClassB': 'B' }
      for className in sorted( self.srClasses ):
         srClass = self.srClasses[ className ]
         table.newRow( _classIdDispFmt[ className ],
                       _classSupportDispFmt[ srClass.supported ],
                       srClass.priority, '%d' % srClass.bandwidth + '%' )
      print table.output()
      print

      printLegend( talkers=True, listeners=True )

      # MSRP information per interface
      headings = [ "Port", ( "Admin", "l", ( "State", ) ),
                   ( "Sr", "l", ( "Pvid", ) ), "Class", "Oper State",
                   ( "Talkers", ( "Adv", "Fail" ) ),
                   ( "Listeners", ( "Rdy", "AskFail" ) ),
                   ( "Bandwidth", ( "Allocated", ) ) ]

      table = createTable( headings, indent=2 )
      f1 = Format( justify="left" )
      f2 = Format( justify="right" )
      table.formatColumns( f1, f1, f2, f2, f1, f2, f2, f2, f2, f2 )

      for intf in Arnet.sortIntf( self.interfaces ):
         interface = self.interfaces[ intf ]
         if interface.operState == 'Disabled':
            # Disabled interfaces are skipped because there can be a lot of them
            continue
         elif interface.operState in [ 'Blocked', 'LinkDown' ]:
            table.newRow( _intfFmt( intf ), interface.operState, interface.srPvid )
         else:
            _bandwidthFmt = lambda srClass: '%d%s' % ( srClass.bandwidth,
                                                       srClass.bandwidthUnit )
            for className in sorted( interface.srClasses ):
               srClass = interface.srClasses[ className ]
               if className == tacSrClassId.SrClassA:
                  table.newRow( _intfFmt( intf ), interface.operState,
                                interface.srPvid, _srClassIdFmt[ className ],
                                srClass.state, srClass.talkerAdvertise,
                                srClass.talkerFailed, srClass.listenerReady,
                                srClass.listenerAskFailed, _bandwidthFmt( srClass ) )
               if className == tacSrClassId.SrClassB:
                  table.newRow( ' ', ' ', ' ', _srClassIdFmt[ className ],
                                srClass.state, srClass.talkerAdvertise,
                                srClass.talkerFailed, srClass.listenerReady,
                                srClass.listenerAskFailed, _bandwidthFmt( srClass ) )
      print table.output()
      print

class MsrpInterfaceStreams( Model ):

   class Intf( Model ):
      
      class Talker( Model ):
         talkerState = Enum( values=tacTalkerState.attributes,
                             help="Talker state on the given stream" )
         talkerDirection = Enum( values=( 'Rx', 'Tx', 'None' ),
                                 help="Rx -- Talker message was received, "
                                      "Tx -- Talker message was transmitted, "
                                      "None -- No Talker present" )
         talkerFailureCode = Enum( values=tacTalkerFailureCode.attributes,
                                   help="Error code behind the "
                                        "Talker Failed message" )
         talkerAge = Str( help="Age since the last state of Talker changed" )
      
      class Listener( Model ):
         listenerState = Enum( values=tacListenerState.attributes,
                               help="Listener state on the given stream" )
         listenerDirection = Enum( values=( 'Rx', 'Tx', 'None' ),
                                   help="Rx -- Listener message was received, "
                                        "Tx -- Listener message was transmitted, "
                                        "None -- No Listener present" )
         listenerAge = Str( help="Age since the last state of Listener changed" )

      streamTalkerData = Dict( keyType=str, valueType=Talker,
                               help="Msrp talker information for a given stream" )
      streamListenerData = Dict( keyType=str, valueType=Listener,
                                 help="Msrp listener information for "
                                      "a given stream" )

   interfaces = Dict( keyType=Interface, valueType=Intf,
                      help="Msrp streams information for a given interface" )

   def render( self ):
      headings = [ "Port", "Stream Id" ]
      formatList = []
      portFormat = Format( minWidth=10, maxWidth=10, justify="left" )
      portFormat.padLimitIs( True )
      streamFormat = Format( minWidth=20, maxWidth=20, justify="left" )
      streamFormat.padLimitIs( True )

      formatList += [ portFormat, streamFormat ]

      interfaces = [ self.interfaces[ intf ] for intf in self.interfaces ]
      displayTalkersInfo = any( [ intf.streamTalkerData
                                  for intf in interfaces ] )
      displayListenersInfo = any( [ intf.streamListenerData
                                    for intf in interfaces ] )

      format1 = Format( minWidth=9, maxWidth=9, justify="left" )
      format1.padLimitIs( True )
      format2 = Format( minWidth=6, maxWidth=6, justify="left" )
      format2.padLimitIs( True )
      format3 = Format( minWidth=7, maxWidth=18, justify="right" )
      format3.padLimitIs( True )

      if( displayTalkersInfo ):
         headings += [ ( "Talkers", ( "Dec", "Dir", "FailCode", "Age" ) ) ]
         formatList += [ format1, format2, format1, format3 ]
      if( displayListenersInfo ):
         headings += [ ( "Listeners", ( "Dec", "Dir", "Age" ) ) ]
         formatList += [ format1, format2, format3 ]

      printLegend( talkers=displayTalkersInfo, listeners=displayListenersInfo )

      table = createTable( headings, indent=2 )
      table.formatColumns( *formatList )

      for intf in Arnet.sortIntf( self.interfaces ):
         interface = self.interfaces[ intf ]
         if not interface.streamTalkerData and not interface.streamListenerData:
            continue
         _dirFmt = lambda direction: '--' if direction == 'None' else direction

         if displayTalkersInfo and displayListenersInfo:
            # Both collections will have same keys
            streamIds = interface.streamTalkerData
         elif displayTalkersInfo:
            streamIds = interface.streamTalkerData
         elif displayListenersInfo:
            streamIds = interface.streamListenerData

         firstLine = True
         for streamId in sorted( streamIds ):
            if firstLine:
               row = [ _intfFmt( intf ), streamId ]
               firstLine = False
            else:
               row = [ ' ', streamId ]

            if displayTalkersInfo:
               tData = interface.streamTalkerData[ streamId ]
               row += [ _tStateFmt[ tData.talkerState ],
                        _dirFmt( tData.talkerDirection ),
                        _failCodeFmt[ tData.talkerFailureCode ],
                        tData.talkerAge ]

            if displayListenersInfo:
               # Print listener streams
               lData = interface.streamListenerData[ streamId ]
               row += [ _lStateFmt[ lData.listenerState ],
                        _dirFmt( lData.listenerDirection ),
                        lData.listenerAge ]
            table.newRow( *row )
         table.newRow() # Add empty line

      print table.output()

class MsrpStreams( Model ):

   class Stream( Model ):
      destinationMacAddr = MacAddress( help="Stream destination MAC address" )
      ingressIntf = Interface( help="Interface on which talker has registered "
                                    "the stream", optional=True )
      talkerState = Enum( values=tacTalkerState.attributes, optional=True,
                          help="State of the talker on the ingress interface" )
      vlanId = Int( help="Vlan Id for the stream", optional=True )
      srClass = Enum( values=tacSrClassId.attributes,
                      help="SR Class of the stream", optional=True )
      bandwidth = Int( help="Bandwidth allocated to the stream" )
      bandwidthUnit = Enum( values=( 'pps', 'bps', 'kbps', 'Mbps' ),
                            help="The unit of bandwidth for SR Classes" )

      failedBridgeId = Int( help="Bridge ID of Bridge that changed Talker Advertise "
                                 "to Talker Failed", optional=True )
      bridgeFailureCode = Enum( values=tacTalkerFailureCode.attributes,
                                optional=True,
                                help="Failure code associated with bridge that "
                                     "changed Talker Advertise to Talker Failed" )
      latency = Int( help="Accumulated latency in nanoseconds for the stream",
                     optional=True )
      maxFrameSize = Int( help="Maximum frame size sent by talker", optional=True )
      maxIntervalFrames = Int( help="Maximum number of frames sent per "
                                    "class measurement", optional=True )

   streams = Dict( keyType=str, valueType=Stream,
                   help="Msrp stream information collected "
                        "from the Talker attributes" )

   def render( self ):
      printLegend( talkers=True, listeners=False )

      headingFmt = '  %-20.20s %-19.19s %-10.10s %-6.6s %4.4s %-5.5s %-10.10s'
      headings = ( "Stream Id", "DMAC", "Port", "Dec", "Vlan", "Class", "Bandwidth" )
      print headingFmt % headings
      print ' ', '-'*20, '-'*19, '-'*10, '-'*6, '-'*4, '-'*5, '-'*10

      rowFmt = '  %-20.20s %-19.19s %-10.10s %-6.6s %4.4s %5.5s %10.10s'
      detailFmt = ' '*17 + '%s'
      macDa = ( lambda stream:
         stream.destinationMacAddr if stream.ingressIntf else '--' )

      for streamId in sorted( self.streams ):
         stream = self.streams[ streamId ]
         print rowFmt % ( streamId, macDa( stream ),
                          _intfFmt( stream.ingressIntf ),
                          _tStateFmt[ stream.talkerState ],
                          _vlanFmt( stream.vlanId ),
                          _srClassIdFmt[ stream.srClass ],
                          '%d%s' % ( stream.bandwidth, stream.bandwidthUnit ) )
         if stream.maxFrameSize is not None:
            # Print detailed statistics
            if stream.failedBridgeId:
               failCode = _failCodeFmt[ stream.bridgeFailureCode ]
               print detailFmt % 'Failure Code:', failCode
               print detailFmt % 'Failed Bridge Id:', stream.failedBridgeId
            print detailFmt % 'Latency (nsec):', stream.latency
            print detailFmt % 'Max Frame Size:', stream.maxFrameSize
            print detailFmt % 'Max Interval Frames:', stream.maxIntervalFrames
            print

class MsrpStreamPropagation( Model ):

   class Stream( Model ):
      destinationMacAddr = MacAddress( help="Stream destination MAC address" )
      vlanId = Int( help="Vlan Id for the stream", optional=True )
      srClass = Enum( values=tacSrClassId.attributes,
                      help="SR Class of the stream", optional=True )
      bandwidth = Int( help="Bandwidth allocated to the stream" )
      bandwidthUnit = Enum( values=( 'pps', 'bps', 'kbps', 'Mbps' ),
                            help="The unit of bandwidth for SR Classes" )

      class TalkerEgressPort( Model ):
         talkerStateEgress = Enum( values=tacTalkerState.attributes,
                                   help="State of the talker propagated "
                                        "on the given egress interface" )
      # Talker propagation information
      talkerIngressIntf = Interface( help="Interface on which the talker message "
                                          "was received", optional=True )
      talkerStateIngressIntf = Enum( values=tacTalkerState.attributes,
                                     help="State of the talker in the message "
                                          "received on the ingress interface" )
      talkerStatePropagated = Enum( values=tacTalkerState.attributes,
                                    help="State of the talker propagated "
                                         "on the given egress interface" )
      talkerEgressInterfaces = Dict( keyType=Interface, valueType=TalkerEgressPort,
                                     help="Egress interfaces on which the talker "
                                          "state of a given stream is propagated" )

      class ListenerIngressPort( Model ):
         listenerStateIngress = Enum( values=tacListenerState.attributes,
                                      help="State of the listener in the message "
                                           "received on the given "
                                           "ingress interface" )
         listenerStatePropagated = Enum( values=tacListenerState.attributes,
                                         optional=True,
                                         help="State of the listener propagated "
                                              "to the egress interface" )
      # Listener propagation information
      listenerIngressInterfaces = Dict( keyType=Interface,
                                        valueType=ListenerIngressPort,
                                        help="Ingress interfaces on which the "
                                             "listener attributes for a given "
                                             "stream are received" )
      listenerEgressIntf = Interface( optional=True,
                                      help="Egress interface on which the "
                                           "listener message was transmitted" )
      listenerStateEgressIntf = Enum( values=tacListenerState.attributes,
                                       help="State of the listener in the message "
                                            "transmitted on the egress interface" )

   streams = Dict( keyType=str, valueType=Stream,
                   help="Shows the propagation of talker and listener attributes "
                        "for a given stream through the switch" )

   def render( self ):
      printLegend( talkers=True, listeners=True )

      headingFmt = '  %-20.20s %-19.19s %-10.10s %-6.6s %4.4s %-5.5s %-10.10s'
      headings = ( "Stream Id", "DMAC", "Port", "Dec", "Vlan", "Class", "Bandwidth" )
      print headingFmt % headings
      print ' ', '-'*20, '-'*19, '-'*10, '-'*6, '-'*4, '-'*5, '-'*10

      streamFmt = '  %-20.20s %-19.19s %-10.10s %-6.6s %4.4s %5.5s %10.10s'

      subHeadingFmt = '%-10.10s     %-10.10s     %-10.10s     %-10.10s     %-10.10s'
      subHeadingFmt = ' '*9 + subHeadingFmt
      rowFmt = ( '%-10.10s %-3.3s %-10.10s %-3.3s %-10.10s %-3.3s %-10.10s %-3.3s '
                 '%-10.10s' )
      rowFmt = ' '*9 + rowFmt

      talkerHeadings1 = ( "Ingress", "Ingress", "Propagated",
                          "Propagated", "Egress" )
      talkerHeadings2 = ( "Dec", "Port", "Dec", "Port", "Dec" )
      listenerHeadings1 = ( "Egress", "Egress", "Propagated", "Listener", "Ingress" )
      listenerHeadings2 = ( "Dec", "Port", "Dec", "Port", "Dec" )

      macDa = ( lambda stream:
         stream.destinationMacAddr if stream.talkerIngressIntf else '--' )

      for streamId in sorted( self.streams ):
         stream = self.streams[ streamId ]
         print streamFmt % ( streamId, macDa( stream ),
                             _intfFmt( stream.talkerIngressIntf ),
                             _tStateFmt[ stream.talkerStateIngressIntf ],
                             _vlanFmt( stream.vlanId ),
                             _srClassIdFmt[ stream.srClass ],
                             '%d%s' % ( stream.bandwidth, stream.bandwidthUnit ) )
         print

         print '    Talker Propagation:'
         print subHeadingFmt % talkerHeadings1
         print subHeadingFmt % talkerHeadings2
         print subHeadingFmt % ( '-'*10, '-'*10, '-'*10, '-'*10, '-'*10 )
         arrow = '-->'
         if stream.talkerIngressIntf:
            line = [ _tStateFmt[ stream.talkerStateIngressIntf ], arrow,
                     _intfFmt( stream.talkerIngressIntf ), arrow,
                     _tStateFmt[ stream.talkerStatePropagated ], arrow ]

            for intfId in Arnet.sortIntf( stream.talkerEgressInterfaces ):
               egressIntf = stream.talkerEgressInterfaces[ intfId ]
               line += [ _intfFmt( intfId ), arrow,
                         _tStateFmt[ egressIntf.talkerStateEgress ] ]
               print rowFmt % tuple( line )
               line = [ ' ', ' ', ' ', ' ', ' ', ' ' ]
         print

         print '    Listener Propagation:'
         print subHeadingFmt % listenerHeadings1
         print subHeadingFmt % listenerHeadings2
         print subHeadingFmt % ( '-'*10, '-'*10, '-'*10, '-'*10, '-'*10 )
         arrow = '<--'
         if stream.listenerEgressIntf:
            line = [ _lStateFmt[ stream.listenerStateEgressIntf ], arrow,
                     _intfFmt( stream.listenerEgressIntf ), arrow ]
         else:
            line = [ ' ', ' ', ' ', ' ' ]

         for intfId in Arnet.sortIntf( stream.listenerIngressInterfaces ):
            ingressIntf = stream.listenerIngressInterfaces[ intfId ]
            line += [ _lStateFmt[ ingressIntf.listenerStatePropagated ], arrow,
                      _intfFmt( intfId ), arrow,
                      _lStateFmt[ ingressIntf.listenerStateIngress ] ]
            print rowFmt % tuple( line )
            line = [ ' ', ' ', ' ', ' ' ]
         print

class MrpDatabase( Model ):

   class IntfDatabase( Model ):

      class AttrTypeDatabase( Model ):

         class Attribute( Model ):
            applicantState = Enum( values=tacMrpAppState.attributes,
                                   help="MRP applicant state" ) 
            registrarState = Enum( values=tacMrpRegState.attributes,
                                   help="MRP registrar state" )
            value = Str( help="String representation of the MRP attribute value",
                         optional=True )

         attributes = Dict( keyType=str, valueType=Attribute,
                            help="MRP attribute information indexed by keys" )
      attributeTypes = Dict( keyType=int, valueType=AttrTypeDatabase,
                             help="MRP type attribute information" )
   interfaces = Dict( keyType=Interface, valueType=IntfDatabase,
                      help="MRP attribute information indexed by interface" )

   def render( self ):
      print ' ' * 51 + 'State'
      headingFmt = '  %-10.10s %-17.17s %-20.20s %-3.3s %-3.3s'
      headings = ( "Port", "Type", "Key", "App", "Reg" )
      print headingFmt % headings
      print headingFmt % ( '-'*10, '-'*17, '-'*20, '-'*3, '-'*3 )
      rowFmt = headingFmt

      for intf in Arnet.sortIntf( self.interfaces ):
         interface = self.interfaces[ intf ]
         for attrType in sorted( interface.attributeTypes ):
            attrTypeInfo = interface.attributeTypes[ attrType ]
            for attrKey in sorted( attrTypeInfo.attributes ):
               attribute = attrTypeInfo.attributes[ attrKey ]

               print rowFmt % ( _intfFmt( intf ), _attrTypeFmt[ attrType ], attrKey,
                                mrpAppStateFmt[ attribute.applicantState ],
                                mrpRegStateFmt[ attribute.registrarState ] )

               if attribute.value:
                  print MsrpCliLib.getAttrValues( attribute.value, attrType )
                  print

class MsrpCounters ( Model ):

   class Intf ( Model ):
      rxCounters = Dict( keyType=str, valueType=int,
                         help="MSRP Rxcounters values for respective interfaces" )
      txCounters = Dict( keyType=str, valueType=int,
                         help="MSRP TxCounters value for respective interfaces" )
      operState = Enum( values=( 'Disabled', 'LinkDown', 'Blocked', 'Active' ),
                        help="Operational Msrp state on this interface: "
                             "Disabled -- Msrp is disabled, "
                             "LinkDown -- Physical link is down, "
                             "Blocked -- STP port state is blocked, "
                             "Active -- Msrp is running successfully" )
   interfaces = Dict( keyType=Interface, valueType=Intf,
                      help="Msrp information for a port" )
   
   def render( self ):
      #MSRP counters value
      print ' ' * 52 + 'Directions'
      headingFmt = '  %-10.10s %-32.32s %12.12s %12.12s'
      headings = ( "Port", "Counters", "Rx", "Tx" )
      print headingFmt % headings
      print headingFmt % ( '-'*10, '-'*32, '-'*12, '-'*12 )
      rowFmt = headingFmt
      for intf in Arnet.sortIntf( self.interfaces ):
         flag = True
         interface = self.interfaces[ intf ]
         if interface.operState != 'Active':
            continue
         else:
            for counters in msrpCountersKeys:
               if flag:
                  print rowFmt % ( _intfFmt( intf ),
                                   countersFmt[ counters ],
                                   interface.rxCounters[ counters ],
                                   interface.txCounters[ counters ] )
                  flag = False
               else:
                  print rowFmt % ( ' ',
                                   countersFmt[ counters ],
                                   interface.rxCounters[ counters ],
                                   interface.txCounters[ counters ] )
            
      print

class MsrpStreamCount( Model ):
   activeAvbStreams = Int( help="Number of established AVB Streams. "
                           "An established AVB stream is one which "
                           "has a registered talker and has bandwidth "
                           "allocated to it " )
   def render( self ):
      print 'Total Number of Active Streams :', self.activeAvbStreams
      print
