# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import BasicCli
import CliCommand
import CliMatcher
import CliMode.Msrp
import CliParser
import CliPlugin.IntfCli as IntfCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.TechSupportCli as TechSupportCli
import CliToken.Clear
import ConfigMount
import Intf
import LazyMount
import MsrpCliLib
from MsrpModels import MsrpStreams, MsrpStreamPropagation, MsrpInterfaceStreams
from MsrpModels import MsrpInfo, MrpDatabase
from MsrpModels import MsrpCounters, MsrpStreamCount
from MsrpStream import matcherStreamPattern, convertStreamIdDisplayToInt
from MsrpStream import matchStreamId, maskForStreamId
from MsrpStream import convertStreamIdToHexString
from MsrpTypes import tacTalkerState, tacListenerState, tacSrClassId
from MsrpTypes import tacSrClassState, tacTalkerFailureCode, intToSrClassId
import ShowCommand
import Tac
import Tracing
import Url

__defaultTraceHandle__ = Tracing.Handle( 'MsrpCli' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

#------------------------------------------------------------------------------------
# The Msrp globals
#------------------------------------------------------------------------------------
mrpConstants = Tac.Type( 'Mrp::MrpTimerConstants' )
msrpShapingConsts = Tac.Type( 'Msrp::MsrpShapingMultiplierConstants' )
msrpApp = Tac.Type( 'Mrp::MrpApplication' ).MSRP

msrpCliConfig = None
msrpConfig = None
msrpStatus = None
mrpConfig = None
mrpStatus = None
msrpHwStatus = None
msrpCounterCheckpoint = None
streamCounters = {} # Stores the talker/listener stream counters, index by intfId
macToDeviceMap = {} # Stores mac to device name mapping
intfToStreamMap = {} # Stores Interface to Stream mapping
entMan = None

def interfaceCountersValue( intfStatus ):
   intfName = intfStatus.intfId
   # Get global snapshot.
   globalSnapshot = msrpCounterCheckpoint.interfaceCounterCheckpoint.get( intfName )

   # Create a dummy snapshot that was logically created at the same time as the
   # intfStatus
   zero = Tac.newInstance(
      "Msrp::InterfaceCounterCheckpoint", intfName )
   zero.timestamp = intfStatus.creationTime
   # Setting counters to 0
   for counterName in intfStatus.txCounters:
      zero.txCounters[ counterName ] = 0
      zero.rxCounters[ counterName ] = 0

   # Choose the newest of the two snapshots.
   snapshots = [ zero ]
   if globalSnapshot is not None:
      snapshots.append( globalSnapshot )
   snapshots.sort( key=lambda snapshot: snapshot.timestamp )
   snapshot = snapshots[ -1 ]
   
   txCounters = {} 
   rxCounters = {}
   # Different loops for both Rx and Tx, just to make it more clear. 
   for counterName in intfStatus.txCounters:
      currentValue = intfStatus.txCounters[ counterName ]
      snapshotValue = snapshot.txCounters[ counterName ]
      assert currentValue >= snapshotValue
      txCounters[ counterName ] = currentValue - snapshotValue
   
   for counterName in intfStatus.rxCounters:
      currentValue = intfStatus.rxCounters[ counterName ]
      snapshotValue = snapshot.rxCounters[ counterName ]
      assert currentValue >= snapshotValue
      rxCounters[ counterName ] = currentValue - snapshotValue
   return txCounters, rxCounters

def initMsrpStatusReactor():
   entMan.msrpStatusReactor = MsrpStatusReactor()
 
def fetchCounter( intfId, srClass ):
   if intfId in streamCounters:
      counterClass = streamCounters[ intfId ]
      if srClass in counterClass:
         counter = counterClass[ srClass ]
      else:
         counter = StreamCounter( intfId, srClass )
         counterClass[ srClass ] = counter
         streamCounters[ intfId ] = counterClass
   else:
      counter = StreamCounter( intfId, srClass )
      counterClass = {}
      counterClass[ srClass ] = counter
      streamCounters[ intfId ] = counterClass
   return counter

def loadMacToDeviceMapFile( configFileName ):
   '''Reads from mac to device name configuration file and stores the
   mapping in macToDeviceMap dictionary'''
   global macToDeviceMap
   lines = open( configFileName, 'r' ).readlines()
   macToDeviceMap = dict( [ tuple( line.split() ) for line in lines ] )

def convertStreamIdToDisplay( streamId ):
   ''' If streamId has a mapping in macToDeviceMap then return <device name>.H
   otherwise return the hex(H.H.H.H) representation of streamId'''
   streamId = convertStreamIdToHexString( streamId )
   #Try matching H.H.H
   if streamId[ : 14 ] in macToDeviceMap:
      return macToDeviceMap[ streamId[ : 14 ] ] + streamId[ 14 : ]
   #Try matching H.H
   elif streamId[ : 9 ] in macToDeviceMap:
      return macToDeviceMap[ streamId[ : 9 ] ] + streamId[ 9 : ]
   else:
      return streamId

class MsrpModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( 'Ethernet' ) and not mode.intf.isSubIntf()

IntfCli.IntfConfigMode.addModelet( MsrpModelet )

class MsrpConfigMode( CliMode.Msrp.MsrpConfigMode, BasicCli.ConfigModeBase ):
   name = "Global MSRP configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CliMode.Msrp.MsrpConfigMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#------------------------------------------------------------------------------------
# Msrp tokens
#------------------------------------------------------------------------------------
def guardMsrp( mode, token ):
   if msrpHwStatus.msrpSupported:
      return None
   return CliParser.guardNotThisPlatform

nodeMsrp = CliCommand.guardedKeyword( 'msrp', 'Configure MSRP parameters',
                                    guardMsrp )
nodeShowMsrp = CliCommand.guardedKeyword( 'msrp', helpdesc='Show MSRP information',
      guard=guardMsrp )
matcherInterfaces = CliMatcher.KeywordMatcher( 'interfaces',
                     helpdesc='Show MSRP information for specific interfaces' )
matcherIntfRange = Intf.IntfRange.IntfRangeMatcher(
      explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType, ) )
matcherStreams = CliMatcher.KeywordMatcher( 'streams', 'MSRP Streams' )
matcherStreamId = CliMatcher.KeywordMatcher( 'stream-id',
      helpdesc='Show MSRP streams information for a specific stream' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='More comprehensive output' )
matcherCounters = CliMatcher.KeywordMatcher( 'counters',
                                      helpdesc='Msrp counters' )
matcherShaping = CliMatcher.KeywordMatcher( 'shaping',
      helpdesc='Configure MSRP shaping parameters' )

#------------------------------------------------------------------------------------
# Mrp tokens
#------------------------------------------------------------------------------------
matcherMrp = CliMatcher.KeywordMatcher( 'mrp',
      helpdesc='Configure MRP parameters' )

#--------------------------------------------------------------------------------
# msrp
# enter msrp-config mode
#--------------------------------------------------------------------------------
class MsrpConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'msrp'
   data = {
      'msrp' : nodeMsrp,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( MsrpConfigMode )
      mode.session_.gotoChildMode( childMode )

BasicCli.GlobalConfigMode.addCommandClass( MsrpConfigCmd )

#--------------------------------------------------------------------------------
# ( no | default ) shaping
#--------------------------------------------------------------------------------
class DisableMsrpShapingCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'shaping'
   data = {
      'shaping' : matcherShaping,
   }

   @staticmethod
   def noHandler( mode, args ):
      msrpCliConfig.disableShaping = True

   @staticmethod
   def defaultHandler( mode, args ):
      msrpCliConfig.disableShaping = False

MsrpConfigMode.addCommandClass( DisableMsrpShapingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shaping multiplier MULTIPLIER
#--------------------------------------------------------------------------------
class MsrpShapingMultiplierCmd( CliCommand.CliCommandClass ):
   syntax = 'shaping multiplier MULTIPLIER'
   noOrDefaultSyntax = 'shaping multiplier ...'
   data = {
      'shaping' : matcherShaping,
      'multiplier' : 'Set the MSRP shaping rate multiplier',
      'MULTIPLIER' : CliMatcher.FloatMatcher(
         msrpShapingConsts.minMultiplier, float( 'Inf' ),
         helpdesc='MSRP shaping rate multiplier. (Min: 1.0, Default: 1.1)',
         precisionString='%.25g' ),
   }

   @staticmethod
   def handler( mode, args ):
      msrpCliConfig.shapingMultiplier = args[ 'MULTIPLIER' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      msrpCliConfig.shapingMultiplier = msrpShapingConsts.defaultMultiplier

MsrpConfigMode.addCommandClass( MsrpShapingMultiplierCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mrp leave-all-timer TIME
#--------------------------------------------------------------------------------
class MrpLeaveAllTimerLeavealltimerCmd( CliCommand.CliCommandClass ):
   syntax = 'mrp leave-all-timer TIME'
   noOrDefaultSyntax = 'mrp leave-all-timer ...'
   data = {
      'mrp' : matcherMrp,
      'leave-all-timer' : 'Mrp leaveAll timer',
      'TIME' : CliMatcher.FloatMatcher(
         mrpConstants.minLeaveAllTime, mrpConstants.maxLeaveAllTime,
         helpdesc='Mrp leaveAll timer value( sec )',
         precisionString='%.25g' ),
   }

   @staticmethod
   def handler( mode, args ):
      mrpConfig.intfLeaveAllTime[ mode.intf.name ] = args[ 'TIME' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = mode.intf.name
      if intf in mrpConfig.intfLeaveAllTime:
         del mrpConfig.intfLeaveAllTime[ intf ]

MsrpModelet.addCommandClass( MrpLeaveAllTimerLeavealltimerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mrp leave-timer TIME
#--------------------------------------------------------------------------------
class MrpLeaveTimerLeavetimerCmd( CliCommand.CliCommandClass ):
   syntax = 'mrp leave-timer TIME'
   noOrDefaultSyntax = 'mrp leave-timer ...'
   data = {
      'mrp' : matcherMrp,
      'leave-timer' : 'Mrp leave timer',
      'TIME' : CliMatcher.FloatMatcher(
         mrpConstants.minLeaveTime, mrpConstants.maxLeaveTime,
         helpdesc='Mrp leave timer value( sec )',
         precisionString='%.25g' ),
   }

   @staticmethod
   def handler( mode, args ):
      mrpConfig.intfLeaveTime[ mode.intf.name ] = args[ 'TIME' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = mode.intf.name
      if intf in mrpConfig.intfLeaveTime:
         del mrpConfig.intfLeaveTime[ intf ]

MsrpModelet.addCommandClass( MrpLeaveTimerLeavetimerCmd )

#------------------------------------------------------------------------------------
# '[no] streams load-file <fileName>' in msrp-config mode.
#------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------
# '[no] msrp streams load-file <fileName>' in global-config mode (for backward
# compatibility).
#------------------------------------------------------------------------------------
def setStreamsMapFile( mode, macToDeviceFileName=None ):
   global macToDeviceMap
   if not macToDeviceFileName:
      msrpCliConfig.macToDeviceNameMapFile = ''
      macToDeviceMap = {}
   else:
      try:
         msrpCliConfig.macToDeviceNameMapFile = macToDeviceFileName.pathname
         loadMacToDeviceMapFile( macToDeviceFileName.realFilename_ )
      except IOError:
         mode.addWarning( "%s file does not exist!" % macToDeviceFileName )
      except ValueError:
         mode.addWarning( "macToDeviceFileName is not in correct format!" )

matcherLoadFile = CliMatcher.KeywordMatcher( 'load-file',
                                        'Mac to device name configuration file' )
urlMatcher = Url.UrlMatcher( lambda fs: True,
                             "Path of MAC to device name config file",
                             notAllowed=[] )

class StreamFileCmd( CliCommand.CliCommandClass ):
   syntax = 'streams load-file URL'
   noOrDefaultSyntax = 'streams load-file ...'
   data = {
      'streams' : matcherStreams,
      'load-file' : matcherLoadFile,
      'URL' : urlMatcher
      }
   @staticmethod
   def handler( mode, args ):
      setStreamsMapFile( mode, args.get( 'URL' ) )

   noOrDefaultHandler = handler

MsrpConfigMode.addCommandClass( StreamFileCmd )

class GlobalModeStreamFileCmd( CliCommand.CliCommandClass ):
   syntax = 'msrp streams load-file URL'
   noOrDefaultSyntax = 'streams load-file ...'
   data = {
      'msrp' : nodeMsrp,
      'streams' : matcherStreams,
      'load-file' : matcherLoadFile,
      'URL' : urlMatcher
      }
   @staticmethod
   def handler( mode, args ):
      setStreamsMapFile( mode, args.get( 'URL' ) )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( GlobalModeStreamFileCmd )

#--------------------------------------------------------------------------------
# [ no | default ] msrp
#--------------------------------------------------------------------------------
class MsrpEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'msrp'
   noOrDefaultSyntax = syntax
   data = {
      'msrp' : nodeMsrp
   }

   @staticmethod
   def handler( mode, args ):
      msrpConfig.intfEnable[ mode.intf.name ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = mode.intf.name
      if intf in msrpConfig.intfEnable:
         del msrpConfig.intfEnable[ intf ]

MsrpModelet.addCommandClass( MsrpEnableCmd )

class StreamCounter( object ):
   def __init__( self, intfId, srClass ):
      self.intfId = intfId
      self.srClass = srClass
      self.talkerAdvertise = 0
      self.talkerFailed = 0
      self.listenerReady = 0
      self.listenerAskFailed = 0

class MsrpStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Msrp::Status"
   def __init__( self ):
      self.streamDataReactor = {}
      Tac.Notifiee.__init__( self, msrpStatus )
      for streamId in msrpStatus.registeredStreams:
         self.handleStream( streamId )

   @Tac.handler( "registeredStreams" )
   def handleStream( self, streamId ):
      if streamId in self.notifier_.registeredStreams:
         self.streamDataReactor[ streamId ] = StreamDataReactor(
            streamId, msrpStatus.registeredStreams[ streamId ] )
      else:
         del self.streamDataReactor[ streamId ]

class StreamDataReactor( Tac.Notifiee ):
   notifierTypeName = "Msrp::StreamData"
   def __init__( self, streamId, streamData ):
      self.streamId = streamId
      Tac.Notifiee.__init__( self, streamData )
      self.streamData = streamData
      self.srClass_ = self.streamData.srClass
      self.localTalkerState = {}
      self.localListenerState = {}
      self.localListenerStateOrig = {}
      for intfId in self.streamData.talkerState:
         self.handleTalker( intfId )
      for intfId in self.streamData.listenerStateOrig:
         self.handleListenerOrig( intfId )
      for intfId in self.streamData.listenerState:
         self.handleListener( intfId )   

   def __del__( self ):
      for intfId in self.localTalkerState.keys():
         self.decrementIntfCount( self.streamId, intfId )
         self.decrementTalkerCounter( intfId )
            
      for intfId in self.localListenerState.keys():
         self.decrementIntfCount( self.streamId, intfId )
         # Checking for the interface in listenerOrig State
         # if the interface is present then we need to increment
         # counters accordingly
         if intfId in self.localListenerStateOrig:
            self.incrementListenerCounter( intfId, orig=True )
         self.decrementListenerCounter( intfId )
         del self.localListenerState[ intfId ]
         
      for intfId in self.localListenerStateOrig.keys():
         self.decrementIntfCount( self.streamId, intfId )
         # Checking for the interface in the listenerState
         # If it is not present then, we need to decrement the counters
         if intfId not in self.localListenerState:
            self.decrementListenerCounter( intfId, orig=True )
         del self.localListenerStateOrig[ intfId ]
  
   def incrementIntfCount( self, streamId, intfId ):
      if intfId in intfToStreamMap:
         if streamId in intfToStreamMap[ intfId ]:
            intfToStreamMap[ intfId ][ streamId ] += 1
         else:
            intfToStreamMap[ intfId ][ streamId ] = 1
      else:
         intfToStreamMap[ intfId ] = {}
         intfToStreamMap[ intfId ][ streamId ] = 1

   def decrementIntfCount( self, streamId, intfId ):
      if intfId not in intfToStreamMap or streamId not in intfToStreamMap[ intfId ]:
         return
      intfToStreamMap[ intfId ][ streamId ] -= 1
      if intfToStreamMap[ intfId ][ streamId ] == 0:
         del intfToStreamMap[ intfId ][ streamId ]
         if not intfToStreamMap[ intfId ]:
            del intfToStreamMap[ intfId ]
            
   def incrementTalkerCounter( self, intfId ):
      if intfId != self.notifier_.ingressIntf:
         return
      counter = fetchCounter( intfId, self.srClass_ )
      tState = self.notifier_.talkerState[ intfId ]
      if tState == tacTalkerState.TalkerAdvertise:
         counter.talkerAdvertise += 1
      elif tState  == tacTalkerState.TalkerFailed:
         counter.talkerFailed += 1
      self.localTalkerState[ intfId ] = tState
 
   def decrementTalkerCounter( self, intfId ):
      counter = fetchCounter( intfId, self.srClass_ )
      tState = self.localTalkerState[ intfId ]
      if tState == tacTalkerState.TalkerAdvertise:
         counter.talkerAdvertise -= 1
      elif tState == tacTalkerState.TalkerFailed:
         counter.talkerFailed -= 1
      del self.localTalkerState[ intfId ]

   def incrementListenerCounter( self, intfId, orig=False ):
      if intfId == self.notifier_.ingressIntf:
         return
      counter = fetchCounter( intfId, self.srClass_ )
      if orig:
         lState = self.notifier_.listenerStateOrig.get( intfId )
      else:
         lState = self.notifier_.listenerState.get( intfId )
         
      if lState == tacListenerState.ListenerReady or \
             lState == tacListenerState.ListenerReadyFailed:
         counter.listenerReady += 1
      elif lState == tacListenerState.ListenerAskingFailed:
         counter.listenerAskFailed += 1
      if orig:
         self.localListenerStateOrig[ intfId ] = lState
      else:
         self.localListenerState[ intfId ] = lState
       
   def decrementListenerCounter( self, intfId, orig=False ):
      counter = fetchCounter( intfId, self.srClass_ )
      if orig:
         lState = self.localListenerStateOrig.get( intfId )
      else:
         lState = self.localListenerState.get( intfId )
      if lState == tacListenerState.ListenerReady or \
             lState == tacListenerState.ListenerReadyFailed:
         counter.listenerReady -= 1
      elif lState == tacListenerState.ListenerAskingFailed:
         counter.listenerAskFailed -= 1
   
   @Tac.handler( "talkerState" )
   def handleTalker( self, intfId ):
      if intfId in self.notifier_.talkerState:
         self.incrementIntfCount( self.streamId, intfId )
         self.incrementTalkerCounter( intfId )
      else:
         if intfId in self.localTalkerState:
            self.decrementIntfCount( self.streamId, intfId )
            self.decrementTalkerCounter( intfId )

   @Tac.handler( "listenerState" )
   def handleListener( self, intfId ):
      if intfId in self.notifier_.listenerState:
         self.incrementIntfCount( self.streamId, intfId )
         # Decrement the previously existing listenerState/listenerStateOrig Count
         if intfId in self.localListenerState:
            self.decrementListenerCounter( intfId )
         elif intfId in self.localListenerStateOrig:
            self.decrementListenerCounter( intfId, orig=True )
         self.incrementListenerCounter( intfId )
      else:
         if intfId in self.localListenerState:
            self.decrementIntfCount( self.streamId, intfId )
            # we need to check for the listenerOrig state before
            # deleting listener . So, that we can update the counters
            # according to the listenerOrig state 
            if intfId in self.localListenerStateOrig:
               self.incrementListenerCounter( intfId, orig=True )
            self.decrementListenerCounter( intfId )
            del self.localListenerState[ intfId ]
            
   @Tac.handler( "listenerStateOrig" )
   def handleListenerOrig( self, intfId ):
      if intfId in self.notifier_.listenerStateOrig:
         self.incrementIntfCount( self.streamId, intfId )
         # we need to check, if interface already exist in listenerState
         # Then, we need not to update counters.
         if intfId not in self.localListenerState:
            self.incrementListenerCounter( intfId, orig=True )
      else:
         if intfId in self.localListenerStateOrig:
            self.decrementIntfCount( self.streamId, intfId )
            # Checking for interface in listener State
            # if it is present, then we should not decrement 
            # counters, as they must have been decremented earlier 
            # when that interface was added to listener state
            if intfId not in self.localListenerState:
               self.decrementListenerCounter( intfId, orig=True )
            del self.localListenerStateOrig[ intfId ]

# Utility function to filter streamData based on search criteria
def filterStreamData( intf=None, streamId=None, talkers=True, listeners=True,
                      dmac=None, smac=None ):
   if streamId:
      streamIdU64, lenStreamId = convertStreamIdDisplayToInt( streamId )
      mask, possibleStreamId = maskForStreamId( lenStreamId, streamIdU64 )
   registeredStreams = msrpStatus.registeredStreams
   streamIds = []
   streamIterList = []
   if intf:
      if intf.name in intfToStreamMap:
         streamIterList = intfToStreamMap[ intf.name ].items()
   else:
      streamIterList.extend( msrpStatus.registeredStreams.items() )
   for sId, _ in streamIterList:
      if streamId:
         if not matchStreamId( mask, possibleStreamId, sId ):
            continue
      if sId not in registeredStreams:
         continue
      streamData = registeredStreams[ sId ]
      if talkers and listeners:
         if not streamData.ingressIntf and not streamData.listenerStateOrig:
            continue
      elif talkers and not streamData.ingressIntf:
         continue
      elif listeners and not streamData.listenerStateOrig:
         continue
      if dmac:
         if not streamData.ingressIntf:
            continue
         else:
            if streamData.dmac != dmac:
               continue
      if smac:
         if intf:
            intfSmac = streamData.smac.get( intf.name )
            match = ( intfSmac == smac )
         else:
            match = ( smac in streamData.smac.values() )
         if not match:
            continue
      if intf:
         tState = streamData.talkerState.get( intf.name )
         lState = streamData.listenerState.get( intf.name )
         if not lState:
            lState = streamData.listenerStateOrig.get( intf.name )
         if talkers and listeners:
            if not tState and not lState:
               continue
         elif talkers and not tState:
            continue
         elif listeners and not lState:
            continue
      streamIds.append( sId )
   return sorted( streamIds )

#--------------------------------------------------------------------------------
# show msrp [ interfaces INTF ]
#--------------------------------------------------------------------------------
def showMsrp( mode, args ):
   # Global information
   mod = None
   initMsrpStatusReactor()

   msrpModel = MsrpInfo()
   msrpModel.msrpEnabled = bool( msrpConfig.intfEnable )
   msrpModel.maxFrameSize = msrpCliConfig.maxFrameSize
   msrpModel.maxFanInPorts = msrpCliConfig.maxFanInPorts

   srClassAModel = msrpModel.SrClassGlobal()
   srClassAModel.supported = msrpCliConfig.srClassASupported
   srClassAModel.priority = msrpCliConfig.srClassAPriority
   srClassAModel.bandwidth = msrpCliConfig.srClassADeltaBandwidth
   msrpModel.srClasses[ tacSrClassId.SrClassA ] = srClassAModel

   srClassBModel = msrpModel.SrClassGlobal()
   srClassBModel.supported = msrpCliConfig.srClassBSupported
   srClassBModel.priority = msrpCliConfig.srClassBPriority
   srClassBModel.bandwidth = msrpCliConfig.srClassBDeltaBandwidth
   msrpModel.srClasses[ tacSrClassId.SrClassB ] = srClassBModel

   def _getSrClassModel( srClass ):
      if srClass in srClasses:
         srClassModel = srClasses[ srClass ]
      else:
         srClassModel = msrpIntfModel.SrClass( talkerAdvertise=0, talkerFailed=0,
                                               listenerReady=0,
                                               listenerAskFailed= 0 )
         srClasses[ srClass ] = srClassModel
      return srClassModel
   # Interface information
   intfs = IntfCli.Intf.getAll( mode, args.get( 'INTF' ), mod, EthIntfCli.EthIntf )

   for intf in intfs:
      if not intf.name.startswith( 'Ethernet' ):
         #and not intf.name.startswith( 'Port-Channel' ):
         continue
      msrpIntfModel = msrpModel.Intf()
      srPvid = msrpCliConfig.srPvid.get( intf.name, msrpCliConfig.defaultSrPvid )
      msrpIntfModel.srPvid = srPvid
      msrpModel.interfaces[ intf.name ] = msrpIntfModel

      srClasses = {}
      if intf.name not in msrpStatus.intfStatus:
         msrpIntfModel.operState = 'Disabled'
      else:
         intfStatus = msrpStatus.intfStatus[ intf.name ]
         if ( intfStatus.linkStatus == 'linkUp' and 
              intfStatus.portState == 'portForwarding' ):
            msrpIntfModel.operState = 'Active'
         elif intfStatus.linkStatus != 'linkUp':
            # Check link status before port state, because if link is down,
            # port state will also be blocked.
            msrpIntfModel.operState = 'LinkDown'
         elif intfStatus.portState != 'portForwarding':
            msrpIntfModel.operState = 'Blocked'

         if msrpIntfModel.operState != 'Active':
            continue

         for srClass in [ tacSrClassId.SrClassA, tacSrClassId.SrClassB ]:
            srClassModel = _getSrClassModel( srClass )
            srClassModel.state = intfStatus.srClassState.get( srClass,
                                          tacSrClassState.NotSupported )
            srClassModel.bandwidth = intfStatus.classBandwidthAllocated[ srClass ]
            srClassModel.bandwidthUnit = 'kbps'
            counterFetched = fetchCounter( intf.name, srClass )
            srClassModel.talkerAdvertise = counterFetched.talkerAdvertise
            srClassModel.talkerFailed = counterFetched.talkerFailed
            srClassModel.listenerReady = counterFetched.listenerReady
            srClassModel.listenerAskFailed = counterFetched.listenerAskFailed

      msrpIntfModel.srClasses = srClasses
   return msrpModel

class ShowMsrpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show msrp [ interfaces INTF ]'
   data = {
      'msrp' : nodeShowMsrp,
      'interfaces' : matcherInterfaces,
      'INTF' : matcherIntfRange
   }

   handler = showMsrp
   cliModel = MsrpInfo

BasicCli.addShowCommandClass( ShowMsrpCmd )

#--------------------------------------------------------------------------------
# show msrp interfaces [ INTF ] ( streams | listeners | talkers )
#                                                            [ stream-id STREAM_ID ]
#--------------------------------------------------------------------------------
def showMsrpIntfStreams( mode, args ):
   talkers = 'talkers' in args
   listeners = 'listeners' in args
   streamId = args.get( 'STREAM_ID' )
   initMsrpStatusReactor()
   msrpModel = MsrpInterfaceStreams()
   intfs = IntfCli.Intf.getAll( mode, args.get( 'INTF' ), None, EthIntfCli.EthIntf )

   for intf in intfs:
      if not intf.name.startswith( 'Ethernet' ):
         #and not intf.name.startswith( 'Port-Channel' ):
         continue
      if not talkers and not listeners:
         # If no keyword is specified, display both talkers and listeners
         # information.
         both = True
         streamIds = filterStreamData( intf, streamId, talkers=True, listeners=True )
      else:
         both =  False
         streamIds = filterStreamData( intf, streamId, talkers, listeners )

      talkerInfo = {}
      listenerInfo = {}
      for sId in streamIds:
         if sId not in msrpStatus.registeredStreams:
            continue
         streamIdString = convertStreamIdToDisplay( sId )
         streamData = msrpStatus.registeredStreams[ sId ]
         if talkers or both:
            state = streamData.talkerState.get( intf.name )
            statusTime = streamData.statusTime.get( intf.name )
            if state:
               tState = state
               if intf.name == streamData.ingressIntf:
                  direction = 'Rx'
                  # Strip off terminating 'ago'
                  tAge = Ark.timestampToStr( statusTime ).strip( ' ago' )
                  # If age is long, strip off the timestamp
                  tAge = tAge.partition( ',' )[ 0 ] if len( tAge ) > 18 else tAge
               else:
                  direction = 'Tx'
                  tAge = '--'
               failCode = streamData.talkerFailureCode.get( 
                  intf.name,
                  tacTalkerFailureCode.noFailure )
            else:
               tState = tacTalkerState.NoTalker
               direction = 'None'
               failCode = tacTalkerFailureCode.noFailure
               tAge = '--'
            msrpTalkerModel = msrpModel.Intf.Talker( talkerState=tState,
                                                     talkerDirection=direction,
                                                     talkerFailureCode=failCode,
                                                     talkerAge=tAge )
            talkerInfo[ streamIdString ] = msrpTalkerModel

         if listeners or both:
            state = streamData.listenerState.get( intf.name )
            statusTime  = streamData.statusTime.get( intf.name )
            if not state:
               state = streamData.listenerStateOrig.get( intf.name )
            if state:
               lState = state
               if intf.name == streamData.ingressIntf:
                  direction = 'Tx'
                  lAge = '--'
               else:
                  direction = 'Rx'
                  lAge = Ark.timestampToStr( statusTime ).strip( ' ago' )
                  # If age is long, strip off the timestamp
                  lAge = lAge.partition( ',' )[ 0 ] if len( lAge ) > 18 else lAge
            else:
               lState = tacListenerState.NoListener
               direction = 'None'
               lAge = '--'
            msrpListenerModel = msrpModel.Intf.Listener( 
               listenerState=lState,
               listenerDirection=direction,
               listenerAge=lAge )
            listenerInfo[ streamIdString ] = msrpListenerModel

      msrpIntfModel = msrpModel.Intf( streamTalkerData=talkerInfo,
                                      streamListenerData=listenerInfo )
      msrpModel.interfaces[ intf.name ] = msrpIntfModel

   return msrpModel

class ShowMsrpIntfStreamsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show msrp interfaces [ INTF ] ( streams | listeners | talkers ) '
                                                         '[ stream-id STREAM_ID ]' )
   data = {
      'msrp' : nodeShowMsrp,
      'interfaces' : matcherInterfaces,
      'INTF' : matcherIntfRange,
      'streams' : matcherStreams,
      'talkers' : 'Show MSRP talkers information',
      'listeners' : 'Show MSRP listeners information',
      'stream-id' : matcherStreamId,
      'STREAM_ID' : matcherStreamPattern,
   }

   handler = showMsrpIntfStreams
   cliModel = MsrpInterfaceStreams

BasicCli.addShowCommandClass( ShowMsrpIntfStreamsCmd )

#--------------------------------------------------------------------------------
# show msrp streams [ stream-id STREAM_ID ] [ detail ]
#--------------------------------------------------------------------------------
def showMsrpStreams( mode, args ):
   streamId = args.get( 'STREAM_ID' )
   initMsrpStatusReactor()
   streams = {}
   streamIds = filterStreamData( None, streamId, None, None )
   for sId in streamIds:
      if sId not in msrpStatus.registeredStreams:
         continue
      streamModel = MsrpStreams.Stream()
      streamData = msrpStatus.registeredStreams[ sId ]
      streamModel.destinationMacAddr = streamData.macDa
      if streamData.ingressIntf:
         streamModel.ingressIntf = streamData.ingressIntf
         streamModel.talkerState = streamData.talkerState[ streamData.ingressIntf ]
         streamModel.vlanId = streamData.vlanId
         streamModel.srClass = streamData.srClass
         streamModel.bandwidth = streamData.bandwidth
      else:
         streamModel.talkerState = tacTalkerState.NoTalker
         streamModel.bandwidth = 0
            
      streamModel.bandwidthUnit = 'kbps'

      if 'detail' in args:
         streamModel.bridgeFailureCode = streamData.talkerFailureCode.get(
            streamData.ingressIntf, tacTalkerFailureCode.noFailure )
         streamModel.failedBridgeId = streamData.failedBridgeId
         streamModel.latency = streamData.accumulatedLatency
         streamModel.maxFrameSize = streamData.maxFrameSize
         streamModel.maxIntervalFrames = streamData.maxIntervalFrames

      streams[ convertStreamIdToDisplay( sId ) ] = streamModel
   return MsrpStreams( streams=streams )

class MsrpStreamsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show msrp streams [ stream-id STREAM_ID ] [ detail ]'
   data = {
      'msrp' : nodeShowMsrp,
      'streams' : matcherStreams,
      'stream-id' : matcherStreamId,
      'STREAM_ID' : matcherStreamPattern,
      'detail' : matcherDetail,
   }

   handler = showMsrpStreams
   cliModel = MsrpStreams

BasicCli.addShowCommandClass( MsrpStreamsCmd )

#--------------------------------------------------------------------------------
# show msrp streams [ stream-id STREAM_ID ] propagation
#--------------------------------------------------------------------------------
def showMsrpStreamPropagation( mode, args ):
   initMsrpStatusReactor() 
   streams = {}
   streamIds = filterStreamData( streamId=args.get( 'STREAM_ID' ) )
   for sId in streamIds:
      if sId not in msrpStatus.registeredStreams:
         continue
      streamData = msrpStatus.registeredStreams[ sId ]
      talkerIngressIntf = listenerEgressIntf = None
      talkerStateIngress = talkerStatePropagated = tacTalkerState.NoTalker
      listenerStateEgress = tacListenerState.NoListener
         
      streamModel = MsrpStreamPropagation.Stream()

      if streamData.ingressIntf:
         talkerIngressIntf = listenerEgressIntf = streamData.ingressIntf
         talkerStateIngress = streamData.talkerState[ talkerIngressIntf ]
         # Propagated talkerState will be TalkerFailed if there exists a
         # TalkerFailureCode for the ingress intf.
         noFailure = tacTalkerFailureCode.noFailure
         if streamData.talkerFailureCode.get( talkerIngressIntf, noFailure ) != \
                noFailure:
            talkerStatePropagated = tacTalkerState.TalkerFailed
         else:
            talkerStatePropagated = talkerStateIngress
         if listenerEgressIntf in streamData.listenerState:
            listenerStateEgress = streamData.listenerState[ listenerEgressIntf ]
               
         streamModel.vlanId = streamData.vlanId
         streamModel.srClass = streamData.srClass
         streamModel.bandwidth = streamData.bandwidth
      else:
         streamModel.bandwidth = 0

      # Talker ingress information
      streamModel.talkerIngressIntf = talkerIngressIntf
      streamModel.talkerStateIngressIntf = talkerStateIngress
      streamModel.talkerStatePropagated = talkerStatePropagated
         
      # Listener egress information
      streamModel.listenerEgressIntf = listenerEgressIntf
      streamModel.listenerStateEgressIntf = listenerStateEgress
      
      streamModel.destinationMacAddr = streamData.macDa
      streamModel.bandwidthUnit = 'kbps'

      talkersEgress = {}
      for intf in streamData.talkerState:
         if intf == streamData.ingressIntf:
            continue
         talkerState = streamData.talkerState[ intf ]
         talkerInfo = streamModel.TalkerEgressPort( talkerStateEgress=talkerState )
         talkersEgress[ intf ] = talkerInfo
            
      listenersIngress = {}
      for intf in streamData.listenerStateOrig:
         if intf == streamData.ingressIntf:
            continue
         listenerStateIngress = streamData.listenerStateOrig[ intf ]
         listenerStatePropagated = streamData.listenerState.get( intf )
         listenerInfo = streamModel.ListenerIngressPort(
            listenerStateIngress=listenerStateIngress,
            listenerStatePropagated=listenerStatePropagated )
         listenersIngress[ intf ] = listenerInfo
         
      streamModel.talkerEgressInterfaces = talkersEgress
      streamModel.listenerIngressInterfaces = listenersIngress
      streams[ convertStreamIdToDisplay( sId ) ] = streamModel

   return MsrpStreamPropagation( streams=streams )

class MsrpStreamsPropagationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show msrp streams [ stream-id STREAM_ID ] propagation'
   data = {
      'msrp' : nodeShowMsrp,
      'streams' : matcherStreams,
      'stream-id' : matcherStreamId,
      'STREAM_ID' : matcherStreamPattern,
      'propagation' : 'Show MSRP stream propagation',
   }

   handler = showMsrpStreamPropagation
   cliModel = MsrpStreamPropagation

BasicCli.addShowCommandClass( MsrpStreamsPropagationCmd )

#--------------------------------------------------------------------------------
# show msrp mrp [ interfaces INTF ] [ talker-advertise ] [ talker-failed ]
#               [ listener ] [ domain ] [ detail ]
#--------------------------------------------------------------------------------
def showMrp( mode, args ):
   talkAdv = 'talker-advertise' in args
   talkFail = 'talker-failed' in args
   listener = 'listener' in args
   domain = 'domain' in args
   detail = 'detail' in args
   intfs = IntfCli.Intf.getAll( mode, args.get( 'INTF' ), None, EthIntfCli.EthIntf )

   # Display all types if no filter is specified
   allTypes = not any( [ talkAdv, talkFail, listener, domain ] )

   mrpModel = MrpDatabase()
   for intf in intfs:
      if intf.name not in mrpStatus.application[ msrpApp ].attrDatabase:
         continue
      intfDatabase = mrpStatus.application[ msrpApp ].attrDatabase[ intf.name ]
      mrpIntfModel = mrpModel.IntfDatabase()
      mrpModel.interfaces[ intf.name ] = mrpIntfModel
      
      for attrType in intfDatabase.intfTypeApplicant:
         # Filter on the basis of attrType
         if attrType == 1 and not allTypes and not talkAdv:
            continue
         elif attrType == 2 and not allTypes and not talkFail:
            continue
         elif attrType == 3 and not allTypes and not listener:
            continue
         elif attrType == 4 and not allTypes and not domain:
            continue

         intfAttrDatabase = intfDatabase.intfTypeApplicant[ attrType ]
         mrpAttrTypeModel = mrpIntfModel.AttrTypeDatabase()
         mrpIntfModel.attributeTypes[ attrType ] = mrpAttrTypeModel

         for mrpAttributeKey in intfAttrDatabase.mrpAttribute:
            # Filter on the basis of attrType
            intfMrpAttrDatabase = intfAttrDatabase.mrpAttribute[ mrpAttributeKey ]
            mrpAttrModel = mrpAttrTypeModel.Attribute()
            mrpAttrModel.applicantState = intfMrpAttrDatabase.applicantState
            mrpAttrModel.registrarState = intfMrpAttrDatabase.registrarState

            if attrType == 4:
               # Key will be srClassId for domain type
               key = intToSrClassId[ mrpAttributeKey ]
            else:
               # Key will be streamId for others
               key = convertStreamIdToDisplay( mrpAttributeKey )

            mrpAttrTypeModel.attributes[ key ] = mrpAttrModel
            
            if detail:
               firstValue = intfMrpAttrDatabase.attributeValue
               values = MsrpCliLib.getAttrs( firstValue, attrType )
               mrpAttrModel.value = str( values )

   return mrpModel

class MsrpMrpInterfacesIntf1Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show msrp mrp [ interfaces INTF ] [ talker-advertise ] '
                            '[ talker-failed ] [ listener ] [ domain ] [ detail ]' )
   data = {
      'msrp' : nodeShowMsrp,
      'mrp' : CliCommand.guardedKeyword( 'mrp', helpdesc='Show MRP information',
         guard=guardMsrp, hidden=True ),
      'interfaces' : matcherInterfaces,
      'INTF' : matcherIntfRange,
      'talker-advertise' : 'Show MRP talker advertise information',
      'talker-failed' : 'Show MRP talker failed information',
      'listener' : 'Show MRP listener information',
      'domain' : 'Show MRP domain information',
      'detail' : matcherDetail,
   }
   cliModel = MrpDatabase
   handler = showMrp

BasicCli.addShowCommandClass( MsrpMrpInterfacesIntf1Cmd )

class MsrpIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( 'MsrpIntfJanitor: interface', self.intf_.name, 'going away' )
      intfEnable = msrpConfig.intfEnable.get( self.intf_.name )
      if intfEnable is not None:
         del msrpConfig.intfEnable[ self.intf_.name ]

      leaveAllIntfs = mrpConfig.intfLeaveAllTime.keys()
      for intf in leaveAllIntfs:
         del mrpConfig.intfLeaveAllTime[ intf ]
         
      leaveIntfs = mrpConfig.intfLeaveTime.keys()
      for intf in leaveIntfs:
         del mrpConfig.intfLeaveTime[ intf ]

#--------------------------------------------------------------------------------
# show msrp streams count
#--------------------------------------------------------------------------------
class MsrpStreamsCountCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show msrp streams count'
   data = {
      'msrp' : nodeShowMsrp,
      'streams' : matcherStreams,
      'count' : 'Number of established AVB Streams',
   }
   cliModel = MsrpStreamCount

   @staticmethod
   def handler( mode, args ):
      msrpModel = MsrpStreamCount()
      msrpModel.activeAvbStreams = len(
         [ stream for stream in msrpStatus.registeredStreams.values()
           if stream.ingressIntf and stream.bandwidth ] )
      return msrpModel

BasicCli.addShowCommandClass( MsrpStreamsCountCmd )

#--------------------------------------------------------------------------------
# show msrp counters [ interfaces INTF ]
#--------------------------------------------------------------------------------
def showMsrpCounters( mode, args ):
   intf = args.get( 'INTF' )
   mod = None
   msrpModel = MsrpCounters()
   # Interface information
   intfs = IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthIntf )
   
   for intf in intfs:
      if not intf.name.startswith( 'Ethernet' ):
         continue
      msrpIntfModel = msrpModel.Intf()
      msrpModel.interfaces[ intf.name ] = msrpIntfModel
      if intf.name not in msrpStatus.intfStatus:
         msrpIntfModel.operState = 'Disabled'
         continue
      else:
         intfStatus = msrpStatus.intfStatus[ intf.name ]
         if ( intfStatus.linkStatus == 'linkUp' and 
              intfStatus.portState == 'portForwarding' ):
            msrpIntfModel.operState = 'Active'
            txCounters, rxCounters = interfaceCountersValue( intfStatus )
            msrpIntfModel.rxCounters = rxCounters
            msrpIntfModel.txCounters = txCounters
         else:
            msrpIntfModel.operState = 'Disabled'
   
   return msrpModel

class MsrpCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show msrp counters [ interfaces INTF ]'
   data = {
      'msrp' : nodeShowMsrp,
      'counters' : matcherCounters,
      'interfaces' : matcherInterfaces,
      'INTF' : matcherIntfRange
   }

   handler = showMsrpCounters
   cliModel = MsrpCounters

BasicCli.addShowCommandClass( MsrpCountersCmd )

#--------------------------------------------------------------------------------
# clear msrp counters [ interfaces INTF ]
#--------------------------------------------------------------------------------
class ClearMsrpCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear msrp counters [ interfaces INTF ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'msrp' : nodeMsrp,
      'counters' : matcherCounters,
      'interfaces' : matcherInterfaces,
      'INTF' : matcherIntfRange,
   }

   @staticmethod
   def handler( mode, args ):
      # Interface information
      intf = args.get( 'INTF' )
      mod = None
      intfs = IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthIntf )

      checkpoint = msrpCounterCheckpoint
      for intf in intfs:
         if( not intf.name.startswith( 'Ethernet' ) or
             intf.name not in msrpStatus.intfStatus ):
            continue
         intfStatus = msrpStatus.intfStatus[ intf.name ]
         if ( intfStatus.linkStatus != 'linkUp' or 
              intfStatus.portState != 'portForwarding' ):
            continue
         snapshot = checkpoint.interfaceCounterCheckpoint.get( intfStatus.intfId )
         if snapshot is None:
            snapshot = checkpoint.interfaceCounterCheckpoint.newMember(
               intfStatus.intfId )
         snapshot.timestamp = Tac.now()
         # Counters names are same for both rx and tx .
         # So, assigning in a single for loop only.
         for counterName in intfStatus.rxCounters:
            snapshot.rxCounters[ counterName ] = intfStatus.rxCounters[ counterName ]
            snapshot.txCounters[ counterName ] = intfStatus.txCounters[ counterName ]

BasicCli.EnableMode.addCommandClass( ClearMsrpCountersCmd )

#--------------------------------------------------------------------------------
# Register to show tech-support
#--------------------------------------------------------------------------------
def msrpSupported():
   return msrpHwStatus.msrpSupported

def _showTechCmds():
   if msrpSupported():
      return [ 'show msrp', 'show msrp interfaces streams' ]
   return []

TechSupportCli.registerShowTechSupportCmdCallback(
   '2014-04-02 00:00:00', _showTechCmds )

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
def Plugin( entityManager ):
   global msrpCliConfig, msrpConfig, msrpStatus, mrpConfig, mrpStatus
   global msrpHwStatus, msrpCounterCheckpoint
   global entMan

   entMan = entityManager
   msrpCliConfig = ConfigMount.mount( entityManager, 'msrp/cliConfig',
                                      'Msrp::CliConfig', 'w' )
   msrpConfig = ConfigMount.mount( entityManager, 'msrp/config',
                                   'Msrp::Config', 'w' )
   msrpStatus = LazyMount.mount( entityManager, 'msrp/status',
                                 'Msrp::Status', 'r' )
   mrpConfig = ConfigMount.mount( entityManager, 'mrp/config',
                                  'Mrp::MrpConfig', 'w' )
   mrpStatus = LazyMount.mount( entityManager, 'mrp/status',
                                 'Mrp::MrpStatus', 'r' )
   msrpHwStatus = LazyMount.mount( entityManager, 'msrp/hwStatus',
                                   "Msrp::HwStatus", "r" )
   msrpCounterCheckpoint = LazyMount.mount( entityManager, 
                                            'msrp/counterCheckpoint',
                                            'Msrp::CounterCheckpoint', 'w' )

   IntfCli.Intf.registerDependentClass( MsrpIntfJanitor )
