# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.MsdpCli as MsdpCli
import CliPlugin.RouteMapCli as RouteMapCli
import CliToken.Ip
from McastCommonCliLib import vrfFromMode
import Tac

def _printNonexistentMeshGroupError( mode, name ):
   mode.addError( "Mesh Group is not configured in vrf %s: %s" % \
                     ( vrfFromMode( mode ), name ) )

#--------------------------------------------------------------------------------
# [ no | default ] default-peer [ prefix-list LISTNAME ]
#
# Deprecated cmd:
# [ no | default ] ip msdp default-peer PEER [ prefix-list LISTNAME ]
#--------------------------------------------------------------------------------
def defaultPeer( ip, prefix ):
   return Tac.newInstance( "Routing::Msdp::DefaultPeer", ip,
         prefix if prefix else "" )

def addDefaultPeer( mode, peerIp, prefix=None ):
   vrfName = vrfFromMode( mode )
   # Create a temporary dP
   dP = defaultPeer( peerIp, prefix )

   # Check and add if it doesn't exist
   exists = False
   for key in MsdpCli.msdpConfig.vrfConfig[ vrfName ].defaultPeer:
      d = MsdpCli.msdpConfig.vrfConfig[ vrfName ].defaultPeer.get( key )
      if d.ip == peerIp:
         exists = True
         MsdpCli.msdpConfig.vrfConfig[ vrfName ].defaultPeer[ key ] = dP

   if not exists:
      MsdpCli.msdpConfig.vrfConfig[ vrfName ].defaultPeer.enq( dP )

def removeDefaultPeer( mode, peerIp ):
   vrfName = vrfFromMode( mode )
   removeKey = None
   for key in MsdpCli.msdpConfig.vrfConfig[ vrfName ].defaultPeer:
      d = MsdpCli.msdpConfig.vrfConfig[ vrfName ].defaultPeer.get( key )
      if d.ip == peerIp:
         removeKey = key
         break
   if removeKey is not None:
      del MsdpCli.msdpConfig.vrfConfig[ vrfName ].defaultPeer[ removeKey ]

defaultPeerMatcher = CliMatcher.KeywordMatcher(
      'default-peer', helpdesc='MSDP default peer' )
prefixListMatcher = CliMatcher.KeywordMatcher(
      'prefix-list', helpdesc='IP Prefix list to apply for this default Peer' )

class DefaultPeerCmd( CliCommand.CliCommandClass ):
   syntax = 'default-peer [ prefix-list LISTNAME ]'
   noOrDefaultSyntax = 'default-peer ...'
   data = {
      'default-peer': defaultPeerMatcher,
      'prefix-list': prefixListMatcher,
      'LISTNAME': RouteMapCli.prefixListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      addDefaultPeer( mode, mode.peer, args.get( 'LISTNAME' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      removeDefaultPeer( mode, mode.peer )

MsdpCli.RouterMsdpPeerMode.addCommandClass( DefaultPeerCmd )

class IpMsdpDefaultPeerPeerCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp default-peer PEER [ prefix-list LISTNAME ]'
   noOrDefaultSyntax = 'ip msdp default-peer PEER ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': MsdpCli.msdpNode,
      'default-peer':
         CliCommand.Node( defaultPeerMatcher,
                          deprecatedByCmd='default-peer in msdp peer mode' ),
      'PEER': MsdpCli.msdpPeerAddrMatcher,
      'prefix-list': prefixListMatcher,
      'LISTNAME': RouteMapCli.prefixListNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      addDefaultPeer( mode, args[ 'PEER' ], args.get( 'LISTNAME' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      removeDefaultPeer( mode, args[ 'PEER' ] )

MsdpCli.RouterMsdpSharedModelet.addCommandClass( IpMsdpDefaultPeerPeerCmd )

#--------------------------------------------------------------------------------
# description DESCRIPTION
#
# Deprecated cmd:
# [ no | default ] ip msdp description PEER DESCRIPTION
#--------------------------------------------------------------------------------
def descriptionPeer( mode, peerIp, desc ):
   vrfName = vrfFromMode( mode )
   c = MsdpCli.peerConfig( peerIp, vrfName=vrfName )
   if c:
      c.description = desc
   else:
      MsdpCli.printNonexistentPeerError( mode, peerIp )

matcherDescription = CliMatcher.KeywordMatcher( 'description',
      helpdesc='Add a description for this peer to the configuration' )
matcherDescriptionString = CliMatcher.StringMatcher( helpname='DESCRIPTION',
      helpdesc='Description of the Peer' )

class DescriptionStringCmd( CliCommand.CliCommandClass ):
   syntax = 'description DESCRIPTION'
   noOrDefaultSyntax = 'description ...'
   data = {
      'description': matcherDescription,
      'DESCRIPTION': matcherDescriptionString,
   }

   @staticmethod
   def handler( mode, args ):
      descriptionPeer( mode, mode.peer, args[ 'DESCRIPTION' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      descriptionPeer( mode, mode.peer, '' )

MsdpCli.RouterMsdpPeerMode.addCommandClass( DescriptionStringCmd )

class IpMsdpDescriptionPeerCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp description PEER DESCRIPTION'
   noOrDefaultSyntax = 'ip msdp description PEER ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': MsdpCli.msdpNode,
      'description':
         CliCommand.Node( matcherDescription,
                          deprecatedByCmd='description in router msdp peer mode' ),
      'PEER': MsdpCli.msdpPeerAddrMatcher,
      'DESCRIPTION': matcherDescriptionString,
   }

   @staticmethod
   def handler( mode, args ):
      descriptionPeer( mode, args[ 'PEER' ], args[ 'DESCRIPTION' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      descriptionPeer( mode, args[ 'PEER' ], '' )

MsdpCli.RouterMsdpSharedModelet.addCommandClass( IpMsdpDescriptionPeerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] disabled
#
# Deprecated cmd:
# [ no | default ] ip msdp shutdown PEER
#--------------------------------------------------------------------------------
def shutdownPeer( mode, peerIp, shutdown ):
   vrfName = vrfFromMode( mode )
   c = MsdpCli.peerConfig( peerIp, vrfName=vrfName )
   if c:
      c.shutdown = shutdown
   else:
      MsdpCli.printNonexistentPeerError( mode, peerIp )

class DisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled': 'Disable the peer but keep the configuration around',
   }

   @staticmethod
   def handler( mode, args ):
      shutdownPeer( mode, mode.peer, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      shutdownPeer( mode, mode.peer, False )

MsdpCli.RouterMsdpPeerMode.addCommandClass( DisabledCmd )

class IpMsdpShutdownPeerCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp shutdown PEER'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': MsdpCli.msdpNode,
      'shutdown':
         CliCommand.Node(
            CliMatcher.KeywordMatcher( 'shutdown',
               helpdesc='Shutdown the peer but keep the configuration around' ),
            deprecatedByCmd='disabled in the msdp peer mode' ),
      'PEER': MsdpCli.msdpPeerAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      shutdownPeer( mode, args[ 'PEER' ], True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      shutdownPeer( mode, args[ 'PEER' ], False )

MsdpCli.RouterMsdpSharedModelet.addCommandClass( IpMsdpShutdownPeerCmd )

#--------------------------------------------------------------------------------
# keepalive KEEPALIVE HOLDTIME
#
# Deprecated cmd:
# [ no | default ] ip msdp keepalive PEER KEEPALIVE HOLDTIME
#--------------------------------------------------------------------------------
def setKeepalive( mode, peerIp, keepalive=None, holdtime=None ):
   vrfName = vrfFromMode( mode )
   mpc = MsdpCli.peerConfig( peerIp, vrfName=vrfName )
   if mpc:
      if keepalive is None:
         keepalive = mpc.keepaliveDefault
      if holdtime is None:
         holdtime = mpc.holdtimeDefault
      if keepalive > holdtime:
         mode.addError( "Keepalive must be shorter than holdtime" )
         return
      mpc.keepalive = keepalive
      mpc.holdtime = holdtime
   else:
      MsdpCli.printNonexistentPeerError( mode, peerIp )

matcherKeepalive = CliMatcher.KeywordMatcher( 'keepalive',
      helpdesc='Set MSDP keepalive and hold times' )
matcherKeepaliveTime = CliMatcher.IntegerMatcher( 1, 65535,
      helpdesc='Keepalive time (in seconds)' )
matcherHoldTime = CliMatcher.IntegerMatcher( 1, 65535,
      helpdesc='Hold time (in seconds)' )

class KeepaliveHoldtimeCmd( CliCommand.CliCommandClass ):
   syntax = 'keepalive KEEPALIVE HOLDTIME'
   noOrDefaultSyntax = 'keepalive ...'
   data = {
      'keepalive': matcherKeepalive,
      'KEEPALIVE': matcherKeepaliveTime,
      'HOLDTIME': matcherHoldTime,
   }

   @staticmethod
   def handler( mode, args ):
      setKeepalive( mode, mode.peer, args[ 'KEEPALIVE' ], args[ 'HOLDTIME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setKeepalive( mode, mode.peer, None, None )

MsdpCli.RouterMsdpPeerMode.addCommandClass( KeepaliveHoldtimeCmd )

class IpMsdpKeepalivePeerKeepaliveHoldtimeCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp keepalive PEER KEEPALIVE HOLDTIME'
   noOrDefaultSyntax = 'ip msdp keepalive PEER ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': MsdpCli.msdpNode,
      'keepalive':
         CliCommand.Node( matcherKeepalive,
            deprecatedByCmd=( 'keepalive <keepalive> <holdtime> in router msdp peer '
                              'mode' ) ),
      'PEER': MsdpCli.msdpPeerAddrMatcher,
      'KEEPALIVE': matcherKeepaliveTime,
      'HOLDTIME': matcherHoldTime,
   }

   @staticmethod
   def handler( mode, args ):
      setKeepalive( mode, args[ 'PEER' ], args[ 'KEEPALIVE' ], args[ 'HOLDTIME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setKeepalive( mode, args[ 'PEER' ], None, None )

MsdpCli.RouterMsdpSharedModelet.addCommandClass(
   IpMsdpKeepalivePeerKeepaliveHoldtimeCmd )

#--------------------------------------------------------------------------------
# local-interface INTF
#--------------------------------------------------------------------------------
def setLocalIntf( mode, localIntf ):
   vrfName = vrfFromMode( mode )
   pc = MsdpCli.msdpConfig.vrfConfig[ vrfName ].peerConfig.get( mode.peer )
   if localIntf is None:
      pc.connSrc = ''
   else:
      pc.connSrc = localIntf.name

class LocalInterfaceIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'local-interface INTF'
   noOrDefaultSyntax = 'local-interface ...'
   data = {
      'local-interface': 'MSDP local interface',
      'INTF': IntfCli.Intf.matcher,
   }

   @staticmethod
   def handler( mode, args ):
      setLocalIntf( mode, args[ 'INTF' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setLocalIntf( mode, None )

MsdpCli.RouterMsdpPeerMode.addCommandClass( LocalInterfaceIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mesh-group MESHGROUP
#
# Deprecated cmd:
# [ no | default ] ip msdp mesh-group MESHGROUP PEER
#--------------------------------------------------------------------------------
def setMeshGroup( mode, peerIp, meshGroup ):
   vrfName = vrfFromMode( mode )
   c = MsdpCli.msdpConfig.vrfConfig[ vrfName ].meshGroup
   if not c.get( meshGroup ):
      mg = MsdpCli.msdpConfig.vrfConfig[ vrfName ].meshGroup.newMember( meshGroup )
   else:
      mg = c.get( meshGroup )

   if not mg.member.get( peerIp ):
      mg.member[ peerIp ] = True

def noMeshGroup( mode, peerIp, meshGroup ):
   vrfName = vrfFromMode( mode )
   c = MsdpCli.msdpConfig.vrfConfig[ vrfName ].meshGroup
   if c:
      mg = c.get( meshGroup )
      if mg:
         if peerIp is None:
            MsdpCli.safeDel( c, meshGroup )
         else:
            # Delete the particular mesh group member.
            MsdpCli.safeDel( mg.member, peerIp )
            if not mg.member:
               MsdpCli.safeDel( c, meshGroup )
      else:
         _printNonexistentMeshGroupError( mode, meshGroup )
   else:
      _printNonexistentMeshGroupError( mode, meshGroup )

matcherMeshGroup = CliMatcher.KeywordMatcher( 'mesh-group',
      helpdesc='MSDP mesh group' )
matcherMeshGroupName = CliMatcher.PatternMatcher( pattern='[-A-Za-z0-9_]+',
      helpdesc='Name of the mesh-group', helpname='WORD' )

class MeshGroupMeshgroupCmd( CliCommand.CliCommandClass ):
   syntax = 'mesh-group MESHGROUP'
   noOrDefaultSyntax = 'mesh-group MESHGROUP'
   data = {
      'mesh-group': matcherMeshGroup,
      'MESHGROUP': matcherMeshGroupName,
   }

   @staticmethod
   def handler( mode, args ):
      setMeshGroup( mode, mode.peer, args[ 'MESHGROUP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noMeshGroup( mode, mode.peer, args[ 'MESHGROUP' ] )

MsdpCli.RouterMsdpPeerMode.addCommandClass( MeshGroupMeshgroupCmd )

class IpMsdpMeshGroupPeerCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp mesh-group MESHGROUP PEER'
   noOrDefaultSyntax = 'ip msdp mesh-group MESHGROUP [ PEER ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': MsdpCli.msdpNode,
      'mesh-group':
         CliCommand.Node( matcherMeshGroup,
            deprecatedByCmd='mesh-group under router msdp peer mode' ),
      'MESHGROUP': matcherMeshGroupName,
      'PEER': MsdpCli.msdpPeerAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      setMeshGroup( mode, args[ 'PEER' ], args[ 'MESHGROUP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      peer = args.get( 'PEER' )
      if peer is None:
         noMeshGroup( mode, None, args[ 'MESHGROUP' ] )
      else:
         noMeshGroup( mode, peer, args[ 'MESHGROUP' ] )

MsdpCli.RouterMsdpSharedModelet.addCommandClass(
   IpMsdpMeshGroupPeerCmd )

class NoMeshGroupMeshgroupCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'mesh-group MESHGROUP'
   data = {
      'mesh-group': matcherMeshGroup,
      'MESHGROUP': matcherMeshGroupName,
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noMeshGroup( mode, None, args[ 'MESHGROUP' ] )

MsdpCli.RouterMsdpBaseMode.addCommandClass( NoMeshGroupMeshgroupCmd )
MsdpCli.RouterMsdpVrfMode.addCommandClass( NoMeshGroupMeshgroupCmd )

#--------------------------------------------------------------------------------
# sa-filter ( in | out ) list ACLNAME
#
# Deprecated cmd:
# [ no | default ] ip msdp sa-filter ( in | out ) PEER list ACLNAME
#--------------------------------------------------------------------------------
def saFilterInOrOutAcl( mode, peerIp, inOrOut, aclName ):
   vrfName = vrfFromMode( mode )
   c = MsdpCli.peerConfig( peerIp, vrfName=vrfName )
   if c:
      if inOrOut == 'in':
         c.saFilterIn = aclName
      else:
         c.saFilterOut = aclName
   else:
      MsdpCli.printNonexistentPeerError( mode, peerIp )

matcherSaFilter = CliMatcher.KeywordMatcher( 'sa-filter',
      helpdesc='Change the SA filter' )
matcherDirection = CliMatcher.EnumMatcher( {
   'in': 'Change the SA Filter In',
   'out': 'Change the SA Filter Out' } )
matcherList = CliMatcher.KeywordMatcher( 'list',
      helpdesc='Specify an access-list' )

class SaFilterInOrOutListAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'sa-filter DIRECTION list ACLNAME'
   noOrDefaultSyntax = 'sa-filter DIRECTION ...'
   data = {
      'sa-filter': matcherSaFilter,
      'DIRECTION': matcherDirection,
      'list': matcherList,
      'ACLNAME': AclCli.ipAclNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      saFilterInOrOutAcl( mode, mode.peer, args[ 'DIRECTION' ], args[ 'ACLNAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      saFilterInOrOutAcl( mode, mode.peer, args[ 'DIRECTION' ], '' )

MsdpCli.RouterMsdpPeerMode.addCommandClass( SaFilterInOrOutListAclnameCmd )

class IpMsdpSAFilterInOutPeerListAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp sa-filter DIRECTION PEER list ACLNAME'
   noOrDefaultSyntax = 'ip msdp sa-filter DIRECTION PEER ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': MsdpCli.msdpNode,
      'DIRECTION': matcherDirection,
      'sa-filter':
         CliCommand.Node( matcherSaFilter,
            deprecatedByCmd='sa-filter under router-msdp-peer mode' ),
      'PEER': MsdpCli.msdpPeerAddrMatcher,
      'list': matcherList,
      'ACLNAME': AclCli.ipAclNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      saFilterInOrOutAcl( mode, args[ 'PEER' ], args[ 'DIRECTION' ],
            args[ 'ACLNAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      saFilterInOrOutAcl( mode, args[ 'PEER' ], args[ 'DIRECTION' ], '' )

MsdpCli.RouterMsdpSharedModelet.addCommandClass(
   IpMsdpSAFilterInOutPeerListAclnameCmd )

#--------------------------------------------------------------------------------
# sa-limit LIMIT
#
# Deprecated cmd:
# [ no | default ] ip msdp sa-limit PEER LIMIT
#--------------------------------------------------------------------------------
def setPeerSALimit( mode, peerIp, limit ):
   vrfName = vrfFromMode( mode )
   c = MsdpCli.peerConfig( peerIp, vrfName=vrfName )
   if c:
      if limit is None:
         limit = c.peerSALimitDefault
      c.peerSALimit = limit
   else:
      MsdpCli.printNonexistentPeerError( mode, peerIp )

matcherSaLimit = CliMatcher.KeywordMatcher( 'sa-limit',
      helpdesc='Limit the number of SAs to be cached from this peer' )
matcherLimit = CliMatcher.IntegerMatcher( 0, 40000,
      helpdesc='Limit for SAs from this peer' )

class SaLimitLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'sa-limit LIMIT'
   noOrDefaultSyntax = 'sa-limit ...'
   data = {
      'sa-limit': matcherSaLimit,
      'LIMIT': matcherLimit,
   }

   @staticmethod
   def handler( mode, args ):
      setPeerSALimit( mode, mode.peer, args[ 'LIMIT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setPeerSALimit( mode, mode.peer, None )

MsdpCli.RouterMsdpPeerMode.addCommandClass( SaLimitLimitCmd )

class IpMsdpSALimitPeerLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp sa-limit PEER LIMIT'
   noOrDefaultSyntax = 'ip msdp sa-limit PEER ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': MsdpCli.msdpNode,
      'sa-limit':
         CliCommand.Node( matcherSaLimit,
            deprecatedByCmd='sa-limit under router msdp peer mode' ),
      'PEER': MsdpCli.msdpPeerAddrMatcher,
      'LIMIT': matcherLimit,
   }

   @staticmethod
   def handler( mode, args ):
      setPeerSALimit( mode, args[ 'PEER' ], args[ 'LIMIT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setPeerSALimit( mode, args[ 'PEER' ], None )

MsdpCli.RouterMsdpSharedModelet.addCommandClass( IpMsdpSALimitPeerLimitCmd )
