#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Arnet
import CliCommand
import CliMatcher
from CliPlugin.IntfCli import Intf
from CliPlugin.IraIpRouteCliLib import isValidPrefixWithError
from CliPlugin.RouterMulticastCliLib import AddressFamily
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.IraIpCli as IraIpCli
import CliPlugin.MsdpCli as MsdpCli
import CliPlugin.VrfCli as VrfCli
import CliToken.Ip
from McastCommonCliLib import vrfFromMode

MAX_REJECTED_SA_CACHE_SIZE = 40000

#-------------------------------------------------------------------------------
# [ no | default ] connection retry interval PERIOD
#
# Deprecated cmd:
# [ no | default ] ip msdp timer PERIOD
#--------------------------------------------------------------------------------
def setConnRetryPeriod( mode, args ):
   vrfName = vrfFromMode( mode )
   period = args.get( 'PERIOD',
         MsdpCli.msdpConfig.vrfConfig[ vrfName ].connRetryPeriodDefault )
   MsdpCli.msdpConfig.vrfConfig[ vrfName ].connRetryPeriod = period

connRetryPeriodMatcher = CliMatcher.IntegerMatcher(
      1, 65535, helpdesc='Connect Retry time (in seconds)' )

class ConnectionRetryIntervalPeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'connection retry interval PERIOD'
   noOrDefaultSyntax = 'connection retry interval ...'
   data = {
      'connection': 'Change the connection retry period for MSDP',
      'retry': 'Change the connection retry period for MSDP',
      'interval': 'Change the connection retry period for MSDP',
      'PERIOD': connRetryPeriodMatcher,
   }
   handler = setConnRetryPeriod
   noOrDefaultHandler = setConnRetryPeriod

MsdpCli.RouterMsdpBaseMode.addCommandClass( ConnectionRetryIntervalPeriodCmd )
MsdpCli.RouterMsdpVrfMode.addCommandClass( ConnectionRetryIntervalPeriodCmd )

class IpMsdpTimerPeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp timer PERIOD'
   noOrDefaultSyntax = 'ip msdp timer ...'
   data = {
         'ip': CliToken.Ip.ipMatcherForConfig,
         'msdp': MsdpCli.msdpNode,
         'timer':
            CliCommand.Node(
               CliMatcher.KeywordMatcher(
                  'timer', helpdesc='Change the connection retry period for MSDP' ),
               deprecatedByCmd=( 'connection retry interval <period> in router '
                                 'msdp mode' ) ),
         'PERIOD': connRetryPeriodMatcher
   }
   handler = setConnRetryPeriod
   noOrDefaultHandler = setConnRetryPeriod

MsdpCli.RouterMsdpSharedModelet.addCommandClass( IpMsdpTimerPeriodCmd )

#--------------------------------------------------------------------------------
# [ no | default ] forward register-packets
#
# Deprecated cmd:
# [ no | default ] ip msdp forward register-packets
#--------------------------------------------------------------------------------
def msdpForward( mode, args ):
   vrfName = vrfFromMode( mode )
   MsdpCli.msdpConfig.vrfConfig[ vrfName ].encapDataPkts = True

def noMsdpForward( mode, args ):
   vrfName = vrfFromMode( mode )
   MsdpCli.msdpConfig.vrfConfig[ vrfName ].encapDataPkts = False

forwardMatcher = CliMatcher.KeywordMatcher(
      'forward', helpdesc='Enable MSDP to forward multicast register packets' )
registerPacketsMatcher = CliMatcher.KeywordMatcher(
      'register-packets',
      helpdesc='Enable MSDP to forward multicast register packets' )

class ForwardRegisterPacketsCmd( CliCommand.CliCommandClass ):
   syntax = 'forward register-packets'
   noOrDefaultSyntax = syntax
   data = {
      'forward': forwardMatcher,
      'register-packets': registerPacketsMatcher
   }
   handler = msdpForward
   noOrDefaultHandler = noMsdpForward

MsdpCli.RouterMsdpBaseMode.addCommandClass( ForwardRegisterPacketsCmd )
MsdpCli.RouterMsdpVrfMode.addCommandClass( ForwardRegisterPacketsCmd )

class IpMsdpForwardRegisterPacketsCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp forward register-packets'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': MsdpCli.msdpNode,
      'forward':
         CliCommand.Node(
            forwardMatcher ),
      'register-packets': registerPacketsMatcher
   }
   handler = msdpForward
   noOrDefaultHandler = noMsdpForward

MsdpCli.RouterMsdpSharedModelet.addCommandClass( IpMsdpForwardRegisterPacketsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] group-limit LIMIT source SOURCE_PREFIX
#
# Deprecated cmd:
# [ no | default ] ip msdp group-limit LIMIT source SOURCE_PREFIX
#--------------------------------------------------------------------------------
def setGroupSALimit( mode, args ):
   sourcePrefixStr = args[ 'SOURCE_PREFIX' ]
   limit = args[ 'LIMIT' ]
   vrfName = vrfFromMode( mode )
   if not isValidPrefixWithError( mode, sourcePrefixStr ):
      return
   MsdpCli.msdpConfig.vrfConfig[ vrfName ].sourceSALimit[ \
      Arnet.Prefix( sourcePrefixStr ) ] = limit

def noSetGroupSALimit( mode, args ):
   sourcePrefixStr = args[ 'SOURCE_PREFIX' ]
   vrfName = vrfFromMode( mode )
   if not isValidPrefixWithError( mode, sourcePrefixStr ):
      return
   del MsdpCli.msdpConfig.vrfConfig[ vrfName ].sourceSALimit[ \
      Arnet.Prefix( sourcePrefixStr ) ]

groupLimitMatcher = CliMatcher.KeywordMatcher(
      'group-limit',
      helpdesc='Limit the number of SAs to be cached matching a source address '
               'prefix' )

limitMatcher = CliMatcher.IntegerMatcher(
      0, 40000, helpdesc='Limit for SAs matching the source address prefix' )

sourceMatcher = CliMatcher.KeywordMatcher(
      'source', helpdesc='Source address prefix' )

sourcePrefixExpr = IpAddrMatcher.ipPrefixExpr(
      'Source prefix', 'Source prefix mask', 'Source prefix with prefix len' )

class GroupLimitLimitSourceCmd( CliCommand.CliCommandClass ):
   syntax = 'group-limit LIMIT source SOURCE_PREFIX'
   noOrDefaultSyntax = syntax
   data = {
      'group-limit': groupLimitMatcher,
      'LIMIT': limitMatcher,
      'source': sourceMatcher,
      'SOURCE_PREFIX': sourcePrefixExpr
   }
   handler = setGroupSALimit
   noOrDefaultHandler = noSetGroupSALimit

MsdpCli.RouterMsdpBaseMode.addCommandClass( GroupLimitLimitSourceCmd )
MsdpCli.RouterMsdpVrfMode.addCommandClass( GroupLimitLimitSourceCmd )

class IpMsdpGroupLimitLimitSourceCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp group-limit LIMIT source SOURCE_PREFIX'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': MsdpCli.msdpNode,
      'group-limit':
         CliCommand.Node(
            groupLimitMatcher,
            deprecatedByCmd='group-limit in router msdp mode' ),
      'LIMIT': limitMatcher,
      'source': sourceMatcher,
      'SOURCE_PREFIX': sourcePrefixExpr
   }
   handler = setGroupSALimit
   noOrDefaultHandler = noSetGroupSALimit

MsdpCli.RouterMsdpSharedModelet.addCommandClass( IpMsdpGroupLimitLimitSourceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] originator-id local-interface INTF
#
# Deprecated cmd:
# [ no | default ] ip msdp originator-id INTF
#--------------------------------------------------------------------------------
def confOriginatorId( mode, args ):
   originatorId = args[ 'INTF' ]
   vrfName = vrfFromMode( mode )
   MsdpCli.msdpConfig.vrfConfig[ vrfName ].originatorId = originatorId.name

def noConfOriginatorId( mode, args ):
   vrfName = vrfFromMode( mode )
   MsdpCli.msdpConfig.vrfConfig[ vrfName ].originatorId = ""

originatorIdMatcher = CliMatcher.KeywordMatcher(
      'originator-id',
      'Configure the interface whose IP Address is advertised as the RP' )

localInterfaceMatcher = CliMatcher.KeywordMatcher(
      'local-interface',
      'Configure the interface whose IP Address is advertised as the RP' )

class OriginatorIdLocalInterfaceIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'originator-id local-interface INTF'
   noOrDefaultSyntax = 'originator-id local-interface ...'
   data = {
      'originator-id': originatorIdMatcher,
      'local-interface': localInterfaceMatcher,
      'INTF': Intf.matcher,
   }
   handler = confOriginatorId
   noOrDefaultHandler = noConfOriginatorId

MsdpCli.RouterMsdpBaseMode.addCommandClass( OriginatorIdLocalInterfaceIntfCmd )
MsdpCli.RouterMsdpVrfMode.addCommandClass( OriginatorIdLocalInterfaceIntfCmd )

class IpMsdpOriginatorIdIntfCmd(  CliCommand.CliCommandClass ):
   syntax = 'ip msdp originator-id INTF'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': MsdpCli.msdpNode,
      'originator-id':
         CliCommand.Node(
            originatorIdMatcher,
            deprecatedByCmd='originator-id under router msdp mode' ),
      'INTF': Intf.matcher,
   }
   handler = confOriginatorId
   noOrDefaultHandler = noConfOriginatorId

MsdpCli.RouterMsdpSharedModelet.addCommandClass( IpMsdpOriginatorIdIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] peer PEERADDR
#
# Deprecated cmd:
# [ no | default ] ip msdp peer PEER [ connect-source INTF ]
#--------------------------------------------------------------------------------
def _getVrfNameFromMode( mode ):
   return getattr( mode, 'vrfName', VrfCli.DEFAULT_VRF )

peerMatcher = CliMatcher.KeywordMatcher( 'peer', 'Configure MSDP peer' )

class PeerPeeraddrCmd( CliCommand.CliCommandClass ):
   syntax = 'peer PEERADDR'
   noOrDefaultSyntax = syntax
   data = {
      'peer': peerMatcher,
      'PEERADDR': MsdpCli.msdpPeerAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      peerAddr = args[ 'PEERADDR' ]
      vrfName = _getVrfNameFromMode( mode )
      if not peerAddr in MsdpCli.msdpConfig.vrfConfig[ vrfName ].peerConfig:
         MsdpCli.msdpConfig.vrfConfig[ vrfName ].newPeerConfig( peerAddr, "" )
      childMode = mode.childMode(
            MsdpCli.RouterMsdpPeerMode, vrfName=vrfName, peer=peerAddr )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      peerAddr = args[ 'PEERADDR' ]
      vrfName = _getVrfNameFromMode( mode )
      vrfConfig = MsdpCli.config( af=AddressFamily.ipv4, vrfName=vrfName )
      del vrfConfig.peerConfig[ peerAddr ]

MsdpCli.RouterMsdpBaseMode.addCommandClass( PeerPeeraddrCmd )
MsdpCli.RouterMsdpVrfMode.addCommandClass( PeerPeeraddrCmd )

class IpMsdpPeerConnectSourceIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp peer PEER [ connect-source INTF ]'
   noOrDefaultSyntax = 'ip msdp peer PEER ...'
   data = {
         'ip': CliToken.Ip.ipMatcherForConfig,
         'msdp': MsdpCli.msdpNode,
         'peer':
            CliCommand.Node(
               peerMatcher,
               deprecatedByCmd='peer in router msdp mode' ),
         'PEER': MsdpCli.msdpPeerAddrMatcher,
         'connect-source': 'MSDP connection source interface',
         'INTF': Intf.matcher,
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = vrfFromMode( mode )
      peerIp = args[ 'PEER' ]
      connSrc = args.get( 'INTF' )
      MsdpCli.peerConfig( peerIp, create=True, connSrc=connSrc, vrfName=vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = vrfFromMode( mode )
      peerIp = args[ 'PEER' ]
      MsdpCli.safeDel( MsdpCli.msdpConfig.vrfConfig[ vrfName ].peerConfig, peerIp ) 

MsdpCli.RouterMsdpSharedModelet.addCommandClass( IpMsdpPeerConnectSourceIntfCmd )
#--------------------------------------------------------------------------------
# [ no | default ] rejected-limit LIMIT
#
# Deprecated cmd:
# [ no | default ] ip msdp rejected-limit LIMIT
#--------------------------------------------------------------------------------
def setRejectedSALimit( mode, args ):
   vrfName = vrfFromMode( mode )
   limit = args.get( 'LIMIT',
         MsdpCli.msdpConfig.vrfConfig[ vrfName ].rejectedSALimitDefault )
   MsdpCli.msdpConfig.vrfConfig[ vrfName ].rejectedSALimit = limit

rejectedLimitMatcher = CliMatcher.KeywordMatcher(
      'rejected-limit', 'Limit the number of SAs in the rejected SA cache' )

limitMatcher = CliMatcher.IntegerMatcher(
      0, MAX_REJECTED_SA_CACHE_SIZE,
      helpdesc='Limit for SAs in the rejected SA cache' )

class RejectedLimitLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'rejected-limit LIMIT'
   noOrDefaultSyntax = 'rejected-limit ...'
   data = {
      'rejected-limit': rejectedLimitMatcher,
      'LIMIT': limitMatcher,
   }
   handler = setRejectedSALimit
   noOrDefaultHandler = setRejectedSALimit

MsdpCli.RouterMsdpBaseMode.addCommandClass( RejectedLimitLimitCmd )
MsdpCli.RouterMsdpVrfMode.addCommandClass( RejectedLimitLimitCmd )

class IpMsdpRejectedLimitLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp rejected-limit LIMIT'
   noOrDefaultSyntax = 'ip msdp rejected-limit ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': MsdpCli.msdpNode,
      'rejected-limit':
         CliCommand.Node(
            rejectedLimitMatcher ),
      'LIMIT': limitMatcher,
   }
   handler = setRejectedSALimit
   noOrDefaultHandler = setRejectedSALimit

MsdpCli.RouterMsdpSharedModelet.addCommandClass( IpMsdpRejectedLimitLimitCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vrf VRFNAME
#--------------------------------------------------------------------------------
class VrfVrfnameCmd( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
      'VRF': VrfCli.VrfExprFactory( helpdesc='Configure MSDP in a VRF' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args[ 'VRF' ]
      if vrfName in VrfCli.RESERVED_VRF_NAMES:
         mode.addError( "vrf name %s is reserved." % vrfName )
         return

      if vrfName not in MsdpCli.msdpConfig.vrfConfig:
         MsdpCli.msdpVrfDefinitionHook( vrfName )
         if not VrfCli.vrfExists( vrfName ):
            mode.addWarning( "vrf name %s is not defined." % vrfName )

      IraIpCli.warnIfRoutingDisabled( mode, vrfName )
      childMode = mode.childMode( MsdpCli.RouterMsdpVrfMode, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      MsdpCli.deleteRouterMsdpVrfMode( mode, args[ 'VRF' ] )

MsdpCli.RouterMsdpBaseMode.addCommandClass( VrfVrfnameCmd )
