# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Enum, Float, Int, List, Str, Dict, Bool, Submodel
from ArnetModel import Ip4Address
from datetime import timedelta
from IntfModels import Interface
import Tac
#------------------------------------------------------------------------------
# CliModels                                                
#------------------------------------------------------------------------------
class SGPair( Model ):
   """Model for ( S, G )"""
   sourceAddress = Ip4Address( help="Source Address" )
   groupAddress = Ip4Address( help="Group Address" )

class SaMsg( Model ):
   sourceGroupPair = Submodel( help="(S,G) Pair", valueType=SGPair )
   rpAddress = Ip4Address( help="Addess of the Rendezvous Point" )
   remoteAddress = Ip4Address( help="Heard from this peer", optional=True )

class PeerSummary( Model ):

   __revision__ = 2
   peerIpAddress = Ip4Address( help="Ip address of the MSDP peer" )
   description = Str( help="Description of MSDP peer", optional=True )
   state = Enum( help="Current status of the peer" ,
                 values= ( 'established',
                           'listen',
                           'connecting',
                           'inactive',
                           'disabled' ) )
   sessionStartTime = Float( help="UTC time at which the peer last switched state\
                                    to established", optional=True )
   saCount = Int( help="Number of SA messages from this MSDP peer in the SA cache",
                  default=0 )
   resetCount = Int( help="MSDP peer connection reset count" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if 'sessionStartTime' not in dictRepr:
            dictRepr['sessionStartTime'] = 0.0
      return dictRepr

class PeerInfo( PeerSummary ):
   """ MSDP peer status and config information """
   __revision__ = 2
   connSourceInterface = Interface( help="Connection source interface used,\
                                          If available", optional=True )
   connSourceAddress = Ip4Address( help="The connection source IP address,\
                                         If Available", optional=True )
   saFilterIn = Str( help="Access list filtering of SA input" )
   saFilterOut = Str( help="Access list filtering of SA output" )
   acceptedSas = List( help="List of Accepted SAs", valueType=SaMsg )

#------------------------------------------------------------------------------
# show msdp mesh-group
#------------------------------------------------------------------------------
class MeshGroupModel( Model ):
   """Cli Model for show msdp mesh-group"""  
   class Group( Model ):
      ipList = List( help="Mesh group members ip", valueType=Ip4Address )
   members = Dict( help="Dictionary of Mesh group names to list of member IPs",
                   valueType=Group )
   def render( self ):
      for (name, member ) in self.members.items():
         print "Mesh Group: %s" % name
         for ip in member.ipList:
            print "\t%s" % ip

#------------------------------------------------------------------------------
# show ip msdp peer
#------------------------------------------------------------------------------
class PeerModel( Model ):    
   """Cli Model for 'show ip msdp peer'"""
   __revision__ = 2
   peerList = List( help="list of peers", valueType=PeerInfo )
   def render( self ):
      for peer in self.peerList:
         print "MSDP Peer %s" % ( peer.peerIpAddress )
         if peer.description:
            print "Description: %s" % peer.description
         print "Connection status:"
         print "   State: %s" % peerStateDisplayStr( peer.state )
         if peer.state == 'established': 
            #Equals to if peer.sessionStartTime:
            print "   Session Time: %s" % formatSessionTime( peer.sessionStartTime )
         print "   Resets: %s" % peer.resetCount
         if peer.connSourceInterface:
            connSrcStr = "   Connection Source: %s ( %s )" % \
                         ( peer.connSourceInterface.stringValue, 
                           peer.connSourceAddress)
            print connSrcStr
         if peer.saFilterIn or peer.saFilterOut:
            print "SA Filtering:"
            if peer.saFilterIn:
               print "Input Filter: %s" % peer.saFilterIn
            if peer.saFilterOut:
               print "Output Filter: %s" % peer.saFilterOut
         if peer.acceptedSas:
            print "SAs accepted:"
            for sa in peer.acceptedSas:
               print "(%s, %s), RP %s" % ( sa.sourceGroupPair.sourceAddress,
                                           sa.sourceGroupPair.groupAddress, 
                                           sa.rpAddress )      
         print 

#------------------------------------------------------------------------------
# show ip msdp pim sa-cache
#------------------------------------------------------------------------------
class PimSaCacheModel( Model ):
   """Cli Model for show ip msdp pim sa-cache"""
   saCache = List( help="MSDP soure active messages for local Pim RP", 
         valueType=SaMsg )
   def render( self ):
      print "MSDP Source Active Messages for local Pim RP" 
      for info in self.saCache:
         print '(%s, %s), RP %s' % ( info.sourceGroupPair.sourceAddress, 
                                     info.sourceGroupPair.groupAddress,
                                     info.rpAddress )
#------------------------------------------------------------------------------
# show msdp rpf-peer <rp address>
#------------------------------------------------------------------------------
class RpfPeerModel( Model ):
   """Cli Model for show msdp rpf-peer <rp address>"""
   rpAddress = Ip4Address( help="RP address" )
   rpfPeer = Ip4Address( help="RPF peer address, if query succeeds", optional=True )
   queryStarted = Bool( help="RPF query started", default=False )
   queryCompleted = Bool( help="If the RPF query completed" )
   def render( self ):
      if self.queryStarted:
         print "Querying the Rpf peer for RP %s" % self.rpAddress
      if self.queryCompleted:
         if str( self.rpfPeer ) == '255.255.255.255': 
            print "RPF peer does not exist" 
         else:
            print "Rpf Peer is %s for RP %s" % ( self.rpfPeer, self.rpAddress )
      else: 
         print 'Unable to find Rpf Peer for %s' % self.rpAddress

#------------------------------------------------------------------------------
# show ip msdp sa-cache [rejected]
#------------------------------------------------------------------------------
class SaCacheModel( Model ):
   """Cli Model for show ip msdp sa-cache [rejected]"""
   _rejected = Bool( help="rejected parameter present" ) 
   acceptedSaMsg = List( help="List of accepted SA Messages", valueType=SaMsg )
   rejectedSaMsg = List( help="List of rejected SA Messages", valueType=SaMsg, 
                         optional=True )
   def setRejected(self, isRejected):
      self._rejected = isRejected
   def render( self ):
      print "MSDP Source Active Cache"
      self.displayRemoteSAMsg( self.acceptedSaMsg )
      print
      if self._rejected:
         print "MSDP Rejected Source Active Cache" 
         self.displayRemoteSAMsg( self.rejectedSaMsg )
         print
   def displayRemoteSAMsg( self, msgList ):
      for saMsg in msgList:
         print '(%s, %s), RP %s, heard from %s' % \
               ( saMsg.sourceGroupPair.sourceAddress, 
                 saMsg.sourceGroupPair.groupAddress, saMsg.rpAddress, 
                 saMsg.remoteAddress )

#------------------------------------------------------------------------------
# show ip msdp summary
#------------------------------------------------------------------------------
class SummaryModel( Model ):
   """Cli Model for 'show ip msdp summary'"""
   __revision__ = 2
   peerList = List( help="list of peers", valueType=PeerSummary )
   def render( self ):
      print "MSDP Peer Status Summary"
      print 'Peer Address'.ljust( 18 ) + 'State'.ljust( 8 ) + \
            'Session time'.ljust( 15 ) + 'SA Count'.ljust( 9 ) + \
            'Reset Count'.ljust( 15 )
      for peer in self.peerList:
         print str( peer.peerIpAddress ).ljust( 18 ) + \
               peerStateDisplayStr( peer.state ).ljust( 8 ),
         if peer.state == 'established':
            print formatSessionTime( peer.sessionStartTime ).ljust(15),
         else:
            print "".ljust(15),
         print str( peer.saCount ).ljust( 9 ) + \
               str( peer.resetCount ).ljust( 15 )

#------------------------------------------------------------------------------
# Helper Methods
#------------------------------------------------------------------------------
def formatSessionTime( time ):
   delta = Tac.utcNow() - time
   td = timedelta( seconds=int( delta ) )
   if td > timedelta( days=1 ) :
      # If session time > 1 day then display the time in XdXXh format
      return str( td.days ) + "d" + str( td.seconds / 3600 ) + "h"
   else:
      #Else display the session time in HH:MM:SS format
      return str( td )

def peerStateDisplayStr( state ):
   stateMap = { 'established' : 'Up',
                'listen' : 'Listen',
                'connecting' : 'Connect',
                'inactive' : 'Down',
                'disabled' : 'Disabled' }
   if state:
      peerState = stateMap[ state ]
   else:
      peerState = 'Down'
   return peerState
