# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli, LazyMount, ConfigMount
import CliParser
import TechSupportCli
from McastCommonCli import mcastRoutingSupportedGuard
import CliToken.Ip
import Tac
from CliMode.Msdp import RoutingMsdpBaseMode, RoutingMsdpVrfMode, \
      RoutingMsdpPeerMode
from McastCommonCliLib import vrfFromMode
from IpLibConsts import DEFAULT_VRF, VRFNAMES_RESERVED
import MrouteCli
import RouterMulticastCliLib
from RouterMulticastCliLib import configGetters, lazyGetters, AddressFamily
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.VrfCli as VrfCli
import CliCommand
import CliMatcher

msdpPeerAddrMatcher = IpAddrMatcher.IpAddrMatcher( 'MSDP peer address' )

vrfExprFactory = VrfCli.VrfExprFactory( helpdesc='VRF name' )

MAX_SA_CACHE_SIZE = 40000

MsdpConfigColl = "Routing::Msdp::ConfigColl"
( configColl,
  configCollFromMode,
  config,
  configFromMode ) = configGetters( MsdpConfigColl )

MsdpClearConfigColl = "Routing::Msdp::ClearConfigColl"
( clearConfigColl,
  clearConfigCollFromMode,
  clearConfig,
  clearConfigFromMode ) = lazyGetters( MsdpClearConfigColl,
                                       writeMount=True )

MfibVrfConfig = "Routing::Multicast::Fib::VrfConfig"
( mfibVrfConfigColl,
  mfibVrfConfigCollFromMode,
  mfibConfig,
  mfibConfigFromMode ) = lazyGetters( MfibVrfConfig, collectionName='config' )

# Globals
msdpConfig = None
msdpStatus = None
msdpPimStatus = None
rpfMapStatusDir = None
groupSourceMapColl = None
entityMgr = None
routingHwStatus = None

def msdpVrfDefinitionHook( vrfName ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      clearConfig( af, vrfName )
      config( af, vrfName )

def _msdpVrfDeletionHook( vrfName ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      _configColl = configColl( af )
      if vrfName != DEFAULT_VRF and vrfName in _configColl.vrfConfig:
         msdpVrfConfig = config( af, vrfName )
         if msdpVrfConfig.originatorId == "" and \
            not msdpVrfConfig.defaultPeer and \
            not msdpVrfConfig.sourceSALimit and \
            not msdpVrfConfig.meshGroup and \
            not msdpVrfConfig.peerConfig and \
            msdpVrfConfig.connRetryPeriod == \
            msdpVrfConfig.connRetryPeriodDefault:
            # only delete if there is no non-default config and the VRF is not
            # defined
            del _configColl.vrfConfig[ vrfName ]
      _clearConfigColl = clearConfigColl( af )
      if vrfName in _clearConfigColl.vrfConfig and vrfName != DEFAULT_VRF:
         del _clearConfigColl.vrfConfig[ vrfName ]

def vrfExists( af, vrfName ):
   if mfibConfig( af, vrfName ):
      return True
   return False

def printNonexistentPeerError( mode, peerIp ):
   mode.addError( "Peer is not configured in vrf %s: %s" % \
                     ( vrfFromMode( mode ), peerIp ) )

class RouterMsdpBaseMode( RoutingMsdpBaseMode, BasicCli.ConfigModeBase ):
   name = 'MSDP configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.vrfName = DEFAULT_VRF
      RoutingMsdpBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterMsdpVrfMode( RoutingMsdpVrfMode, BasicCli.ConfigModeBase ):
   name = 'MSDP VRF configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      assert vrfName not in VRFNAMES_RESERVED
      self.vrfName = vrfName
      RoutingMsdpVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterMsdpPeerMode( RoutingMsdpPeerMode, BasicCli.ConfigModeBase ):
   name = 'MSDP Peer configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName=DEFAULT_VRF, peer=None ):
      RoutingMsdpPeerMode.__init__( self, vrfName, peer )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterMsdpSharedModelet( CliParser.SingletonModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.SingletonModelet.__init__( self )
      # GlobalConfigMode enables instanceRule sharing, so addRule()
      # will be performed separately.
      self.mode_ = mode
      if hasattr( self.mode_, 'vrfName' ):
         self.vrfName = self.mode_.vrfName
      else:
         self.vrfName = DEFAULT_VRF

RouterMsdpBaseMode.addModelet( RouterMsdpSharedModelet )
RouterMsdpVrfMode.addModelet( RouterMsdpSharedModelet )
BasicCli.GlobalConfigMode.addModelet( RouterMsdpSharedModelet )

#------------------------------------------------------------------------------------
# Clean-up VRF config 
#------------------------------------------------------------------------------------

def cleanUpMsdpVrfConfig( vrfName ):
   if vrfName is None or vrfName == '':
      vrfName = DEFAULT_VRF
   if vrfName in msdpConfig.vrfConfig:
      msdpVrfConfig = msdpConfig.vrfConfig[ vrfName ]
      msdpVrfConfig.originatorId = ""
      msdpVrfConfig.defaultPeer.clear()
      msdpVrfConfig.sourceSALimit.clear() 
      msdpVrfConfig.meshGroup.clear()
      msdpVrfConfig.peerConfig.clear()
      msdpVrfConfig.connRetryPeriod = \
            msdpVrfConfig.connRetryPeriodDefault 

def deleteRouterMsdpVrfMode( mode, vrfName ):
   if vrfName in VRFNAMES_RESERVED:
      return
   if vrfName in msdpConfig.vrfConfig:
      cleanUpMsdpVrfConfig( vrfName )
      if vrfName != DEFAULT_VRF:
         _msdpVrfDeletionHook( vrfName )

#-------------------------------------------------------------------------------
# Common command utilities and 'msdp' node (note that it requires multicast to be
# supported).
#-------------------------------------------------------------------------------
msdpNode = CliCommand.Node(
      CliMatcher.KeywordMatcher(
         'msdp', helpdesc='MSDP protocol commands' ),
      guard=mcastRoutingSupportedGuard )

def peerConfig( peerIp, create=False, connSrc=None, vrfName=DEFAULT_VRF ):
   existingPeerConfig = msdpConfig.vrfConfig[ vrfName ].peerConfig.get( peerIp )
   if create:
      if existingPeerConfig:
         del msdpConfig.vrfConfig[ vrfName ].peerConfig[ peerIp ]
      if connSrc is not None:
         peer = \
             msdpConfig.vrfConfig[ vrfName ].peerConfig.newMember(
            peerIp, connSrc.name )
      else:
         peer = msdpConfig.vrfConfig[ vrfName ].peerConfig.newMember( peerIp, "" )
      return peer
   else:
      return existingPeerConfig

def safeDel( coll, key ):
   if key and key in coll:
      try:
         del coll[ key ]
      except KeyError:
         # this was a race to delete
         pass

#-------------------------------------------------------------------------------
# Deprecated cmd:
# [ no | default ] ip msdp cache-sa-state
#-------------------------------------------------------------------------------
class IpMsdpCacheSaStateCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp cache-sa-state'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': msdpNode,
      'cache-sa-state':
         CliCommand.Node( 
            CliMatcher.KeywordMatcher( 'cache-sa-state',
               helpdesc='Cache SA state (EOS always does this)' ),
            deprecatedByCmd='none (no longer supported)' )
   }

   @staticmethod
   def handler( mode, args ):
      # Do nothing.
      pass

RouterMsdpSharedModelet.addCommandClass( IpMsdpCacheSaStateCmd )

#-------------------------------------------------------------------------------
# Tech Support commands
#-------------------------------------------------------------------------------
def msdpSupported():
   return routingHwStatus.multicastRoutingSupported

def _showTechCmds():
   if msdpSupported():
      return [ 
            'show ip msdp peer',
            'show ip msdp sa-cache rejected',
            'show ip msdp pim sa-cache',
            'show ip msdp summary',
           ]
   return []

TechSupportCli.registerShowTechSupportCmdCallback( '2012-02-24 10:07:00',
               _showTechCmds,
               summaryCmdCallback=lambda: [ 'show ip msdp summary' ] )

def Plugin( entityManager ):
   global msdpConfig
   global msdpStatus
   global msdpPimStatus
   global rpfMapStatusDir
   global groupSourceMapColl
   global routingHwStatus
   global entityMgr

   entityMgr = entityManager
   RouterMulticastCliLib.doConfigMounts( entityManager, [ MsdpConfigColl ] )
   RouterMulticastCliLib.doLazyMounts( entityManager, [ MsdpClearConfigColl ], 
                                       useWriteMount=True )
   RouterMulticastCliLib.doLazyMounts( entityManager, [ MfibVrfConfig ], 
                                       useWriteMount=False )

   msdpConfig = ConfigMount.mount( entityManager, 'routing/msdp/config',
                                   'Routing::Msdp::ConfigColl', 'w' )
   msdpStatus = LazyMount.mount( entityManager, 'routing/msdp/status',
                                 'Routing::Msdp::StatusColl', 'r' )
   msdpPimStatus = LazyMount.mount( entityManager, 'routing/msdp/pimstatus',
                                 'Routing::Msdp::PimStatusColl', 'r' )
   rpfMapStatusDir = LazyMount.mount( entityManager, 'routing/msdp/rpfMapStatus',
         'Tac::Dir', 'ri' )
   groupSourceMapColl = LazyMount.mount( entityManager,
                            "routing/pim/sparsemode/groupsourcemap",
                            "Routing::Pim::SparseMode::GroupSourceMapColl",
                                         "r" )
   routingHwStatus = LazyMount.mount( entityManager, 'routing/hardware/status',
         'Routing::Hardware::Status', 'r' )

   MrouteCli.routerMcastVrfDeletionHook.addExtension( _msdpVrfDeletionHook )
   MrouteCli.routerMcastVrfDefinitionHook.addExtension( msdpVrfDefinitionHook )
