# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericAddress
from CliModel import Model, List, Dict
import TableOutput
from TypeFuture import TacLazyType
from functools import cmp_to_key

IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )

class MrouteVxlanDecapGroup( Model ):
   sources = List( help="List of sources for this group",
         valueType=IpGenericAddress )

class MrouteVxlanDecapInterface( Model ):
   groups = Dict( help="A mapping of group to it's sources",
         keyType=IpGenericAddress,
         valueType=MrouteVxlanDecapGroup )

class MrouteVxlanDecapJoinedInterfaces( Model ):
   interfaces = Dict( help="A mapping of interface to joined groups", keyType=str,
         valueType=MrouteVxlanDecapInterface )

   def render( self ):
      def compareAddress( addr1, addr2 ):
         if addr1 == addr2:
            return 0
         elif addr1 < addr2:
            return -1
         return 1

      def groupSort( addr1, addr2 ):
         return compareAddress( IpGenAddr( addr1 ), IpGenAddr( addr2 ) )

      def sourceSort( addr1, addr2 ):
         return compareAddress( addr1, addr2 )
      if self.interfaces:
         headers = ( "Interface", "Group", "Source" )
         formatLeft = TableOutput.Format( justify="left" )
         formatLeft.noPadLeftIs( True )

         table = TableOutput.createTable( headers )
         table.formatColumns( formatLeft, formatLeft, formatLeft )
         for interface, groups in sorted( self.interfaces.items() ):
            for group in sorted( groups.groups, key=cmp_to_key( groupSort ) ):
               for source in sorted( groups.groups[ group ].sources,
                     key=cmp_to_key( sourceSort ) ):
                  table.newRow( interface, group, source )
         print table.output()
