# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
import CliToken.Ip
import CliToken.Ipv6
import CliToken.McastCommon
import CliToken.McastVpn
import IpGenAddrMatcher
import ShowCommand
import Tac
import Toggles.McastVpnLibToggleLib
from MrouteVxlanDecapShowModel import (
   MrouteVxlanDecapJoinedInterfaces,
   MrouteVxlanDecapInterface,
)
from MrouteVxlanDecapShowModel import MrouteVxlanDecapGroup
import MrouteVxlanDecapShowModel
from VirtualIntfRule import IntfMatcher
import VlanIntfCli
import EthIntfCli
import LagIntfCli
import SubIntfCli
import SmashLazyMount
import LazyMount
from operator import attrgetter

_entityManager = None
_mrouteVxlanDecapTable4 = None
_mrouteVxlanDecapTable6 = None
_bridgingConfig = None

joinedKwMatcher = CliMatcher.KeywordMatcher( 'joined', helpdesc='Joined SP Trees' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Interface keyword' )
intfMatcher = IntfMatcher()
intfMatcher |= VlanIntfCli.VlanIntf.matcher
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= LagIntfCli.EthLagIntf.matcher
intfMatcher |= SubIntfCli.subMatcher
intfMatcher |= LagIntfCli.subMatcher
groupIpGenAddrMatcher = IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='Group address' )
srcIpGenAddrMatcher = IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='Source address' )

def showMulticastDecapJoined( mode, args ):
   """Display the multicast decap routes, optionally limiting
      output to matching sources, groups, and interfaces"""
   src = args.get( 'SOURCE' )
   grp = args.get( 'GROUP' )
   intf = args.get( 'INTERFACE' )
   af = 'ipv4' if 'ipv4' in args else 'ipv6'
   addr = attrgetter( 'v4Addr' if af == 'ipv4' else 'v6Addr' )

   model = MrouteVxlanDecapJoinedInterfaces()
   if af == 'ipv4':
      mrouteVxlanDecapTable = _mrouteVxlanDecapTable4
   else:
      mrouteVxlanDecapTable = _mrouteVxlanDecapTable6

   if not mrouteVxlanDecapTable:
      return model

   for rKey, decap in mrouteVxlanDecapTable.mrouteVxlanDecap.items():
      vlanConversion = "Vlan" + str( decap.v )
      vc = _bridgingConfig.vlanConfig.get( decap.v )
      if intf:
         if vc and vc.internal and str( intf ) != vc.configuredName:
            continue
         elif vc and not vc.internal and str( intf ) != vlanConversion:
            continue
      if grp and grp.stringValue != str( addr( rKey.g ) ):
         continue
      if src and src.stringValue != str( addr( rKey.s ) ):
         continue
      # Get the InterfaceModel if it exists, else create it
      if vc and vc.internal:
         vlanConversion = vc.configuredName
      interfaceModel = model.interfaces.get( vlanConversion )
      if not interfaceModel:
         interfaceModel = MrouteVxlanDecapInterface()
         model.interfaces[ vlanConversion ] = interfaceModel
      groupModel = interfaceModel.groups.get( rKey.g.ipGenAddr )
      if not groupModel:
         groupModel = MrouteVxlanDecapGroup()
         interfaceModel.groups[ rKey.g.ipGenAddr ] = groupModel
      groupModel.sources.append( rKey.s.ipGenAddr )
   return model

#-----------------------------------------------------------------------------------
# show multicast ( ipv4 | ipv6 ) evpn decap joined [ INTERFACE ] [ GROUP [ SOURCE ] ]
#-----------------------------------------------------------------------------------
class ShowMulticastDecapJoined( ShowCommand.ShowCliCommandClass ):
   syntax = (
         'show multicast (  ipv4 | ipv6 ) evpn decap joined [ interface INTERFACE ] '
         '[ GROUP [ SOURCE ] ]' )
   data = {
            'multicast': CliToken.McastCommon.multicastMatcherForShow,
            'ipv4': CliToken.Ip.ipv4MatcherForShow,
            'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'evpn': CliToken.McastVpn.evpnForShow,
            'decap': CliToken.McastVpn.decapForShow,
            'joined': joinedKwMatcher,
            'interface': matcherInterface,
            'INTERFACE': intfMatcher,
            'GROUP': groupIpGenAddrMatcher,
            'SOURCE': srcIpGenAddrMatcher,
          }
   handler = showMulticastDecapJoined
   cliModel = MrouteVxlanDecapShowModel.MrouteVxlanDecapJoinedInterfaces

if Toggles.McastVpnLibToggleLib.toggleMcastVpnOISMEnabled():
   BasicCli.addShowCommandClass( ShowMulticastDecapJoined )

def Plugin( entityManager ):
   global _entityManager
   global _mrouteVxlanDecapTable4
   global _mrouteVxlanDecapTable6
   global _bridgingConfig

   _entityManager = entityManager
   smashUrl = Tac.Type( "MrouteLib::MrouteVxlanDecapTable" ).mountPath( 'ipv4' )
   _mrouteVxlanDecapTable4 = SmashLazyMount.mount( _entityManager, smashUrl,
         'MrouteLib::MrouteVxlanDecapTable',
         SmashLazyMount.mountInfo( 'reader' ) )
   smashUrl = Tac.Type( "MrouteLib::MrouteVxlanDecapTable" ).mountPath( 'ipv6' )
   _mrouteVxlanDecapTable6 = SmashLazyMount.mount( _entityManager, smashUrl,
         'MrouteLib::MrouteVxlanDecapTable',
         SmashLazyMount.mountInfo( 'reader' ) )
   _bridgingConfig = LazyMount.mount( _entityManager, "bridging/config",
         'Bridging::Config', 'r' )
