#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Multicast Routing related commands.
# Configure commands:
# -  [no|default] ip multicast-routing
# *  ip route multicast G [S] iif intfName oif intfName1 intfName2 ...
# *  no ip route multicast G [S] [iif intfName oif intfName1 intfName2 ...]
# *  ip multicast local-source-notify intfName
# *  no ip multicast fastdrop
# -  [no] ip mfib max-fastdrops <n>
# -  ip multicast multipath none
# -  [no] ip multicast multipath deterministic
# -  [no] ip multicast disable-l2-flooding
#
# Show/Enable-level commands:
# -  show ip mfib [ vrf <vrf-name> ]
# -  show mfib { ipv4 | ipv6 } [ vrf <vrf-name> ]
# -  clear ip mfib [ vrf <vrf-name> ] fastdrop
# -  clear mfib { ipv4 | ipv6 } [ vrf <vrf-name> ] fastdrop
# *  clear ip mfib [ vrf <vrf-name> ] route
# *  clear mfib { ipv4 | ipv6 } [ vrf <vrf-name> ] route
#
# The commands marked '*' are hidden commands.
#-------------------------------------------------------------------------------

import CliParser, BasicCli, Tac, Arnet
import CliToken.Clear, CliToken.Ip
from CliToken.Ip import addressFamilyExprForClear
from CliToken.Router import routerMatcherForConfig
import LazyMount, IraIpCli, IraIpIntfCli, ConfigMount, SmashLazyMount
import IntfCli, VlanIntfCli, EthIntfCli, LagIntfCli, SubIntfCli
import IraVrfCli
from IraIpRouteCliLib import isValidPrefixWithError, routeMatcherForConfig
from IpLibConsts import DEFAULT_VRF, VRFNAMES_RESERVED
import MrouteConsts
import MrouteDfModel
from McastCommonModel import MfibGroups
import SharedMem
import CliPlugin.ConfigConvert
import CliPlugin.VrfCli as VrfCli
import CliExtensions
import Tracing
import RouterMulticastCliLib
from RouterMulticastCliLib import ( legacyCliCallback,
                                    configGetters,
                                    RouterMulticastMode,
                                    getAddressFamilyFromMode )
from McastCommonCliLib import ( mcastRoutingSupported,
                                mcastRoutingSupportedGuard,
                                mcast6RoutingSupportedGuard,
                                mcastGenRoutingSupportedGuard,
                                AddressFamily,
                                validateRouting,
                                getAfFromIpFamilyRule,
                                IpFamilyExpr )
import sys
from VirtualIntfRule import IntfMatcher
from CliCommand import CliCommandClass, Node, CliExpression, isNoCmd, isDefaultCmd
from CliMatcher import ( KeywordMatcher, IntegerMatcher, DynamicIntegerMatcher,
                         EnumMatcher )
from IpAddrMatcher import IpAddrMatcher
from Ip6AddrMatcher import Ip6AddrMatcher
from itertools import chain

traceHandle = Tracing.Handle( "MrouteCli" )
t0 = traceHandle.trace0

mfibDir = None
mfibDir6 = None
mfibSmashDir = Tac.newInstance( "Smash::Multicast::Fib::MfibSmashDir" )
mfibSmashDir6 = Tac.newInstance( "Smash::Multicast::Fib::MfibSmashDir" )
mfibDirSm = None
mfibDirSm6 = None
mfibVrfConfig = None
mfib6VrfConfig = None
mfibHardwareStatus = None
mfib6HardwareStatus = None
routingVrfInfoDir = None
routing6VrfInfoDir = None
routingHardwareStatus = None
routing6HardwareStatus = None
mfibCounterStatus = None
ipStatus = None
bidirStatus = None
bidirVrf = None
_entityManager = None
_smashMount = None
_multicastLegacyConfig = None

enterSubmode = RouterMulticastMode.enterSubmode
deleteSubmode = RouterMulticastMode.deleteSubmode

# Af Independent Config Types
MfibVrfConfig = "Routing::Multicast::Fib::VrfConfig"

# RouterMcastVrfDeletion CLI hook is used to allow agents to clean up on
# "no router multicast" (for default VRF) or "no vrf <vrfName>" (for non-default VRF)
# This function is called as:
#
# hook ( vrfName )
#
# vrfName is the name of relevant vrf.

routerMcastVrfDeletionHook = CliExtensions.CliHook()

# RouterMcastVrfDefinition CLI hook is used to allow multicast agents to create
# their config directories for this VRF in the CliPlugin.
# "router multicast" (for default VRF) or "vrf <vrfName>" (for non-default VRF)
# This function is called as:
#
# hook ( vrfName )
#
# vrfName is the name of relevant vrf.
# enabled is True when vrf definition is configured.
# and False when vrf definition is deleted using 'no vrf definition <vrfName>'

routerMcastVrfDefinitionHook = CliExtensions.CliHook()

legacyConfigHook = CliExtensions.CliHook()

# showMfibCounterHook CLI hook is used to allow multicast agent to
# attach counter information to CLI output
# This function is called as:
#
# hook( capiModel )
#
# capiModel is a TAC type that can accept a counter hook
showMfibCounterHook = CliExtensions.CliHook()

# clearMfibCounterHook CLI hook is used to allow multicast agent to
# clear the counter for a given mroute.
# This function is called as:
#
# hook( vrfName, key )
#
# vrfName is the name of relevant vrf, key is a representation of S and G
clearMfibCounterHook = CliExtensions.CliHook()

( mfibConfigRoot,
  mfibConfigRootFromMode,
  mfibConfig,
  mfibConfigFromMode ) = configGetters( MfibVrfConfig, collectionName='config' )

def mcastFibIpv4SupportedGuard( mode=None, token=None ):
   if mcastRoutingSupported( _entityManager.root(), routingHardwareStatus=None ):
      return None
   return CliParser.guardNotThisPlatform

def _mfibConfig( vrfName, af=AddressFamily.ipv4 ):
   return mfibConfig( af, vrfName )

def _mfibStatus( vrfName, af ):
   if af == 'ipv4':
      if vrfName in mfibSmashDir.smashColl:
         return mfibSmashDir.smashColl[ vrfName ]
   else:
      if vrfName in mfibSmashDir6.smashColl:
         return mfibSmashDir6.smashColl[ vrfName ]

   return None

def _mfibHwVrfStatus( vrfName, af ):
   if af == AddressFamily.ipv4:
      if vrfName in mfibHardwareStatus.vrfStatus:
         return mfibHardwareStatus.vrfStatus[ vrfName ]
   else:
      if vrfName in mfib6HardwareStatus.vrfStatus:
         return mfib6HardwareStatus.vrfStatus[ vrfName ]

   return None

def _mfibHwStatus( af=AddressFamily.ipv4 ) :
   if af == AddressFamily.ipv4:
      return mfibHardwareStatus
   else:
      return mfib6HardwareStatus

def _mrouteVrfCreation( vrfName, status ):
   t0( "_mrouteVrfCreation( %s, %s )" %( vrfName, status ) )
   # TODO: Create commonConfig
   ipv4Root = mfibConfigRoot( af=AddressFamily.ipv4 )
   ipv6Root = mfibConfigRoot( af=AddressFamily.ipv6 )
   if status:
      t0( "Initializing vrf %s" % vrfName )
      if vrfName not in ipv4Root.config:
         ipv4Root.config.newMember( vrfName )

      if vrfName not in ipv6Root.config:
         ipv6Root.config.newMember( vrfName )
      for hook in routerMcastVrfDefinitionHook.extensions():
         t0( "Creating Hook: %s" % hook )
         hook( vrfName )
   else:
      if vrfName != DEFAULT_VRF:
         mvc = ipv4Root.config.get( vrfName )
         if mvc:
            t0( "Cleaning up ipv4 vrf %s" % vrfName )
            del ipv4Root.config[ vrfName ]

         mvc = ipv6Root.config.get( vrfName )
         if mvc:
            t0( "Cleaning up ipv6 vrf %s" % vrfName )
            del ipv6Root.config[ vrfName ]
      else:
         ipv4Root.config.get( vrfName ).reset()
         ipv6Root.config.get( vrfName ).reset()

      for hook in routerMcastVrfDeletionHook.extensions():
         t0( " Deleting Hook : %s" % hook )
         hook( vrfName )

def vrfFromIntf( intfName ):
   vrfName = DEFAULT_VRF
   if intfName in ipStatus.ipIntfConfig:
      vrfName = ipStatus.ipIntfConfig[ intfName ].vrf
   return vrfName

def minSupportVersion( minVersion ):
   if _multicastLegacyConfig.version < minVersion:
      _multicastLegacyConfig.version = minVersion

   if _multicastLegacyConfig.version > _multicastLegacyConfig.globalMode:
      _multicastLegacyConfig.legacy = False
   else:
      _multicastLegacyConfig.legacy = True

def convertLegacyConfigMulticast( mode ):
   minSupportVersion( _multicastLegacyConfig.ipMode )

defaultSourceV4 = Arnet.IpGenPrefix( "0.0.0.0/32" )
defaultSourceV6 = Arnet.IpGenPrefix( "0::0/128" )

def sourceGroupKey( source, group ):
   if source:
      source = str( source )
   group = str( group )

   if ':' in group:
      mlen = 128
      defaultSource = defaultSourceV6
   else:
      mlen = 32
      defaultSource = defaultSourceV4

   sourcePfx = Arnet.IpGenPrefix( source + "/%d" % mlen ) if source else \
      defaultSource
   groupPfx = Arnet.IpGenPrefix( group + "/%d" % mlen )
   return Tac.Value( "Routing::Multicast::Fib::IpGenRouteKey", sourcePfx, groupPfx )

#-------------------------------------------------------------------------------
#Register convertLegacyConfigMulticast via "config convert new-syntax"
#-------------------------------------------------------------------------------
CliPlugin.ConfigConvert.registerConfigConvertCallback( convertLegacyConfigMulticast )

multicastRoutingKwMatcher = KeywordMatcher( 'multicast-routing',
   helpdesc='General multicast routing configuration commands' )
multicastRoutingNode = Node( multicastRoutingKwMatcher,
   guard=mcastRoutingSupportedGuard )

def setMulticastRouting( mode, args, legacy=False ):
   mvc = mfibConfigFromMode( mode, legacy=legacy )
   mvc.routing = True

   vrfName = RouterMulticastCliLib.getVrfNameFromMode( mode )
   if vrfName == DEFAULT_VRF:
      configRoot = mfibConfigRootFromMode( mode=mode, legacy=legacy )
      configRoot.routing = True
   else:
      if vrfName not in VrfCli.getVrfNames():
         mode.addWarning( "vrf name %s is not defined." % vrfName )

   af = RouterMulticastCliLib.getAddressFamilyFromMode( mode, legacy=legacy )
   vrfInfoDir = routingVrfInfoDir if af == AddressFamily.ipv4 else routing6VrfInfoDir

   routingInfo = vrfInfoDir.get( vrfName )
   if not ( routingInfo and routingInfo.routing ):
      mode.addWarning( "unicast routing is not configured for VRF %s in %s," \
                       "packets will not be forwarded" % ( vrfName, af ) )

def noMulticastRouting( mode, args, legacy=False ):
   mvc = mfibConfigFromMode( mode, legacy=legacy )
   mvc.routing = False

   vrfName = RouterMulticastCliLib.getVrfNameFromMode( mode )
   if vrfName == DEFAULT_VRF:
      configRoot = mfibConfigRootFromMode( mode=mode, legacy=legacy )
      configRoot.routing = False

class MulticastRouting( CliCommandClass ):
   syntax = 'routing'
   noOrDefaultSyntax = syntax
   data = {
      'routing': "Enable Multicast Routing"
   }

   handler = setMulticastRouting
   noOrDefaultHandler = noMulticastRouting

RouterMulticastCliLib.RouterMulticastAfSharedModelet.addCommandClass(
   MulticastRouting )

class MulticastRoutingLegacy( CliCommandClass ):
   syntax = 'ip multicast-routing'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'multicast-routing': multicastRoutingNode
   }

   handler = legacyCliCallback( setMulticastRouting )
   noOrDefaultHandler = legacyCliCallback( noMulticastRouting )

BasicCli.GlobalConfigMode.addCommandClass( MulticastRoutingLegacy )
RouterMulticastCliLib.RouterMulticastSharedModelet.addCommandClass(
   MulticastRoutingLegacy )

multicastKwMatcher = KeywordMatcher( 'multicast',
   helpdesc='Multicast routing commands' )
multicastNode = Node( multicastKwMatcher, guard=mcastGenRoutingSupportedGuard )

intfMatcher = IntfMatcher()
intfMatcher |= VlanIntfCli.VlanIntf.matcher
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= LagIntfCli.EthLagIntf.matcher
intfMatcher |= SubIntfCli.subMatcher
intfMatcher |= LagIntfCli.subMatcher

def setLocalSource( mode, args ):
   intf = args[ 'INTF' ]
   mfibStatus = _mfibStatus( vrfFromIntf( intf.name ), af=AddressFamily.ipv4 )
   if mfibStatus:
      mfibStatus.notifyLocalSources[ intf.name ] = True

class LocalSource( CliCommandClass ):
   syntax = 'ip multicast local-source-notify INTF'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'multicast': multicastNode,
      'local-source-notify': "Configure notification of local sources",
      'INTF': intfMatcher
   }
   hidden = True

   handler = setLocalSource

BasicCli.GlobalConfigMode.addCommandClass( LocalSource )

#-------------------------------------------------------------------------------
# The MfibIntf class is used to remove the IntfConfig object when
# an interface is deleted. The Intf class will create a new instance
# of MfibIntf and call destroy when the interface is deleted
# -------------------------------------------------------------------------------
class MfibIntf( IntfCli.IntfDependentBase ):
   #----------------------------------------------------------------------------
   # Destroys the IntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      name = self.intf_.name
      del mfibVrfConfig.intfConfig[ name ]
      del mfib6VrfConfig.intfConfig[ name ]

fastdropKwMatcher = KeywordMatcher( 'fastdrop', helpdesc='Fastdrop routes' )

mfibKwMatcher = KeywordMatcher( 'mfib', helpdesc='Multicast FIB' )
mfibNode = Node( mfibKwMatcher, guard=mcastRoutingSupportedGuard )

def setIntfFastdrop( mode, args, legacy=False ):
   ipFamily = args.get( 'AF' )
   name = mode.intf.name
   af = getAfFromIpFamilyRule( ipFamily, legacy )

   vrfName = vrfFromIntf( mode.intf.name )
   validateRouting( mode, vrfName, af, msgType='warn' )

   mvc = mfibVrfConfig if af == AddressFamily.ipv4 else mfib6VrfConfig

   if isNoCmd( args ):
      intfConfig = mvc.intfConfig.newMember( name )
      intfConfig.noFastdrop = True
   else:
      intfConfig = mvc.intfConfig.get( name )
      if intfConfig:
         intfConfig.noFastdrop = False
         del mvc.intfConfig[ intfConfig.name ]

class IntfFastdrop( CliCommandClass ):
   syntax = 'mfib AF fastdrop'
   noOrDefaultSyntax = syntax
   data = {
      'mfib': mfibNode,
      'AF': IpFamilyExpr,
      'fastdrop': fastdropKwMatcher
   }

   handler = setIntfFastdrop
   noOrDefaultHandler = setIntfFastdrop

IraIpIntfCli.RoutingProtocolIntfConfigModelet.addCommandClass( IntfFastdrop )

class IntfFastdropLegacy( CliCommandClass ):
   syntax = 'ip mfib fastdrop'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfigIf,
      'mfib': mfibNode,
      'fastdrop': fastdropKwMatcher
   }

   handler = legacyCliCallback( setIntfFastdrop )
   noOrDefaultHandler = legacyCliCallback( setIntfFastdrop )

IraIpIntfCli.RoutingProtocolIntfConfigModelet.addCommandClass( IntfFastdropLegacy )

maxFastdropsKwMatcher = KeywordMatcher( 'max-fastdrops',
   helpdesc='Maximum number of fastdrop routes' )
maxFastdropsIntMatcher = IntegerMatcher( 0, 1000000,
   helpdesc='Maximum number of fastdrop routes' )

def setMaxFastdrops( mode, args, legacy=False ):
   mvc = mfibConfigFromMode( mode, legacy=legacy )
   maxFastdrops = args.get( 'MAX_FASTDROPS', mvc.maxFastdropsDefault )
   mvc.maxFastdrops = maxFastdrops

class MaxFastdrops( CliCommandClass ):
   syntax = 'max-fastdrops MAX_FASTDROPS'
   noOrDefaultSyntax = 'max-fastdrops ...'
   data = {
      'max-fastdrops': maxFastdropsKwMatcher,
      'MAX_FASTDROPS': maxFastdropsIntMatcher
   }

   handler = setMaxFastdrops
   noOrDefaultHandler = handler

RouterMulticastCliLib.RouterMulticastAfCommonSharedModelet.addCommandClass(
   MaxFastdrops )

class MaxFastdropsLegacy( CliCommandClass ):
   syntax = 'ip mfib max-fastdrops MAX_FASTDROPS'
   noOrDefaultSyntax = 'ip mfib max-fastdrops ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'mfib': mfibNode,
      'max-fastdrops': maxFastdropsKwMatcher,
      'MAX_FASTDROPS': maxFastdropsIntMatcher
   }

   handler = legacyCliCallback( setMaxFastdrops )
   noOrDefaultHandler = handler

RouterMulticastCliLib.RouterMulticastSharedModelet.addCommandClass(
   MaxFastdropsLegacy )
BasicCli.GlobalConfigMode.addCommandClass( MaxFastdropsLegacy )

def setFastdrop( mode, args=None, legacy=False ):
   mvc = mfibConfigFromMode( mode, legacy=legacy )
   if isDefaultCmd( args ):
      value = mvc.fastdropDefault
   else:
      value = not isNoCmd( args )
   mvc.fastdrop = value

setFastdropLegacy = legacyCliCallback( setFastdrop )

class Fastdrop( CliCommandClass ):
   syntax = 'fastdrop'
   noOrDefaultSyntax = syntax
   data = {
      'fastdrop': fastdropKwMatcher
   }

   handler = setFastdrop
   noOrDefaultHandler = handler

RouterMulticastCliLib.RouterMulticastAfCommonSharedModelet.addCommandClass(
   Fastdrop )

class FastdropLegacy( CliCommandClass ):
   syntax = 'ip multicast fastdrop'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'multicast': multicastNode,
      'fastdrop': fastdropKwMatcher
   }
   hidden = True

   handler = legacyCliCallback( setFastdrop )
   noOrDefaultHandler = handler

RouterMulticastCliLib.RouterMulticastSharedModelet.addCommandClass( FastdropLegacy )
BasicCli.GlobalConfigMode.addCommandClass( FastdropLegacy )

counterKwMatcher = KeywordMatcher( 'counters', helpdesc='Counter for bytes/packets' )

counterOptMatcher = EnumMatcher( {
   'bytes': "Count in bytes",
   'packets': "Count in packets"
} )

def isMcastFlexCounterSupported( af=AddressFamily.ipv4 ):
   if af == AddressFamily.ipv4:
      return mfibHardwareStatus.mcastFlexCounterSupport
   else:
      return mfib6HardwareStatus.mcastFlexCounterSupport

def priorityCountingSupportedGuard( mode, token ):
   if isMcastFlexCounterSupported():
      return CliParser.guardNotThisPlatform
   return None

def priorityCountingSupportedV6Guard( mode, token ):
   if isMcastFlexCounterSupported( af=AddressFamily.ipv6 ):
      return CliParser.guardNotThisPlatform
   return None

groupAddrNode = Node( IpAddrMatcher( "Group address" ),
                      guard=priorityCountingSupportedGuard )
groupV6AddrNode = Node( Ip6AddrMatcher( "Group address" ),
                        guard=priorityCountingSupportedV6Guard )
sourceAddrNode = Node( IpAddrMatcher( "Source address" ),
                       guard=priorityCountingSupportedGuard )
sourceV6AddrNode = Node( Ip6AddrMatcher( "Source address" ),
                       guard=priorityCountingSupportedV6Guard )

class McastGroupSourceExpr( CliExpression ):
   expression = 'GROUP [ SOURCE ]'
   data = {
      'GROUP': groupAddrNode,
      'SOURCE': sourceAddrNode
   }

class Mcast6GroupSourceExpr( CliExpression ):
   expression = 'GROUP [ SOURCE ]'
   data = {
      'GROUP': groupV6AddrNode,
      'SOURCE': sourceV6AddrNode
   }

def setCounter( mode, args, legacy=False ):
   mcastCounterOption = args.get( 'CTR_OPTION' )

   configRoot = mfibConfigRootFromMode( mode, legacy=legacy )
   af = getAddressFamilyFromMode( mode )
   CounterType = Tac.Type( "Routing::Multicast::Fib::CounterMode" )
   if not mcastCounterOption:
      if isMcastFlexCounterSupported( af=af ):
         mode.addWarning(
            "Use 'hardware counter feature multicast %s' "
            "instead to configure multicast counters" % af )
      else:
         configRoot.allCounter = True
   elif mcastCounterOption == 'bytes':
      configRoot.counterMode = CounterType.byteMode
   elif mcastCounterOption == 'packets':
      configRoot.counterMode = CounterType.packetMode
   else:
      if not configRoot.allCounter:
         if af == AddressFamily.ipv4:
            mode.addWarning(
               "'ip multicast counters' not configured, packet will not be counted" )
         else:
            mode.addWarning(
               "'counters' not configured, packet will not be counted" )

      if not len( mcastCounterOption ) == 2:
         mode.addError( "need to have both Group and Source" )
         return

      routeKey = sourceGroupKey( mcastCounterOption[ 1 ], mcastCounterOption[ 0 ] )
      mvc = mfibConfigFromMode( mode, legacy=legacy )
      if not mvc.counterRoute.has_key( routeKey ):
         mvc.counterRoute[ routeKey ] = True

def noSetCounter( mode, args, legacy=False ):
   mcastCounterOption = args.get( 'CTR_OPTION' )

   configRoot = mfibConfigRootFromMode( mode=mode, legacy=legacy )
   vrfName = RouterMulticastCliLib.getVrfNameFromMode( mode )
   if legacy:
      af = AddressFamily.ipv4
   else:
      af = getAddressFamilyFromMode( mode )
   if not mcastCounterOption:
      configRoot.counterMode = configRoot.counterModeDefault
      if isMcastFlexCounterSupported( af=af ):
         mode.addWarning(
            "Use 'no hardware counter feature multicast %s' "
            "instead to unconfigure multicast counters" % af )
      else:
         configRoot.allCounter = configRoot.allCounterDefault
   elif mcastCounterOption == 'bytes' or mcastCounterOption == 'packets':
      configRoot.counterMode = configRoot.counterModeDefault
   else:
      if not len( mcastCounterOption ) == 2:
         mode.addError( "need to have both Group and Source" )
         return
      routeKey = sourceGroupKey( mcastCounterOption[ 1 ], mcastCounterOption[ 0 ] )
      if vrfName in configRoot.config:
         del _mfibConfig( vrfName, af=af ).counterRoute[ routeKey ]

# 'ip multicast counters bytes|packets|<cr>' is only configurable in default vrf.
# It is configured once and applies to all VRFs.
# 'ip multicast counters <group> <source> is configurable in default and non-default
# vrfs and requires 'ip multicast counters' to be configured to have any effect.

class Multicast4Counters( CliCommandClass ):
   syntax = 'counter [ CTR_OPTION | GROUP_SOURCE ]'
   noOrDefaultSyntax = syntax
   data = {
      'counter': counterKwMatcher,
      'CTR_OPTION': counterOptMatcher,
      'GROUP_SOURCE': McastGroupSourceExpr
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'GROUP' in args:
         group = args.pop( 'GROUP', None )
         source = args.pop( 'SOURCE', None )
         args[ 'CTR_OPTION' ] = ( group, source )

   handler = setCounter
   noOrDefaultHandler = noSetCounter

RouterMulticastCliLib.RouterMulticastIpv4Modelet.addCommandClass(
   Multicast4Counters )

class Multicast6Counters( CliCommandClass ):
   syntax = 'counter [ CTR_OPTION | GROUP_SOURCE ]'
   noOrDefaultSyntax = syntax
   data = {
      'counter': counterKwMatcher,
      'CTR_OPTION': counterOptMatcher,
      'GROUP_SOURCE': Mcast6GroupSourceExpr
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'GROUP' in args:
         group = args.pop( 'GROUP', None )
         source = args.pop( 'SOURCE', None )
         args[ 'CTR_OPTION' ] = ( group, source )

   handler = setCounter
   noOrDefaultHandler = noSetCounter

RouterMulticastCliLib.RouterMulticastIpv6Modelet.addCommandClass(
   Multicast6Counters )

class MulticastCounterLegacyBase( CliCommandClass ):
   baseData = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'multicast': multicastNode,
      'counters': counterKwMatcher,
      'GROUP_SOURCE': McastGroupSourceExpr
   }

   handler = legacyCliCallback( setCounter )
   noOrDefaultHandler = legacyCliCallback( noSetCounter )

class MulticastCountersLegacy( MulticastCounterLegacyBase ):
   syntax = 'ip multicast counters [ CTR_OPTION | GROUP_SOURCE ]'
   noOrDefaultSyntax = syntax
   data = {
      'CTR_OPTION': counterOptMatcher
   }
   data.update( MulticastCounterLegacyBase.baseData )

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'GROUP' in args:
         group = args.pop( 'GROUP', None )
         source = args.pop( 'SOURCE', None )
         args[ 'CTR_OPTION' ] = ( group, source )

RouterMulticastCliLib.RouterMulticastMode.addCommandClass( MulticastCountersLegacy )
BasicCli.GlobalConfigMode.addCommandClass( MulticastCountersLegacy )

class MulticastCountersVrfLegacy( MulticastCounterLegacyBase ):
   syntax = 'ip multicast counters [ GROUP_SOURCE ]'
   noOrDefaultSyntax = syntax
   data = {}
   data.update( MulticastCounterLegacyBase.baseData )

   @staticmethod
   def adapter( mode, args, argsList ):
      group = args.pop( 'GROUP', None )
      if group:
         source = args.pop( 'SOURCE', None )
         args[ 'CTR_OPTION' ] = ( group, source )

RouterMulticastCliLib.RouterMulticastVrfMode.addCommandClass(
   MulticastCountersVrfLegacy )

multipathKwMatcher = KeywordMatcher( 'multipath',
   helpdesc='Multiple reverse paths' )

MultipathControlType = Tac.Type(
   "Routing::Multicast::Fib::Config::MultipathControl" )

def setMultipath( mode, args, legacy=False ):
   multipathOption = args[ 'MULTIPATH_OPT' ]
   mvc = mfibConfigFromMode( mode, legacy=legacy )
   if multipathOption == 'none':
      multipathControl = MultipathControlType.multipathNone
   elif multipathOption == 'deterministic':
      multipathControl = MultipathControlType.multipathDeterministic
   mvc.multipathControl = multipathControl

def defaultMultipath( mode, args, legacy=False ):
   mvc = mfibConfigFromMode( mode, legacy=legacy )
   mvc.multipathControl = MultipathControlType.multipathDeterministic

def noMultipath( mode, args, legacy=False ):
   multipathOption = args[ 'MULTIPATH_OPT' ]
   mvc = mfibConfigFromMode( mode, legacy=legacy )
   if multipathOption == 'none':
      multipathControl = MultipathControlType.multipathDeterministic
   elif multipathOption == 'deterministic':
      multipathControl = MultipathControlType.noMultipathDeterministic
   mvc.multipathControl = multipathControl

multipathOptMatcher = EnumMatcher( {
   'deterministic': 'Enable consistent hashing in ECMP routes.',
   'none': 'Select RPF with highest PIM neighbor address'
} )

class Multipath( CliCommandClass ):
   syntax = 'multipath MULTIPATH_OPT'
   noOrDefaultSyntax = syntax
   data = {
      'multipath': multipathKwMatcher,
      'MULTIPATH_OPT': multipathOptMatcher
   }

   handler = setMultipath
   noHandler = noMultipath
   defaultHandler = defaultMultipath

RouterMulticastCliLib.RouterMulticastAfCommonSharedModelet.addCommandClass(
   Multipath )

class MultipathLegacy( CliCommandClass ):
   syntax = 'ip multicast multipath MULTIPATH_OPT'
   noOrDefaultSyntax = syntax
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'multicast': multicastNode,
      'multipath': multipathKwMatcher,
      'MULTIPATH_OPT': multipathOptMatcher
   }

   handler = legacyCliCallback( setMultipath )
   noHandler = legacyCliCallback( noMultipath )
   defaultHandler = legacyCliCallback( defaultMultipath )

RouterMulticastCliLib.RouterMulticastSharedModelet.addCommandClass( MultipathLegacy )
BasicCli.GlobalConfigMode.addCommandClass( MultipathLegacy )

class MultipathRouterIdCmd( CliCommandClass ):
   syntax = 'multipath deterministic router-id'
   noOrDefaultSyntax = syntax
   data = {
      'multipath': 'Multiple reverse paths',
      'deterministic': 'Enable consistent hashing in ECMP routes.',
      'router-id': 'Deterministically pick the same upstream router'
   }

   @staticmethod
   def handler( mode, args ):
      mvc = mfibConfigFromMode( mode, legacy=False )
      mvc.multipathControl = MultipathControlType.multipathRouterId

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mvc = mfibConfigFromMode( mode, legacy=False )
      mvc.multipathControl = MultipathControlType.multipathDeterministic

RouterMulticastCliLib.RouterMulticastIpv4Modelet.addCommandClass(
   MultipathRouterIdCmd )

def pollingIntervalRange( mode, af='ipv4' ):
   mfibHwStatus = _mfibHwStatus( af )
   minPollingInterval = mfibHwStatus.pollingMinInterval
   maxPollingInterval = 60
   return( minPollingInterval, maxPollingInterval )

intervalMatcher = DynamicIntegerMatcher( pollingIntervalRange,
   helpdesc='MFIB entry activity polling interval' )

def pollingIntervalRange6( mode ):
   return pollingIntervalRange( mode, 'ipv6' )

intervalMatcher6 = DynamicIntegerMatcher( pollingIntervalRange6,
   helpdesc='MFIB entry activity polling interval' )

activityKwMatcher = KeywordMatcher( 'activity', helpdesc='Multicast activity' )

intervalKwMatcher = KeywordMatcher( 'polling-interval',
   helpdesc='Multicast activity polling interval' )

def setIpMfibActivityInterval( mode, args, legacy=False ):
   interval = args[ 'INTERVAL' ]
   config = mfibConfigRootFromMode( mode, legacy=legacy )
   config.activityPollingInterval = interval

def noIpMfibActivityInterval( mode, args, legacy=False ):
   config = mfibConfigRootFromMode( mode, legacy=legacy )
   config.activityPollingInterval = config.activityPollingIntervalDefault

class MfibActivityInterval( CliCommandClass ):
   syntax = 'activity polling-interval INTERVAL'
   noOrDefaultSyntax = 'activity polling-interval ...'
   data = {
      'activity': activityKwMatcher,
      'polling-interval': intervalKwMatcher,
      'INTERVAL': intervalMatcher6
   }

   handler = setIpMfibActivityInterval
   noOrDefaultHandler = noIpMfibActivityInterval

RouterMulticastCliLib.RouterMulticastNonVrfAfModelet.addCommandClass(
   MfibActivityInterval )

class MfibActivityIntervalLegacy( CliCommandClass ):
   syntax = 'ip mfib activity polling-interval INTERVAL'
   noOrDefaultSyntax = 'ip mfib activity polling-interval ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'mfib': mfibNode,
      'activity': activityKwMatcher,
      'polling-interval': intervalKwMatcher,
      'INTERVAL': intervalMatcher
   }

   handler = legacyCliCallback( setIpMfibActivityInterval )
   noOrDefaultHandler = legacyCliCallback( noIpMfibActivityInterval )

# BUG153820: this probably needs to be made per-vrf. If so, the command needs to be
# added to RouterMulticastSharedModelet
RouterMulticastCliLib.RouterMulticastMode.addCommandClass(
   MfibActivityIntervalLegacy )
BasicCli.GlobalConfigMode.addCommandClass( MfibActivityIntervalLegacy )

#-------------------------------------------------------------------------------
# show commands
#-------------------------------------------------------------------------------
warningHeader = "======================================================\n" \
                "WARNING: Some of the routes are not programmed in     \n" \
                "hardware, and they are marked with '*'.               \n" \
                "======================================================"

def isMatchingSG( s1, g1, s2, g2 ):
   """Determine whether one (s,g) matches a second (s,g), taking into
      account possible wildcard values in the second (s,g).  If either
      of the values of the second (s,g) is "None", it is considered a
      wildcard and matches any corresponding value in the first (s,g).
      The all-zeros IPv4 address is also considered a legal wildcard
      for the second source value only. Returns True if the two (s,g)
      values match and False if they do not match."""

   if g2 and g2 != g1:
      return False
   if s2 and s2 != "0.0.0.0" and s2 != s1:
      return False
   return True

def countRouteType (
      vrfName, sKey, key, fastdrop, sg, star_g, prefix, iifFrr, failure, af ):
   """Decide the type of a given route, return with the incrementation on
      the count number of correponding types (fastdrop/(S,G)/(*,G)/prefix).
      If an argument of a type is -1, it won't be taken a consideration"""
   # get the source/group IP of this route from key
   source, group = key.s, key.g

   # Verify the IP address for both group and source
   if not group.isMulticast or source.isMulticast:
      return fastdrop, sg, star_g, prefix, iifFrr

   # In most cases, the routes is in normal state
   state = 0
   # Verify whether the route is in unprogrammed state
   mfibHwStatus = _mfibHwVrfStatus( vrfName, af )
   if mfibHwStatus is None:
      return fastdrop, sg, star_g, prefix, iifFrr

   rs = mfibHwStatus.route.get( key )
   failed = rs and rs.hwFlags
   # If one route is unprogrammed, take it in another record
   if failed:
      state = 1
      failure = True

   # get the length of prefix on both source and group
   sPrefixLen = source.len
   gPrefixLen = group.len

   # if group is "224.0.0.0/4", the route is multicast
   if group.af == 'ipv4' and group.stringValue == '224.0.0.0/4':
      # The source of prefix routing should not be S/32
      # The prefix should be considered
      if sPrefixLen < 32 and prefix >= 0:
         prefix[ state ] += 1

   elif gPrefixLen == 32 or gPrefixLen == 128:
      # if source is S/32, the route is (S,G)
      if sPrefixLen == 32 or sPrefixLen == 128:
         sg[ state ] += 1
      # if source is ::/0, the route is (*,G)
      elif star_g >= 0 and source.isDefaultRoutePrefix:
         star_g[ state ] += 1

      mfibSmashColl = _mfibStatus( vrfName, af )
      if mfibSmashColl and sKey in mfibSmashColl.smashPtr:
         mfibSmash = mfibSmashColl.smashPtr[ sKey ]
         # acquire the interface of route
         r = mfibSmash.route.get( key )
         # depending on the static fastdrop on iif, increase/keep fastdrop
         if r.fastdrop != '':
            fastdrop[state] += 1
         if r.iifFrr != '':
            iifFrr[ state ] += 1

   return fastdrop, sg, star_g, prefix, iifFrr, failure

def displayRouteCount( sg, star_g, fastdrop, prefix, iifFrr, failure ):
   # print required count of (S,G), (*,G), fastdrop, iifFrr, prefix
   print "(S,G) routes: ", sg[ 0 ]
   if star_g[ 0 ] >= 0:
      print "(*,G) routes: ", star_g[ 0 ]
   if fastdrop[ 0 ] >= 0:
      print "Fastdrop routes: ", fastdrop[ 0 ]
   if iifFrr[ 0 ] > 0:
      print "MoFrr routes: ", iifFrr[ 0 ]
   if prefix[ 0 ] >= 0:
      print "Prefix routes: ", prefix[ 0 ]
   if failure:
      print "***Unprogrammed Routes Count:"
      print "Unprogrammed (S,G) routes:", sg[ 1 ]
      if star_g[ 1 ] >= 0:
         print "Unprogrammed (*,G) routes:", star_g[ 1 ]
      if fastdrop[ 1 ] >= 0:
         print "Unprogrammed fastdrop routes:", fastdrop[ 1 ]
      if iifFrr[ 1 ] > 0:
         print "Unprogrammed MoFrr routes: ", iifFrr[ 1 ]
      if prefix[ 1 ] >= 0:
         print "Unprogrammed prefix routes:", prefix[ 1 ]

def initMfibDirSm():
   global mfibDirSm, mfibDirSm6
   mfibDirSm = Tac.newInstance( "Smash::Multicast::Fib::MfibDirSm",
                                mfibDir, mfibSmashDir,
                                _entityManager.cEntityManager() )
   mfibDirSm6 = Tac.newInstance( "Smash::Multicast::Fib::MfibDirSm",
                                mfibDir6, mfibSmashDir6,
                                _entityManager.cEntityManager() )

def showIpMfib( mode, args ):
   """Display the contents of the multicast fib, optionally limiting
      output to maching sources and groups"""
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   src = args.get( 'SOURCE' )
   grp = args.get( 'GROUP' )
   counter = args.get( 'counters' )
   af = AddressFamily.ipv6 if 'ipv6' in args else AddressFamily.ipv4

   defaultAddr = Tac.Value( "Arnet::IpGenAddr" )

   sourceGenAddr = Arnet.IpGenAddr( str( src ) ) if src is not None else defaultAddr
   groupGenAddr = Arnet.IpGenAddr( str( grp ) ) if grp is not None else defaultAddr

   # If only one argument is specified, check to see if argument is a source
   if not( src or groupGenAddr.isMulticast ):
      sourceGenAddr = groupGenAddr
      groupGenAddr = defaultAddr

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()

   mfibSmashColl = _mfibStatus( vrfName, af )

   if not mfibSmashColl:
      mfibSmashColl = Tac.newInstance( "Smash::Multicast::Fib::MfibSmashColl",
                                       vrfName )

   if af == AddressFamily.ipv4:
      mfibHwStatus = mfibHardwareStatus
   else:
      mfibHwStatus = mfib6HardwareStatus
   LazyMount.force( mfibHwStatus )

   showMfibRoutes = Tac.newInstance( "McastCommonCapiModel::ShowMfib",
         mfibHwStatus, mfibVrfConfig, mfibCounterStatus, mfibSmashColl )

   if bool( counter ) and showMfibCounterHook.extensions():
      showMfibCounterHook.notifyExtensions( showMfibRoutes )

   showMfibRoutes.render( fd, fmt, groupGenAddr, sourceGenAddr,
                          vrfName, bool( counter ) )
   return MfibGroups

def showIpMfibByPimMode( mode, args ):
   """Display the contents of the mode-specific multicast fib, optionally limiting
      output to maching sources and groups"""
   pimMode = args.get( 'PIM_MODE' )
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   src = args.get( 'SOURCE' )
   grp = args.get( 'GROUP' )
   counter = args.get( 'counters' )
   af = AddressFamily.ipv6 if 'ipv6' in args else AddressFamily.ipv4

   defaultAddr = Tac.Value( "Arnet::IpGenAddr" )

   sourceGenAddr = Arnet.IpGenAddr( str( src ) ) if src is not None else defaultAddr
   groupGenAddr = Arnet.IpGenAddr( str( grp ) ) if grp is not None else defaultAddr

   # If only one argument is specified, check to see the argument is a source
   if not( src or groupGenAddr.isMulticast ):
      sourceGenAddr = groupGenAddr
      groupGenAddr = defaultAddr

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()

   if pimMode == 'sparse-mode':
      pimModeSuffix = 'sparsemode'
   elif pimMode == 'bidirectional':
      pimModeSuffix = 'bidir'
   elif pimMode == 'static':
      pimModeSuffix = 'static'
   else:
      assert False, 'Invalid PIM mode'

   smashUrl = Tac.Type( "Smash::Multicast::Fib::Status" ).mountPath( af, vrfName,
                                                                    pimModeSuffix )
   mfibSmash = _smashMount.getTacEntity( smashUrl )
   if not mfibSmash:
      mfibSmash = Tac.newInstance( "Smash::Multicast::Fib::Status", "test" )

   LazyMount.force( mfibHardwareStatus )

   showMfibRoutes = Tac.newInstance( "McastCommonCapiModel::ShowMfibProtocol",
         mfibHardwareStatus, mfibVrfConfig,
         mfibCounterStatus, mfibSmash )

   if bool( counter ) and showMfibCounterHook.extensions():
      showMfibCounterHook.notifyExtensions( showMfibRoutes )

   showMfibRoutes.render( fd, fmt, groupGenAddr, sourceGenAddr,
                          vrfName, bool( counter ) )
   return MfibGroups

def showIpMfibCount( mode, args ):
   """Display the contents of the multicast fib, optionally limiting
      output to maching sources and groups"""
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   source = args.get( 'SOURCE' )
   group = args.get( 'GROUP' )
   af = AddressFamily.ipv6 if 'ipv6' in args else AddressFamily.ipv4

   if mfibHardwareStatus.hwResourceFull:
      print warningHeader

   mfibSmash = _mfibStatus( vrfName, af )
   if not mfibSmash:
      return

   if mfibSmash.routes():
      print "Activity poll time:", mfibHardwareStatus.pollingInterval, "seconds"

   # set up the argument for different conditions:
   # 1. no source & no group; 2. only source; 3. only group; 4. both ip's
   # 2-d array for both normal and unprogrammed states
   fastdrop, sg, star_g, prefix, iifFrr = [ 0, 0 ], [ 0, 0 ], [ 0, 0 ], \
                                                            [ -1, -1 ], [ 0, 0 ]
   failure = False

   # Group and source are required to exist simultaneously
   if group and source:
      if af == 'ipv4' and source == '0.0.0.0':
         source = None
      elif af == 'ipv6' and source.isZero:
         source = None

      k = sourceGroupKey( source, group )
       # with fix source and group address, only (S,G) route possible
      star_g = [ -1, -1 ]

      if mfibSmash.route( k ) != Tac.Value( "Smash::Multicast::Fib::Route" ):
         # invalid route has been filter by the previous if statement
         fastdrop, sg, star_g, prefix, iifFrr, failure = countRouteType(vrfName,
                                                      mfibSmash.smashKey( k ), k,
                                                      fastdrop, sg, star_g,
                                                      prefix, iifFrr, failure, af)
   else:
      # With one given IP address, determine whether it is source or group
      if group and \
            not ( Arnet.IpGenAddr( group ) if af == 'ipv4' else group ).isMulticast:
         source = group
         group = None

      if source:
         star_g = [ -1, -1 ]
      elif not group:
         prefix = [ 0, 0 ]
      for ptr, status in mfibSmash.smashPtr.iteritems():
         for routeKey in status.route:
            src, grp = routeKey.s, routeKey.g
            if not ( source or group ) or \
               ( af == 'ipv4' and \
                  ( source == src.v4Addr or group == grp.v4Addr ) ) or \
               ( af == 'ipv6' and \
                  ( source == src.v6Addr or group == grp.v6Addr ) ):
               fastdrop, sg, star_g, prefix, iifFrr, failure = countRouteType(
                  vrfName, ptr, routeKey,
                  fastdrop, sg, star_g,
                  prefix, iifFrr, failure, af )

   displayRouteCount( sg, star_g, fastdrop, prefix, iifFrr, failure )

def showIpMfibByPimModeCount( mode, args ):
   """Display the contents of the mode-specific multicast fib, optionally limiting
      output to maching sources and groups"""
   pimMode = args.get( 'PIM_MODE' )
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   source = args.get( 'SOURCE' )
   group = args.get( 'GROUP' )
   af = AddressFamily.ipv6 if 'ipv6' in args else AddressFamily.ipv4

   if mfibHardwareStatus.hwResourceFull:
      print warningHeader

   mfibSmashColl = _mfibStatus( vrfName, af )
   if not mfibSmashColl:
      return

   if pimMode == 'sparse-mode':
      pimModeSuffix = 'sparsemode'
   elif pimMode == 'bidirectional':
      pimModeSuffix = 'bidir'
   elif pimMode == 'static':
      pimModeSuffix = 'static'
   else:
      assert False, 'Invalid PIM mode'

   smashUrl = Tac.Type( "Smash::Multicast::Fib::Status" ).mountPath( af, vrfName,
                                                                    pimModeSuffix )
   mfibSmash = _smashMount.getTacEntity( smashUrl )
   if not mfibSmash:
      return

   if mfibSmash.route:
      print "Activity poll time:", mfibHardwareStatus.pollingInterval, "seconds"
   # set up the argument for different conditions:
   # 1. no source & no group; 2. only source; 3. only group; 4. both ip's
   # 2-d array for both normal and unprogrammed states
   fastdrop, sg, star_g, prefix, iifFrr = [ 0, 0 ], [ 0, 0 ], [ 0, 0 ], \
                                                         [ -1, -1 ], [ 0, 0 ]
   failure = False

   if group and source:
      if af == 'ipv4' and source == '0.0.0.0':
         source = None
      elif af == 'ipv6' and source.isZero:
         source = None

      k = sourceGroupKey( source, group )
      # with fix source and group address, only (S,G) route possible
      star_g = [-1, -1]
      if k in mfibSmash.route:
         # invalid route has been filter by the previous if statement
         fastdrop, sg, star_g, prefix, iifFrr, failure = countRouteType(vrfName,
                                                      mfibSmash, k, fastdrop,
                                                      sg, star_g,
                                                      prefix, iifFrr, failure, af)
   else:
      # For one IP address, determine whether it is source or group
      if group and \
            not ( Arnet.IpGenAddr( group ) if af == 'ipv4' else group ).isMulticast:
         source = group
         group = None

      if source:
         star_g = [ -1, -1 ]
      elif not group:
         prefix = [ 0, 0 ]

      for routeKey in mfibSmash.route:
         src, grp = routeKey.s, routeKey.g
         if not ( source or group ) or \
            ( af == 'ipv4' and \
               ( source == src.v4Addr or group == grp.v4Addr ) ) or \
            ( af == 'ipv6' and \
               ( source == src.v6Addr or group == grp.v6Addr ) ):
            fastdrop, sg, star_g, prefix, iifFrr, failure = countRouteType(
               vrfName, mfibSmash, routeKey,
               fastdrop, sg, star_g,
               prefix, iifFrr, failure, af )

   displayRouteCount( sg, star_g, fastdrop, prefix, iifFrr, failure )

#-------------------------------------------------------------------------------
# show ip mfib [ vrf < vrf-name > ] summary
#-------------------------------------------------------------------------------
def showIpMfibSummary( mode, args ):
   """Display the summary of multicast FIB."""
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   af = AddressFamily.ipv6 if 'ipv6' in args else AddressFamily.ipv4

   mroutesCount = 0
   mfibSmash = _mfibStatus( vrfName, af )
   mfibSmashConfig = _mfibConfig( vrfName, af=af )
   if mfibSmash:
      mroutesCount = mfibSmash.routes()
   else:
      # VRF is not present. Just return
      return

   fastdropRoutesCount = 0

   mfibHwStatus = _mfibHwVrfStatus( vrfName, af )
   if mfibHwStatus:
      # fastDropCount comes from vrfStatus::fastDropCount
      # which is mirror of MrouteTable:fastdropCount
      fastdropRoutesCount = mfibHwStatus.fastdropCount

   print "Number of multicast routes: %d" % ( mroutesCount )
   print "Number of fastdrop routes : %d" % ( fastdropRoutesCount )
   if fastdropRoutesCount >= mfibSmashConfig.maxFastdrops:
      print "Warning: number of fastdrops has reached maximum: %d " % \
         fastdropRoutesCount

#-------------------------------------------------------------------------------
# clear commands
#-------------------------------------------------------------------------------

vrfExprFactory = VrfCli.VrfExprFactory( helpdesc='VRF name' )
mfibClearNode = Node( mfibKwMatcher, guard=mcastGenRoutingSupportedGuard )

class ClearMfibBase( CliCommandClass ):
   baseData = {
      'clear': CliToken.Clear.clearKwNode,
      'VRF': vrfExprFactory,
      'route': routeMatcherForConfig
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      af = args.get( 'AF', 'ipv4' )
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      if not vrfName in mfibConfigRoot( af=af ).config:
         return
      mvc = mfibConfig( af, vrfName )
      if mvc.clearMfibCount < 255:
         mvc.clearMfibCount += 1
      else:
         mvc.clearMfibCount = 0
      mvc.fastdrop = True

class ClearMfib( ClearMfibBase ):
   syntax = 'clear mfib AF [ VRF ] route'
   data = {
      'mfib': mfibClearNode,
      'AF': addressFamilyExprForClear,
   }
   data.update( ClearMfibBase.baseData )

BasicCli.EnableMode.addCommandClass( ClearMfib )

class ClearMfibLegacy( ClearMfibBase ):
   syntax = 'clear ip mfib [ VRF ] route'
   data = {
      'ip': CliToken.Ip.ipMatcherForClear,
      'mfib': mfibNode,
   }
   data.update( ClearMfibBase.baseData )

BasicCli.EnableMode.addCommandClass( ClearMfibLegacy )

class ClearMfibFastdropBase( CliCommandClass ):
   baseData = {
      'clear': CliToken.Clear.clearKwNode,
      'VRF': vrfExprFactory,
      'fastdrop': fastdropKwMatcher
   }

   @staticmethod
   def handler( mode, args ):
      af = args.get( 'AF', 'ipv4' )
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      if not vrfName in mfibConfigRoot( af=af ).config:
         return
      mvc = mfibConfig( af, vrfName )
      mvc.clearFastdrop = Tac.now()

class ClearMfibFastdrop( ClearMfibFastdropBase ):
   syntax = 'clear mfib AF [ VRF ] fastdrop'
   data = {
      'mfib': mfibClearNode,
      'AF': addressFamilyExprForClear
   }
   data.update( ClearMfibFastdropBase.baseData )

BasicCli.EnableMode.addCommandClass( ClearMfibFastdrop )

class ClearMfibFastdropLegacy( ClearMfibFastdropBase ):
   syntax = 'clear ip mfib [ VRF ] fastdrop'
   data = {
      'ip': CliToken.Ip.ipMatcherForClear,
      'mfib': mfibNode
   }
   data.update( ClearMfibFastdropBase.baseData )

BasicCli.EnableMode.addCommandClass( ClearMfibFastdropLegacy )

class ClearMulticastCounter( CliCommandClass ):
   syntax = """clear ( ip multicast [ VRF ] counters [ GROUP [ SOURCE ] ] )
                | ( ipv6 multicast [ VRF ] counters [ GROUP6 [ SOURCE6 ] ] )"""

   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ip': CliToken.Ip.ipMatcherForClear,
      'ipv6': CliToken.Ipv6.ipv6MatcherForClear,
      'multicast': multicastNode,
      'VRF': vrfExprFactory,
      'counters': counterKwMatcher,
      'GROUP': IpAddrMatcher( "Group address" ),
      'SOURCE': IpAddrMatcher( "Source address" ),
      'GROUP6': Ip6AddrMatcher( "IPv6 Group address" ),
      'SOURCE6': Ip6AddrMatcher( "IPv6 Source address" )
   }

   @staticmethod
   def handler( mode, args ):
      group = args.get( 'GROUP' ) or args.get( 'GROUP6' )
      source = args.get( 'SOURCE' ) or args.get( 'SOURCE6' )
      vrfName = args.get( 'VRF', DEFAULT_VRF )

      if group and source:
         routeKey = sourceGroupKey( source, group )
      else:
         routeKey = None

      if clearMfibCounterHook.extensions() == []:
         mvc = _mfibConfig( vrfName )
         if mvc:
            routeKey = Tac.Value( "Routing::Multicast::Fib::IpGenRouteKey",
               Arnet.IpGenPrefix( "::/0" ), Arnet.IpGenPrefix( "::/0" ) )
            del mvc.clearCounterRoute[ routeKey ]
            mvc.clearCounterRoute[ routeKey ] = True
      else:
         clearMfibCounterHook.notifyExtensions( vrfName, routeKey )

BasicCli.EnableMode.addCommandClass( ClearMulticastCounter )

#-------------------------------------------------------------------------------
# Multicast routing hook for "show ip" command in Ira
#-------------------------------------------------------------------------------
def getMcastRoutingSupport():
   mCastRoutingSupport = {}
   if routingHardwareStatus.multicastRoutingSupported:
      mCastRoutingSupport[ 'v4' ] = mfibVrfConfig.routing
   if routing6HardwareStatus.multicastRoutingSupported:
      mCastRoutingSupport[ 'v6' ] = mfib6VrfConfig.routing
   return mCastRoutingSupport

IraIpCli.mCastRoutingHook.addExtension( getMcastRoutingSupport )
#-------------------------------------------------------------------------------
# show ip mfib [ vrf <vrfName> ] df
#-------------------------------------------------------------------------------
def showIpMfibBidirDf( mode, args ):
   """Display df profile per interface"""
   vrfName = args.get( "VRF", DEFAULT_VRF )
   intf = args.get( "INTF" )

   model = MrouteDfModel.DfProfile()
   if vrfName not in bidirVrf.bidirEnabledVrf:
      model.bidirEnabled = False
      return model

   if vrfName not in bidirStatus:
      bidirStatus[ vrfName ] = \
          SmashLazyMount.mount( _entityManager,
                             'routing/multicast/bidirstatus/%s'%vrfName, \
                                   "Smash::Multicast::Fib::BidirStatus", \
                                SmashLazyMount.mountInfo( 'reader' ) )

   if intf:
      df = bidirStatus[ vrfName ].dfProfile.get( intf.name )
      if not df:
         return model
      model.dfProfiles[ df.key ] = long( df.dfBitmap )
      return model

   for df in bidirStatus[ vrfName ].dfProfile:
      model.dfProfiles[ df ] = long( bidirStatus[vrfName].dfProfile[df].dfBitmap )
   return model

#-------------------------------------------------------------------------------
# show ip mfib [ vrf <vrfName> ] rpa
#-------------------------------------------------------------------------------
def showIpMfibBidirRpa( mode, args ):
   """Display rpa index for bidir perfix"""
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   prefix = args.get( 'PREFIX' )

   model = MrouteDfModel.BidirGroup()

   if vrfName not in bidirVrf.bidirEnabledVrf:
      model.bidirEnabled = False
      return model

   if vrfName not in bidirStatus:
      bidirStatus[ vrfName ] = \
          SmashLazyMount.mount( _entityManager,
                             'routing/multicast/bidirstatus/%s'%vrfName, \
                                   "Smash::Multicast::Fib::BidirStatus", \
                                SmashLazyMount.mountInfo( 'reader' ) )

   if prefix:
      if not isValidPrefixWithError( mode, prefix ):
         return model
      group = bidirStatus[ vrfName ].bidirGroup.get( Arnet.IpGenPrefix( prefix ) )
      if not group:
         return model
      model.groups[ group.key.stringValue ] = group.rpa
   else:
      for group in bidirStatus[ vrfName ].bidirGroup:
         model.groups[ group.stringValue ] = \
                        bidirStatus[vrfName].bidirGroup[group].rpa

   return model

#------------------------------------------------------------------------------
# Multicast routing hook for "show vrf" command in Ira
#------------------------------------------------------------------------------
def showVrf( mode, vrfName, af ):

   #check if vrf exists
   if vrfName not in VrfCli.getAllVrfNames():
      return False

   #check if multicasting is supported
   rhs = routingHardwareStatus if af == 'v4' else routing6HardwareStatus
   if not rhs.multicastRoutingSupported:
      return False

   mvc = mfibConfig( AddressFamily.ipv4 if af == 'v4' else AddressFamily.ipv6,
                     vrfName )
   if mvc and mvc.routing:
      return True

   return False

IraVrfCli.showMCastHook.addExtension( showVrf )

#------------------------------------------------------------------------------------
# Config tokens
#------------------------------------------------------------------------------------

class RouterMulticastVrf( CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
      'VRF': VrfCli.VrfExprFactory( helpdesc='Configure multicast for a VRF' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args[ 'VRF' ]
      if vrfName in VRFNAMES_RESERVED:
         mode.addError( "VRF name %s is reserved." % vrfName )
         return
      _mrouteVrfCreation( vrfName, True )

      if vrfName not in VrfCli.getVrfNames():
         mode.addWarning( "VRF name %s is not defined." % vrfName )

      childMode = mode.childMode( RouterMulticastCliLib.RouterMulticastVrfMode,
                                     vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )
      enterSubmode( "vrf", vrfName=vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args[ 'VRF' ]
      if vrfName in VRFNAMES_RESERVED:
         return
      t0( "Deleting vrf:", vrfName )
      _mrouteVrfCreation( vrfName, False )
      deleteSubmode( "vrf", vrfName=vrfName )

RouterMulticastCliLib.RouterMulticastMode.addCommandClass( RouterMulticastVrf )

class RouterMulticast( CliCommandClass ):
   syntax = 'router multicast'
   noOrDefaultSyntax = syntax
   data = {
      'router': routerMatcherForConfig,
      'multicast': multicastNode
   }

   @staticmethod
   def handler( mode, args ):
      minSupportVersion( _multicastLegacyConfig.routerMode )
      for hook in legacyConfigHook.extensions():
         hook()
      childMode = mode.childMode( RouterMulticastCliLib.RouterMulticastMode )
      mode.session_.gotoChildMode( childMode )
      RouterMulticastMode.enterMode()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
         configRoot = mfibConfigRoot( af=af )
         # delete non-default config all for all VRFs
         for vrfName in configRoot.config:
            t0( "Deleting vrf:", vrfName )
            _mrouteVrfCreation( vrfName, False )
         configRoot.reset()

      _multicastLegacyConfig.legacy = True
      _multicastLegacyConfig.version = _multicastLegacyConfig.globalMode
      for hook in legacyConfigHook.extensions():
         hook()
      RouterMulticastMode.deleteMode()

BasicCli.GlobalConfigMode.addCommandClass( RouterMulticast )

ipv6KwMatcher = KeywordMatcher( 'ipv6',
   helpdesc='Configure multicast for address family IPv6' )
ipv6Node = Node( ipv6KwMatcher, guard=mcast6RoutingSupportedGuard )

class RouterMulticastIpv6( CliCommandClass ):
   syntax = 'ipv6'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': ipv6Node
   }

   @staticmethod
   def handler( mode, args ):
      minSupportVersion( _multicastLegacyConfig.ipMode )
      vrfName = RouterMulticastCliLib.getVrfNameFromMode( mode )
      if vrfName == DEFAULT_VRF:
         childMode = mode.childMode(
               RouterMulticastCliLib.RouterMulticastDefaultIpv6Mode )
      else:
         childMode = mode.childMode(
               RouterMulticastCliLib.RouterMulticastVrfIpv6Mode,
               vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )
      enterSubmode( "af", vrfName=vrfName, af=AddressFamily.ipv6 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = RouterMulticastCliLib.getVrfNameFromMode( mode )
      vrfConfig = mfibConfig( af=AddressFamily.ipv6, vrfName=vrfName )
      vrfConfig.reset()

      mfibConfigColl = mfibConfigRoot( af=AddressFamily.ipv4 ).config.values()
      mfib6ConfigColl = mfibConfigRoot( af=AddressFamily.ipv6 ).config.values()
      if all( config.isDefault() for config in \
            chain( mfibConfigColl, mfib6ConfigColl ) ):
         _multicastLegacyConfig.version = _multicastLegacyConfig.routerMode

      deleteSubmode( "af", vrfName=vrfName, af=AddressFamily.ipv6 )

RouterMulticastCliLib.RouterMulticastSharedModelet.addCommandClass(
   RouterMulticastIpv6 )

ipv4KwMatcher = KeywordMatcher( 'ipv4',
   helpdesc='Configure multicast for address family IPv4' )
ipv4Node = Node( ipv4KwMatcher, guard=mcastRoutingSupportedGuard )

class RouterMulticastIpv4( CliCommandClass ):
   syntax = 'ipv4'
   noOrDefaultSyntax = syntax
   data = {
      'ipv4': ipv4Node
   }

   @staticmethod
   def handler( mode, args ):
      minSupportVersion( _multicastLegacyConfig.ipMode )
      vrfName = RouterMulticastCliLib.getVrfNameFromMode( mode )
      if vrfName == DEFAULT_VRF:
         childMode = mode.childMode(
               RouterMulticastCliLib.RouterMulticastDefaultIpv4Mode )
      else:
         childMode = mode.childMode(
               RouterMulticastCliLib.RouterMulticastVrfIpv4Mode,
               vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )
      enterSubmode( "af", vrfName=vrfName, af=AddressFamily.ipv4 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = RouterMulticastCliLib.getVrfNameFromMode( mode )
      vrfConfig = mfibConfig( af=AddressFamily.ipv4, vrfName=vrfName )
      vrfConfig.reset()

      mfibConfigColl = mfibConfigRoot( af=AddressFamily.ipv4 ).config.values()
      mfib6ConfigColl = mfibConfigRoot( af=AddressFamily.ipv6 ).config.values()
      if all( config.isDefault() for config in \
            chain( mfibConfigColl, mfib6ConfigColl ) ):
         _multicastLegacyConfig.version = _multicastLegacyConfig.routerMode

      deleteSubmode( "af", vrfName=vrfName, af=AddressFamily.ipv4 )

RouterMulticastCliLib.RouterMulticastSharedModelet.addCommandClass(
   RouterMulticastIpv4 )

def Plugin ( entityManager ):
   global mfibHardwareStatus, mfib6HardwareStatus, mfibDir, mfibDir6
   global routingHardwareStatus, routing6HardwareStatus
   global mfibCounterStatus
   global ipStatus, mfibVrfConfig, mfib6VrfConfig, _entityManager
   global routingVrfInfoDir, routing6VrfInfoDir
   global bidirStatus, bidirVrf
   global _smashMount
   global _multicastLegacyConfig

   _entityManager = entityManager
   _smashMount = SharedMem.entityManager( sysdbEm=entityManager )

   #Af independent Config mounts
   configTypes = [ MfibVrfConfig, ]
   RouterMulticastCliLib.doConfigMounts( entityManager, configTypes )

   _multicastLegacyConfig = ConfigMount.mount( entityManager,
      'routing/multicast/legacyconfig',
      'Routing::Multicast::MulticastLegacyConfig', 'w' )
   mfibVrfConfig = ConfigMount.mount(
      entityManager, MrouteConsts.routingMulticastVrfConfigSysdbPath,
      'Routing::Multicast::Fib::VrfConfig', 'w' )
   mfib6VrfConfig = ConfigMount.mount(
      entityManager, 'routing6/multicast/vrf/config',
      'Routing::Multicast::Fib::VrfConfig', 'w' )

   # XXX Cli writing into status does look unusual, but for now we have no agent
   # to copy multicast static routes from config to status, and we still want
   # to have static routes.

   mfibHardwareStatus = LazyMount.mount( entityManager,
      'routing/hardware/multicast/status',
      'Routing::Multicast::Fib::Hardware::Status', 'r' )
   mfib6HardwareStatus = LazyMount.mount( entityManager,
      'routing6/hardware/multicast/status',
      'Routing::Multicast::Fib::Hardware::Status', 'r' )

   mg = entityManager.mountGroup()
   mfibDir = mg.mount( 'routing/multicast/fib', 'Tac::Dir', 'ri' )
   mfibDir6 = mg.mount( 'routing6/multicast/fib', 'Tac::Dir', 'ri' )
   mg.close( initMfibDirSm )

   mfibCounterStatus = SmashLazyMount.mount( entityManager,
      'routing/multicast/counterStatus', "Routing::Multicast::Fib::CounterStatus",
      SmashLazyMount.mountInfo( 'reader' ) )

   bidirStatus = {}
   bidirStatus[ DEFAULT_VRF ] = \
      SmashLazyMount.mount( entityManager, 'routing/multicast/bidirstatus/default',
         "Smash::Multicast::Fib::BidirStatus", SmashLazyMount.mountInfo( 'reader' ) )

   bidirVrf = LazyMount.mount( entityManager, "routing/multicast/bidirstatus",
      "Routing::Multicast::Fib::BidirStatus", "r" )

   routingVrfInfoDir = \
       LazyMount.mount( entityManager, "routing/vrf/routingInfo/status",
                        "Tac::Dir", "r" )
   routing6VrfInfoDir = \
       LazyMount.mount( entityManager, "routing6/vrf/routingInfo/status",
                        "Tac::Dir", "r" )

   routingHardwareStatus = LazyMount.mount( entityManager,
      "routing/hardware/status", "Routing::Hardware::Status", "r" )
   routing6HardwareStatus = LazyMount.mount( entityManager,
      "routing6/hardware/status", "Routing6::Hardware::Status", "r" )
   ipStatus = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )

   # xxx I don't know if 22 is right
   IntfCli.Intf.registerDependentClass( MfibIntf, priority=22 )
