#!/usr/bin/env python
# Copyright (c) 2010, 2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli, ConfigMount, LazyMount
import Tac, Arnet
import CliToken.Ip
import IntfCli
from IraIpRouteCliLib import isValidPrefixWithError
from IraIp6RouteCliLib import isValidIpv6PrefixWithError as isValidPrefix6WithError
from IpLibConsts import DEFAULT_VRF
from RoutingConsts import defaultStaticRoutePreference
import RouterMulticastCliLib
from RouterMulticastCliLib import legacyCliCallback
import MrouteCli
import Tracing
from McastCommonCli import mcastRoutingSupportedGuard
import Toggles.McastCommonToggleLib
import CliCommand
import IpAddrMatcher, Ip6AddrMatcher
import CliMatcher

traceHandle = Tracing.Handle( "MribConfigCli" )
t0 = traceHandle.trace0

AddressFamily = Tac.Type( "Arnet::AddressFamily" )

mribConfigColl = {}
_allVrfConfig = None

def _mribVrfCreationHook( vrfName ):
   t0( "_mribVrfCreationHook : %s" % vrfName )
   if vrfName != DEFAULT_VRF:
      for af in [ "ipv4", "ipv6" ]:
         if vrfName not in mribConfigColl[ af ].vrfConfig:
            mcc = mribConfigColl[ af ].newVrfConfig( vrfName )
            assert mcc is not None

def _mribVrfDeletionHook( vrfName ):
   t0( "_mribVrfDeletionHook :  %s" % vrfName )
   if vrfName == DEFAULT_VRF:
      for af in [ "ipv4", "ipv6" ]:
         mribConfigColl[ af ].vrfConfig[ vrfName ].route.clear()
      return

   for af in [ "ipv4", "ipv6" ]:
      if vrfName in mribConfigColl[ af ].vrfConfig:
         mribConfigColl[ af ].vrfConfig[ vrfName ].route.clear()
         del mribConfigColl[ af ].vrfConfig[ vrfName ]

def isValidPrefix( mode, prefix, af ):
   if af == AddressFamily.ipv4:
      return isValidPrefixWithError( mode, prefix )
   elif af == AddressFamily.ipv6:
      return isValidPrefix6WithError( mode, prefix )
   return False

rpfKwMatcher = CliMatcher.KeywordMatcher( 'rpf', helpdesc='RPF config commands' )

mrouteKwMatcher = CliMatcher.KeywordMatcher( 'mroute',
      helpdesc='Manage IP static multicast routes' )
mrouteNode = CliCommand.Node( matcher=mrouteKwMatcher,
                              guard=mcastRoutingSupportedGuard )

routeKwMatcher = CliMatcher.KeywordMatcher( 'route',
   helpdesc='Manage IP static multicast routes' )
routeNode = CliCommand.Node( matcher=routeKwMatcher,
      guard=mcastRoutingSupportedGuard )

prefixExpr = IpAddrMatcher.ipPrefixExpr( 'Source prefix',
                                         'Source prefix mask',
                                         'Source address with prefix',
                                         partial=True )

prefix6Expr = Ip6AddrMatcher.Ip6PrefixValidMatcher( 'IPv6 address prefix',
   overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )

addressMatcher = IpAddrMatcher.IpAddrMatcher( "Upstream router's address" )
address6Matcher = Ip6AddrMatcher.Ip6AddrMatcher( "Upstream router's address" )

prefMatcher = CliMatcher.IntegerMatcher( 1, 255,
                                 helpdesc='Administrative distance for this route' )

def ipMroute( mode, args, legacy=False ):
   prefix = args[ 'PREFIX' ]
   rpf = args.get( 'INTF' ) or args.get( 'ADDR' )
   preference = args.get( 'PREF' )
   af = RouterMulticastCliLib.getAddressFamilyFromMode( mode, legacy=legacy )
   if not isValidPrefix( mode, prefix, af ):
      return

   arnetPrefix = Arnet.IpGenPrefix( str( prefix ) )
   if arnetPrefix.isMulticast:
      mode.addError( "Destination prefix must be unicast" )
      return

   vrfName = RouterMulticastCliLib.getVrfNameFromMode( mode )
   if not vrfName :
      vrfName = DEFAULT_VRF
   if vrfName != DEFAULT_VRF and vrfName not in _allVrfConfig.vrf:
      mode.addWarning( "vrf name %s is not defined." % vrfName )

   mribConfig = mribConfigColl[ af ].vrfConfig[ vrfName ]
   assert mribConfig.route is not None

   if isinstance( rpf, ( str, Tac.Type( "Arnet::Ip6Addr" ) ) ):
      # Its an address
      rpfType = 'address'
      address = Arnet.IpGenAddr( str( rpf ) )
      intfName = ""
   else:
      # Its an interface
      intf = rpf

      if not intf.config():
         mode.addError( "Interface %s does not exist" % intf.name )
         return

      if not intf.routingSupported():
         mode.addError( "Interface %s is not routable" % intf.name )
         return

      rpfType = 'intf'
      address = Tac.Value( "Arnet::IpGenAddr" )
      address.af = af

      intfName = intf.name

   if preference is None:
      preference = defaultStaticRoutePreference

   rkPrefix = mribConfig.route.get( arnetPrefix )
   if not rkPrefix:
      rkPrefix = mribConfig.route.newMember( arnetPrefix )
   rkPreference = rkPrefix.pref.get( preference )
   if not rkPreference:
      rkPreference = rkPrefix.pref.newMember( preference )
   RPF = Tac.Value( "Routing::Multicast::Rib::Rpf",
                    rpfType, Tac.Value( "Arnet::IntfId", intfName ),
                    address )
   if RPF not in rkPreference.rpf:
      rkPreference.rpf[ RPF ] = True

ipMrouteLegacy = legacyCliCallback( ipMroute )

def noIpMroute( mode, args, legacy=False ):
   prefix = args[ 'PREFIX' ]
   rpf = args.get( 'INTF' ) or args.get( 'ADDR' )
   preference = args.get( 'PREF' )
   af = RouterMulticastCliLib.getAddressFamilyFromMode( mode, legacy=legacy )
   if not isValidPrefix( mode, prefix, af ):
      return

   vrfName = RouterMulticastCliLib.getVrfNameFromMode( mode )
   if not vrfName :
      vrfName = DEFAULT_VRF

   if vrfName not in mribConfigColl[ af ].vrfConfig:
      return

   mribConfig = mribConfigColl[ af ].vrfConfig[ vrfName ]
   routes = mribConfig.route

   assert routes is not None

   arnetPrefix = Arnet.IpGenPrefix( str( prefix ) )
   route = routes.get( arnetPrefix )
   if not route:
      return

   if rpf:
      if isinstance( rpf, ( str, Tac.Type( "Arnet::Ip6Addr" ) ) ):
         rpfType = 'address'
      else:
         rpfType = 'intf'

   # Delete all routes matching prefix
   if not rpf and not preference:
      del routes[ arnetPrefix ]
   # Delete all routes matching prefix and preference
   elif not rpf and preference:
      if preference in route.pref:
         del route.pref[ preference ]
      if not route.pref:
         del routes[ arnetPrefix ]
   # Delete all routes matching prefix and rpf
   elif rpf and not preference:
      delPref = []
      for pref in route.pref:
         rkPref = route.pref[ pref ]
         for RPF in rkPref.rpf:
            if RPF.rpfType == rpfType:
               if rpfType == 'address' and str( RPF.address ) == str( rpf ):
                  del rkPref.rpf[ RPF ]
                  break
               elif rpfType == 'intf' and RPF.intfId == str( rpf ):
                  del rkPref.rpf[ RPF ]
                  break
         if not rkPref.rpf:
            delPref.append( pref )
      for pref in delPref:
         del route.pref[ pref ]
      if not route.pref:
         del routes[ arnetPrefix ]
   # Delete all routes matching prefix, preference and rpf
   else:
      rkPref = routes[ arnetPrefix ].pref.get( preference )
      if not rkPref:
         return
      for RPF in rkPref.rpf:
         if rpfType == RPF.rpfType:
            if ( ( rpfType == 'address' and str( RPF.address ) == str( rpf ) ) or
                  ( rpfType == 'intf' and RPF.intfId == str( rpf ) ) ):
               del rkPref.rpf[ RPF ]
               del routes[ arnetPrefix ].pref[ preference ].rpf[ RPF ]
               break
      if not rkPref.rpf:
         del routes[ arnetPrefix ].pref[ preference ]
      if not routes[ arnetPrefix ].pref:
         del routes[ arnetPrefix ]

noIpMrouteLegacy = legacyCliCallback( noIpMroute )

class LegacyIpMroute( CliCommand.CliCommandClass ):
   syntax = 'ip mroute PREFIX ( INTF | ADDR ) [ PREF ]'
   noOrDefaultSyntax = 'ip mroute PREFIX [ ( INTF | ADDR ) ] [ PREF ] ...'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'mroute': mrouteNode,
      'PREFIX': prefixExpr,
      'INTF': IntfCli.Intf.matcher,
      'ADDR': addressMatcher,
      'PREF': prefMatcher
   }
   handler = ipMrouteLegacy
   noOrDefaultHandler = noIpMrouteLegacy

class RpfRouteBase( CliCommand.CliCommandClass ):
   syntax = 'rpf route PREFIX ( INTF | ADDR ) [ PREF ]'
   noOrDefaultSyntax = 'rpf route PREFIX [ ( INTF | ADDR ) ] [ PREF ] ...'
   baseData = {
      'rpf': rpfKwMatcher,
      'route': routeNode,
      'INTF': IntfCli.Intf.matcher,
      'PREF': prefMatcher
   }
   handler = ipMroute
   noOrDefaultHandler = noIpMroute

class RpfRoute4( RpfRouteBase ):
   data = {
      'PREFIX': prefixExpr,
      'ADDR': addressMatcher
   }
   data.update( RpfRouteBase.baseData )

class RpfRoute6( RpfRouteBase ):
   data = {
      'PREFIX': prefix6Expr,
      'ADDR': address6Matcher
   }
   data.update( RpfRouteBase.baseData )

RouterMulticastCliLib.RouterMulticastSharedModelet.addCommandClass(
   LegacyIpMroute )

BasicCli.GlobalConfigMode.addCommandClass( LegacyIpMroute )

RouterMulticastCliLib.RouterMulticastIpv4Modelet.addCommandClass(
      RpfRoute4 )

if Toggles.McastCommonToggleLib.toggleMrib6CliEnabled():
   RouterMulticastCliLib.RouterMulticastIpv6Modelet.addCommandClass(
         RpfRoute6 )

def Plugin( entityManager ):
   global _allVrfConfig

   typeStr = 'Routing::Multicast::Rib::ConfigColl'
   configType = Tac.Type( typeStr )
   for af in [ "ipv4", "ipv6" ]:
      mountPath = configType.mountPath( af )
      mribConfigColl[ af ] = ConfigMount.mount( entityManager, mountPath, typeStr,
                                                'w' )

   _allVrfConfig = LazyMount.mount( entityManager, 'ip/vrf/config',
                                   'Ip::AllVrfConfig', 'r' )

   MrouteCli.routerMcastVrfDefinitionHook.addExtension( _mribVrfCreationHook )
   MrouteCli.routerMcastVrfDeletionHook.addExtension( _mribVrfDeletionHook )
