#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from BgpVpnCli import ( ExtCommValuesAndExactExpression,
      LargeCommValuesAndExactExpression,
      IpPrefixShowExpression,
      Ipv6PrefixShowExpression,
      RdCliExpression,
      NexthopExpression,
      )
from CliToken.RoutingBgpShowCliTokens import (
      bgpAfterShow,
      CommunityValuesAndExactRule,
      detail as kwDetail,
      internal as kwInternal,
      neighbors,
      RouteTypeMaybeFilteredMatcher,
      summary,
      )
from CliPlugin import TechSupportCli
from CliPlugin.RoutingBgpCli import (
      V4V6PeerKeyCliExpression )
from CliPlugin.ArBgpCli import doShowIpBgpSummaryAcrImpl
from CliPlugin.MplsVpnCliModels import mplsVpnRoutesVrfModel
from CliPlugin.IpAddrMatcher import ipPrefixMatcher
from CliPlugin.Ip6AddrMatcher import ip6PrefixMatcher
from MplsVpnCli import tokenVpnV4, tokenVpnV6, mplsVpnAfExpression
from RoutingBgpShowCli import summaryVrfModel
from RoutingBgpShowCli import ArBgpShowOutput
from RoutingBgpShowCli import BgpVrfRoutingContextDefaultOnly
import ShowCommand

#-------------------------------------------------------------------------------
# Helper Methods
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpMplsVpn', arBgpModeOnly=True )
@BgpVrfRoutingContextDefaultOnly( cliModel=mplsVpnRoutesVrfModel )
def doShowBgpMplsVpn( mode, nlriType=None, peerAddrValue=None,
                      bgpRouteTypeValue=None,
                      commValuesAndExact=None,
                      extCommValuesAndExact=None,
                      largeCommValuesAndExact=None,
                      prefixValue=None, longerPrefixes=False,
                      rdValue=None, nexthopValue=None,
                      detail=None, internal=None ):
   assert False, "This method is abstract. " + \
      "If this was reached then something went wrong with MplsVpnHelperCli loading"


# For 'show bgp vpn-ipv4/vpn-ipv6 summary' cmd, we are using ArBgp
# implementation of 'show ip[v6] bgp summary' cmd.
@ArBgpShowOutput( 'doShowBgpMplsVpnV4Summary', arBgpModeOnly=True )
@BgpVrfRoutingContextDefaultOnly( cliModel=summaryVrfModel )
def doShowBgpMplsVpnV4Summary( mode ):
   return doShowIpBgpSummaryAcrImpl( mode, nlriAfiSafi='ipv4MplsVpn' )

@ArBgpShowOutput( 'doShowBgpMplsVpnV6Summary', arBgpModeOnly=True )
@BgpVrfRoutingContextDefaultOnly( cliModel=summaryVrfModel )
def doShowBgpMplsVpnV6Summary( mode ):
   return doShowIpBgpSummaryAcrImpl( mode, nlriAfiSafi='ipv6MplsVpn' )

#-------------------------------------------------------------------------------
# "show bgp ( vpn-ipv4 | vpn-ipv6 ) summary"
#-------------------------------------------------------------------------------
class MplsVpnSummaryCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp AF summary"
   data = {
         "bgp" : bgpAfterShow,
         "AF" : mplsVpnAfExpression,
         "summary" : summary,
   }
   cliModel = summaryVrfModel

   @staticmethod
   def handler( mode, args ):
      if args[ 'AF' ] == 'vpn-ipv4':
         doShowBgpMplsVpnV4Summary( mode )
      else:
         doShowBgpMplsVpnV6Summary( mode )
      return summaryVrfModel()

BasicCli.addShowCommandClass( MplsVpnSummaryCommand )

def handlerVpnV4( mode, args ):
   return doShowBgpMplsVpn( mode,
            nlriType='mplsVpnIpv4Unicast',
            peerAddrValue=args.get( "PEER" ),
            bgpRouteTypeValue=args.get( "route-type" ),
            commValuesAndExact=args.get( "communityValuesAndExact" ),
            extCommValuesAndExact=args.get( "EXT_COMMUNITIES" ),
            largeCommValuesAndExact=args.get( "LARGE_COMMUNITIES" ),
            prefixValue=args.get( "PREFIX" ),
            longerPrefixes=args.get( "longer-prefixes" ),
            rdValue=args.get( "RD_VAL" ),
            nexthopValue=args.get( "nexthop-addr" ),
            detail=args.get( "detail" ),
            internal=args.get( "internal" ) )

def handlerVpnV6( mode, args ):
   return doShowBgpMplsVpn( mode,
            nlriType='mplsVpnIpv6Unicast',
            peerAddrValue=args.get( "PEER" ),
            bgpRouteTypeValue=args.get( "route-type" ),
            commValuesAndExact=args.get( "communityValuesAndExact" ),
            extCommValuesAndExact=args.get( "EXT_COMMUNITIES" ),
            largeCommValuesAndExact=args.get( "LARGE_COMMUNITIES" ),
            prefixValue=args.get( "PREFIX" ),
            longerPrefixes=args.get( "longer-prefixes" ),
            rdValue=args.get( "RD_VAL" ),
            nexthopValue=args.get( "nexthop-addr" ),
            detail=args.get( "detail" ),
            internal=args.get( "internal" ) )
#-------------------------------------------------------------------------------
# "show bgp vpn-ipv4 [ community <communities> [ exact ] ]
#                    [ extcommunity <extcommunities> [ exact ] ]
#                    [ largecommunity <largecommunities> [ exact ] ]
#                    [ prefix <ipv4-prefix> [ longer-prefixes ] ]
#                    [ rd <rd> ]
#                    [ next-hop ( <v4addr> | <v6addr> ) ]
#                    [ detail ]"
#-------------------------------------------------------------------------------

class ShowBgpVpnV4Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show bgp vpn-ipv4 [ COMM ] [ EXT_COMM ] [ LARGE_COMM ]"
              " [ IP_PREFIX ] [ RD ] [ NEXT_HOP ] [ detail ]" )
   data = {
         "bgp" : bgpAfterShow,
         "vpn-ipv4" : tokenVpnV4,
         "COMM" : CommunityValuesAndExactRule,
         "EXT_COMM" : ExtCommValuesAndExactExpression,
         "LARGE_COMM" : LargeCommValuesAndExactExpression,
         "IP_PREFIX" : IpPrefixShowExpression,
         "RD" : RdCliExpression,
         "NEXT_HOP" : NexthopExpression,
         "detail" : kwDetail
   }
   cliModel = mplsVpnRoutesVrfModel
   handler = handlerVpnV4

BasicCli.addShowCommandClass( ShowBgpVpnV4Cmd )

#-------------------------------------------------------------------------------
# "show bgp vpn-ipv4 <ipv4-prefix> rd <rd> internal"
#-------------------------------------------------------------------------------
# Internal command for specific ipv4 prefix & RD.
class ShowBgpVpnV4InternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp vpn-ipv4 PREFIX RD internal"
   data = {
         "bgp" : bgpAfterShow,
         "vpn-ipv4" : tokenVpnV4,
         "PREFIX" : ipPrefixMatcher,
         "RD" : RdCliExpression,
         "internal" : kwInternal
   }
   cliModel = mplsVpnRoutesVrfModel
   handler = handlerVpnV4

BasicCli.addShowCommandClass( ShowBgpVpnV4InternalCmd )

#-------------------------------------------------------------------------------
# "show bgp vpn-ipv6 [ community <communities> [ exact ] ]
#                    [ extcommunity <extcommunities> [ exact ] ]
#                    [ largecommunity <larecommunities> [ exact ] ]
#                    [ prefix <ipv6-prefix> [ longer-prefixes ] ]
#                    [ rd <rd> ]
#                    [ next-hop ( <v4addr> | <v6addr> ) ]
#                    [ detail ]"
#-------------------------------------------------------------------------------
class ShowBgpVpnV6Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show bgp vpn-ipv6 [ COMM ] [ EXT_COMM ] [ LARGE_COMM ]"
              " [ IPv6_PREFIX ] [ RD ] [ NEXT_HOP ] [ detail ]" )
   data = {
         "bgp" : bgpAfterShow,
         "vpn-ipv6" : tokenVpnV6,
         "COMM" : CommunityValuesAndExactRule,
         "EXT_COMM" : ExtCommValuesAndExactExpression,
         "LARGE_COMM" : LargeCommValuesAndExactExpression,
         "IPv6_PREFIX" : Ipv6PrefixShowExpression,
         "RD" : RdCliExpression,
         "NEXT_HOP" : NexthopExpression,
         "detail" : kwDetail
   }
   cliModel = mplsVpnRoutesVrfModel
   handler = handlerVpnV6

BasicCli.addShowCommandClass( ShowBgpVpnV6Cmd )

#-------------------------------------------------------------------------------
# "show bgp vpn-ipv6 <ipv6-prefix> rd <rd> internal"
#-------------------------------------------------------------------------------
# Internal command for specific ipv6 prefix & RD.
class ShowBgpVpnV6InternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp vpn-ipv6 PREFIX RD internal"
   data = {
         "bgp" : bgpAfterShow,
         "vpn-ipv6" : tokenVpnV6,
         "PREFIX" : ip6PrefixMatcher,
         "RD" : RdCliExpression,
         "internal" : kwInternal
   }
   cliModel = mplsVpnRoutesVrfModel
   handler = handlerVpnV6

BasicCli.addShowCommandClass( ShowBgpVpnV6InternalCmd )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> vpn-ipv4
#                    ( routes | received-routes | advertised-routes )
#                    [ community <communities> [ exact ] ]
#                    [ extcommunity <extcommunities> [ exact ] ]
#                    [ largecommunity <largecommunities> [ exact ] ]
#                    [ <ipv4-prefix> [ longer-prefixes ] ]
#                    [ rd <rd> ]
#                    [ next-hop ( <v4addr> | <v6addr> ) ]
#                    [ detail ]"
#-------------------------------------------------------------------------------
# Show BGP vpn-ipv4 routes filtered by neighbors IP, and optionally filter by
# communities, extcommunities, largecommunities, ipv4-prefix, RD, and/or next-hop.
class ShowBgpNbrVpnV4Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show bgp neighbors PEER_ADDR vpn-ipv4 ROUTE_TYPE"
              " [ COMM ] [ EXT_COMM ] [ LARGE_COMM ]"
              " [ IP_PREFIX ] [ RD ] [ NEXT_HOP ] [ detail ]" )
   data = {
         "bgp" : bgpAfterShow,
         "neighbors" : neighbors,
         "PEER_ADDR" : V4V6PeerKeyCliExpression,
         "vpn-ipv4" : tokenVpnV4,
         "ROUTE_TYPE" : RouteTypeMaybeFilteredMatcher,
         "COMM" : CommunityValuesAndExactRule,
         "EXT_COMM" : ExtCommValuesAndExactExpression,
         "LARGE_COMM" : LargeCommValuesAndExactExpression,
         "IP_PREFIX" : IpPrefixShowExpression,
         "RD" : RdCliExpression,
         "NEXT_HOP" : NexthopExpression,
         "detail" : kwDetail
   }
   cliModel = mplsVpnRoutesVrfModel
   handler = handlerVpnV4

BasicCli.addShowCommandClass( ShowBgpNbrVpnV4Cmd )


#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> vpn-ipv4
#                    ( routes | received-routes | advertised-routes )
#                    <ipv4-prefix> rd <rd> internal"
#-------------------------------------------------------------------------------
# Internal command for specific ipv4 prefix & RD.
class ShowBgpNbrVpnV4InternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = (
      "show bgp neighbors PEER_ADDR vpn-ipv4 ROUTE_TYPE PREFIX RD internal" )
   data = {
         "bgp" : bgpAfterShow,
         "neighbors" : neighbors,
         "PEER_ADDR" : V4V6PeerKeyCliExpression,
         "vpn-ipv4" : tokenVpnV4,
         "ROUTE_TYPE" : RouteTypeMaybeFilteredMatcher,
         "PREFIX" : ipPrefixMatcher,
         "RD" : RdCliExpression,
         "internal" : kwInternal
   }
   cliModel = mplsVpnRoutesVrfModel
   handler = handlerVpnV4

BasicCli.addShowCommandClass( ShowBgpNbrVpnV4InternalCmd )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> vpn-ipv6
#                    ( routes | received-routes | advertised-routes )
#                    [ community <communities> [ exact ] ]
#                    [ extcommunity <extcommunities> [ exact ] ]
#                    [ largecommunity <largecommunities> [ exact ] ]
#                    [ <ipv6-prefix> [ longer-prefixes ] ]
#                    [ rd <rd> ]
#                    [ next-hop ( <v4addr> | <v6addr> ) ]
#                    [ detail ]"
#-------------------------------------------------------------------------------
# Show BGP vpn-ipv6 routes filtered by neighbors IP, and optionally filter by
# communities, extcommunities, largecommunities, ipv6-prefix, RD, and/or next-hop.
class ShowBgpNbrVpnV6Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show bgp neighbors PEER_ADDR vpn-ipv6 ROUTE_TYPE"
              " [ COMM ] [ EXT_COMM ] [ LARGE_COMM ]"
              " [ IPv6_PREFIX ] [ RD ] [ NEXT_HOP ] [ detail ]" )
   data = {
         "bgp" : bgpAfterShow,
         "neighbors" : neighbors,
         "PEER_ADDR" : V4V6PeerKeyCliExpression,
         "vpn-ipv6" : tokenVpnV6,
         "ROUTE_TYPE" : RouteTypeMaybeFilteredMatcher,
         "COMM" : CommunityValuesAndExactRule,
         "EXT_COMM" : ExtCommValuesAndExactExpression,
         "LARGE_COMM" : LargeCommValuesAndExactExpression,
         "IPv6_PREFIX" : Ipv6PrefixShowExpression,
         "RD" : RdCliExpression,
         "NEXT_HOP" : NexthopExpression,
         "detail" : kwDetail
   }
   cliModel = mplsVpnRoutesVrfModel
   handler = handlerVpnV6

BasicCli.addShowCommandClass( ShowBgpNbrVpnV6Cmd )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> vpn-ipv6
#                    ( routes | received-routes | advertised-routes )
#                    <ipv6-prefix> rd <rd> internal"
#-------------------------------------------------------------------------------
# Internal command for specific ipv6 prefix & RD.
class ShowBgpNbrVpnV6InternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = (
      "show bgp neighbors PEER_ADDR vpn-ipv6 ROUTE_TYPE PREFIX RD internal" )
   data = {
         "bgp" : bgpAfterShow,
         "neighbors" : neighbors,
         "PEER_ADDR" : V4V6PeerKeyCliExpression,
         "vpn-ipv6" : tokenVpnV6,
         "ROUTE_TYPE" : RouteTypeMaybeFilteredMatcher,
         "PREFIX" : ip6PrefixMatcher,
         "RD" : RdCliExpression,
         "internal" : kwInternal
   }
   cliModel = mplsVpnRoutesVrfModel
   handler = handlerVpnV6

BasicCli.addShowCommandClass( ShowBgpNbrVpnV6InternalCmd )

#-------------------------------------------------------------------------------
# register mpls vpn commands for show tech-support
#
# Not registering the "show bgp vpn-ipvX" commands as the output could be
# overwhelming in scaled setups.
#-------------------------------------------------------------------------------
def _showTechMplsVpnCmds():
   return [
            'show bgp vpn-ipv4 summary',
            'show bgp vpn-ipv6 summary',
         ]

TechSupportCli.registerShowTechSupportCmdCallback( '2019-09-19 10:51:00',
      _showTechMplsVpnCmds,
      summaryCmdCallback=_showTechMplsVpnCmds )
