# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import ( DeferredModel, Submodel, Str, Int, Float, List, Enum,
                       GeneratorDict, Bool )
from ArnetModel import Ip4Address, IpGenericPrefix
from CliPlugin.BgpCliModels import ( BgpRoutePeerEntry, EncapLabel,
                                     REASON_NOT_BEST_LIST )
from CliPlugin.VrfCli import generateVrfCliModel

ROUTE_ORIGIN_ENUM_VALUES = ( 'Igp', 'Egp', 'Incomplete' )

class MplsVpnRouteASPathEntry( DeferredModel ):
   asPath = Str( optional=True, help='AS path string (if absent,  '
                 'then the route was originated locally)' )
   asPathType = Enum( values=( 'Internal', 'External', 'Confed-External', 'Local',
                               'Invalid' ),
                      help='AS path type: '
                            'Internal - originated by I-BGP '
                            'External - originated by E-BGP '
                            'Confed-External - originated by a E-BGP confederation '
                            'Local - originated locally '
                            'Invalid - AS path is invalid' )

class MplsVpnRouteTypeEntry ( DeferredModel ):
   stale = Bool( default=False, help='Route is stale' )
   valid = Bool( default=False, help='Route is valid' )
   suppressed = Bool( default=False,
                      help='Route is suppressed from entering the Rib' )
   active = Bool( default=False, help='Route is active' )
   backup = Bool( default=False, help='Route is backup' )
   ecmpHead = Bool( default=False, help='Route is the ECMP head' )
   ecmp = Bool( default=False, help='Route is an ECMP route' )
   ucmp = Bool( optional=True, default=False, help='Route is an UCMP route' )
   ecmpContributor = Bool( default=False, help='Route contributes to ECMP' )
   atomicAggregator = Bool( default=False, help='Route is an atomic-aggregate' )
   queued = Bool( default=False, help='Route is queued for advertisement' )
   localAgg = Bool( optional=True, help='Route is locally aggregated' )
   notInstalledReason = Enum( values=( 'routeBestInactive', 'routeInstallDenied',
                                       'labeledRoutePresent' ),
                              optional=True,
                              help="Reason for route not being installed" )
   origin = Enum( optional=True, values=ROUTE_ORIGIN_ENUM_VALUES,
                  help='Route origin' )
   waitForConvergence = Bool( optional=True,
                              help='Route is pending BGP convergence' )

class CustomerAttributesDetailEntry( DeferredModel ):
   originAsn = Int( help="Origin AS number" )
   origin = Enum( values=ROUTE_ORIGIN_ENUM_VALUES, help='Route origin' )
   aggregator = Str( optional=True, help='Aggregator of the route' )
   originator = Ip4Address( optional=True,
                            help='Router ID of the originator of this route' )
   clusterList = Str( optional=True,
                       help='Cluster list for the route' )
   domainPath = List( optional=True, valueType=str, help="Domain path attribute" )
   med = Int( optional=True, help='Multi Exit Discriminator for the route' )
   localPreference = Int( optional=True, help='I-BGP Local preference indicator' )
   communities = List( valueType=str, help='Standard community attributes' )
   extCommunities = List( valueType=str, help='Extended community attributes' )
   largeCommunities = List( valueType=str, help='Large community attributes' )

class MplsVpnRouteDetailEntry( DeferredModel ):
   aggregator = Str( optional=True, help='Aggregator of the route' )
   communities = List( valueType=str, help='Route community' )
   extCommunities = List( valueType=str, help='Extended community for route' )
   rxPathId = Int( optional=True, help='Received path ID of this route' )
   txPathId = Int( optional=True, help='Advertised path ID for this route' )
   rxSafi = Str( optional=True, help='Received SAFI of this route' )
   origin = Enum( values=ROUTE_ORIGIN_ENUM_VALUES, help='Route origin' )
   originator = Ip4Address( optional=True,
                            help='Router ID of the originator of this route' )
   clusterList = Str( optional=True,
                       help='Cluster list for the route' )
   peerEntry = Submodel( valueType=BgpRoutePeerEntry, optional=True,
                         help='Peer information for the route' )
   fromRRClient = Bool( default=False,
                        help='Route received from route reflector client' )
   seqNumber = Int( optional=True, help='Route sequence number' )
   pendingTimeToAdv = Float( optional=True, help='Timestamp of route advertisement' )
   # the redistibution protocol strings here need to be kept in sync
   # with the strings returned by
   # gated/gated-ctk/src/bgp/bgp_dget_route.c:redist_proto_str()
   # note these are usually transformed by maybeCamelize()
   redistributionProtocol = Enum( values=( 'Connected', 'Static', 'Ospf3', 'Ospf',
                                           'Rip', 'Is-Is', 'unknown' ),
                                  optional=True,
                                  help='Protocol from which route got '
                                     'redistributed into BGP' )
   labelStack = List( optional=True, valueType=int,
                      help='MPLS label stack information' )
   tunnelRibEligible = Bool( default=False,
                             help='Route eligible to be installed in Tunnel RIB' )
   remoteLabel = Submodel( optional=True, valueType=EncapLabel,
                           help='Remote MPLS label' )
   localLabel = Submodel( optional=True, valueType=EncapLabel,
                          help='Local MPLS label' )
   domainPath = List( optional=True, valueType=str, help="Domain path attribute" )
   customerAttributeSet = Submodel( optional=True,
                                    valueType=CustomerAttributesDetailEntry,
                                    help="Customer Attributes" )

class MplsVpnRoutePath( DeferredModel ):
   nextHop = Str( optional=True, help='Route next hop address' )
   asPathEntry = Submodel( valueType=MplsVpnRouteASPathEntry,
                           help='AS path information' )
   importedMplsVpnPathRd = Str( optional=True,
                                help='Imported MPLS VPN path RouteDistinguisher' )
   med = Int( optional=True, help='Multi Exit Discriminator for the route' )
   localPreference = Int( optional=True, help='I-BGP Local preference indicator' )
   routeType = Submodel( valueType=MplsVpnRouteTypeEntry, help='Route type' )
   weight = Int( optional=True, help='Weight for the route' )
   timestamp = Int( optional=True,
                    help="UTC seconds since epoch when the route was received."
                          "Only returned with detail output" )
   routeDetail = Submodel( valueType=MplsVpnRouteDetailEntry, optional=True,
                           help='Route details' )
   reasonNotBestpath = Enum( values=REASON_NOT_BEST_LIST,
                             help='Reason route was not selected as BGP best path' )

class MplsVpnRouteKeyDetail( DeferredModel ):
   rd = Str( optional=True, help='Route distinguisher' )
   ipGenPrefix = IpGenericPrefix( optional=True, help='IPv4 or IPv6 address prefix' )

class MplsVpnRouteEntry( DeferredModel ):
   routeKey = Str( help='MPLS VPN Network Layer Reachability Information (NLRI)' )
   routeKeyDetail = Submodel( valueType=MplsVpnRouteKeyDetail, help='NLRI details' )
   totalPaths = Int( optional=True, help='Total number of paths for this route' )
   mplsVpnRoutePaths = List( valueType=MplsVpnRoutePath,
                             help='List of BGP MPLS VPN route ECMP paths' )

class MplsVpnRoutes( DeferredModel ):
   vrf = Str( help='VRF name' )
   routerId = Ip4Address( help='BGP Router Identity' )
   asn = Int( help='Autonomous System Number' )
   lastProcessedSeqNum = Int( optional=True,
                              help='Last route sequence number acknowledged' )
   currentSeqNum = Int( optional=True, help='Current route sequence number' )
   _detail = Bool( optional=True, help='Detailed output is requested' )
   _advRoutes = Bool( optional=True, help='Advertised routes output is requested' )
   mplsVpnRoutes = GeneratorDict( keyType=str, valueType=MplsVpnRouteEntry,
         help='Dictionary of BGP MPLS VPN route entries indexed by the route key' )

# Wrap MplsVpnRoutes model under "vrfs" key
mplsVpnRoutesVrfModel = generateVrfCliModel( MplsVpnRoutes,
                                             "MPLS VPN route summary" )
