#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliParser
import CliCommand
import CliMatcher
import IntfCli
from BgpLib import updateConfigAttrsAfMap, vpnAfTypeMapInv
from CliMode.BgpCommon import RoutingBgpBaseAfMode
import CliPlugin.BgpVpnDomainIdCli as BgpVpnDomainIdCli
from CliPlugin.BgpVpnCli import RouteImportMatchFailureDiscardCmd
from CliPlugin.RoutingBgpCli import (
   afModeExtensionHook,
   bgpNeighborConfig,
   configForVrf,
   delNeighborConfigIfDefault,
   BgpCmdBaseClass,
   PeerCliExpression,
   RouterBgpAfBaseModelet,
   RouterBgpAfSharedModelet,
   RouterBgpAfVpnModelet,
   RouterBgpBaseMode,
)
from CliPlugin.RoutingBgpInstanceCli import (
   NexthopResolutionDisabledCmd,
   nhResRibsConfigHelper,
   removeNexthopSelfReceivedVpnRoutesConfig,
   resetBgpAfModeConfig,
   SetAddpathSendCmd,
)
from CliPlugin.RoutingBgpNeighborCli import (
   neighborAfActivateHelper,
   PeerAfStateEnum,
   SetNeighborAddPathSendCmd,
)
from CliPlugin.IpRibLib import ResolutionRibsExpr, ResolutionRibsHiddenUnicastExpr
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
import CliToken.RoutingBgp as bgpTokens
import CliToken.IpRibLibCliTokens
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType
from Toggles import BgpCommonToggleLib
# pylint: disable-msg=W0621

updateConfigAttrsAfMap( { 'vpn-ipv4' : [ 'VpnV4', 'AfVpnV4' ],
                          'vpn-ipv6' : [ 'VpnV6', 'AfVpnV6' ] } )

# Note: Defining this directly in the RouterBgpBaseMode,
# so this command is not available in Nd - VRF submode.
# Mpls in the MplsVpn below refers to the service label used for the VPN and not the
# EncapType( the transport label ).
# EncapType may be Mpls, GRE etc. but we only support Mpls at present.
class RouterBgpBaseAfMplsVpnV4Mode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family MPLS-VPN V4 configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily='vpn-ipv4' ):
      self.vrfName = DEFAULT_VRF
      self.addrFamily = addrFamily
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfMplsVpnV6Mode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family MPLS-VPN V6 configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily='vpn-ipv6' ):
      self.vrfName = DEFAULT_VRF
      self.addrFamily = addrFamily
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfMplsVpnSharedModelet( RouterBgpAfBaseModelet ):
   """This modelete has all the commands which are shared between vpn-ipv4
   and vpn-ipv6 address-families"""
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

RouterBgpBaseAfMplsVpnV4Mode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfMplsVpnV4Mode.addModelet( RouterBgpBaseAfMplsVpnSharedModelet )
RouterBgpBaseAfMplsVpnV4Mode.addModelet( RouterBgpAfVpnModelet )
RouterBgpBaseAfMplsVpnV6Mode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfMplsVpnV6Mode.addModelet( RouterBgpBaseAfMplsVpnSharedModelet )
RouterBgpBaseAfMplsVpnV6Mode.addModelet( RouterBgpAfVpnModelet )

#------------------------------------------------------------------------------------
# "[ no | default ] neighbor PEER ( activate | deactivate )
#------------------------------------------------------------------------------------
neighborActivateCmdRegistration = \
      [ RouterBgpBaseAfMplsVpnSharedModelet ]

class MplsVpnNeighborActDeact( CliCommand.CliCommandClass ):
   syntax = 'neighbor PEER ( activate | deactivate )'
   noOrDefaultSyntax = 'neighbor PEER activate'
   data = {
         'neighbor' : bgpTokens.neighbor,
         'PEER' : PeerCliExpression,
         'activate' : bgpTokens.activate,
         'deactivate' : bgpTokens.deactivate,
   }

   @staticmethod
   # Special handling for NED-mode, where 'deactivate' is used in place of 'no'
   def handler( mode, args ):
      if 'activate' in args:
         neighborAfActivateHelper( mode, args[ 'PEER' ],
                                   PeerAfStateEnum.afActive )
      else:
         neighborAfActivateHelper( mode, args[ 'PEER' ],
                                   PeerAfStateEnum.afInactive )

   @staticmethod
   def noHandler( mode, args ):
      neighborAfActivateHelper( mode, args[ 'PEER' ], PeerAfStateEnum.afInactive )

   @staticmethod
   def defaultHandler( mode, args ):
      neighborAfActivateHelper( mode, args[ 'PEER' ], PeerAfStateEnum.afDefault )

for mode in neighborActivateCmdRegistration:
   mode.addCommandClass( MplsVpnNeighborActDeact )


#------------------------------------------------------------------------------------
# "[ no | default ] address-family vpn-ipv4 | vpn-ipv6" config mode
#------------------------------------------------------------------------------------
tokenVpnV4 = CliMatcher.KeywordMatcher( 'vpn-ipv4',
                     helpdesc='MPLS L3 VPN IPv4 unicast address family' )
tokenVpnV6 = CliMatcher.KeywordMatcher( 'vpn-ipv6',
                     helpdesc='MPLS L3 VPN IPv6 unicast address family' )


class mplsVpnAfExpression( CliCommand.CliExpression ):
   expression = '( vpn-ipv4 | vpn-ipv6 )'
   data = {
         'vpn-ipv4' : tokenVpnV4,
         'vpn-ipv6' : tokenVpnV6
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'vpn-ipv4' in args:
         args[ 'AF' ] = 'vpn-ipv4'
      elif 'vpn-ipv6' in args:
         args[ 'AF' ] = 'vpn-ipv6'

class MplsVpnAfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'address-family FAMILY'
   noOrDefaultSyntax = syntax
   data = {
         'address-family' : bgpTokens.addrFamily,
         'FAMILY' : mplsVpnAfExpression,
         }

   @staticmethod
   def handler( mode, args ):
      vpnAddrFamily = args[ 'AF' ]
      if getEffectiveProtocolModel( mode ) == ProtocolAgentModelType.ribd:
         mode.addWarning( "Routing protocols model multi-agent must be "
                          "configured for MPLS-VPN address-family" )
      assert vpnAddrFamily in afModeExtensionHook.afModeExtensions()
      childMode = mode.childMode(
         afModeExtensionHook.afModeExtension[ vpnAddrFamily ] )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      bgpConfig = configForVrf( mode.vrfName )
      resetBgpAfModeConfig( bgpConfig, args[ 'AF' ], mode.vrfName )

RouterBgpBaseMode.addCommandClass( MplsVpnAfModeCmd )

#------------------------------------------------------------------------------------
# [ no | default ] bgp additional-paths send any
#------------------------------------------------------------------------------------
RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( SetAddpathSendCmd )

#--------------------------------------------------------------------------------
# ( no | default ) neighbor ( ADDR | V6ADDR | LLV6ADDR | PEERGROUPNAME )
# additional-paths send any
#--------------------------------------------------------------------------------
RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( SetNeighborAddPathSendCmd )

#--------------------------------------------------------------------------------
# ( no | default ) neighbor default encapsulation mpls next-hop-self
# source-interface INTF
#--------------------------------------------------------------------------------
def setMplsNexthopSelfSourceInterface( mode, args ):
   peer = args.get( 'PEER' )
   if peer:
      config = bgpNeighborConfig( peer, mode.vrfName )
   else:
      config = configForVrf( mode.vrfName )

   if args[ 'mpls' ] == 'mpls':
      tacEncapType = 'encapMpls'
   else:
      raise ValueError()
   if mode.addrFamily == 'vpn-ipv4':
      config.afVpnV4Encap = tacEncapType
      config.nexthopSelfSrcIntfVpnV4 = args[ 'INTF' ].name
   elif mode.addrFamily == 'vpn-ipv6':
      config.afVpnV6Encap = tacEncapType
      config.nexthopSelfSrcIntfVpnV6 = args[ 'INTF' ].name
   else:
      raise ValueError()

def noMplsNexthopSelfSourceInterface( mode, args ):
   peer = args.get( 'PEER' )
   if peer:
      config = bgpNeighborConfig( peer, mode.vrfName, create=False )
   else:
      config = configForVrf( mode.vrfName )

   if not config:
      return

   if mode.addrFamily == 'vpn-ipv4':
      config.afVpnV4Encap = config.afVpnV4EncapDefault
      config.nexthopSelfSrcIntfVpnV4 = config.nexthopSelfSrcIntfVpnV4Default
   elif mode.addrFamily == 'vpn-ipv6':
      config.afVpnV6Encap = config.afVpnV6EncapDefault
      config.nexthopSelfSrcIntfVpnV6 = config.nexthopSelfSrcIntfVpnV6Default
   else:
      raise ValueError()

   if peer:
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

class NeighborDefaultEncapMplsNhselfSourceInterfaceCmd( BgpCmdBaseClass ):
   syntax = 'neighbor default encapsulation mpls next-hop-self source-interface INTF'
   noOrDefaultSyntax = 'neighbor default encapsulation mpls next-hop-self \
         source-interface ...'
   data = {
      'neighbor' : bgpTokens.neighbor,
      'default' : bgpTokens.default,
      'encapsulation' : bgpTokens.encap,
      'mpls' : bgpTokens.mpls,
      'next-hop-self' : bgpTokens.nextHopSelf,
      'source-interface' : bgpTokens.sourceInterface,
      'INTF' : IntfCli.Intf.matcherWithIpSupport,
   }

   @staticmethod
   def _handleNormal( mode, args ):
      setMplsNexthopSelfSourceInterface( mode, args )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      noMplsNexthopSelfSourceInterface( mode, args )

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass(
      NeighborDefaultEncapMplsNhselfSourceInterfaceCmd )

#--------------------------------------------------------------------------------
# ( no | default ) neighbor ( addr | peer group ) encapsulation mpls next-hop-self
# source-interface INTF
#--------------------------------------------------------------------------------
class NeighborPeerEncapMplsNhselfSourceInterfaceCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER encapsulation mpls next-hop-self source-interface INTF'
   noOrDefaultSyntax = syntax.replace( 'INTF', '...' )
   data = {
      'neighbor' : bgpTokens.neighbor,
      'PEER' : PeerCliExpression,
      'encapsulation' : bgpTokens.encap,
      'mpls' : bgpTokens.mpls,
      'next-hop-self' : bgpTokens.nextHopSelf,
      'source-interface' : bgpTokens.sourceInterface,
      'INTF' : IntfCli.Intf.matcherWithIpSupport,
   }

   @staticmethod
   def _handleNormal( mode, args ):
      setMplsNexthopSelfSourceInterface( mode, args )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      noMplsNexthopSelfSourceInterface( mode, args )

if BgpCommonToggleLib.toggleMplsVpnPerPeerNexthopEnabled():
   RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass(
         NeighborPeerEncapMplsNhselfSourceInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mpls label allocation disabled
#--------------------------------------------------------------------------------
def setLabelAllocation( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.labelAllocationDisabledAfVpnV4 = 'isTrue'
   elif mode.addrFamily == 'vpn-ipv6':
      config.labelAllocationDisabledAfVpnV6 = 'isTrue'
   else:
      raise ValueError()

def noLabelAllocation( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.labelAllocationDisabledAfVpnV4 = 'isInvalid'
   elif mode.addrFamily == 'vpn-ipv6':
      config.labelAllocationDisabledAfVpnV6 = 'isInvalid'
   else:
      raise ValueError()

class MplsLabelAllocationDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls label allocation disabled'
   noOrDefaultSyntax = syntax
   data = {
      'mpls' : bgpTokens.mpls,
      'label' : 'VPN label',
      'allocation' : 'VPN label allocation scheme',
      'disabled' : 'Use explicit-null as local label',
   }

   @staticmethod
   def handler( mode, args ):
      setLabelAllocation( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noLabelAllocation( mode, args )

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( MplsLabelAllocationDisabledCmd )

#-------------------------------------------------------------------------------
# "[no|default] next-hop resolution ribs ( vrf-unicast-rib | RIBS )"
#-------------------------------------------------------------------------------
def setNexthopResolutionInVrfUnicastRib( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.nexthopResolutionVrfUniRibAfVpnV4 = 'isTrue'
   elif mode.addrFamily == 'vpn-ipv6':
      config.nexthopResolutionVrfUniRibAfVpnV6 = 'isTrue'
   else:
      raise ValueError()

def noNexthopResolutionInVrfUnicastRib( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.nexthopResolutionVrfUniRibAfVpnV4 = 'isInvalid'
   elif mode.addrFamily == 'vpn-ipv6':
      config.nexthopResolutionVrfUniRibAfVpnV6 = 'isInvalid'
   else:
      raise ValueError()

def nhResRibsConfigVpnHelper( mode, args, no=False ):
   if 'vrf-unicast-rib' in args:
      if no:
         noNexthopResolutionInVrfUnicastRib( mode, args )
      else:
         setNexthopResolutionInVrfUnicastRib( mode, args )
   else:
      nhResRibsConfigHelper( mode.addrFamily, args, no=no,
            vrfName=mode.vrfName )

def setNhResRibsConfigVpnHandler( mode, args ):
   nhResRibsConfigVpnHelper( mode, args )

def noNhResRibsConfigVpnHandler( mode, args ):
   nhResRibsConfigVpnHelper( mode, args, no=True )

if BgpCommonToggleLib.toggleHideUnicastRibEnabled():
   ribsExpr = ResolutionRibsHiddenUnicastExpr
else:
   ribsExpr = ResolutionRibsExpr

class ResolutionVpnRibCmd( CliCommand.CliCommandClass ):
   syntax = 'next-hop resolution ribs ( vrf-unicast-rib | RIBS )'
   noOrDefaultSyntax = 'next-hop resolution ribs [ vrf-unicast-rib ]'
   data = {
      'next-hop' : bgpTokens.nextHopKwForResolutionPolicy,
      'resolution' : CliToken.IpRibLibCliTokens.matcherResolution,
      'ribs' : CliToken.IpRibLibCliTokens.matcherRibs,
      'vrf-unicast-rib' :
      'Resolve the next-hop in the imported-vrf using unicast RIB',
      'RIBS' : ribsExpr,
   }
   handler = setNhResRibsConfigVpnHandler
   noOrDefaultHandler = noNhResRibsConfigVpnHandler

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( ResolutionVpnRibCmd )

#-------------------------------------------------------------------------------
# "[no|default] domain identifier global_admin:local_admin
# under 'address-family vpn-ipv4|vpn-ipv6' mode
#-------------------------------------------------------------------------------

def setVpnDomainId( mode, vpnDomainId ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.domainIdVpnV4 = vpnDomainId
   elif mode.addrFamily == 'vpn-ipv6':
      config.domainIdVpnV6 = vpnDomainId
   else:
      raise ValueError()

def noVpnDomainId( mode ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      defaultVal = config.domainIdVpnV4Default
      config.domainIdVpnV4 = defaultVal
   elif mode.addrFamily == 'vpn-ipv6':
      defaultVal = config.domainIdVpnV6Default
      config.domainIdVpnV6 = defaultVal
   else:
      raise ValueError()

# VPN domain id configuration under MPLS VPN address-family.
# "domain identifier ASN:LOCAL_ADMIN"
class SetMplsVpnDomainIdCmd( CliCommand.CliCommandClass ):
   syntax = 'domain identifier DOMAIN_ID'
   noOrDefaultSyntax = 'domain identifier ...'
   data = { 'domain' : BgpVpnDomainIdCli.tokenDomain,
            'identifier' : BgpVpnDomainIdCli.tokenIdentifier,
            'DOMAIN_ID' : BgpVpnDomainIdCli.DomainIdExpression }
   @staticmethod
   def handler( mode, args ):
      setVpnDomainId( mode, args[ 'DOMAIN_ID' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noVpnDomainId( mode )

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( SetMplsVpnDomainIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] neighbor default encapsulation mpls next-hop-self
# received-vpnv4-routes
# in "address-family vpn-ipv4" mode
#--------------------------------------------------------------------------------

def setMplsNexthopSelfReceivedVpnRoutes( mode, args ):
   config = configForVrf( mode.vrfName )
   if mode.addrFamily in [ 'vpn-ipv4', 'vpn-ipv6' ]:
      afiSafi = vpnAfTypeMapInv[ mode.addrFamily ]
      config.nexthopSelfLocalLabelAlloc[ afiSafi ] = True
   else:
      raise ValueError()

def noMplsNexthopSelfReceivedVpnRoutes( mode, args ):
   config = configForVrf( mode.vrfName )
   removeNexthopSelfReceivedVpnRoutesConfig( config, mode.addrFamily,
                                             vrfName=mode.vrfName )

matcherReceivedVpnv4Routes = CliMatcher.KeywordMatcher( 'received-vpnv4-routes',
      helpdesc='Use local labels for advertising received VPN-IPv4 routes' )

class NeighborDefaultEncapMplsNhselfReceivedVpnv4RoutesCmd(
      CliCommand.CliCommandClass ):
   syntax = 'neighbor default encapsulation mpls next-hop-self received-vpnv4-routes'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor' : bgpTokens.neighbor,
      'default' : bgpTokens.default,
      'encapsulation' : bgpTokens.encap,
      'mpls' : bgpTokens.mpls,
      'next-hop-self' : bgpTokens.nextHopSelf,
      'received-vpnv4-routes' : matcherReceivedVpnv4Routes,
   }

   @staticmethod
   def handler( mode, args ):
      setMplsNexthopSelfReceivedVpnRoutes( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noMplsNexthopSelfReceivedVpnRoutes( mode, args )

RouterBgpBaseAfMplsVpnV4Mode.addCommandClass(
    NeighborDefaultEncapMplsNhselfReceivedVpnv4RoutesCmd )

#--------------------------------------------------------------------------------
# [ no | default ] neighbor default encapsulation mpls next-hop-self
# received-vpnv6-routes
# in "address-family vpn-ipv6" mode
#--------------------------------------------------------------------------------

matcherReceivedVpnv6Routes = CliMatcher.KeywordMatcher( 'received-vpnv6-routes',
      helpdesc='Use local labels for advertising received VPN-IPv6 routes' )

class NeighborDefaultEncapMplsNhselfReceivedVpnv6RoutesCmd(
      CliCommand.CliCommandClass ):
   syntax = 'neighbor default encapsulation mpls next-hop-self received-vpnv6-routes'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor' : bgpTokens.neighbor,
      'default' : bgpTokens.default,
      'encapsulation' : bgpTokens.encap,
      'mpls' : bgpTokens.mpls,
      'next-hop-self' : bgpTokens.nextHopSelf,
      'received-vpnv6-routes' : matcherReceivedVpnv6Routes,
   }

   @staticmethod
   def handler( mode, args ):
      setMplsNexthopSelfReceivedVpnRoutes( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noMplsNexthopSelfReceivedVpnRoutes( mode, args )

RouterBgpBaseAfMplsVpnV6Mode.addCommandClass(
    NeighborDefaultEncapMplsNhselfReceivedVpnv6RoutesCmd )

#---------------------------------------------------------------------------------
# "[no|default] next-hop resolution disabled" in "address-family vpn-ipv4" and
# "address-family vpn-ipv6" modes
#---------------------------------------------------------------------------------
RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( NexthopResolutionDisabledCmd )

#---------------------------------------------------------------------------------
# "[no|default] route import match-failure action discard"
# in "address-family vpn-ipv4" and "address-family vpn-ipv6" modes.
#
# Enables the discarding of MPLSVPN paths that won't be imported into any VRF
# (AID6625).
#---------------------------------------------------------------------------------

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass(
   RouteImportMatchFailureDiscardCmd )


def Plugin( entityManager ):
   afModeExtensionHook.addAfModeExtension( 'vpn-ipv4', RouterBgpBaseAfMplsVpnV4Mode )
   afModeExtensionHook.addAfModeExtension( 'vpn-ipv6', RouterBgpBaseAfMplsVpnV6Mode )
