#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from enum import Enum
from eunuchs.in_h import IPPROTO_UDP
import os
import sys

import AclCliLib
import Arnet
import Arnet.MplsLib
import BasicCli
import ConfigMount
import CmdExtension
import CliCommand
from ClientCommonLib import LspPingTypeBgpLu
import CliMatcher
from CliMode.MplsUtilsMode import MplsUtilsMode
import CliParser
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliModel as AclCliModel
import CliPlugin.IntfCli as IntfCli
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import CliPlugin.IraNexthopGroupCli as IraNexthopGroupCli
import CliPlugin.MacAddr as MacAddr
import CliPlugin.MplsCli as MplsCli
import CliPlugin.NetworkToolsCli as NetworkToolsCli
import CliPlugin.VrfCli as VrfCli
import CliToken.Ip
import CliToken.Clear
import LazyMount
import ShowCommand
from SrTePolicyCommonLib import tacColor
import Toggles.ArBgpToggleLib
import Toggles.MplsUtilsToggleLib
from TypeFuture import TacLazyType

allNhgStatus = None
mplsUtilsConfig = None
mplsHwCapability = None
aclConfig = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None

ArnetIpAddr = TacLazyType( 'Arnet::IpAddr' )
ArnetIp6Addr = TacLazyType( 'Arnet::Ip6Addr' )
ArnetIpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
AddressFamily = TacLazyType( 'Arnet::AddressFamily' )
Ip6Prefix = TacLazyType( 'Arnet::Ip6Prefix' )
MplsOamStandard = TacLazyType( 'MplsUtils::MplsOamStandard' )
Prefix = TacLazyType( 'Arnet::Prefix' )
RsvpCliId = TacLazyType( 'Rsvp::RsvpCliId' )
LspPingConstant = TacLazyType( 'LspPing::LspPingConstant' )
U32_MAX_VALUE = 0xFFFFFFFF
U8_MAX_VALUE = 0xFF
maxTtlVal = 64
LspPingUtil = 'LspPing'
LspPingService = LspPingConstant.lspPingServiceName
LspPingPort = LspPingConstant.lspPingUdpPort
LspTracerouteUtil = 'LspTraceroute'
TacDscp = TacLazyType( 'Arnet::DscpValue' )

def mplsLabelRange( mode ):
   return ( Arnet.MplsLib.labelMin, Arnet.MplsLib.labelMax )

def getNhgNames( mode ):
   # We could just reutrn all MPLS nexthop-groups here.
   # But since LspPingUtil will check this anyway, it should be fine to
   # return all nexthop-groups.
   return sorted( allNhgStatus.nexthopGroupNameToId.members() )

def nhgEntryIndexRange( mode ):
   return ( 0, IraNexthopGroupCli.nexthopGroupSizeMax( mode ) - 1 )

# ping and traceroute mpls currently don't support non-default VRF.
# This guard will prevent to use 'vrf' option in the cli until the VRF support is
# implemented.
def mplsUtilvVrfSupportedGuard( mode, token ):
   return CliParser.guardNonDefaultVrf

nodeMpls = CliCommand.Node( CliMatcher.KeywordMatcher( 'mpls',
      helpdesc='Send echo messages over a LSP' ),
      guard=MplsCli.mplsSupportedGuard )

# As long as the only purpose of the 'mpls ping' CLI is to configure ACLs, this
# token will be guarded by the hwEpoch of service ACLs. This will prevent both
# configuring ACLs and showing/clearing counters.
matcherPing = CliMatcher.KeywordMatcher( 'ping',
      helpdesc='MPLS Ping and Traceroute configuration' )

matcherAddrOrPrefix = IpAddrMatcher.IpAddrOrPrefixExprFactory(
   'Destination address', 'Destination prefix mask', 'Destination prefix',
   overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT )
matcherAddrOrPrefix6 = Ip6AddrMatcher.Ip6AddrOrPrefixExprFactory(
   overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT )
matcherBgp = CliMatcher.KeywordMatcher( 'bgp', helpdesc='BGP route' )
matcherBgpLu = CliMatcher.KeywordMatcher( 'labeled-unicast',
                                          helpdesc='Labeled Unicast route' )
matcherBgpNexthop = CliMatcher.KeywordMatcher( 'bgp', helpdesc='BGP next hop' )
matcherRaw = CliMatcher.KeywordMatcher( 'raw', helpdesc='Raw route' )
matcherLdp = CliMatcher.KeywordMatcher( 'ldp', helpdesc='LDP route' )
matcherRsvp = CliMatcher.KeywordMatcher( 'rsvp', helpdesc='RSVP route' )
matcherSegmentRouting = CliMatcher.KeywordMatcher( 'segment-routing',
      helpdesc='Segment route' )
matcherStatic = CliMatcher.KeywordMatcher(
                             'static', helpdesc='Static MPLS push label route' )
matcherNhg = CliMatcher.KeywordMatcher( 'nexthop-group',
      helpdesc='MPLS nexthop group' )
matcherTunnel = CliMatcher.KeywordMatcher( 'tunnel',
      helpdesc='Tunnel interface' )
matcherNhgForTunnel = CliMatcher.KeywordMatcher( 'nexthop-group',
      helpdesc='Nexthop group tunnel' )
matcherSrte = CliMatcher.KeywordMatcher( 'srte', helpdesc='SR-TE Policy' )
matcherEndpoint = CliMatcher.KeywordMatcher( 'endpoint',
      helpdesc='SR-TE Policy endpoint' )
matcherColor = CliMatcher.KeywordMatcher( 'color', helpdesc='SR-TE Policy color' )
matcherColorVal = CliMatcher.IntegerMatcher( tacColor.min, tacColor.max,
      helpdesc='Color Value' )
matcherTrafficAf = CliMatcher.KeywordMatcher( 'traffic-af',
      helpdesc='The type of traffic to be steered through the policy' )
matcherV4 = CliMatcher.KeywordMatcher( 'v4', helpdesc='IPv4 traffic' )
matcherV6 = CliMatcher.KeywordMatcher( 'v6', helpdesc='IPv6 traffic' )
matcherP2mp = CliMatcher.KeywordMatcher( 'p2mp',
      helpdesc='Point To MultiPoint Route' )
matcherMldp = CliMatcher.KeywordMatcher( 'mldp', helpdesc='Mldp Route' )
matcherLsdId = CliMatcher.KeywordMatcher( 'lsp-id',
      helpdesc='LSP ID opaque type' )
matcherMulticast = CliMatcher.KeywordMatcher( 'multicast',
      helpdesc='Multicast opaque type' )
matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4', helpdesc='Ipv4 type' )
matcherIp = CliMatcher.KeywordMatcher( 'ip', helpdesc='IPv4 route' )
matcherIp6 = CliMatcher.KeywordMatcher( 'ipv6', helpdesc='IPv6 route' )
matcherPrefix = IpAddrMatcher.ipPrefixExpr( 'Destination address',
                                            'Destination prefix mask',
                                            'Destination prefix',
                                       overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT )
matcherPrefix6 = Ip6AddrMatcher.Ip6PrefixValidMatcher( 'IPv6 Address prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_REJECT )
matcherSession = CliMatcher.KeywordMatcher( 'session', helpdesc='RSVP session' )
matcherId = CliMatcher.KeywordMatcher( 'id', helpdesc='Specify session by ID' )
matcherIdVal = CliMatcher.IntegerMatcher( RsvpCliId.minValue, RsvpCliId.maxValue,
      helpdesc='RSVP session ID' )
matcherName = CliMatcher.KeywordMatcher( 'name', helpdesc='Specify session by name' )
matcherNameVal = CliMatcher.PatternMatcher( '[!-~]+', helpname='WORD',
      helpdesc='RSVP session name' )
matcherLsp = CliMatcher.KeywordMatcher( 'lsp', helpdesc='Specify LSP' )
matcherLspVal = CliMatcher.IntegerMatcher( RsvpCliId.minValue, RsvpCliId.maxValue,
      helpdesc='RSVP LSP ID' )
nodeEgressValidate = CliCommand.singleKeyword( 'egress-validate',
      helpdesc='Perform validation for this Label Switched Path at the egress' )
nodeAddress = CliCommand.singleKeyword( 'address',
      helpdesc=( 'endpoint address of the Label Switched Path that needs to be '
                 'validated' ) )
nodeLabel = CliCommand.guardedKeyword( 'label',
      helpdesc='Push an MPLS label', guard=MplsCli.mplsSupportedGuard )
matcherLabelValue = CliMatcher.DynamicIntegerMatcher( mplsLabelRange,
      helpdesc='Value of the MPLS label' )
matcherOpaqueVal = CliMatcher.IntegerMatcher( 1, U32_MAX_VALUE,
      helpdesc='lsp-id value' )
matcherNexthopGroup = CliMatcher.DynamicNameMatcher( getNhgNames,
      'Nexthop group name' )
matcherEntry = CliMatcher.KeywordMatcher( 'entry',
      helpdesc='Index value for the entry' )
matcherEntryIndex = CliMatcher.DynamicIntegerMatcher( nhgEntryIndexRange,
      helpdesc='Index value for the entry' )
matcherNexthop = CliMatcher.KeywordMatcher( 'nexthop',
      helpdesc='IP Address of nexthop' )
matcherNexthopIpAddr = IpAddrMatcher.IpAddrMatcher( 'Nexthop IP Address' )
matcherNexthopIp6Addr = Ip6AddrMatcher.Ip6AddrMatcher( 'Nexthop IPv6 Address' )
lspPingVrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='LSP ping in VRF',
      guard=mplsUtilvVrfSupportedGuard )
lspTracerouteVrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='LSP traceroute in VRF',
      guard=mplsUtilvVrfSupportedGuard )

class LspOptionType( Enum ):
   ping = 'ping'
   mldpPing = 'mldpPing'
   traceroute = 'traceroute'
   mldpTraceroute = 'mldpTraceroute'
   mplsTracerouteSr = 'mplsTracerouteSr'
   ldpTraceroute = 'ldpTraceroute'

class SourceExpr( CliCommand.CliExpression ):
   expression = 'source SRC_IP'
   data = {
      'source' : CliCommand.singleKeyword( 'source',
         helpdesc='Specify source address' ),
      'SRC_IP' : IpGenAddrMatcher.IpGenAddrMatcher(
         'IPv4 or IPv6 source address',
         helpdesc4='IPv4 address used as a source',
         helpdesc6='IPv6 source address' )
   }

class TrafficClassExpr( CliCommand.CliExpression ):
   expression = 'traffic-class TRAFFIC_CLASS'
   data = {
      'traffic-class' : CliCommand.singleKeyword( 'traffic-class',
         helpdesc='Specify MPLS traffic class field' ),
      'TRAFFIC_CLASS' : CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='MPLS traffic class field value' )
   }

class StandardExpr( CliCommand.CliExpression ):
   expression = 'standard STANDARD'
   data = {
      'standard' : CliCommand.singleKeyword( 'standard',
         helpdesc='Set the standard to comply with' ),
      'STANDARD' : CliMatcher.EnumMatcher( {
         'arista' : ( 'Comply with the Arista standard, '
                      'which allows interworking with older EOS devices' ),
         'ietf' : 'Comply with the IETF standard'
      } ),
   }

def dscpRange():
   return TacDscp.min, TacDscp.max

def dscpAclNames( mode ):
   return { k : v[ 1 ] for k, v in AclCliLib.dscpAclNames.items() }

class TosExpr( CliCommand.CliExpression ):
   expression = 'tos ( TOS_VAL | ( dscp ( DSCP | DSCP_ACL ) ) )'
   data = {
      'tos' : CliCommand.singleKeyword( 'tos',
         helpdesc='Specify ToS value' ),
      'TOS_VAL' : CliMatcher.IntegerMatcher( 0, U8_MAX_VALUE,
         helpdesc='ToS value' ),
      'dscp' : CliCommand.singleKeyword( 'dscp',
         helpdesc='Specify DSCP value' ),
      'DSCP' : CliMatcher.IntegerMatcher( *dscpRange(),
         helpdesc='DSCP value' ),
      'DSCP_ACL' : CliMatcher.DynamicKeywordMatcher( dscpAclNames )
   }

class LspOptionExprFactory( CliCommand.CliExpressionFactory ):
   '''
   The LSP protocols won't necessarily have the same options across all of them. This
   generator class lets us customize this by passing in other specific options that
   will be present in the generated CliExpression object.
   '''
   def __init__( self, optionType, multiLabel=False, egressValidate=False ):
      self.optionType = optionType
      self.multiLabel = multiLabel
      self.egressValidate = egressValidate
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      # Base option expr/data that's used by all protocols' CLI traceroute/ping cmds
      optionExpr = ( 'SOURCE | STANDARD | TRAFFIC_CLASS | TOS | '
                     'pad-reply | ( size SIZE ) ' )
      optionData = {
         'SOURCE' : SourceExpr,
         'STANDARD' : StandardExpr,
         'TRAFFIC_CLASS' : TrafficClassExpr,
         'TOS' : TosExpr,
         'pad-reply' : CliCommand.singleKeyword( 'pad-reply',
            helpdesc='Indicates that the reply should copy the Pad TLV' ),
         'size' : CliCommand.singleKeyword( 'size',
            helpdesc='Specify packet size in bytes' ),
         'SIZE' : CliMatcher.IntegerMatcher( 120, 10000,
            helpdesc='Packet size in bytes' ),
      }

      # Add the specific expr/data fields based on the given option type
      if ( self.optionType == LspOptionType.ping or
           self.optionType == LspOptionType.mldpPing ):
         optionExpr += '| ( repeat REPEAT ) '
         optionData.update( { 'repeat' : NetworkToolsCli.repeatKwNode,
                              'REPEAT' : NetworkToolsCli.countRangeMatcher } )
      elif ( self.optionType == LspOptionType.traceroute or
             self.optionType == LspOptionType.mldpTraceroute ):
         optionExpr += ( '| ( max-ttl TTL ) '
                         '| ( timeout TIME ) ' )
         optionData.update( {
            'max-ttl' : CliCommand.singleKeyword( 'max-ttl',
               helpdesc='Maximum Time-To-Live' ),
            'TTL' : CliMatcher.IntegerMatcher( 1, maxTtlVal, helpdesc='TTL range' ),
            'timeout' : CliCommand.singleKeyword( 'timeout',
               helpdesc='Time for echo reply to be received' ),
            'TIME' : CliMatcher.IntegerMatcher( 0, U32_MAX_VALUE,
               helpdesc='timeout for echo reply' )

         } )
      elif self.optionType == LspOptionType.ldpTraceroute:
         optionExpr += '| multipath | DS_TYPE | ( base BASE ) | ( count COUNT )'
         optionData.update( {
            'multipath' : CliCommand.singleKeyword( 'multipath',
               helpdesc='Enable multipath traceroute' ),
            'DS_TYPE' : CliCommand.singleNode(
               matcher=CliMatcher.EnumMatcher( {
                  'dsmap' : 'Use the Downstream Mapping TLV',
                  'ddmap' : 'Use the Downstream Detailed Mapping TLV',
               } ) ),
            'base' : CliCommand.singleKeyword( 'base',
               helpdesc='Base IP to use in multipath traceroute' ),
            'BASE' : IpGenAddrMatcher.IpGenAddrMatcher(
               'IP base address',
               helpdesc4='IPv4 base address',
               helpdesc6='IPv6 base address' ),
            'count' : CliCommand.singleKeyword( 'count',
               helpdesc='Number of IPs to probe in multipath traceroute' ),
            'COUNT' : CliMatcher.IntegerMatcher( 1, 512,
               helpdesc='Number of IPs to probe in multipath traceroute' )
         } )
      elif self.optionType == LspOptionType.mplsTracerouteSr:
         # don't add any options right now
         pass
      else:
         raise ValueError( 'Invalid optionType: {}'.format( self.optionType ) )
      # extra args for mldp ping and traceroute
      if ( self.optionType == LspOptionType.mldpPing or
           self.optionType == LspOptionType.mldpTraceroute ):
         optionExpr += ( '| ( jitter JITTER ) '
                         '| ( node-responder-id RESPADDR ) ' )
         optionData.update( {
            'jitter' : CliCommand.singleKeyword( 'jitter', helpdesc='Echo Jitter' ),
            'JITTER' : CliMatcher.IntegerMatcher( 1, U32_MAX_VALUE,
               helpdesc='jitter value in millisecs' ),
            'node-responder-id' : CliCommand.singleKeyword( 'node-responder-id',
               helpdesc='Node responder address' ),
            'RESPADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
               helpdesc='Node Responder Address' )
         } )
      if self.multiLabel:
         optionExpr += '| ( multi-label MULTI_LABEL ) '
         optionData.update( {
            'multi-label' : CliCommand.singleKeyword( 'multi-label',
               helpdesc=( 'Perform validation using the specified number of '
                          'top-labels during route lookups' ) ),
            # Change this to use an integer matcher if we ever support
            # more than 2 labels
            'MULTI_LABEL' : CliCommand.singleKeyword(
               str( MplsCli.maxIngressMplsTopLabels ),
               helpdesc=( 'Number of top-labels to use for validation '
                          'during route lookups' ) )
         } )

      if self.egressValidate:
         optionExpr += '| ( egress-validate [ address EGRESS_ADDRESS ] ) '
         optionData.update( { 'egress-validate' : nodeEgressValidate,
                             'address' : nodeAddress,
                             'EGRESS_ADDRESS' : IpGenAddrMatcher.IpGenAddrMatcher(
                                helpdesc='Expected egress address' ) } )

      class LspOptionExpr( CliCommand.CliExpression ):
         expression = '{ %s }' % optionExpr
         data = optionData

         @staticmethod
         def adapter( mode, args, argsList ):
            options = set()
            # No need to conditionally create this dict - just put all possible args
            # in here, since they're gated by the actual expression syntax
            optionValDict = {
               'base' : 'BASE',
               'count' : 'COUNT',
               'multi-label' : 'MULTI_LABEL',
               'multipath' : 'multipath',
               'pad-reply' : 'pad-reply',
               'repeat' : 'REPEAT',
               'size' : 'SIZE',
               'source' : 'SRC_IP',
               'standard' : 'STANDARD',
               'traffic-class' : 'TRAFFIC_CLASS',
               'max-ttl' : 'TTL',
               'timeout' : 'TIME',
               'jitter' : 'JITTER',
               'node-responder-id' : 'RESPADDR',
               }
            for option, optionVal in optionValDict.iteritems():
               if option in args:
                  val = args[ optionVal ]
                  val = val[ 0 ] if isinstance( val, list ) else val
                  options.add( ( option, val ) )
            args[ name ] = options
            if 'DS_TYPE' in args:
               options.add( ( 'dstype', args[ 'DS_TYPE' ] ) )
            if 'egress-validate' in args:
               if 'address' in args:
                  egressAddress = args[ 'EGRESS_ADDRESS' ]
                  if isinstance( egressAddress, list ):
                     egressAddress = egressAddress[ 0 ]
                  options.add( ( 'egress-validate', egressAddress ) )
               else:
                  options.add( ( 'egress-validate', 'default' ) )
            if 'tos' in args:
               tosVal, dscpVal = 0, None
               if 'dscp' in args:
                  dscpVal = args.get( 'DSCP' )
                  if isinstance( dscpVal, list ):
                     dscpVal = dscpVal[ 0 ]
                  if not dscpVal:
                     dscpAclVal = args.get( 'DSCP_ACL' )
                     if isinstance( dscpAclVal, list ):
                        dscpAclVal = dscpAclVal[ 0 ]
                     dscpVal, _ = AclCliLib.dscpValueFromCli( mode, dscpAclVal )
                  tosVal = dscpVal << 2 # dscp val is first 6 bits of tos
               else:
                  tosVal = args[ 'TOS_VAL' ]
                  if isinstance( tosVal, list ):
                     tosVal = tosVal[ 0 ]
               options.add( ( 'tos', tosVal ) )
      return LspOptionExpr

lspPingOptionsExpr = LspOptionExprFactory( LspOptionType.ping )
lspTracerouteOptionsExpr = LspOptionExprFactory( LspOptionType.traceroute )

def buildLspPingArgs( options ):
   defaultArgs = {}
   for o in options:
      defaultArgs[ o[ 0 ] ] = o[ 1 ]
   optionToArg = { 'repeat' : '--count',
                   'pad-reply' : '--padReply',
                   'tos' : '--tos',
                   'size' : '--size',
                   'source' : '--src',
                   'standard' : '--standard',
                   'traffic-class' : '--tc',
                   'multi-label' : '--multiLabel',
                   'egress-validate' : '--egressValidateAddress',
                   'jitter' : '--jitter',
                   'node-responder-id' : '--responderAddr' }
   args = []
   for arg, val in defaultArgs.items():
      args.append( optionToArg[ arg ] )
      args.append( str( val ) )
   return args

def validateLspTracerouteArgs( options, mode ):
   defaultArgs = {}
   for o in options:
      defaultArgs[ o[ 0 ] ] = o[ 1 ]

   if 'base' in defaultArgs:
      testGenAddr = defaultArgs[ 'base' ]

      if testGenAddr.af == AddressFamily.ipv4:
         loopbackPrefix = Prefix( ArnetIpAddr.ipAddrLoopbackBase, 8 )
         testAddr = testGenAddr.v4Addr
      else:
         loopbackPrefix = Ip6Prefix( ArnetIp6Addr( 0, 0, 0xffff, 0x7f000000 ), 108 )
         testAddr = testGenAddr.v6Addr

      if not loopbackPrefix.contains( testAddr ):
         mode.addError( 'base must be within %s' % loopbackPrefix )
         return False

   return True

def buildLspTracerouteArgs( options ):
   defaultArgs = {}
   for o in options:
      defaultArgs[ o[ 0 ] ] = o[ 1 ]
   optionToArg = { 'source' : '--src',
                   'traffic-class' : '--tc',
                   'pad-reply' : '--padReply',
                   'tos' : '--tos',
                   'size' : '--size',
                   'standard' : '--standard',
                   'base' : '--multipathbase',
                   'count' : '--multipathcount',
                   'timeout' : '--interval',
                   'max-ttl' : '--maxTtl',
                   'dstype' : '--dstype',
                   'multi-label' : '--multiLabel',
                   'egress-validate' : '--egressValidateAddress',
                   'jitter' : '--jitter',
                   'node-responder-id' : '--responderAddr',
                 }
   optionToArglessArg = { 'multipath' : '--multipath', }
   args = []
   for arg, val in defaultArgs.items():
      if arg in optionToArg:
         args.append( optionToArg[ arg ] )
         args.append( str( val ) )
      if arg in optionToArglessArg:
         args.append( optionToArglessArg[ arg ] )
   return args

def doCallLspUtil( mode, lspUtil, args, vrfName=VrfCli.DEFAULT_VRF, ipv=4 ):
   cliCmdExt = CmdExtension.getCmdExtender()
   args = [ 'sudo', lspUtil ] + args
   try:
      # Flush stdout before spawning subprocess as the output to stdout is
      # buffered: that is, print statements actually write to a buffer, and
      # this buffer is only occassionally flushed to the terminal. Each
      # process has a separate buffer, which is why writes from different
      # processes can appear out of order
      sys.stdout.flush()
      # breadth test env
      # If testing arguments passed to the script, calling it with print-arg option.
      if os.environ.get( 'MOCK_LSP_UTIL' ) == 'printArg':
         args.append( '--print-arg' )
         p = cliCmdExt.subprocessPopen( args, mode.session, vrfName=vrfName,
                                        stdout=sys.stdout, stderr=sys.stderr )
         p.communicate()
      elif 'MOCK_LSP_UTIL' in os.environ:
         cliCmdExt.extendCmd( args, mode.session, vrfName=vrfName )
      else:
         p = cliCmdExt.subprocessPopen( args, mode.session, vrfName=vrfName,
                                        stdout=sys.stdout, stderr=sys.stderr )
         p.communicate()
   except EnvironmentError as e:
      mode.addError( e.strerror )

#--------------------------------------------------------------------------------
# ping mpls [ VRF ] raw label LABEL ( ( nexthop NEXTHOP ) | ( MACADDR INTF ) )
#                                                                        [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingRawCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ping mpls [ VRF ] raw label LABEL '
                           '( ( nexthop NEXTHOP ) | ( MACADDR INTF ) ) [ OPTIONS ]' )
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspPingVrfExprFactory,
      'raw' : matcherRaw,
      'label' : nodeLabel,
      'LABEL' : matcherLabelValue,
      'nexthop' : matcherNexthop,
      'NEXTHOP' : matcherNexthopIpAddr,
      'MACADDR' : MacAddr.macAddrMatcher,
      'INTF' : IntfCli.Intf.matcher,
      'OPTIONS' : lspPingOptionsExpr
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF' )
      options = args.get( 'OPTIONS' )
      cmdArgs = [ '--type', 'raw', '--label', str( args[ 'LABEL' ] ) ]
      cmdArgs += buildLspPingArgs( options )
      if 'MACADDR' in args:
         cmdArgs += [ '--dmac', str( args[ 'MACADDR' ] ),
                   '--interface', str( args[ 'INTF' ] ), '127.0.0.1' ]
         print 'LSP raw ping to {}'.format( args[ 'MACADDR' ] )
      else:
         destAddr = str( args[ 'NEXTHOP' ] )
         cmdArgs += [ '--nexthop', destAddr, destAddr ]
         print 'LSP raw ping to {}'.format( destAddr )
      doCallLspUtil( mode, LspPingUtil, cmdArgs, vrfName=vrfName )

BasicCli.EnableMode.addCommandClass( PingRawCmd )

# ------------------------------------------------------------------------------
# Mpls Utils configuration mode
# ------------------------------------------------------------------------------
class MplsUtilsConfigMode( MplsUtilsMode, BasicCli.ConfigModeBase ):
   name = 'Mpls Utils Configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName='default' ):
      self.vrfName = vrfName
      self.mode = MplsUtilsConfigMode
      MplsUtilsMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   # pylint: disable-msg=E1103
   def doSetLspPingAclName( self, aclType, aclName, vrfName ):
      if vrfName is None:
         vrfName = VrfCli.DEFAULT_VRF
      AclCliLib.setServiceAcl( self, LspPingService, IPPROTO_UDP,
                               aclConfig, aclCpConfig, aclName,
                               aclType=aclType, vrfName=vrfName,
                               port=[ LspPingPort ], sport=[ LspPingPort ] )

   # pylint: disable-msg=E1103
   def doDelLspPingAclName( self, aclType, vrfName ):
      if vrfName is None:
         vrfName = VrfCli.DEFAULT_VRF
      AclCliLib.noServiceAcl( self, LspPingService, aclConfig, aclCpConfig,
                              None, aclType=aclType, vrfName=vrfName )

def noLspPingServiceAcl( mode, vrfName=VrfCli.DEFAULT_VRF ):
   mode.doDelLspPingAclName( 'ip', vrfName )

def noLspPingServiceAclV6( mode, vrfName=VrfCli.DEFAULT_VRF ):
   mode.doDelLspPingAclName( 'ipv6', vrfName )

#--------------------------------------------------------------------------------
# ping mpls [ VRF ] ldp ip IP_ADDRESS [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingLdpCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ping mpls [ VRF ] ldp ip IP_ADDRESS [ OPTIONS ]' )
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspPingVrfExprFactory,
      'ldp' : matcherLdp,
      'ip' : matcherIp,
      'IP_ADDRESS' : matcherPrefix,
      'OPTIONS' : lspPingOptionsExpr
   }

   @staticmethod
   def handler( mode, args ):
      prefix = args[ 'IP_ADDRESS' ]
      if not prefix:
         mode.addError( 'Invalid prefix' )
         return
      vrfName = args.get( 'VRF' )
      options = args.get( 'OPTIONS' )
      args = [ prefix if isinstance( prefix, str ) else prefix.stringValue,
               '--type', 'ldp' ] + buildLspPingArgs( options )
      print 'LSP ping to LDP route {}'.format( prefix )
      doCallLspUtil( mode, LspPingUtil, args, vrfName=vrfName )

BasicCli.EnableMode.addCommandClass( PingLdpCmd )

#--------------------------------------------------------------------------------
# ping mpls p2mp mldp ROOTADDR ( ( lsp-id OPQVAL ) |
#                                ( multicast ipv4 SOURCEADDR GROUPADDR ) )
#                                [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingMplsP2mpMldpCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ping mpls p2mp mldp ROOTADDR '
              '( ( lsp-id OPQVAL ) | ( multicast ipv4 SOURCEADDR GROUPADDR ) ) '
              '[ OPTIONS ]'
            )
   data = {
           'ping' : NetworkToolsCli.pingMatcher,
           'mpls' : nodeMpls,
           'p2mp' : matcherP2mp,
           'mldp' : matcherMldp,
           'ROOTADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                             'Root Ip Address',
                             helpdesc4='IPv4 Root Address',
                             helpdesc6='IPv6 Root Address' ),
           'lsp-id' : matcherLsdId,
           'OPQVAL' : matcherOpaqueVal,
           'multicast' : matcherMulticast,
           'ipv4' : matcherIpv4,
           'SOURCEADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                             'IP address of the source',
                             helpdesc4='IPv4 Source Address',
                             helpdesc6='IPv6 Source Address' ),
           'GROUPADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                             'Group Ip Address',
                             helpdesc4='IPv4 Group Address',
                             helpdesc6='IPv6 Group Address' ),
           'OPTIONS' : LspOptionExprFactory( LspOptionType.mldpPing )
          }

   @staticmethod
   def handler( mode, args ):
      rootAddr = args.get( 'ROOTADDR' )
      if rootAddr:
         opqVal = args.get( 'OPQVAL' )
         sourceAddr = args.get( 'SOURCEADDR' )
         groupAddr = args.get( 'GROUPADDR' )
         lspArgs = [ rootAddr.stringValue ]
         lspArgs += [ '--type', 'mldp' ]
         if opqVal:
            lspArgs += [ '--genOpqVal', str( opqVal ) ]
         elif sourceAddr and groupAddr:
            lspArgs += [ '--sourceAddrOpqVal', sourceAddr.stringValue ]
            lspArgs += [ '--groupAddrOpqVal', groupAddr.stringValue ]
         else:
            mode.addError( 'Invalid Opaque Value' )
            return
         lspArgs += buildLspPingArgs( args[ 'OPTIONS' ] )
         print 'LSP ping to Mldp route {}'.format( rootAddr )
         doCallLspUtil( mode, LspPingUtil, lspArgs )
      else:
         mode.addError( 'Invalid Root Addr' )

BasicCli.EnableMode.addCommandClass( PingMplsP2mpMldpCmd )

#--------------------------------------------------------------------------------
# traceroute mpls p2mp mldp ROOTADDR ( ( lsp-id OPQVAL ) |
#                                      ( multicast ipv4 SOURCEADDR GROUPADDR ) )
#                                    [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteMplsP2mpMldpCmd( CliCommand.CliCommandClass ):
   syntax = ( 'traceroute mpls p2mp mldp ROOTADDR '
              '( ( lsp-id OPQVAL ) | ( multicast ipv4 SOURCEADDR GROUPADDR ) ) '
              '[ OPTIONS ]'
            )
   data = {
            'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
            'mpls' : nodeMpls,
            'p2mp' : matcherP2mp,
            'mldp' : matcherMldp,
            'ROOTADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                              'Root Ip Address',
                              helpdesc4='IPv4 Root Address',
                              helpdesc6='IPv6 Root Address' ),
            'lsp-id' : matcherLsdId,
            'OPQVAL' : matcherOpaqueVal,
            'multicast' : matcherMulticast,
            'ipv4' : matcherIpv4,
            'SOURCEADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                                'IP address of the source',
                                helpdesc4='IPv4 Source Address',
                                helpdesc6='IPv6 Source Address' ),
            'GROUPADDR' : IpGenAddrMatcher.IpGenAddrMatcher(
                               'Group Ip Address',
                                helpdesc4='IPv4 Source Address',
                                helpdesc6='IPv6 Source Address' ),
            'OPTIONS' : LspOptionExprFactory( LspOptionType.mldpTraceroute )
           }

   @staticmethod
   def handler( mode, args ):
      rootAddr = args[ 'ROOTADDR' ]
      if rootAddr:
         opqVal = args.get( 'OPQVAL' )
         sourceAddr = args.get( 'SOURCEADDR' )
         groupAddr = args.get( 'GROUPADDR' )
         lspArgs = [ rootAddr.stringValue ]
         lspArgs += [ '--type', 'mldp' ]
         if opqVal:
            lspArgs += [ '--genOpqVal', str( args[ 'OPQVAL' ] ) ]
         elif sourceAddr and groupAddr:
            lspArgs += [ '--sourceAddrOpqVal', sourceAddr.stringValue ]
            lspArgs += [ '--groupAddrOpqVal', groupAddr.stringValue ]
         else:
            mode.addError( 'Invalid Opaque Value' )
            return
         lspArgs += buildLspTracerouteArgs( args[ 'OPTIONS' ] )
         print 'LSP traceroute to Mldp route {}'.format( rootAddr )
         doCallLspUtil( mode, LspTracerouteUtil, lspArgs )
      else:
         mode.addError( 'Invalid Root Addr' )

BasicCli.EnableMode.addCommandClass( TracerouteMplsP2mpMldpCmd )

#--------------------------------------------------------------------------------
# ping mpls segment-routing ( ( ip IP_ADDRESS ) | ( ipv6 IPV6_ADDRESS ) ) [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingSrCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ping mpls [ VRF ] segment-routing '
                        '( ( ip IP_ADDRESS ) | ( ipv6 IPV6_ADDRESS ) ) [ OPTIONS ]' )
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspPingVrfExprFactory,
      'segment-routing' : matcherSegmentRouting,
      'ip' : matcherIp,
      'IP_ADDRESS' : matcherPrefix,
      'ipv6' : matcherIp6,
      'IPV6_ADDRESS' : matcherPrefix6,
      'OPTIONS' : lspPingOptionsExpr
   }

   @staticmethod
   def handler( mode, args ):
      prefix = args.get( 'IP_ADDRESS', args.get( 'IPV6_ADDRESS' ) )
      if not prefix:
         mode.addError( 'Invalid prefix' )
         return
      vrfName = args.get( 'VRF' )
      options = args.get( 'OPTIONS' )
      args = [ prefix if isinstance( prefix, str ) else prefix.stringValue,
               '--type', 'segment-routing' ] + buildLspPingArgs( options )
      print 'LSP ping to Segment-Routing route {}'.format( prefix )
      doCallLspUtil( mode, LspPingUtil, args, vrfName=vrfName )

BasicCli.EnableMode.addCommandClass( PingSrCmd )

#--------------------------------------------------------------------------------
# ping mpls static ( ip IP_ADDRESS | ipv6 IPV6_ADDRESS )
#                             [ egress-validate address EGRESS_ADDRESS ] [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingMplsStaticCmd( CliCommand.CliCommandClass ):
   syntax = '''ping mpls static ( ( ip IP_ADDRESS ) | ( ipv6 IPV6_ADDRESS ) )
               [ egress-validate address EGRESS_ADDRESS ] [ OPTIONS ]'''
   data = { 'ping' : NetworkToolsCli.pingMatcher,
            'mpls' : nodeMpls,
            'static' : matcherStatic,
            'ip' : matcherIp,
            'IP_ADDRESS' : matcherPrefix,
            'ipv6' : matcherIp6,
            'IPV6_ADDRESS' : matcherPrefix6,
            'egress-validate' : nodeEgressValidate,
            'address' : nodeAddress,
            'EGRESS_ADDRESS' : IpGenAddrMatcher.IpGenAddrMatcher(
                                         helpdesc='Expected egress address' ),
            'OPTIONS' : LspOptionExprFactory( LspOptionType.ping,
                                              multiLabel=True )
          }

   @staticmethod
   def handler( mode, args ):
      fec = args[ 'IP_ADDRESS' ] if 'ip' in args else args[ 'IPV6_ADDRESS' ]
      if not fec:
         mode.addError( 'Invalid prefix' )
         return
      fec = str( fec )
      pingArgs = [ fec, '--type', 'static', ] + buildLspPingArgs( args[ 'OPTIONS' ] )
      print 'LSP ping to static MPLS push label route {}'.format( fec )
      doCallLspUtil( mode, LspPingUtil, pingArgs )

BasicCli.EnableMode.addCommandClass( PingMplsStaticCmd )

#--------------------------------------------------------------------------------
# ping mpls nexthop-group name [entry] [options]
# XXXXX NO VRF
#--------------------------------------------------------------------------------
class PingMplsNhgCmd( CliCommand.CliCommandClass ):
   syntax = 'ping mpls nexthop-group NHG [ entry ENTRY ] [ OPTIONS ]'
   data = { 'ping': NetworkToolsCli.pingMatcher,
            'mpls' : nodeMpls,
            'nexthop-group' : matcherNhg,
            'NHG' : matcherNexthopGroup,
            'entry' : matcherEntry,
            'ENTRY' : matcherEntryIndex,
            'OPTIONS' : LspOptionExprFactory( LspOptionType.ping, multiLabel=True )
          }

   @staticmethod
   def handler( mode, args ):
      nhgName = args[ 'NHG' ]
      lspArgs = [ nhgName, '--type', 'nexthop-group' ]
      entry = args.get( 'ENTRY' )
      if entry is not None:
         lspArgs += [ '--entry', str( entry ) ]
      lspArgs += buildLspPingArgs( args[ 'OPTIONS' ] )
      print 'LSP ping to nexthop-group {}'.format( nhgName )
      doCallLspUtil( mode, LspPingUtil, lspArgs )

BasicCli.EnableMode.addCommandClass( PingMplsNhgCmd )

#--------------------------------------------------------------------------------
# ping mpls tunnel nexthop-group ENDPOINT [entry] [options]
#--------------------------------------------------------------------------------
class PingMplsNhgTunnelCmd( CliCommand.CliCommandClass ):
   syntax = 'ping mpls tunnel nexthop-group ENDPOINT [ entry ENTRY ] [ OPTIONS ]'
   data = { 'ping': NetworkToolsCli.pingMatcher,
            'mpls' : nodeMpls,
            'tunnel' : matcherTunnel,
            'nexthop-group' : matcherNhgForTunnel,
            'ENDPOINT': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
                          ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                          ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
            'entry' : matcherEntry,
            'ENTRY' : matcherEntryIndex,
            'OPTIONS' : LspOptionExprFactory( LspOptionType.ping, multiLabel=True,
                                              egressValidate=True )
          }

   @staticmethod
   def handler( mode, args ):
      lspArgs = [ str( args[ 'ENDPOINT' ] ), '--type', 'nexthop-group-tunnel' ]
      if 'entry' in args:
         lspArgs += [ '--entry', str( args[ 'ENTRY' ] ) ]
      lspArgs += buildLspPingArgs( args[ 'OPTIONS' ] )
      print 'LSP ping to nexthop-group tunnel {}'.format( str( args[ 'ENDPOINT' ] ) )
      doCallLspUtil( mode, LspPingUtil, lspArgs )

BasicCli.EnableMode.addCommandClass( PingMplsNhgTunnelCmd )

#--------------------------------------------------------------------------------
# ping mpls srte endpoint ENDPOINT color COLOR [ traffic-af ( v4 | v6 ) ] [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingMplsSrTeCmd( CliCommand.CliCommandClass ):
   syntax = '''ping mpls srte endpoint ENDPOINT color COLOR
               [ traffic-af ( v4 | v6 ) ] [ OPTIONS ]'''
   data = { 'ping' : NetworkToolsCli.pingMatcher,
            'mpls' : nodeMpls,
            'srte' : matcherSrte,
            'endpoint' : matcherEndpoint,
            'ENDPOINT' : IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='Endpoint '
                                                            'IP/IPv6 address' ),
            'color' : matcherColor,
            'COLOR' : matcherColorVal,
            'traffic-af' : matcherTrafficAf,
            'v4' : matcherV4,
            'v6' : matcherV6,
            'OPTIONS' : LspOptionExprFactory( LspOptionType.ping,
                                              egressValidate=True ) }

   @staticmethod
   def handler( mode, args ):
      lspArgs = [ str( args[ 'ENDPOINT' ] ), '--type', 'SrTePolicy',
                  '--color', str( args[ 'COLOR' ] ) ]
      if 'traffic-af' in args:
         trafficAf = 'v4' if 'v4' in args else 'v6'
         lspArgs += [ '--trafficAf', trafficAf ]
      lspArgs += buildLspPingArgs( args[ 'OPTIONS' ] )
      displayStr = 'LSP ping to SR-TE Policy endpoint {} color {}'.format(
                                                        args[ 'ENDPOINT' ],
                                                        args[ 'COLOR' ] )
      if '--egressValidateAddress' in lspArgs:
         egressValidateAddress = \
                     lspArgs[ lspArgs.index( '--egressValidateAddress' ) + 1 ]
         displayStr += ' egress-validate: '
         displayStr += egressValidateAddress

      print displayStr

      doCallLspUtil( mode, LspPingUtil, lspArgs )

BasicCli.EnableMode.addCommandClass( PingMplsSrTeCmd )

#--------------------------------------------------------------------------------
# ping mpls rsvp session ( ( id SESSION_ID [ lsp LSP ] ) |
#                          ( name SESSION_NAME ) ) [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingRsvpCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ping mpls rsvp session ( ( id SESSION_ID [ lsp LSP ] ) | '
                                       '( name SESSION_NAME ) ) [ OPTIONS ]' )
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'rsvp' : matcherRsvp,
      'session' : matcherSession,
      'id' : matcherId,
      'SESSION_ID' : matcherIdVal,
      'lsp' : matcherLsp,
      'LSP' : matcherLspVal,
      'name' : matcherName,
      'SESSION_NAME' : matcherNameVal,
      'OPTIONS' : lspPingOptionsExpr,
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = None
      rsvpId = args.get( 'SESSION_ID' )
      name = args.get( 'SESSION_NAME' )
      lspOption = args.get( 'LSP' )
      options = args.get( 'OPTIONS' )
      if rsvpId:
         args = [ '--type', 'rsvp', '--session-id', str( rsvpId ) ]
         if lspOption:
            args += [ '--lsp', str( lspOption ) ]
         if options:
            args += buildLspPingArgs( options )
         if '--lsp' in args:
            idx = args.index( '--lsp' ) + 1
            print 'LSP ping to RSVP session #{} LSP #{}'.format( rsvpId,
                  args[ idx ] )
         else:
            print 'LSP ping to RSVP session #{}'.format( rsvpId )
         doCallLspUtil( mode, LspPingUtil, args, vrfName=vrfName )
      elif name:
         args = [ '--type', 'rsvp', '--session-name', str( name ) ]
         if options:
            args += buildLspPingArgs( options )
         print 'LSP ping to session {}'.format( name )
         doCallLspUtil( mode, LspPingUtil, args, vrfName=vrfName )
      else:
         mode.addError( 'No session' )

BasicCli.EnableMode.addCommandClass( PingRsvpCmd )

#--------------------------------------------------------------------------------
# ping mpls pseudowire ldp <pw-name> [options]
#--------------------------------------------------------------------------------
class PingPseudowireLdpCmd( CliCommand.CliCommandClass ):
   syntax = 'ping mpls pseudowire ldp PW_NAME [ OPTIONS ]'
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'pseudowire' : 'MPLS pseudowire',
      'ldp' : 'LDP pseudowire',
      'PW_NAME' : CliMatcher.PatternMatcher( '[!-~]+', helpname='WORD',
         helpdesc='Ldp pseudowire name' ),
      'OPTIONS' : lspPingOptionsExpr,
   }

   @staticmethod
   def handler( mode, args ):
      pwName = args[ 'PW_NAME' ]
      options = args.get( 'OPTIONS' )
      args = [ pwName, '--type', 'pwLdp' ]
      args += buildLspPingArgs( options )
      print 'LSP ping to LDP pseudowire {}'.format( pwName )
      doCallLspUtil( mode, LspPingUtil, args )

BasicCli.EnableMode.addCommandClass( PingPseudowireLdpCmd )

#--------------------------------------------------------------------------------
# ping mpls bgp labeled-unicast ( ip | ipv6 ) PREFIX
# [ bgp nexthop NEXTHOP [ label-stack { LABELS } ] ] [ OPTIONS ]
#--------------------------------------------------------------------------------
class PingBgpLuCmd( CliCommand.CliCommandClass ):
   syntax = '''ping mpls bgp labeled-unicast
               ( ip PREFIX [ BGP_N nexthop NEXTHOP [ label-stack { LABELS } ] ] ) |
               ( ipv6 PREFIX6 [ BGP_N nexthop NEXTHOP6 [ label-stack { LABELS } ] ] )
               [ OPTIONS ]'''
   data = {
      'ping' : NetworkToolsCli.pingMatcher,
      'mpls' : nodeMpls,
      'bgp' : matcherBgp,
      'labeled-unicast' : matcherBgpLu,
      'ip' : matcherIp,
      'PREFIX' : matcherAddrOrPrefix,
      'ipv6' : matcherIp6,
      'PREFIX6' : matcherAddrOrPrefix6,
      'BGP_N' : matcherBgpNexthop,
      'nexthop' : matcherNexthop,
      'NEXTHOP' : matcherNexthopIpAddr,
      'NEXTHOP6' : matcherNexthopIp6Addr,
      'label-stack' : MplsCli.labelStackKeywordNode,
      'LABELS' : MplsCli.labelStackValNode,
      'OPTIONS' : lspPingOptionsExpr,
   }

   @staticmethod
   def handler( mode, args ):
      prefix = args.get( 'PREFIX' ) or args.get( 'PREFIX6' )
      if not prefix:
         mode.addError( 'Invalid prefix' )
         return
      prefix = ArnetIpGenPrefix( str( prefix ) ).stringValue
      nexthop = args.get( 'NEXTHOP' ) or args.get( 'NEXTHOP6' )
      labels = args.get( 'LABELS' )
      if labels and not MplsCli.validateLabelStackSize( mode, labels ):
         return
      options = args.get( 'OPTIONS' )
      lspArgs = [ prefix, '--type', LspPingTypeBgpLu ]
      if nexthop:
         if not isinstance( nexthop, str ):
            nexthop = nexthop.stringValue
         lspArgs += [ '--nexthop', nexthop ]
         if labels:
            lspArgs += [ '--label', ','.join( str( l ) for l in labels ) ]
      lspArgs += buildLspPingArgs( options )
      print 'LSP ping to BGP labeled unicast tunnel', prefix
      doCallLspUtil( mode, LspPingUtil, lspArgs )

if Toggles.ArBgpToggleLib.toggleBgpLuLspPingEnabled():
   BasicCli.EnableMode.addCommandClass( PingBgpLuCmd )

#--------------------------------------------------------------------------------
# traceroute mpls [ VRF ] raw label LABEL ( ( nexthop NEXTHOP ) | ( MACADDR INTF ) )
#                                                                        [ OPTIONS ]
# IPv4 only at the moment
#--------------------------------------------------------------------------------
class TracerouteRawCmd( CliCommand.CliCommandClass ):
   syntax = ( 'traceroute mpls [ VRF ] raw label LABEL '
                           '( ( nexthop NEXTHOP ) | ( MACADDR INTF ) ) [ OPTIONS ]' )
   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspPingVrfExprFactory,
      'raw' : matcherRaw,
      'label' : nodeLabel,
      'LABEL' : matcherLabelValue,
      'nexthop' : matcherNexthop,
      'NEXTHOP' : matcherNexthopIpAddr,
      'MACADDR' : MacAddr.macAddrMatcher,
      'INTF' : IntfCli.Intf.matcher,
      'OPTIONS' : lspTracerouteOptionsExpr
   }

   @staticmethod
   def handler( mode, args ):
      cmdArgs = [ '--type', 'raw', '--label', str( args[ 'LABEL' ] ) ]
      cmdArgs += buildLspTracerouteArgs( args[ 'OPTIONS' ] )
      if 'MACADDR' in args:
         cmdArgs += [ '--dmac', args[ 'MACADDR' ],
                   '--interface', str( args[ 'INTF' ] ), '127.0.0.1' ]
         print 'LSP traceroute to {}'.format( args[ 'MACADDR' ] )
      else:
         destAddr = args[ 'NEXTHOP' ]
         cmdArgs += [ '--nexthop', destAddr, destAddr ]
         print 'LSP traceroute to {}'.format( destAddr )
      doCallLspUtil( mode, LspTracerouteUtil, cmdArgs, vrfName=args.get( 'VRF' ) )

BasicCli.EnableMode.addCommandClass( TracerouteRawCmd )

#--------------------------------------------------------------------------------
# traceroute mpls [ VRF ] ldp ip IP_ADDRESS [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteLdpCmd( CliCommand.CliCommandClass ):
   syntax = 'traceroute mpls [ VRF ] ldp ip IP_ADDRESS [ OPTIONS ]'
   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspTracerouteVrfExprFactory,
      'ldp' : matcherLdp,
      'ip' : matcherIp,
      'IP_ADDRESS' : matcherPrefix,
      'OPTIONS' : LspOptionExprFactory( LspOptionType.ldpTraceroute )
   }

   @staticmethod
   def handler( mode, args ):
      prefix = args[ 'IP_ADDRESS' ]
      if not prefix:
         mode.addError( 'Invalid prefix' )
         return
      vrfName = args.get( 'VRF' )
      options = args.get( 'OPTIONS' )

      args = [ prefix if isinstance( prefix, str ) else prefix.stringValue,
               '--type', 'ldp' ]

      if validateLspTracerouteArgs( options, mode ):
         args += buildLspTracerouteArgs( options )
         print 'LSP traceroute to {}'.format( prefix )
         doCallLspUtil( mode, LspTracerouteUtil, args, vrfName=vrfName )

BasicCli.EnableMode.addCommandClass( TracerouteLdpCmd )

#--------------------------------------------------------------------------------
# traceroute mpls [ VRF ] segment-routing
#                          ( ( ip IP_ADDRESS ) | ( ipv6 IPV6_ADDRESS ) ) [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteSrCmd( CliCommand.CliCommandClass ):
   syntax = ( 'traceroute mpls [ VRF ] segment-routing '
                     '( ( ip IP_ADDRESS ) | ( ipv6 IPV6_ADDRESS ) ) [ OPTIONS ]' )
   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'VRF' : lspTracerouteVrfExprFactory,
      'segment-routing' : matcherSegmentRouting,
      'ip' : matcherIp,
      'IP_ADDRESS' : matcherPrefix,
      'ipv6' : matcherIp6,
      'IPV6_ADDRESS' : matcherPrefix6,
      'OPTIONS' : lspTracerouteOptionsExpr,
   }

   @staticmethod
   def handler( mode, args ):
      prefix = args.get( 'IP_ADDRESS', args.get( 'IPV6_ADDRESS' ) )
      if not prefix:
         mode.addError( 'Invalid prefix' )
         return
      vrfName = args.get( 'VRF' )
      options = args.get( 'OPTIONS' )
      args = [ prefix if isinstance( prefix, str ) else prefix.stringValue,
               '--type', 'segment-routing' ]
      if not validateLspTracerouteArgs( options, mode ):
         return

      args += buildLspTracerouteArgs( options )
      print 'LSP traceroute to {}'.format( prefix )
      doCallLspUtil( mode, LspTracerouteUtil, args, vrfName=vrfName )

BasicCli.EnableMode.addCommandClass( TracerouteSrCmd )

#--------------------------------------------------------------------------------
# traceroute mpls static ( ip | ipv6 ) prefix [ nexthop <nexthop-ip> label <label> ]
# [ options ]
#--------------------------------------------------------------------------------
class TracerouteMplsStaticCmd( CliCommand.CliCommandClass ):
   syntax = '''traceroute mpls static
               ( ( ip IP_ADDRESS [ nexthop NEXTHOP label LABEL ] ) |
                 ( ipv6 IPV6_ADDRESS [ nexthop NEXTHOP6 label LABEL ] ) )
               [ egress-validate address EGRESS_ADDRESS ] [ OPTIONS ]'''
   data = { 'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
            'mpls' : nodeMpls,
            'static' : matcherStatic,
            'ip' : matcherIp,
            'IP_ADDRESS' : matcherPrefix,
            'ipv6' : matcherIp6,
            'IPV6_ADDRESS' : matcherPrefix6,
            'nexthop' : matcherNexthop,
            'NEXTHOP' : matcherNexthopIpAddr,
            'NEXTHOP6' : matcherNexthopIp6Addr,
            'label' : nodeLabel,
            'LABEL' : matcherLabelValue,
            'egress-validate' : nodeEgressValidate,
            'address' : nodeAddress,
            'EGRESS_ADDRESS' : IpGenAddrMatcher.IpGenAddrMatcher(
                                         helpdesc='Expected egress address' ),
            'OPTIONS' : LspOptionExprFactory( LspOptionType.traceroute,
                                              multiLabel=True )
          }

   @staticmethod
   def handler( mode, args ):
      prefix = args.get( 'IP_ADDRESS', args.get( 'IPV6_ADDRESS' ) )
      if not prefix:
         mode.addError( 'Invalid prefix' )
         return
      prefix = str( prefix )
      tracerouteArgs = [ prefix, '--type', 'static' ]
      if validateLspTracerouteArgs( args[ 'OPTIONS' ], mode ):
         tracerouteArgs += buildLspTracerouteArgs( args[ 'OPTIONS' ] )
         if 'nexthop' in args:
            nexthopIp = args[ 'NEXTHOP' ] if 'NEXTHOP' in args else \
                        args[ 'NEXTHOP6' ]
            nexthopIp = str( nexthopIp )
            label = str( args[ 'LABEL' ] )
            tracerouteArgs += [ '--label', label, '--nexthop', nexthopIp ]
         print 'LSP traceroute to {}'.format( prefix )
         doCallLspUtil( mode, LspTracerouteUtil, tracerouteArgs )

BasicCli.EnableMode.addCommandClass( TracerouteMplsStaticCmd )

#--------------------------------------------------------------------------------
# traceroute mpls nexthop-group name entry entry-index [options]
# XXXXX NO VRF
#--------------------------------------------------------------------------------
class TracerouteMplsNhgCmd( CliCommand.CliCommandClass ):
   syntax = 'traceroute mpls nexthop-group NHG entry ENTRY [ OPTIONS ]'
   data = { 'traceroute': NetworkToolsCli.tracerouteKwMatcher,
            'mpls' : nodeMpls,
            'nexthop-group' : matcherNhg,
            'NHG' : matcherNexthopGroup,
            'entry' : matcherEntry,
            'ENTRY' : matcherEntryIndex,
            'OPTIONS' : LspOptionExprFactory( LspOptionType.traceroute,
                                              multiLabel=True )
          }

   @staticmethod
   def handler( mode, args ):
      nhgName = args[ 'NHG' ]
      entry = args[ 'ENTRY' ]
      lspArgs = [ nhgName, '--type', 'nexthop-group', '--entry', str( entry ) ]
      lspArgs += buildLspTracerouteArgs( args[ 'OPTIONS' ] )
      print 'LSP traceroute to nexthop-group %s, entry %d' % ( nhgName, entry )
      doCallLspUtil( mode, LspTracerouteUtil, lspArgs )

BasicCli.EnableMode.addCommandClass( TracerouteMplsNhgCmd )

#--------------------------------------------------------------------------------
# traceroute mpls tunnel nexthop-group ENDPOINT [entry] [options]
#--------------------------------------------------------------------------------
class TracerouteMplsNhgTunnelCmd( CliCommand.CliCommandClass ):
   syntax = ( 'traceroute mpls tunnel nexthop-group ENDPOINT' 
              '[ entry ENTRY ] [ OPTIONS ]' )
   data = { 'traceroute': NetworkToolsCli.tracerouteKwMatcher,
            'mpls' : nodeMpls,
            'tunnel' : matcherTunnel,
            'nexthop-group' : matcherNhgForTunnel,
            'ENDPOINT': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
                           ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                           ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
            'entry' : matcherEntry,
            'ENTRY' : matcherEntryIndex,
            'OPTIONS' : LspOptionExprFactory( LspOptionType.traceroute,
                                              multiLabel=True,
                                              egressValidate=True )
          }

   @staticmethod
   def handler( mode, args ):
      lspArgs = [ str( args[ 'ENDPOINT' ] ), '--type', 'nexthop-group-tunnel' ]
      if 'entry' in args:
         lspArgs += [ '--entry', str( args[ 'ENTRY' ] ) ]
      lspArgs += buildLspTracerouteArgs( args[ 'OPTIONS' ] )
      print 'LSP traceroute to nexthop-group tunnel {}'.format(
            str( args[ 'ENDPOINT' ] ) )
      doCallLspUtil( mode, LspTracerouteUtil, lspArgs )

BasicCli.EnableMode.addCommandClass( TracerouteMplsNhgTunnelCmd )

#--------------------------------------------------------------------------------
# traceroute mpls srte endpoint ENDPOINT color COLOR [ traffic-af ( v4 | v6 ) ]
#                                                    [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteMplsSrTeCmd( CliCommand.CliCommandClass ):
   syntax = '''traceroute mpls srte endpoint ENDPOINT color COLOR
               [ traffic-af ( v4 | v6 ) ] [ OPTIONS ]'''
   data = { 'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
            'mpls' : nodeMpls,
            'srte' : matcherSrte,
            'endpoint' : matcherEndpoint,
            'ENDPOINT' : IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='Endpoint '
                                                            'IP/IPv6 address' ),
            'color' : matcherColor,
            'COLOR' : matcherColorVal,
            'traffic-af' : matcherTrafficAf,
            'v4' : matcherV4,
            'v6' : matcherV6,
            'OPTIONS' : LspOptionExprFactory( LspOptionType.traceroute,
                                              multiLabel=False,
                                              egressValidate=True ) }

   @staticmethod
   def handler( mode, args ):
      lspArgs = [ str( args[ 'ENDPOINT' ] ), '--type', 'SrTePolicy',
                  '--color', str( args[ 'COLOR' ] ) ]
      if 'traffic-af' in args:
         trafficAf = 'v4' if 'v4' in args else 'v6'
         lspArgs += [ '--trafficAf', trafficAf ]
      lspArgs += buildLspPingArgs( args[ 'OPTIONS' ] )
      print 'LSP traceroute to SR-TE Policy endpoint {} color {}'.format(
                                                         str( args[ 'ENDPOINT' ] ),
                                                         str( args[ 'COLOR' ] ) )
      doCallLspUtil( mode, LspTracerouteUtil, lspArgs )

BasicCli.EnableMode.addCommandClass( TracerouteMplsSrTeCmd )

#--------------------------------------------------------------------------------
# traceroute mpls rsvp session ( ( id SESSION_ID lsp LSP ) |
#                                ( name SESSION_NAME ) ) [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteRsvpCmd( CliCommand.CliCommandClass ):
   syntax = ( 'traceroute mpls rsvp session ( ( id SESSION_ID lsp LSP ) | '
                                             '( name SESSION_NAME ) ) [ OPTIONS ]' )
   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'rsvp' : matcherRsvp,
      'session' : matcherSession,
      'id' : matcherId,
      'SESSION_ID' : matcherIdVal,
      'lsp' : matcherLsp,
      'LSP' : matcherLspVal,
      'name' : matcherName,
      'SESSION_NAME' : matcherNameVal,
      'OPTIONS' : lspTracerouteOptionsExpr,
   }

   @staticmethod
   def handler( mode, args ):
      rsvpId = args.get( 'SESSION_ID' )
      lsp = args.get( 'LSP' )
      name = args.get( 'SESSION_NAME' )
      options = args.get( 'OPTIONS' )
      vrfName = None

      if rsvpId and lsp:
         args = [ '--type', 'rsvp', '--session-id', str( rsvpId ),
               '--lsp', str( lsp ) ]
         if options:
            args += buildLspTracerouteArgs( options )
         print 'LSP traceroute to RSVP session #{} LSP #{}'.format( rsvpId, lsp )
         doCallLspUtil( mode, LspTracerouteUtil, args, vrfName=vrfName )
      elif name:
         args = [ '--type', 'rsvp', '--session-name', str( name ) ]
         if options:
            args += buildLspPingArgs( options )
         print 'LSP traceroute to RSVP session {}'.format( name )
         doCallLspUtil( mode, LspTracerouteUtil, args, vrfName=vrfName )
      else:
         mode.addError( 'No session' )

BasicCli.EnableMode.addCommandClass( TracerouteRsvpCmd )

#--------------------------------------------------------------------------------
# traceroute mpls bgp labeled-unicast ( ip | ipv6 ) PREFIX
# [ bgp nexthop NEXTHOP [ label-stack { LABELS } ] ] [ OPTIONS ]
#--------------------------------------------------------------------------------
class TracerouteBgpLuCmd( CliCommand.CliCommandClass ):
   syntax = '''traceroute mpls bgp labeled-unicast
               ( ip PREFIX [ BGP_N nexthop NEXTHOP [ label-stack { LABELS } ] ] ) |
               ( ipv6 PREFIX6 [ BGP_N nexthop NEXTHOP6 [ label-stack { LABELS } ] ] )
               [ OPTIONS ]'''
   data = {
      'traceroute' : NetworkToolsCli.tracerouteKwMatcher,
      'mpls' : nodeMpls,
      'bgp' : matcherBgp,
      'labeled-unicast' : matcherBgpLu,
      'ip' : matcherIp,
      'PREFIX' : matcherAddrOrPrefix,
      'ipv6' : matcherIp6,
      'PREFIX6' : matcherAddrOrPrefix6,
      'BGP_N' : matcherBgpNexthop,
      'nexthop' : matcherNexthop,
      'NEXTHOP' : matcherNexthopIpAddr,
      'NEXTHOP6' : matcherNexthopIp6Addr,
      'label-stack' : MplsCli.labelStackKeywordNode,
      'LABELS' : MplsCli.labelStackValNode,
      'OPTIONS' : lspTracerouteOptionsExpr,
   }

   @staticmethod
   def handler( mode, args ):
      prefix = args.get( 'PREFIX' ) or args.get( 'PREFIX6' )
      if not prefix:
         mode.addError( 'Invalid prefix' )
         return
      prefix = ArnetIpGenPrefix( str( prefix ) ).stringValue
      nexthop = args.get( 'NEXTHOP' ) or args.get( 'NEXTHOP6' )
      labels = args.get( 'LABELS' )
      if labels and not MplsCli.validateLabelStackSize( mode, labels ):
         return
      options = args.get( 'OPTIONS' )
      lspArgs = [ prefix, '--type', LspPingTypeBgpLu ]
      if nexthop:
         if not isinstance( nexthop, str ):
            nexthop = nexthop.stringValue
         lspArgs += [ '--nexthop', nexthop ]
         if labels:
            lspArgs += [ '--label', ','.join( str( l ) for l in labels ) ]
      lspArgs += buildLspTracerouteArgs( options )
      print 'LSP traceroute to BGP labeled unicast tunnel', prefix
      doCallLspUtil( mode, LspTracerouteUtil, lspArgs )

if Toggles.ArBgpToggleLib.toggleBgpLuLspPingEnabled():
   BasicCli.EnableMode.addCommandClass( TracerouteBgpLuCmd )

# ------------------------------------------------------------------------------
# Mpls Utils configuration commands
# ------------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# [ no | default ] ip access-group ACL_NAME [ VRF ] [ in ]
#--------------------------------------------------------------------------------
class IpAccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL_NAME [ VRF ] [ in ]'
   noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] [ VRF ] [ in ]'
   data = {
      'ip' : AclCli.ipKwForServiceAclMatcher,
      'access-group' : AclCli.accessGroupKwMatcher,
      'ACL_NAME' : AclCli.ipAclNameMatcher,
      'VRF' : AclCli.vrfKwAndNameExprFactory,
      'in' : AclCli.inKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.doSetLspPingAclName( 'ip', args[ 'ACL_NAME' ], args.get( 'VRF' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noLspPingServiceAcl( mode, args.get( 'VRF' ) )

MplsUtilsConfigMode.addCommandClass( IpAccessGroupAclnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 access-group ACL_NAME [ VRF ] [ in ]
#--------------------------------------------------------------------------------
class Ipv6AccessGroupAclNameCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-group ACL_NAME [ VRF ] [ in ]'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] [ VRF ] [ in ]'
   data = {
      'ipv6' : AclCli.ipv6KwMatcherForServiceAcl,
      'access-group' : AclCli.accessGroupKwMatcher,
      'ACL_NAME' : AclCli.ip6AclNameMatcher,
      'VRF' : AclCli.vrfKwAndNameExprFactory,
      'in' : AclCli.inKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.doSetLspPingAclName( 'ipv6', args[ 'ACL_NAME' ], args.get( 'VRF' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noLspPingServiceAclV6( mode, args.get( 'VRF' ) )

MplsUtilsConfigMode.addCommandClass( Ipv6AccessGroupAclNameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mpls oam standard ( arista | ietf )
#--------------------------------------------------------------------------------
class MplsOamStandardCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls oam standard ( arista | ietf )'
   noOrDefaultSyntax = 'mpls oam standard [ arista | ietf ]'
   data = { 'mpls' : MplsCli.mplsNodeForConfig,
            'oam' : 'MPLS Operations, Administration, and Management configuration',
            'standard' : ( 'Set the default standard to comply with for '
                           'ping/traceroute commands' ),
            'arista' : ( 'Comply with the Arista standard for ping/traceroute, '
                         'which allows interworking with older EOS devices' ),
            'ietf' : 'Comply with the IETF standard'
          }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'arista' ):
         mplsUtilsConfig.oamStandard = MplsOamStandard.arista
      else:
         mplsUtilsConfig.oamStandard = MplsOamStandard.ietf

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if ( args.get( 'arista' ) and
           mplsUtilsConfig.oamStandard != MplsOamStandard.arista ):
         mode.addWarning( 'Given standard is not configured' )
      elif ( args.get( 'ietf' ) and
             mplsUtilsConfig.oamStandard != MplsOamStandard.ietf ):
         mode.addWarning( 'Given standard is not configured' )
      else:
         mplsUtilsConfig.oamStandard = MplsOamStandard.arista

BasicCli.GlobalConfigMode.addCommandClass( MplsOamStandardCmd )

# ------------------------------------------------------------------------------
# show mpls ping [ip|ipv6] access-list [ ACL_NAME ] [ summary ] [ dynamic] [ detail ]
# NB: ACL_NAME_AND_OPTS expands out to have extra options
# ------------------------------------------------------------------------------
class ShowMplsPingAcl( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls ping [ ip | ipv6 ] access-list [ ACL_NAME_AND_OPTS ]'
   data = {
      'mpls' : MplsCli.mplsForShowNode,
      'ping' : matcherPing,
      'ip' : AclCli.ipKwForShowServiceAcl,
      'ipv6' : AclCli.ipv6KwForShowServiceAcl,
      'access-list' : AclCli.accessListKwMatcherForServiceAcl,
      'ACL_NAME_AND_OPTS' : AclCli.ipOrIpv6AclNameExpression,

   }
   cliModel = AclCliModel.AllAclList

   @staticmethod
   def handler( mode, args ):
      aclType = args.get( 'ip', args.get( 'ipv6' ) )
      return AclCli.showServiceAcl( mode,
                                    aclCpConfig,
                                    aclStatus,
                                    aclCheckpoint,
                                    aclType,
                                    args[ '<aclNameExpr>' ],
                                    serviceName=LspPingService )

BasicCli.addShowCommandClass( ShowMplsPingAcl )

#----------------------------------------------------------------
# clear mpls ping counters [ ip | ipv6 ] access-list
#----------------------------------------------------------------
class ClearIpAclCounters( CliCommand.CliCommandClass ):
   syntax = 'clear mpls ping counters [ ip | ipv6 ] access-list'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'mpls' : MplsCli.mplsMatcherForClear,
      'ping' : matcherPing,
      'counters' : AclCli.countersKwMatcher,
      'ip' : AclCli.ipKwForClearServiceAclMatcher,
      'ipv6' : AclCli.ipv6KwMatcherForClearServiceAcl,
      'access-list' : AclCli.accessListKwMatcherForServiceAcl,
   }

   @staticmethod
   def handler( mode, args ):
      aclType = args.get( 'ip', args.get( 'ipv6' ) )
      return AclCli.clearServiceAclCounters( mode,
                                             aclStatus,
                                             aclCheckpoint,
                                             aclType )

BasicCli.EnableMode.addCommandClass( ClearIpAclCounters )

#--------------------------------------------------------------------------------
# [ no | default ] mpls icmp ( fragmentation-needed | ttl-exceeded ) tunneling
# MPLS ICMP configuration commands
#
# Commands should follow the format:
# [no|default] mpls icmp <type of ICMP msg> <type of handling>
#--------------------------------------------------------------------------------

# Enable tunneling of ICMP 'Fragmentation Needed', 'Packet Too Big' messages
# [no|default] mpls icmp fragmentation-needed tunneling
def fragNeededIcmpTunnelingGuard( mode, token ):
   if mplsHwCapability.mplsFragNeededIcmpTunnelingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

class MplsIcmpTunnelingCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls icmp ( fragmentation-needed | ttl-exceeded ) tunneling'
   noOrDefaultSyntax = syntax
   data = {
      'mpls' : MplsCli.mplsNodeForConfig,
      'icmp' : 'MPLS ICMP configuration',
      'fragmentation-needed' : CliCommand.guardedKeyword( 'fragmentation-needed',
         helpdesc='Fragmentation needed, or packet too big, for LSP',
         guard=fragNeededIcmpTunnelingGuard ),
      'ttl-exceeded' : 'Time-to-live exceeded in transit',
      'tunneling' : 'Enable MPLS tunneling',
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      if 'fragmentation-needed' in args:
         mplsUtilsConfig.fragNeededTunneling = not noOrDefault
      elif 'ttl-exceeded' in args:
         mplsUtilsConfig.icmpTtlTunneling = not noOrDefault
      else:
         assert False, 'Uknown ICMP type'

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( MplsIcmpTunnelingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mpls ping
#--------------------------------------------------------------------------------
class MplsPingCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ping'
   noOrDefaultSyntax = syntax
   data = {
      'mpls' : MplsCli.mplsNodeForConfig,
      'ping' : matcherPing,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( MplsUtilsConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      childMode = mode.childMode( MplsUtilsConfigMode )
      noLspPingServiceAcl( childMode )
      noLspPingServiceAclV6( childMode )

BasicCli.GlobalConfigMode.addCommandClass( MplsPingCmd )

def Plugin( entityManager ):
   global allNhgStatus
   global mplsUtilsConfig
   global aclConfig
   global aclCpConfig
   global aclStatus
   global aclCheckpoint
   global mplsHwCapability

   allNhgStatus = LazyMount.mount( entityManager, 
                                   'routing/nexthopgroup/status',
                                   'Routing::NexthopGroup::Status', 'r' )
   mplsUtilsConfig = ConfigMount.mount( entityManager, 'mplsutils/config',
                                        'MplsUtils::Config', 'w' )
   aclConfig = ConfigMount.mount( entityManager, 'acl/config/cli',
                                  'Acl::Input::Config', 'w' )
   aclCpConfig = ConfigMount.mount( entityManager, 'acl/cpconfig/cli',
                                  'Acl::Input::CpConfig', 'w' )
   aclStatus = LazyMount.mount( entityManager, 'acl/status/all',
                                'Acl::Status', 'r' )
   aclCheckpoint = LazyMount.mount( entityManager, 'acl/checkpoint',
                                   'Acl::CheckpointStatus', 'w' )
   mplsHwCapability = LazyMount.mount( entityManager,
                                       'routing/hardware/mpls/capability',
                                       'Mpls::Hardware::Capability',
                                       'r' )
