#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericPrefix
from CliModel import Bool, Int, List, Model
from TableOutput import createTable, Format
import TunnelModels

class MplsTunnelCountersEntry( Model ):
   tunnelIndex = Int( help="Index of the tunnel entry in the source tunnel table" )
   tunnelType = TunnelModels.TunnelType
   endpoint = IpGenericPrefix( help="Endpoint of the tunnel", optional=True )
   vias = List( valueType=TunnelModels.IpVia, help="List of nexthops" )
   txPackets = Int( help="Total number of transmitted packets", optional=True )
   txBytes = Int( help="Total number of transmitted bytes", optional=True )

   def renderMplsTunnelCountersTableRow( self, table ):
      printTunnelMetadata = True
      for via in self.vias:
         txPacketsStr = '-'
         txBytesStr = '-'
         tunnelIndexStr = '-'
         tunnelTypeStr = '-'
         tunnelEndpointStr = '-'
         if printTunnelMetadata:
            txPacketsStr = self.txPackets if self.txPackets is not None else '-'
            txBytesStr = self.txBytes if self.txBytes is not None else '-'
            tunnelIndexStr = self.tunnelIndex
            tunnelTypeStr = self.tunnelType
            if self.endpoint:
               tunnelEndpointStr = str( self.endpoint )
            printTunnelMetadata = False
         table.newRow( tunnelTypeStr, tunnelIndexStr, tunnelEndpointStr,
                       str( via.nexthop ), via.interface.stringValue,
                       txPacketsStr, txBytesStr )

   def renderMplsTunnelCountersListRecord( self ):
      print "  Tunnel type: %s, index: %d, endpoint: %s" % (
               self.tunnelType, self.tunnelIndex, self.endpoint or "-" )
      for via in self.vias:
         print "    Nexthop: %s, interface: %s" % (
                    via.nexthop, via.interface.stringValue )
      if self.txPackets is not None and self.txBytes is not None:
         print "  %d packets, %d bytes\n" % (
                  self.txPackets, self.txBytes )
      else:
         print "  Tunnel counters not available\n"

class MplsTunnelCounters( Model ):
   entries = List( valueType=MplsTunnelCountersEntry,
                   help="MPLS tunnel hardware counters" )
   _tableOutput = Bool( help="True if table output format was requested" )

   def renderTable( self ):
      headings = ( "Tunnel Type", "Index", "Endpoint",
                   "Nexthop", "Interface", "Packets", "Bytes" )
      fl = Format( justify='left' )
      table = createTable( headings, tableWidth=200 )
      table.formatColumns( fl, fl, fl, fl, fl, fl, fl )

      for mplsTunnelCountersEntry in self.entries:
         mplsTunnelCountersEntry.renderMplsTunnelCountersTableRow( table )

      print table.output()

   def renderList( self ):
      for mplsTunnelCountersEntry in self.entries:
         mplsTunnelCountersEntry.renderMplsTunnelCountersListRecord()

   def render( self ):
      if self._tableOutput:
         self.renderTable()
      else:
         self.renderList()

class MplsTunnelInterfaceCountersEntry( Model ):
   txPackets = Int( help="Total number of transmitted packets" )
   txBytes = Int( help="Total number of transmitted bytes" )

