# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from ArnetModel import IpGenericPrefix
from CliModel import List, Dict, Enum, Int, Bool, Float, Str
from CliModel import Model, Submodel, DeferredModel
from CliPlugin.IraIpModel import RouteSrTePolicy
from CliPlugin.TunnelCli import (
   getNhAndIntfAndLabelStrs,
   getViaModelFromViaDict,
)
from CliPlugin.TunnelFibModel import (
   getTunnelViaStatusStr,
   tunnelViaStatusCapiEnumVals,
)
from TableOutput import createTable, Format
import Tac, ArnetModel, IntfModel
import TunnelModels
import Toggles.MplsToggleLib

AddressFamily = Tac.Type( "Arnet::AddressFamily" )
LabelAction = Tac.Type( "Arnet::MplsLabelAction" )
LfibSource = Tac.Type( "Mpls::LFib::Source" )
LfibViaType = Tac.Type( "Mpls::LfibViaType" )
LfibViaSetType = Tac.Type( "Mpls::LfibViaSetType" )
PayloadType = Tac.Type( "Mpls::PayloadType" )
TtlMode = Tac.Type( "Mpls::TtlMode" )
IpGenPrefix = Tac.Type( "Arnet::IpGenPrefix" )
Ip6Type = Tac.Type( "Arnet::Ip6Prefix" )
PwType = Tac.Type( "Pseudowire::PseudowireType" )

LABEL_RANGE_STATIC = 'static'
LABEL_RANGE_DYNAMIC = 'dynamic'
LABEL_RANGE_UNASSIGNED = 'unassigned'

import datetime
import itertools

def printt( val, indent=0 ):
   print ' ' * 3 * indent + val

def renderLabel( label, conflict=False ):
   if label == 0 or label == 2:
      renderedLabel = 'exp-null'
   elif label == 1:
      renderedLabel = 'rtr-alert'
   elif label == 3:
      renderedLabel = 'imp-null'
   else:
      renderedLabel = str( label )
   if conflict:
      renderedLabel += ' **conflict**'
   return renderedLabel

def plural( collection ):
   return "s" if len( collection ) != 1 else ""
   
def uptimeStr( uptime ):
   return str( datetime.timedelta( seconds=int( Tac.utcNow() - uptime ) ) )

def warningHeader():
   warningHeaderContent = \
         "======================================================\n" \
         "WARNING: Some of the routes are not programmed in     \n" \
         "hardware, and they are marked with '*'.               \n" \
         "======================================================"
   return warningHeaderContent

class Dot1qTagToVlanKey( Model ):
   index = Int( help="Index of the Dot1Q to VLAN key" )
   source = Enum( values=LfibSource.attributes, help="Lfib source" )

class Dot1qTagToVlan( Model ):
   dot1qTagToVlanKey = Submodel( valueType=Dot1qTagToVlanKey,
                                 help="Dot1Q to VLAN map key" )
   dot1qTagToVlanId = Dict( keyType=int, valueType=int,
                            help="Mapping of Dot1Q tags to VLAN IDs" )

class MplsHwVia( Model ):
   recursiveNextHop = \
       ArnetModel.IpGenericAddress( help="Recursive nexthop IP address" )
   resolvedNextHop = \
       ArnetModel.IpGenericAddress( help="Resolved nexthop IP address" )
   l3Intf = IntfModel.Interface( help="Egress L3 interface" )
   labelAction = Enum( values=LabelAction.attributes, help="Label action" )
   outLabel = Int( help="Out label for the packet" )
   payloadType = Enum( values=PayloadType.attributes,
                      help="Type of payload being carried" )
   ttlMode = Enum( values=TtlMode.attributes, help="TTL mode" )
   dscpMode = Enum( values=TtlMode.attributes, help="DSCP mode", optional=True )
   vlanId = Int( help="VLAN ID" )
   l2Intf = IntfModel.Interface( help="Egress L2 interface" )
   macAddr = ArnetModel.MacAddress( help="Destination MAC address" )
   skipEgressAcl = Bool( optional=True,
      help="True if egress ACL is bypassed for the inner IP packets after MPLS pop" )
   # Only present when via is a tunnel.
   vias = List( valueType=TunnelModels.TunnelViaInfo,
                help="List of nested tunnel vias",
                optional=True )
   # Only present when there are backup vias for the tunnel
   backupVias = List( valueType=TunnelModels.TunnelViaInfo,
                      help="List of backup tunnel vias",
                      optional=True )
   tunnelDescriptor = Submodel( valueType=TunnelModels.TunnelInfo,
                                help="Tunnel information", optional=True )

class IpLookupVia( Model ):
   vrfName = Str( help="Name of VRF" )
   addressFamily = Enum( values=[ 'ipv4', 'ipv6', 'autoDecide' ],
                         help="Address family. 'autoDecide' indicates that the "
                              "lookup decision will be handled dynamically by the "
                              "platform based on the header following the label." )

class PwVia( Model ):
   pwType = Enum( values=PwType.attributes, help="Pseudowire type" )
   controlWordEnabled = Bool( help="Control word is enabled" )
   flowLabelEnabled = Bool( help="Flow label is enabled" )
   patch = Str( help="Patch associated with PW" )
   connector = Str( help="Local connector for PW" )

class VlanVia( Model ):
   vlanId = Int( help="VLAN ID" )
   controlWordEnabled = Bool( help="Control word is enabled" )

class EvpnVlanAwareVia( Model ):
   dot1qTagToVlan = Submodel( valueType=Dot1qTagToVlan, help="Dot1Q to VLAN map" )
   controlWordEnabled = Bool( help="Control word is enabled" )

class EvpnVlanFloodVia( Model ):
   vlanId = Int( help="VLAN ID" )
   controlWordEnabled = Bool( help="Control word is enabled" )

class EvpnVlanAwareFloodVia( Model ):
   dot1qTagToVlan = Submodel( valueType=Dot1qTagToVlan, help="Dot1Q to VLAN map" )
   controlWordEnabled = Bool( help="Control word is enabled" )

class EvpnEthernetSegmentFilterVia( Model ):
   intf = IntfModel.Interface( help="Interface mapping to ESI label" )

class MplsHwAdjacency( Model ):
   viaType = Enum( values=LfibViaType.attributes, help="Type of the adjacency" )
   mplsVias = List( valueType=MplsHwVia, optional=True, help="List of MPLS vias" )
   backupMplsVias = List( valueType=MplsHwVia, optional=True,
                          help="List of backup MPLS vias" )
   ipLookupVias = List( valueType=IpLookupVia, optional=True,
                        help="List of MPLS vias" )
   vlanVias = List( valueType=VlanVia, optional=True, help="List of VLAN vias" )
   evpnVlanAwareVias = List( valueType=EvpnVlanAwareVia, optional=True,
                             help="List of EVPN VLAN Aware vias" )
   evpnVlanFloodVias = List( valueType=EvpnVlanFloodVia, optional=True,
                             help="List of EVPN VLAN Flood vias" )
   evpnVlanAwareFloodVias = List( valueType=EvpnVlanAwareFloodVia, optional=True,
                                  help="List of EVPN VLAN Aware Flood vias" )
   evpnEthernetSegmentFilterVias = List( valueType=EvpnEthernetSegmentFilterVia,
                                         optional=True,
                                         help="List of EVPN Ethernet Segment Filter "
                                              "vias" )

class MplsHwRoute( Model ):
   label = Int( help="Incoming MPLS label, set to 1048576 (N/A) if labels "
                     "attribute is present" )
   labels = Str( help="Incoming MPLS labels for multi-label routes, "
                      "comma-separated, ordered top-most to bottom-most",
                 optional=True )
   adjacencies = Dict( keyType=int, valueType=MplsHwAdjacency,
                       help="Dictionary of adjacencies keyed by metric" )
   bestMetric = Int( help="Metric for the best available path" )
   unprogrammed = Bool( help="Route is unprogrammed in hardware" )
   adjacencySetUnprogrammed = Bool( optional=True,
                                    help="Adjacency set is programmed in hardware" )
   viaSetType = Enum( optional=True,
                      values=LfibViaSetType.attributes, help="Via set type" )

class MplsHwStatus( DeferredModel ):
   __revision__ = 2
   routes = Dict( keyType=int, valueType=MplsHwRoute,
                  help="Dictionary of programmed routes, keyed by their incoming "
                       "MPLS label" )
   multiLabelRoutes = Dict( keyType=str, valueType=MplsHwRoute,
                            help="Dictionary of programmed multi-label routes, "
                                 "keyed by their incoming MPLS labels, "
                                 "comma-separated, ordered top-most to bottom-most",
                            optional=True )
   mplsSupported = Bool( help="True if MPLS routing is supported" )
   allowDefaultRoute = Bool(
              help="True if MPLS next-hop is allowed to resolve over default route",
              optional=True )

class MplsHwStatusSummary( Model ):
   numLabels = Int( help='Number of labels' )
   numUnprogrammedLabels = Int( help='Number of unprogrammed labels' )
   numHwAdjs = Int( help='Number of adjacencies (via groups) used in hardware' )
   numBackupAdjs = Int( help='Number of backup adjacencies (via groups)' )

   def render( self ):
      n = self.numUnprogrammedLabels
      if n:
         unprogrammed = '(%d unprogrammed)' % n
      else:
         unprogrammed = ''
      print 'Number of Labels: %d %s' % ( self.numLabels, unprogrammed )
      print 'Number of adjacencies in hardware: %d' % self.numHwAdjs
      print 'Number of backup adjacencies: %d' % self.numBackupAdjs

class MplsVia( Model ):
   nexthop = ArnetModel.IpGenericAddress( help="Next hop IP address" )
   intf = IntfModel.Interface( help="Egress L3 interface of next hop" )
   outLabel = Int( help="Out label for the packet" )
   labelAction = Enum( values=LabelAction.attributes, help="Label action" )
   payloadType = Enum( values=( "ipv4", "ipv6", "mpls", "autoDecide",
                                "undefinedPayload" ),
                       help="Type of payload being carried" )
   skipEgressAcl = Bool( optional=True, help="Bypass egress ACL for the inner IP "
                         "packets after MPLS pop" )
   ttlMode = Enum( values=TtlMode.attributes, help="TTL mode" )
   dscpMode = Enum( values=TtlMode.attributes, help="DSCP mode", optional=True )
   status = Enum( values=( "up", "down" ), optional=True,
                           help="Resolution status of entry" )
   # Only present when via is a tunnel.
   vias = List( valueType=TunnelModels.TunnelViaInfo,
                help="List of nested tunnel vias",
                optional=True )
   # Only present when there are backup vias for the tunnel
   backupVias = List( valueType=TunnelModels.TunnelViaInfo,
                      help="List of backup tunnel vias",
                      optional=True )
   tunnelDescriptor = Submodel( valueType=TunnelModels.TunnelInfo,
                                help="Tunnel information", optional=True )

   def degrade( self, mplsVia, revision ):
      if revision < 2:
         if mplsVia[ 'payloadType' ] == 'autoDecide':
            mplsVia[ 'payloadType' ] = 'ipv4'
      return mplsVia

# TODO: move this to the gribi package when the CliPlugins are created (BUG389370)
class RouteGribi( Model ):
   nhgId = Int( help="gRIBI Nexthop Group ID" )

lfibSources = [ 'staticMpls', 'ldp', 'isisSrV4Adj', 'isisSrV6Adj',
                'isisSrV4Prefix', 'isisSrV6Prefix', 'ldpToIsisSr',
                'isisSrToLdp', 'bgpL2Evpn', 'bgpL3Vpn', 'pseudowire',
                'bgpLu', 'srtepolicy', 'rsvp', 'mldp', 'debug',
                'staticMcast', 'gribi' ]

if Toggles.MplsToggleLib.toggleOspfSegmentRoutingMplsEnabled():
   lfibSources.extend( [ 'ospfSrV4Adj', 'ospfSrV4Prefix',
                         'ldpToOspfSr', 'ospfSrToLdp' ] )

if Toggles.MplsToggleLib.toggleIsisAreaProxyEnabled():
   lfibSources.extend( [ 'isisSrV4Area', 'isisSrV6Area' ] )

class MplsAdjacency( Model ):
   viaType = Enum( values=LfibViaType.attributes, help="Type of the adjacency" )
   source = Enum( values=lfibSources,
                  optional=True,
                  help="LFIB source" )
   fecForRoute = Str( help="The FEC for the route", optional=True )
   srTePolicy = Submodel( valueType=RouteSrTePolicy, help="SR-TE policy",
                          optional=True )
   gribi = Submodel( valueType=RouteGribi, help="gRIBI LFIB route info",
                          optional=True )
   fecId = Int( help='FEC ID', optional=True )
   mplsVias = List( valueType=MplsVia, optional=True, help="List of MPLS vias" )
   backupMplsVias = List( valueType=MplsVia, optional=True,
                          help="List of backup MPLS vias" )
   ipLookupVias = List( valueType=IpLookupVia, optional=True,
                        help="List of IP lookup vias" )
   vlanVias = List( valueType=VlanVia, optional=True, help="List of VLAN vias" )
   evpnVlanAwareVias = List( valueType=EvpnVlanAwareVia, optional=True,
                             help="List of EVPN VLAN Aware vias" )
   evpnVlanFloodVias = List( valueType=EvpnVlanFloodVia, optional=True,
                             help="List of EVPN VLAN Flood vias" )
   evpnVlanAwareFloodVias = List( valueType=EvpnVlanAwareFloodVia, optional=True,
                                  help="List of EVPN VLAN Aware Flood vias" )
   evpnEthernetSegmentFilterVias = List( valueType=EvpnEthernetSegmentFilterVia,
                                         optional=True,
                                         help="List of EVPN Ethernet Segment Filter "
                                              "vias" )
   pseudowireVias = List( valueType=PwVia, optional=True, help="List of PW vias" )

class MplsRoute( Model ):
   topLabel = Int( help="Incoming MPLS label, set to 1048576 (N/A) if topLabels "
                        "attribute is present" )
   topLabels = Str( help="Incoming MPLS labels for multi-label routes, "
                         "comma-separated, ordered top-most to bottom-most",
                    optional=True )
   adjacencies = Dict( keyType=int, valueType=MplsAdjacency,
                       help="Dictionary of adjacencies keyed by metric" )

class MplsRoutes( Model ):
   __revision__ = 3
   routes = Dict( keyType=int,
                  valueType=MplsRoute, help="Dictionary of MPLS routes configured, "
                                            "keyed by their top label" )
   multiLabelRoutes = Dict( keyType=str, valueType=MplsRoute,
                            help="Dictionary of multi-label MPLS routes configured, "
                                 "keyed by their top labels, comma-separated, "
                                 "ordered top-most to bottom-most",
                            optional=True )

   def degrade( self, mplsRoutes, revision ):
      if revision < 3:
         mplsRoutesDegrade = []
         for r in mplsRoutes[ 'routes' ].values():
            mplsRouteDegrade = dict( topLabel=r[ 'topLabel' ] )
            for metric, adj in r[ 'adjacencies' ].iteritems():
               mplsRouteDegrade[ 'metric' ] = int( metric )
               mplsRouteDegrade[ 'via' ] = adj.get( 'mplsVias', [] )
               if 'source' in adj:
                  mplsRouteDegrade[ 'source' ] = adj[ 'source' ]
               if 'fecForRoute' in adj:
                  mplsRouteDegrade[ 'fecForRoute' ] = adj[ 'fecForRoute' ]
               mplsRoutesDegrade.append( mplsRouteDegrade )
         mplsRoutes[ 'routes' ] = mplsRoutesDegrade
      return mplsRoutes

   def render( self ):
      # MplsRoutes for 'show mpls config route' don't specify a source or FEC,
      # so we don't show those columns if all of the entries have neither.
      adjs = itertools.chain.from_iterable( r.adjacencies.values()
                                            for r in self.routes.values() )
      displaySource = any( adj.source is not None or adj.fecForRoute is not None 
                           for adj in adjs if adj.mplsVias )

      lfibSourceMap = {
         'staticMpls': 'S',
         'isisSrV4Prefix': 'IP',
         'isisSrV4Adj': 'IA',
         'isisSrV6Prefix': 'IP',
         'isisSrV6Adj': 'IA',
         'ldp': 'L',
         'isisSrToLdp': 'IL',
         'ldpToIsisSr': 'LI',
         'bgpL2Evpn' : 'B2',
         'bgpL3Vpn' : 'B3',
         'pseudowire' : 'P',
         'bgpLu' : 'BL',
         'srtepolicy' : 'ST',
         'rsvp' : 'R',
         'mldp' : 'M',
      }

      headings = \
            ( "In-Label", "Out-Label", "Metric", "Payload",
              "NextHop", "Egress-ACL", "Status" )
      if displaySource:
         headings += ( "Source", "FEC" )

      fmt = Format( justify='left' )
      fmt.padLimitIs( True )
      t = createTable( headings, tableWidth=140 )
      t.formatColumns( *( [ fmt ] * len( headings ) ) )

      def _fillTable( routeDictLabelKey, route ):
         for metric, adj in sorted( route.adjacencies.iteritems() ):
            for via in adj.mplsVias:
               t.startRow()

               nexthop = via.nexthop
               intf = via.intf
               outLabel = str( via.outLabel ) \
                   if via.labelAction == 'swap' else via.labelAction
               payloadType = via.payloadType if via.labelAction == 'pop' else 'mpls'
               skipEgressAcl = 'bypass' if via.skipEgressAcl else 'apply'
               status = via.status

               nexthopAndIntf = nexthop.stringValue
               # bool method on the Interface model always returns True
               if intf.stringValue != '':
                  nexthopAndIntf += ',%s' % intf.shortName

               # Format must be specified for each cell to properly apply align left
               t.newFormattedCell( routeDictLabelKey, format=fmt )
               t.newFormattedCell( outLabel, format=fmt )
               t.newFormattedCell( metric, format=fmt )
               t.newFormattedCell( payloadType, format=fmt )
               t.newFormattedCell( nexthopAndIntf, format=fmt )
               t.newFormattedCell( skipEgressAcl, format=fmt )
               t.newFormattedCell( status, format=fmt )
               if displaySource:
                  source = lfibSourceMap[ adj.source ] if adj.source else ''
                  fec = adj.fecForRoute if adj.fecForRoute else ''

                  t.newFormattedCell( source, format=fmt )
                  t.newFormattedCell( fec, format=fmt )

      for topLabel, route in sorted( self.routes.iteritems() ):
         _fillTable( topLabel, route )
      if self.multiLabelRoutes:
         for topLabels, route in sorted( self.multiLabelRoutes.iteritems() ):
            _fillTable( topLabels, route )

      print '''Codes: %s - Static MPLS Route, %s - IS-IS SR Adjacency Segment,
       %s - IS-IS SR Prefix Segment, %s - LDP,
       %s - IS-IS SR Segment to LDP, %s - LDP to IS-IS SR Segment, %s - RSVP''' % \
            tuple( lfibSourceMap[ k ] for k in [ 'staticMpls', 'isisSrV4Adj',
                                                 'isisSrV4Prefix', 'ldp',
                                                 'isisSrToLdp', 'ldpToIsisSr',
                                                 'rsvp' ] )
      print ''

      print t.output()
      return

class MplsPeerRouterId( Model ):
   protocol = Enum( values=[ 'unknown', 'mplste', 'ldp', 'isis', 'ospf', 'rsvp' ],
                    help='Protocol in which router has this ID' )
   routerId = Str( help='Router ID' )
   hostname = Str( help='Hostname', optional=True )

class MplsPeerIdAndLabel( Model ):
   peer = Submodel( MplsPeerRouterId, help="Peer ID")
   labels = List( valueType=int, help="Peer labels" )
   elc = Bool( help='Entropy label capability', optional=True )

   ## The following attributes are included iff 'detail' variant of show command
   bindingUpdateTimestamp = Float( 
                               help='UTC timestamp of when the labels were assigned',
                              optional=True )
   # In non-detail mode: filtered FEC is excluded from model and filtered is None
   # In detail mode: all FECs included in model and filtered is always set
   filtered = Bool( help='FEC is filtered', optional=True )

class MplsLocalLabels( Model ):
   labels = List( valueType=int, help="Local labels" )
   elc = Bool( help='Entropy label capability', optional=True )

   ## The following attributes are included iff 'detail' variant of show command
   bindingUpdateTimestamp = Float( 
                               help='UTC timestamp of when the labels were assigned',
                               optional=True )
   # In non-detail mode: filtered FEC is excluded from model and filtered is None
   # In detail mode: all FECs included in model and filtered is always set
   filtered = Bool( help='FEC is filtered', optional=True )

class MplsLabelBinding( Model ):
   local = Submodel( valueType=MplsLocalLabels, 
                     help='Locally defined label bindings', 
                     optional=True )
   remote = List( valueType=MplsPeerIdAndLabel, 
                  help='Remote label bindings',
                  optional=True )

afMap = { 'ipv4':0, 'ipv6':1 }
def sortIpGenPrefix( x ):
   # The sortKey implementation of IpGenPrefixes generates a 64 bit integer
   # to compare that masks the 32 bit word in a IPv4 address with all 1's in 
   # the top 32-bits and the address value in the lower 32-bits.
   # The IPv6 sortKey is simple a OR of the first 32-bits and
   # the last 32-bits. Thus a IPv6 prefix is considered lesser than an 
   # IPv4 prefix. This routine is to bypass sortKey and end up with IPv4
   # prefixes first and then IPV6 prefixes
   # Argument to sortIpGenPrefix is a tuple: ( fec, binding )
   fec = x[ 0 ]
   pfx = IpGenPrefix( fec )
   return ( afMap[ pfx.af ] << 64 | pfx.sortKey )

class MplsBindingsModel( Model ):
   # pylint: disable-msg=E1101
   bindings = Dict( keyType=str,
                    valueType=MplsLabelBinding,
                    help=' Label bindings' )
   _summary = Bool( help='Summary of bindings' )

def viaKey( via ):
   return ( via.nexthop.stringValue,
            Arnet.intfNameKey( via.interface.shortName ),
            via.labels )

class StaticTunnelTableEntry( TunnelModels.TunnelTableEntry ):
   vias = List( valueType=TunnelModels.MplsVia, help="List of nexthops" )
   name = Str( help="Static tunnel name" )

   def renderStaticTunnelTableEntry( self, table, tunnelIndex ):
      nhStr = intfStr = labelsStr = '-'
      vias = sorted( self.vias, key=viaKey )
      if vias:
         firstVia = vias[ 0 ]
         nhStr, intfStr, labelsStr = getNhAndIntfAndLabelStrs( firstVia )
      table.newRow( tunnelIndex, self.name, str( self.endpoint ), nhStr, intfStr,
                    labelsStr )
      for via in vias[ 1 : ]:
         nhStr, intfStr, labelsStr = getNhAndIntfAndLabelStrs( via )
         table.newRow( '-', '-', '-', nhStr, intfStr, labelsStr )

class StaticTunnelTable( Model ):
   __revision__ = 3
   entries = Dict( keyType=long, valueType=StaticTunnelTableEntry,
                   help="Static tunnel table entries keyed by tunnel index" )

   def render( self ):
      headings = ( "Index", "Name", "Endpoint", "Nexthop", "Interface",
                   "Labels" )
      fl = Format( justify='left' )
      infinity = int(2**31 - 1)
      table = createTable( headings, tableWidth=infinity )
      table.formatColumns( fl, fl, fl, fl, fl, fl )
      for tunnelIndex, staticTunnelTableEntry in sorted( self.entries.iteritems() ):
         staticTunnelTableEntry.renderStaticTunnelTableEntry( table, tunnelIndex )

      print table.output()

   # "show mpls tunnel static | json revision 1" will invoke this degrade 
   # function which will delete the via[ 'type' ] from the Dict.
   # There is no compatibility issue between revision 1 and 2. But we 
   # added an extra type field to rev 2 in Via(Model) and removing it 
   # for consistency reasons.
   # "show mpls tunnel static | json revision 2" will have 'labels' removed from
   # every via, and the 'labels' value from the first via added to the entry.
   def degrade( self, dictRepr, revision ):
      if revision == 1:
         for entry in dictRepr[ 'entries' ].itervalues():
            for via in entry[ 'vias' ]:
               if via: 
                  getViaModelFromViaDict( via ).degradeToV1( via )
      if revision < 3:
         for entry in dictRepr[ 'entries' ].itervalues():
            labels = []
            vias = entry.get( 'vias' )
            if vias:
               labels = vias[ 0 ][ 'labels' ]
            for via in vias:
               del via[ 'labels' ]
            entry[ 'labels' ] = labels
      return dictRepr

class MplsTunnelFibEntry( Model ):
   tunnelIndex = Int( help="Index of the tunnel entry in the source tunnel table" )
   tunnelType = TunnelModels.TunnelType
   endpoint = IpGenericPrefix( help="Endpoint of the tunnel", optional=True )
   vias = List( valueType=TunnelModels.MplsVia, help="List of nexthops" ) 
   tunnelViaStatus = Enum( values=tunnelViaStatusCapiEnumVals,
                           help="Tunnel programming status", optional=True )

   def renderMplsTunnelFibEntry( self, table ):
      printTunnelMetadata = True
      for via in self.vias:
         tunnelIndexStr = '-'
         tunnelTypeStr = '-'
         tunnelEndpointStr = '-'
         tunnelViaStatusStr = '-'
         if printTunnelMetadata:
            tunnelIndexStr = self.tunnelIndex
            tunnelTypeStr = self.tunnelType
            if self.endpoint:
               tunnelEndpointStr = str( self.endpoint )
            tunnelViaStatusStr = getTunnelViaStatusStr( self.tunnelViaStatus )
            printTunnelMetadata = False
         labelsStr = '[ ' + ' '.join( via.labels ) + ' ]'
         table.newRow( tunnelTypeStr, tunnelIndexStr, tunnelEndpointStr,
                       str( via.nexthop ), via.interface.stringValue, labelsStr,
                       tunnelViaStatusStr )

class MplsTunnelFib( Model ):
   __revision__ = 3
   entries = List( valueType=MplsTunnelFibEntry,
                   help="List of mpls tunnel FIB entries" )
   def render( self ):
      headings = ( "Tunnel Type", "Index", "Endpoint", "Nexthop", "Interface",
                   "Labels", "Forwarding" )
      fl = Format( justify='left' )
      table = createTable( headings, tableWidth=200 )
      table.formatColumns( fl, fl, fl, fl, fl, fl, fl )
      for mplsTunnelFibEntry in self.entries:
         mplsTunnelFibEntry.renderMplsTunnelFibEntry( table )
        
      print table.output()

   # "show mpls tunnel fib | json revision 1" will invoke this degrade 
   # function which will delete the via[ 'type' ] from the Dict.
   # There is no compatibility issue between revision 1 and 2. But we 
   # added an extra type field in rev 2 in Via(Model) and removing it 
   # for consistency reasons
   # Up to revision 2, the endpoint attribute was mandatory for MplsTunnelFibEntry
   # model, it has become optional in revision 3, therefore we should send an N/A
   # address for revisions below 3.
   def degrade( self, dictRepr, revision ):
      if revision == 1:
         for entry in dictRepr[ 'entries' ]:
            for via in entry[ 'vias' ]:
               if via: 
                  getViaModelFromViaDict( via ).degradeToV1( via )
      if revision < 3:
         for entry in dictRepr[ 'entries' ]:
            if entry.get( 'endpoint' ) is None:
               entry[ 'endpoint' ] = Ip6Type.ip6PrefixUnspecified.stringValue
      return dictRepr

class MplsLabelRange( Model ):
   blockStart = Int( help='The first MPLS label in the range' )
   blockSize = Int( help='The number of labels in the range' )
   protocol = Str( optional=True, help='The protocol which uses the range, '
         'or absent if the range is unused'  )
   rangeType = Enum( values=( LABEL_RANGE_STATIC,
                              LABEL_RANGE_DYNAMIC,
                              LABEL_RANGE_UNASSIGNED ),
                     help='How the range was added' )

   def blockEnd( self ):
      return self.blockStart + self.blockSize - 1

class MplsLabelRanges( Model ):
   ranges = List( valueType=MplsLabelRange,
                  help='Set of operationally active label ranges' )
   conflictingRanges = List( valueType=MplsLabelRange,
                             help='Set of operationally inactive label ranges '
                             'due to conflict with some other range' )

   def render( self ):
      lineFmt = "%-9s %-9s %-9s %-18s"
      header = lineFmt % ( "Start", "End", "Size", "Usage" )
      print header
      print '-' * len( header )

      # Assuming that the ranges are already sorted by blockStart
      allRanges = []
      for r in self.ranges:
         usage = r.protocol
         if r.rangeType == LABEL_RANGE_DYNAMIC:
            usage = '%s (dynamic)' % ( usage if usage else 'free' )
         elif r.rangeType == LABEL_RANGE_UNASSIGNED:
            usage = r.rangeType
         elif r.protocol == LABEL_RANGE_STATIC:
            usage = 'static mpls'

         allRanges.append( {
            'start': r.blockStart,
            'size': r.blockSize,
            'usage': usage } )

      for rangeVal in allRanges:
         print lineFmt % ( rangeVal[ 'start' ],
               rangeVal[ 'start' ] + rangeVal[ 'size' ] - 1,
               rangeVal[ 'size' ],
               rangeVal[ 'usage' ] )

class MplsNexthopResolutionConfig( Model ):
   nexthopResolutionAllowDefaultRoute = Bool(
                    help="Show MPLS nexthop resolution configuration" )

   def render( self ):
      print 'Allow default route: %s' % self.nexthopResolutionAllowDefaultRoute

class MplsFecRequestTableEntry( Model ):
   requestedFecId = Int( help="Requested FEC ID for MPLS route" )
   installedFecId = Int( help="Installed FEC ID for MPLS route" )
   programStatus = Str( help="Programming status of the requested FEC ID" )

class MplsFecRequestTable( Model ):
   routes = Dict( keyType=str, valueType=MplsFecRequestTableEntry,
                  help="MPLS FEC ID keyed by labels" )

   def render( self ):
      headings = ( "Labels", "Requested FEC ID",
                   "Installed FEC ID", "Programming Status" )
      fl = Format( justify='left' )
      infinity = int( 2**31 - 1 )
      table = createTable( headings, tableWidth=infinity )
      table.formatColumns( fl, fl, fl, fl )
      for labels, entry in sorted( self.routes.iteritems() ):
         table.newRow( str( labels ),
                       str( entry.requestedFecId ),
                       str( entry.installedFecId ),
                       entry.programStatus )

      print table.output()
