# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import sys

import Arnet
import BasicCli
import CliCommand
import CliMatcher
from CliModel import cliPrinted
from CliPlugin import MplsModel
from CliPlugin.IntfCli import Intf
import CliPlugin.Ip6AddrMatcher
import CliPlugin.IpAddrMatcher
import CliPlugin.IpGenAddrMatcher
from CliPlugin.MplsCli import (
      copyStaticTunnelConfigEntry,
      getMplsTunnelNhTepAndIntfId,
      handleMplsLabelConfig,
      handleNoMplsLabelConfig,
      impNullTunnelNode,
      intfValMatcher,
      labelOperation,
      labelsAdapter,
      labelStackKeywordNode,
      labelStackValNode,
      metricMatcher,
      metricValMatcher,
      mplsForShowNode,
      mplsNodeForConfig,
      mplsSupported,
      outLabelValMatcher,
      payloadTypeMatcher,
      payloadTypeNoBypassMatcher,
      popMatcher,
      showMplsConfigWarnings,
      staticMatcher,
      staticTunnelMatcher,
      swapLabelMatcher,
      topLabelMatcher,
      topLabelValMatcher,
      tunnelNameValMatcher,
      tunnelNode,
      validateLabelStackSize
)
from CliPlugin.TunnelCli import (
      getTunnelIdFromIndex,
      tunnelIndexMatcher,
)
from CliPlugin.BridgingCli import (
      MacAddrTableExprForConfig,
      staticKwMatcher as macStaticKwMatcher,
      matcherVlan as matcherVlanKw,
      MacAddr,
      setStaticHelper,
)
from CliPlugin.VlanCli import vlanIdMatcher
from CliToken.Bridging import nodeMacAddressTableDebugHidden
import ConfigMount
import FibUtils
import LazyMount
from MplsLib import tunTypeEnumDict
from MplsTypeLib import (
   tunnelTypes,
   tunnelTypeXlate,
)
import ShowCommand
import SharedMem
import Smash
import Tac
from TypeFuture import TacLazyType

AddressFamily = TacLazyType( 'Arnet::AddressFamily' )
MplsLabel = Tac.Type( 'Arnet::MplsLabel' )
MplsVia = TacLazyType( 'Tunnel::TunnelTable::MplsVia' )
PayloadType = Tac.Type( 'Mpls::PayloadType' )
IntfId = Tac.Type( 'Arnet::IntfId' )
VlanId = Tac.Type( 'Bridging::VlanId' )
LfibFlag = Tac.Type( 'Mpls::LfibFlag' )
LfibSysdbIpLookupVia = Tac.Type( 'Mpls::LfibSysdbIpLookupVia' )
LfibSysdbVlanVia = Tac.Type( 'Mpls::LfibSysdbVlanVia' )
LfibSysdbVlanFloodVia = Tac.Type( 'Mpls::LfibSysdbVlanFloodVia' )
LfibSysdbESFilterVia = Tac.Type( 'Mpls::LfibSysdbEthernetSegmentFilterVia' )
LfibSysdbExtFecVia = Tac.Type( 'Mpls::LfibSysdbExtFecVia' )
FecId = Tac.Type( 'Smash::Fib::FecId' )
FecType = Tac.Type( "Smash::Fib::AdjType" )
StaticTunnelConfigEntry = TacLazyType( 'Tunnel::Static::StaticTunnelConfigEntry' )

bgpLuLfibInput = None
debugLfib = None
lfibInfo = None
routing6Status = None
routingStatus = None
staticTunnelConfig = None
tunnelFib = None

debugNodeHidden = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'debug', helpdesc='' ),
      hidden=True )
labelValMatcher = CliMatcher.IntegerMatcher(
      MplsLabel.unassignedMin, MplsLabel.max,
      helpdesc="MPLS label" )
vlanValMatcher = CliMatcher.IntegerMatcher(
      VlanId.min, VlanId.max,
      helpdesc="Identifier for a Virtual LAN" )
trueMatcher = CliMatcher.KeywordMatcher( 'true',
      helpdesc='Enable control-word' )
falseMatcher = CliMatcher.KeywordMatcher( 'false',
      helpdesc='Disable control-word' )
controlWordPresentMatcher = CliMatcher.KeywordMatcher( 'control-word-present',
      helpdesc='Enable/disable control-word' )
tunnelTypeValMatcher = CliMatcher.EnumMatcher( tunnelTypes )

#-----------------------------------------------------------
# [ no | default ] mpls debug ip-lookup-via LABEL VRF ( ipv4 | ipv6 )
#-----------------------------------------------------------
class MplsDebugIpLookupVia( CliCommand.CliCommandClass ):
   syntax = 'mpls debug ip-lookup-via LABEL VRF ( ipv4 | ipv6 )'
   noOrDefaultSyntax = 'mpls debug ip-lookup-via LABEL ...'
   data = {
      'mpls': mplsNodeForConfig,
      'debug': debugNodeHidden,
      'ip-lookup-via': 'IpLookupVia related (label, VRF, and address family)',
      'LABEL': labelValMatcher,
      'VRF': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                                        helpname='WORD',
                                        helpdesc='Name of VRF' ),
      'ipv4': 'IPv4 related',
      'ipv6': 'IPv6 related',
   }

   @staticmethod
   def handler( mode, args ):
      label = args[ 'LABEL' ]
      vrfName = args[ 'VRF' ]
      payloadType = PayloadType.ipv4 if 'ipv4' in args else PayloadType.ipv6

      # Instantiate a via which contains info for vrf-name and addrFamily
      ipLookupVia = LfibSysdbIpLookupVia( label, vrfName, payloadType )

      # Actually add it into Sysdb.
      debugLfib.ipLookupVia.addMember( ipLookupVia )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del debugLfib.ipLookupVia[ args[ 'LABEL' ] ]

#-----------------------------------------------------------
# [ no | default ] mpls debug vlan-via LABEL VLAN
#                  control-word-present ( true | false )
#-----------------------------------------------------------
class MplsDebugVlanVia( CliCommand.CliCommandClass ):
   syntax = 'mpls debug vlan-via LABEL VLAN control-word-present ( true | false )'
   noOrDefaultSyntax = 'mpls debug vlan-via LABEL ...'
   data = {
      'mpls': mplsNodeForConfig,
      'debug': debugNodeHidden,
      'vlan-via': 'VlanVia related (label, vlanId, and control word)',
      'LABEL': labelValMatcher,
      'VLAN': vlanValMatcher,
      'control-word-present': controlWordPresentMatcher,
      'true': trueMatcher,
      'false': falseMatcher
   }

   @staticmethod
   def handler( mode, args ):
      label = args[ 'LABEL' ]
      vlanId = args[ 'VLAN' ]
      controlWordPresent = LfibFlag( int( 'true' in args ) )

      # Instantiate a via which contains info for vlanId and control-word
      vlanVia = LfibSysdbVlanVia( label, vlanId, controlWordPresent )

      # Actually add it into Sysdb
      debugLfib.vlanVia.addMember( vlanVia )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del debugLfib.vlanVia[ args[ 'LABEL' ] ]

#-----------------------------------------------------------
# [ no | default ] mpls debug vlan-flood-via LABEL VLAN
#                  control-word-present ( true | false )
#-----------------------------------------------------------
class MplsDebugVlanFloodVia( CliCommand.CliCommandClass ):
   syntax = 'mpls debug vlan-flood-via LABEL VLAN control-word-present (true|false)'
   noOrDefaultSyntax = 'mpls debug vlan-flood-via LABEL ...'

   data = {
      'mpls': mplsNodeForConfig,
      'debug': debugNodeHidden,
      'vlan-flood-via': 'VlanFloodVia related (label, vlanId, and control word)',
      'LABEL': labelValMatcher,
      'VLAN': vlanValMatcher,
      'control-word-present': controlWordPresentMatcher,
      'true': trueMatcher,
      'false': falseMatcher
   }

   @staticmethod
   def handler( mode, args ):
      label = args[ 'LABEL' ]
      vlanId = args[ 'VLAN' ]
      controlWordPresent = LfibFlag( int( 'true' in args ) )

      # Instantiate a via which contains info for vlanId and control-word
      vlanFloodVia = LfibSysdbVlanFloodVia( label, vlanId, controlWordPresent )

      # Actually add it into Sysdb
      debugLfib.vlanFloodVia.addMember( vlanFloodVia )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del debugLfib.vlanFloodVia[ args[ 'LABEL' ] ]

# -------------------------------------------------------------
# [ no | default ] mpls debug es-filter-via LABEL INTF
# -------------------------------------------------------------
class MplsDebugEthernetSegmentFilterVia( CliCommand.CliCommandClass ):
   syntax = 'mpls debug es-filter-via LABEL INTF'
   noOrDefaultSyntax = 'mpls debug es-filter-via LABEL ...'
   data = {
      'mpls': mplsNodeForConfig,
      'debug': debugNodeHidden,
      'es-filter-via' : 'EthernetSegmentFilterVia related (label and interface)',
      'LABEL': labelValMatcher,
      'INTF': Intf.matcher,
   }

   @staticmethod
   def handler( mode, args ):
      label = args[ 'LABEL' ]
      intf = args[ 'INTF' ]

      eSFilterVia = LfibSysdbESFilterVia( label, IntfId( intf.name ) )
      debugLfib.ethernetSegmentFilterVia.addMember( eSFilterVia )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del debugLfib.ethernetSegmentFilterVia[ args[ 'LABEL' ] ]

BasicCli.GlobalConfigMode.addCommandClass( MplsDebugIpLookupVia )
BasicCli.GlobalConfigMode.addCommandClass( MplsDebugVlanVia )
BasicCli.GlobalConfigMode.addCommandClass( MplsDebugVlanFloodVia )
BasicCli.GlobalConfigMode.addCommandClass( MplsDebugEthernetSegmentFilterVia )

#-------------------------------------------------------------------------
# [no|default] mac address-table debug static <mac_addr> vlan <vlan_id> label
# <label_value> tunnel <tunnel_type> <tunnel_index>  (hidden from user)
#-------------------------------------------------------------------------
def getTunnelIntfId( tunnelType, tunnelIndex ):
   tunTypeEnum = tunTypeEnumDict[ tunnelType ]
   tunnelId = Tac.Type( 'Tunnel::TunnelTable::TunnelId' ).convertToTunnelValue(
                 tunTypeEnum, tunnelIndex )
   tunnelIntfId = Tac.Type( 'Arnet::DynamicTunnelIntfId' ).tunnelIdToIntfId(
                     tunnelId )
   return tunnelIntfId

tunnelTypeMatcher = CliMatcher.EnumMatcher( {
   'static': 'Static tunnel',
   'ldp': 'LDP tunnel',
   'segment-routing': 'Segment Routing tunnel',
} )

class MacAddrDebugStaticTunnel( CliCommand.CliCommandClass ):
   syntax = ( 'MAC_ADDR_TABLE debug static MACADDR vlan VLAN_ID label LABEL '
              'tunnel TUN_TYPE TUN_IDX' )
   noOrDefaultSyntax = syntax
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForConfig,
      'debug': nodeMacAddressTableDebugHidden,
      'static': macStaticKwMatcher,
      'MACADDR': MacAddr.MacAddrMatcher(),
      'vlan': matcherVlanKw,
      'VLAN_ID': vlanIdMatcher,
      'label': 'Specify the MPLS label to push',
      'LABEL': labelValMatcher,
      'INTF': Intf.matcher,
      'tunnel': 'Tunnel interface',
      'TUN_TYPE': tunnelTypeMatcher,
      'TUN_IDX': tunnelIndexMatcher,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      tunnelIntfId = getTunnelIntfId( args[ 'TUN_TYPE' ], args[ 'TUN_IDX' ] )
      setStaticHelper( mode, args[ 'MACADDR' ], args[ 'VLAN_ID' ],
                       [ tunnelIntfId ], args[ 'LABEL' ], no=no )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( MacAddrDebugStaticTunnel )

#------------------------------------------
#  [ no | default ] mpls static top-label LABEL debug
#                   tunnel-type TTYPE tunnel-index TINDEX
#                   [ ( swap-label OUT_LABEL)
#                   | ( pop [ payload-type PTYPE ] ) ]
#                   [ metric METRIC ]
# ( Hidden )
#------------------------------------------
def getTunnelIntfIdFromTunnelInfo( tunnelType, tunnelIndex ):
   tunnelId = getTunnelIdFromIndex( tunnelTypeXlate[ tunnelType ], tunnelIndex )
   tunnelIntfId = Tac.Type( 'Arnet::DynamicTunnelIntfId' ).tunnelIdToIntfId(
                     tunnelId )
   return tunnelIntfId

class MplsStaticTopLabelTunnel( CliCommand.CliCommandClass ):
   syntax = ( 'mpls static top-label LABEL debug '
              'tunnel-type TTYPE tunnel-index TINDEX '
              '[ ( swap-label OUT_LABEL) '
              '| ( pop [ payload-type PTYPE ] ) ] '
              '[ metric METRIC ]' )
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'LABEL': topLabelValMatcher,
      'debug': debugNodeHidden,
      'tunnel-type': 'Tunnel type',
      'TTYPE': tunnelTypeValMatcher,
      'tunnel-index': 'Tunnel index',
      'TINDEX': tunnelIndexMatcher,
      'swap-label': swapLabelMatcher,
      'OUT_LABEL': outLabelValMatcher,
      'pop': popMatcher,
      'payload-type': payloadTypeMatcher,
      'PTYPE': payloadTypeNoBypassMatcher,
      'metric': metricMatcher,
      'METRIC': metricValMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      # Relevant arguments.
      label = args[ 'LABEL' ]
      tunnelIntfId = getTunnelIntfIdFromTunnelInfo( args[ 'TTYPE' ],
                                                    args[ 'TINDEX' ] )
      if 'pop' in args:
         labelOp = {
               'pop': ( args.get( 'PTYPE', 'autoDecide' ), False ),
         }
      elif 'swap-label' in args:
         labelOp = {
               'swap': {
                     'outLabel': args.get( 'OUT_LABEL' ),
               },
         }
      else:
         labelOp = None

      via = {
            'intfNexthop': {
                  'intf': tunnelIntfId,
            },
            'metric': args.get( 'METRIC' ),
            'labelOp': labelOp,
      }
      handleMplsLabelConfig( mode, [ label ], via )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Relevant arguments.
      label = args[ 'LABEL' ]
      tunnelType = args.get( 'TTYPE' )
      tunnelIndex = args.get( 'TINDEX' )
      if tunnelType and tunnelIndex:
         tunnelIntfId = getTunnelIntfIdFromTunnelInfo( tunnelType, tunnelIndex )
         via = {
               'intfNexthop': {
                     'intf': tunnelIntfId,
               },
               'metric': args.get( 'METRIC' ),
         }
         if 'pop' in args:
            labelOp = {
                  'pop': ( args.get( 'PTYPE', 'autoDecide' ), False ),
            }
         elif 'swap-label' in args:
            labelOp = {
                  'swap': {
                        'outLabel': args.get( 'OUT_LABEL' ),
                  },
            }
         else:
            labelOp = None

         via = {
               'intfNexthop': {
                     'intf': tunnelIntfId,
               },
               'metric': args.get( 'METRIC' ),
               'labelOp': labelOp,
         }
      else:
         via = None
      handleNoMplsLabelConfig( mode, [ label ], via )

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticTopLabelTunnel )

#----------------------------------------------------
# show mpls lfib input bgp labeled-unicast
#----------------------------------------------------
class ShowMplsLfibInputBgpLuCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls lfib input bgp labeled-unicast'
   data = {
      'mpls': mplsForShowNode,
      'lfib': 'Show MPLS LFIB',
      'input' : 'Show MPLS LFIB inputs',
      'bgp' : 'Show BGP LFIB inputs',
      'labeled-unicast' : 'Show BGP labeled-unicast LFIB inputs',
   }
   hidden = True
   cliModel = MplsModel.MplsRoutes

   @staticmethod
   def handler( mode, args ):
      showMplsConfigWarnings( mode )
      capiRevision = mode.session_.requestedModelRevision()

      LazyMount.force( lfibInfo )
      if not lfibInfo:
         return cliPrinted( MplsModel.MplsRoutes() )

      labelStack = Tac.Value( "Arnet::BoundedMplsLabelStack" )
      dispFilter = Tac.Type( "Mpls::DisplayFilter" ).bgpLuInputFilter

      sys.stdout.flush()
      fd = sys.stdout.fileno()
      fmt = mode.session_.outputFormat()
      helper = Tac.newInstance( "Mpls::MplsRouteHelper", labelStack, capiRevision,
                                mplsSupported() )
      helper.lfibInfo = lfibInfo
      helper.tunnelFib = tunnelFib
      helper.render( fd, fmt, None, None, None, None, None, None,
                     bgpLuLfibInput, dispFilter, True, False )
      return cliPrinted( MplsModel.MplsRoutes )

BasicCli.addShowCommandClass( ShowMplsLfibInputBgpLuCmd )

#------------------------------------------
# [ no | default ] mpls static top-label LABEL debug
#                  fec-type FTYPE fec-index FINDEX
#                  ( ( pop [ payload-type PTYPE ] )
#                  | ( forward ) )
# ( Hidden )
#------------------------------------------
def getFecTypeTokens():
   tokens = {}
   shortNameToCompleteName = {}
   completeNameToShortName = {}
   for adj in FecType.attributes:
      if adj.endswith( 'Adj' ):
         trimmedAdj = adj.lower().replace( "adj", "" )
         shortNameToCompleteName[ trimmedAdj ] = adj
         completeNameToShortName[ adj ] = trimmedAdj
         tokens[ trimmedAdj ] = 'Match %s fec type' % trimmedAdj

   return tokens, shortNameToCompleteName, completeNameToShortName

fecTypes, shortFecTypeToCompleteType, _ = getFecTypeTokens()

class MplsStaticTopLabelFec( CliCommand.CliCommandClass ):
   syntax = ( 'mpls static top-label LABEL debug '
              'fec-type FTYPE fec-index FINDEX '
              '( ( pop [ payload-type PTYPE ] ) '
              '| ( forward ) )' )
   noOrDefaultSyntax = 'mpls static top-label LABEL debug ...'
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'LABEL': topLabelValMatcher,
      'debug': debugNodeHidden,
      'fec-type': 'FEC type',
      'FTYPE': CliMatcher.EnumMatcher( fecTypes ),
      'fec-index': 'FEC index',
      'FINDEX': CliMatcher.IntegerMatcher( 1, 2 ** 56 - 1, helpdesc='FEC index' ),
      'pop': popMatcher,
      'payload-type': payloadTypeMatcher,
      'PTYPE': payloadTypeNoBypassMatcher,
      'forward': 'Forward the top label',
   }

   @staticmethod
   def handler( mode, args ):
      label = args[ 'LABEL' ]
      fecType = args[ 'FTYPE' ]
      fecIndex = args[ 'FINDEX' ]
      fecId = FecId.fecIdForAdjType( shortFecTypeToCompleteType[ fecType ],
                                     fecIndex )
      labelAction = 'pop' if 'pop' in args else 'forward'

      payloadType = args.get( 'PTYPE', 'autoDecide' )

      extFecVia = LfibSysdbExtFecVia( label, fecId, labelAction )
      extFecVia.payloadType = payloadType

      # Add it into Sysdb.
      debugLfib.extFecVia.addMember( extFecVia )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del debugLfib.extFecVia[ args[ 'LABEL' ] ]

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticTopLabelFec )

#-------------------------------------------------------------------------
# The "[no] mpls debug [ backup ] tunnel static <name> <endpoint> <nexthop>
# <interface> label-stack <label-stack>" command
# Note: This command shares syntax with MplsStaticTunnel in MplsCli.py. If
# updating it here, please update the non-debug command accordingly.
#-------------------------------------------------------------------------
class MplsDebugStaticTunnel( CliCommand.CliCommandClass ):
   syntax = '''mpls debug [ backup ] tunnel static NAME
               ( ( ( ( ( TEP_V4_ADDR | TEP_V4_PREFIX ) NEXTHOP_V4 ) |
                     ( ( TEP_V6_ADDR | TEP_V6_PREFIX ) NEXTHOP_V6 ) )
                   INTF ) |
                 ( ( resolving R_TEP ) |
                   ( resolving-tunnel RT_TEP tunnel-type RT_TYPE
                     tunnel-index RT_INDEX ) ) )
               ( ( label-stack { LABELS } ) | imp-null-tunnel )
            '''

   noOrDefaultSyntax = '''mpls debug [ backup ] tunnel static NAME ...'''

   data = {
      'mpls': mplsNodeForConfig,
      'debug': debugNodeHidden,
      'backup': 'Backup static tunnel via',
      'tunnel': tunnelNode,
      'static': staticTunnelMatcher,
      'NAME': tunnelNameValMatcher,
      'TEP_V4_ADDR': CliPlugin.IpAddrMatcher.ipAddrMatcher,
      'TEP_V4_PREFIX': CliPlugin.IpAddrMatcher.ipPrefixExpr(
         'IP address',
         "Subnet's mask value",
         'IP address with mask length',
         overlap=CliPlugin.IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
      ),
      'NEXTHOP_V4': CliPlugin.IpAddrMatcher.ipAddrMatcher,
      # FIXME BUG200408: Un-hide IPv6 endpoints and nexthop in "mpls static tunnel"
      'TEP_V6_ADDR': CliCommand.Node(
         matcher=CliPlugin.Ip6AddrMatcher.ip6AddrMatcher,
         hidden=True ),
      'TEP_V6_PREFIX': CliCommand.Node(
         matcher=CliPlugin.Ip6AddrMatcher.ip6PrefixMatcher,
         hidden=True ),
      'NEXTHOP_V6': CliCommand.Node(
         matcher=CliPlugin.Ip6AddrMatcher.ip6AddrMatcher,
         hidden=True ),
      'INTF': intfValMatcher,
      'resolving': 'Static tunnel over resolving prefix',
      'R_TEP': CliPlugin.IpGenAddrMatcher.IpGenPrefixMatcher( 'Resolving prefix' ),
      'resolving-tunnel': 'Static tunnel over tunnel',
      'RT_TEP': CliPlugin.IpGenAddrMatcher.IpGenPrefixMatcher(
         'Resolving tunnel prefix' ),
      'tunnel-type': 'Tunnel type',
      'RT_TYPE': tunnelTypeValMatcher,
      'tunnel-index': 'Tunnel index',
      'RT_INDEX': tunnelIndexMatcher,
      'label-stack': labelStackKeywordNode,
      'LABELS': labelStackValNode,
      'imp-null-tunnel': impNullTunnelNode,
   }

   adapter = labelsAdapter

   @staticmethod
   def handler( mode, args ):
      tunnelName = args[ 'NAME' ]
      endpoint = ( args.get( 'TEP_V4_ADDR' ) or args.get( 'TEP_V4_PREFIX' ) or
                   args.get( 'TEP_V6_ADDR' ) or args.get( 'TEP_V6_PREFIX' ) )
      nexthop = args.get( 'NEXTHOP_V4' ) or args.get( 'NEXTHOP_V6' )
      intf = args.get( 'INTF' )
      resolvingPrefix = args.get( 'R_TEP' ) or args.get( 'RT_TEP' )
      resolvingTunnelType = args.get( 'RT_TYPE' )
      resolvingTunnelIndex = args.get( 'RT_INDEX' )
      if 'imp-null-tunnel' in args:
         labels = [ 3 ]
      else:
         labels = args.get( 'LABELS' )
         if not validateLabelStackSize( mode, labels ):
            return

      if 'resolving' in args:
         resolvingPrefixAddr = Arnet.IpGenPrefix( str( resolvingPrefix ) )
         if resolvingPrefixAddr.af == AddressFamily.ipv4:
            prefixKey = Arnet.Prefix( resolvingPrefix )
            route = routingStatus.route.get( prefixKey )
         elif resolvingPrefixAddr.af == AddressFamily.ipv6:
            prefixKey = Arnet.Ip6Prefix( resolvingPrefix )
            route = routing6Status.route.get( prefixKey )
         else:
            mode.addError( "Invalid resolving prefix: neither IPv4 nor IPv6." )
            return
         if route is None:
            mode.addError( "Route to %s not found." % resolvingPrefix )
            return
         nexthopAddr = Arnet.IpGenAddr( '0.0.0.0' )
         fecId = route.fecId
         usedByTunnelFecId = FecId.convertFecIdForUseByTunnel( fecId )
         intfId = FecId.fecIdToHierarchicalIntfId( usedByTunnelFecId )
         tunnelEndpoint = resolvingPrefixAddr
      elif 'resolving-tunnel' in args:
         resolvingPrefixAddr = Arnet.IpGenPrefix( str( resolvingPrefix ) )
         nexthopAddr = Arnet.IpGenAddr( '0.0.0.0' )
         intfId = getTunnelIntfIdFromTunnelInfo( resolvingTunnelType,
                                                 resolvingTunnelIndex )
         tunnelEndpoint = resolvingPrefixAddr
      else:
         nexthopAddr, tunnelEndpoint, intfId = getMplsTunnelNhTepAndIntfId( mode,
                                                                            nexthop,
                                                                            endpoint,
                                                                            intf )
         if ( nexthopAddr, tunnelEndpoint, intfId ) == ( None, None, None ):
            return

      staticTunnelConfigEntry = staticTunnelConfig.entry.get( tunnelName )
      if staticTunnelConfigEntry and staticTunnelConfigEntry.tep == tunnelEndpoint:
         newStaticTunnelConfigEntry = \
            copyStaticTunnelConfigEntry( staticTunnelConfigEntry )
      else:
         newStaticTunnelConfigEntry = StaticTunnelConfigEntry( tunnelName )
         newStaticTunnelConfigEntry.tep = tunnelEndpoint
      # add newly configured via to either via or backupVia
      newMplsVia = MplsVia( nexthopAddr, intfId, labelOperation( labels ) )
      if 'backup' in args:
         newStaticTunnelConfigEntry.backupVia = newMplsVia
      else:
         # Clear existing vias if any, and add newly created via
         newStaticTunnelConfigEntry.via.clear()
         newStaticTunnelConfigEntry.via[ newMplsVia ] = True
         newStaticTunnelConfigEntry.inStaticTunnelMode = False

      if staticTunnelConfigEntry:
         # Check for duplicate and returns if it is the case
         if staticTunnelConfigEntry == newStaticTunnelConfigEntry:
            return
      staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tunnelName = args[ 'NAME' ]

      staticTunnelConfigEntry = staticTunnelConfig.entry.get( tunnelName )
      if staticTunnelConfigEntry:
         newStaticTunnelConfigEntry = \
            copyStaticTunnelConfigEntry( staticTunnelConfigEntry )
         if 'backup' in args:
            newStaticTunnelConfigEntry.backupVia = MplsVia()
         else:
            newStaticTunnelConfigEntry.via.clear()
         # if neither primary via or backup via is still valid, then delete the
         # entry, else update it in the static tunnel config table
         if ( not newStaticTunnelConfigEntry.via and
              newStaticTunnelConfigEntry.backupVia == MplsVia() ):
            del staticTunnelConfig.entry[ tunnelName ]
         else:
            staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )

BasicCli.GlobalConfigMode.addCommandClass( MplsDebugStaticTunnel )

def Plugin( entityManager ):
   global bgpLuLfibInput
   global debugLfib
   global lfibInfo
   global routing6Status
   global routingStatus
   global staticTunnelConfig
   global tunnelFib
   # Mpls::LfibSysdbStatus is essentially a Sysdb-compatible version of
   # Mpls::LfibStatus. (Currently, we only care about the ipLookupVia field in it.)
   # This way, we can write to the debugLfib in CLI, and then have a SyncSm to
   # copy the relevant info from this debugLfib into a Mpls::LfibStatus. This
   # Mpls::LfibStatus will not be persistently stored, but will just be passed
   # into the LfibSmashMergeSm downstream. This way, we can add and remove
   # entries in the debugLfib and have changes propagate through the system.
   # We're only interested in the IpLookupVia for now. See aid/3901.
   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   readerInfo = Smash.mountInfo( 'reader' )
   bgpLuLfibInput = smashEm.doMount( "mpls/protoLfibInputDir/bgpLu",
                                     "Mpls::LfibStatus", readerInfo )
   debugLfib = ConfigMount.mount( entityManager, 'mpls/debug/lfib',
                                  'Mpls::LfibSysdbStatus', 'w' )
   lfibInfo = LazyMount.mount( entityManager, 'routing/mpls/lfibInfo',
                               'Mpls::LfibInfo', 'r' )
   routing6Status = smashEm.doMount( 'routing6/status', 'Smash::Fib6::RouteStatus',
                                     FibUtils.routeStatusInfo( 'reader' ) )
   routingStatus = smashEm.doMount( 'routing/status', 'Smash::Fib::RouteStatus',
                                    FibUtils.routeStatusInfo( 'reader' ) )
   staticTunnelConfig = ConfigMount.mount( entityManager, "tunnel/static/config",
                                          "Tunnel::Static::Config", "w" )
   tunnelFib = smashEm.doMount( 'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib',
                                readerInfo )
