#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from CliModel import (
   Model,
   Dict,
   Enum,
   Bool,
   Int,
)
import IntfCli
from IntfModel import (
   Interface,
   prettyIntfStatusMap,
)
from TableOutput import (
   createTable,
   Format,
)

def renderBool( value ):
   if value is None:
      return '-'
   elif value:
      return 'Yes'
   return 'No'

class MplsInterfaceInfo( Model ):
   status = Enum( values=prettyIntfStatusMap.keys(),
                  help='Operational status of the interface' )
   mplsConfigured = Bool( help='Configuration of MPLS on interface' )
   ldpConfigured = Bool( help='Configuration of LDP on interface' )
   rsvpConfigured = Bool( help='Configuration of RSVP on interface' )

   def render( self ):
      return [ prettyIntfStatusMap[ self.status ],
               renderBool( self.mplsConfigured ),
               renderBool( self.ldpConfigured ),
               renderBool( self.rsvpConfigured ) ]

class MplsInterfaceModel( Model ):
   intfs = Dict( keyType=Interface, valueType=MplsInterfaceInfo,
                 help="A mapping between an Interface and MPLS interface info" )

   def render( self ):
      headers = [ 'Interface', 'Status', 'MPLS', 'LDP', 'RSVP' ]
      f1 = Format( justify='left' )
      f1.padLimitIs( True )
      outputTable = createTable( headers )
      for interface in Arnet.sortIntf( self.intfs ):
         interfaceInfo = self.intfs[ interface ]
         outputTable.newRow( interface, *interfaceInfo.render() )
      outputTable.formatColumns( *[ f1 for _ in headers ] )
      print outputTable.output()

# show mpls interfaces <i> counters
class MplsInterfaceIngressCounters( MplsInterfaceModel ):
   # Mpls per interface ingress counters
   class MplsInterfaceIngressCounters( Model ):
      mplsInOctets = Int( help="Number of input MPLS octets", default=0 )
      mplsInPkts = Int( help="Number of input MPLS packets", default=0 )

   interfaces = Dict( keyType=Interface, valueType=MplsInterfaceIngressCounters,
                      help="MPLS interface ingress counters" )

   def printRow( self, intf, inOctets, inPkts ):
      print "{:10} {:>18} {:>15}".format( intf, inOctets, inPkts )

   def render( self ):
      if not self.interfaces:
         return
      self.printRow( "Interface", "MPLSInOctets", "MPLSInPkts" )

      for intf in Arnet.sortIntf( self.interfaces ):
         x = self.interfaces[ intf ]
         self.printRow( IntfCli.Intf.getShortname( intf ), x.mplsInOctets,
                        x.mplsInPkts )
