#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: rpmwith %{_libdir}/libRsvp.so*

from CliModel import UnknownEntityError
from CliPlugin.AleCountersCli import (
   checkCounterFeatureEnabled,
   checkCounterFeatureSupported,
)
from CliPlugin.MplsCliModel import (
   MplsInterfaceModel,
   MplsInterfaceInfo,
   MplsInterfaceIngressCounters,
)
from EthIntfCli import (
   EthPhyAutoIntfType,
   EthIntf,
)
from IntfRangePlugin.LoopbackIntf import LoopbackAutoIntfType
from IntfRangePlugin.VlanIntf import VlanAutoIntfType
from IpLibConsts import DEFAULT_VRF
from RoutingIntfUtils import isManagement
from LagIntfCli import LagAutoIntfType
from VlanIntfCli import VlanIntf
from SubIntfCli import SubIntf
from TypeFuture import TacLazyType
import CliParser
import IntfCli
import LazyMount
import SmashLazyMount
import Tac

FapId = TacLazyType( 'FlexCounters::FapId' )
FeatureId = TacLazyType( 'FlexCounters::FeatureId' )
RsvpCliConfig = TacLazyType( 'Rsvp::RsvpCliConfig' )

ldpConfigColl = None
mplsInterfaceCounterTable = None
mplsInterfaceSnapshotTable = None
mplsRoutingConfig = None
rsvpCliConfig = None

#--------------------------------------------------------------------------------
# show mpls interface [INTERFACE]
#--------------------------------------------------------------------------------

intfTypes = ( EthPhyAutoIntfType, LagAutoIntfType,
              LoopbackAutoIntfType, VlanAutoIntfType )
# Classes of interfaces to filter in 'show mpls interfaces' command
intfTypeClassesForShowMplsIntfs = [ EthIntf, VlanIntf, SubIntf ]

def generateInterfaceModel( mode, args ):
   intf = args.get( 'INTERFACE' )
   # FIXME: Pass "module" argument mod=xxx to getAll(), BUG374639
   intfs = IntfCli.Intf.getAll( mode, intf=intf,
                                intfType=intfTypeClassesForShowMplsIntfs )
   # Filter Management interfaces and those that do not support routing
   intfs = [ i for i in intfs if i.routingCurrentlySupported()
                                 and not isManagement( i.name ) ]
   if not intfs:
      if intf is None:
         msg = "No MPLS/IP capable interfaces"
      else:
         names = list( intf.intfNames() )
         msg = "%s does not support MPLS/IP" % names[ 0 ]
      raise UnknownEntityError( msg )

   ldpConfig = ldpConfigColl.config.get( DEFAULT_VRF )
   ldpEnabled = ldpConfig is not None and ldpConfig.enabled
   rsvpEnabled = rsvpCliConfig is not None and rsvpCliConfig.enabled
   mplsInterfaceInfo = {}
   for intf in intfs:
      mplsConfigured = True
      if not mplsRoutingConfig.mplsRouting or \
         mplsRoutingConfig.mplsRoutingDisabledIntf.get( intf.name ):
         mplsConfigured = False
      # both Ldp and Mpls must be enabled to consider this interface as Ldp enabled
      ldpIntfEnabled = mplsConfigured and ldpEnabled

      # check the interface is in the selected list, if only running on select intfs
      if ldpIntfEnabled and ldpConfig.onlyLdpEnabledIntfs:
         ldpIntfEnabled = intf.name in ldpConfig.ldpEnabledIntf

      mplsInterfaceInfo[ intf.name ] = MplsInterfaceInfo(
         status=intf.getIntfState(),
         mplsConfigured=mplsConfigured,
         ldpConfigured=ldpIntfEnabled,
         rsvpConfigured=rsvpEnabled )

   return MplsInterfaceModel( intfs=mplsInterfaceInfo )

#--------------------------------------------------------------------------------
# show mpls interface [INTERFACE] counters
#--------------------------------------------------------------------------------

def mplsInterfaceCountersGuard( mode, token ):
   if not checkCounterFeatureSupported( FeatureId.MplsInterfaceIngress ):
      return CliParser.guardNotThisPlatform
   return None

def mplsInterfaceSmashCounter( intf ):
   counter = mplsInterfaceCounterTable.counterEntry.get( intf )
   snapshot = mplsInterfaceSnapshotTable.counterEntry.get( intf )
   if counter is None:
      pkts = 0
      octets = 0
   elif snapshot is None:
      pkts = counter.pkts
      octets = counter.octets
   else:
      pkts = counter.pkts - snapshot.pkts
      octets = counter.octets - snapshot.octets
   return pkts, octets

def showInterfacesMplsCounter( mode, args ):
   intf = args.get( 'INTERFACE' )
   mod = None # FIXME, BUG374639
   if not checkCounterFeatureEnabled( FeatureId.MplsInterfaceIngress ):
      feature = "hardware counter feature mpls interface in"
      mode.addError( '"{}" should be enabled first'.format( feature ) )
      return None
   mplsInterfaceCounters = MplsInterfaceIngressCounters()
   intfs = IntfCli.counterSupportedIntfs( mode, intf, mod )
   if not intfs:
      return mplsInterfaceCounters
   # Filter out non-'Ethernet' interfaces.
   intfs = ( i.name for i in intfs if i.name.startswith( 'Ethernet' ) )
   for intf in intfs:
      if not mplsRoutingConfig.mplsRouting or \
          mplsRoutingConfig.mplsRoutingDisabledIntf.get( intf ):
         continue
      intfCounters = mplsInterfaceCounters.MplsInterfaceIngressCounters()
      intfCounters.mplsInPkts, intfCounters.mplsInOctets = \
            mplsInterfaceSmashCounter( intf )
      mplsInterfaceCounters.interfaces[ intf ] = intfCounters
   return mplsInterfaceCounters

def Plugin( em ):
   global ldpConfigColl
   global mplsInterfaceCounterTable
   global mplsInterfaceSnapshotTable
   global mplsRoutingConfig
   global rsvpCliConfig

   # Mount the ldp config.
   ldpConfigColl = LazyMount.mount( em,
                                    "mpls/ldp/ldpConfigColl",
                                    "Ldp::LdpConfigColl",
                                    "r" )

   mplsRoutingConfig = LazyMount.mount( em,
                                        "routing/mpls/config",
                                        "Mpls::Config",
                                        "r" )

   # Mount the RSVP config
   rsvpCliConfig = LazyMount.mount( em,
                                    RsvpCliConfig.mountPath,
                                    "Rsvp::RsvpCliConfig",
                                    "r" )

   readerInfo = SmashLazyMount.mountInfo( 'reader' )

   # Mount the Mpls interface current counter smash.
   mountPath = 'flexCounters/counterTable/MplsInterfaceIngress/%u' % (
         FapId.allFapsId )
   mplsInterfaceCounterTable = SmashLazyMount.mount( em, mountPath,
         "Ale::FlexCounter::MplsInterfaceIngressCounterTable", readerInfo )

   # Mount the Mpls interface snapshot counter smash.
   mountPath = 'flexCounters/snapshotTable/MplsInterfaceIngress/%u' % (
         FapId.allFapsId )
   mplsInterfaceSnapshotTable = SmashLazyMount.mount( em, mountPath,
         "Ale::FlexCounter::MplsInterfaceIngressCounterTable", readerInfo )
