# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements MPLS configuration.
#-------------------------------------------------------------------------------
'''Configuration commands supported for MPLS'''

import sys

import Arnet.MplsLib
import BasicCli
import CliParser
import CliCommand
import CliMatcher
import ConfigMount
import IntfCli
import IpAddrMatcher
import Ip6AddrMatcher
import IpGenAddrMatcher
import IraIpIntfCli
import LazyMount
import MplsLib
import ShowCommand
import MplsModel
from MplsModel import LABEL_RANGE_STATIC, LABEL_RANGE_DYNAMIC, LABEL_RANGE_UNASSIGNED
from MplsTypeLib import (
   showTunnelFibIgnoredTunnelTypes,
)
import SharedMem
import Smash
import Tac
import TacSigint
import TechSupportCli
import AgentCommandRequest
import Toggles.MplsToggleLib
import Toggles.EvpnLibToggleLib
import TunnelModels
from CliModel import cliPrinted
from IpLibConsts import DEFAULT_VRF
from IraIpCli import vrfKwMatcher, vrfNameMatcher
from IraNexthopGroupCli import nexthopGroupNameMatcher
from SrTePolicyCommonLib import srTePolicyStatusPath
from TunnelCli import (
   TunnelTableIdentifier,
   getEndpointFromTunnelId,
   getMplsViaModelFromTunnelVia,
   getTunnelIdFromIndex,
   getTunnelIndexFromId,
   getTunnelTypeFromTunnelId,
   getTunnelViaStatusFromTunnelId,
   getViaModels,
   readMountTunnelTable,
   tunnelIndexMatcher,
)
from TypeFuture import TacLazyType
from CliMode.Mpls import StaticMulticastModeBase, TunnelStaticModeBase

MAX_LABEL_EXCEEDED = "More labels are specified than the hardware can handle"

# pkgdeps library RoutingLib

AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
BoundedMplsLabelStack = TacLazyType( 'Arnet::BoundedMplsLabelStack' )
FecId = TacLazyType( 'Smash::Fib::FecId' )
FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
LabelRangeInfo = Tac.Type( 'Mpls::LabelRangeInfo' )
LfibHwProgrammingReturnCode = Tac.Type( 'Mpls::LfibHwProgrammingReturnCode' )
LfibSource = Tac.Type( "Mpls::LFib::Source" )
MplsLabel = Tac.Type( 'Arnet::MplsLabel' )
MplsLabelAction = Tac.Type( 'Arnet::MplsLabelAction' )
MplsStackEntryIndex = TacLazyType( 'Arnet::MplsStackEntryIndex' )
MplsVia = Tac.Type( 'Tunnel::TunnelTable::MplsVia' )
PayloadType = TacLazyType( 'Mpls::PayloadType' )
RouteKey = Tac.Type( 'Mpls::RouteKey' )
RouteKeyAndMetric = Tac.Type( 'Mpls::RouteKeyAndMetric' )
RouteMetric = Tac.Type( 'Mpls::RouteMetric' )
StaticTunnelConfigEntry = TacLazyType( 'Tunnel::Static::StaticTunnelConfigEntry' )
TacDyTunIntfId = Tac.Type( 'Arnet::DynamicTunnelIntfId' )
TtlMode = Tac.Type( 'Mpls::TtlMode' )

implicitNullLabel = MplsLabel.implicitNull

em = None
cliChurnTestHelper = None
cliMplsRouteConfig = None
cliMplsRouteConfigReadOnly = None
configMode = BasicCli.GlobalConfigMode
decapLfib = None
decapLfibConsumer = None
decapLfibHw = None
evpnEthernetSegmentLfib = None
evpnEthernetSegmentLfibHw = None
fecModeSm = None
fecModeStatus = None
forwarding6Status = None
forwardingStatus = None
ip6Config = None
ipConfig = None
l3Config = None
l3ConfigDir = None
l3NHResolver = None
lfibInfo = None
lfibStatus = None
mldpOpaqueValueTable = None
gribiAfts = None
mplsHwCapability = None
mplsHwStatus = None
mplsRouteConfig = None
mplsRouteConfigDir = None
mplsRouteConfigMergeSm = None
mplsRoutingConfig = None
mplsTunnelConfig = None
mplsStatus = None
mplsRoutingStatus = None
mplsVrfLabelConfig = None
pseudowireLfib = None
pwRouteHelper = None
routingHardwareStatus = None
routing6VrfInfoDir = None
routingVrfInfoDir = None
srTePolicyStatus = None
srteForwardingStatus = None
staticMcastLfib = None
staticTunnelConfig = None
staticTunnelTable = None
transitLfib = None
tunnelFib = None
unifiedForwarding6Status = None
unifiedForwardingStatus = None
vrfGlobalTunPref = None

def mplsSupported():
   return mplsHwCapability.mplsSupported

def mplsSupportedGuard( mode, token ):
   if mplsHwCapability.mplsSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def evpnMplsSupportedGuard( mode, token ):
   if mplsHwCapability.evpnMplsSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsStaticVrfLabelSupported( mode, token ):
   if mplsHwCapability.mplsStaticVrfLabelSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsIntfSupportedGuard( mode, token ):
   if mplsHwCapability.mplsIntfCfgSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsSkipEgressAclSupportedGuard( mode, token ):
   if mplsHwCapability.mplsSkipEgressAclSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsForwardIntoNexthopGroupSupportedGuard( mode, token ):
   if mplsHwCapability.mplsForwardIntoNexthopGroupSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsMultiLabelLookupGuard( mode, token ):
   if mplsHwCapability.mplsMultiLabelLookupSupported():
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsTunnelTerminationGuard( mode, token ):
   if mplsHwCapability.mplsTunnelTerminationSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsPhpTtlGuard( mode, token ):
   if mplsHwCapability.mplsPhpTtlSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsPhpDscpUniformGuard( mode, token ):
   if token == 'pipe' or mplsHwCapability.mplsPhpDscpUniformSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsMulticastSupportedGuard( mode, token ):
   if mplsHwCapability.mplsMulticastSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def getForwardingStatus():
   unifiedMode = Tac.Type( 'Smash::Fib::FecMode' ).fecModeUnified
   if fecModeStatus.fecMode == unifiedMode:
      return( unifiedForwardingStatus, unifiedForwarding6Status )
   else:
      return( forwardingStatus, forwarding6Status )

def mplsEntropyLabelSupportedGuard( mode, token ):
   if mplsHwCapability.mplsEntropyLabelSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def getMaxMplsLabelStack():
   return Tac.Type( 'Arnet::MplsStackEntryIndex' ).max + 1

def getPlatformMaxMplsLabelStack():
   return routingHardwareStatus.nexthopGroupMplsLabelStackSize

tunnelKwMatcherMplsShow = CliMatcher.KeywordMatcher( 'tunnel',
   helpdesc="MPLS tunnel information and counters" )

mplsNodeForConfig = CliCommand.guardedKeyword( 'mpls',
      helpdesc="Global MPLS configuration commands",
      guard=mplsSupportedGuard )
_ipForMplsMatcher = CliMatcher.KeywordMatcher( 'ip',
                      helpdesc="Enable MPLS IP routing globally",
                      # legacy support: "mpls routing"
                      alternates=[ 'routing' ] )
staticMatcher = CliMatcher.KeywordMatcher( 'static',
      helpdesc='Static MPLS configuration commands' )
topLabelMatcher = CliMatcher.KeywordMatcher( 'top-label',
      helpdesc="Specify the top-most MPLS labels" )

# Used in Ldp and Isis.
bindingsKw = CliMatcher.KeywordMatcher( 'bindings', helpdesc='Label bindings' )

# Used in Ldp and Rsvp.
mplsMatcherForClear = CliMatcher.KeywordMatcher( 'mpls',
      helpdesc='Clear MPLS information' )

nhStr = "Address of the nexthop router"
intfValMatcher = IntfCli.Intf.matcher

# This is an arbitrary number >= the largest value we support on any HW platform
# for multi-pop. It is used to limit the number of labels we will accept for
# static MPLS routes. The reason for making this a constant value is for the
# startup config scenario, where these config commands are loaded before the MPLS
# HW config has been initialized.
maxIngressMplsTopLabels = 2

def topLabelRangeFn( mode ):
   # when guards are disabled, allow configured labels that are outside of
   # the static label range configuration
   if mode.session_.guardsEnabled():
      labelRange = labelRangeValue( LabelRangeInfo.rangeTypeStatic )
      labelMin = labelRange.base
      labelMax = labelRange.base + labelRange.size - 1
   else:
      labelMin = Arnet.MplsLib.labelMin
      labelMax = Arnet.MplsLib.labelMax
   return labelMin, labelMax

topLabelValMatcher = CliMatcher.DynamicIntegerMatcher(
      rangeFn=topLabelRangeFn,
      helpdesc='Value of the MPLS label' )
# This is used to register a second command for multi-label routes that only
# support pop. If swap is ever supported in the future, this can be modified.
# Decrement maxiter by one since one label is already given from topLabelValMatcher
topLabelsValNode = CliCommand.Node(
      matcher=topLabelValMatcher,
      maxMatches=maxIngressMplsTopLabels - 1,
      guard=mplsMultiLabelLookupGuard )

# for no/default mpls static top-label <xxxx>, there's no need to check the
# staticLabelRange configuration
_noTopLabelValMatcher = Arnet.MplsLib.labelValMatcher

_nexthopMatcher = IpGenAddrMatcher.IpGenAddrMatcher(
      helpdesc="Address of the nexthop router" )
_nexthopOnIntfMatcher = IpGenAddrMatcher.IpGenAddrMatcher(
      helpdesc="Forwarding router's address on destination interface" )

swapLabelMatcher = CliMatcher.KeywordMatcher( 'swap-label',
      helpdesc="Specify the label to swap the top label with" )
outLabelValMatcher = Arnet.MplsLib.labelValMatcher
metricMatcher = CliMatcher.KeywordMatcher( 'metric',
      helpdesc='Specify the metric for the route' )
metricValMatcher = CliMatcher.IntegerMatcher(
      RouteMetric.min, RouteMetric.max,
      helpdesc='Metric for the route' )

_accessListMatcher = CliCommand.guardedKeyword( 'access-list',
                      helpdesc="Egress access-list",
                      guard=mplsSkipEgressAclSupportedGuard )

popMatcher = CliMatcher.KeywordMatcher( 'pop',
      helpdesc='Pop the top label' )
payloadTypeMatcher = CliMatcher.KeywordMatcher( 'payload-type',
      helpdesc='Specify the type of the underlying payload' )

_ipv4PayloadHelp = "Underlying payload type is Ipv4"
_ipv6PayloadHelp = "Underlying payload type is Ipv6"
_mplsPayloadHelp = "Underlying payload type is MPLS"
_autoPayloadHelp = "Auto detect Underlying payload type"
_autoPayloadTypeMatcher = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'auto',
                                         helpdesc=_autoPayloadHelp ),
      hidden=True )

class PayloadTypeWithBypassMatcher( CliCommand.CliExpressionFactory ):
   def __init__( self ):
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      payloadTypeKwName = name + '_payload-type'
      accessListKwName = name + '_access-list'
      bypassKwName = name + '_bypass'
      mplsKwName = name + '_mpls'
      ipv4KwName = name + '_ipv4'
      ipv6KwName = name + '_ipv6'
      autoKwName = name + '_auto'

      class PayloadTypeWithBypassExpr( CliCommand.CliExpression ):
         expression = ( '{plt} ( '
                        ' {mpls} | '
                        ' ( ( {ipv4} | {ipv6} | {auto} ) '
                        '   [ {al} {bypass} ] ) )'.format(
                           plt=payloadTypeKwName,
                           mpls=mplsKwName,
                           ipv4=ipv4KwName,
                           ipv6=ipv6KwName,
                           auto=autoKwName,
                           bypass=bypassKwName,
                           al=accessListKwName,
                           ) )
         data = {
            payloadTypeKwName: payloadTypeMatcher,
            mplsKwName: CliMatcher.KeywordMatcher( 'mpls',
               helpdesc=_mplsPayloadHelp ),
            ipv4KwName: CliMatcher.KeywordMatcher( 'ipv4',
               helpdesc=_ipv4PayloadHelp ),
            ipv6KwName: CliMatcher.KeywordMatcher( 'ipv6',
               helpdesc=_ipv6PayloadHelp ),
            autoKwName: _autoPayloadTypeMatcher,
            accessListKwName: _accessListMatcher,
            bypassKwName: CliMatcher.KeywordMatcher( "bypass",
               helpdesc="Bypass egress access-list" ),
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            # Verify that no other arg was added with the same name we want to
            # write the adaptor output to.
            assert name not in args

            if mplsKwName in args:
               args[ name ] = PayloadType.mpls
            else:
               bypass = bypassKwName in args
               if ipv4KwName in args:
                  args[ name ] = ( PayloadType.ipv4, bypass )
               elif ipv6KwName in args:
                  args[ name ] = ( PayloadType.ipv6, bypass )
               elif autoKwName in args:
                  args[ name ] = ( PayloadType.autoDecide, bypass )

      return PayloadTypeWithBypassExpr

# The NoBypass version is used for multi-pop routes and forwarding into NHGs,
# since we don't support bypassing egress ACLs for these
payloadTypeNoBypassMatcher = CliMatcher.EnumMatcher( {
      'ipv4': _ipv4PayloadHelp,
      'ipv6': _ipv6PayloadHelp,
      'mpls': _mplsPayloadHelp,
} )

mplsForShowNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'mpls', helpdesc='Show MPLS information' ),
      guard=mplsSupportedGuard )

nextHopMatcher = CliMatcher.KeywordMatcher( 'next-hop',
                                            helpdesc="MPLS next-hop configuration" )

def ipRoutingEnabledAny( ):
   routingInfo = routingVrfInfoDir.get( DEFAULT_VRF )
   routing6Info = routing6VrfInfoDir.get( DEFAULT_VRF )
   return ( routingInfo and routingInfo.routing ) or \
          ( routing6Info and routing6Info.routing )

def showMplsConfigWarnings( mode ):
   if not mplsRoutingConfig.mplsRouting:
      mode.addWarning( "Mpls routing is not enabled" )
   if not ipRoutingEnabledAny():
      mode.addWarning( "Neither IP nor IPv6 routing is enabled" )

#---------------------------------------------------------
# [no|default] mpls ip
#---------------------------------------------------------
class MplsIpCmd( CliCommand.CliCommandClass ):
   syntax = "mpls ip"
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsNodeForConfig,
      'ip': _ipForMplsMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mplsRoutingConfig.mplsRouting = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mplsRoutingConfig.mplsRouting = False

configMode.addCommandClass( MplsIpCmd )

#---------------------------------------------------------
# [no|default] mpls next-hop resolution allow default-route
#---------------------------------------------------------
nexthopResolutionConfigChangeWarning = \
   "Change will take effect only after switch reboot or forwarding agent restart."

class MplsNextHopResolutionAllowDefaultRouteCmd( CliCommand.CliCommandClass ):
   syntax = "mpls next-hop resolution allow default-route"
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsNodeForConfig,
      'next-hop': nextHopMatcher,
      'resolution': "MPLS next-hop resolution configuration",
      'allow': "Select when possible",
      'default-route': "Allow resolving MPLS nexthops over default route",
   }

   @staticmethod
   def handler( mode, args ):
      allow = not CliCommand.isNoOrDefaultCmd( args )
      mplsRoutingConfig.nexthopResolutionAllowDefaultRoute = allow
      mode.addWarning( nexthopResolutionConfigChangeWarning )

   noOrDefaultHandler = handler

configMode.addCommandClass( MplsNextHopResolutionAllowDefaultRouteCmd )

# Entropy label command tokens
tunnelNodeHelpStr = 'Tunnel configuration commands'
tunnelNode = CliMatcher.KeywordMatcher( 'tunnel', helpdesc=tunnelNodeHelpStr )

entropyLabelNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'entropy-label',
                                         helpdesc="Entropy label configuration" ),
      guard=mplsEntropyLabelSupportedGuard )

#-------------------------------------------------------------
# [no|default] mpls entropy-label pop in "config" mode
#-------------------------------------------------------------
class MplsEntropyLabelPopCmd( CliCommand.CliCommandClass ):
   syntax = "mpls entropy-label pop"
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsNodeForConfig,
      'entropy-label': entropyLabelNode,
      'pop': 'Enable popping entropy label at the PHP',
   }

   @staticmethod
   def handler( mode, args ):
      mplsRoutingConfig.entropyLabelPop = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mplsRoutingConfig.entropyLabelPop = False

#-------------------------------------------------------------
# [no|default] mpls tunnel entropy-label push in "config" mode
#-------------------------------------------------------------
class MplsTunnelEntropyLabelPushCmd( CliCommand.CliCommandClass ):
   syntax = "mpls tunnel entropy-label push"
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'entropy-label': entropyLabelNode,
      'push': 'Enable pushing entropy label on ingress LER for all MPLS tunnels',
   }

   @staticmethod
   def handler( mode, args ):
      mplsTunnelConfig.entropyLabelPush = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mplsTunnelConfig.entropyLabelPush = False

if Toggles.MplsToggleLib.toggleMplsEntropyLabelSupportEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( MplsTunnelEntropyLabelPushCmd )
   BasicCli.GlobalConfigMode.addCommandClass( MplsEntropyLabelPopCmd )

def initCliHelper():
   return Tac.newInstance( "Mpls::CliHelper", "MplsCli",
                           cliMplsRouteConfig.force() )

def initMplsRouteConfigMergeSm( entityManager ):
   global mplsRouteConfigMergeSm
   global mplsRouteConfig
   mplsRouteConfig = Tac.newInstance( 'Mpls::RouteConfig', 'routeConfig' )
   if Toggles.MplsToggleLib.toggleArexMplsMergeSmEnabled():
      ctrl = Tac.Value( 'Arx::SmControl' )
      ctrl.runToCompletion = True
      mplsRouteConfigMergeSm = Tac.newInstance( 'Mpls::RouteConfigMergerSm',
                                                mplsRouteConfig,
                                                mplsRouteConfigDir,
                                                cliMplsRouteConfigReadOnly,
                                                ctrl )
   else:
      mplsRouteConfigMergeSm = Tac.newInstance( 'Mpls::RouteConfigMergeSm',
                                                mplsRouteConfig,
                                                mplsRouteConfigDir,
                                                cliMplsRouteConfigReadOnly )

def initFecModeSm():
   global fecModeSm
   global fecModeStatus
   fecModeStatus = Tac.newInstance( 'Smash::Fib::FecModeStatus', 'fms' )
   fecModeSm = Tac.newInstance( 'Ira::FecModeSm', l3Config, fecModeStatus )

def validatePopViaPayload( vias, newVia ):
   '''
   This function checks all the vias for a the label to see if there are any pop vias
   configured. If there are any then validates that the payload type is the same for
   all the pop vias. If not then it returns and error.
   In addition if there is a pop via configured with the same 
   nexthop, payload-type as the new incoming via, but a different option for the 
   access-list bypass parameter then that via is deleted. The new via is going to 
   replace the deleted via. eg.
   'mpls static top-label x 10.1.1.1 pop payload-type ipv4' then if the user 
   configures 'mpls static top-label x 10.1.1.1 pop payload-type ipv4 access-list 
   bypass' then the old via gets replaced with the new via configured.
   '''

   for via in vias:
      if via.labelAction == MplsLabelAction.pop:
         if newVia.payloadType != via.payloadType:
            # return error string indicating payload mismatch
            return ( "Cannot configure pop operations for the same label with"
                     " different payload type" )
         else:
            break

   viaToCheck = Tac.Value( "Mpls::Via",
                            nextHop=newVia.nextHop,
                            labelAction=newVia.labelAction,
                            outLabel=newVia.outLabel,
                            skipEgressAcl=not newVia.skipEgressAcl)
   viaToCheck.payloadType = newVia.payloadType

   if viaToCheck in vias:
      del vias[ viaToCheck ]

   return None

topLabelCountWarning = 'Maximum number of top-labels supported is {}'
labelStackSizeWarning = 'Maximum label stack size is {} labels'.format(
      MplsStackEntryIndex.max + 1 )

def handleMplsLabelConfig( mode, inLabels, via ):
   cliHelper = initCliHelper()
   intfNexthop = via[ 'intfNexthop' ]
   nexthop = intfNexthop.get( 'nexthop' )
   if ( nexthop is not None and
        not isinstance( nexthop, Tac.Type( 'Arnet::IpGenAddr' ) ) ):
      nexthop = Tac.Value( 'Arnet::IpGenAddr', nexthop )
   intf = intfNexthop.get( 'intf' )
   nexthopGroup = intfNexthop.get( 'nexthopGroup' )
   isDynamicTunnel = False
   if isinstance( intf, str ):
      isDynamicTunnel = TacDyTunIntfId.isDynamicTunnelIntfId( intf )
   # You can set at most one of the following
   assert sum( [ int( nexthop is not None),
                 int( nexthopGroup is not None),
                 int( isDynamicTunnel ) ] ) == 1
   labelOp = via[ 'labelOp' ]
   metric = via[ 'metric' ]
   error = None

   assert inLabels
   isMultiLabel = len( inLabels ) > 1

   staticLabelRange = labelRangeValue( LabelRangeInfo.rangeTypeStatic )
   base = staticLabelRange.base
   size = staticLabelRange.size
   minLabel = base
   maxLabel = base + size - 1
   for inLabel in inLabels:
      if inLabel < minLabel or inLabel > maxLabel:
         # Just display a warning and continue. In interactive configuration,
         # the plugin will not get here since the topLabelValMatcher will not
         # match out of range labels. This warning will just be displayed
         # in the startup config output, or during config-replace, etc.
         mode.addWarning( 'mpls static top-label ' + str( inLabel ) +
                          ' is outside of the mpls label range, base: ' +
                          str( base ) + ', size: ' + str( size ) )

   # Check for multi-label pop routes and display warnings if necessary
   if len( inLabels ) > MplsStackEntryIndex.max:
      mode.addWarning( labelStackSizeWarning )
      return

   labelStack = BoundedMplsLabelStack()
   for labelVal in inLabels:
      labelVal = int( labelVal )
      labelStack.prepend( labelVal )

   # During startup config, mplsLookupLabelMaxCount is not initialized, so we
   # check some arbitrary "large" number to ensure we aren't trying to configure
   # a gargantuan, non-sensical label stack. This is needed because we aren't
   # enforcing the number of labels we can inhale from 'mpls static top-label'
   if mode.session_.startupConfig():
      if labelStack.stackSize > maxIngressMplsTopLabels:
         return
   else:
      if labelStack.stackSize > mplsHwCapability.mplsLookupLabelMaxCount:
         mode.addWarning( topLabelCountWarning.format(
            mplsHwCapability.mplsLookupLabelMaxCount ) )
         return

   outLabel = None
   skipEgressAcl = False
   if metric is None:
      metric = Tac.Type( 'Mpls::Constants' ).metricDefault

   if labelOp is None:
      assert nexthopGroup or isMultiLabel, ( "Label action none is only allowed "
                                             "with forwarding into nexthop-group "
                                             "or static multi-label pop routes" )
      outLabel = implicitNullLabel
      action = 'forward'
      payloadType = 'undefinedPayload'
   elif 'swap' in labelOp:
      outLabel = labelOp[ 'swap' ][ 'outLabel' ]
      payloadType = 'mpls'
      action = 'swap'
      skipEgressAcl = True
      assert nexthopGroup is None
   elif 'pop' in labelOp:
      outLabel = implicitNullLabel
      action = 'pop'
      if labelOp[ action ] == 'mpls':
         payloadType = labelOp[ action ]
         skipEgressAcl = False
      else:
         payloadType, skipEgressAcl = labelOp[ 'pop' ]
   else:
      assert False, "Label op has to be swap, pop, or None"

   l = cliHelper.addMplsRoute( labelStack, metric )

   nexthop = nexthop or Tac.Value( 'Arnet::IpGenAddr' )
   via = Tac.Value( "Mpls::Via",
                    nextHop=nexthop,
                    labelAction=action,
                    outLabel=outLabel,
                    skipEgressAcl=skipEgressAcl,
                    payloadType=payloadType )
   if isDynamicTunnel:
      via.intf = intf
   elif intf:
      via.intf = intf.name

   if nexthopGroup:
      via.nexthopGroup = nexthopGroup

   maxEcmpSize = Tac.Type( "Mpls::Constants" ).maxEcmpSize
   if via not in l.via and len(l.via) >= maxEcmpSize:
      #return error back to the user
      mode.addError( "Maximum number of next hops supported is %d" % maxEcmpSize )
   else:
      #mplsHwCapability.mplsSkipEgressAclSupported = True
      if ( mplsHwCapability.mplsSkipEgressAclSupported and
           l.via and
           action == 'pop' ):
         error = validatePopViaPayload( l.via, via )
      if error is not None:
         # return the validation error
         mode.addError( error )
      else:
         l.via[ via ] = True

def handleNoMplsLabelConfig( mode, inLabels, via ):
   cliHelper = initCliHelper()

   assert inLabels
   nullMetric = Tac.Type( "Mpls::RouteMetric" ).null
   skipEgressAcl = False

   if len( inLabels ) > MplsStackEntryIndex.max:
      mode.addWarning( labelStackSizeWarning )
      return

   labelStack = BoundedMplsLabelStack()
   for labelVal in inLabels:
      labelVal = int( labelVal )
      labelStack.prepend( labelVal )

   if via is not None:
      intfNexthop = via[ 'intfNexthop' ]
      nexthop = intfNexthop.get( 'nexthop' ) or ''
      if not isinstance( nexthop, Tac.Type( 'Arnet::IpGenAddr' ) ):
         nexthop = Tac.Value( 'Arnet::IpGenAddr', nexthop )
      intf = intfNexthop.get( 'intf' )
      intfName = ''
      if isinstance( intf, str ) and TacDyTunIntfId.isDynamicTunnelIntfId( intf ):
         intfName = intf
      elif intf:
         intfName = intf.name
      nexthopGroup = intfNexthop.get( 'nexthopGroup', '' )
      labelOp = via[ 'labelOp' ]
      metric = via[ 'metric' ] if via[ 'metric' ] is not None else nullMetric

      if labelOp is None:
         outLabel = implicitNullLabel
         action = 'forward'
         payloadType = 'undefinedPayload'
      elif 'swap' in labelOp:
         outLabel = labelOp[ 'swap' ][ 'outLabel' ]
         payloadType = 'mpls'
         action = 'swap'
         skipEgressAcl = True
      elif 'pop' in labelOp:
         outLabel = implicitNullLabel
         action = 'pop'
         if labelOp[ action ] == 'mpls':
            payloadType = labelOp[ action ]
            skipEgressAcl = False
         else:
            payloadType, skipEgressAcl = labelOp[ 'pop' ]
      else:
         assert False, "Label op has to be swap, pop, or None"

      cliHelper.delMplsRouteVia( labelStack, metric, nexthop, nexthopGroup, action,
                                 outLabel, skipEgressAcl, payloadType, intfName )
   else:
      cliHelper.delMplsRoute( labelStack, nullMetric )

#------------------------------------------
# mpls static top-label <inlabel> [ intf ] <nhaddr> ...
# ( no | default ) mpls static top-label <inlabel> [ ... ]
#------------------------------------------
class MplsStaticRoutePopOrSwapCmd( CliCommand.CliCommandClass ):
   _viaSyntax = ( '( ( INTF ADDR_ON_INTF ) | ADDR ) '
                  '( ( swap-label OUT_LABEL ) | ( pop PAYLOAD_TYPE_AND_BYPASS ) ) '
                  '[ metric METRIC ]' )

   syntax = 'mpls static top-label IN_LABEL ' + _viaSyntax
   noOrDefaultSyntax = 'mpls static top-label NO_IN_LABEL [ %s ]' % _viaSyntax

   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'IN_LABEL': topLabelValMatcher,
      'NO_IN_LABEL': _noTopLabelValMatcher,
      'INTF': intfValMatcher,
      'ADDR_ON_INTF': _nexthopOnIntfMatcher,
      'ADDR': _nexthopMatcher,
      'pop': popMatcher,
      'PAYLOAD_TYPE_AND_BYPASS': PayloadTypeWithBypassMatcher(),
      'swap-label': swapLabelMatcher,
      'OUT_LABEL': outLabelValMatcher,
      'metric': metricMatcher,
      'METRIC': metricValMatcher,
   }

   @staticmethod
   def _getVia( args ):
      via = None
      addr = args.get( 'ADDR_ON_INTF' ) or args.get( 'ADDR' )
      if addr is not None:
         via = { 'intfNexthop': { 'intf': args.get( 'INTF' ),
                                  'nexthop': addr },
                 'metric': args.get( 'METRIC' ) }
         if 'pop' in args:
            via[ 'labelOp' ] = { 'pop': args[ 'PAYLOAD_TYPE_AND_BYPASS' ] }
         else:
            assert 'swap-label' in args
            via[ 'labelOp' ] = { 'swap': { 'outLabel': args[ 'OUT_LABEL' ] } }

      return via

   @staticmethod
   def handler( mode, args ):
      handleMplsLabelConfig( mode, [ args[ 'IN_LABEL' ] ],
                             MplsStaticRoutePopOrSwapCmd._getVia( args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      handleNoMplsLabelConfig( mode, [ args[ 'NO_IN_LABEL' ] ],
                               MplsStaticRoutePopOrSwapCmd._getVia( args ) )

configMode.addCommandClass( MplsStaticRoutePopOrSwapCmd )

#--------------------------------------------------------------------------
# [ no | default ] mpls lookup label count <1-2>
#--------------------------------------------------------------------------
def getLabelRange( mode ):
   if mplsHwCapability.mplsLookupLabelMaxCount == 1:
      # This is to avoid issues caused when startup config is loaded before the
      # hardware capability is set up, such as in the CliSaveTestLib and on a
      # real dut. This implementation won't write false data to the products because
      # mplsLookupLabelCount variable in Config is checked against product's
      # hardware capability before it is wrote to Status.
      return ( 1, sys.maxint )
   else:
      return ( 1, mplsHwCapability.mplsLookupLabelMaxCount )

class MplsLookupLabelCountCmd( CliCommand.CliCommandClass ):
   syntax = "mpls lookup label count LABEL_COUNT"
   noOrDefaultSyntax = "mpls lookup label count ..."

   data = {
      'mpls': mplsNodeForConfig,
      'lookup': CliCommand.guardedKeyword( 'lookup',
                   helpdesc="MPLS route lookup options",
                   guard=mplsMultiLabelLookupGuard ),
      'label': "MPLS label configuration commands",
      'count': "Specify the number of labels considered for route lookup",
      'LABEL_COUNT': CliMatcher.DynamicIntegerMatcher(
         rangeFn=getLabelRange,
         helpname=None,
         helpdesc="Number of labels considered for route lookup" )
   }

   @staticmethod
   def handler( mode, args ):
      mplsRoutingConfig.mplsLookupLabelCount = args[ 'LABEL_COUNT' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mplsRoutingConfig.mplsLookupLabelCount = 1

configMode.addCommandClass( MplsLookupLabelCountCmd )

class MplsLabelsMatcher( CliCommand.CliExpressionFactory ):
   '''
   Matches a single or multiple labels.
   Will always yield a list of labels, regardless of if one or multiple is matched.
   '''
   def __init__( self, labelMatcher, labelsMatcher, multiLabelOptional=True ):
      self.multiLabelOptional = multiLabelOptional
      self.labelMatcher = labelMatcher
      self.labelsMatcher = labelsMatcher
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      labelKwName = name + '_label'
      labelsKwName = name + '_labels'
      multiLabelOptional = self.multiLabelOptional
      labelMatcher = self.labelMatcher
      labelsMatcher = self.labelsMatcher

      class PayloadTypeWithBypassExpr( CliCommand.CliExpression ):
         if multiLabelOptional:
            expression = '( %s [ { %s } ] )' % ( labelKwName, labelsKwName )
         else:
            expression = '( %s { %s } )' % ( labelKwName, labelsKwName )

         data = {
            labelKwName: labelMatcher,
            labelsKwName: labelsMatcher,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            # Verify that no other arg was added with the same name we want to
            # write the adaptor output to.
            assert name not in args

            label = args.get( labelKwName )
            if label is None:
               return

            labels = args.get( labelsKwName )
            # There's a weird secnario where a `Node` with `maxMatches=1`
            # isn't a `list`.
            if isinstance( labels, int ):
               labels = [ labels ]
            labels = labels or []
            labels.insert( 0, label )
            args[ name ] = labels

      return PayloadTypeWithBypassExpr

_multiTopLabelValMatcher = MplsLabelsMatcher(
      topLabelValMatcher, topLabelsValNode, multiLabelOptional=False )

def labelsAdapter( mode, args, argsList ):
   # There's a weird secnario where a `Node` with `maxMatches=1` isn't a `list`.
   labels = args.get( 'LABELS' )
   if labels is not None and not isinstance( labels, list ):
      args[ 'LABELS' ] = [ labels ]

#--------------------------------------------------------------------------
# [ no | default ] mpls static top-label LABEL { LABELS } ADDR
#                  [ pop payload-type PTYPE ]
#                  [ metric METRIC ]
#--------------------------------------------------------------------------
class MplsStaticMultiLabelPop( CliCommand.CliCommandClass ):
   # BUG455910 There should be an optional interface here, effectively the same
   # as the single label mpls static config.
   syntax = ( 'mpls static top-label LABELS ADDR '
              '[ pop payload-type PTYPE ] '
              '[ metric METRIC ]' )
   noOrDefaultSyntax = ( 'mpls static top-label LABELS [ ADDR ] '
                         '[ pop payload-type PTYPE ] '
                         '[ metric METRIC ]' )
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'LABELS': _multiTopLabelValMatcher,
      'ADDR': _nexthopMatcher,
      'pop': popMatcher,
      'payload-type': payloadTypeMatcher,
      'PTYPE': payloadTypeNoBypassMatcher,
      'metric': metricMatcher,
      'METRIC': metricValMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      # Relevant arguments.
      via = { 'intfNexthop': { 'intf': None,
                               'nexthop': args[ 'ADDR' ] },
              'metric': args.get( 'METRIC' ) }
      if 'pop' in args:
         payloadType = args.get( 'PTYPE', 'autoDecide' )
         skipEgressAcl = False
         via[ 'labelOp' ] = { 'pop': ( payloadType, skipEgressAcl ) }
      else:
         via[ 'labelOp' ] = None
      handleMplsLabelConfig( mode, args[ 'LABELS' ], via )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Relevant arguments.
      if 'ADDR' in args:
         via = { 'intfNexthop': { 'intf': None,
                                  'nexthop': args.get( 'ADDR' ) },
                 'metric': args.get( 'METRIC' ) }
         if 'pop' in args:
            payloadType = args.get( 'PTYPE', 'autoDecide' )
            skipEgressAcl = False
            via[ 'labelOp' ] = { 'pop': ( payloadType, skipEgressAcl ) }
         else:
            via[ 'labelOp' ] = None
      else:
         via = None
      handleNoMplsLabelConfig( mode, args[ 'LABELS' ], via )

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticMultiLabelPop )

#--------------------------------------------------------------------------
# [ no | default ] mpls static top-label LABEL nexthop-group GROUP
#                  [ pop [ payload-type PTYPE ] ] [ metric METRIC ]
#--------------------------------------------------------------------------

_nexthopGroupNode = CliCommand.guardedKeyword( 'nexthop-group',
      helpdesc="Forwarding into nexthop-group",
      guard=mplsForwardIntoNexthopGroupSupportedGuard )

class MplsStaticTopLabelNexthopGroup( CliCommand.CliCommandClass ):
   syntax = ( 'mpls static top-label LABEL nexthop-group GROUP '
              '[ pop [ payload-type PTYPE ] ] [ metric METRIC ]' )
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'LABEL': topLabelValMatcher,
      'nexthop-group': _nexthopGroupNode,
      'GROUP': nexthopGroupNameMatcher,
      'pop': popMatcher,
      'payload-type': payloadTypeMatcher,
      'PTYPE': payloadTypeNoBypassMatcher,
      'metric': metricMatcher,
      'METRIC': metricValMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      # Relevant arguments.
      label = args[ 'LABEL' ]
      via = { 'intfNexthop': { 'intf': None,
                               'nexthop': None,
                               'nexthopGroup': args[ 'GROUP' ] },
              'metric': args.get( 'METRIC', None ) }
      if 'pop' in args:
         payloadType = args.get( 'PTYPE', 'autoDecide' )
         via[ 'labelOp' ] = { 'pop' : ( payloadType, False ) }
      else:
         via[ 'labelOp' ] = None
      handleMplsLabelConfig( mode, [ label ], via )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Relevant arguments.
      label = args[ 'LABEL' ]
      nexthopGroup = args.get( 'GROUP' )
      if nexthopGroup:
         via = { 'intfNexthop': { 'intf': None,
                                  'nexthop': None,
                                  'nexthopGroup': nexthopGroup },
                 'metric': args.get( 'METRIC', None ) }
         if 'pop' in args:
            payloadType = args.get( 'PTYPE', 'autoDecide' )
            via[ 'labelOp' ] = { 'pop' : ( payloadType, False ) }
         else:
            via[ 'labelOp' ] = None
      else:
         via = None
      handleNoMplsLabelConfig( mode, [ label ], via )

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticTopLabelNexthopGroup )

#--------------------------------------------------------------------------
# [ no | default ] mpls static top-label LABEL { LABELS } nexthop-group GROUP
#                  [ pop [ payload-type PTYPE ] ] [ metric METRIC ]
#--------------------------------------------------------------------------
class MplsStaticMultiLabelNexthopGroup( CliCommand.CliCommandClass ):
   syntax = ( 'mpls static top-label LABELS nexthop-group GROUP '
              '[ pop [ payload-type PTYPE ] ] [ metric METRIC ]' )
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'top-label': topLabelMatcher,
      'LABELS': _multiTopLabelValMatcher,
      'nexthop-group': _nexthopGroupNode,
      'GROUP': nexthopGroupNameMatcher,
      'pop': popMatcher,
      'payload-type': payloadTypeMatcher,
      'PTYPE': payloadTypeNoBypassMatcher,
      'metric': metricMatcher,
      'METRIC': metricValMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      # Relevant arguments.
      via = { 'intfNexthop': { 'intf': None,
                               'nexthop': None,
                               'nexthopGroup': args[ 'GROUP' ] },
              'metric': args.get( 'METRIC', None ) }
      if 'pop' in args:
         payloadType = args.get( 'PTYPE', 'autoDecide' )
         skipEgressAcl = False
         via[ 'labelOp' ] = { 'pop': ( payloadType, skipEgressAcl ) }
      else:
         via[ 'labelOp' ] = None
      handleMplsLabelConfig( mode, args[ 'LABELS' ], via )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Relevant arguments.
      nexthopGroup = args.get( 'GROUP' )
      if nexthopGroup:
         via = { 'intfNexthop': { 'intf': None,
                                  'nexthop': None,
                                  'nexthopGroup': nexthopGroup },
                 'metric': args.get( 'METRIC', None ) }
         if 'pop' in args:
            payloadType = args.get( 'PTYPE', 'autoDecide' )
            skipEgressAcl = False
            via[ 'labelOp' ] = { 'pop': ( payloadType, skipEgressAcl ) }
         else:
            via[ 'labelOp' ] = None
      else:
         via = None
      handleNoMplsLabelConfig( mode, args[ 'LABELS' ], via )

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticMultiLabelNexthopGroup )

#------------------------------------------
# [no] mpls static multicast top-label <inLabel>
#     next-hop <nhAddr> swap-label <outLabel>
#     next-hop <nhAddr> swap-label <outLabel>
#     ...
#------------------------------------------

LfibSysdbStaticMldpViaSet = Tac.Type( 'Mpls::LfibSysdbStaticMldpViaSet' )
LfibSysdbMplsVia = Tac.Type( 'Mpls::LfibSysdbMplsVia' )
multicastNode = CliCommand.guardedKeyword( 'multicast',
      helpdesc="MPLS multicast configuration commands",
      guard=mplsMulticastSupportedGuard )

class StaticMulticastMode( StaticMulticastModeBase, BasicCli.ConfigModeBase ):
   name = "Mpls static multicast route configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, inLabel ):
      StaticMulticastModeBase.__init__( self, inLabel )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.inLabel = MplsLabel( inLabel )

      self.vias = set() # a local version of viaSet, for easier entry handling
      if staticMcastLfib.mldpStaticViaSet.get( inLabel ):
         self._syncFromSysdbViaSet()
      else:
         newViaSet = LfibSysdbStaticMldpViaSet( self.inLabel )
         staticMcastLfib.mldpStaticViaSet.addMember( newViaSet )
      self.viasUpdated = False

   def onExit( self ):
      if self.viasUpdated:
         self._publishLocalViasToSysdb()

      BasicCli.ConfigModeBase.onExit( self )

   # convert staticMcastLfib.mldpStaticViaSet[ inLabel ] into local viaSet self.vias
   def _syncFromSysdbViaSet( self ):
      sysdbViaSet = staticMcastLfib.mldpStaticViaSet[ self.inLabel ]
      for idx in range( sysdbViaSet.getViaCount() ):
         via = sysdbViaSet.mplsVia[ idx ]
         nhAddr = via.nextHop.stringValue
         outLabel = via.outLabel.topLabel()
         self.vias.add( ( nhAddr, outLabel ) )
      assert sysdbViaSet.getViaCount() == len( self.vias )

   # publish entries in self.vias to staticMcastLfib.mldpStaticViaSet[ self.inLabel ]
   def _publishLocalViasToSysdb( self ):
      del staticMcastLfib.mldpStaticViaSet[ self.inLabel ]
      newViaSet = LfibSysdbStaticMldpViaSet( self.inLabel )
      for idx, ( nhAddr, outLabel ) in enumerate( self.vias ):
         nextHop = Arnet.IpGenAddr( nhAddr )
         outLabelStack = Tac.Value( 'Arnet::MplsLabelOperation' )
         outLabelStack.appendLabel( MplsLabel( outLabel ) )
         newVia = LfibSysdbMplsVia( self.inLabel, nextHop, outLabelStack )
         newViaSet.mplsVia[ idx ] = newVia
      staticMcastLfib.mldpStaticViaSet.addMember( newViaSet )
      assert len( self.vias ) == \
         staticMcastLfib.mldpStaticViaSet[ self.inLabel ].getViaCount()

#--------------------------------------------------------------------------
# [ no | default ] mpls static multicast top-label LABEL
#--------------------------------------------------------------------------
class MplsStaticMulticastConfigRoute( CliCommand.CliCommandClass ):
   syntax = 'mpls static multicast top-label LABEL'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'multicast': multicastNode,
      'top-label': topLabelMatcher,
      'LABEL': topLabelValMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      inLabel = args[ 'LABEL' ]
      childMode = mode.childMode( StaticMulticastMode, inLabel=inLabel )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del staticMcastLfib.mldpStaticViaSet[ args[ 'LABEL' ] ]

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticMulticastConfigRoute )

#--------------------------------------------------------------------------
# [ no | default ] mpls fec ip sharing disabled
#--------------------------------------------------------------------------
class MplsStaticFecSharing( CliCommand.CliCommandClass ):
   syntax = 'mpls fec ip sharing disabled'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'fec': 'FEC configuration',
      'ip': 'IP configuration',
      'sharing': 'IP sharing configuration',
      'disabled': 'Disable FEC sharing',
   }

   @staticmethod
   def handler( mode, args ):
      mplsRoutingConfig.optimizeStaticRoutes = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mplsRoutingConfig.optimizeStaticRoutes = True

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticFecSharing )

#--------------------------------------------------------------------------
# [ no | default ] next-hop ADDR swap-label LABEL
#--------------------------------------------------------------------------
class MplsStaticMulticastRouteConfigVia( CliCommand.CliCommandClass ):
   syntax = "next-hop ADDR swap-label LABEL"
   noOrDefaultSyntax = syntax
   data = {
      'next-hop': nextHopMatcher,
      'ADDR': IpAddrMatcher.IpAddrMatcher( "Address of the nexthop router" ),
      'swap-label': swapLabelMatcher,
      'LABEL': outLabelValMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      nhAddr = args[ 'ADDR' ]
      outLabel = args[ 'LABEL' ]
      if ( nhAddr, outLabel ) not in mode.vias:
         mode.viasUpdated = True
         mode.vias.add( ( nhAddr, outLabel ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nhAddr = args[ 'ADDR' ]
      outLabel = args[ 'LABEL' ]
      if ( nhAddr, outLabel ) in mode.vias:
         mode.viasUpdated = True
         mode.vias.remove( ( nhAddr, outLabel ) )

StaticMulticastMode.addCommandClass( MplsStaticMulticastRouteConfigVia )

#------------------------------------------
# [no] mpls static vrf-label <label> vrf <vrf-name>
#------------------------------------------
VrfLabel = TacLazyType( 'Mpls::VrfLabel' )

class MplsStaticVrfLabelCmd( CliCommand.CliCommandClass ):
   syntax = "mpls static vrf-label IN_LABEL vrf VRF_NAME"
   noOrDefaultSyntax = "mpls static vrf-label NO_IN_LABEL [ vrf VRF_NAME ]"
   data = {
      'mpls': mplsNodeForConfig,
      'static': staticMatcher,
      'vrf-label': CliCommand.guardedKeyword( 'vrf-label',
                      helpdesc="Specify the MPLS label to vrf mapping",
                      guard=mplsStaticVrfLabelSupported ),
      'IN_LABEL': topLabelValMatcher,
      'NO_IN_LABEL': _noTopLabelValMatcher,
      'vrf': vrfKwMatcher,
      'VRF_NAME': vrfNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      # Instantiate a via which contains info for vrf-name
      vrfLabel = VrfLabel( args[ 'IN_LABEL' ], args[ 'VRF_NAME' ] )

      # add it to the sysdb
      mplsVrfLabelConfig.vrfLabel.addMember( vrfLabel )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del mplsVrfLabelConfig.vrfLabel[ args[ 'NO_IN_LABEL' ] ]

configMode.addCommandClass( MplsStaticVrfLabelCmd )

def showMplsLfibRoute( mode, labels, typeFilter, detail=False ):
   showMplsConfigWarnings( mode )

   LazyMount.force( gribiAfts )
   LazyMount.force( lfibInfo )
   LazyMount.force( mldpOpaqueValueTable )
   LazyMount.force( cliChurnTestHelper )
   if not transitLfib or not lfibInfo:
      return cliPrinted( MplsModel.MplsRoutes() )

   capiRevision = mode.session_.requestedModelRevision()

   labelStack = Arnet.MplsLib.labelListToBoundedMplsLabelStack( labels or [] )

   DisplayFilter = Tac.Type( "Mpls::DisplayFilter" )

   if typeFilter:
      if typeFilter == 'mpls':
         dispFilter = DisplayFilter.mplsFilter
      if typeFilter == 'evpn':
         dispFilter = DisplayFilter.evpnFilter
      if typeFilter == 'pseudowire':
         dispFilter = DisplayFilter.pseudowireFilter
   else:
      dispFilter = DisplayFilter.all

   sys.stdout.flush()
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   helper = Tac.newInstance( "Mpls::MplsRouteHelper", labelStack, capiRevision,
                             mplsHwCapability.mplsSupported )
   helper.lfibInfo = lfibInfo
   helper.forwardingStatus, helper.forwarding6Status = getForwardingStatus()
   helper.churnTestHelper = cliChurnTestHelper
   helper.srteForwardingStatus = srteForwardingStatus
   helper.srTePolicyStatus = srTePolicyStatus
   helper.mplsAft = gribiAfts.mplsAft
   helper.mldpOpaqueValueTable = mldpOpaqueValueTable
   helper.pwRouteHelper = pwRouteHelper
   helper.tunnelFib = tunnelFib

   helper.render( fd, fmt, pseudowireLfib, transitLfib, decapLfib, decapLfibHw,
                  evpnEthernetSegmentLfib, evpnEthernetSegmentLfibHw,
                  None, dispFilter, True, detail )
   return cliPrinted( MplsModel.MplsRoutes )

def showMplsConfigRoute( mode, labels ):
   showMplsConfigWarnings( mode )

   LazyMount.force( l3NHResolver )

   labelStack = Arnet.MplsLib.labelListToBoundedMplsLabelStack( labels or [] )

   if labels:
      routeKeys = [ RouteKeyAndMetric.fromKey( RouteKey( labelStack ), i )
                    for i in range( RouteMetric.min, RouteMetric.max + 1 ) ]
   else:
      routeKeys = mplsRouteConfig.route.keys()

   routeDict = {}
   multiLabelRouteDict = None
   for rk in sorted( routeKeys ):
      route = mplsRouteConfig.route.get( rk )
      if route is None:
         continue

      # lookup LFIB and Ira RouteResolver to identify if route or its vias are down
      lfibRoute = transitLfib.lfibRoute
      lRoute = lfibRoute.get( rk.routeKey )
      if lRoute is None or lRoute.source != LfibSource.lfibSourceStaticMpls:
         rkStatus = "down"
      else:
         rkStatus = "up"
      nhvs = l3NHResolver.vrf.get( DEFAULT_VRF )

      vias = [ ]
      viaSet = route.via
      for via in viaSet:
         viaModel = MplsModel.MplsVia()
         viaModel.nexthop = via.nextHop
         viaModel.intf = via.intf
         viaModel.labelAction = via.labelAction
         viaModel.outLabel = via.outLabel
         viaModel.payloadType = via.payloadType \
                                 if via.labelAction == 'pop' else 'mpls'
         viaModel.skipEgressAcl = via.skipEgressAcl
         viaModel.ttlMode = via.ttlMode
         viaModel.dscpMode = via.dscpMode
         # Lookup in Nexthop Resolver if the next-hop has been resolved
         if nhvs is None or via.nextHop not in nhvs.nexthop:
            rkStatus = "down"
         viaModel.status = rkStatus
         vias.append( viaModel )

      adjacencyModel = MplsModel.MplsAdjacency(
            viaType=MplsModel.LfibViaType.viaTypeMplsIp, mplsVias=vias )
      if rk.routeKey.labelStack.stackSize < 2:
         curRouteDict = routeDict
         dictKey = rk.topLabel
         topLabel = dictKey
         topLabels = None
      else:
         if multiLabelRouteDict is None:
            multiLabelRouteDict = {}
         curRouteDict = multiLabelRouteDict
         dictKey = rk.routeKey.labelStack.cliShowString()
         topLabel = MplsLabel.null
         topLabels = dictKey
      routeModel = curRouteDict.get( dictKey, None )
      if routeModel:
         routeModel.adjacencies[ rk.metric ] = adjacencyModel
      else:
         routeModel = MplsModel.MplsRoute( topLabel=topLabel, topLabels=topLabels,
                                         adjacencies={ rk.metric : adjacencyModel } )
         curRouteDict[ dictKey ] = routeModel
   return MplsModel.MplsRoutes( routes=routeDict,
                                multiLabelRoutes=multiLabelRouteDict )

#--------------------------------------------------------------------------
# [ no | default ] mpls label range RTYPE BASE SIZE [ force ]'
#--------------------------------------------------------------------------
labelRangeKeywords = {
   LabelRangeInfo.rangeTypeStatic: 'Specify labels reserved for static MPLS routes',
   LabelRangeInfo.rangeTypeDynamic: 'Specify labels reserved for dynamic assignment',
   LabelRangeInfo.rangeTypeSrgb:
       'Specify labels reserved for IS-IS SR global segment identifiers (SIDs)',
   LabelRangeInfo.rangeTypeBgpSrgb:
       'Specify labels reserved for BGP SR global segment identifiers (SIDs)',
   LabelRangeInfo.rangeTypeSrlb:
       'Specify labels reserved for SR local segment identifiers (SIDs)',
}

ospfSrLabelRangeKeyword = {
   LabelRangeInfo.rangeTypeOspfSrgb:
       'Specify labels reserved for OSPF SR global segment identifiers (SIDs)'
   }

def labelRangeValue( rangeType ):
   return MplsLib.labelRange( mplsRoutingConfig, rangeType )

def defaultLabelRange( rangeType ):
   return MplsLib.labelRangeDefault( mplsRoutingConfig, rangeType )

def labelRangeKeywordsFn():
   result = labelRangeKeywords
   if Toggles.MplsToggleLib.toggleOspfSegmentRoutingMplsEnabled():
      result = dict( result ) # copy
      result.update( ospfSrLabelRangeKeyword )
   return result

class RangeTypeExpression( CliCommand.CliExpression ):
   expression = 'RANGE_TYPE'
   data = {
      'RANGE_TYPE': CliMatcher.EnumMatcher( labelRangeKeywords ),
      'l2evpn': 'Specify labels reserved for L2 EVPN routes',
   }
   if Toggles.MplsToggleLib.toggleOspfSegmentRoutingMplsEnabled():
      expression = expression + ' | ospf-sr'
      data[ 'ospf-sr' ] = 'Specify labels reserved for OSPF SR global segment ' \
                          'identifiers (SIDs)'
   if Toggles.EvpnLibToggleLib.toggleSharedEsiLabelEnabled():
      expression = expression + ' | ( l2evpn [ ethernet-segment ] )'
      data[ 'ethernet-segment' ] = 'Specify labels reserved for L2 EVPN A-D ' \
                                   'per ES routes for split-horizon filtering'
   else:
      expression = expression + ' | l2evpn'

   @staticmethod
   def adapter( mode, args, argsList ):
      if args.get( 'ethernet-segment' ):
         rangeType = LabelRangeInfo.rangeTypeL2evpnSharedEs
      else:
         rangeType = args.get( 'RANGE_TYPE' ) or \
                     args.get( 'ospf-sr' ) or \
                     args.get( 'l2evpn' )
      args[ 'RTYPE' ] = rangeType

class CfgMplsLabelRange( CliCommand.CliCommandClass ):
   syntax = 'mpls label range RTYPE BASE SIZE [ force ]'
   noOrDefaultSyntax = 'mpls label range RTYPE ...'
   data = {
      'mpls': mplsNodeForConfig,
      'label': 'Specify label range allocations',
      'range': 'Specify label range allocations',
      'RTYPE': RangeTypeExpression,
      'BASE': CliMatcher.IntegerMatcher( MplsLabel.unassignedMin,
                                         MplsLabel.max,
                                         helpdesc='Starting label to reserve' ),
      'SIZE': CliMatcher.IntegerMatcher( 0,
                                         MplsLabel.max - MplsLabel.unassignedMin + 1,
                                         helpdesc='Number of labels to reserve' ),
      'force': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher( 'force', helpdesc='' ),
                  hidden=True ),
   }

   @staticmethod
   def _rangeOverlap( rangeX, rangeY ):
      if rangeX[ 1 ] == 0 or rangeY[ 1 ] == 0:
         return False
      # Convert ( base, size ) to inclusive ranges [ label1, label2 ] for x,y
      x = ( rangeX[ 0 ], rangeX[ 0 ] + rangeX[ 1 ] - 1 )
      y = ( rangeY[ 0 ], rangeY[ 0 ] + rangeY[ 1 ] - 1 )
      return x[ 0 ] <= y[ 1 ] and y[ 0 ] <= x[ 1 ]

   @staticmethod
   def _allowOverlap( *args ):
      allowedOverlapList = ( LabelRangeInfo.rangeTypeSrgb,
                             LabelRangeInfo.rangeTypeBgpSrgb,
                             LabelRangeInfo.rangeTypeOspfSrgb )
      return all( arg in allowedOverlapList for arg in args )

   @staticmethod
   def _checkForRangeError( mode, rangeType, base, size ):
      if not mode.session_.guardsEnabled():
         return False

      others = ( set( mplsRoutingConfig.labelRangeDefault.keys() ) - 
                 set( [ rangeType ] ) )
      for otherName in others:
         other = labelRangeValue( otherName )
         if ( CfgMplsLabelRange._rangeOverlap( ( base, size ),
                                               ( other.base, other.size ) ) and
              not CfgMplsLabelRange._allowOverlap( otherName, rangeType ) ):
            mode.addError( 'Label range %s overlaps with mpls label range %s' % (
                                                  ( rangeType, otherName ) ) )
            return True

      if rangeType != LabelRangeInfo.rangeTypeStatic or not mplsRouteConfig.route:
         return False

      confMin = MplsLabel.max
      confMax = MplsLabel.min
      for routeKey in mplsRouteConfig.route:
         if routeKey.topLabel < confMin:
            confMin = routeKey.topLabel
         if routeKey.topLabel > confMax:
            confMax = routeKey.topLabel

      if confMin != MplsLabel.max and confMax != MplsLabel.min:
         staticBase = confMin
         staticSize = confMax - confMin + 1
         if confMin < base or confMax >= base + size:
            mode.addError( 'Label range needs to encompass existing static '
                           'MPLS routes '
                           '(existing range is %d, size %d).' % ( staticBase,
                                                                  staticSize ) )
            return True
      return False

   @staticmethod
   def _saveRange( rangeType, base, size ):
      value = Tac.Value( 'Mpls::LabelRange', base, size )
      mplsRoutingConfig.labelRange[ rangeType ] = value

   @staticmethod
   def handler( mode, args ):
      base = args[ 'BASE' ]
      size = args[ 'SIZE' ]
      rangeType = args[ 'RTYPE' ]
      force = 'force' in args
      if size > MplsLabel.max - base + 1:
         mode.addError( 'Label range base + size must be less than %d' %
                        ( MplsLabel.max + 1 ) )
         return

      if ( rangeType == LabelRangeInfo.rangeTypeDynamic and
           size != 0 and size < LabelRangeInfo.dynamicMinSize and
           mode.session_.guardsEnabled() and not force ):
         mode.addError( 'Dynamic label range size must be at least %d' %
                        LabelRangeInfo.dynamicMinSize )
         return

      # check for overlap
      if CfgMplsLabelRange._checkForRangeError( mode, rangeType, base, size ):
         return
      CfgMplsLabelRange._saveRange( rangeType, base, size )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      rangeType = args[ 'RTYPE' ]
      rangeDefault = defaultLabelRange( rangeType )
      base = rangeDefault.base
      size = rangeDefault.size
      # check for overlap
      if CfgMplsLabelRange._checkForRangeError( mode, rangeType, base, size ):
         return
      CfgMplsLabelRange._saveRange( rangeType, base, size )

configMode.addCommandClass( CfgMplsLabelRange )

#----------------------------------------------
# [no] mpls ip command in config-if mode
#----------------------------------------------
modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

mplsIntfModeKw = CliCommand.guardedKeyword( 'mpls',
      helpdesc='Interface level MPLS configuration commands',
      guard=mplsIntfSupportedGuard )

class IntfMplsIpCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ip'
   noOrDefaultSyntax = syntax

   data = {
      'mpls': mplsIntfModeKw,
      'ip': 'Enable MPLS traffic on interface if MPLS enabled globally',
   }

   @staticmethod
   def handler( mode, args ):
      disable = CliCommand.isNoCmd( args )
      intf = mode.intf.name
      if disable:
         mplsRoutingConfig.mplsRoutingDisabledIntf[ intf ] = True
      else:
         del mplsRoutingConfig.mplsRoutingDisabledIntf[ intf ]

   noOrDefaultHandler = handler

modelet.addCommandClass( IntfMplsIpCmd )

class IntfMplsConfigCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      # Need to set the config to default i.e. enable mpls
      intf = self.intf_.name
      del mplsRoutingConfig.mplsRoutingDisabledIntf[ intf ]

mplsNodeForShow = CliCommand.guardedKeyword( 'mpls',
      helpdesc='Show MPLS information',
      guard=mplsSupportedGuard )
_mplsRouteKwForShowMatcher = CliMatcher.KeywordMatcher( 'route',
                                                        helpdesc='Show MPLS routes' )
_mplsHwRouteKwForShowMatcher = CliMatcher.KeywordMatcher(
      'route', helpdesc='Show MPLS routes (detail)' )
matcherLfib = CliMatcher.KeywordMatcher( 'lfib', helpdesc='Show MPLS LFIB' )
labelValMatcher = Arnet.MplsLib.labelValMatcher
labelsValNode = CliCommand.Node(
      matcher=labelValMatcher,
      guard=mplsMultiLabelLookupGuard,
      maxMatches=maxIngressMplsTopLabels - 1 )

_filterViaTypeKwMatcher = CliMatcher.KeywordMatcher(
      'type', helpdesc="Filter via types" )

_tunnelTypeFilterValMatcher = CliMatcher.EnumMatcher( {
   'mpls': 'MPLS IP vias',
   'evpn': 'EVPN vias',
   'pseudowire': 'Pseudowire vias',
} )

_singleOrMultiLabelValMatcher = MplsLabelsMatcher( labelValMatcher, labelsValNode,
                                                   multiLabelOptional=True )

#-------------------------------------------------------------
# show mpls lfib route [type {mpls | evpn | pseudowire }] [<label> ...] [detail]
#-------------------------------------------------------------
class ShowMplsLfibRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show mpls lfib route [ type TYPE ] [ LABELS ] [ detail ]"
   data = {
      'mpls': mplsNodeForShow,
      'lfib': matcherLfib,
      'route': _mplsRouteKwForShowMatcher,
      'type': _filterViaTypeKwMatcher,
      'TYPE': _tunnelTypeFilterValMatcher,
      'LABELS': _singleOrMultiLabelValMatcher,
      'detail': "Display information in detail",
   }
   cliModel = MplsModel.MplsRoutes

   @staticmethod
   def handler( mode, args ):
      return showMplsLfibRoute( mode, args.get( 'LABELS' ), args.get( 'TYPE' ),
                                detail='detail' in args )

BasicCli.addShowCommandClass( ShowMplsLfibRouteCmd )

#------------------------------------------------------
# show mpls config route [<labels> ...]
#------------------------------------------------------
class ShowMplsConfigRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show mpls config route [ LABELS ]"
   data = {
      'mpls': mplsNodeForShow,
      'config': "Show MPLS configuration",
      'route': _mplsRouteKwForShowMatcher,
      'LABELS': _singleOrMultiLabelValMatcher,
   }
   cliModel = MplsModel.MplsRoutes

   @staticmethod
   def handler( mode, args ):
      return showMplsConfigRoute( mode, args.get( 'LABELS' ) )

BasicCli.addShowCommandClass( ShowMplsConfigRouteCmd )

def showMplsHwRoute( mode, labels, typeFilter ):
   showMplsConfigWarnings( mode )

   LazyMount.force( mplsHwStatus )
   LazyMount.force( cliChurnTestHelper )
   if not mplsHwStatus:
      return MplsModel.MplsHwStatus( mplsHwStatus=[] )

   labelStack = Arnet.MplsLib.labelListToBoundedMplsLabelStack( labels or [] )

   DisplayFilter = Tac.Type( "Mpls::DisplayFilter" )

   if typeFilter:
      if typeFilter == 'mpls':
         dispFilter = DisplayFilter.mplsFilter
      if typeFilter == 'evpn':
         dispFilter = DisplayFilter.evpnFilter
      if typeFilter == 'pseudowire':
         dispFilter = DisplayFilter.pseudowireFilter
   else:
      dispFilter = DisplayFilter.all

   sys.stdout.flush()
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   capiRevision = mode.session_.requestedModelRevision()
   helper = Tac.newInstance( "Mpls::MplsRouteHelper", labelStack, capiRevision,
                             mplsHwCapability.mplsSupported )
   helper.mplsHwStatus =  mplsHwStatus
   helper.forwardingStatus, helper.forwarding6Status = getForwardingStatus()
   helper.srteForwardingStatus = srteForwardingStatus
   helper.srTePolicyStatus = srTePolicyStatus
   helper.tunnelFib = tunnelFib
   helper.churnTestHelper = cliChurnTestHelper

   with TacSigint.immediateMode():
      helper.render( fd, fmt, pseudowireLfib, transitLfib, decapLfib, decapLfibHw,
                     evpnEthernetSegmentLfib, evpnEthernetSegmentLfibHw,
                     None, dispFilter, False, False )
   return MplsModel.MplsHwStatus

#-----------------------------------------------
# show mpls route [type (mpls | evpn | pseudowire)] [<label> ...]
#-----------------------------------------------
class ShowMplsRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show mpls route [ type TYPE ] [ LABELS ]"
   data = {
      'mpls': mplsNodeForShow,
      'route': _mplsHwRouteKwForShowMatcher,
      'type': _filterViaTypeKwMatcher,
      'TYPE': _tunnelTypeFilterValMatcher,
      'LABELS': _singleOrMultiLabelValMatcher,
   }
   cliModel = MplsModel.MplsHwStatus

   @staticmethod
   def handler( mode, args ):
      return showMplsHwRoute( mode, args.get( 'LABELS' ), args.get( 'TYPE' ) )

BasicCli.addShowCommandClass( ShowMplsRouteCmd )

def showMplsHwRouteSummary( mode, args ):
   showMplsConfigWarnings( mode )

   numUnprogrammedLabels = numLabels = numHwAdjs = numBackupAdjs = 0
   if mplsHwStatus:
      numLabels = 0
      l2Adjs = set()
      backupAdjs = set()
      for r in mplsHwStatus.route.values():
         adj = mplsHwStatus.adjacencyBase( r.adjBaseKey )
         if not adj:
            # If an adj does not exist for the route, then we ignore the route,
            # as is done in the other show commands.
            continue
         numLabels += 1
         if r.unprogrammed or adj.unprogrammed:
            numUnprogrammedLabels += 1
         if adj.l2Adjacency:
            l2Adjs.add( adj.l2Adjacency )
      numHwAdjs = len( l2Adjs )
      # there will not be any backup adjs for now, preserve the show commands
      # for backward compatibility
      numBackupAdjs = len( backupAdjs )

   numLabelsSet = set()
   numUnprogrammedLabelsSet = set()
   numHwAdjsSet = set()
   if decapLfib and decapLfibHw:
      numLabelsSet.update( r.topLabel for r in decapLfib.lfibRoute )
      numUnprogrammedLabelsSet.update( r.key.topLabel
            for r in decapLfibHw.unprogrammedRoute.values()
            if r.status == LfibHwProgrammingReturnCode.routeNotProgrammed or
               r.status == LfibHwProgrammingReturnCode.adjacencyNotProgrammed )
      numHwAdjsSet.update( decapLfib.lfibRoute[ r.key ].viaSetKey
            for r in decapLfibHw.unprogrammedRoute.values()
            if r.status == LfibHwProgrammingReturnCode.programmed )

   if evpnEthernetSegmentLfib and evpnEthernetSegmentLfibHw:
      numLabelsSet.update( r.topLabel for r in evpnEthernetSegmentLfib.lfibRoute )
      numUnprogrammedLabelsSet.update( r.key.topLabel
            for r in evpnEthernetSegmentLfibHw.unprogrammedRoute.values()
            if r.status == LfibHwProgrammingReturnCode.routeNotProgrammed or
               r.status == LfibHwProgrammingReturnCode.adjacencyNotProgrammed )
      numHwAdjsSet.update( evpnEthernetSegmentLfib.lfibRoute[ r.key ].viaSetKey
            for r in evpnEthernetSegmentLfibHw.unprogrammedRoute.values()
            if r.status == LfibHwProgrammingReturnCode.programmed )

   numLabels += len( numLabelsSet )
   numUnprogrammedLabels += len( numUnprogrammedLabelsSet )
   numHwAdjs += len( numHwAdjsSet )
   
   return MplsModel.MplsHwStatusSummary( numLabels=numLabels,
                                         numUnprogrammedLabels=numUnprogrammedLabels,
                                         numHwAdjs=numHwAdjs,
                                         numBackupAdjs=numBackupAdjs )

#------------------------------------------
# show mpls route summary
#------------------------------------------
class ShowMplsRouteSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show mpls route summary"
   data = {
      'mpls': mplsNodeForShow,
      'route': _mplsHwRouteKwForShowMatcher,
      'summary': "Show brief statistics of MPLS routes",
   }
   cliModel = MplsModel.MplsHwStatusSummary
   handler = showMplsHwRouteSummary

BasicCli.addShowCommandClass( ShowMplsRouteSummaryCmd )

def _showTechCmds():
   if not mplsHwCapability.mplsSupported:
      return []
   cmds = [ 'show mpls route summary',
            'show mpls route',
            'show mpls lfib route detail',
          ]
   return cmds

def _showTechSummaryCmds():
   if not mplsHwCapability.mplsSupported:
      return []
   return [ 'show mpls route summary' ]

# Timestamps are made up to maintain historical order within show tech-support
TechSupportCli.registerShowTechSupportCmdCallback( 
      '2013-11-22 11:35:37',
      _showTechCmds,
      summaryCmdCallback=_showTechSummaryCmds )

#-------------------------------------------------------------------------
# The "[no] mpls tunnel static <name> <endpoint> <nexthop> <interface>
#    label-stack <label-stack>" command.
#-------------------------------------------------------------------------
# Note: This token has been duplicated using the new parser above (as tunnelNode)
staticTunnelMatcherHelpStr = 'Static tunnel configuration'
staticTunnelMatcher = CliMatcher.KeywordMatcher(
   'static',
   helpdesc=staticTunnelMatcherHelpStr )
# XXX: Limit of name to 128 characters
tunnelNameValMatcher = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]{1,128}',
                                                  helpdesc='Name of static tunnel',
                                                  helpname='WORD' )

labelStackHelpStr = 'Specify the MPLS label(s) to push'
labelStackKeywordNode = CliMatcher.KeywordMatcher( 'label-stack',
                                                   helpdesc=labelStackHelpStr )
labelStackValNode = CliCommand.Node(
   matcher=CliMatcher.DynamicIntegerMatcher( topLabelRangeFn,
                                             helpdesc='Value of the MPLS label' ),
   maxMatches=getMaxMplsLabelStack() )
impNullTunnelNode = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                                   'imp-null-tunnel', helpdesc='' ),
                                     hidden=True )
endpointMatcher = CliMatcher.KeywordMatcher( 'endpoint',
      helpdesc='Match tunnel endpoint' )

invalidEndpointErrMsg = 'Invalid endpoint address: neither IPv4 nor IPv6.'
endpointNextHopAfErrMsg = ( 'Endpoint and nexthop addresses must belong to same'
                            ' address family.' )
def cliDisplayOrderLabelList( mplsLabelStack ):
   """Converts a BoundedMplsLabelStack to a list of labels (int) in the order that
   they should be shown in show commands.

   BoundedMplsLabelStack reserves label index 0 for the bottom of the stack, however
   the CLI generally displays labels with the outermost label on the wire
   (top of stack) on the left, so the list representation for that purpose is in
   reverse order when compared to BoundedMplsLabelStack.
   """
   return [ mplsLabelStack.label( i )
            for i in xrange( mplsLabelStack.stackSize - 1, -1, -1 ) ]

def labelOperation( labels ):
   mplsLabel = Tac.Value( 'Arnet::MplsLabelOperation' )
   mplsLabel.stackSize = len( labels )
   mplsLabel.operation = MplsLabelAction.push
   indexes = range( len( labels ) )
   for labelStackIndex, labelIndex in zip( reversed( indexes ), indexes ):
      mplsLabel.labelStackIs( labelStackIndex, labels[ labelIndex ] )

   return mplsLabel

def copyStaticTunnelConfigEntry( staticTunnelConfigEntry ):
   if staticTunnelConfigEntry is None:
      return None

   newConfigEntry = StaticTunnelConfigEntry( staticTunnelConfigEntry.name )
   newConfigEntry.tep = staticTunnelConfigEntry.tep
   # copy existing primary vias, if any
   for via in staticTunnelConfigEntry.via:
      newConfigEntry.via[ via ] = True
   # copy existing backup via
   newConfigEntry.backupVia = staticTunnelConfigEntry.backupVia
   newConfigEntry.inStaticTunnelMode = staticTunnelConfigEntry.inStaticTunnelMode

   return newConfigEntry

def validateLabelStackSize( mode, labels ):
   platformMaxLabels = getPlatformMaxMplsLabelStack()
   if mode.session.isInteractive() and len( labels ) > platformMaxLabels:
      errStr = ( 'The label stack depth exceeds the capability of the hardware.'
                 ' Maximum label stack depth supported: %d' % platformMaxLabels )
      mode.addError( errStr )
      return False
   return True

def getMplsTunnelNhTepAndIntfId( mode, nexthop, endpoint, intf ):
   nexthopAddr = Arnet.IpGenAddr( str( nexthop ) )
   endpointAddr = Arnet.IpGenPrefix( str( endpoint ) )
   if nexthopAddr.af != endpointAddr.af:
      mode.addError( endpointNextHopAfErrMsg )
      return None, None, None
   if endpointAddr.af == AddressFamily.ipv4:
      ipIntf = ipConfig.ipIntfConfig.get( intf.name )
   elif endpointAddr.af == AddressFamily.ipv6:
      ipIntf = ip6Config.intf.get( intf.name )
   else:
      mode.addError( invalidEndpointErrMsg )
      return None, None, None
   intfId = ipIntf.intfId if ipIntf else Tac.Value( "Arnet::IntfId",
                                                    intf.name )
   return nexthopAddr, endpointAddr, intfId

# Note: This command shares syntax with MplsDebugStaticTunnel in
# MplsDebugCli.py. If updating it here, please update the debug
# command accordingly.
class MplsStaticTunnel( CliCommand.CliCommandClass ):
   syntax = '''mpls tunnel static NAME
               ( ( ( TEP_V4_ADDR | TEP_V4_PREFIX ) NEXTHOP_V4 ) |
                 ( ( TEP_V6_ADDR | TEP_V6_PREFIX ) NEXTHOP_V6 ) )
               INTF ( ( label-stack { LABELS } ) | imp-null-tunnel )
            '''

   noOrDefaultSyntax = '''mpls tunnel static NAME ...'''

   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'static': staticTunnelMatcher,
      'NAME': tunnelNameValMatcher,
      'TEP_V4_ADDR': IpAddrMatcher.ipAddrMatcher,
      'TEP_V4_PREFIX': IpAddrMatcher.ipPrefixExpr(
         'IP address',
         "Subnet's mask value",
         'IP address with mask length',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
      ),
      'NEXTHOP_V4': IpAddrMatcher.ipAddrMatcher,
      # FIXME BUG200408: Un-hide IPv6 endpoints and nexthop in "mpls static tunnel"
      'TEP_V6_ADDR': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher,
                                      hidden=True ),
      'TEP_V6_PREFIX': CliCommand.Node( matcher=Ip6AddrMatcher.ip6PrefixMatcher,
                                        hidden=True ),
      'NEXTHOP_V6': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher,
                                     hidden=True ),
      'INTF': intfValMatcher,
      'label-stack': labelStackKeywordNode,
      'LABELS': labelStackValNode,
      'imp-null-tunnel': impNullTunnelNode,
   }

   adapter = labelsAdapter

   @staticmethod
   def handler( mode, args ):
      tunnelName = args[ 'NAME' ]
      endpoint = ( args.get( 'TEP_V4_ADDR' ) or args.get( 'TEP_V4_PREFIX' ) or
                   args.get( 'TEP_V6_ADDR' ) or args.get( 'TEP_V6_PREFIX' ) )
      nexthop = args.get( 'NEXTHOP_V4' ) or args.get( 'NEXTHOP_V6' )
      intf = args.get( 'INTF' )
      if 'imp-null-tunnel' in args:
         labels = [ 3 ]
      else:
         labels = args.get( 'LABELS' )
         if not validateLabelStackSize( mode, labels ):
            return

      nexthopAddr, tunnelEndpoint, intfId = getMplsTunnelNhTepAndIntfId( mode,
                                                                         nexthop,
                                                                         endpoint,
                                                                         intf )
      if ( nexthopAddr, tunnelEndpoint, intfId ) == ( None, None, None ):
         return

      staticTunnelConfigEntry = staticTunnelConfig.entry.get( tunnelName )
      if staticTunnelConfigEntry and staticTunnelConfigEntry.tep == tunnelEndpoint:
         newStaticTunnelConfigEntry = \
            copyStaticTunnelConfigEntry( staticTunnelConfigEntry )
      else:
         newStaticTunnelConfigEntry = StaticTunnelConfigEntry( tunnelName )
         newStaticTunnelConfigEntry.tep = tunnelEndpoint
      # add newly configured via
      newMplsVia = MplsVia( nexthopAddr, intfId, labelOperation( labels ) )
      # Clear existing vias if any, and add newly created via
      newStaticTunnelConfigEntry.via.clear()
      newStaticTunnelConfigEntry.via[ newMplsVia ] = True
      newStaticTunnelConfigEntry.inStaticTunnelMode = False

      if staticTunnelConfigEntry:
         # Check for duplicate and returns if it is the case
         if staticTunnelConfigEntry == newStaticTunnelConfigEntry:
            return
      staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tunnelName = args[ 'NAME' ]

      staticTunnelConfigEntry = staticTunnelConfig.entry.get( tunnelName )
      if staticTunnelConfigEntry:
         newStaticTunnelConfigEntry = \
            copyStaticTunnelConfigEntry( staticTunnelConfigEntry )
         newStaticTunnelConfigEntry.via.clear()
         # if neither primary via or backup via is still valid, then delete the
         # entry, else update it in the static tunnel config table
         if ( not newStaticTunnelConfigEntry.via and
              newStaticTunnelConfigEntry.backupVia == MplsVia() ):
            del staticTunnelConfig.entry[ tunnelName ]
         else:
            staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )

BasicCli.GlobalConfigMode.addCommandClass( MplsStaticTunnel )

#------------------------------------------
# [no] mpls tunnel static <name> <endpoint>
#     via <nexthop> <interface> label-stack <label-stack>
#     via <nexthop> <interface> label-stack <label-stack>
#     ...
#------------------------------------------

class TunnelStaticMode( TunnelStaticModeBase, BasicCli.ConfigModeBase ):
   name = "Mpls static tunnel configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, tunName, tep ):
      TunnelStaticModeBase.__init__( self, tunName, tep )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------
# mpls tunnel static <name> <endpoint>
# [ no | default ] behavior is handled by the MplsStaticTunnel class
#--------------------------------------------------------------------------
class MplsTunnelStaticConfig( CliCommand.CliCommandClass ):
   syntax = '''mpls tunnel static NAME
               ( ENDPOINTV4ADDR | ENDPOINTV4PREFIX |
                 ENDPOINTV6ADDR | ENDPOINTV6PREFIX )'''

   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'static': staticTunnelMatcher,
      'NAME': tunnelNameValMatcher,
      'ENDPOINTV4ADDR': IpAddrMatcher.ipAddrMatcher,
      'ENDPOINTV4PREFIX': IpAddrMatcher.ipPrefixExpr(
         'IP address',
         "Subnet's mask value",
         'IP address with mask length',
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
      ),
      # FIXME BUG200408: Un-hide IPv6 endpoints and nexthop in "mpls static tunnel"
      'ENDPOINTV6ADDR': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher,
                                         hidden=True ),
      'ENDPOINTV6PREFIX': CliCommand.Node( matcher=Ip6AddrMatcher.ip6PrefixMatcher,
                                           hidden=True ),
   }

   @staticmethod
   def handler( mode, args ):
      tunnelName = args[ 'NAME' ]
      endpoint = ( args.get( 'ENDPOINTV4ADDR' ) or args.get( 'ENDPOINTV4PREFIX' ) or
                   args.get( 'ENDPOINTV6ADDR' ) or args.get( 'ENDPOINTV6PREFIX' ) )

      tunnelEndpoint = Arnet.IpGenPrefix( str( endpoint ) )
      if ( tunnelEndpoint.af != AddressFamily.ipv4 and
           tunnelEndpoint.af != AddressFamily.ipv6 ):
         mode.addError( invalidEndpointErrMsg )
         return
      staticTunnelConfigEntry = staticTunnelConfig.entry.get( tunnelName )
      if staticTunnelConfigEntry:
         if staticTunnelConfigEntry.tep != tunnelEndpoint:
            errMsg = ( "Existing config for tunnel '{}' endpoint {}"
                     ).format( tunnelName, str( tunnelEndpoint ) )
            mode.addError( errMsg )
            return
         for via in staticTunnelConfigEntry.via:
            # If via.intfId is a FecIdIntfId, this is a resolving tunnel (hidden
            # argument), and not supported for ECMP
            if FecIdIntfId.isFecIdIntfId( via.intfId ):
               errMsg = ( "Existing resolving via config for tunnel '{}' endpoint {}"
                        ).format( tunnelName, str( tunnelEndpoint ) )
               mode.addError( errMsg )
               return
         newStaticTunnelConfigEntry = \
            copyStaticTunnelConfigEntry( staticTunnelConfigEntry )
      else:
         newStaticTunnelConfigEntry = \
            Tac.Value( 'Tunnel::Static::StaticTunnelConfigEntry', tunnelName )
         newStaticTunnelConfigEntry.tep = tunnelEndpoint
      newStaticTunnelConfigEntry.inStaticTunnelMode = True
      staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )
      childMode = mode.childMode( TunnelStaticMode, tunName=tunnelName,
                                  tep=tunnelEndpoint )
      mode.session_.gotoChildMode( childMode )

BasicCli.GlobalConfigMode.addCommandClass( MplsTunnelStaticConfig )

#--------------------------------------------------------------------------
# [ no | default ] via <nexthop> <interface> label-stack <label-stack>
#--------------------------------------------------------------------------

class MplsTunnelStaticConfigVia( CliCommand.CliCommandClass ):
   syntax = '''via ( NEXTHOPV4 | NEXTHOPV6 ) INTF
               ( ( label-stack { LABELS } ) | imp-null-tunnel )'''
   noOrDefaultSyntax = syntax
   data = {
      'via': 'MPLS via configuration',
      'NEXTHOPV4': IpAddrMatcher.ipAddrMatcher,
      'NEXTHOPV6': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher,
                                    hidden=True ),
      'INTF': intfValMatcher,
      'label-stack': labelStackKeywordNode,
      'LABELS': labelStackValNode,
      'imp-null-tunnel': impNullTunnelNode,
   }

   @staticmethod
   def _checkVia( tunName, staticTunnelConfigEntry, nexthopAddr=None, no=False ):
      if not no and not staticTunnelConfigEntry:
         errMsg = ( "Config for tunnel '{}' not found" ).format( tunName )
         return errMsg
      if nexthopAddr and nexthopAddr.af != staticTunnelConfigEntry.tep.af:
         errMsg = endpointNextHopAfErrMsg
         return errMsg
      if ( staticTunnelConfigEntry.tep.af != AddressFamily.ipv4 and
           staticTunnelConfigEntry.tep.af != AddressFamily.ipv6 ):
         errMsg = invalidEndpointErrMsg
         return errMsg
      return None

   @staticmethod
   def _makeVia( staticTunnelConfigEntry, nexthopAddr, intf, labels ):
      if staticTunnelConfigEntry.tep.af == AddressFamily.ipv4:
         ipIntf = ipConfig.ipIntfConfig.get( intf.name )
      else: # staticTunnelConfigEntry.tep.af == AddressFamily.ipv6
         ipIntf = ip6Config.intf.get( intf.name )
      intfId = ipIntf.intfId if ipIntf else Tac.Value( "Arnet::IntfId", intf.name )
      via = MplsVia( nexthopAddr, intfId, labelOperation( labels ) )
      return via

   @staticmethod
   def handler( mode, args ):
      nexthop = args.get( 'NEXTHOPV4' ) or args.get( 'NEXTHOPV6' )
      intf = args.get( 'INTF' )
      if 'imp-null-tunnel' in args:
         labels = [ 3 ]
      else:
         labels = args.get( 'LABELS' )
         if not validateLabelStackSize( mode, labels ):
            return
      nexthopAddr = Arnet.IpGenAddr( str( nexthop ) )
      staticTunnelConfigEntry = staticTunnelConfig.entry.get( mode.tunName )

      error = MplsTunnelStaticConfigVia._checkVia( mode.tunName,
                                                   staticTunnelConfigEntry,
                                                   nexthopAddr=nexthopAddr )
      if error:
         mode.addError( error )
         return

      newStaticTunnelConfigEntry = \
         copyStaticTunnelConfigEntry( staticTunnelConfigEntry )

      newMplsVia = MplsTunnelStaticConfigVia._makeVia( staticTunnelConfigEntry,
                                                       nexthopAddr, intf, labels )
      newStaticTunnelConfigEntry.via[ newMplsVia ] = True

      # Check for duplicate and returns if it is the case
      if staticTunnelConfigEntry == newStaticTunnelConfigEntry:
         return
      staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nexthop = args.get( 'NEXTHOPV4' ) or args.get( 'NEXTHOPV6' )
      intf = args.get( 'INTF' )
      if 'imp-null-tunnel' in args:
         labels = [ 3 ]
      else:
         labels = args.get( 'LABELS' )
         if not validateLabelStackSize( mode, labels ):
            return
      nexthopAddr = Arnet.IpGenAddr( str( nexthop ) )
      staticTunnelConfigEntry = staticTunnelConfig.entry.get( mode.tunName )

      error = MplsTunnelStaticConfigVia._checkVia( mode.tunName,
                                                   staticTunnelConfigEntry,
                                                   no=True )
      if error:
         mode.addError( error )
         return

      if staticTunnelConfigEntry:
         newStaticTunnelConfigEntry = \
            copyStaticTunnelConfigEntry( staticTunnelConfigEntry )
      else:
         return # entry already doesn't exist

      newMplsVia = MplsTunnelStaticConfigVia._makeVia( staticTunnelConfigEntry,
                                                       nexthopAddr, intf, labels )
      del newStaticTunnelConfigEntry.via[ newMplsVia ]
      # Check for duplicate and returns if it is the case
      if staticTunnelConfigEntry == newStaticTunnelConfigEntry:
         return
      staticTunnelConfig.entry.addMember( newStaticTunnelConfigEntry )

TunnelStaticMode.addCommandClass( MplsTunnelStaticConfigVia )

#-------------------------------------------------------------------------
# [no] mpls tunnel termination model ttl (pipe | uniform) dscp (pipe | uniform)
#-------------------------------------------------------------------------
_tunnelTerminationMatcher = CliCommand.guardedKeyword( 'termination',
                              helpdesc="Tunnel termination configuration commands",
                              guard=mplsTunnelTerminationGuard )
_tunnelTermModelMatcher = CliMatcher.KeywordMatcher(
   "model", helpdesc="tunnel termination ttl and dscp model" )
_ttlMatcher = CliMatcher.KeywordMatcher( 'ttl', helpdesc="model for ttl" )
_dscpMatcher = CliMatcher.KeywordMatcher( 'dscp', helpdesc="model for dscp" )

_pipeHelpdesc = "Preserve the inner value"

_ttlAndDscpTunnelModeValMatcher = CliMatcher.EnumMatcher( {
   'pipe': _pipeHelpdesc,
   'uniform': "Propagate value from outer header",
} )

class MplsTunnelTerminationModelCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls tunnel termination model ttl TTL_MODEL dscp DSCP_MODEL'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'termination': _tunnelTerminationMatcher,
      'model': _tunnelTermModelMatcher,
      'ttl': _ttlMatcher,
      'TTL_MODEL': _ttlAndDscpTunnelModeValMatcher,
      'dscp': _dscpMatcher,
      'DSCP_MODEL': _ttlAndDscpTunnelModeValMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      ttlModel = args[ 'TTL_MODEL' ]
      dscpModel = args[ 'DSCP_MODEL' ]

      if ttlModel != dscpModel and ( ttlModel == 'pipe'
            or not mplsHwCapability.mplsTtlUniformDscpPipeModelSupported ):
         errStr = "ttl model %s is not supported with dscp model %s" % \
                  ( ttlModel, dscpModel )
         mode.addError( errStr )
      mplsRoutingConfig.labelTerminationTtlMode = ttlModel
      mplsRoutingConfig.labelTerminationDscpMode = dscpModel
      mplsRoutingConfig.labelTerminationModeVersion += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mplsRoutingConfig.labelTerminationTtlMode = "uniform"
      mplsRoutingConfig.labelTerminationDscpMode = "uniform"
      mplsRoutingConfig.labelTerminationModeVersion += 1

configMode.addCommandClass( MplsTunnelTerminationModelCmd )

#-------------------------------------------------------------------------
# [no|default] mpls tunnel termination php model
#    ttl ( pipe | uniform | ) dscp ( pipe | uniform )
#-------------------------------------------------------------------------

_phpDscpTunnelModeValMatcher = CliCommand.Node(
   matcher=CliMatcher.EnumMatcher( {
      'pipe': _pipeHelpdesc,
      'uniform': "Propagate value from outer header",
   } ),
   guard=mplsPhpDscpUniformGuard )

class MplsTunnelTerminationPhpModelCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls tunnel termination php model ttl TTL_MODEL dscp DSCP_MODEL'
   noOrDefaultSyntax = 'mpls tunnel termination php model ...'
   data = {
      'mpls': mplsNodeForConfig,
      'tunnel': tunnelNode,
      'termination': _tunnelTerminationMatcher,
      'php': CliCommand.guardedKeyword( 'php',
         helpdesc="Penultimate hop popping configuration",
         guard=mplsPhpTtlGuard ),
      'model': _tunnelTermModelMatcher,
      'ttl': _ttlMatcher,
      'TTL_MODEL': _ttlAndDscpTunnelModeValMatcher,
      'dscp': _dscpMatcher,
      'DSCP_MODEL': _phpDscpTunnelModeValMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if CliCommand.isNoOrDefaultCmd( args ):
         mplsRoutingConfig.labelPhpTtlMode = TtlMode.undefinedTtlMode
         mplsRoutingConfig.labelPhpDscpMode = TtlMode.undefinedTtlMode
      else:
         mplsRoutingConfig.labelPhpTtlMode = args[ 'TTL_MODEL' ]
         mplsRoutingConfig.labelPhpDscpMode = args[ 'DSCP_MODEL' ]
      mplsRoutingConfig.labelPhpModeVersion += 1

   noOrDefaultHandler = handler

configMode.addCommandClass( MplsTunnelTerminationPhpModelCmd )

#-------------------------------------------------------------------------
# show mpls tunnel static
# [ ( index INDEX )
# | ( endpoint ( ADDR4 | PREFIX4 | ADDR6 | PREFIX6 ) )
# | ( name NAME ) ]
#-------------------------------------------------------------------------
tunnelAfterMplsMatcherForShow = CliMatcher.KeywordMatcher( 'tunnel',
      helpdesc="MPLS tunnel information and counters" )

def getStaticTunnelTableEntryModel( tunnelId, endpoint=None, name=None ):
   vias = []

   # fix once the capi model for StaticTunnelTableEntry is fixed
   # to support backup vias
   staticTunnelTableEntry = staticTunnelTable.entry.get( tunnelId )
   if staticTunnelTableEntry:
      if endpoint and staticTunnelTableEntry.tep != endpoint:
         return None
      if name and staticTunnelTableEntry.getName() != name:
         return None
      for via in staticTunnelTableEntry.via.itervalues():
         viaModels = getViaModels( via, useMplsVias=True )
         # The 'resolving-tunnel' hidden command uses MplsTunnelVia model, which is
         # not supported by MplsModel.StaticTunnelTableEntry, so skip those
         if viaModels:
            vias.extend( via for via in viaModels
                         if isinstance( via, TunnelModels.MplsVia ) )

      return MplsModel.StaticTunnelTableEntry(
         endpoint=staticTunnelTableEntry.tep, name=staticTunnelTableEntry.getName(),
         vias=vias )
   return None

def getStaticTunnelTable( args ):
   staticEntries = {}

   tunnelIds = staticTunnelTable.entry
   endpoint = None
   name = None
   if 'index' in args:
      # For each index query, generate a V4 index (44th bit set to zero) and a
      # V6 index (44th bit set to one) with getTunnelIdFromIndex. We'll combine
      # both indices into tunnelIds, and pass it along to the for loop below.
      # Note that indices are generated with a monotonically increasing
      # counter. So we should never have a case where we see both a V4 tunnel
      # and a V6 tunnel within the same  "show mpls tunnel static index
      # <tunnel-index>" command.
      tunnelV4Ids = [ getTunnelIdFromIndex( 'staticV4Tunnel',
                                            args[ 'INDEX' ] ) ]
      tunnelV6Ids = [ getTunnelIdFromIndex( 'staticV6Tunnel',
                                            args[ 'INDEX' ] ) ]
      tunnelIds = tunnelV4Ids + tunnelV6Ids
   elif 'name' in args:
      name = args[ 'NAME' ]
   elif 'endpoint' in args:
      endpoint = ( args.get( 'ADDR4' ) or
                   args.get( 'PREFIX4' ) or
                   args.get( 'ADDR6' ) or
                   args.get( 'PREFIX6' ) )
      endpoint = Arnet.IpGenPrefix( getattr( endpoint, 'stringValue', endpoint ) )

   for tunnelId in tunnelIds:
      staticEntryModel = getStaticTunnelTableEntryModel( tunnelId, endpoint, name )
      if staticEntryModel:
         staticEntries[ getTunnelIndexFromId( tunnelId ) ] = staticEntryModel

   return MplsModel.StaticTunnelTable( entries=staticEntries )

class ShowMplsTunnelStaticCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mpls tunnel static '
               '[ ( index INDEX ) '
               '| ( endpoint ( ADDR4 | PREFIX4 | ADDR6 | PREFIX6 ) ) '
               '| ( name NAME ) ]' )
   # FIXME BUG200408: Un-hide IPv6 endpoints and nexthop in "mpls static tunnel"
   data = {
      'mpls': mplsNodeForShow,
      'tunnel': tunnelAfterMplsMatcherForShow,
      'static': 'Static MPLS tunnel information',
      'index': 'Match tunnel index',
      'INDEX': tunnelIndexMatcher,
      'endpoint': endpointMatcher,
      'ADDR4': IpAddrMatcher.ipAddrMatcher,
      'PREFIX4': IpAddrMatcher.ipPrefixMatcher,
      'ADDR6': CliCommand.Node( matcher=Ip6AddrMatcher.ip6AddrMatcher,
                                hidden=True ),
      'PREFIX6': CliCommand.Node( matcher=Ip6AddrMatcher.ip6PrefixMatcher,
                                  hidden=True ),
      'name': 'Match tunnel name',
      'NAME': tunnelNameValMatcher,
   }
   cliModel = MplsModel.StaticTunnelTable

   @staticmethod
   def handler( mode, args ):
      return getStaticTunnelTable( args=args )

BasicCli.addShowCommandClass( ShowMplsTunnelStaticCmd )

#-------------------------------------------------------------------------
# The "show mpls tunnel fib" command
#-------------------------------------------------------------------------
showMplsTunnelFibDeprecatedWarning = (
    "'show mpls tunnel fib' has been deprecated. Please use "
    "'show tunnel fib [options]' moving forward." )

def getMplsTunnelFibEntryModel( tunnelId ):
   tunnelType = getTunnelTypeFromTunnelId( tunnelId )
   if tunnelType in showTunnelFibIgnoredTunnelTypes:
      # Do not show ignored entry types in 'show mpls tunnel fib'
      return None

   # NOTE: The tunnel fib entry is now obtained from the new refactored
   # TunnelFib and not MplsTunnelFib. To defer changing the Mpls CAPI
   # models, we convert the new TunnelFib entry data into the current
   # CAPI model that represents the old MplsTunnelFib. At some point in
   # the future, the CAPI model could be upgraded (again).
   vias = []
   tunnelFibEntry = tunnelFib.entry.get( tunnelId )
   if tunnelFibEntry:
      endpoint = getEndpointFromTunnelId( tunnelId )
      tunnelViaStatus = getTunnelViaStatusFromTunnelId( tunnelId )
      for tunVia in tunnelFibEntry.tunnelVia.itervalues():
         mplsViaModels = getMplsViaModelFromTunnelVia( tunVia )
         for mplsViaModel in mplsViaModels:
            vias.append( mplsViaModel )
      return MplsModel.MplsTunnelFibEntry(
            tunnelIndex=getTunnelIndexFromId( tunnelId ), 
            tunnelType=TunnelModels.tunnelTypeStrDict[ tunnelType ],
            endpoint=endpoint, vias=vias, tunnelViaStatus=tunnelViaStatus )
   return None

class ShowMplsTunnelFib( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls tunnel fib'
   data = {
            'mpls': mplsForShowNode,
            'tunnel': tunnelKwMatcherMplsShow,
            'fib': 'Tunnel forwarding information'
          }
   cliModel = MplsModel.MplsTunnelFib
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.addWarning( showMplsTunnelFibDeprecatedWarning )
      mplsTunnelFibEntries = []

      for tunnelId in tunnelFib.entry:
         mplsTunnelFibEntryModel = getMplsTunnelFibEntryModel( tunnelId=tunnelId )
         if mplsTunnelFibEntryModel:
            mplsTunnelFibEntries.append( mplsTunnelFibEntryModel )

      # Sort entries by address family, endpoint, index
      mplsTunnelFibEntries.sort( key=lambda entry:
         ( entry.endpoint.af if entry.endpoint else 0,
           entry.endpoint.address if entry.endpoint else 0,
           entry.tunnelIndex ) )
      return MplsModel.MplsTunnelFib( entries=mplsTunnelFibEntries )

BasicCli.addShowCommandClass( ShowMplsTunnelFib )

#------------------------------------------
# show mpls label ranges
#------------------------------------------
class ShowMplsLabelRangeCommand( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls label ranges'
   data = {
      'mpls': mplsForShowNode,
      'label': 'Show static and dynamically allocated labels',
      'ranges': 'Show static and dynamically allocated label ranges',
   }
   cliModel = MplsModel.MplsLabelRanges

   @staticmethod
   def handler( mode, args ):
      static, dynamicBlock = ShowMplsLabelRangeCommand._getStaticRanges()
      dynamic = ShowMplsLabelRangeCommand._getDynamicRanges()
      static = [ item for item in static if item.blockSize > 0 ]
      # 1. Unassigned ranges: Ranges left after combining operational and
      # conflicting label-ranges ( both static and dynamic blocks ).
      # 2. Unused dynamic ranges: Free ranges within the dynamic block.
      # Note that unused dynamic ranges make sense only if the dynamic
      # block is operationally active. Unassigned ranges are by nature
      # conflict-free and appear in operational ranges available for
      # assignments.
      unusedStatic = ShowMplsLabelRangeCommand._getUnusedRanges(
            static + [ dynamicBlock ],
            0,
            MplsLabel.max,
            LABEL_RANGE_STATIC )
      unusedDynamic = ShowMplsLabelRangeCommand._getUnusedRanges(
            dynamic, dynamicBlock.blockStart,
            dynamicBlock.blockEnd(),
            LABEL_RANGE_DYNAMIC )

      allRanges = static + unusedStatic + dynamic + unusedDynamic
      allRanges = ShowMplsLabelRangeCommand._sortedRanges( allRanges )
      return MplsModel.MplsLabelRanges( ranges=allRanges )

   @staticmethod
   def _getDynamicRanges():
      rangeList = []
      for name, lba in mplsStatus.labelBlockAllocation.iteritems():
         for rangeVal in lba.block.itervalues():
            rangeList.append( MplsModel.MplsLabelRange(
                  protocol=name,
                  rangeType=LABEL_RANGE_DYNAMIC,
                  blockSize=rangeVal.blockSize,
                  blockStart=rangeVal.blockStart ) )
      return rangeList

   @staticmethod
   def _getCliProtocolName( rangeType ):
      if rangeType == LabelRangeInfo.rangeTypeL2evpnSharedEs:
         return "l2evpn shared ethernet-segment"
      else:
         return rangeType

   @staticmethod
   def _getStaticRanges():
      rangeList = []
      # MPLS Standard
      rangeList.append( MplsModel.MplsLabelRange(
         protocol='reserved',
         rangeType=LABEL_RANGE_STATIC,
         blockSize=MplsLabel.unassignedMin - MplsLabel.min,
         blockStart=MplsLabel.min ) )

      dynamicBlock = None
      labelRangeEntity = mplsRoutingConfig if not Toggles.MplsToggleLib.\
                         toggleLabelRangeStatusEnabled() else \
                         mplsRoutingStatus
      for name, lr in labelRangeEntity.labelRange.iteritems():
         protocol = ShowMplsLabelRangeCommand._getCliProtocolName( name )
         rangeVal = MplsModel.MplsLabelRange(
            protocol=protocol,
            rangeType=LABEL_RANGE_STATIC,
            blockSize=lr.size,
            blockStart=lr.base )

         if rangeVal.protocol == LabelRangeInfo.rangeTypeDynamic:
            dynamicBlock = rangeVal
         else:
            rangeList.append( rangeVal )
      return ( rangeList, dynamicBlock )

   @staticmethod
   def _getUnusedRanges( labelRanges, start, end, typ ):
      labelRanges = ShowMplsLabelRangeCommand._sortedRanges( labelRanges )
      mergedRanges = []
      for labelRange in labelRanges:
         rangeBounds = ( labelRange.blockStart, labelRange.blockEnd() )
         l = len( mergedRanges )
         if l == 0 or mergedRanges[ l - 1 ][ 1 ] < rangeBounds[ 0 ]:
            # The range is distinct from the known ranges
            mergedRanges.append( rangeBounds )
         else:
            # The range overlaps the last range, and only the last range
            mergedRanges[ l - 1 ] = ( mergedRanges[ l - 1 ][ 0 ],
                  max( mergedRanges[ l - 1 ][ 1 ], rangeBounds[ 1 ] ) )


      unusedRanges = []
      lastEnd = start - 1
      for r in mergedRanges:
         # Will be true except when first range is at start
         if r[ 0 ] > lastEnd + 1:
            unusedRanges.append(
                  ShowMplsLabelRangeCommand._createUnusedRange(
                     lastEnd + 1, r[ 0 ] - 1, typ ) )
         lastEnd = r[ 1 ]

      if lastEnd < end:
         unusedRanges.append( ShowMplsLabelRangeCommand._createUnusedRange(
            lastEnd + 1, end, typ ) )

      return unusedRanges

   @staticmethod
   def _sortedRanges( labelRanges ):
      return sorted( labelRanges, key=lambda r: r.blockStart )

   @staticmethod
   def _createUnusedRange( start, end, typ ):
      return MplsModel.MplsLabelRange(
            protocol=None,
            rangeType=typ if typ != LABEL_RANGE_STATIC else LABEL_RANGE_UNASSIGNED,
            blockSize=end - start + 1,
            blockStart=start )

BasicCli.addShowCommandClass( ShowMplsLabelRangeCommand )

#-----------------------------------------------------------------------------------
# show mpls hardware ale adj [ ( ( LABEL [ { LABELS } ] ) | pending ) ]
#-----------------------------------------------------------------------------------
class ShowMplsFecRequestCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls hardware ale adj [ LABELS | pending ]'
   data = {
      'mpls': mplsForShowNode,
      'hardware': 'Show MPLS hardware command',
      'ale': 'Show MPLS hardware ale command',
      'adj': 'Show MPLS ale adjacency command',
      'LABELS': _singleOrMultiLabelValMatcher,
      'pending': ( 'Show MPLS routes whose ale adjacency are not programmed or in '
                   'sync with hardware' ),
   }
   cliModel = MplsModel.MplsFecRequestTable

   @staticmethod
   def handler( mode, args ):
      labels = args.get( 'LABELS' )
      pending = 'pending' in args
      showMplsConfigWarnings( mode )

      LazyMount.force( mplsHwStatus )
      if not mplsHwStatus:
         return MplsModel.MplsFecRequestTable( entries=[] )

      routes = {}
      if labels:
         labelStack = Arnet.MplsLib.labelListToBoundedMplsLabelStack( labels )
         routeKeys = [ RouteKey( labelStack ) ]
      else:
         routeKeys = mplsHwStatus.route.keys()

      # for each route, create the table entry
      for rk in routeKeys:
         route = mplsHwStatus.route.get( rk )
         if route is None:
            continue
         labels = None
         if rk.labelStack.stackSize < 2:
            labels = str( rk.topLabel )
         else:
            labels = rk.labelStack.cliShowString()
         requestedFecId = route.requestedFecId
         installedFecId = route.installedFecId
         programStatus = 'synced' if requestedFecId == installedFecId else 'pending'
         if pending and programStatus == 'synced':
            continue
         entry = MplsModel.MplsFecRequestTableEntry(
                                             requestedFecId=requestedFecId,
                                             installedFecId=installedFecId,
                                             programStatus=programStatus )
         routes[ labels ] = entry
      return MplsModel.MplsFecRequestTable( routes=routes )

BasicCli.addShowCommandClass( ShowMplsFecRequestCmd )

#--------------------------------------------------------------------------------
# show tech-support mpls graceful-restart
#--------------------------------------------------------------------------------
class ShowTechMplsGr( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show tech-support mpls graceful-restart' )
   data = {
      'tech-support': TechSupportCli.techSupportKwMatcher,
      'mpls': "Show aggregated status details for Mpls",
      'graceful-restart': "Graceful restart information for Mpls agent",
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      # hardcoding 'Mpls' to avoid importing MplsAgent.name
      AgentCommandRequest.runSocketCommand( em, 'Mpls', 'debug', 'DUMP_GR_STATE' )

BasicCli.addShowCommandClass( ShowTechMplsGr )

# register it to show tech support
def _showTechMplsHwAleAdjCmds():
   if not mplsHwCapability.mplsSupported:
      return []
   return [
            'show mpls hardware ale adj',
          ]
TechSupportCli.registerShowTechSupportCmdCallback( '2019-09-27 14:32:19',
                                                   _showTechMplsHwAleAdjCmds )
#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global em
   global cliChurnTestHelper
   global cliMplsRouteConfig
   global cliMplsRouteConfigReadOnly
   global decapLfib
   global decapLfibConsumer
   global decapLfibHw
   global evpnEthernetSegmentLfib
   global evpnEthernetSegmentLfibHw
   global ip6Config
   global ipConfig
   global l3Config
   global l3ConfigDir
   global l3NHResolver
   global lfibInfo
   global mldpOpaqueValueTable
   global gribiAfts
   global mplsHwCapability
   global mplsHwStatus
   global mplsRouteConfigDir
   global mplsRoutingConfig
   global mplsTunnelConfig
   global mplsStatus
   global mplsRoutingStatus
   global mplsVrfLabelConfig
   global pseudowireLfib
   global pwRouteHelper
   global routingHardwareStatus
   global routing6VrfInfoDir
   global routingVrfInfoDir
   global srTePolicyStatus
   global srteForwardingStatus
   global staticMcastLfib
   global staticTunnelConfig
   global staticTunnelTable
   global transitLfib
   global tunnelFib
   global vrfGlobalTunPref

   def doMountsComplete():
      global forwardingStatus
      global forwarding6Status
      global unifiedForwardingStatus
      global unifiedForwarding6Status
      unifiedForwardingStatus = smashEm.doMount( "forwarding/unifiedStatus",
                                                 "Smash::Fib::ForwardingStatus",
                                                 readerInfo )
      unifiedForwarding6Status = smashEm.doMount( "forwarding6/unifiedStatus",
                                                  "Smash::Fib6::ForwardingStatus",
                                                  readerInfo )
      forwardingStatus = smashEm.doMount( "forwarding/status",
                                          "Smash::Fib::ForwardingStatus",
                                          readerInfo )
      forwarding6Status = smashEm.doMount( "forwarding6/status",
                                           "Smash::Fib6::ForwardingStatus",
                                           readerInfo )
      initFecModeSm()
      initMplsRouteConfigMergeSm( entityManager )

   em = entityManager

   mplsRoutingConfig = ConfigMount.mount( entityManager, "routing/mpls/config",
                                          "Mpls::Config", "w" )
   cliMplsRouteConfig = ConfigMount.mount( entityManager,
                                           "routing/mpls/route/input/cli",
                                           "Mpls::RouteConfigInput", "wi" )
   mplsVrfLabelConfig = ConfigMount.mount( entityManager,
                                           "routing/mpls/vrfLabel/input/cli",
                                           "Mpls::VrfLabelConfigInput", "wi" )
   mplsTunnelConfig = ConfigMount.mount( entityManager,
                                         "routing/mpls/tunnel/config",
                                         "Tunnel::MplsTunnelConfig", "w" )
   lfibInfo = LazyMount.mount( entityManager, "routing/mpls/lfibInfo",
                               "Mpls::LfibInfo", "r" )
   mplsHwStatus = LazyMount.mount( entityManager, "routing/hardware/mpls/status",
                                   "Mpls::Hardware::Status", "rS" )
   mplsHwCapability = LazyMount.mount( entityManager,
                                       "routing/hardware/mpls/capability",
                                       "Mpls::Hardware::Capability",
                                       "r" )
   mplsStatus = LazyMount.mount( entityManager, "mpls/status",
                                 "Mpls::Api::Status", "rS" )
   mplsRoutingStatus = LazyMount.mount( entityManager, "routing/mpls/status",
                                        "Mpls::Status", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   routingVrfInfoDir = LazyMount.mount( entityManager,
                                    "routing/vrf/routingInfo/status",
                                    "Tac::Dir", "ri" )
   routing6VrfInfoDir = LazyMount.mount( entityManager,
                                    "routing6/vrf/routingInfo/status",
                                    "Tac::Dir", "ri" )
   ipConfig = ConfigMount.mount( entityManager, "ip/config", "Ip::Config", "w" )
   ip6Config = ConfigMount.mount( entityManager, "ip6/config", "Ip6::Config", "w" )
   l3ConfigDir = ConfigMount.mount( entityManager, "l3/intf/config",
                                    "L3::Intf::ConfigDir", "w" )
   gribiAfts = LazyMount.mount( entityManager, "routing/gribi/afts",
                                 "Gribi::GribiAfts", "r" )
   staticTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.staticTunnelTable, entityManager )
   staticTunnelConfig = ConfigMount.mount( entityManager, "tunnel/static/config",
                                          "Tunnel::Static::Config", "w" )
   staticMcastLfib = ConfigMount.mount( entityManager, 'mpls/staticMcast/lfib',
                                  'Mpls::LfibSysdbStatus', 'w' )
   IntfCli.Intf.registerDependentClass( IntfMplsConfigCleaner, 20 )
   cliChurnTestHelper = LazyMount.mount( entityManager,
                                         "routing/mpls/cli/churnHelper",
                                         "Mpls::CliChurnTestHelper", "w" )

   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   keyshadowInfo = Smash.mountInfo( 'keyshadow' )
   readerInfo = Smash.mountInfo( 'reader' )
   srTePolicyStatus = smashEm.doMount( srTePolicyStatusPath(),
                                       'SrTePolicy::PolicyStatus', readerInfo )
   tunnelFib = smashEm.doMount( 'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib',
                                readerInfo )
   pseudowireLfib = smashEm.doMount( 'mpls/pseudowireLfib', 'Mpls::LfibStatus',
                                     readerInfo )
   transitLfib = smashEm.doMount( 'mpls/transitLfib', 'Mpls::LfibStatus',
                                  readerInfo )
   decapLfib = smashEm.doMount( "mpls/decapLfib", "Mpls::LfibStatus", keyshadowInfo )
   decapLfibHw = smashEm.doMount( "hardware/mpls/decapLfib", "Mpls::LfibHwStatus",
                                  readerInfo )
   evpnEthernetSegmentLfib = smashEm.doMount( "mpls/evpnEthernetSegmentFilterLfib",
                                              "Mpls::LfibStatus", readerInfo )
   evpnEthernetSegmentLfibHw = smashEm.doMount(
                                    "hardware/mpls/evpnEthernetSegmentFilterLfib",
                                    "Mpls::LfibHwStatus", readerInfo )
   srteForwardingStatus = smashEm.doMount( "forwarding/srte/status",
                                       "Smash::Fib::ForwardingStatus", readerInfo )
   decapLfibConsumer = Tac.newInstance( 'Mpls::LfibStatusConsumerSm', decapLfib )
   mldpOpaqueValueTable = LazyMount.mount( entityManager,
                                           "mpls/ldp/mldpOpaqueValueTable",
                                           "Mpls::MldpOpaqueValueTable", "r" )
   # We need to use a mount group because:
   # 1) we need to defer initMplsRouteConfigMergeSm() until the mounts are mounted,
   # 2) we need to access a config mounted entity
   #
   # We do not use the ConfigMount instance of the config mounted entity since we
   # just need read access to the entity at the specified path for the merge sm.
   # The flag needs to be "w" even though it's read-only so avoid conflicting
   # permissions with ConfigMount
   mg = entityManager.mountGroup()
   l3Config = mg.mount( "l3/config", "L3::Config", "ri" )
   mplsRouteConfigDir = mg.mount( 'routing/mpls/route/input/dynamic',
                                  'Tac::Dir', 'ri' )
   cliMplsRouteConfigReadOnly = mg.mount( 'routing/mpls/route/input/cli',
                                          'Mpls::RouteConfigInput', 'w' )
   l3NHResolverPath = Tac.Type( 'Mpls::RouteConfigMergeSm' ).l3NHResolverPath()
   l3NHResolver = LazyMount.mount( entityManager, l3NHResolverPath,
                                   'Routing::NexthopStatus', 'r' )
   mg.close( doMountsComplete )

   ctx = Tac.newInstance( "Mpls::PluginContext" )
   ctx.entityManager = entityManager.cEntityManager()
   pluginLoader = Tac.newInstance( "Mpls::PluginLoader" )
   pluginLoader.loadPlugins( "MplsCliPlugin", ctx, "", "" )
   pwRouteHelper = ctx.pwRouteHelper

#--------------------------------------------------------------------------
# register show tech-support extended evpn
#--------------------------------------------------------------------------
def _showTechEvpnCmds():
   return [
            'show mpls route',
            'show mpls label range',
          ]

TechSupportCli.registerShowTechSupportCmdCallback( '2017-11-03 12:06:08',
                                                   _showTechEvpnCmds,
                                                   extended = 'evpn' )
