# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import FruCli
import IntfCli

""" This module extends the interface cli commands defined in
Arnet/CliPlugin/IntfCli with the following module-specific interface
commands:
   - 'show interfaces module <path>' command, in enable mode
   - 'show interfaces module <path> description' command, in enable mode 
These commands are only installed on modular systems.
"""
moduleKw = CliMatcher.KeywordMatcher(
   'module',
   helpdesc='Limit display to interfaces on module' )

def modValueFunc( mode, slotDesc ):
   # slotDesc is a tuple of slot object, tag, and label. 
   # convert a slot to a customer-visible name of the card (i.e., 1, 3 or Fabric1 )
   slot = slotDesc.slot
   if slot is None:
      return None
   else:
      cardSlot = slot.parent.defaultChildTags.get( 'CardSlot' )
      if cardSlot and ( slot.tag in cardSlot.tag.values() ):
         return int( slot.label )
      else:
         return "%s%s" % ( slot.tag, slot.label )

class ModuleExpressionFactory( CliCommand.CliExpressionFactory ):
   def __init__( self ):
      self.slotExpr_ = FruCli.SlotExpressionFactory()
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      moduleName = name + "_module"
      slotName = name + "SLOT"
      class ModuleExpression( CliCommand.CliExpression ):
         expression = "%s %s" % ( moduleName, slotName )
         data = {
            moduleName : CliCommand.Node( moduleKw,
                                          guard=FruCli.modularSystemGuard ),
            slotName : self.slotExpr_
            }
         @staticmethod
         def adapter( mode, args, argsList ):
            slotDesc = args.pop( slotName, None )
            if slotDesc is not None:
               if isinstance( slotDesc, list ):
                  args[ name ] = [ modValueFunc( mode, i ) for i in slotDesc ]
               else:
                  args[ name ] = modValueFunc( mode, slotDesc )
      return ModuleExpression

IntfCli.registerModularExpression( ModuleExpressionFactory() )

