# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliPlugin.FruCli as FruCli
import CliMatcher
import LazyMount
import ModuleModels
import MultiRangeRule
import ShowCommand
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "ModuleCli" )
t0 = Tracing.trace0

cardStatus = None
entityMib = None

# ------------------------------------------------------
# The "show module" command, in "enable" mode
#    show module [ MODNAME | all ]
#-------------------------------------------------------

def _getNumMacAddrs( startAddr, endAddr ):
   # Return the number of MAC addresses in the given ( startAddr, endAddr ) range
   startAddrInt = int( startAddr.replace( ":", "" ), 16 )
   endAddrInt = int( endAddr.replace( ":", "" ), 16 )
   return int( endAddrInt - startAddrInt + 1 )

def _slotName( slot ):
   # slot is an EntityMib::CardSlot object
   defaultTags = slot.parent.defaultChildTags.get( 'CardSlot', {} )
   if defaultTags.tag.has_key( slot.tag ):
      return slot.label
   else:
      return "%s%s" % ( slot.tag, slot.label )

def _getAllSlots( mode ):
   slots = []
   entityMibRoot = entityMib.root
   if entityMibRoot is not None and entityMibRoot.initStatus == "ok":
      return entityMibRoot.cardSlot.values()
   return slots

# Dictionary of functions that return a status string and uptime
# for the card, given the card entity mib and cli mode,
# keyed by card tag.
_cardStatusHook = {}
_cardUptimeHook = {}

def registerCardStatusHook( cardTag, hook ):
   _cardStatusHook[ cardTag ] = hook

def registerCardUptimeHook( cardTag, hook ):
   _cardUptimeHook[ cardTag ] = hook

_stateToStatusEnumMap = {
      "ok" : "unknown",
      "unknown" : "unknown",
      "booting" : "poweringOn",
      "running" : "ok",
      "powerFailed" : "failed",
      "poweredOff" : "poweredOff",
      "poweringOn" : "poweringOn",
      "unknownInitStatus" : "unknown",
      "partialFailure" : "unknown",
      "epochRebootRequired" : "disabledUntilReboot",
      "epochUpgradeRequired" : "disabledUntilSystemUpgrade",
      "active" : "active",
      "standby" : "standby",
      "disabled" : "disabled",
      "upgradingFpga" : "upgradingFpga",
}

def getCardStatus( mode, card ):
   if card.tag in _cardStatusHook:
      state = _cardStatusHook[ card.tag ]( mode, card )
      ret = _stateToStatusEnumMap.get( state )
      if not ret:
         assert False, "Unknown cardState value %s" % state
   else:
      status = cardStatus
      cardName = "%s%s" % ( card.tag, card.label )
      # Figure out the status of the card. There are several possibilities:
      hwCardStatus = status.cardStatus.get( cardName )
      if not hwCardStatus:
         initStatus = card.initStatus
         ret = _stateToStatusEnumMap.get( initStatus )
         if not ret:
            assert False, "Unknown initStatus value %s" % initStatus
      else:
         state = hwCardStatus.state
         ret = _stateToStatusEnumMap.get( state )
         if not ret:
            assert False, "Unknown cardState value %s" % state
   return ret

def getCardUptime( mode, card ):
   if card.tag in _cardUptimeHook:
      return _cardUptimeHook[ card.tag ]( mode, card )
   else:
      if getCardStatus( mode, card ) != 'ok':
         return None

      status = cardStatus
      cardName = "%s%s" % ( card.tag, card.label )
      hwCardStatus = status.cardStatus.get( cardName )

      if not hwCardStatus:
         return None 
      uptime = Tac.now() - hwCardStatus.lastCardStateChangeTime
      return Tac.utcNow() - uptime

def showModule( mode, args ):
   slotDesc = args.get( 'MODNAME' )
   redundancyStatus = mode.session_.entityManager_.redundancyStatus()
   modules = ModuleModels.Modules()
   # "show module" cannot be ran in "standby rpr" or "disabled simplex" as in those
   # situations Sysdb does not have the required entities mounted that provide
   # the requested information.
   modules.redundancyMode = redundancyStatus.mode
   modules.redundancyProtocol = redundancyStatus.protocol
   
   # slots is the list of EntityMib::CardSlot objects
   if slotDesc is None:
      slots = _getAllSlots( mode )
   else:
      slots = [ slotDesc.slot ] if slotDesc.slot else []
   t0( 'slots are', slots )

   for slot in slots:
      card = slot.card
      if card:
         m = ModuleModels.Module()
         m.portCount = len( card.port.keys() )
         m.typeDescription = card.description
         m.modelName = card.modelName
         m.serialNumber = card.serialNum
         m.minMacAddress = card.minAddr
         m.maxMacAddress = card.maxAddr
         m.numMacAddresses = _getNumMacAddrs( card.minAddr, card.maxAddr )
         m.hardwareRev = card.hardwareRev
         m.softwareRev = card.softwareRev.split( '-' )[ 0 ]
         m.status = getCardStatus( mode, card )        
         m.uptime = getCardUptime( mode, card )
         modules.modules[ _slotName( slot ) ] = m

   return modules

moduleMatcher = CliMatcher.KeywordMatcher( 'module',
                                           helpdesc='Limit display to a module' )

moduleNode = CliCommand.Node( moduleMatcher, guard=FruCli.modularSystemGuard )

class ShowModuleCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show module [ all | MODNAME ]'
   data = { 'module': moduleNode,
            'all': 'Display all module status and information', 
            'MODNAME': FruCli.SlotExpressionFactory(),
          }
   cliModel = ModuleModels.Modules
   handler = showModule
BasicCli.addShowCommandClass( ShowModuleCmd )

# Register 'show module' with 'show tech support'
def _registerShowModuleInShowTechSupport():
   import CliPlugin.TechSupportCli
   CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      '2010-06-11 00:00:00',
      lambda: [ "show module" ],
      summaryCmdCallback=lambda : [ "show module" ] )

FruCli.registerModularSystemCallback( _registerShowModuleInShowTechSupport )

# ------------------------------------------------------
# The "power module" command, in "config" mode
#
# The full syntax of this command is:
#     power enable module <module path>
#     no power enable module <module path>
#-------------------------------------------------------
# { key: ( tagLong, helpDesc ) }
cardTypes = { 'NUMS': ( '', 'Linecard and Supervisor numbers' ), 
              'LINECARD': ( 'Linecard', 'Linecards' ),
              'SUP': ( 'Supervisor', 'Supervisors' ),
              'FABRIC': ( 'Fabric', 'Fabric Cards' ),
              'SWITCHCARD': ( 'Switchcard', 'Switchcards' ),
            } 
def handleCardPower( mode, args, powerOn ):
   cardList = None
   for cardType, cardInfo in cardTypes.items():
      if cardType in args:
         cardList = args.get( cardType )
         assert isinstance( cardList, MultiRangeRule.GenericRangeList )
         cardList = FruCli.getCardsFromIdList( mode, 
                                               cardList.values(),
                                               cardInfo[ 0 ] )
         for card in cardList:
            if powerOn:
               card.powerOn()
            else:
               card.powerOff()
         break

def doNoPowerEnable( mode, args ):
   handleCardPower( mode, args, powerOn=False )
   
def doPowerEnable( mode, args ):
   handleCardPower( mode, args, powerOn=True )

class PowerModuleExpression( CliCommand.CliExpression ):
   expression = ' | '.join( cardTypes.keys() )
   data = {}
   for _cardType, _cardInfo in cardTypes.items():
      # pylint: disable=cell-var-from-loop
      _matcher =  MultiRangeRule.MultiRangeMatcher(
                     noSingletons=False,
                     helpdesc=_cardInfo[ 1 ],
                     tagLong=_cardInfo[ 0 ],
                     rangeFn=lambda x=_cardInfo[ 0 ]: FruCli.rangeFn( x ),
                  )
      _node = CliCommand.Node( _matcher, 
                               guard=FruCli.cardTypeOrModularRprActiveSupeGuard )
      data[ _cardType ] = _node 

moduleMatcher = CliMatcher.KeywordMatcher( 'module',
                                           helpdesc='Configure module power' )
powerModuleNode = CliCommand.Node( moduleMatcher,
                                   guard=FruCli.modularSystemGuard )

class PowerModuleCmd( CliCommand.CliCommandClass ):
   syntax = 'power enable module CARDLIST'
   noOrDefaultSyntax = 'power enable module CARDLIST'
   data = { 'power': 'Configure power supplies',
            'enable': 'Configure power',
            'module': powerModuleNode,
            'CARDLIST': PowerModuleExpression 
   }
   handler = doPowerEnable
   noHandler = doNoPowerEnable
   defaultHandler = doPowerEnable

BasicCli.GlobalConfigMode.addCommandClass( PowerModuleCmd )

def Plugin( entityManager ):
   global cardStatus, entityMib
   cardStatus = LazyMount.mount( entityManager, "hardware/card/status",
                                 "Hardware::Card::Status", "r" )
   entityMib = LazyMount.mount( entityManager, "hardware/entmib",
                                "EntityMib::Status", "r" )
