# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import copy
import Arnet
import BasicCli
import CliParser
import CliCommand
import CliMatcher
import LazyMount
from VlanCli import VlanConfigMode, Vlan, vlanSetMatcher
from CliMode.MldSnooping import MldSnoopingMode, MldSnoopingVlanMode
from CliToken.Clear import clearKwNode
import IntfCli
import ConfigMount
import FileUrl
import Url
import Tac
import ShowCommand
import Ip6AddrMatcher
import IgmpSnoopingLib
from IgmpSnoopingCli import doShowGmpSnoopingGroupsVlansHelper, igmpVersionTable
from IgmpSnoopingCli import doShowGmpSnoopingMrouter, IgmpSnoopingMrouter
from IgmpSnoopingCli import memberMatcher, interfaceMatcher, vlanIdMatcher
from IgmpSnoopingCli import setSnoopingVlanMrouter
from IgmpSnoopingCli import clearSnoopingCounters, syncCounterCheckpointDir
from IgmpSnoopingCli import intfRangeConfigMatcher, doSetSnoopingVlanMember
from IgmpSnoopingCli import showClientSnoopingsGroupsHandler
import IgmpSnoopingCliLib
from IgmpSnoopingModel import GmpMembershipInfo, GmpMembershipDetailInfo
from IgmpSnoopingModel import GmpMembershipCountInfo
# Add explicit dependency to load MlagIgmpSnoopingCli plugin. MlagIgmpSnoopingCli
# adds an IgmpSnoopingCliHelper extension to provide apis required to find peer-link
# intf
import MlagIgmpSnoopingCli # pylint: disable=unused-import
from MldSnoopingModel import MldSnoopingCounters, MldSnoopingCountersInterface, \
                             MldSnoopingQuerier, MldSnoopingVlanQuerier, \
                             MldSnoopingInfo, MldSnoopingVlanInfo
from MldSnooping import agentName
import Toggles.MldSnoopingToggleLib

# All globals are dictionaries with index as entityManager. This is to support Cli
# with multiple entity managers in a cohab test environment
mldSnoopingConfigMap = {}
bridgingHwCapabilitiesMap = {}
forwardingStatusMap = {}
protocolStatusMap = {}
mldSnoopingCounterConfigMap = {}
counterDirMap = {}
counterCheckpointDirMap = {}
bridgingInputCliMap = {}
peerQuerierStatusMap = {}

mldMatcher = CliMatcher.KeywordMatcher( 'mld',
                  helpdesc='Multicast Listener Discovery commands' )

def mldSnoopingSupportedGuard( mode, token ):
   bridgingHwCapabilities = bridgingHwCapabilitiesMap[ mode.entityManager ]
   if bridgingHwCapabilities.mldSnoopingSupported:
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# The "config-mld-snooping" mode.
#-------------------------------------------------------------------------------
class MldSnoopingConfigMode( MldSnoopingMode, BasicCli.ConfigModeBase ):
   name = 'mld snooping configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      MldSnoopingMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# The "[no|default] mld snooping" command, in "config" mode.
#-------------------------------------------------------------------------------
class EnterMldSnoopingConfigMode( CliCommand.CliCommandClass ):
   syntax = 'mld snooping'
   noOrDefaultSyntax = syntax
   data = {
            'mld': 'Multicast Listener Discovery commands',
            'snooping': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'snooping',
                  helpdesc='Configure snooping parameters' ),
               guard=mldSnoopingSupportedGuard )
          }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      if not mldSnoopingConfig.mldSnoopingConfigured:
         mldSnoopingConfig.mldSnoopingConfigured = True
         mldSnoopingConfig.vlanDefaultEnabled = True
      childMode = mode.childMode( MldSnoopingConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      mldSnoopingConfig.mldSnoopingConfigured = False
      mldSnoopingConfig.vlanDefaultEnabled = False
      mldSnoopingConfig.vlanConfig.clear()

#-------------------------------------------------------------------------------
# The "[no|default] disabled" command, in "config-mld-snooping" mode.
#-------------------------------------------------------------------------------
class MldSnoopingDisabled( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = { 'disabled': 'Disable MLD snooping' }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      mldSnoopingConfig.vlanDefaultEnabled = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      mldSnoopingConfig.reset()

#-------------------------------------------------------------------------------
# The "config-mld-snooping-vlan" mode.
#-------------------------------------------------------------------------------
class MldSnoopingVlanConfigMode( MldSnoopingVlanMode, BasicCli.ConfigModeBase ):
   name = 'vlan configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vlan ):
      # vlan is used to instantiate VlanConfigMode and need
      # original vlan.individualVlans_
      self.vlan = copy.copy( vlan )
      MldSnoopingVlanMode.__init__( self, self.vlan.promptText() )
      self.multiInstance = False
      if len( self.vlan.individualVlans_ ) > 1:
         self.multiInstance = True
         self.constructIndividualVlanConfigModes( parent, session )
      self.vlanConfigMode = VlanConfigMode( parent, session, vlan )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def constructIndividualVlanConfigModes( self, parent, session ):
      newList = []
      for m in self.vlan.individualVlans_:
         newList.append( MldSnoopingVlanConfigMode( parent, session, m ) )
      self.vlan.individualVlans_ = newList

   def showActive( self ):
      runningConfigUrl = FileUrl.localRunningConfig( *Url.urlArgsFromMode( self ) )
      self.vlanConfigMode.printShowActiveForVlanRange( runningConfigUrl,
                                                       configmode='mld snooping',
                                                       prefix='   ' )

   def showActiveAll( self, showDetail ):
      runningConfigAllUrl = FileUrl.localRunningConfigAll(
         *Url.urlArgsFromMode( self ), showDetail=showDetail )
      self.vlanConfigMode.printShowActiveForVlanRange( runningConfigAllUrl,
                                                       configmode='mld snooping',
                                                       prefix='   ' )


#-------------------------------------------------------------------------------
# The "vlan <vlan range>" command, in "config-mld-snooping" mode.
#-------------------------------------------------------------------------------
class EnterMldSnoopingVlanConfigMode( CliCommand.CliCommandClass ):
   syntax = 'vlan VLANSET'
   noOrDefaultSyntax = syntax
   data = {
            'vlan': 'Specify VLAN/VLAN range',
            'VLANSET': vlanSetMatcher
          }

   @staticmethod
   def handler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      vlanSet = args[ 'VLANSET' ]
      vlanConfig = mldSnoopingConfig.vlanConfig
      for vlanId in vlanSet.ids:
         vlan = vlanConfig.get( vlanId )
         if not vlan:
            vlan = mldSnoopingConfig.newVlanConfig( vlanId )
            vlan.enabled = 'vlanStateDefault'

      if len( vlanSet.ids ) == 1:
         vlan = Vlan( list( vlanSet.ids )[ 0 ] )
      else:
         vlan = vlanSet

      childMode = mode.childMode( MldSnoopingVlanConfigMode, vlan=vlan )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      vlanSet = args[ 'VLANSET' ]
      vlanConfig = mldSnoopingConfig.vlanConfig
      for vlanId in vlanSet.ids:
         vlan = vlanConfig.get( vlanId )
         if vlan:
            del vlanConfig[ vlanId ]

def setMldSnoopingVlans( mode, no=False ):
   mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
   for vlanId in mode.vlan.ids:
      vlan = mldSnoopingConfig.vlanConfig.get( vlanId )
      vlan.enabled = 'vlanStateDisabled' if no else 'vlanStateDefault'

def setMldSnoopingVlan( mode, no=False ):
   mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
   vlanId = mode.vlan.id_
   vlan = mldSnoopingConfig.vlanConfig.get( vlanId )
   vlan.enabled = 'vlanStateDisabled' if no else 'vlanStateDefault'

#-------------------------------------------------------------------------------
# The "[no|default] disabled" command, in "config-mld-snooping-vlan" mode.
#-------------------------------------------------------------------------------
class MldSnoopingVlanDisabled( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = { 'disabled': 'Disable mld snooping on vlans' }

   @staticmethod
   def handler( mode, args ):
      if mode.multiInstance:
         setMldSnoopingVlans( mode, no=True )
      else:
         setMldSnoopingVlan( mode, no=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.multiInstance:
         setMldSnoopingVlans( mode, no=False )
      else:
         setMldSnoopingVlan( mode, no=False )

#-------------------------------------------------------------------------------
# The "[no|default] member <IPV6ADDR> interface <intf>" command,
# in "config-mld-snooping-vlan" mode.
#-------------------------------------------------------------------------------
class VlanMemberGroupCommand( CliCommand.CliCommandClass ):
   syntax = 'member IPV6ADDR interface INTF'
   noOrDefaultSyntax = syntax
   data = {
      'member': memberMatcher,
      'IPV6ADDR': Ip6AddrMatcher.ip6AddrMatcher,
      'interface': interfaceMatcher,
      'INTF': intfRangeConfigMatcher
   }

   @staticmethod
   def handler( mode, args ):
      ipv6Addr = args[ 'IPV6ADDR' ]
      intfs = args[ 'INTF' ]
      no = CliCommand.isNoOrDefaultCmd( args )
      if mode.multiInstance:
         vlanIds = mode.vlan.ids
      else:
         vlanIds = [ mode.vlan.id_ ]
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      for vlanId in vlanIds:
         doSetSnoopingVlanMember( mode, mldSnoopingConfig, vlanId, ipv6Addr, intfs,
                                  no=no )

   noOrDefaultHandler = handler

#-------------------------------------------------------------------------------
# The "[no|default] multicast-router interface <intf>" command,
# in "config-mld-snooping-vlan" mode.
#-------------------------------------------------------------------------------
class VlanMulticastRouterCommand( CliCommand.CliCommandClass ):
   syntax = "multicast-router interface INTF"
   noOrDefaultSyntax = syntax
   data = {
      "multicast-router": "Multicast router related configuration",
      "interface": interfaceMatcher,
      "INTF": intfRangeConfigMatcher
   }

   @staticmethod
   def handler( mode, args ):
      intfs = args[ 'INTF' ]
      no = CliCommand.isNoOrDefaultCmd( args )
      if mode.multiInstance:
         vlanIds = mode.vlan.ids
      else:
         vlanIds = [ mode.vlan.id_ ]
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      setSnoopingVlanMrouter( mode, mldSnoopingConfig, vlanIds, intfs, no=no )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( EnterMldSnoopingConfigMode )
MldSnoopingConfigMode.addCommandClass( MldSnoopingDisabled )
MldSnoopingConfigMode.addCommandClass( EnterMldSnoopingVlanConfigMode )
MldSnoopingVlanConfigMode.addCommandClass( MldSnoopingVlanDisabled )
MldSnoopingVlanConfigMode.addCommandClass( VlanMemberGroupCommand )
MldSnoopingVlanConfigMode.addCommandClass( VlanMulticastRouterCommand )

#-------------------------------------------------------------------------------
# "show mld snooping groups"
#-------------------------------------------------------------------------------
class ShowMldSnoopingGroups( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping groups'
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'groups': 'MLD group information'
   }

   @staticmethod
   def handler( mode, args ):
      warningSentence = \
         ( "* Warning: MLD Snooping Pruning is not active. Flooding traffic"
          " to vlan.\nPlease check for MLD Querier." )
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      protocolStatus = protocolStatusMap[ mode.entityManager ]
      forwardingStatus = forwardingStatusMap[ mode.entityManager ]
      doShowGmpSnoopingGroupsVlansHelper( mode, mldSnoopingConfig.vlanConfig,
                                          protocolStatus.contextStatus,
                                          forwardingStatus.vlanStatus,
                                          agentName(), warningSentence )

detailHelpStr = 'More comprehensive output'

#-------------------------------------------------------------------------------
# "show mld snooping mrouter"
#-------------------------------------------------------------------------------
class ShowMldSnoopingMrouter( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping mrouter [detail] [vlan VLANID]'
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'mrouter': 'MLD multicast router information',
      'detail': detailHelpStr,
      'vlan': 'Specify VLAN',
      'VLANID': vlanIdMatcher,
   }
   cliModel = IgmpSnoopingMrouter

   @staticmethod
   def handler( mode, args ):
      vlanId = args.get( 'VLANID' )
      mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
      protocolStatus = protocolStatusMap[ mode.entityManager ]
      forwardingStatus = forwardingStatusMap[ mode.entityManager ]
      return doShowGmpSnoopingMrouter( mode, vlanId=vlanId,
            detail='detail' in args, sysdbNodes=( None,
                                                  mldSnoopingConfig.vlanConfig,
                                                  protocolStatus.contextStatus,
                                                  forwardingStatus.vlanStatus ) )

BasicCli.addShowCommandClass( ShowMldSnoopingMrouter )

def updatePortCounters( mode, args ):
   mldSnoopingCounterConfig = mldSnoopingCounterConfigMap[ mode.entityManager ]
   mldSnoopingCounterConfig.counterUpdateRequestTime = Tac.now() # monotonic time
   forwardingStatus = forwardingStatusMap[ mode.entityManager ]

   def countersUpdated():
      return forwardingStatus.counterUpdateTime >= \
             mldSnoopingCounterConfig.counterUpdateRequestTime
   try:
      Tac.waitFor( countersUpdated, description="counter update", maxDelay=0.2,
                   sleep=True, timeout=30.0 )
   except Tac.Timeout:
      mode.addWarning( "Displaying stale counters" )

def doShowMldSnoopingCounter( mode, args ):
   ''' show multicast counter entries. '''
   counterDir = counterDirMap[ mode.entityManager ]
   counterCheckpointDir = counterCheckpointDirMap[ mode.entityManager ]
   syncCounterCheckpointDir( counterCheckpointDir, counterDir )
   intfs = counterCheckpointDir.intfCounter

   if not intfs:
      return MldSnoopingCounters()
   errorSpecific = "errors" in args
   c = Tac.newInstance( "Bridging::IgmpSnooping::IgmpContextIntfCounter", "" )
   interfaces = { }
   for intf in Arnet.sortIntf( intfs ):
      counter = counterDir.intfCounter.get( intf )
      counterCheckpoint = counterCheckpointDir.intfCounter.get( intf )
      if counter is None or counterCheckpoint is None:
         continue
      # print difference since last checkpointed counter
      c.doCopy( counter )
      c.doSubtract( counterCheckpoint )
      interfaceCounters = MldSnoopingCountersInterface()
      interfaceCounters.shortPacketsReceived = c.inPktTooShort
      interfaceCounters.nonIpPacketsReceived = c.inPktNonIp
      interfaceCounters.badChecksumIpPacketsReceived = c.inPktBadIpChecksum
      interfaceCounters.unknownIpPacketsReceived = c.inPktUnknown
      interfaceCounters.badChecksumPimPacketsReceived = c.inPktBadPimChecksum

      interfaceCounters.badChecksumIcmpV6PacketsReceived = c.inPktBadIcmpV6Checksum
      interfaceCounters.badMldQueryReceived = c.inPktBadMldQuery
      interfaceCounters.badMldV2ReportReceived = c.inPktBadMldV2Report
      if not errorSpecific:
         interfaceCounters.pimPacketsReceived = c.inPktPim
         interfaceCounters.otherPacketsSent = c.outPktOther

         interfaceCounters.mldV1QueryReceived = c.inPktMldV1Query
         interfaceCounters.mldV2QueryReceived = c.inPktMldV2Query
         interfaceCounters.mldV2ReportReceived = c.inPktMldV2Report
         interfaceCounters.otherIcmpPacketsReceived = c.inPktIcmpV6Other
         interfaceCounters.mldQuerySend = c.outPktGmpQuery
         interfaceCounters.mldReportSend = c.outPktGmpReport

      interfaces[ intf ] = interfaceCounters
   return MldSnoopingCounters( interfaces=interfaces,
         _errorSpecific=errorSpecific )

#-------------------------------------------------------------------------------
# "show mld snooping counters"
#-------------------------------------------------------------------------------
class ShowMldSnoopingCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping counters [ errors ]'
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'counters': 'MLD counter information',
      'errors': 'Error counters',
   }
   cliModel = MldSnoopingCounters
   prepareFunction = updatePortCounters
   handler = doShowMldSnoopingCounter

BasicCli.addShowCommandClass( ShowMldSnoopingCounters )

def doShowMldSnoopingQuerier( mode, args ):
   # pylint: disable-msg=W0612
   vlanId = args.get( 'VLANID' )
   protocolStatus = protocolStatusMap[ mode.entityManager ]
   contextStatuses = protocolStatus.contextStatus
   peerQuerierStatus = peerQuerierStatusMap[ mode.entityManager ]

   vlans = {}
   if vlanId:
      contextStatus = contextStatuses.get( vlanId )
      if contextStatus:
         vlanInfo = showVlanQuerierHelper( contextStatus, peerQuerierStatus )
         if vlanInfo:
            vlans[ vlanId ] = vlanInfo
      return MldSnoopingQuerier( vlans=vlans, _vlanSpecific=True )
   else:
      vlanIds = contextStatuses.keys()
      vlanIds.sort( key=int )
      for vlanId in vlanIds:
         contextStatus = contextStatuses.get( vlanId )
         if contextStatus:
            vlanInfo = showVlanQuerierHelper( contextStatus, peerQuerierStatus )
            if vlanInfo:
               vlans[ vlanId ] = vlanInfo
      return MldSnoopingQuerier( vlans=vlans )

def showVlanQuerierHelper( status, peerQuerierStatus ):
   querier = status.querier
   if querier is None:
      return None
   querierIntf = querier.intf
   helper = IgmpSnoopingCliLib.IgmpSnoopingCliHelper()
   if querierIntf is None:
      return None
   elif helper.isPeerIntf( querierIntf ): # pylint: disable-msg=no-member
      # Peer-link is local querier intf. Check if the address matches peer's querier
      # address
      if status.vlanId in peerQuerierStatus.vlanQuerierStatus:
         peerVlanQuerier = peerQuerierStatus.vlanQuerierStatus[ status.vlanId ]
         if querier.addr == peerVlanQuerier.addr:
            # Local querier address is same as peer's querier address. Check if there
            # is a corresponding local mlag intf
            localMlagIntf = peerVlanQuerier.localMlagIntfId
            querierIntf = localMlagIntf if localMlagIntf else querierIntf

   vlanQuerierInfo = MldSnoopingVlanQuerier()
   vlanQuerierInfo.querierAddress = str( querier.addr )
   vlanQuerierInfo.mldVersion = \
               igmpVersionTable[ status.querierVersion ]
   vlanQuerierInfo.querierInterface = querierIntf
   vlanQuerierInfo.queryResponseInterval = status.queryResponseInterval
   return vlanQuerierInfo

#-------------------------------------------------------------------------------
# "show mld snooping querier [STATUS][VLAN_ID][DATA]"
#-------------------------------------------------------------------------------
class ShowMldSnoopingQuerier( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping querier [ vlan VLANID ]'
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'querier': 'MLD querier information',
      'vlan': 'Specify VLAN',
      'VLANID': vlanIdMatcher,
   }
   cliModel = MldSnoopingQuerier
   handler = doShowMldSnoopingQuerier

BasicCli.addShowCommandClass( ShowMldSnoopingQuerier )

def getVlanSnoopingState( config, vlanId ):
   vlan = config.vlanConfig.get( vlanId )
   if vlan and vlan.enabled == 'vlanStateDefault':
      return 'enabled' if config.vlanDefaultEnabled else 'disabled'
   return 'disabled'

def doShowMldSnooping( mode, args ):
   ''' the function shows singleton attributes for the global settings,
       and per-vlan settings. Only the active vlans are shown here, as per
       industry-standard.
   '''
   mldSnooping = MldSnoopingInfo()
   mldSnoopingConfig = mldSnoopingConfigMap[ mode.entityManager ]
   forwardingStatus = forwardingStatusMap[ mode.entityManager ]
   protocolStatus = protocolStatusMap[ mode.entityManager ]
   bridgingInputCli = bridgingInputCliMap[ mode.entityManager ]
   if not mldSnoopingConfig.mldSnoopingConfigured:
      return mldSnooping
   vlanId = args.get( 'VLANID' )
   if vlanId:
      if not IgmpSnoopingLib.isVlanActive( bridgingInputCli, vlanId ):
         mode.addWarning( "VLAN %s is not active." % vlanId )
         return mldSnooping
      vlanIds = [ vlanId ]
   else:
      vlanIds = IgmpSnoopingLib.activeVlans( bridgingInputCli )

   mldSnooping.mldSnoopingState = \
         'enabled' if mldSnoopingConfig.vlanDefaultEnabled else 'disabled'

   mldSnooping.robustness = mldSnoopingConfig.robustness
   for vlanId in vlanIds:
      mldSnooping.vlans[ vlanId ] = \
            doShowMldSnoopingVlan( mldSnoopingConfig, forwardingStatus,
                                   protocolStatus, vlanId )
   return mldSnooping

def doShowMldSnoopingVlan( config, forwardingStatus, protocolStatus, vlanId ):
   ''' show singleton attributes for a given vlan. '''

   vlans = MldSnoopingVlanInfo()
   vlans.mldSnoopingState = getVlanSnoopingState( config, vlanId )
   vlanConfig = config.vlanConfig.get( vlanId )

   if vlanConfig and ( vlanConfig.maxGroups < vlanConfig.unlimitedGroups ):
      vlans.maxGroups = vlanConfig.maxGroups

   contextStatus = protocolStatus.contextStatus.get( vlanId )
   if contextStatus and contextStatus.groupLimitExceeded:
      vlans.groupsOverrun = True

   statuses = forwardingStatus.vlanStatus
   statusVlan = statuses.get( vlanId )
   if contextStatus:
      vlans.pruningActive = contextStatus.confident
   if statusVlan:
      vlans.floodingTraffic = statusVlan.useVlanFloodset
   return vlans

#-------------------------------------------------------------------------------
# "show mld snooping [vlan VLAN_ID]"
#-------------------------------------------------------------------------------
class ShowMldSnooping( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld snooping [ vlan VLANID ] '
   data = {
      'mld': mldMatcher,
      'snooping': 'Show snooping information',
      'vlan': 'Specify VLAN',
      'VLANID': vlanIdMatcher,
   }
   cliModel = MldSnoopingInfo
   handler = doShowMldSnooping

BasicCli.addShowCommandClass( ShowMldSnooping )

#-------------------------------------------------------------------------------
# clear mld snooping counters [intfs] in enable mode
#-------------------------------------------------------------------------------
class ClearMldSnoopingCounters( CliCommand.CliCommandClass ):
   syntax = 'clear mld snooping counters [INTF]'
   data = {
            'clear': clearKwNode,
            'mld': 'Multicast Listener Discovery commands',
            'snooping': 'Snooping related',
            'counters': 'Counters',
            'INTF': intfRangeConfigMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      intfs = args.get( 'INTF' )
      interfaces = []
      if intfs:
         interfaces = IntfCli.Intf.getAll( mode, intfs, config=True )
         if not interfaces:
            return
      counterCheckpointDir = counterCheckpointDirMap[ mode.entityManager ]
      counterDir = counterDirMap[ mode.entityManager ]
      clearSnoopingCounters( mode, counterCheckpointDir, counterDir, igmp=False,
                             intfs=interfaces )

BasicCli.EnableMode.addCommandClass( ClearMldSnoopingCounters )

# -----------------------------------------------------------------------------------
# Display MembershipJoinStatus from different clients - user, local, mlag.
# If no filter is provided, display the merged state.
#
# show ip igmp snooping groups [ local | mlag | user ] \
#                              [ vlan <vlanid> ] [ A.B.C.D ] [ detail ] [ count ]
#
# Example of output:
# IGMP Snooping Group Membership
# EX: Filter mode Exclude
# IN: Filter mode Include
# VLAN  Group            Members
#----  ---------------  ----------------------------------------
# 10    225.0.0.1        Et1, Et2, Et3, Et4
#                       10.1.1.1(IR), 10.1.1.2(IR), 10.1.1.3(IR)
# 10    225.0.0.2        Et1, Et2, Et3, Et4, 10.1.1.1(IR),
#                       10.1.1.2(IR), 10.1.1.3(IR)
#10    *                Et3, Et4

clientMatcher = CliMatcher.EnumMatcher( {
   'local': 'Show groups learnt locally via MLD',
   'mlag': 'Show groups learnt via MLAG peer',
   'user': 'Show groups configured by user' } )

vlanMatcher = CliMatcher.KeywordMatcher( 'vlan', helpdesc='Specify VLAN' )
showSnoopingMatcher = CliMatcher.KeywordMatcher( 'snooping',
                         helpdesc='Show snooping information' )
showGroupsMatcher = CliMatcher.KeywordMatcher( 'groups',
                       helpdesc='MLD group information' )

class ShowClientSnoopingGroups( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mld snooping groups [ CLIENT ] [ vlan VLANID ] [ GRPADDR ]' )
   data = {
         'mld': mldMatcher,
         'snooping': showSnoopingMatcher,
         'groups': showGroupsMatcher,
         'CLIENT': clientMatcher,
         'vlan': vlanMatcher,
         'VLANID': vlanIdMatcher,
         'GRPADDR': Ip6AddrMatcher.ip6AddrMatcher,
         }
   cliModel = GmpMembershipInfo

   @staticmethod
   def handler( mode, args ):
      return showClientSnoopingsGroupsHandler( mode, agentName(), args,
                                               'showIpIgmpSnoopingGroups',
                                               GmpMembershipInfo )

class ShowClientSnoopingGroupsDetail( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mld snooping groups [ CLIENT ] [ vlan VLANID ] [ GRPADDR ] '
              'detail' )
   data = {
         'mld': mldMatcher,
         'snooping': showSnoopingMatcher,
         'groups': showGroupsMatcher,
         'CLIENT': clientMatcher,
         'vlan': vlanMatcher,
         'VLANID': vlanIdMatcher,
         'GRPADDR': Ip6AddrMatcher.ip6AddrMatcher,
         'detail': detailHelpStr,
         }
   cliModel = GmpMembershipDetailInfo

   @staticmethod
   def handler( mode, args ):
      return showClientSnoopingsGroupsHandler( mode, agentName(), args,
                                               'showIpIgmpSnoopingGroups',
                                               GmpMembershipDetailInfo )

class ShowClientSnoopingGroupsCount( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mld snooping groups [ CLIENT ] [ vlan VLANID ] count' )
   data = {
         'mld': mldMatcher,
         'snooping': showSnoopingMatcher,
         'groups': showGroupsMatcher,
         'CLIENT': clientMatcher,
         'vlan': vlanMatcher,
         'VLANID': vlanIdMatcher,
         'count': 'Show membership count',
         }
   cliModel = GmpMembershipCountInfo

   @staticmethod
   def handler( mode, args ):
      return showClientSnoopingsGroupsHandler( mode, agentName(), args,
                                               'showIpIgmpSnoopingGroupsCount',
                                               GmpMembershipCountInfo )

if Toggles.MldSnoopingToggleLib.toggleMlagMldSnoopingEnabled():
   BasicCli.addShowCommandClass( ShowClientSnoopingGroups )
   BasicCli.addShowCommandClass( ShowClientSnoopingGroupsDetail )
   BasicCli.addShowCommandClass( ShowClientSnoopingGroupsCount )
else:
   BasicCli.addShowCommandClass( ShowMldSnoopingGroups )

def Plugin( entityManager ):
   global mldSnoopingConfigMap
   global bridgingHwCapabilitiesMap
   global forwardingStatusMap
   global protocolStatusMap
   global counterDirMap
   global counterCheckpointDirMap
   global mldSnoopingCounterConfigMap
   global bridgingInputCliMap
   global protocolStatusMap

   mldSnoopingConfigMap[ entityManager ] = ConfigMount.mount(
      entityManager,
      "bridging/mldsnooping/config",
      "Bridging::IgmpSnooping::Config", "w" )
   bridgingHwCapabilitiesMap[ entityManager ] = LazyMount.mount(
      entityManager,
      "bridging/hwcapabilities",
      "Bridging::HwCapabilities", "r" )
   forwardingStatusMap[ entityManager ] = LazyMount.mount(
      entityManager,
      "bridging/mldsnooping/forwarding/status",
      "Bridging::IgmpSnooping::Status", "r" )
   protocolStatusMap[ entityManager ] = LazyMount.mount(
      entityManager,
      "bridging/mldsnooping/protocol/status",
      "Bridging::IgmpSnooping::IgmpProtocolStatus", "r" )
   counterDirMap[ entityManager ] = LazyMount.mount(
      entityManager,
      'bridging/mldsnooping/counterDir',
      'Bridging::IgmpSnooping::IgmpCounterDir', 'r' )
   counterCheckpointDirMap[ entityManager ] = LazyMount.mount(
      entityManager,
      'bridging/mldsnooping/counterCheckpointDir',
      'Bridging::IgmpSnooping::IgmpCounterDir', 'w' )
   mldSnoopingCounterConfigMap[ entityManager ] = LazyMount.mount(
      entityManager,
      "bridging/mldsnooping/counterConfig",
      "Bridging::IgmpSnooping::CounterConfig", "w" )
   bridgingInputCliMap[ entityManager ] = LazyMount.mount(
      entityManager,
      'bridging/input/config/cli',
      'Bridging::Input::CliConfig', 'r' )
   peerQuerierStatusMap[ entityManager ] = LazyMount.mount(
      entityManager,
      'bridging/mldsnooping/mlag/peerQuerierStatus',
      'Bridging::IgmpSnooping::PeerQuerierStatus', 'r' )
