#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys

import BasicCli
import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.McastCommonCli as McastCommonCli
import CliPlugin.MldCliLib as MldCliLib
import CliPlugin.MldCliModel as MldCliModel
import CliPlugin.TechSupportCli
import CliPlugin.VrfCli as VrfCli
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ShowCommand
import Tac

mldStatus = None
mldConfig = None
mldAddrs = None
ip6Status = None
aclConfigCli = None

nodeMld = CliCommand.guardedKeyword( 'mld',
      helpdesc='Multicast Listener Discovery commands',
      guard=McastCommonCli.mcastv6RoutingSupportedGuard )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Show MLD information for specified interface' )

def useCurrentVrfIfNone( func ):
   '''Decorator for show commands. If the vrfName argument is None replace
   it with the Current VRF. '''
   def newFunc( mode, args ):
      if 'VRF' in args and args[ 'VRF' ] is None:
         args[ 'VRF' ] = VrfCli.vrfMap.getCliSessVrf( mode.session )
      return func( mode, args )
   return newFunc

#--------------------------------------------------------------------------------
# show mld [ VRF ] membership [ FILTER ] [ interface INTF ] [ group GROUP_ADDR ]
#--------------------------------------------------------------------------------
class MldMembershipCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mld [ VRF ] membership [ FILTER ] [ interface INTF ] '
                                          '[ group GROUP_ADDR ]' )
   data = {
      'mld' : nodeMld,
      'VRF' : MldCliLib.vrfExprFactory,
      'membership' : 'Show MLD group membership information',
      'FILTER' : CliMatcher.EnumMatcher( {
         'static' : 'Show statically configured groups',
         'dynamic' : 'Show dynamically learned groups',
      } ),
      'interface' : matcherInterface,
      'INTF' : IntfCli.Intf.matcher,
      'group' : 'Show MLD information for specified group',
      'GROUP_ADDR' : Ip6AddrMatcher.Ip6AddrMatcher( helpdesc='IPv6 group address' ),
   }
   cliModel = MldCliModel.MldMembershipInterfaces

   @staticmethod
   @useCurrentVrfIfNone
   def handler( mode, args ):
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      intf = args.get( 'INTF' )
      filterType = args.get( 'FILTER' )
      groupAddr = args.get( 'GROUP_ADDR' )
      gmpStatus = mldStatus.getGmpStatus( vrfName )

      fd = sys.stdout.fileno()
      outputFormat = mode.session_.outputFormat()

      groupFilterType = "groupFilterTypeNone"
      if filterType == 'static':
         groupFilterType = "groupFilterTypeStaticGroups"
      elif filterType == 'dynamic':
         groupFilterType = "groupFilterTypeDynamicGroups"

      intfIdFilter = Tac.Value( "Arnet::IntfId" )
      if intf:
         intfIdFilter = intf.name

      groupAddrFilter = Tac.Value( "Arnet::Ip6Addr" )
      if groupAddr:
         groupAddrFilter = groupAddr

      dynamicStatus = mldStatus.getMldStatus( vrfName, 'dynamicGroup' )

      staticStatus = mldStatus.getMldStatus( vrfName, 'staticGroup' )

      groupRenderer = Tac.newInstance( "Routing::Mld::GroupRenderer", vrfName,
                                       groupFilterType, intfIdFilter,
                                       groupAddrFilter, gmpStatus, dynamicStatus,
                                       staticStatus )

      groupRenderer.render( fd, outputFormat )

      # Deferred Model - return an empty model
      return MldCliModel.MldMembershipInterfaces

BasicCli.addShowCommandClass( MldMembershipCmd )

#--------------------------------------------------------------------------------
# show mld [ VRF ] summary [ interface INTF ]
#--------------------------------------------------------------------------------
class MldSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld [ VRF ] summary [ interface INTF ]'
   data = {
      'mld' : nodeMld,
      'VRF' : MldCliLib.vrfExprFactory,
      'summary' : 'Show MLD summary',
      'interface' : matcherInterface,
      'INTF' : IntfCli.Intf.matcher,
   }
   cliModel =  MldCliModel.MldSummaryInterfaces

   @staticmethod
   @useCurrentVrfIfNone
   def handler( mode, args ):
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      intf = args.get( 'INTF' )
      querierStatus = mldStatus.getMldQuerierStatus( vrfName )
      dynamicStatus = mldStatus.getMldStatus( vrfName, 'dynamicGroup' )
      staticStatus = mldStatus.getMldStatus( vrfName, 'staticGroup' )

      fd = sys.stdout.fileno()
      outputFormat = mode.session_.outputFormat()

      intfIdFilter = Tac.Value( "Arnet::IntfId" )
      if intf:
         intfIdFilter = intf.name

      # Must do a force mount before passing to C++ for rendering
      LazyMount.force( ip6Status )

      renderer = Tac.newInstance( "Routing::Mld::SummaryRenderer", vrfName,
                                  intfIdFilter,
                                  dynamicStatus,
                                  staticStatus,
                                  querierStatus,
                                  ip6Status )

      renderer.render( fd, outputFormat )

      # Deferred Model - return an empty model
      return MldCliModel.MldSummaryInterfaces

BasicCli.addShowCommandClass( MldSummaryCmd )

#--------------------------------------------------------------------------------
# show mld [ VRF ] statistics [ version VERSION ]
#--------------------------------------------------------------------------------
# By default, the show command will show total stats.
# See QuerierStatsMldVersion in MldCapi.tac.
MLDSTATSVERSIONDEFAULT = 100

class MldStatisticsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld [ VRF ] statistics [ version VERSION ]'
   data = {
      'mld' : nodeMld,
      'VRF' : MldCliLib.vrfExprFactory,
      'statistics' : 'Show MLD statistics information',
      'version' : 'MLD version',
      'VERSION' : CliMatcher.IntegerMatcher( 1, 2, helpdesc='MLD version value' ),
   }
   cliModel = MldCliModel.MldStatisticsInterfaces

   @staticmethod
   @useCurrentVrfIfNone
   def handler( mode, args ):
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      version = args.get( 'VERSION', MLDSTATSVERSIONDEFAULT )

      querierStatus = mldStatus.getMldQuerierStatus( vrfName )

      fd = sys.stdout.fileno()
      outputFormat = mode.session_.outputFormat()

      statsRenderer = Tac.newInstance( "Routing::Mld::QuerierStatsRenderer", vrfName,
                                       querierStatus )

      statsRenderer.renderQuerierStatus( fd, outputFormat, version )

      # Deferred Model - return an empty model
      return MldCliModel.MldStatisticsInterfaces

BasicCli.addShowCommandClass( MldStatisticsCmd )

#--------------------------------------------------------------------------------
# show mld [ VRF ] querier [ parameters ] [ interface INTF ]
#--------------------------------------------------------------------------------
class MldQuerierCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld [ VRF ] querier [ parameters ] [ interface INTF ]'
   data = {
      'mld' : nodeMld,
      'VRF' : MldCliLib.vrfExprFactory,
      'querier' : 'Show MLD querier information',
      'parameters' : 'Show querier parameters',
      'interface' : matcherInterface,
      'INTF' : IntfCli.Intf.matcher,
   }

   cliModel = MldCliModel.MldQuerierInterfaces

   @staticmethod
   @useCurrentVrfIfNone
   def handler( mode, args ):
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      intf = args.get( 'INTF' )
      parameters = 'parameters' in args

      intfIdFilter = Tac.Value( "Arnet::IntfId" )
      if intf:
         intfIdFilter = intf.name

      querierStatus = mldStatus.getMldQuerierStatus( vrfName )
      querierConfig = mldConfig.intfConfig

      fd = sys.stdout.fileno()
      outputFormat = mode.session_.outputFormat()

      renderer = Tac.newInstance( "Routing::Mld::QuerierInfoRenderer",
                                  vrfName,
                                  intfIdFilter,
                                  parameters,
                                  querierStatus,
                                  querierConfig )

      renderer.renderQuerierInfo( fd, outputFormat)

      # Deferred Model - return an empty model
      return MldCliModel.MldQuerierInterfaces

BasicCli.addShowCommandClass( MldQuerierCmd )

#--------------------------------------------------------------------------------
# show mld [ VRF ] static-group access-list ACL_NAME
#--------------------------------------------------------------------------------
class MldStaticGroupAccessListNameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mld [ VRF ] static-group access-list ACL_NAME'
   data = {
      'mld' : nodeMld,
      'VRF' : MldCliLib.vrfExprFactory,
      'static-group' : 'Show MLD static groups',
      'access-list' : 'IPv6 access-list for use as static group list',
      'ACL_NAME' : AclCli.userIp6AclNameMatcher,
   }
   cliModel = MldCliModel.MldAclInfo

   @staticmethod
   @useCurrentVrfIfNone
   def handler( mode, args ):
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      name = args[ 'ACL_NAME' ]
      aclConfig = aclConfigCli.config[ 'ipv6' ].acl.get( name )
      if aclConfig is None:
         mode.addError( "Access-list '%s' not configured" % name )
         return None

      acl = aclConfig.currCfg
      intfConfig = mldConfig.intfConfig
      gmpStatus = mldStatus.getGmpStatus( vrfName )
      if not gmpStatus:
         mode.addError( "Invalid vrf name %s " % vrfName )
         return None

      renderer = Tac.newInstance( "Routing::Mld::StaticAclRenderer",
                                  vrfName, acl, name, intfConfig, gmpStatus )
      fd = sys.stdout.fileno()
      outputFormat = mode.session_.outputFormat()
      renderer.renderStaticAclInfo( fd, outputFormat )

      # Deferred Model - return an empty model
      return MldCliModel.MldAclInfo

BasicCli.addShowCommandClass( MldStaticGroupAccessListNameCmd )

def _showTechCmds():
   if McastCommonCli.mcastv6RoutingSupportedGuard():
      return []

   return [
         "show mld summary",
         "show mld membership",
         "show mld statistics",
         ]

def _showTechSummaryCmds():
   if McastCommonCli.mcastv6RoutingSupportedGuard():
      return []
   return [ "show mld summary" ]

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( '2019-07-16 17:30:00',
            _showTechCmds, summaryCmdCallback=_showTechSummaryCmds )

def Plugin( entityManager ):
   global mldStatus
   global mldConfig
   global ip6Status
   global aclConfigCli
   global mldAddrs

   mldStatus = MldCliLib.MldStatus( entityManager )
   mldConfig = MldCliLib.MldConfig( entityManager )
   mldAddrs = MldCliLib.MldAddrs()

   ip6Status = LazyMount.mount( entityManager, 'ip6/status', 'Ip6::Status', 'r' )
   aclConfigCli = LazyMount.mount( entityManager, 'acl/config/cli',
                                  'Acl::Input::Config', 'r' )

