#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import DeferredModel, Submodel
from CliModel import Enum, Float, Dict, Int, Str, List
from IntfModels import Interface
from ArnetModel import Ip6Address, IpGenericAddrAndFullMask

class MldSource( DeferredModel ):
   '''MLD local listener state for an (S,G) or (*,G)'''

   filterMode = Enum( help="The filter mode may be include or exclude. In include "
                      "mode, packet reception is enabled for this source. In exclude"
                      " mode, it is not", values=( 'include', 'exclude' ) )

   creationTime = Float( help="UTC time at which this source was created",
                         optional=True )
   expiryTime = Float( help="UTC time at which this source will expire",
                       optional=True )

class MldGroup( DeferredModel ):
   '''MLD per group membership state'''

   filterMode = Enum( help="The filter mode may be include or exclude. "  
                      "This mode is based on modes of sources. "
                      "In case there are no sources, the mode is exclude "
                      "and it means to exclude no sources",
                      values=( 'include', 'exclude' ), optional=True )

   creationTime = Float( help="UTC time at which this group was created",
                         optional=True )
   expiryTime = Float( help="UTC time at which this group will expire",
                       optional=True )
   lastReporter = Ip6Address( help="The last host to send an MLD report for this "
                              "group", optional=True )
   compatibilityVersion = Enum( help="The compatibility version is set to version1 "
                                "or version2. It is set to version2 by default, but "
                                "set to version1 whenever an MLDv1 report is "
                                "received for the group",
                                values=( 'version1', 'version2' ), optional=True )
   sources = Dict( help="Map of include or exclude sources for this group",
                   keyType=Ip6Address, valueType=MldSource, optional=True )

class MldMembershipInterface( DeferredModel ):
   '''MLD per interface group membership state'''

   groups = Dict( help="Map of groups for which there are local listeners",
                  keyType=Ip6Address, valueType=MldGroup )

class MldMembershipInterfaces( DeferredModel ):
   '''MLD interfaces that have local listeners for any multicast group'''

   interfaces = Dict( help="Map of interfaces for which there are listeners "
                      "attached", keyType=Interface,
                      valueType=MldMembershipInterface )

class MldV1StatisticsInterface( DeferredModel ):
   '''MLD per interface statistics for version 1'''
   generalQueriesSent = Int( help='MLD v1 general queries sent' )

   gsQueriesSent = Int( help='MLD v1 multicast address specific queries sent' )

   generalQueriesReceived = Int( help='MLD v1 general queries received' )

   gsQueriesReceived = \
            Int( help='MLD v1 multicast address specific queries received' )

   reportsReceived = Int( help='MLD v1 multicast listener reports received' )

   donesReceived = Int( help='MLD v1 multicast listener dones received' )

class MldV2StatisticsInterface( DeferredModel ):
   '''MLD per interface statistics for version 2'''
   generalQueriesSent = Int( help='MLD v2 general queries sent' )

   gsQueriesSent = Int( help='MLD v2 multicast address specific queries sent' )

   gssQueriesSent = \
            Int( help='MLD v2 multicast address and source specific queries sent' )


   generalQueriesReceived = Int( help='MLD v2 general queries received' )

   gsQueriesReceived = \
            Int( help='MLD v2 multicast address specific queries received' )

   gssQueriesReceived = \
            Int( help='MLD multicast address and source specific queries received' )

   reportsReceived = Int( help='MLD v2 multicast listener reports received' )

class MldStatisticsInterface( DeferredModel ):
   '''MLD per interface statistics for all versions'''

   v1Statistics = Submodel( help='MLD version 1 statistics',
                            valueType=MldV1StatisticsInterface, optional=True )

   v2Statistics = Submodel( help='MLD version 2 statistics',
                            valueType=MldV2StatisticsInterface, optional=True )

   errorPacketsReceived = Int( help='Error packets received', optional=True )

   otherPacketsReceived = Int( help='Other packets received', optional=True )

class MldStatisticsInterfaces( DeferredModel ):
   '''MLD interfaces that are enabled'''

   interfaces = Dict( help="Map of statistics for MLD interfaces",
                      keyType=Interface,
                      valueType=MldStatisticsInterface )

class MldQuerierInterface( DeferredModel ):
   '''MLD per interface information related to querier and querier parameters'''
   querier = Ip6Address( 
      help='Ip Address of the querier on this interface' )
   version = Enum( help="The MLD version configured on this interface.", 
                      values=( '1', '2' ) )
   robustness = Int( help='Active robustness on this interface' )
   queryInterval = Float( help='Active query interval in seconds' )
   queryResponseInterval = Float( 
         help='Active maximum query response interval in seconds' )
   startupQueryInterval = Float( 
         help='Effective startup query interval in seconds' )
   startupQueryCount = Int( help='Configured startup query count' )
   lastListenerQueryInterval = Float( 
         help='Configured last listener query response interval in seconds' )
   lastListenerQueryCount = Int( help='Configured last listener query count' )
   generalQueryExpiryTime = Float( 
         help='UTC time at which general query will be sent', optional=True )
   otherQuerierExpiryTime = Float( 
         help='UTC time at which the other querier will expire', optional=True )

class MldQuerierInterfaces( DeferredModel ):
   '''MLD interfaces that are enabled'''
   interfaces = Dict( help="Map of querier information for MLD interfaces",
                      keyType=Interface,
                      valueType=MldQuerierInterface )


class MldSummaryInterface( DeferredModel ):
   '''MLD per interface summary information'''
   ipAddress = Ip6Address(
      help='Link Local Ip Address of this interface used in sending queries', 
      optional=True )

   numGroups = Int( help='Number of MLD groups joined on this interface' )

   querierState = Enum( help="The querier state may be init, querier, " 
                        "or non-querier. ",  
                      values=( 'init', 'querier', 'non-querier'  ) )


class MldSummaryInterfaces( DeferredModel ):
   '''MLD interfaces that are enabled'''

   interfaces = Dict( help="Map of summary information for MLD interfaces",
                      keyType=Interface,
                      valueType=MldSummaryInterface )


class RuleInfo( DeferredModel ):
   source = Ip6Address( help='IP Address of the source in this rule' )
   group = IpGenericAddrAndFullMask( help='IP Address of the group and mask '
                                           'in this rule' )

   errorMessage = Str( help="Error message for improperly defined rule",
                           optional=True )
   groupsInRule = Int( help="Number of groups in the specified rule" )

class MldAclInfo( DeferredModel ):
   errorMessage = Str( help="Error message for improperly configured access-list",
                       optional=True )
   totalSGs = Int( help="Total number of valid (S,G) pairs within the access-list "
                            "across all rules" )
   totalStarGs = Int( help="Total number of (*,G) pairs within the access-list "
                            "across all rules" )
   ruleDetails = Dict( keyType=int,
                      valueType=RuleInfo,
                      help="Map of sequence number to rule data for this "
                            "access-list" )
   interfaces = List( valueType=Interface,
                      help="List of the interface names in the VRF using this "
                            "access-list for static groups" )

