#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliExtensions
import CliPlugin.LagCli as LagCli
import CliPlugin.IntfRangeCli as IntfRangeCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IraIpIntfCli as IraIpIntfCli
import CliPlugin.VlanCli as VlanCli
import CliPlugin.VrfCli as VrfCli
import CliPlugin.ReloadCli as ReloadCli
import CliPlugin.TapAggIntfCli as TapAggIntfCli
# Bogus import to generate dependency on TapAgg types.
# pkgdeps: import MirroringAgent
import LazyMount
from .MlagShowCli import portCategorySummary, showMlagInt 
from CliPlugin.BridgingCli import switchIntfConfigIfEnabled
from CliPlugin.VlanCli import Vlan
from StpStableUtil import stableFileExists
from StpCliUtil import stpStatusIs, stpConfigIs
from textwrap import fill
import Tracing, Tac
import re
from .MlagModel import IssuWarnings, IssuCompatibility, compatCheckMessage
import Url
import FileUrl
import EosVersion
from UrlPlugin.NetworkUrl import HttpUrl
from MlagMountHelper import MlagStatusLazyMounter
from MlagIssuCompatUtils import getSwiVersionAndCompatibleImages

__defaultTraceHandle__ = Tracing.Handle( "MlagWarning" )
t0 = Tracing.trace0

reloadDelayLacpConfigMsg = "'reload-delay mode lacp standby' is configured and "\
                           "the Non-MLAG reload delay of %.0f seconds is less "\
                           "than the MLAG reload delay of %.0f seconds. This "\
                           "configuration could cause traffic loss after reboot."
mlagInterfaceArgs = { 'states': 'states',
                      'interfaces': 'interfaces', 
                      'mlag': 'mlag', 
                      'STATES': ['active-partial-local-up'], 
                      'show': 'show' }

mlagConfig = mlagStatus = lagIntfStatusDir = ethIntfStatusDir = stpStatus = \
    stpInputConfig = stpConfig = mlagProtoStatus = ethIntfConfigDir = \
    mlagHwStatus = ipConfig = None 
def isPrimaryOrSecondaryMlag():
   return ( mlagStatus is not None and
            mlagStatus.mlagState in ( 'primary', 'secondary' ) )

def linkIsUp( intf ):
   config = ethIntfConfigDir.get( intf )
   status = ethIntfStatusDir.get( intf )
   if not config or not status:
      return False
   return config.enabled and status.operStatus == 'intfOperUp'

def canAllowIpIntfChange( intfName, msgTrailer, newAddrWithMask=None ):
   # Returns an error message and False if configured IP address for an
   # interface is the same as configured peerAddress if the interface is in the
   # same VRF as Mlag local interface.
   #
   # Returns an error message and False if we are currently primary or
   # secondary and the interface being modified is the local MLAG interface.
   # Returns True otherwise.
   msg = ""
   cfg = mlagConfig
   if cfg is None:
      # If Mlag agent isn't mounted, this extension doesn't matter
      return [ True, msg ]
   localInterface = cfg.localIntfId
   mlagLocalIntfVrf = VrfCli.getVrfNameFromIntfConfig( ipConfig, localInterface )
   vrf = VrfCli.getVrfNameFromIntfConfig( ipConfig, intfName )

   if vrf == mlagLocalIntfVrf and \
      newAddrWithMask and cfg.peerAddress != '0.0.0.0' and \
      newAddrWithMask.address == cfg.peerAddress:
      msg += 'Interface address cannot be the MLAG peer address'
      msg += msgTrailer
      return [ False, msg ]

   if intfName == localInterface and isPrimaryOrSecondaryMlag():
      msg += '%s is configured as the MLAG local interface' % localInterface
      msg += msgTrailer
      return [ False, msg ]
   else:
      return [ True, msg ]

def canDeleteIntfIp( intfName, addrWithMask=None ):
   return canAllowIpIntfChange( intfName, '; ip address not modified' )

def canSetIntfIp( intfName, newAddrWithMask=None, secondary=False, mode=None ):
   return canAllowIpIntfChange( intfName, '; ip address not modified', 
                                newAddrWithMask )

# similar to above, but handles VRF changes on the interface
def canSetVrf( intfName, oldVrf, newVrf, vrfDelete ):
   return canAllowIpIntfChange( intfName, '; VRF not modified' )

# This function checks whether an interface specified by user to be 
# deleted in an mlag dut can be deleted or not.
def canDeleteIntf( mode, intfs ):
   if not isPrimaryOrSecondaryMlag():
      return True 
   cfg = mlagConfig
   if cfg is None:
      return True
   peerLinkName = cfg.peerLinkIntfId
   localInterface = cfg.localIntfId
   canDelete = True
   intfNames = map( lambda intf: intf.name, intfs ) 
   msg = ''
   if ( localInterface in intfNames ) and linkIsUp( localInterface ):
      msg += 'Interface %s is configured as the MLAG local interface' \
          % localInterface
      canDelete = False
   elif ( peerLinkName in intfNames ) and linkIsUp( peerLinkName ):
      msg += 'Interface %s is configured as the MLAG peer link' % peerLinkName
      canDelete = False
   if not canDelete:
      mode.addError( msg + '; no interfaces deleted' )
   return canDelete
   
def canShutdownOrRemoveIfs( mode, intfs, isError=False, errorText='' ):
   if not isPrimaryOrSecondaryMlag():
      return True
   cfg = mlagConfig
   peerLinkName = cfg.peerLinkIntfId
   localInterface = cfg.localIntfId
   canShutdown = True
   intfNames = map( lambda intf: intf.name, intfs )
   msg = ''
   for intf in intfNames:
      if localInterface == intf and ( not isError or linkIsUp( intf ) ):
         msg += '%s is configured as the MLAG local interface' % intf
         canShutdown = False
      if peerLinkName == intf and ( not isError or linkIsUp( intf ) ):
         msg += 'Interface %s is configured as the MLAG peer link' % intf
         canShutdown = False

   elis = lagIntfStatusDir.get( peerLinkName )
   if elis:
      if isError and elis.member:
         linkUpMembers = set( filter( linkIsUp, elis.member ) )
         if linkUpMembers and linkUpMembers.issubset( intfNames ):
            if len( intfNames ) > 1:
               msg += 'Specified interface range would break the MLAG peer link %s' \
                   % peerLinkName
            else:
               msg += 'Interface %s is a member of the MLAG peer link %s' \
                   % ( intfNames[ 0 ], peerLinkName )
            canShutdown = False
      elif not isError:
         peerLinkMembers = set( elis.member ).intersection( intfNames )
         if len( peerLinkMembers ) > 1:
            msg += 'Interfaces %s are members of the MLAG peer link %s' \
                % ( ', '.join( sorted ( list(peerLinkMembers) ) ), peerLinkName )
         elif len( peerLinkMembers ) == 1:
            msg += 'Interface %s is a member of the MLAG peer link %s' \
                % ( peerLinkMembers.pop(), peerLinkName )

   if msg:
      if isError and not canShutdown:
         msg += '; ' + errorText
         mode.addError( msg )
      else:
         mode.addWarning( msg )
   return canShutdown

IraIpIntfCli.canDeleteIntfIpHook.addExtension( canDeleteIntfIp )
IraIpIntfCli.canSetIntfIpHook.addExtension( canSetIntfIp )
IraIpIntfCli.canSetVrfHook.addExtension( canSetVrf )
IntfRangeCli.extraGotoIntfRangeModeHook.addExtension( canShutdownOrRemoveIfs )
IntfCli.canShutdownIfHook.addExtension( lambda intfModes: canShutdownOrRemoveIfs( 
      intfModes[0], [ m.intf for m in intfModes ], True,
      'no interface was shutdown' ) )
IntfRangeCli.canDeleteIntfRangeHook.addExtension( canDeleteIntf )

def canSetChannelGroup( intfModes, channelGroupId, oldLagIds,
                        sessionOldLagIds ):

   if channelGroupId and mlagConfig.peerLinkIntfId:
      reject = False
      for mode in intfModes:
         if mode.intf.name == mlagConfig.peerLinkIntfId:
            intfModes[0].addError( 'Cannot add %s to a port channel '
                           '(Mlag peer-link cannot be member of a port channel)'
                           % mode.intf.shortname )
            reject = True
      if reject:
         return False

   if channelGroupId is None or \
          any( channelGroupId.id != olid for olid in oldLagIds ):
      return canShutdownOrRemoveIfs(
         intfModes[0], [ m.intf for m in intfModes ], True,
         'channel-group %s not changed' %
         ( 'statuses' if len( intfModes ) > 1 else 'status' ) )

   return True

LagCli.registerLagConfigCallback( canSetChannelGroup )

def handleVlanConfiguration( mode, vlanSet, toBeDeleted=False ):
   if not isPrimaryOrSecondaryMlag():
      return True
   cfg = mlagConfig
   localInterface = cfg.localIntfId
   peerLinkTrunkGroups = set()
   doesWarn = False
   if cfg.peerLinkIntfId:
      peerLinkIntf = IntfCli.Intf( cfg.peerLinkIntfId, mode )
      swIntfConfig = switchIntfConfigIfEnabled( mode, peerLinkIntf )
      if swIntfConfig:
         peerLinkTrunkGroups.update( swIntfConfig.trunkGroup.keys() )
   msg = ''
   if len( peerLinkTrunkGroups ) > 0:
      vlans = Vlan.getAll( mode )
      vlans = filter( lambda v: v.id in vlanSet.ids, vlans )
      vlans = filter( lambda v: 
                      not peerLinkTrunkGroups.isdisjoint( v.trunkGroups( mode ) ),
                      vlans )
      vlans.sort()
      for v in vlans:
         tgs = ', '.join( peerLinkTrunkGroups.intersection( v.trunkGroups( mode ) ) )
         msg = ( 'VLAN%s is a member of the trunk group "%s" assigned to ' +
                 'the MLAG peer link %s' ) % ( v.id, tgs, cfg.peerLinkIntfId )
         doesWarn = True
   for vlan in vlanSet.ids:
      if ( 'Vlan' + str( vlan ) ) == localInterface:
         msg = ( '%s is configured as the MLAG local interface' % localInterface )
         doesWarn = True
   if doesWarn:
      if toBeDeleted:
         mode.addError( msg + '; no vlans deleted'  )
      else:
         mode.addWarning( msg )
   return not doesWarn

VlanCli.extraGotoVlanModeHook.addExtension( handleVlanConfiguration )

VlanCli.canDeleteVlanHook.addExtension( 
   lambda mode, vlanSet: handleVlanConfiguration(mode, vlanSet, True) )

def canConfigureSwitchport( mode, intfName, no ):
   if no and intfName in mlagConfig.intfConfig:
      mode.addError( "Cannot configure routed port on a MLAG" )
      return False
   else:
      return True

VlanCli.switchportCmdHook.addExtension( canConfigureSwitchport )

def canConfigureTapTool( mode, intfName ):
   peerLinkName = mlagConfig.peerLinkIntfId
   if peerLinkName and peerLinkName == intfName:
      mode.addError( 'Cannot configure tap or tool on ' + \
            'MLAG peer link interface : ' + str( intfName ) )
      return False
   return True

TapAggIntfCli.switchportModeCmdHook.addExtension( canConfigureTapTool )

def canConfigureDot1q( mode, intfName, switchportMode ):
   if switchportMode != 'dot1qTunnel' :
      return True
   peerLinkName = mlagConfig.peerLinkIntfId
   if peerLinkName and peerLinkName == intfName:
      mode.addError( 'Cannot configure dot1q-tunnel on ' + \
            'MLAG peer link interface : ' + str( intfName ) )
      return False
   return True

VlanCli.switchportModeCmdHook.addExtension( canConfigureDot1q )

def canDeletePortTrunkGroup( mode, group ):
   '''This function determines whether to allow deletion of trunk group or not
   If the current link is peer link then and given group is on Mlag Peerlink,
   then does not allow deletion.
   parameters
   mode  - Current mode(ConfigIf mode).
   group - Delete only the group that is being passed. In case, no
           group is passed then delete all the groups present.
   '''
   if  not isPrimaryOrSecondaryMlag() or not mlagConfig:
      return True
   cfg = mlagConfig
   peerLinkName = cfg.peerLinkIntfId
   if peerLinkName != mode.intf.name:
      return True
   localInterface = cfg.localIntfId
   patt = re.search ( '[0-9]+' , localInterface )
   vlanId = int( patt.group( 0 ) ) if patt else None
   if not vlanId:
      return True
   VlanObj = Vlan( vlanId )
   currVlan = VlanObj if VlanObj in Vlan.getAll( mode ) else None
   if not currVlan:
      return True
   peerLinkTrunkGroups = set()
   commonTrunkGroup = []
   if cfg.peerLinkIntfId:
      peerLinkIntf = IntfCli.Intf( cfg.peerLinkIntfId, mode )
      swIntfConfig = switchIntfConfigIfEnabled( mode, peerLinkIntf )
      if swIntfConfig:
         peerLinkTrunkGroups.update( swIntfConfig.trunkGroup.keys() )
      commonTrunkGroup = list( peerLinkTrunkGroups.intersection(
                               currVlan.trunkGroups( mode ) ) )
   if not commonTrunkGroup:
      return True
   else:
      groupStr = ' '
      if group:
         if group not in commonTrunkGroup:
            return True
         else:
            groupStr = ' Mlag Trunk Group ' + group + ' is '
      else:
         groupStr = ' Mlag Trunk Groups are '
      msg = ( 'Interface %s is configured as the MLAG' +
              ' peer link,' + groupStr + 'not' +
              ' deleted' ) % ( mode.intf.name )
      mode.addError( msg )
      return False

VlanCli.canDeletePortTrunkGroupHook.addExtension( canDeletePortTrunkGroup )

def noSpeedChange( mode, intfList, linkmode, advertisedModes ):
   '''This function determines whether to allow change in link speed
   If link is Mlag peer link or it's part do not allow speed change
   parameters
   mode - the current mode (ConfigIf Mode)
   intfList - List of interfaces on which speed change required
   linkmode - The new speed to which change is required'''
   if  not isPrimaryOrSecondaryMlag() or not mlagConfig:
      return False
   cfg = mlagConfig
   commonIntfs = set()
   msg = 'no speed change done'
   peerLinkName = cfg.peerLinkIntfId
   if not peerLinkName:
      return False
   if peerLinkName.startswith( 'Ethernet' ):
      if peerLinkName not in [ intf.name for intf in intfList ]:
         return False
      commonIntfs.add( peerLinkName )
      outputMsg = ( 'Interface %s is configured as the MLAG peer link' +
               '; ' + msg ) % ( peerLinkName )
   elif peerLinkName.startswith( 'Port-Channel' ):
      portChannel = lagIntfStatusDir.get( peerLinkName )
      commonIntfs = set( portChannel.member.keys() ).intersection(
                     set( [ intf.name for intf in intfList ] ) )
      if not commonIntfs:
         return False
      outputMsg = ( 'Specified interface range would break the MLAG peer link ' +
               peerLinkName + '; ' + msg )
      if len( commonIntfs ) == 1:
         outputMsg = ( 'Interface %s is a member of the MLAG peer link ' +
                  peerLinkName + '; ' + msg ) % ( list( commonIntfs )[ 0 ] )
   mode.addError( outputMsg )
   return True

EthIntfCli.canPromptForAbortHook.addExtension( noSpeedChange )

def canSetSystemPriority( mode, newPriority ):
   '''This function determines whether to allow change in LACP system-priority
      on MlagDut with respect to Mlag State.
      mode - the current mode (Config Mode)
   '''
   if not isPrimaryOrSecondaryMlag():
      return True
   if LagCli.getCurrentSystemPriority() == newPriority:
      return True
   msg = 'LACP system-priority configuration is not allowed if '\
         'Mlag State is Active( primary/secondary )'
   mode.addError( msg )
   return False

LagCli.canSetSystemPriorityHook.addExtension( canSetSystemPriority )

def canSetFlowcontrolHook( mode, intf, direction, val ):
   if val == 'flowControlConfigOff':
      return True

   cfg = mlagConfig
   peerLinkName = cfg.peerLinkIntfId
   elis = lagIntfStatusDir.get( peerLinkName )
   if elis and intf in elis.member:
      mode.addError( 'Flow control cannot be enabled on members of the '
                     'MLAG peer link %s' % peerLinkName )
      return False
   if intf == peerLinkName:
      mode.addError( 'Flow control cannot be enabled on the MLAG peer link' )
      return False
   return True

EthIntfCli.canSetFlowcontrolHook.addExtension( canSetFlowcontrolHook )

def canSetUnidirectionalLinkMode( mode, intfs ):
   if not isPrimaryOrSecondaryMlag():
      return True
   canSetUniLinkMode = True
   if Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId( \
         mlagConfig.peerLinkIntfId ):
      elis = lagIntfStatusDir.get( mlagConfig.peerLinkIntfId )
      peerLinkMembers = set( elis.member )
   else:
      peerLinkMembers = set( [ mlagConfig.peerLinkIntfId ] )
   intfNames = map( lambda intf: intf.name , intfs )
   validPeerLinkMembers = peerLinkMembers.intersection( intfNames )
   if validPeerLinkMembers:
      mode.addError( 'Unidirectional link configuration is restricted on MLAG' \
              ' peer-link interface ' + ', '.join( sorted( validPeerLinkMembers ) ) )
      canSetUniLinkMode = False
   return canSetUniLinkMode
     
EthIntfCli.canSetUnidirectionalLinkModeHook.addExtension( \
      lambda intfModes: canSetUnidirectionalLinkMode( intfModes[ 0 ], \
             [ m.intf for m in intfModes ] ) )

canShutdownMlagHook = CliExtensions.CliHook()

def addMlagShutdownWarnings( mode ):
   for hook in canShutdownMlagHook.extensions():
      msg = hook( isPrimaryOrSecondaryMlag() )
      if msg:
         mode.addWarning( msg )

#-------------------------------------------------------------------------------
# Reload Hook
#-------------------------------------------------------------------------------

class Worry:
   def __init__( self ):
      pass

   def shouldBeWorried( self ):
      raise NotImplementedError

   def __str__( self ):
      raise NotImplementedError

class GenericWorry( Worry ):
   def __init__( self, mode ):
      Worry.__init__( self )
      self.mode = mode
      vi = EosVersion.VersionInfo( sysdbRoot=None )
      self.version_ = vi.version()
      fileUrl = FileUrl.bootConfig( mode )
      bootFile = fileUrl.get( 'SWI','(not set)' ) 
      if bootFile != '(not set)':
         context = Url.Context( *Url.urlArgsFromMode( mode ) )
         url = Url.parseUrl( bootFile, context )
         self.issuCompatibility = showMlagIssuCompatibility( mode=None, 
                                                             args={ 'URL': url } )
      else:
         # This should only happen in breadth tests.
         self.issuCompatibility = None
      
   def shouldBeWorried( self ):
      primaryOrSecondary = isPrimaryOrSecondaryMlag()
      t0( "isPrimaryOrSecondaryMlag=" + str( primaryOrSecondary ) )
      return primaryOrSecondary

   def __str__( self ):
      genericMsg = "If you are performing an upgrade, and the "\
                   "Release Notes for the new version of EOS "\
                   "indicate that MLAG is not backwards"\
                   "-compatible with the currently installed "\
                   "version (%s), the upgrade will result in "\
                   "packet loss." % self.version_ 

      msg = compatCheckMessage if self.issuCompatibility is None \
                               else self.issuCompatibility.renderStr()
      msg = msg + '\n' + fill( genericMsg ) 
      return msg

class ActivePartialWorry( Worry ):
   def __init__( self, mlagStat ):
      Worry.__init__( self )
      summary = portCategorySummary( mlagStat )
      ( disabledPorts, configuredPorts, inactivePorts,
        activePartialLocalUpPorts, activePartialRemoteUpPorts, 
        activeFullPorts ) = summary

      self.disabledPorts = disabledPorts
      self.configuredPorts = configuredPorts
      self.inactivePorts = inactivePorts
      self.activePartialLocalUpPorts = activePartialLocalUpPorts
      self.activePartialRemoteUpPorts = activePartialRemoteUpPorts
      self.activeFullPorts = activeFullPorts

   def shouldBeWorried( self ):
      primaryOrSecondary = isPrimaryOrSecondaryMlag()
      t0( "isPrimaryOrSecondaryMlag=" + str( primaryOrSecondary ) )
      t0( "activePartialLocalUpPorts=" + str( self.activePartialLocalUpPorts ) )
      return primaryOrSecondary and self.activePartialLocalUpPorts > 0

   def __str__( self ):
      # Only care about active partial local up
      notActiveModeWarning = "The following MLAGs are not in Active mode. " \
      "Traffic to or from these ports will be lost during the upgrade process:\n"
      detailedOutput = showMlagInt( None, mlagInterfaceArgs ).renderStr()
      return fill( notActiveModeWarning ) + "\n" + detailedOutput

class NotRestartableWorry( Worry ):
   def __init__( self, stpInputConf, stpConf, stpStat ):
      Worry.__init__( self )
      self.forceProtocolVersion = stpInputConf.forceProtocolVersion
      self.stableFileExists = stableFileExists( stpConf.bridgeName )
      self.recentlyRestarted = stpStat.recentlyRestarted

   def shouldBeWorried( self ):
      stpRunning = not ( self.forceProtocolVersion == 'none' or
                         self.forceProtocolVersion == 'backup' )
      primaryOrSecondary = isPrimaryOrSecondaryMlag()
      t0( "stpRunning=" + str( stpRunning ) )
      t0( "isPrimaryOrSecondaryMlag=" + str( primaryOrSecondary ) )
      t0( "stableFileExists=" + str( self.stableFileExists ) )
      t0( "recentlyRestarted=" + str( self.recentlyRestarted ) )
      stpRestartable = self.stableFileExists and not \
          self.recentlyRestarted
      return stpRunning and not stpRestartable and primaryOrSecondary

   def __str__( self ):
      return fill( "Stp is not restartable. Topology changes will occur "
                   "during the upgrade process." )

class ReloadDelayWorry( Worry ):
   def __init__( self, mlagConf, mlagHwStat ):
      Worry.__init__( self )
      self.reloadDelay = mlagConf.reloadDelay
      self.reloadDelayDefault = mlagHwStat.reloadDelay
      self.reloadDelayConfigured = \
          ( self.reloadDelay.reloadDelayType == "reloadDelayConfigured" )

   def shouldBeWorried( self ):
      primaryOrSecondary = isPrimaryOrSecondaryMlag()
      t0( "isPrimaryOrSecondaryMlag=" + str( primaryOrSecondary ) )
      t0( "reloadDelay=" + str( self.reloadDelay ) )
      t0( "reloadDelayDefault=" + str( self.reloadDelayDefault ) )
      t0( "reloadDelayConfigured ", self.reloadDelayConfigured )
      shouldBeWorried = primaryOrSecondary and \
          self.reloadDelay.delay < self.reloadDelayDefault and \
          self.reloadDelayConfigured
      return shouldBeWorried
   
   def __str__( self ):
      return fill( "The configured reload delay of %.0f seconds is below the "
                   "recommended value of %.0f seconds. A longer reload delay allows "
                   "more time to rollback an unsuccessful upgrade due to "
                   "incompatibility." %
                   ( self.reloadDelay.delay, self.reloadDelayDefault ) )

class ReloadDelayLacpConfigWorry( Worry ):
   def __init__( self, mlagConf ):
      Worry.__init__( self )
      self.reloadDelayMlag = mlagConf.reloadDelay
      self.reloadDelayNonMlag = mlagConf.reloadDelayNonMlag
      self.lacpStandby = mlagConf.lacpStandby

   def shouldBeWorried( self ):
      primaryOrSecondary = isPrimaryOrSecondaryMlag()
      t0( "isPrimaryOrSecondaryMlag=" + str( primaryOrSecondary ) )
      t0( "reloadDelayMlag=" + str( self.reloadDelayMlag ) )
      t0( "reloadDelayNonMlag=" + str( self.reloadDelayNonMlag ) )
      t0( "lacpStandby=" + str( self.lacpStandby ) )
      shouldBeWorried = primaryOrSecondary and self.lacpStandby and \
          self.reloadDelayNonMlag.delay < self.reloadDelayMlag.delay
      return shouldBeWorried
   
   def __str__( self ):
      return fill( reloadDelayLacpConfigMsg % ( self.reloadDelayNonMlag.delay, 
                                                self.reloadDelayMlag.delay ) )

class PeerPortsDisabledWorry( Worry ):
   def __init__( self, mlagStat, protoStat ):
      Worry.__init__( self )
      self.mlagState = mlagStat.mlagState
      self.portsErrdisabled = protoStat.portsErrdisabled
   
   def shouldBeWorried( self ):
      t0( "mlagState=" + str( self.mlagState ) )
      t0( "portsErrDisabled=" + str( self.portsErrdisabled ) )
      peerPortsErrdisabled = self.mlagState == 'primary' and \
          self.portsErrdisabled
      return peerPortsErrdisabled
   
   def __str__( self ):
      return fill( "The other MLAG peer has errdisabled interfaces. Traffic "
                   "loss will occur during the upgrade process." )

#-------------------------------------------------------------------------------
# show mlag issu warnings
#-------------------------------------------------------------------------------

def showMlagIssuWarnings( mode, args ):
   warnings = IssuWarnings()
   redStatus = mode.session_.entityManager_.redundancyStatus()
   # Warnings are only needed if I am the active supervisor and a
   # switchover would not be stateful.
   if redStatus.mode != 'standby' and \
          ( redStatus.protocol != 'sso' or redStatus.peerMode != 'standby' ):
      for w in ( GenericWorry( mode ),
                 ActivePartialWorry( mlagStatus ),
                 NotRestartableWorry( stpInputConfig, stpConfig, stpStatus ),
                 ReloadDelayWorry( mlagConfig, mlagHwStatus ),
                 ReloadDelayLacpConfigWorry( mlagConfig ),
                 PeerPortsDisabledWorry( mlagStatus, mlagProtoStatus ) ):
         if w.shouldBeWorried():
            warnings.issuWarnings.append( str( w ) ) # pylint: disable-msg=E1101
   return warnings

def mlagWarningsBeforeReload( mode, power, now ):
   if now:
      return True

   showMlagIssuWarnings( mode, {} ).render()
   return True

ReloadCli.registerReloadHook( mlagWarningsBeforeReload, 'ReloadMlagCheckCli', 
                              'RUN_FIRST' )

#-------------------------------------------------------------------------------
# Mounts
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global mlagConfig, mlagStatus, lagIntfStatusDir, ethIntfStatusDir, mlagHwStatus
   global stpStatus, stpInputConfig, stpConfig, mlagProtoStatus, ethIntfConfigDir
   global ipConfig
   # LazyMount mlag/config, Mlag::Config and its dependent paths
   mlagConfig = LazyMount.mount( entityManager, 'mlag/config', 'Mlag::Config',
                                 'r' )
   # LazyMount mlag/status, Mlag::Status and its dependent paths
   mlagStatus = MlagStatusLazyMounter( entityManager )
   mlagHwStatus = LazyMount.mount( entityManager, "mlag/hardware/status",
                                   "Mlag::Hardware::Status", "r" )
   mlagProtoStatus = LazyMount.mount( entityManager, 
                                      "mlag/proto", "Mlag::ProtoStatus", "rO" )
   lagIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/lag",
                                       "Interface::EthLagIntfStatusDir", "r" )
   ethIntfConfigDir = LazyMount.mount( entityManager, "interface/config/eth/intf",
                                       "Interface::EthIntfConfigDir", "r" )
   ethIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/intf",
                                       "Interface::EthIntfStatusDir", "r" )
   stpInputConfig = LazyMount.mount( entityManager, "stp/input/config/cli", 
                                     "Stp::Input::Config", "r" )
   stpStatus = LazyMount.mount( entityManager, "stp/status", "Stp::Status", "r" )
   stpStatusIs( stpStatus )
   stpConfig = LazyMount.mount( entityManager, "stp/config", "Stp::Config", "r" )
   stpConfigIs( stpConfig )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )

#-------------------------------------------------------------------------------
# show mlag issu compatibility <image>
#-------------------------------------------------------------------------------

def showMlagIssuCompatibility( mode, args ):
   url = args.get( 'URL' )
   import zipfile
   issuComp = IssuCompatibility()
   issuComp.isValid = False
   issuComp.currentVersion = EosVersion.VersionInfo( None ).version()
   assert issuComp.currentVersion
   newFileName = url.localFilename()
   issuComp.newImagePath = newFileName
   issuComp.detailedErrMessage = ""

   if isinstance( url, HttpUrl ):
      issuComp.detailedErrMessage = compatCheckMessage
      return issuComp

   try:
      if not url.open() or not FileUrl.urlIsSwiFile( url ):
         return issuComp
   except ( IOError, EnvironmentError ), e:
      issuComp.detailedErrMessage = "Error opening SWI file: " + \
          "".join( str(e).split("\n") )
      return issuComp
   if not zipfile.is_zipfile( newFileName ):
      return issuComp
   issuComp.newVersion, _ = getSwiVersionAndCompatibleImages( newFileName )
   issuComp.isValid = bool( issuComp.newVersion )
   return issuComp

