# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliParser
import CliToken.Ip
import ConfigMount
import Tracing
import LazyMount
from MlagVrrpModels import MlagPeerRouterMac

import importlib

# Importing dynamically because only FhrpCli.pyc
# is included in Fhrp-cli
FhrpCli = importlib.import_module( 'CliPlugin.FhrpCli' )

traceHandle = Tracing.Handle( 'MlagVrrp' )
t0 = traceHandle.trace0

mlagConfig = None
mlagStatus = None
mlagHwStatus = None
hwConfig = None

#-------------------------------------------------------------------------------
# [no|default] ip virtual-router mac-address mlag-peer
#-------------------------------------------------------------------------------
def isL2EntryBasedPerVlanRoutingEnabled():
   if 'strata' in hwConfig.keys():
      strataHwConfig = hwConfig[ 'strata' ][ 'hwConfig' ][ 'slice' ]
      strataMiscConfig = hwConfig[ 'strata' ][ 'globalConfig' ][ 'misc' ]
      if strataMiscConfig.perVlanRoutingEnabled:
         for key in strataHwConfig.keys():
            if strataHwConfig[ key ].features.l2EntryBasedPerVlanRoutingSupported:
               return True
   return False

def peerMacRoutingSupportedGuard( mode, token ):
   if mlagHwStatus.mlagPeerMacRoutingSupported and \
      not isL2EntryBasedPerVlanRoutingEnabled():
      return None
   return CliParser.guardNotThisPlatform

class PeerMacRoutingCmd( CliCommand.CliCommandClass ):
   syntax = 'ip virtual-router mac-address mlag-peer'
   noOrDefaultSyntax = syntax
   data = {
            'ip': CliToken.Ip.ipMatcherForConfig,
            'virtual-router': FhrpCli.nodeVirtualRouter,
            'mac-address': FhrpCli.matcherMacAddress,
            'mlag-peer': CliCommand.guardedKeyword( 'mlag-peer',
               helpdesc='Mlag peer router MAC', guard=peerMacRoutingSupportedGuard )
          }

   @staticmethod
   def handler( mode, args ):
      t0( 'Enable routing using Mlag peer MAC.' )
      mlagConfig.peerMacRoutingEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'Disable routing using Mlag peer MAC.' )
      mlagConfig.peerMacRoutingEnabled = False

BasicCli.GlobalConfigMode.addCommandClass( PeerMacRoutingCmd )

#-----------------------------------
# Hook for 'show ip virtual-router' 
#-----------------------------------
def showMlagPeerRouterMac( mode ):
   if not mlagHwStatus.mlagPeerMacRoutingSupported:
      return None
   return MlagPeerRouterMac( macAddress=mlagStatus.peerMacAddr,
                             macType='mlag',
                             configured=mlagStatus.peerMacRoutingEnabled )

# Register hook for displaying Mlag peer MAC address routing status.
FhrpCli.showVirtualRouterMacHooks.addExtension( showMlagPeerRouterMac )

def Plugin( entityManager ):
   global mlagConfig, mlagHwStatus, mlagStatus, hwConfig
   mlagConfig = ConfigMount.mount( entityManager, "mlag/config", "Mlag::Config",
                                   "w" )
   mlagStatus = LazyMount.mount( entityManager, "mlag/status", "Mlag::Status", "r" )
   mlagHwStatus = LazyMount.mount( entityManager, "mlag/hardware/status",
                                   "Mlag::Hardware::Status", "r" )
   hwConfig = LazyMount.mount( entityManager, "hardware", "Tac::Dir", "r" )
